#!/devtool/local/bin/perl

BEGIN
{
	if ($^O !~ /win32/i) {
		unshift @INC, ("/devtool/TOOL/tool/lib/perl5", "/devtool/TOOL/tool/lib/Perl", "/devtool/TOOL/tool/bin", "/SAP_DB/TESTDB");
	}
	unshift @INC, ($^O =~ /win32/i ? "\\SAP_DB\\TESTDB\\lib" : "/SAP_DB/TESTDB/lib");
}

use strict;
use Getopt::Long;
use Sys::Hostname;
use Net::HTTP;
use HTTP::Status;
use XML::Simple;
use File::Basename;
use QAConnect;

my $CHECK_FREQ 		= 3;
my $hostname		= lc(hostname());
my $counter		= 0;
my $forcable		= 1;
my %Opts;

$| = 1;

Usage() if (!GetOptions( \%Opts, 'h|help|?', 'name=s', 'init:i', 'lock', 'unlock', 'timeout:i', 'force', 'forcable=i', 'debug', 'profile=s', 'mul_chr=i') || $Opts{'h'} || !$Opts{'name'} || @ARGV);

QAConnect::setdbg($Opts{'debug'});

if (!defined($Opts{'init'}) && !-e "$Opts{'name'}.data")
{
	QAConnect::dbgout("Sema file '$Opts{'name'}.data' does not exist. Going to initial mode.");
	$Opts{'init'} = 0;
}

my $delay_str = ($Opts{'profile'} =~ /weekend/i ? "_WEEKEND" : "_WORKDAY");

my ($ok, $href) = QAConnect::httpsql_request("QADB_HTTPSQL", "SELECT DELAY_WORKDAY, DELAY_WEEKEND, TESTGROUPS FROM SERVERS WHERE HOST like '$hostname'");
(ref($href) and $href->{'Rows'}->{'Row'}[0]->{'TESTGROUPS'}) or QAConnect::throw_error("UNKNOWN HOSTNAME", "This host ($hostname) has no entry in the SERVERS table!");
if ($ok)
{
	my $timeout = ($Opts{'profile'} =~ /weekend/i ? $href->{'Rows'}->{'Row'}[0]->{'DELAY_WEEKEND'} : $href->{'Rows'}->{'Row'}[0]->{'DELAY_WORKDAY'});
	if (defined($Opts{'timeout'}) && ($Opts{'timeout'} == 0))
	{
		$Opts{'timeout'} = $timeout;
		QAConnect::dbgout("Timeout set to $timeout.");
	}
	if (defined($Opts{'init'}) && ($Opts{'init'} == 0))
	{
		$Opts{'init'} = $href->{'Rows'}->{'Row'}[0]->{'TESTGROUPS'};
		QAConnect::dbgout("Initial counter set to $href->{'Rows'}->{'Row'}[0]->{'TESTGROUPS'}.");
	}
}
else
{
	QAConnect::dbgout("QADB connection failed! Setting defaults...");
	if (defined($Opts{'timeout'}) && ($Opts{'timeout'} == 0))
	{
		$Opts{'timeout'} = 3600;
		QAConnect::dbgout("Timeout set to '$Opts{'timeout'}'.");
	}
	if (defined($Opts{'init'}) && ($Opts{'init'} == 0))
	{
		$Opts{'init'} = 2;
		QAConnect::dbgout("Initial counter set to '$Opts{'init'}'");
	}
}

if ($Opts{'init'})
{
	QAConnect::dbgout("Creating new semaphore '$Opts{'name'}'");
	unlink("$Opts{'name'}.lock", "$Opts{'name'}.queue", "$Opts{'name'}.data");
	$counter = $Opts{'init'};
	$forcable = defined($Opts{'forcable'}) ? $Opts{'forcable'} : 1;
	write_sema();
}
else
{
	QAConnect::dbgout("Using existing semaphore '$Opts{'name'}'");
	read_sema();
}

if (($Opts{'timeout'} > 0) && ($Opts{'mul_chr'} > 1)) {
	$Opts{'timeout'} = ($Opts{'timeout'} * ($Opts{'mul_chr'} - 1));
	QAConnect::dbgout("chr_mul: Multiplicated timeout is now $Opts{'timeout'}.");
}

$Opts{'timeout'} = -1 unless (defined($Opts{'timeout'}));

QAConnect::dbgout("STATUS:");
QAConnect::dbgout("    counter  = $counter");
QAConnect::dbgout("    forcable = $forcable");
QAConnect::dbgout("    force    = " . ($Opts{'force'} ? "yes" : "no"));
QAConnect::dbgout("    init     = " . (defined($Opts{'init'}) ? "$Opts{'init'}" : "undef"));
QAConnect::dbgout("    timeout  = " . (defined($Opts{'timeout'}) ? "$Opts{'timeout'}" : "undef"));


if ($Opts{'lock'})
{
	QAConnect::dbgout("Locking semaphore.");
	my ($done, $res) = run_timeout($Opts{'timeout'}, \&lock_sema);
	#read_sema();
	if ($done) {
		QAConnect::dbgout("Lock " . ($res ? "Succeeded.\n" : "Failed!"));
		exit ($res ? 0 : 1); 
	}
	elsif ($Opts{'force'} && $forcable == 1)
	{
		QAConnect::dbgout("Timeout! Enforcing semaphore below zero!");
		read_sema();
		$counter --;
		$forcable = defined($Opts{'forable'}) ? $Opts{'forcable'} : $forcable;
		write_sema();
		QAConnect::dbgout("Enforced semaphore succeeded!");
		exit(0);
	}
	else
	{
		QAConnect::dbgout("Timeout! Exiting...");
		if (defined($Opts{'forcable'}))
		{
			QAConnect::dbgout("Actualizing command line parameter 'forcable'.");
			read_sema();
			write_sema();
		}
		exit(88);
	}
}

if ($Opts{'unlock'})
{
	QAConnect::dbgout("Unlocking semaphore.");
	read_sema();
	$counter ++;
	$forcable = defined($Opts{'forable'}) ? $Opts{'forcable'} : $forcable;
	write_sema();
	QAConnect::dbgout("Semaphore increased one level.");
	exit(0);
}

exit(0);

##############################################################################
# lock_sema() - Waits for and locks a semaphore.
##############################################################################
sub lock_sema
{
	QAConnect::dbgout("Opening lock file");
	open (LOCK_FILE, ">$Opts{'name'}.queue") or (QAConnect::throw_error("LOCK FILE OPEN ERROR", "Can't open queue file '$Opts{'name'}.queue'!\n$!\n") and return 0);
	QAConnect::dbgout("Appending myself to locking queue...");
	flock (LOCK_FILE, 2) or (QAConnect::throw_error("LOCK FILE LOCK ERROR", "Can't lock lock file '$Opts{'name'}.queue'!\n$!\n") and return 0);
	QAConnect::dbgout("Now i'm first in queue! Let's get semaphore data.");
	read_sema();
	QAConnect::dbgout("Entering wait loop until counter becomes positive...");
	while($counter <= 0) { sleep($CHECK_FREQ); read_sema(1);	}
	QAConnect::dbgout("Counter is above zero!");
	$counter --;
	QAConnect::dbgout("Writing decreased counter to sema file.");
	write_sema();
	QAConnect::dbgout("Removing myself from wait queue.");
	flock(LOCK_FILE, 8);
	QAConnect::dbgout("Closing queue file.");
	close(LOCK_FILE);
	return 1;
}


##############################################################################
# read_sema() - reads out sema file.
##############################################################################
sub read_sema
{
	my ($ignore_dbg) = @_;
	
	QAConnect::dbgout("locking data file.");
	open (LOCK_IN, ">$Opts{'name'}.lock") or QAConnect::throw_error("LOCK FILE WRITE ERROR", "Can't open lock file '$Opts{'name'}.lock' for writing!\n$!\n", 1);
	flock(LOCK_IN, 2) or QAConnect::throw_error("FILE LOCK ERROR", "Can't lock sema file '$Opts{'name'}.lock'!");
	QAConnect::dbgout("Opening data file.");
	open (FILE_IN, "<$Opts{'name'}.data") or QAConnect::throw_error("DATA FILE READ ERROR", "Can't open sema file '$Opts{'name'}.data' for reading!\n$!\n", 1);
	QAConnect::dbgout("reading counter.");
	chomp ($counter = <FILE_IN>);
	$counter = $Opts{'init'} ? $Opts{'init'} : $counter;
	QAConnect::dbgout("reading forcable.");
	chomp ($forcable = <FILE_IN>);
	$forcable = defined($Opts{'forcable'}) ? $Opts{'forcable'} : $forcable;
	QAConnect::dbgout("closing data file.");
	close (FILE_IN);
	QAConnect::dbgout("unlocking data file.");
	flock(LOCK_IN, 8);
	close (LOCK_IN);
}

##############################################################################
# read_sema() - reads out sema file.
##############################################################################
sub write_sema
{
	QAConnect::dbgout("locking data file.");
	open (LOCK_IN, ">$Opts{'name'}.lock") or QAConnect::throw_error("LOCK FILE WRITE ERROR", "Can't open lock file '$Opts{'name'}.lock' for writing!\n$!\n", 1);
	flock(LOCK_IN, 2) or QAConnect::throw_error("FILE LOCK ERROR", "Can't lock sema file '$Opts{'name'}.lock'!");
	QAConnect::dbgout("Opening data file.");	
	open (FILE_OUT, ">$Opts{'name'}.data") or QAConnect::throw_error("FILE READ ERROR", "Can't open sema file '$Opts{'name'}.data' for reading!\n$!\n", 1);
	QAConnect::dbgout("locking data file.");
	flock(FILE_OUT, 2) or QAConnect::throw_error("FILE LOCK ERROR", "Can't lock sema file '$Opts{'name'}.data'!");
	QAConnect::dbgout("writing counter ($counter).");
	print FILE_OUT "$counter\n";
	QAConnect::dbgout("writing forcable ($forcable).");
	print FILE_OUT "$forcable\n";
	QAConnect::dbgout("unlocking data file.");
	flock(FILE_OUT, 8);
	QAConnect::dbgout("closing data file.");
	close (FILE_OUT);
	QAConnect::dbgout("unlocking data file.");
	flock(LOCK_IN, 8);
	close (LOCK_IN);
}

##############################################################################
# run_timeout() - Calls a function with timeout.
##############################################################################
sub run_timeout
{
	my ($timeout, $func_ptr, @args) = @_;
	
	eval {
		use Socket;
		use IO::Handle;
		use POSIX ":sys_wait_h";
		
		pipe(PARENT_RDR, CHILD_WTR); pipe(CHILD_RDR,  PARENT_WTR);
   		CHILD_WTR->autoflush(1); PARENT_WTR->autoflush(1);
    
		if (my $pid = fork)	{
			QAConnect::dbgout("parent forked.");
			close PARENT_RDR; close PARENT_WTR;
			while ($timeout != 0) {
				my $kill_pid = waitpid(-1,&WNOHANG);
				if ($kill_pid == $pid || $kill_pid == -1) {
					QAConnect::dbgout("child exited.");
					my $chld_res = <CHILD_RDR>;
					close CHILD_RDR; close CHILD_WTR;
					die "DONE: $chld_res\n";
				}
				sleep(1);
				$timeout --;
			}
			close CHILD_RDR; close CHILD_WTR;
			kill(9,$pid); die "TIMEOUT";
		} else {
			QAConnect::dbgout("child forked.");
			close CHILD_RDR; close CHILD_WTR;
			my $res = &{$func_ptr}(@args);
			print PARENT_WTR "$res\n";
			close PARENT_RDR; close PARENT_WTR;
			QAConnect::dbgout("child done.");
			exit;
		}
	};
	
	if($@)	{
		if    ($@ =~ /^DONE:\s(.*)\n/) { return (1, $1); }
		elsif ($@ =~ /TIMEOUT/)        { return (0, undef); }
		else			               { return (-1, $@); }
	}
	return (-1, undef);
}



##############################################################################
# Usage() - prints out something usable for a change.
##############################################################################
sub Usage
{
	print <<HELPEND;

Usage:	checklock -name <FILE> [ -init [VAL] ] [ -lock ] [ -unlock ] 
                  [ -timeout [VAL] ] [ -force ] [ -(h|help|?) ]
                  [ -forcable <0|1>]

Where:  -help           Shows this extremely helpful output.
        -name <FILE>    Sets the name of the semaphore, mandatory.
        -init [VAL]     Initializes the semaphore to VAL or Machine default.
        -lock           Locks the semaphore (sema.p()).
        -unlock         Unlocks the semaphore again (sema.v()).
        -timeout [VAL]  Sets timeout to VAL or Machine default.
        -force          Continues after timeout. (Sets sema below zero!)
        -forcable <0|1> Makes sema forcable, if set to 1. See above.
		
NOTE: If no value is given for init or timeout, the machine default will be
      read from QADB servers table.
	
HELPEND
	exit(1);
}
