/* Copyright (C) 1995 Bjoern Beutel. */

/* Description. =============================================================*/

/* Common routines for all Malaga GTK windows. */

/* types ====================================================================*/

typedef struct canvas canvas_t;

/* Colors that can be set by "set_color". */
typedef enum {BLACK, WHITE, RED} color_t; 

/* A value in the canvas together with its position and dimensions. */
typedef struct 
{ 
  int_t x, y, width, height, ascent;
  /* Private items follow. */
} pos_value_t;

/* A string in the canvas together with its position and dimensions. */
typedef struct 
{ 
  int_t x, y, width;
  /* Private items follow. */
} pos_string_t;

typedef struct 
{
  int_t x, y, width, height;
} rectangle_t;

typedef void (*expose_func_t)( canvas_t *canvas, rectangle_t *area );
/* Callback handler to draw exposed AREA in CANVAS. */

typedef void (*configure_func_t)( canvas_t *canvas, 
				  int_t *width_p, int_t *height_p );
/* Callback handler to compute *WIDTH_P and *HEIGHT_P for CANVAS. */

typedef bool_t (*mouse_func_t)( canvas_t *canvas, 
				int_t x, int_t y, int_t button );
/* Callback handler for mouse actions in CANVAS:
 * BUTTON != 0 => mouse button BUTTON pressed at (X,Y).
 * BUTTON == 0, X >= 0, Y >= 0 => mouse has moved to (X,Y). */

typedef void (*close_func_t)( canvas_t *canvas );
/* Callback hander to release memory when CANVAS is closed. */

/* Variables. ===============================================================*/

extern bool_t hangul; /* Set to TRUE if hangul output is used. */
extern string_t font_name, char_set;
extern int_t font_size;

/* Functions. ===============================================================*/

extern canvas_t *create_canvas( string_t title,
				string_t ps_file_name,
				rectangle_t *geometry,
				configure_func_t configure,
				expose_func_t expose,
				close_func_t close,
				mouse_func_t mouse_event,
				bool_t show_hanging_option,
				GtkItemFactoryEntry menu_items[], 
				int_t item_count );
/* Create a new canvas with TITLE and initial GEOMETRY.
 * When exporting Postscript, use PS_FILE_NAME as default file name.
 * Install the callbacks CONFIGURE, EXPOSE, CLOSE, and MOUSE_EVENT.
 * Add the "Hanging" option to the "Style" menu if SHOW_HANGING_OPTION == TRUE.
 * Add ITEM_COUNT additional MENU_ITEMS. 
 * The canvas is automatically configured and shown. */

extern void configure_canvas( canvas_t *canvas );
/* Configure CANVAS. This must be done when its layout has changed. */

extern void redraw_canvas( canvas_t *canvas );
/* Redraw CANVAS. This must be done when the layout has remained, 
 * but the content (e.g. marking) has changed. */

extern void show_canvas( canvas_t *canvas );
/* Show CANVAS and raise it. */

extern void hide_canvas( canvas_t *canvas );
/* Hide CANVAS. It may be shown again with "show_canvas". */

extern void make_visible( canvas_t *canvas, int_t x, int_t y );
/* Make sure the point (X,Y) is in the displayed part of CANVAS. */

extern void set_cursor( canvas_t *canvas, bool_t alternate );
/* Set cursor for CANVAS to alternate shape if ALTERNATE == TRUE. */

extern void go_canvas_bottom( canvas_t *canvas );
/* Display the bottom left of CANVAS. */

/* Primitives to lay out and draw a canvas. ---------------------------------*/

extern int_t get_space_width( canvas_t *canvas );
/* Get width of space character in CANVAS. */

extern int_t get_font_height( canvas_t *canvas );
/* Get font height in CANVAS. */

extern int_t get_font_ascent( canvas_t *canvas );
/* Get ascent of font in CANVAS. */

extern int_t get_border_width( canvas_t *canvas );
/* Get border with of CANVAS in pixels. */

extern void set_color( color_t color );
/* Set the COLOR for the following graphics operations. */

extern void draw_lines( int_t count, int_t x, int_t y, ... );
/* Called as "draw_lines( COUNT, x_1, y_1, x_2, y_2, ..., x_COUNT, y_COUNT )".
 * Draw lines:
 * from (x_1, y_1) to (x_2, y_2),
 * from (x_2, y_2) to (x_3, y_3),
 * ...
 * from (x_COUNT-1, y_COUNT-1) to (x_COUNT, y_COUNT). */

extern void draw_rectangle( int_t x, int_t y, int_t width, int_t height );
/* Draw a filled rectangle with upper left corner (X,Y), WIDTH and HEIGHT. */

extern void draw_circle( bool_t filled, int_t x, int_t y, int_t r );
/* Draw an arc with center (X,Y) and radius R. Fill it if FILLED == TRUE. */

/* Functions for strings to draw in a canvas.--------------------------------*/

extern pos_string_t *new_pos_string( string_t string );
/* Create new POS_STRING_T with value STRING. */

extern void config_pos_string( pos_string_t *pos_string, canvas_t *canvas );
/* Compute size of POS_STRING when drawn in CANVAS. */

extern void draw_pos_string( pos_string_t *pos_string, canvas_t *canvas );
/* Draw POS_STRING in CANVAS. */

extern void free_pos_string( pos_string_t **pos_string_p );
/* Free **POS_STRING_P. */

/* Functions for malaga values to draw in a canvas. -------------------------*/

extern pos_value_t *parse_pos_value( void );
/* Parse a value and return it as a "pos_value_t". */

extern void config_pos_value( pos_value_t *pos_value, canvas_t *canvas );
/* Compute size of POS_VALUE when drawn in CANVAS. */

extern void draw_pos_value( pos_value_t *pos_value, canvas_t *canvas );
/* Draw POS_VALUE in CANVAS. */

extern void free_pos_value( pos_value_t **pos_value_p );
/* Free **POS_VALUE_P. */

/* End of file. =============================================================*/
