#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <syslog.h>
#include <pwd.h>
#include <selinux/selinux.h>

int permanent = 0;

int setbool(char **list, int start, int end);
void rollback(char *list[], int start, int end);


void usage(void)
{
	fputs("\nUsage:  setsebool [ -P ] boolean value | bool1=val1 bool2=val2...\n\n", stderr);
	exit(1);
}

int main(int argc, char **argv)
{
	int rc, start;

	if (argc < 2) 
		usage();

	if (is_selinux_enabled() <= 0) {
		fputs("setsebool:  SELinux is disabled.\n", stderr);
		return 1;
	}

	if (strcmp(argv[1], "-P") == 0) {
		permanent = 1;
		if (argc < 3) 
			usage();
		start = 2;
	}
	else
		start = 1;

	/* Check to see which way we are being called. If a '=' is passed,
	   we'll enforce the list syntax. If not we'll enforce the original
	   syntax for backward compatibility. */
	if (strchr(argv[start], '=') == 0) {
		int len;
		char *bool_list[1];

		if ((argc - start) != 2)
			usage();

		/* Add 1 for the '=' */
		len = strlen(argv[start]) + strlen(argv[start+1]) + 2;
		bool_list[0]=(char *)malloc(len);
		if (bool_list[0] == 0) {
			fputs("Out of memory - aborting\n", stderr);
			return 1;
		}
		snprintf(bool_list[0], len, "%s=%s", argv[start], 
							argv[start+1]);
		rc = setbool(bool_list, 0, 1);
		free(bool_list[0]);
	}
	else 
		rc = setbool(argv, start, argc);

	return rc;
}

/* Given an array of strings in the form "boolname=value", a start index,
   and a finish index...walk the list and set the bool. */
int setbool(char **list, int start, int end)
{
	char *name, *value_ptr;
	int i=start, value;
	struct passwd *pwd;

	while (i < end) {
		name = list[i];
		value_ptr = strchr(list[i], '=');
		if (value_ptr == 0) {
			fprintf(stderr, 
			"setsebool: '=' not found in boolean expression %s\n",
				list[i]);
			rollback(list, start, i);
			return 4;
		}
		*value_ptr = 0;
		value_ptr++;
		if (strcmp(value_ptr, "1") == 0 || 
				strcasecmp(value_ptr, "true") == 0)
			value = 1;
		else if (strcmp(value_ptr, "0") == 0 || 
				strcasecmp(value_ptr, "false") == 0)
			value = 0;
		else {
			fprintf(stderr, "setsebool: illegal boolean value %s\n",
				value_ptr);
			rollback(list, start, i);
			return 1;
		}

		if(security_set_boolean(name, value)) {
			fprintf(stderr, 
				"Error setting boolean %s to value %d (%s)\n", 
				name, value, strerror(errno));
			rollback(list, start, i);
			return 2;
		}
		i++;

		/* Now put it back */
		value_ptr--;
		*value_ptr = '=';
	}

	/* At this point we know that everything is good. Let's write
	   the file if the -P option was given. */
	if (permanent) {
		char **names;
		const char *bool_file;
		char *tmp_bool_file;
		int rc, len, fd, j;

		rc = security_get_boolean_names(&names, &len);
		if (rc) {
			fprintf(stderr,
				"Unable to get boolean names:  %s\n",
				strerror(errno));
			rollback(list, start, i);
			return 5;
		}

		if (!len) {
			fprintf(stderr, 
			"Unable to get the boolean list from kernel - exiting\n"
				);
			rollback(list, start, i);
			return 6;
		}

		/* Open file */
		bool_file = selinux_booleans_path();
		tmp_bool_file = (char *) alloca (strlen(bool_file) + 8);
		strcpy(stpcpy(tmp_bool_file, bool_file), ".XXXXXX");
		fd = mkstemp(tmp_bool_file);
		if (fd < 0) {
			fprintf(stderr, 
				"Error creating boolean file %s\n", 
				bool_file);
			rollback(list, start, i);
			return 7;
			
		}

		/* Walk the list in pending memory, writing each to the file */
		for (j=0; j<len; j++) {
			char val_str[72];
			int len;
			int pending = security_get_boolean_pending(names[j]);
			len = snprintf(val_str, sizeof(val_str), "%s=%d\n", 
							names[j], pending);
			if (write(fd, val_str, len) != len) {
			close_remove_fail:
				close(fd);
			remove_fail:
				unlink(tmp_bool_file);
				rollback(list, start, i);
				return 8;
			}
		}

		if (fchmod(fd, S_IRUSR | S_IWUSR) != 0)
			goto close_remove_fail;
		close(fd);
		if (rename(tmp_bool_file, bool_file) != 0)
			goto remove_fail;
		syslog(LOG_NOTICE, "%s has been updated.", bool_file);
	}
	
	/* OK, let's do the commit */
	if (security_commit_booleans()) {
		fputs("Error committing booleans\n", stderr);
		return 3;
	}

	/* Now log what was done */
	pwd = getpwuid(getuid());
	i = start;
	while (i < end) {
		/* Error checking shouldn't be needed since we just did
		   this above and aborted if something went wrong. */
		name = list[i];
		value_ptr = strchr(name, '=');
		*value_ptr = 0;
		value_ptr++;
		if (pwd && pwd->pw_name)
			syslog(LOG_NOTICE, 
			    "The %s policy boolean was changed to %s by %s",
				name, value_ptr, pwd->pw_name);
		else
			syslog(LOG_NOTICE, 
			    "The %s policy boolean was changed to %s by uid:%d",
				name, value_ptr, getuid());
		i++;
	}

	return 0;
}

void rollback(char *list[], int start, int end)
{
	int i;

        for(i=start; i<end; i++)
                security_set_boolean(list[i],
                        security_get_boolean_active(list[i]));
}

