#ifndef PACKAGE_NAMESPACE_PHOENIX_H_INCLUDED
#define PACKAGE_NAMESPACE_PHOENIX_H_INCLUDED 1
////////////////////////////////////////////////////////////////////////////////
// phoenix.h
// phoenix<> provides "context singletons" with "phoexing" capabilities.
//
// Author: stephan beal <stephan@s11n.net>
// License: Public Domain
// CVS Revision: $Revision: 1.1.1.1 $
////////////////////////////////////////////////////////////////////////////////

#include <stdlib.h> // atexit()

#ifndef PHOENIX_SINGLETON_DEBUG
// enable debuggering to see when phoenixes are (re)created.
#define PHOENIX_SINGLETON_DEBUG 0
#endif

#include <iostream>
#if PHOENIX_SINGLETON_DEBUG
#  include <PACKAGE_NAMESPACE/cl_debuggering_macros.h> // COUT/CERR
#  include <typeinfo>
#  define PHOENIXCERR CERR << "phoenix<"<<typeid((base_type *)NULL).name()<<" , " << typeid((context_type *)NULL).name()<<"> "
#else
#  define PHOENIXCERR if(0) std::cout
#endif


namespace PACKAGE_NAMESPACE {

        
        /**
           Internal helper class to provide a default no-op
           initializer for phoenixed objects.

           See the phoenix<> class.
        */
        struct no_op_phoenix_initializer
        {
                /** Does nothing: This class is called no_op for a reason ;) */
                template <typename T>
                void operator()( T & ) { return; }
        };

        /**
           phoenix is class for holding singleton-style instances of
           BaseType objects. Rather than requiring that BaseType be a
           Singleton type, phoenix subclasses BaseType to add the
           phoenix-like capabilities. Phoenixing makes the shared
           object post-main() safe, in terms of object destruction
           order.

           Parameterized on:

           - BaseType: must be struct or class type and must be
           default-constructable. i have no clue what is supposed to
           happen if BaseType's dtor is not virtual. That said,
           phoenix has been successfully demonstrated with a BaseType
           of std::map, which has no virtual dtor.

           - ContextType: These objects are only singletons within the
           given ContextType.  That is, phoenix&lt;T,X&gt;::instance()
           will return a different object than phoenix&lt;T,Y&gt;
           will.

           - InitializerType: must be a unary functor accepting a
           BaseType &. It's return value is ignored. The default
           functor does nothing.  The InitializerType is called when a
           to-be-phoenixed object is initially created and whenever it
           is phoenixed. This is intended to be used, e.g., for
           re-populating a phoenixed shared object.
           TODO: investigate the implications of a predicate
           initializer, which would return false if the object could
           not be initialized.


           Whether or not BaseType is technically a singleton depends
           on entirely BaseType itself. This class is more often used
           to provide easy access to context-dependent shared objects,
           rather than pure singletons. The phoenix class itself is a
           true Singleton, but each combination of template arguments
           provides a different Singleton *type*, so the end effect is
           "context singletons."


           This is another attempt to solve the classic
           Keyboard-Console-Log problem, as discussed at length in
           <i>Modern C++ Design</i>. It relies on sane behaviour in
           the C library's atexit() function, which, as is shown in
           MC++D, is not the case on all systems. That said, the
           phoenix-specific behaviours are undefined on those systems,
           which is only to say that it might not be post-main() safe.


           Caveats:

           i am not 100% clear on all of the implications of this
           implementation's approach... my gut tells me i'm missing
           some significant bits. i mean, it <i>can't</i> have been
           this straightforward to solve ;). The very nature of the
           Phoenix Singleton problem makes it difficult to reliably
           test in real-world applications. That said, i have seen a
           objects be successfully phoenixed and atexit()ed, so it is
           known to at least "basically" work.

           There's a paper about "context singletons", this class,
           and some of it's implications, at:

           http://s11n.net/misccode/context_singletons.html


           [Much later: i've gotten more re-use out of this class than
           probably any other single class i've ever written.]
        */
        template <
                typename BaseType,
                typename ContextType = BaseType,
                typename InitializerType = no_op_phoenix_initializer
        >
        struct phoenix : public BaseType
        {
                /**
                   context_type is unused by this class, but might be useful
                   for type identification at some point.
                */
                typedef ContextType context_type;
                /**
                   The BaseType parameterized type.
                 */
                typedef BaseType base_type;

                /**
                   The functor type used to initialize this phoenixed object.
                */
                typedef InitializerType initializer_type;

                /**
                   Returns a shared instance of this object.

                   The instance() method will always return the same
                   address, though it is potentially possible
                   (post-main()) that the actual object living at that
                   address is different from previous calls.

                   It is never a good idea to hold on to the returned
                   reference for the life of an object, as that bypasses
                   the phoenixing capabilities.

                   If you ever delete it you're on you're own. That's
                   a Bad Idea.
                 */
                static base_type & instance()
                {
                        static this_type meyers;
                        static bool donethat = false;
                        if( this_type::m_destroyed )
                        {
                                PHOENIXCERR << "Phoenixing!" << std::endl;
                                donethat = false;
                                new( &meyers ) this_type;
                                atexit( this_type::do_atexit );
                        }
                        if( !donethat )
                        {
                                PHOENIXCERR << "initializing instance" << std::endl;
                                donethat = true;
                                initializer_type()( meyers );
                        }
                        PHOENIXCERR << "instance() == " <<std::hex<<&meyers<<std::endl;
                        return meyers;
                }

        private:

                /** A convenience typedef. */
                typedef phoenix<base_type,context_type,initializer_type> this_type;

                static bool m_destroyed;

                phoenix()
                {
                        PHOENIXCERR << "phoenix() @" << std::hex<< this << std::endl;
                        m_destroyed = false;
                }

                ~phoenix()
                {
                        PHOENIXCERR << "~phoenix() @" << std::hex<< this << std::endl;
                        m_destroyed = true;
                }
                /**
                   Destroys the shared object via a manual call to it's dtor.
                 */
                static void do_atexit()
                {
                        if( m_destroyed ) return;
                        PHOENIXCERR << "::do_atexit() @ " << std::hex << &instance() << std::endl;
                        static_cast<this_type &>(instance()).~phoenix(); // will eventually trigger the BaseType dtor
                }
        
        };
        template <typename T, typename C, typename I> bool phoenix<T,C,I>::m_destroyed = false;

} // namespace

#undef PHOENIX_SINGLETON_DEBUG
#undef PHOENIXCERR
#endif // PACKAGE_NAMESPACE_PHOENIX_H_INCLUDED


/*********************************************************************************
todo: check this out more closely:

Program received signal SIGSEGV, Segmentation fault.
0x4015e2d0 in std::string::compare(std::string const&) const () from /usr/lib/libstdc++.so.5
(gdb) bt
#0  0x4015e2d0 in std::string::compare(std::string const&) const () from /usr/lib/libstdc++.so.5
#1  0x0807eae2 in operator< <char, std::char_traits<char>, std::allocator<char> > (__lhs=@0x80a1b30, __rhs=@0xbfffec10)
    at basic_string.h:948
#2  0x0807eac4 in std::less<std::string>::operator()(std::string const&, std::string const&) const (this=0x4008cad0,
    __x=@0x80a1b30, __y=@0xbfffec10) at stl_function.h:197
#3  0x40064b34 in std::_Rb_tree<std::string, std::pair<std::string const, s11n::basic_serializer* (*)()>, std::_Select1st<std::pair<std::string const, s11n::basic_serializer* (*)()> >, std::less<std::string>, std::allocator<std::pair<std::string const, s11n::basic_serializer* (*)()> > >::find(std::string const&) (this=0x4008cac8, __k=@0xbfffec10) at stl_tree.h:1271
#4  0x400647ff in std::map<std::string, s11n::basic_serializer* (*)(), std::less<std::string>, std::allocator<std::pair<std::string const, s11n::basic_serializer* (*)()> > >::find(std::string const&) (this=0x4008cac8, __x=@0xbfffec10) at stl_map.h:468
#5  0x4006477b in s11n::instantiator<s11n::basic_serializer, std::string>::instantiate(std::string const&) (key=@0xbfffec10)
    at instantiator.h:151
#6  0x40064712 in s11n::class_loader<s11n::basic_serializer, std::string, false>::load(std::string const&) const (this=0x4008caf0,
    key=@0xbfffede0) at class_loader.h:298
#7  0x40067f6e in s11n::serializer_loader::load(std::string const&) const (this=0x4008caf0, key=@0xbfffede0)
    at serializer_loader.cpp:31
#8  0x40063454 in s11n::node_loader::load(std::istream&) const (this=0x4008cad4, is=@0x80cf5d0) at node_loader.cpp:69
#9  0x400639ea in s11n::node_loader::load(std::string const&) const (this=0x4008cad4, key=@0xbfffef10) at node_loader.cpp:95
#10 0x400632e3 in s11n::node_loader::load_node(std::string const&) (key=@0xbfffef10) at node_loader.cpp:40
#11 0x40060007 in s11n::libconfig_initializer::operator()(s11n::libconfig&) (this=0xbfffef67, conf=@0x4008ca60) at libconfig.cpp:53
#12 0x4005fda3 in s11n::phoenix<s11n::libconfig, s11n::libconfig, s11n::libconfig_initializer>::instance() () at phoenix.h:152
#13 0x4005fb53 in s11n::config() () at libconfig.cpp:65
#14 0x40069327 in s11n::basic_serializer::use_indentation() const (this=0xbffff0d0) at serializer.cpp:99
#15 0x4006957e in s11n::funtxt_serializer::serialize(s11n::s11n_node const&, std::ostream&) const (this=0xbffff0d0,
    node=@0x4008ca60, os=@0x80a4350) at serializer.cpp:153
#16 0x08073532 in s11n::s11n_io<s11n::funtxt_serializer>::save(s11n::s11n_node const&, std::ostream&) (node=@0x4008ca60,
    os=@0x80a4350) at s11n_io.h:77
#17 0x400605d7 in s11n::s11n_io<s11n::funtxt_serializer>::save(s11n::s11n_node const&, std::string const&) (node=@0x4008ca60,
    filename=@0x4008ca84) at s11n_io.h:94
#18 0x400603ac in ~libconfig (this=0x4008ca60) at libconfig.cpp:33
#19 0x40060153 in ~phoenix (this=0x4008ca60) at phoenix.h:174
#20 0x4005fc63 in __tcf_0 () at phoenix.h:140
#21 0x401ee010 in exit () from /lib/i686/libc.so.6
#22 0x401d7d1f in __libc_start_main () from /lib/i686/libc.so.6
#23 0x0806d4b1 in _start () at ../sysdeps/i386/elf/start.S:102


 *********************************************************************************/

