#ifndef PACKAGE_NAMESPACE_INSTANTIATOR_H_INCLUDED
#define PACKAGE_NAMESPACE_INSTANTIATOR_H_INCLUDED 1
// Author: stephan beal <stephan@s11n.net>
// License: Public Domain

#include <string>
#include <map>

#include <functional>

#include "phoenix.h" // i don't like this dep, but i also don't like
                     // what happens in some cases when i don't use
                     // phoenix. :/

namespace PACKAGE_NAMESPACE
{

	/**
           object_factory is a helper object factory for the classes
           instantiator and class_loader.

           SubT must derive from (or be) T and must be Default
           Constructuable on the heap. In short, the following must be
           able to succeed:

<pre>
T * foo = new SubT;
</pre>
        */

	template < class T, class SubT = T >
        struct object_factory
	{
                /**
                   A typedef for the first template parameter for this
                   type.
                */
                typedef T result_type;

                /**
                   A typedef for the second template parameter for this
                   type.
                */
                typedef SubT actual_type;

		/**
                   This creates a new SubT, which is assumed to be a
                   subclass of T.  It can be used as a factory for
                   instantiator & class_loader.
                */
		static result_type *new_instance()
		{
			return new actual_type;
		}

                /**
                   A convenience factory to assist in the registration
                   of abstract base types. It can be used as a factory
                   for such types, avoiding the problem that the default
                   factory will try to use 'new T', which won't work
                   with abstract types.

                   It always returns 0.
                */
		static result_type * no_instance()
		{
			return 0;
		}


                /**
                   Same as new_instance();
                */
                result_type * operator()() const
                {
                        return new_instance();
                }
	};

        /** Internal marker class */
        struct instantiator_sharing_context {};

        /**
           instantiator is essentially a static classloader, capable of
           loading classes by using registered factories for a given
           set of keys (e.g., class names).

           Classloaders, at least in my experience, need to be able to
           load all classes which derive from some given type. Without
           a common base class, one can't safely attempt to cast from
           an arbitrary pointer to the type we want to load. That's
           where the BaseType parameter comes in. All objects
           instantiated via this loader must inherit from BaseType.

           KeyType is a type which specifies the type of key used to
           look up classes, defaulting to std::string.

           Both BaseType and KeyType must be Default Constructable on
           the heap (e.g., via new BaseType()).

           This class holds no state, thus it can be copied quickly.

           Sample usage:
<pre>
typedef instantiator&lt;MyClass&gt; CL;
CL::register_factory( "my_key" );
MyClass *foo = CL::load( "some_key" ); // == NULL
foo = CL::instantiate( "my_key" ); // == a new object
</pre>

          Note that all instantiators of the same type use the same
          object factories. The ContextType template parameter can be
          used to limit the scope of the object factory registrations
          to a specific context: instantiators with different Contexts
          use different maps. ContextType is only used as a type, and
          is never instantiated by this class.
        */
	template < class BaseType,
                   class KeyType = std::string,
                   class ContextType = instantiator_sharing_context
                   >
        class instantiator
	{
        public:
                /**
                   A typedef for the BaseType used by this class.
                */
		typedef BaseType value_type;

                /**
                   A typedef for the KeyType used by this class.
                */
		typedef KeyType key_type;

                /** Same as ContextType */
                typedef ContextType context_type;

                /**
                   Same as KeyType, for conformance with the the
                   Adaptable Unary Functor model.
                 */
                typedef KeyType argument_type;

                /**
                   Same as (BaseType *), for conformance with the
                   Adaptable Unary Functor model.
                */
                typedef BaseType * result_type;

                /**
                   Convenience typedef.
                 */
		typedef instantiator< BaseType, KeyType, ContextType > ThisType;


                /**
                   The type of factories used by this class: a
                   function taking void and returning (value_type
                   *). See factory_map().

                   todo: implement proper functor support.
                */
                typedef value_type *( *factory_type ) ();

                /**
                   Internal container type used for mapping keys to
                   factories.
                */
		typedef std::map < key_type, factory_type > object_factory_map;


		/**
                   Tries to instantiate an instance of value_type
                   using the given key. Returns NULL if no class could
                   be loaded for the given key.

                   The caller takes responsibility for the returned pointer.
                */
		static value_type * instantiate( const key_type & key )
		{
			typename object_factory_map::const_iterator it = factory_map().find( key );
			if ( it != factory_map().end() )	// found a factory?
			{
				return ( it->second ) ();	// run our factory.
			}
			return 0;
		}

                /**
                   Same as instantiate( key ).
                 */
                result_type operator()( const argument_type & key )
                {
                        return instantiate( key );
                }


		/**
                   Registers a factory using the given key. If fp is
                   NULL then a default factory is used. Note that fp
                   may not return a type other than
                   ThisType::value_type *, but the actual object it
                   creates may be a polymorphic subclass of
                   value_type. See the object_factory class for a
                   factory which does this subtype-to-base conversion.
                */
		static void register_factory( const key_type & key, factory_type fp = 0 )
		{
                        if( ! fp ) fp = object_factory<value_type>::new_instance;
                        factory_map().insert( object_factory_map::value_type( key, fp ) );
                }

		/**
                   Convenience wrapper around register_factory( key,
                   factory_for_SubOfBaseType ).

                   Registers a factory for ThisType::value_type, using
                   the given key and a default factory for producing
                   SubOfBaseType objects. SubOfBaseType must be a
                   subtype of ThisType::value_type.
                */
                template <typename SubOfBaseType>
		static void register_subtype( const key_type & key, factory_type fp = 0 )
		{
                        if( ! fp ) fp = object_factory<value_type,SubOfBaseType>::new_instance;
                        register_factory( key, fp );
		}


                /**
                   Returns the internal key-to-factory map. It is safe
                   for clients to modify this except in multi-threaded
                   environments, and then all guarantees go out the
                   window. That said, it should never be necessary for
                   clients to use this.

                   It is safe to call this post-main(), but such calls
                   may return an empty map!
                */
		static object_factory_map & factory_map()
		{
			return phoenix<object_factory_map,ThisType>::instance();
		}

                /**
                   Returns true if the given key is registered. This is sometimes useful
                   for checking whether a factory needs to be re-registered, which
                   is sometimes necessary post-main(), when the internal map gets hosed
                   before clients are done using it.
                */
                static bool is_registered( const key_type & key )
                {
                        return factory_map().end() != factory_map().find( key );
                }


	}; // class instantiator

} // namespace
#endif // PACKAGE_NAMESPACE_INSTANTIATOR_H_INCLUDED
