#ifndef simplexml_SERIALIZER_H_INCLUDED
#define simplexml_SERIALIZER_H_INCLUDED 1

////////////////////////////////////////////////////////////////////////
// data_node_serializers.h: some file parsers for the s11n framework
//
// License: Public Domain
// Author: stephan@s11n.net
////////////////////////////////////////////////////////////////////////

#define MAGIC_COOKIE_SIMPLEXML "<!DOCTYPE S11N_NS::simplexml>"

namespace S11N_NS {
	namespace io {


                /***
                    The sharing namespace defines some "sharing contexts"
                    for use with S11N_NS::phoenix. They are used to
                    provide contexts in which disparate framework components
                    can share data.
                */
                namespace sharing {
                        /**
                           Sharing context used by simplexml_serializer.
                         */
                        struct simplexml_sharing_context {};
                }
                typedef std::map<std::string,std::string> entity_translation_map;

                /**
                   The entity translations map used by simplexml_serializer.
                 */
                entity_translation_map & simplexml_serializer_translations();



// TAB() is a helper macro for some serializers.
#define TAB(LEVEL,ECHO) indent = ""; for( size_t i = 0; i < depth + LEVEL; i++ ) { indent += '\t'; if(ECHO) dest << '\t'; }

                /**
                   De/serializes objects from/to a simple XML grammar,
                   with properties stored as XML attibutes and children
                   stored as subnodes.
                */
                template <typename NodeType>
                class simplexml_serializer : public tree_builder_lexer<NodeType,sharing::simplexml_sharing_context>
                {
                public:
                        typedef NodeType node_type;

                        typedef simplexml_serializer<node_type> this_type; // convenience typedef
                        typedef tree_builder_lexer<node_type,sharing::simplexml_sharing_context> parent_type; // convenience typedef

                        simplexml_serializer() : parent_type( "simplexml_data_nodeFlexLexer" ), m_depth(0)
                        {
                                this->magic_cookie( MAGIC_COOKIE_SIMPLEXML );
                        }

                        virtual ~simplexml_serializer() {}

                        /**
                           Reimplemented to return this type's entity
                           translation map.
                         */
                        virtual const entity_translation_map & entity_translations() const
                        {
                                return simplexml_serializer_translations();
                        }


                        /**
                           Writes src out to dest.
                        */
                        virtual bool serialize( const node_type & src, std::ostream & dest )
                        {
                                size_t depth = this->m_depth++;
                                if ( 0 == depth )
                                {
                                        dest << this->magic_cookie() << '\n';
                                }


                                std::string nname = src.name();
                                std::string impl = src.impl_class();
                                std::string indent;
                                const entity_translation_map & trans = this->entity_translations();

                                std::string ximpl = impl;
                                S11N_NS::translate_entities( ximpl, trans, false );

                                TAB(0,1);
                                dest << "<" << nname << " s11n_class=\"" << ximpl << "\"";

                                std::string propval;
                                std::string key;

                                typename node_type::const_iterator it = src.begin(), et = src.end();
                                if ( it != et )
                                {
                                        for ( ; it != et; ++it )
                                        {
                                                if ( "CDATA" == ( *it ).first )
                                                        continue;	// special treatment later on
                                                key = (*it).first;
                                                propval = (*it).second;
                                                S11N_NS::translate_entities( propval, trans, false );
                                                dest << " " << key << "=\"" << propval << "\"";
                                        }
                                }

                                bool use_closer = false; // if false then an element can <close_itself />
                                if ( src.is_set( "CDATA" ) )
                                {
                                        dest << ">";
                                        use_closer = true;
                                        dest << "<![CDATA[" << src.get_string( "CDATA" ) << "]]>";
                                }


                                bool tailindent = false;

                                if( src.children().end() != src.children().begin() )
                                {
                                        if( ! use_closer ) dest << '>';
                                        use_closer = true;
                                        tailindent = true;
                                        dest << '\n';
                                        std::for_each( src.children().begin(),
                                                       src.children().end(),
                                                       node_child_simple_formatter<this_type>( *this,
                                                                                         dest,
                                                                                         "",
                                                                                         "" )
                                                       );
                                }

                                dest << ( tailindent ? indent : "" );
                                if( use_closer )
                                {
                                        dest << "</" << nname << '>';
                                }
                                else
                                {
                                        dest << " />";
                                }
                                dest << '\n';

                                if( 0 == depth )
                                {
                                        dest.flush();
                                        // if we don't do this then the client is possibly forced to flush() the stream :/
                                }
                                --this->m_depth;
                                return true;
                        }


                private:
                        size_t m_depth;
                };


	} // namespace io
} // namespace S11N_NS
#undef TAB

#endif // simplexml_SERIALIZER_H_INCLUDED
