/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include "pxmchem-fragment.h"
#include "pxmchem-plugin.h"
#include "pxmchem-formula.h"
#include "pxmchem-polchemdef.h"
#include "pxmchem-masscalc.h"
#include "pxmchem-monomer.h"




/* returns the number of oligomers generated, or -1 if an error
   occurred.
 */
gint
pxmchem_fragment_polymer (PxmPolymer *polymer,
			  GPtrArray *GPA,
			  PxmPolchemdef *polchemdef,			
			  PxmFragOpt *fragopt, 
			  PxmCalcOpt *calcopt)
{
  gint count = -1;
  
  g_assert (polymer != NULL && polymer->monomerGPA != NULL);

  g_assert (polchemdef != NULL);
  
  g_assert (GPA != NULL);

  g_assert (fragopt != NULL);
  g_assert (fragopt->fragspec != NULL);
  
  if (polymer->monomerGPA->len <= 0)
    return 0;

  /* We get a fragopt from the parameters in which a fragspec can be
     accessed. This fragspec object contains all the data necessary to
     fragment the polymer. The calcopt contains the delimitation of
     the polymer to be actually fragmented (i.e. start_idx and
     end_idx). The generated oligomers (fragments) are stored in the
     GPtrArray GPA passed as parameter.
  */

  /* Depending on the fragspec in the fragopt parameter, the function
     that will actually be doing the work changes. This is because the
     fragmentation process is significantly different depending on the
     end of the polymer to fragment from which the fragments are
     generated. For example fragmenting a, b, c fragments for proteins
     is significantly different than fragmenting immonium ions or x,
     y, z fragments (always from the same polymer sequence).
  */
  if (fragopt->fragspec->end == PXM_FRAG_END_NONE)
    {
      count = pxmchem_fragment_polymer_end_none (polymer, GPA, polchemdef,
						 fragopt, calcopt);
      
      return count;
    }
  else if (fragopt->fragspec->end == PXM_FRAG_END_LEFT)
    {
      count = pxmchem_fragment_polymer_end_left (polymer, GPA, polchemdef,
						 fragopt, calcopt);
      
      return count;
    }
  else if (fragopt->fragspec->end == PXM_FRAG_END_RIGHT)
    {
      count = pxmchem_fragment_polymer_end_right (polymer, GPA, polchemdef,
						  fragopt, calcopt);
      
      return count;
    }
  else
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: the end member of the fragspec is invalid.\n"),
	     __FILE__, __LINE__);
  
      return -1;
    }
  
  return -1;
}



gint
pxmchem_fragment_polymer_end_none (PxmPolymer *polymer,
				   GPtrArray *GPA,
				   PxmPolchemdef *polchemdef,			
				   PxmFragOpt *fragopt, 
				   PxmCalcOpt *calcopt)
{
  gint count = 0;
  gint iter = 0;
  gint jter = 0;
  
  
  PxmOligomer *oligomer_1 = NULL;
  PxmOligomer *oligomer_2 = NULL;
  
  PxmMonomer *monomer = NULL;
  
  PxmProp *prop = NULL;
  PxmProp *prop_new = NULL;
  
  PxmFragSpec *fragspec = NULL;
  PxmFragRule *fragrule = NULL;
  
  

  /* "end_none" means that the fragments that we generate for the
     polymer sequence stretch considered are only one
     monomer-long. See the example of immonium ions in the
     proteinaceous world.

     That means that for each iteration in the for loop below we
     create an oligomer.
  */



  g_assert (polymer != NULL && polymer->monomerGPA != NULL);

  g_assert (polchemdef != NULL);
  
  g_assert (GPA != NULL);

  g_assert (fragopt != NULL);
  
  fragspec = fragopt->fragspec;
  g_assert (fragspec != NULL);
  
  if (polymer->monomerGPA->len <= 0)
    return 0;

  /* We get a calcopt->end_idx index that truly is the index of the last
     monomer to take into account during the fragmentation, which
     is why we want to include, thence the "iter < calcopt->end_idx + 1"
     below.
  */
  for (iter = calcopt->start_idx; iter < calcopt->end_idx + 1; iter++)
    {
      /* Allocate a new oligomer that we'll feed with data garnered
	 from the function calls below.
      */
      oligomer_1 = pxmchem_oligomer_new ();
      
      oligomer_1->polymer = polymer;

      oligomer_1->masspair = libpolyxmass_masspair_new ();

      monomer = g_ptr_array_index (polymer->monomerGPA, iter);
      g_assert (monomer != NULL);
      
      /* Account for the mass of the iterated monomer. *******************
       */
      if (FALSE ==
	  pxmchem_monomer_account_mass_by_name (monomer->name,
						polchemdef->monomerGPA,
						polchemdef->atomGPA,
						1, /*times*/
						oligomer_1->masspair))
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: failed accounting mass for monomer '%s'\n"),
		 __FILE__, __LINE__, monomer->name);
	  
	  pxmchem_oligomer_free (oligomer_1);
	  
	  return -1;
	}
      
      /* Account for the mass of the iterated monomer's modifs ***********
       */
      if (calcopt->mnm_chement & PXMCHEMENT_MNM_MODIF)
	{
	  if (FALSE == 
	      pxmchem_monomer_account_mass_for_modif (monomer,
						      polchemdef->modifGPA,
						      polchemdef->atomGPA,
						      1, /*times*/
						      oligomer_1->masspair))
	    {	  
	      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		     _("%s@%d: failed accounting for modif of "
		       "monomer named '%s'\n"),
		     __FILE__, __LINE__, monomer->name);
	      
	      pxmchem_oligomer_free (oligomer_1);
	      
	      return -1;
	    }
	}
      
      /* Also, if the user wanted that the modification eventually found
	 in the monomer be signalled, we can do this.
      */
      if (fragopt->mnm_chement != PXMCHEMENT_MNM_NONE)
	{
	  prop =  libpolyxmass_prop_find_prop (monomer->propGPA,
					   NULL,
					   NULL,
					   "MODIF",
					   NULL,
					   PXM_UNDEF_PROP_CMP);
	  
	  if (NULL != prop)
	    {
	      /* The monomer is modified. Just set a prop stating this.
	       */
	      prop_new = libpolyxmass_prop_new ();

	      /* The nomenclature here is more complex than with the
		 polymer cleavage case, because the fact the monomer
		 number depends on the PxmEnd of the fragmentation
		 complicates much the work. So we simply construct a
		 prop name by telling what is the offset position of
		 the monomer that is modified with respect to the end
		 at which the fragmentation occurs, and also by
		 telling what the true index of the monomer is in the
		 whole polymer sequence. We need this datum so that
		 later we can access the monomer in question and thus
		 get to the name of the modification.... This is when
		 GUI programs need to display... For this case this is
		 not particularly meaningful, but when PxmEnd is LEFT
		 or right, then it is.
	      */
	      libpolyxmass_prop_set_name (prop_new, "POS/IDX/MODIF");
	      prop_new->data =g_strdup_printf ("%d/%d/%s",
					        /* pos in the oligomer */
					       iter + 1,
					       /* index in the sequence */
					       iter,
					       (gchar *) prop->data);

	      g_ptr_array_add (oligomer_1->propGPA, prop_new);
	    }
	}
  
      /* If the fragspec indicated that a chemical action should be
	 performed onto the monomer being fragmented, then do
	 it. Remember that some fragmentation patterns do not involve
	 a chemical modification of the monomer. This is why the
	 actform may be NULL.
      */
      if (fragspec->actform != NULL)
	{
	  if (FALSE == 
	      pxmchem_actform_account_mass (fragspec->actform,
					    polchemdef->atomGPA,
					    1 /*times*/, 
					    oligomer_1->masspair))
	    {
	      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		     _("%s@%d: failed accounting for fragspec's "
		       "actform: '%s'\n"),
		     __FILE__, __LINE__, fragspec->actform);

	      pxmchem_oligomer_free (oligomer_1);
	      
	      return -1;
	    }
	}
      
      /* We may have to account for the left end modification or the
	 right end modification. We only take into account the left
	 end modif if the currently iterated monomer is indeed the
	 left end monomer of the initial polymer sequence. We only
	 take into account the right end modif if the currently
	 iterated monomer is indeed the right end monomer of the
	 initial polymer sequence. We test this now:
      */
      if (calcopt->plm_chement & PXMCHEMENT_PLM_LEFT_MODIF 
	  && iter == 0)
	{
	  /* Take into account the left end modif.
	   */
	  if (FALSE == 
	      pxmchem_masscalc_polymer_account_left_end_modif (polymer,
							       oligomer_1->
							       masspair,
							       polchemdef))
	    {
	      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		     _("%s@%d: failed accounting for polymer's "
		       "left end modif.\n"),
		     __FILE__, __LINE__);
	      
	      pxmchem_oligomer_free (oligomer_1);
	      
	      return -1;
	    }
	  
	  /*
	    debug_printf (("accounted left end modif "
	    "- mono=%f and avg=%f\n",
	    masspair->mono, masspair->avg));
	  */
	}
      
      if (calcopt->plm_chement & PXMCHEMENT_PLM_RIGHT_MODIF
	  && iter == polymer->monomerGPA->len - 1)
	{
	  /* Take into account the right end modif.
	   */
	  if (FALSE == 
	      pxmchem_masscalc_polymer_account_right_end_modif (polymer,
								oligomer_1->
								masspair,
								polchemdef))
	    {
	      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		     _("%s@%d: failed accounting for polymer's "
		       "right end modif.\n"),
		     __FILE__, __LINE__);
	      
	      pxmchem_oligomer_free (oligomer_1);
	      
	      return -1;
	    }
	  
	  /*
	    debug_printf (("accounted right end modif "
	    "- mono=%f and avg=%f\n",
	    masspair->mono, masspair->avg));
	  */
	}
      
      if (fragopt-> plm_chement & PXMCHEMENT_PLM_LEFT_MODIF 
	  && iter == 0)
	{
	  prop = libpolyxmass_prop_find_prop (polymer->propGPA,
					  NULL,
					  NULL,
					  "LEFT_END_MODIF",
					  NULL,
					  PXM_UNDEF_PROP_CMP);
	      
	  if (NULL != prop)
	    {
	      if (prop->data != NULL)
		{
		  prop_new = 
		    libpolyxmass_prop_dup (prop, PXM_MODIF_PROP_DUP_DEEP_YES);
		      
		  g_ptr_array_add (oligomer_1->propGPA, prop_new);
		}
	    }
	}

      if (fragopt-> plm_chement & PXMCHEMENT_PLM_RIGHT_MODIF 
	  && iter == polymer->monomerGPA->len - 1)
	{
	  prop = libpolyxmass_prop_find_prop (polymer->propGPA,
					  NULL,
					  NULL,
					  "RIGHT_END_MODIF",
					  NULL,
					  PXM_UNDEF_PROP_CMP);
	      
	  if (NULL != prop)
	    {
	      if (prop->data != NULL)
		{
		  prop_new = 
		    libpolyxmass_prop_dup (prop, PXM_MODIF_PROP_DUP_DEEP_YES);
		      
		  g_ptr_array_add (oligomer_1->propGPA, prop_new);
		}
	    }
	}


      /* If the caller wants that the sequence of the fragment be
	 stored in it, just put the single monomer in it.
      */
      if (fragopt->put_sequence == TRUE)
	{
	  prop_new = libpolyxmass_prop_both_strings_new ("SEQUENCE", 
						     monomer->code);

	  g_ptr_array_add (oligomer_1->propGPA, prop_new);
	}


      /* We now have to account for the fragrules...
       */
      for (jter = 0; jter < fragspec->fgrGPA->len; jter++)
	{
	  fragrule = g_ptr_array_index (fragspec->fgrGPA, jter);
	  g_assert (fragrule != NULL);
	  
	  /* Depending on the specifics of the currently iterated
	     fragrule, we may have or not to account for it. We test
	     this right now:
	  */
	  if (FALSE == 
	      pxmchem_fragment_account_fragrule (fragrule,
						 TRUE, /* check */
						 iter,
						 fragspec->end,
						 NULL,
						 polymer,
						 polchemdef->atomGPA))
	    continue;
	  
	  /* Each time a fragrule is found to be accounted for, we 
	     create a new oligomer, because each fragrule is considered
	     to generate a fragment that is different from the others.
	  */
	  oligomer_2 = pxmchem_oligomer_dup (oligomer_1,
					     PXM_OLIGOMER_DUP_DEEP_YES);
	  
	  /* With this brand new oligomer, we can make the chemical
	     work related to accounting the fragrule that is being
	     currently iterated.
	  */
	  
	  if (FALSE ==
	      pxmchem_fragment_account_fragrule (fragrule,
						 FALSE, /* check */
						 iter,
						 fragspec->end,
						 oligomer_2->masspair,
						 polymer,
						 polchemdef->atomGPA))
	    {
	      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		     _("%s@%d: failed accounting for the fragrule: '%s'\n"),
		     __FILE__, __LINE__, fragrule->name);
	      
	      pxmchem_oligomer_free (oligomer_1);
	      pxmchem_oligomer_free (oligomer_2);
	      
	      return -1;
	    }
	  
	  /* We can construct a name for this fragment:
	   */
	  oligomer_1->name = g_strdup_printf ("%s-%s-(%s)",
					      fragspec->name, 
					      monomer->code,
					      fragrule->name);

	  /* Finally finished working on the accounting of the
	     fragrule, which means that we can add this fragment to
	     the array of oligomers that was passed as parameter.
	  */
	  g_ptr_array_add (GPA, oligomer_2);

	  count++;
	}
    
      /* Note that we may be here for two reasons:
	 ----------------------------------------- 
	 
	 1. because the array of fragrules did not contain any fragrule,
	 in which case the oligomer_1 still needs to be validated.
	 2. because we finished dealing with fragrules, in which case
	 we do not add oligomer_1 to the array of fragments, because
	 if there were fragrules, then (even if none matched) the
	 oligomer_1 is not considered valid.
      */
      if (fragspec->fgrGPA->len <= 0)
	{
	  /* We have to make a complete oligomer with oligomer_1:
	   */
	  oligomer_1->name = g_strdup_printf ("%s-%s",
					      fragspec->name, 
					      monomer->code);
	  
	  g_ptr_array_add (GPA, oligomer_1);
	
	  count++;
	}
      else
	/* We do not need oligomer_1.
	 */
	pxmchem_oligomer_free (oligomer_1);
    }
  /* end of 
     for (iter = calcopt->start_idx; iter < calcopt->end_idx + 1; iter++)
  */

  return count;
}


gint
pxmchem_fragment_polymer_end_left (PxmPolymer *polymer,
				   GPtrArray *GPA,
				   PxmPolchemdef *polchemdef,			
				   PxmFragOpt *fragopt, 
				   PxmCalcOpt *calcopt)
{
  gint count = 0;
  gint numb = 0;

  gint iter = 0;
  gint jter = 0;
  gint kter = 0;

  gboolean fragrule_succeed = FALSE;

  GString *seq = NULL;

  PxmOligomer *oligomer_1 = NULL;
  PxmOligomer *oligomer_2 = NULL;
  
  PxmMonomer *monomer = NULL;

  PxmMasspair *masspair = NULL;
  PxmMasspair *masspair_tmp = NULL;
  
  PxmProp *prop = NULL;
  PxmProp *prop_new = NULL;
  
  PxmFragSpec *fragspec = NULL;
  PxmFragRule *fragrule = NULL;
  
  GPtrArray *modifpropGPA = NULL;
  

  /* "end_left" means that the fragments that we generate for the
     polymer sequence stretch considered contain the left end of the
     initial polymer (which actually may be an oligomer) sequence.
     See the example of the a,b,c ions in the proteinaceous world.
  */



  g_assert (polymer != NULL && polymer->monomerGPA != NULL);

  g_assert (polchemdef != NULL);
  
  g_assert (GPA != NULL);

  g_assert (fragopt != NULL);
  
  fragspec = fragopt->fragspec;
  g_assert (fragspec != NULL);
  
  if (polymer->monomerGPA->len <= 0)
    return 0;


  /* Allocate a masspair instance that will hold the masses of the
     oligomer during its elongation. There will be another masspair
     instance that will hold the masses when these masses change due
     to taking into account the mass calculation options passed as
     parameters.
  */
  masspair = libpolyxmass_masspair_new ();
  
  /* If the caller asks that the oligomer have their sequence in them,
     seed the string that will be elongating later as we iterate
     in the oligomer...
  */
  if (fragopt->put_sequence == TRUE)
    seq = g_string_new ("");
  
  /* Allocate the array that we will use to store all the
     modifications that we find in any of the monomer that enter in
     the composition of the oligomer being generated. Note that each
     oligomer will have a copy of the prop instances located in this
     array as long as the oligomer effectively comprises the monomers
     the modif of which is described in the prop instances.
  */
  modifpropGPA = g_ptr_array_new ();
  
  /* We get a calcopt->end_idx index that truly is the index of the last
     monomer to take into account during the fragmentation, which
     is why we want to include, thence the "iter < calcopt->end_idx + 1"
     below.
  */
  for (iter = calcopt->start_idx; iter < calcopt->end_idx + 1; iter++, 
	 numb++)
    {
      monomer = g_ptr_array_index (polymer->monomerGPA, iter);
      g_assert (monomer != NULL);
      
      if (fragopt->put_sequence == TRUE) /* note the _append_, since
					    we are iterating from left
					    to right.
					 */
	{
	  seq = g_string_append (seq, monomer->code);
	}
      
      /* Account for the mass of the iterated monomer. *******************
       */
      if (FALSE ==
	  pxmchem_monomer_account_mass_by_name (monomer->name,
						polchemdef->monomerGPA,
						polchemdef->atomGPA,
						1, /*times*/
						masspair))
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: failed accounting mass for monomer '%s'\n"),
		 __FILE__, __LINE__, monomer->name);
	  
	  libpolyxmass_masspair_free (masspair);

	  if (seq != NULL)
	    g_string_free (seq, TRUE);
	  
	  return -1;
	}
      
      /* Account for the mass of the iterated monomer's modifs ***********
       */
      if (calcopt->mnm_chement & PXMCHEMENT_MNM_MODIF)
	{
	  if (FALSE == 
	      pxmchem_monomer_account_mass_for_modif (monomer,
						      polchemdef->modifGPA,
						      polchemdef->atomGPA,
						      1, /*times*/
						      masspair))
	    {	  
	      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		     _("%s@%d: failed accounting for modif of "
		       "monomer named '%s'\n"),
		     __FILE__, __LINE__, monomer->name);
	      
	      pxmchem_oligomer_free (oligomer_1);
	      libpolyxmass_masspair_free (masspair);
	      
	      if (seq != NULL)
		g_string_free (seq, TRUE);
	      
	      return -1;
	    }
	}
      
      /* Also, if the user wanted that the modification eventually found
	 in the monomer be signalled, we can do this.
      */
      if (fragopt->mnm_chement != PXMCHEMENT_MNM_NONE)
	{
	  prop =  libpolyxmass_prop_find_prop (monomer->propGPA,
					   NULL,
					   NULL,
					   "MODIF",
					   NULL,
					   PXM_UNDEF_PROP_CMP);
	  
	  if (NULL != prop)
	    {
	      prop_new = libpolyxmass_prop_new ();

	      /* The nomenclature here is more complex than with the
		 polymer cleavage case, because the fact the monomer
		 number depends on the PxmEnd of the fragmentation
		 complicates much the work. So we simply construct a
		 prop name by telling what is the offset position of
		 the monomer that is modified with respect to the end
		 at which the fragmentation occurs, and also by
		 telling what the true index of the monomer is in the
		 whole polymer sequence. We need this datum so that
		 later we can access the monomer in question and thus
		 get to the name of the modification.... This is when
		 GUI programs need to display...
	      */
	      libpolyxmass_prop_set_name (prop_new, "POS/IDX/MODIF");
	      prop_new->data =g_strdup_printf ("%d/%d/%s",
					       /* pos in the oligomer */
					       iter - calcopt->start_idx + 1,
					       /* index in the sequence */
					       iter,
					       (gchar *) prop->data);

	      g_ptr_array_add (modifpropGPA, prop_new);
	    }
	}
  

      /* Make a "local" copy of the masspair instance that will be
	 stuck in the oligomer at the end of mass computing. This is
	 required, because masspair holds the masses of the elongating
	 fragment and should not be incremented with masses from 
	 evaluation of logical conditions and other fragrules...
      */
      masspair_tmp = libpolyxmass_masspair_dup (masspair);
      
      /* If the fragspec indicated that a chemical action should be
	 performed onto the monomer being fragmented, then do
	 it. Remember that some fragmentation patterns do not involve
	 a chemical modification of the monomer. This is why the
	 actform may be NULL.
      */
      if (fragspec->actform != NULL)
	{
	  if (FALSE == 
	      pxmchem_actform_account_mass (fragspec->actform,
					    polchemdef->atomGPA,
					    1 /*times*/, 
					    masspair_tmp))
	    {
	      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		     _("%s@%d: failed accounting for fragspec's "
		       "actform: '%s'\n"),
		     __FILE__, __LINE__, fragspec->actform);

	      libpolyxmass_masspair_free (masspair);
	      libpolyxmass_masspair_free (masspair_tmp);
	      
	      if (seq != NULL)
		g_string_free (seq, TRUE);
	      
	      return -1;
	    }
	}


      /* LEFT/RIGHT end chemistry.
	 ---------------------------
      */

      /* Because we are dealing with a fragmentation pattern that
	 starts at the LEFT END of the parent polymer sequence, we
	 must left-cap the fragment.
      */
      if (FALSE == 
	  pxmchem_actform_account_mass (polchemdef->leftcap,
					polchemdef->atomGPA,
					1 /*times*/, masspair_tmp))
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: failed accounting for left cap: '%s'\n"),
		 __FILE__, __LINE__, polchemdef->leftcap);
	 
	  libpolyxmass_masspair_free (masspair);
	  libpolyxmass_masspair_free (masspair_tmp);
	  
	  if (seq != NULL)
	    g_string_free (seq, TRUE);
	  
	  return -1;
	}
      
      /* Since we are constructing a fragment by iterating from left
	 to right, this fragment may encompasses the real left end of
	 the parent polymer sequence. This is why we check if the user
	 wanted to account for the left end modif and to give
	 indication of this. 

	 Note that we only account for the mass of the left end modif
	 if the fragment actually contains the left end of the polymer
	 (remember that the left end modif is the permanent
	 modification, which by definition lies at the LEFT END of the
	 polymer sequence). This is why we test that the start_idx be
	 0.

	 Also, since we are construction fragments that ALL have the
	 left end of the initial polymer sequence (encompassing or not
	 the true left end of the parent polymer sequence), we do not
	 need to check if iter == 0 because if (calcopt->start_idx ==
	 0), then all the fragments are going to have this left end...
      */
      if (calcopt->plm_chement & PXMCHEMENT_PLM_LEFT_MODIF
	  && calcopt->start_idx == 0)
	{
	  /* Take into account the left end modif.
	   */
	  if (FALSE == 
	      pxmchem_masscalc_polymer_account_left_end_modif (polymer,
							       masspair_tmp,
							       polchemdef))
	    {
	      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		     _("%s@%d: failed accounting for polymer's "
		       "left end modif.\n"),
		     __FILE__, __LINE__);
	      
	      libpolyxmass_masspair_free (masspair);
	      libpolyxmass_masspair_free (masspair_tmp);
	      
	      if (seq != NULL)
		g_string_free (seq, TRUE);
	      
	      return -1;
	    }
	}

      /* The same rationale applies to the RIGHT END and its chemical
	 permanent modification, which is why we test that end_idx ==
	 the size of the polymer less 1 (index and not position).

	 Note that while we always added the mass of the left end
	 modif, because EACH fragment is generated from the left end,
	 and thus by definition has the left end... the right end is
	 only contained by the fragment that actually encompasses the
	 whole polymer sequence since it starts from the left end and
	 arrives to the right end. This is why we ensure that the
	 modification is taken into account ONLY IF the
	 [calcopt->end_idx == polymer->monomerGPA->len - 1] AND IF
	 [iter == calcopt->end_idx].
      */
      if (calcopt->plm_chement & PXMCHEMENT_PLM_RIGHT_MODIF 
	  && calcopt->end_idx == polymer->monomerGPA->len - 1
	  && iter == calcopt->end_idx)
	{
	  /* Take into account the right end modif.
	   */
	  if (FALSE == 
	      pxmchem_masscalc_polymer_account_right_end_modif (polymer,
								masspair_tmp,
								polchemdef))
	    {
	      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		     _("%s@%d: failed accounting for polymer's "
		       "right end modif.\n"),
		     __FILE__, __LINE__);
	      
	      libpolyxmass_masspair_free (masspair);
	      libpolyxmass_masspair_free (masspair_tmp);
	      
	      if (seq != NULL)
		g_string_free (seq, TRUE);
	      
	      return -1;
	    }
	}
      

      /* Allocate an oligomer now.
       */
      oligomer_1 = pxmchem_oligomer_new ();

      oligomer_1->polymer = polymer;

      oligomer_1->start_idx = calcopt->start_idx;
      oligomer_1->end_idx = calcopt->end_idx;
            
      /* The fragment that we are constructing may encompass the whole
	 polymer sequence or only the left end (since it is a PxmEnd
	 LEFT). So we test if we are required to let the oligomer know
	 what were its ends...
      */
      if (fragopt-> plm_chement & PXMCHEMENT_PLM_LEFT_MODIF 
	  && calcopt->start_idx == 0)
	{
	  prop = libpolyxmass_prop_find_prop (polymer->propGPA,
					  NULL,
					  NULL,
					  "LEFT_END_MODIF",
					  NULL,
					  PXM_UNDEF_PROP_CMP);
	      
	  if (NULL != prop)
	    {
	      if (prop->data != NULL)
		{
		  prop_new = 
		    libpolyxmass_prop_dup (prop, PXM_MODIF_PROP_DUP_DEEP_YES);
		      
		  g_ptr_array_add (oligomer_1->propGPA, prop_new);
		}
	    }
	}

      if (fragopt-> plm_chement & PXMCHEMENT_PLM_RIGHT_MODIF 
	  && calcopt->end_idx == polymer->monomerGPA->len - 1
	  && iter == calcopt->end_idx)
	{
	  prop = libpolyxmass_prop_find_prop (polymer->propGPA,
					  NULL,
					  NULL,
					  "RIGHT_END_MODIF",
					  NULL,
					  PXM_UNDEF_PROP_CMP);
	      
	  if (NULL != prop)
	    {
	      if (prop->data != NULL)
		{
		  prop_new = 
		    libpolyxmass_prop_dup (prop, PXM_MODIF_PROP_DUP_DEEP_YES);
		      
		  g_ptr_array_add (oligomer_1->propGPA, prop_new);
		}
	    }
	}


      /* We now can put the masspair calculated so far in this
	 oligomer.
      */
      oligomer_1->masspair = masspair_tmp;
      
      /* If we want to store the sequence of the oligomer, time has
	 come to do it.
      */
      if (fragopt->put_sequence == TRUE)
	{
	  prop_new = libpolyxmass_prop_both_strings_new ("SEQUENCE", seq->str);
	  
	  g_ptr_array_add (oligomer_1->propGPA, prop_new);
	}
      
      /* At this point we can put in the oligomer->propGPA all the
	 prop instances that were prepared until this moment along the
	 fragmentation path with all the monomers that were found to
	 be modified. The current fragment will thus be
	 self-knowledgeable of all the monomers that it has inside
	 itself that are modified (see the cleavage process which is
	 simpler, but very similar).
      */
      for (kter = 0; kter < modifpropGPA->len; kter++)
	{
	  prop = g_ptr_array_index (modifpropGPA, kter);
	  g_assert (prop != NULL);
	  
	  prop_new = libpolyxmass_prop_dup (prop, PXM_MODIF_PROP_DUP_DEEP_YES);
	  
	  g_ptr_array_add (oligomer_1->propGPA, prop_new);
	}
      

      /* We now have to account for the fragrules...
       */
      fragrule_succeed = FALSE;
      
      for (jter = 0; jter < fragspec->fgrGPA->len; jter++)
	{
	  fragrule = g_ptr_array_index (fragspec->fgrGPA, jter);
	  g_assert (fragrule != NULL);
	  
	  /* Depending on the specifics of the currently iterated
	     fragrule, we may have or not to account for it. We test
	     this right now:
	  */
	  if (FALSE == 
	      pxmchem_fragment_account_fragrule (fragrule,
						 TRUE, /* check */
						 iter,
						 fragspec->end,
						 NULL,
						 polymer,
						 polchemdef->atomGPA))
	    continue;
	  
	  /* Each time a fragrule is found to be accounted for, we 
	     create a new oligomer, because each fragrule is considered
	     to generate a fragment that is different from the others.
	  */
	  oligomer_2 = pxmchem_oligomer_dup (oligomer_1,
					     PXM_OLIGOMER_DUP_DEEP_YES);
	  
	  /* With this brand new oligomer, we can make the chemical
	     work related to accounting the fragrule that is being
	     currently iterated.
	  */
	  if (FALSE ==
	      pxmchem_fragment_account_fragrule (fragrule,
						 FALSE, /* check */
						 iter,
						 fragspec->end,
						 oligomer_2->masspair,
						 polymer,
						 polchemdef->atomGPA))
	    {
	      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		     _("%s@%d: failed accounting for the fragrule: '%s'\n"),
		     __FILE__, __LINE__, fragrule->name);
	      
	      pxmchem_oligomer_free (oligomer_1);
	      pxmchem_oligomer_free (oligomer_2);
	      
	      return -1;
	    }
	  
	  /* We can construct a name for this fragment:
	   */
	  oligomer_2->name = g_strdup_printf ("%s-%d-(%s)",
					      fragspec->name, 
					      numb + 1,
					      fragrule->name);
	  
	  /* Finally finished working on the accounting of the
	     fragrule, which means that we can add this fragment to
	     the array of oligomers that was passed as parameter.
	  */
	  g_ptr_array_add (GPA, oligomer_2);

	  fragrule_succeed = TRUE;
	  
	  count++;
	}
    
      /* Note that we may be here for two reasons:
	 ----------------------------------------- 
	 
	 1. because the array of fragrules did not contain any
	 fragrule, in which case the oligomer_1 still needs to be
	 validated.  

	 2. because we finished dealing with fragrules, in
	 which case we add oligomer_1 to the array of fragments, only
	 if no one of the fragrules analyzed gave a successfully
	 generated fragment.
      */
      if (fragspec->fgrGPA->len <= 0)
	{
	  /* We have to make a complete oligomer with oligomer_1: note
	     that this time we do not use any fragrule whatsoever to
	     construct the name of the oligomer because there are no
	     fragrules!
	   */
	  oligomer_1->name = g_strdup_printf ("%s-%d",
					      fragspec->name, 
					      numb + 1);
	  g_ptr_array_add (GPA, oligomer_1);
	  
	  count++;
	}
      else
	/* We do not need oligomer_1 only if at least one fragrule
	   succeeded in generating a valid fragment.
	 */
	{
	  if (fragrule_succeed == TRUE)
	    pxmchem_oligomer_free (oligomer_1);
	  else
	    {
	      oligomer_1->name = g_strdup_printf ("%s-%d",
						  fragspec->name, 
						  numb + 1);
	      g_ptr_array_add (GPA, oligomer_1);
	      
	      count++;
	    }
	}
    }
  /* end of 
     for (iter = calcopt->start_idx; iter < calcopt->end_idx + 1; iter++)
  */

  /* At this point, we have to free the array of modifpropGPA:
   */
  libpolyxmass_prop_GPA_free (modifpropGPA);
  
  /* And the masspair, which is of no use anymore.
   */
  libpolyxmass_masspair_free (masspair);

  if (fragopt->put_sequence == TRUE)
    g_string_free (seq, TRUE);

  return count;
}


gint
pxmchem_fragment_polymer_end_right (PxmPolymer *polymer,
				    GPtrArray *GPA,
				    PxmPolchemdef *polchemdef,			
				    PxmFragOpt *fragopt, 
				    PxmCalcOpt *calcopt)
{
  gint count = 0;
  gint numb = 0;

  gint iter = 0;
  gint jter = 0;
  gint kter = 0;

  gboolean fragrule_succeed = FALSE;


  GString *seq = NULL;

  PxmOligomer *oligomer_1 = NULL;
  PxmOligomer *oligomer_2 = NULL;
  
  PxmMonomer *monomer = NULL;
  
  PxmMasspair *masspair = NULL;
  PxmMasspair *masspair_tmp = NULL;
  
  PxmProp *prop = NULL;
  PxmProp *prop_new = NULL;
  
  PxmFragSpec *fragspec = NULL;
  PxmFragRule *fragrule = NULL;
  
  GPtrArray *modifpropGPA = NULL;


  /* "end_right" means that the fragments that we generate for the
     polymer sequence stretch considered contain the right end of the
     initial polymer (which actually may be an oligomer) sequence.
     See the example of the x,y,z ions in the proteinaceous world.
  */



  g_assert (polymer != NULL && polymer->monomerGPA != NULL);

  g_assert (polchemdef != NULL);
  
  g_assert (GPA != NULL);

  g_assert (fragopt != NULL);
  
  fragspec = fragopt->fragspec;
  g_assert (fragspec != NULL);
  
  if (polymer->monomerGPA->len <= 0)
    return 0;


  /* Allocate a masspair instance that will hold the masses of the
     oligomer during its elongation. There will be another masspair
     instance that will hold the masses when these masses change due
     to taking into account the mass calculation options passed as
     parameters.
  */
  masspair = libpolyxmass_masspair_new ();
  
  /* If the caller asks that the oligomer have their sequence in them,
     seed the string that will be elongating later as we iterate
     in the oligomer...
  */
  if (fragopt->put_sequence == TRUE)
    seq = g_string_new ("");
  
  /* Allocate the array that we will use to store all the
     modifications that we find in any of the monomer that enter in
     the composition of the oligomer being generated. Note that each
     oligomer will have a copy of the prop instances located in this
     array as long as the oligomer effectively comprises the monomers
     the modif of which is described in the prop instances.
  */
  modifpropGPA = g_ptr_array_new ();

  /* Note that we get a calcopt->start_idx index that truly
     corresponds to the index of the last monomer that we want to
     integrate in the fragmentation process, which means that we want
     to include it in the for loop below by issuing "iter >
     calcopt->start_idx - 1".
  */
  for (iter = calcopt->end_idx; iter > calcopt->start_idx - 1; iter--, 
	 numb++)
    {
      monomer = g_ptr_array_index (polymer->monomerGPA, iter);
      g_assert (monomer != NULL);
      
      if (fragopt->put_sequence == TRUE) /* note the _prepend_, since
					    we are iterating from left
					    to right.
					 */
	{
	  seq = g_string_prepend (seq, monomer->code);
	}
      
      /* Account for the mass of the iterated monomer. *******************
       */
      if (FALSE ==
	  pxmchem_monomer_account_mass_by_name (monomer->name,
						polchemdef->monomerGPA,
						polchemdef->atomGPA,
						1, /*times*/
						masspair))
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: failed accounting mass for monomer '%s'\n"),
		 __FILE__, __LINE__, monomer->name);
	  
	  libpolyxmass_masspair_free (masspair);

	  if (seq != NULL)
	    g_string_free (seq, TRUE);
	  
	  return -1;
	}
      
      /* Account for the mass of the iterated monomer's modifs ***********
       */
      if (calcopt->mnm_chement & PXMCHEMENT_MNM_MODIF)
	{
	  if (FALSE == 
	      pxmchem_monomer_account_mass_for_modif (monomer,
						      polchemdef->modifGPA,
						      polchemdef->atomGPA,
						      1, /*times*/
						      masspair))
	    {	  
	      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		     _("%s@%d: failed accounting for modif of "
		       "monomer named '%s'\n"),
		     __FILE__, __LINE__, monomer->name);
	      
	      pxmchem_oligomer_free (oligomer_1);
	      libpolyxmass_masspair_free (masspair);
	      
	      if (seq != NULL)
		g_string_free (seq, TRUE);
	      
	      return -1;
	    }
	}
      
      /* Also, if the user wanted that the modification eventually found
	 in the monomer be signalled, we can do this.
      */
      if (fragopt->mnm_chement != PXMCHEMENT_MNM_NONE)
	{
	  prop =  libpolyxmass_prop_find_prop (monomer->propGPA,
					   NULL,
					   NULL,
					   "MODIF",
					   NULL,
					   PXM_UNDEF_PROP_CMP);
	  
	  if (NULL != prop)
	    {
	      prop_new = libpolyxmass_prop_new ();

	      /* The nomenclature here is more complex than with the
		 polymer cleavage case, because the fact the monomer
		 number depends on the PxmEnd of the fragmentation
		 complicates much the work. So we simply construct a
		 prop name by telling what is the offset position of
		 the monomer that is modified with respect to the end
		 at which the fragmentation occurs, and also by
		 telling what the true index of the monomer is in the
		 whole polymer sequence. We need this datum so that
		 later we can access the monomer in question and thus
		 get to the name of the modification.... This is when
		 GUI programs need to display...
	      */
	      libpolyxmass_prop_set_name (prop_new, "POS/IDX/MODIF");
	      prop_new->data =g_strdup_printf ("%d/%d/%s",
					       /* pos in the oligomer */
					       calcopt->end_idx - iter + 1,
					       /* index in the sequence */
					       iter,
					       (gchar *) prop->data);
	      
	      g_ptr_array_add (modifpropGPA, prop_new);
	    }
	}
  

      /* Make a "local" copy of the masspair instance that will be
	 stuck in the oligomer at the end of mass computing. This is
	 required, because masspair holds the masses of the elongating
	 fragment and should not be incremented with masses from 
	 evaluation of fragrules...
      */
      masspair_tmp = libpolyxmass_masspair_dup (masspair);
      
      /* If the fragspec indicated that a chemical action should be
	 performed onto the monomer being fragmented, then do
	 it. Remember that some fragmentation patterns do not involve
	 a chemical modification of the monomer. This is why the
	 actform may be NULL.
      */
      if (fragspec->actform != NULL)
	{
	  if (FALSE == 
	      pxmchem_actform_account_mass (fragspec->actform,
					    polchemdef->atomGPA,
					    1 /*times*/, masspair_tmp))
	    {
	      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		     _("%s@%d: failed accounting for fragspec's "
		       "actform: '%s'\n"),
		     __FILE__, __LINE__, fragspec->actform);

	      libpolyxmass_masspair_free (masspair);
	      libpolyxmass_masspair_free (masspair_tmp);
	      
	      if (seq != NULL)
		g_string_free (seq, TRUE);
	      
	      return -1;
	    }
	}


      /* LEFT/RIGHT end chemistry.
	 ---------------------------
      */

      /* Because we are dealing with a fragmentation pattern that
	 starts at the RIGHT END of the parent polymer sequence, we
	 must right-cap the fragment.
      */
      if (FALSE == 
	  pxmchem_actform_account_mass (polchemdef->rightcap,
					polchemdef->atomGPA,
					1 /*times*/, masspair_tmp))
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: failed accounting for right cap: '%s'\n"),
		 __FILE__, __LINE__, polchemdef->rightcap);
	 
	  libpolyxmass_masspair_free (masspair);
	  libpolyxmass_masspair_free (masspair_tmp);
	  
	  if (seq != NULL)
	    g_string_free (seq, TRUE);
	  
	  return -1;
	}
      
      /* Since we are constructing a fragment by iterating from right
	 to left, this fragment may encompass the real right end of the
	 parent polymer sequence. This is why we check if the user
	 wanted to account for the right end modif and to give
	 indication of this.

	 Note that we only account for the mass of the right end modif
	 if the fragment actually contains the right end of the
	 polymer (remember that the right end modif is the permanent
	 modification, which by definition lies at the RIGHT END of
	 the polymer sequence). This is why we test that the end_idx
	 be polymer->monomerGPA->len - 1 (index, not position).

	 Also, since we are construction fragments that ALL have the
	 right end of the initial polymer sequence (encompassing or
	 not the true right end of the parent polymer sequence), we do
	 not need to check if iter == [polymer->monomerGPA->len - 1]
	 because if (calcopt->end_idx == polymer->monomerGPA->len -
	 1), then all the fragments are going to have this right
	 end...
      */
      if (calcopt->plm_chement & PXMCHEMENT_PLM_RIGHT_MODIF 
	  && calcopt->end_idx == polymer->monomerGPA->len - 1)
	{
	  /* Take into account the right end modif.
	   */
	  if (FALSE == 
	      pxmchem_masscalc_polymer_account_right_end_modif (polymer,
								masspair_tmp,
								polchemdef))
	    {
	      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		     _("%s@%d: failed accounting for polymer's "
		       "right end modif.\n"),
		     __FILE__, __LINE__);
	      
	      libpolyxmass_masspair_free (masspair);
	      libpolyxmass_masspair_free (masspair_tmp);
	      
	      if (seq != NULL)
		g_string_free (seq, TRUE);
	      
	      return -1;
	    }
	}
      
      /* The same rationale applies to the LEFT END and its chemical
	 permanent modification, which is why we test that start_idx
	 == 0.

	 Note that while we always added the mass of the right end
	 modif, because EACH fragment is generated from the right end,
	 and thus by definition has the right end... the left end is
	 only contained by the fragment that actually encompasses the
	 whole polymer sequence since it starts from the right end and
	 arrives to the left end. This is why we ensure that the
	 modification is taken into account ONLY IF the
	 [calcopt->start_idx == 0] AND IF [iter == calcopt->start_idx].
      */
      if (calcopt->plm_chement & PXMCHEMENT_PLM_LEFT_MODIF
	  && calcopt->start_idx == 0
	  && iter == calcopt->start_idx)
	{
	  /* Take into account the left end modif.
	   */
	  if (FALSE == 
	      pxmchem_masscalc_polymer_account_left_end_modif (polymer,
							       masspair_tmp,
							       polchemdef))
	    {
	      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		     _("%s@%d: failed accounting for polymer's "
		       "left end modif.\n"),
		     __FILE__, __LINE__);
	      
	      libpolyxmass_masspair_free (masspair);
	      libpolyxmass_masspair_free (masspair_tmp);
	      
	      if (seq != NULL)
		g_string_free (seq, TRUE);
	      
	      return -1;
	    }
	}


      /* Allocate an oligomer now.
       */
      oligomer_1 = pxmchem_oligomer_new ();
      
      oligomer_1->polymer = polymer;
      
      oligomer_1->start_idx = calcopt->start_idx;
      oligomer_1->end_idx = calcopt->end_idx;

      /* The fragment that we are construction may encompass the whole
	 polymer sequence or only the right end (since it is a PxmEnd
	 RIGHT). So we test if we are required to let the oligomer
	 know what were its ends...
      */
      if (fragopt-> plm_chement & PXMCHEMENT_PLM_RIGHT_MODIF 
	  && calcopt->end_idx == polymer->monomerGPA->len - 1)
	{
	  prop = libpolyxmass_prop_find_prop (polymer->propGPA,
					  NULL,
					  NULL,
					  "RIGHT_END_MODIF",
					  NULL,
					  PXM_UNDEF_PROP_CMP);
	      
	  if (NULL != prop)
	    {
	      if (prop->data != NULL)
		{
		  prop_new = 
		    libpolyxmass_prop_dup (prop, PXM_MODIF_PROP_DUP_DEEP_YES);
		      
		  g_ptr_array_add (oligomer_1->propGPA, prop_new);
		}
	    }
	}

      if (fragopt-> plm_chement & PXMCHEMENT_PLM_LEFT_MODIF 
	  && calcopt->start_idx == 0
	  && iter == calcopt->start_idx)
	{
	  prop = libpolyxmass_prop_find_prop (polymer->propGPA,
					  NULL,
					  NULL,
					  "LEFT_END_MODIF",
					  NULL,
					  PXM_UNDEF_PROP_CMP);
	      
	  if (NULL != prop)
	    {
	      if (prop->data != NULL)
		{
		  prop_new = 
		    libpolyxmass_prop_dup (prop, PXM_MODIF_PROP_DUP_DEEP_YES);
		      
		  g_ptr_array_add (oligomer_1->propGPA, prop_new);
		}
	    }
	}


      /* We now can put the masspair calculated so far in this
	 oligomer.
      */
      oligomer_1->masspair = masspair_tmp;
      
      /* If we want to store the sequence of the oligomer, time has
	 come to do it.
      */
      if (fragopt->put_sequence == TRUE)
	{
	  prop_new = 
	    libpolyxmass_prop_both_strings_new ("SEQUENCE", seq->str);
	  
	  g_ptr_array_add (oligomer_1->propGPA, prop_new);
	}
      
      /* At this point we can put in the oligomer->propGPA all the
	 prop instances that were prepared until this moment along the
	 fragmentation path with all the monomers that were found to
	 be modified. The current fragment will thus be
	 self-knowledgeable of all the monomers that it has inside
	 itself that are modified (see the cleavage process which is
	 simpler, but very similar).
      */
      for (kter = 0; kter < modifpropGPA->len; kter++)
	{
	  prop = g_ptr_array_index (modifpropGPA, kter);
	  g_assert (prop != NULL);
	  
	  prop_new = 
	    libpolyxmass_prop_dup (prop, PXM_MODIF_PROP_DUP_DEEP_YES);
	  
	  g_ptr_array_add (oligomer_1->propGPA, prop_new);
	}
      
      /* We now have to account for the fragrules...
       */
      fragrule_succeed = FALSE;
      
      for (jter = 0; jter < fragspec->fgrGPA->len; jter++)
	{
	  fragrule = g_ptr_array_index (fragspec->fgrGPA, jter);
	  g_assert (fragrule != NULL);
	  
	  /* Depending on the specifics of the currently iterated
	     fragrule, we may have or not to account for it. We test
	     this right now:
	  */
	  if (FALSE == 
	      pxmchem_fragment_account_fragrule (fragrule,
						 TRUE, /* check */
						 iter,
						 fragspec->end,
						 NULL,
						 polymer,
						 polchemdef->atomGPA))
	    continue;
	  
	  /* Each time a fragrule is found to be accounted for, we 
	     create a new oligomer, because each fragrule is considered
	     to generate a fragment that is different from the others.
	  */
	  oligomer_2 = pxmchem_oligomer_dup (oligomer_1,
					     PXM_OLIGOMER_DUP_DEEP_YES);
	  
	  /* With this brand new oligomer, we can make the chemical
	     work related to accounting the fragrule that is being
	     currently iterated.
	  */
	  if (FALSE ==
	      pxmchem_fragment_account_fragrule (fragrule,
						 FALSE, /* check */
						 iter,
						 fragspec->end,
						 oligomer_2->masspair,
						 polymer,
						 polchemdef->atomGPA))
	    {
	      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		     _("%s@%d: failed accounting for the fragrule: '%s'\n"),
		     __FILE__, __LINE__, fragrule->name);
	      
	      pxmchem_oligomer_free (oligomer_1);
	      pxmchem_oligomer_free (oligomer_2);
	      
	      return -1;
	    }
	  
	  /* We can construct a name for this fragment:
	   */
	  oligomer_2->name = g_strdup_printf ("%s-%d-(%s)",
					      fragspec->name, 
					      numb + 1,
					      fragrule->name);
	  
	  /* Finally finished working on the accounting of the
	     fragrule, which means that we can add this fragment to
	     the array of oligomers that was passed as parameter.
	  */
	  g_ptr_array_add (GPA, oligomer_2);

	  fragrule_succeed = TRUE;
	  
	  count++;
	}
    
      /* Note that we may be here for two reasons:
	 ----------------------------------------- 
	 
	 1. because the array of fragrules did not contain any
	 fragrule, in which case the oligomer_1 still needs to be
	 validated.  

	 2. because we finished dealing with fragrules, in
	 which case we add oligomer_1 to the array of fragments, only
	 if no one of the fragrules analyzed gave a successfully
	 generated fragment.
      */
      if (fragspec->fgrGPA->len <= 0)
	{
	  /* We have to make a complete oligomer with oligomer_1: note
	     that this time we do not use any fragrule whatsoever to
	     construct the name of the oligomer because there are no
	     fragrules!
	   */
	  oligomer_1->name = g_strdup_printf ("%s-%d",
					      fragspec->name, 
					      numb + 1);
	  g_ptr_array_add (GPA, oligomer_1);
	  
	  count++;
	}
      else
	/* We do not need oligomer_1 only if at least one fragrule
	   succeeded in generating a valid fragment.
	 */
	{
	  if (fragrule_succeed == TRUE)
	    pxmchem_oligomer_free (oligomer_1);
	  else
	    {
	      oligomer_1->name = g_strdup_printf ("%s-%d",
						  fragspec->name, 
						  numb + 1);
	      g_ptr_array_add (GPA, oligomer_1);
	      
	      count++;
	    }
	}
    }
  /* end of 
     for (iter = calcopt->start_idx; iter < calcopt->end_idx + 1; iter++)
  */

  /* At this point, we have to free the array of modifpropGPA:
   */
  libpolyxmass_prop_GPA_free (modifpropGPA);
  
  /* And the masspair, which is of no use anymore.
   */
  libpolyxmass_masspair_free (masspair);

  if (fragopt->put_sequence == TRUE)
    g_string_free (seq, TRUE);
  
  return count;
}


gboolean
pxmchem_fragment_account_fragrule (PxmFragRule *fragrule,
				   gboolean check,
				   gint idx,
				   PxmEnd end,
				   PxmMasspair *masspair,
				   PxmPolymer *polymer,
				   GPtrArray *atom_refGPA)
{
  PxmMonomer *prev = NULL;
  PxmMonomer *this = NULL;
  PxmMonomer *next = NULL;

  gint len = 0;
  

  g_assert (fragrule != NULL);
  g_assert (fragrule->actform != NULL);
  
  g_assert (polymer != NULL);
  g_assert (atom_refGPA != NULL);
  
  len = polymer->monomerGPA->len;
  g_assert (idx < len);
  g_assert (idx >= 0);
  
  /* If function is called to actually account for masses, masspair
     cannot be NULL.
  */
  if (check == FALSE)
    g_assert (masspair != NULL);
  

  /* A fragrule specifies an actform (actionformula) that must be
     applied to a fragment as soon as it has been verified that the
     conditional elements in its prev, this, next member(s) are
     verified. Note that any of these members that may be NULL are
     considered to be a verified condition.
  */

  /* If the check parameter is TRUE, then we do not make the actual
     mass accounting, but only return TRUE if the conditions are met,
     and FALSE if not. If 'check' is FALSE, then this function is
     called to perform the actual mass accounting. In this case the
     function return FALSE only if the accounting has failed, and not
     if the conditions are not met (which is not in this case
     considered a failure).
  */

  /* We can immediately check the 'this' condition, because it does
     not depend on the PxmEnd 'end' parameter (of course prev and next
     depend on the PxmEnd 'end' parameter, as 'prev' corresponds to
     'this'-1 if the fragment is elongating from left end, and
     'this'+1 if the fragment is elongating from right end. But we'll see
     that later.
  */
  this = g_ptr_array_index (polymer->monomerGPA, idx);
  
  if (fragrule->this != NULL)
    {
      if (0 != strcmp (fragrule->this, this->code))
	return FALSE;
    }
  
  /* Now check the two other optional conditions if this is possible.
     Attention with the "next"/"prev" orientation, depending on the
     kind of oligomer we are generating ('end' param).
  */
  
  if (fragrule->prev != NULL && fragrule->next != NULL)
    {
      if (end & PXM_FRAG_END_LEFT || end & PXM_FRAG_END_NONE)
	{
	  if (idx == 0)
	    /* There cannot be any "previous code" since "this code"
	       is the first of the fragmentation series.
	     */	
	    return FALSE;
	  
	  /* Since we know that 'end' is either PXM_FRAG_END_LEFT or
	     PXM_FRAG_END_NONE, we know what monomer is the 'prev':
	  */
	  prev = g_ptr_array_index (polymer->monomerGPA, idx - 1);

	  if (idx == len - 1)
	    /* There cannot be any "next code" since "this code" is
	       the last of the fragmentation series.
	     */
	    return FALSE;

	  next = g_ptr_array_index (polymer->monomerGPA, idx + 1);
	}
      
      else if (end & PXM_FRAG_END_RIGHT)
	{
	  if (idx == 0)
	    /* There cannot be any "next code" since "this code" is
	       the last of the fragmentation series.
	     */	  
	    return FALSE;

	  next = g_ptr_array_index (polymer->monomerGPA, idx - 1);

	  if (idx == len - 1)
	    /* There cannot be any "previous code" since "this code"
	       is the first of the fragmentation series.
	     */
	    return FALSE;

	  prev = g_ptr_array_index (polymer->monomerGPA, idx+ 1);
	}
      
      else
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: bad 'end' member found in fragrule '%s'\n"),
		 __FILE__, __LINE__, fragrule->name);

	  return FALSE;
	}
      
      /* Now that we have correctly identified the 'next' and 'prev' monomers
	 in the sequence, we can make the comparisons to ensure if the
	 conditions are met.
      */
      if (0 == strcmp (fragrule->prev, prev->code) 
	  && 0 == strcmp (fragrule->next, next->code))
	{
	  if (check == TRUE)
	    return TRUE;
	  
	  /* Apparently the conditions are such as we can do the
	   * chemical calculation.
	   */
	  if (FALSE ==
	      pxmchem_actform_account_mass (fragrule->actform,
					    atom_refGPA,
					    1 /*times*/, masspair))
	    {
	      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		     _("%s@%d: failed accounting fragrule's "
		       "actform: '%s'\n"),
		     __FILE__, __LINE__, fragrule->actform);
	      
	      return FALSE;
	    }
	  else
	    return TRUE;
	}
      
      else
	{
	  if (check == TRUE)
	    return FALSE;
	  else
	    return TRUE;
	}
    }
  /* end of 
     if (fragrule->prev != NULL && fragrule->next != NULL)
  */

  else if (fragrule->prev != NULL)
    {
      if (end & PXM_FRAG_END_LEFT || end & PXM_FRAG_END_NONE)
	{
	  if (idx == 0)
	    /* There cannot be any "previous code" since "this code"
	       is the first of the fragmentation series.
	     */	
	    return FALSE;
	  
	  /* Since we know that 'end' is either PXM_FRAG_END_LEFT or
	     PXM_FRAG_END_NONE, we know what monomer is the 'prev':
	  */
	  prev = g_ptr_array_index (polymer->monomerGPA, idx - 1);
	}
      
      else if (end & PXM_FRAG_END_RIGHT)
	{
	  if (idx == len - 1)
	    /* There cannot be any "previous code" since "this code"
	       is the first of the fragmentation series.
	     */
	    return FALSE;

	  prev = g_ptr_array_index (polymer->monomerGPA, idx+ 1);
	}
      
      else
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: bad 'end' member found in fragrule '%s'\n"),
		 __FILE__, __LINE__, fragrule->name);

	  return FALSE;
	}
      
      /* Now that we have correctly identified the 'next' and 'prev' monomers
	 in the sequence, we can make the comparisons to ensure if the
	 conditions are met.
      */
      if (0 == strcmp (fragrule->prev, prev->code))
	{
	  if (check == TRUE)
	    return TRUE;
	  
	  /* Apparently the conditions are such as we can do the
	   * chemical calculation.
	   */
	  if (FALSE ==
	      pxmchem_actform_account_mass (fragrule->actform,
					    atom_refGPA,
					    1 /*times*/, masspair))
	    {
	      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		     _("%s@%d: failed accounting fragrule's "
		       "actform: '%s'\n"),
		     __FILE__, __LINE__, fragrule->actform);
	      
	      return FALSE;
	    }
	  else
	    return TRUE;
	}
      
      else
	{
	  if (check == TRUE)
	    return FALSE;
	  else
	    return TRUE;
	}
    }
  /* end of 
     else if (fragrule->prev != NULL)
  */

  else if (fragrule->next != NULL)
    {
      if (end & PXM_FRAG_END_LEFT || end & PXM_FRAG_END_NONE)
	{
	  if (idx == len - 1)
	    /* There cannot be any "next code" since "this code" is
	       the last of the fragmentation series.
	    */
	    return FALSE;
	  
	  next = g_ptr_array_index (polymer->monomerGPA, idx + 1);
	}
      
      else if (end & PXM_FRAG_END_RIGHT)
	{
	  if (idx == 0)
	    /* There cannot be any "next code" since "this code" is
	       the last of the fragmentation series.
	     */	  
	    return FALSE;

	  next = g_ptr_array_index (polymer->monomerGPA, idx - 1);
	}
      
      else
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: bad 'end' member found in fragrule '%s'\n"),
		 __FILE__, __LINE__, fragrule->name);

	  return FALSE;
	}
      
      /* Now that we have correctly identified the 'next' and 'prev' monomers
	 in the sequence, we can make the comparisons to ensure if the
	 conditions are met.
      */
      if (0 == strcmp (fragrule->next, next->code))
	{
	  if (check == TRUE)
	    return TRUE;
	  
	  /* Apparently the conditions are such as we can do the
	   * chemical calculation.
	   */
	  if (FALSE ==
	      pxmchem_actform_account_mass (fragrule->actform,
					    atom_refGPA,
					    1 /*times*/, masspair))
	    {
	      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		     _("%s@%d: failed accounting fragrule's "
		       "actform: '%s'\n"),
		     __FILE__, __LINE__, fragrule->actform);
	      
	      return FALSE;
	    }
	  else
	    return TRUE;
	}
      
      else
	{
	  if (check == TRUE)
	    return FALSE;
	  else
	    return TRUE;
	}
    }
  /* end of 
     else if (fragrule->next != NULL)
  */
  
  else
    /* All the 'prev' 'next' are NULL, which means that we consider
       the conditions verified.
    */
    {
      if (check == TRUE)
	return TRUE;

      /* Apparently the conditions are such as we can do the
       * chemical calculation.
       */
      if (FALSE ==
	  pxmchem_actform_account_mass (fragrule->actform,
					atom_refGPA,
					1 /*times*/, masspair))
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: failed accounting fragrule's "
		   "actform: '%s'\n"),
		 __FILE__, __LINE__, fragrule->actform);
	  
	  return FALSE;
	}
      else
	return TRUE;
    }
  
  /* We should never reach this point!
   */
  g_assert (1 == 0);
  
  return FALSE;
}

