/***************************************************************************
                          hbcimessagequeue.cpp  -  description
                             -------------------
    begin                : Thu Jul 26 2001
    copyright            : (C) 2001 by fabian kaiser
    email                : fabian.kaiser@gmx.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/
/*
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h> // debug
//#include <iostream.h>

#include "messagequeue.h"
#include "hbcistring.h"
#include "adminsegs.h"
#include "mediumrdhbase.h"


namespace HBCI {

GeneralSegResponse::GeneralSegResponse(int RefSEG,
                                       int RefDE,
                                       int ReturnCode,
                                       string Message):
    refSEG(RefSEG),
    refDE(RefDE),
    returnCode(ReturnCode),
    message(Message)
{
}


GeneralSegResponse::GeneralSegResponse(string &segment) {
    unsigned int pos = 0;
    unsigned int spos = 0;
    string tmp;

    refDE = 0;
    refSEG = 0;
    returnCode = 0;

    // overall return-message
    if (String::nextDEG(segment, pos) == "HIRMG") {
        // skip seg head
        pos+=String::nextDE(segment, pos).length() + 1;
        // read return code
        tmp=String::nextDEG(segment, pos);
        returnCode=atoi(tmp.c_str());
        pos += tmp.length() + 1;
        // reference-de must not be set so inc pos by 1
        pos++;
        // read return-message
        message = String::unEscape(String::nextDEG(segment, pos));
    } else if (String::nextDEG(segment, pos) == "HIRMS") {
        // return message for a segment that does not belong to any job
        // skip segid, segversion and segnumber
        string segHead = String::nextDE(segment, pos);
        pos = segHead.length() + 1;
        spos = 0;
        spos += String::nextDEG(segHead, spos).length() + 1;
        spos += String::nextDEG(segHead, spos).length() + 1;
        spos += String::nextDEG(segHead, spos).length() + 1;

        // get the ref-seg
        tmp = String::nextDEG(segment, spos);
        refSEG=atoi(tmp.c_str());

        // get the response code
        tmp = String::nextDEG(segment, pos);
        pos += tmp.length() + 1;
        returnCode=atoi(tmp.c_str());

        // get the ref-de
        tmp = String::nextDEG(segment, pos);
        pos += tmp.length() + 1;
        refDE = atoi(tmp.c_str());

        // get the textmessage
        message = String::nextDE(segment, pos);
    } else
        throw Error("GeneralResponse::GeneralResponse(segment)",
                        "Parse-error",0);
}





MessageQueue::MessageQueue(Pointer<Customer> cust)
:_syncmode(false)
{
    _customer=cust;
    _msgnumber=0;
    _dialogid="0";
    _customer.setDescription("MessageQueue::_customer");
    _medium.setDescription("MessageQueue::_medium (Medium)");
    reset();
}


MessageQueue::~MessageQueue(){
    reset();
    if (_medium.isValid())
        _medium.ref().unmountMedium();
}


void MessageQueue::reset(){
    _jobqueue.clear();
    _signers.clear();
    _response.erase();
    //_bankmessages.clear();
    _msgnumber++;
    _result=-1;
}


bool MessageQueue::addJob(Pointer<Job> newJob){
    _jobqueue.push_back(newJob);
    return true;
}


bool MessageQueue::addSigner(Pointer<Customer> newSigner){
    list<Pointer<Customer> >::iterator it;

    for (it=_signers.begin();
	 it!=_signers.end();
	 it++) {
	if ((*it)==newSigner) {
	    if (Hbci::debugLevel()>0) {
		fprintf(stderr,
			"Signer already exists, not added");
		return false;
	    }
	}
    } // for
    _signers.push_back(newSigner);
    return true;
}


Error MessageQueue::_mountCustomersMedium(Pointer<Customer> cust) {
  Pointer<Bank> bank;
  Error err;

  bank.setDescription("MessageQueue::_mountCustomersMedium::bank");
  bank=cust.ref().user().ref().bank();
  if (_medium!=cust.ref().user().ref().medium()) {
    if (Hbci::debugLevel()>4)
      fprintf(stderr,"MessageQueue: Mounting new medium.\n");

    // unmount previous medium, if any
    if (_medium.isValid())
      _medium.ref().unmountMedium();
    _medium=cust.ref().user().ref().medium();
    err=_medium.ref().mountMedium();
    if (!err.isOk()){
      _medium=0;
      return Error("MessageQueue::_mountCustomersMedium()",err);
    }
  }

  // select context on the medium
  //_customer->hbci->msgStateResponse("MessageQueue: Selecting context.");
  err=_medium.ref().selectContext(bank.ref().countryCode(),
				  bank.ref().bankCode(),
				  cust.ref().user().ref().userId());
  if (!err.isOk()) {
    _medium.ref().unmountMedium();
    _medium=0;
    if (Hbci::debugLevel()>0)
      fprintf(stderr,
	      "Could not select this context:\n"
	      " Bank: %d/%s User:%s\n",
	      bank.ref().countryCode(),
	      bank.ref().bankCode().c_str(),
	      cust.ref().user().ref().userId().c_str());
    return Error("MessageQueue::_mountCustomersMedium", err);
  }
  return Error();
}


string MessageQueue::toString(int message_number){
  string pureMessage;
    string result;
    string seg;
    int firstseg;
    int currseg;
    list<Pointer<Job> >::const_iterator job;
    list<Pointer<Customer> >::const_iterator signer;
    bool needsign;
    bool needcrypt;
    Error herr;

    // check if there is anything to do
    if (_jobqueue.empty())
        throw Error("MessageQueue::toString()",
                        "empty jobqueue.",0);

    // check if we need to encrypt and to sign the message
    needsign=false;
    needcrypt=false;
    for (job=_jobqueue.begin(); job!=_jobqueue.end(); job++) {
        needsign|=(*job).ref().needsToBeSigned();
        needcrypt|=(*job).ref().needsToBeEncrypted();
    }

    // add signer, if needed
    if (needsign)
        if (!addSigner(_customer))
            throw Error("MessageQueue::toString()",
			"could not add signer.",0);

    // ok, calculate the first segment number
    firstseg=_signers.size()+2;

    // now get the jobs into a string
    currseg=firstseg;
    for (job=_jobqueue.begin(); job!=_jobqueue.end(); job++) {
      // mount and select corresponding medium and context
      herr=_mountCustomersMedium((*job).ref().owner());
      if (!herr.isOk())
	throw herr;
      pureMessage+=(*job).ref().toString(currseg);
      currseg+=(*job).ref().segments();
    }
    if (Hbci::debugLevel()>5)
      fprintf(stderr,
	      "PURE MESSAGE:\n%s\n",
	      String::dumpToString(pureMessage).c_str());
    result=pureMessage;

    // now let all signers sign this message
    if (needsign) {
        //_customer->hbci->msgStateResponse("MessageQueue: Signature needed.");
        for (signer=_signers.begin(); signer!=_signers.end(); signer++) {
            string controlref;
            string sign;
            string tosign;

            herr=_mountCustomersMedium(*signer);
            if (!herr.isOk())
                throw herr;

            // create Signature Head
            //_customer->hbci->msgStateResponse("MessageQueue: Creating signature head.");
            SEGSignatureHead sighead(_customer, _syncmode);
            controlref=String::date2string()+String::time2string();
            controlref=controlref.substr(0,14);
            sighead.setData(controlref);
            seg=sighead.toString(--firstseg);

            // sign head AND pure message !
            //_customer->hbci->msgStateResponse("MessageQueue: Signing message.");
            tosign=seg+pureMessage;
            sign=_medium.ref().sign(tosign);

            // store head before message
            result=seg+result;

            // create Signature Tail
            SEGSignatureTail sigtail(_customer);
            sigtail.setData(controlref,sign);
            seg=sigtail.toString(currseg++);
            result+=seg;
        } // for
    } // if needsign

    if (Hbci::debugLevel()>2)
      fprintf(stderr,
	      "MessageQueue:Raw Message\n%s\n",
	      String::dumpToString(result).c_str());

    if (Hbci::debugLevel()>4) {
        char DBG_BUFFER[128];
        sprintf(DBG_BUFFER,"/tmp/request_%d_.dump",_msgnumber);
	FILE *fDBG=fopen(DBG_BUFFER,"w+");
	if (fDBG!=0) {
	    if (result.length())
		fwrite(result.data(),1,result.length(),fDBG);
	    fclose(fDBG);
	}
    }

    // mount owner of messagebox
    herr=_mountCustomersMedium(_customer);
    if (!herr.isOk())
        throw herr;

    // now encrypt the message
    if (needcrypt) {
        string msgkey;
        string cryptedkey;
        string crypteddata;

        // create message key
        msgkey=_medium.ref().createMessageKey();
        // encrypt message
        crypteddata=_medium.ref().encrypt(result,msgkey);
        // encrypt message key
        cryptedkey=_medium.ref().encryptKey(msgkey);
        // create crypt head
        SEGCryptedHead crypthead(_customer, _syncmode);
        crypthead.setData(cryptedkey);
        result=crypthead.toString(0);
        // create crypt data segment
        SEGCryptedData cryptdata(_customer);
        cryptdata.setData(crypteddata);
        result+=cryptdata.toString(0);
    }

    // prepend head
    SEGMessageHead msghead(_customer);
    msghead.setData(message_number,_dialogid);
    seg=msghead.toString(1);
    result=seg+result;
    // append tail
    SEGMessageTail msgtail(_customer);
    msgtail.setData(message_number);
    seg=msgtail.toString(currseg);
    result+=seg;

    // set the size in message head
    msghead.setSize(result);

    // show encrypted message, for key reviewing
    if (Hbci::debugLevel()>3)
      fprintf(stderr,
	      "MessageQueue: Encrypted Message\n%s\n",
	      String::dumpToString(result).c_str());

    // To say it with Merlin: "That's it" ;-)

    if (Hbci::debugLevel()>4) {
        char DBG_BUFFER[128];
        sprintf(DBG_BUFFER,"/tmp/request_%d_crypt.dump",_msgnumber);
	FILE *fDBG=fopen(DBG_BUFFER,"w+");
	if (fDBG!=0) {
	    if (result.length())
		fwrite(result.data(),1,result.length(),fDBG);
	    fclose(fDBG);
	}
    }
    return result;
}


string MessageQueue::dialogId(){
    return _dialogid;
}


bool MessageQueue::setResponse(string response){
  string segment;
  string signature;
  string signedData;
  string tmp;
  string msgkey;
  unsigned int pos=0;
  unsigned int startOfSignedData=0;
  unsigned int endOfSignedData=0;
  list<Pointer<Job> >::const_iterator iter;
  int segNumber;
  Error herr;

  // DEBUG
  if (Hbci::debugLevel() > 4) {
      char DBG_BUFFER[128];
      sprintf(DBG_BUFFER,"/tmp/answer_%d_.crypt.dump",_msgnumber);
      FILE *fDBG=fopen(DBG_BUFFER,"w+");
      if (fDBG!=0) {
	  fwrite(response.data(),1,response.length(),fDBG);
	  fclose(fDBG);
      }
  }
  // /DEBUG

  if (Hbci::debugLevel() > 3)
    fprintf(stderr, "ENCRYPTED MESSAGE:\n%s\n",
	    String::dumpToString(response).c_str());

  herr=_mountCustomersMedium(_customer);
  if (!herr.isOk())
    throw herr;

  // first segment must be a "HNHBK"
  if (String::nextDEG(response,pos)!="HNHBK") {
    return false;
  }

  // run through all segments
  while (pos<response.length()) {
    // current segment
    segment=String::nextSEG(response, pos);
    pos+=segment.length() + 1;

    // get segment reference number
    string head = String::nextDE(segment, 0);
    unsigned int i=String::nextDEG(head, 0).length() + 1;
    i+=String::nextDEG(head, i).length() + 1;
    i+=String::nextDEG(head, i).length() + 1;
    head=String::nextDEG(head, i);
    segNumber = atoi(head.c_str());

    // now handle the segment

    // Message Head ?
    if (String::nextDEG(segment,0) == "HNHBK") {
      SEGMessageHead msghead(_customer);
      unsigned int spos=0;
      if (!msghead.parse(segment,spos)) {
	_medium.ref().unmountMedium();
	_medium=(Medium*)0;
	throw Error("MessageQueue::setResponse",
		    "HNHBK: Bad segment.",0);
      }
      // get dialog id
      _dialogid=msghead.dialogId();
    }

    // overall result ?
    else if (String::nextDEG(segment,0) == "HIRMG") {
      GeneralSegResponse resp(segment);
      generalMsg.push_back(resp);
      _result = resp.returnCode;
    }

    // Crypt head ?
    else if (String::nextDEG(segment, 0) == "HNVSK") {
      SEGCryptedHead crypthead(_customer);
      string crkey;
      unsigned int spos=0;
      if (!crypthead.parse(segment,spos)) {
	_medium.ref().unmountMedium();
	_medium=(Medium*)0;
	throw Error("MessageQueue::setResponse",
		    "HNVSK: Bad segment.",0);
      }
      // read the encrypted key
      crkey=crypthead.cryptedKey();

      // decrypt the key and store it for later use
      msgkey=_medium.ref().decryptKey(crkey);
    }

    // crypt data ?
    else if (String::nextDEG(segment, 0) == "HNVSD") {
      SEGCryptedData cryptdata(_customer);
      string cdata;
      string tail;
      unsigned int spos=0;

      if (!cryptdata.parse(segment,spos)) {
	_medium.ref().unmountMedium();
	_medium=(Medium*)0;
	throw Error("MessageQueue::setResponse",
		    "HNVSD: Bad segment.",0);
      }
      // save rest behind this segment
      tail=response.substr(pos);
      // get encrypted data
      cdata=cryptdata.data();
      // decrypt it and add the tail to it
      response=_medium.ref().decrypt(cdata,msgkey);

      response+=tail;
      pos=0;
      segment=String::nextSEG(response,pos);

      // DEBUG
      if (Hbci::debugLevel() > 4) {
	char DBG_BUFFER[128];
	sprintf(DBG_BUFFER,"/tmp/answer_%d_.dump",_msgnumber);
	FILE *fDBG=fopen(DBG_BUFFER,"w+");
	if (fDBG!=0) {
	  fwrite(response.data(),1,response.length(),fDBG);
	  fclose(fDBG);
	}
      }
      // /DEBUG

      if (Hbci::debugLevel() > 2)
	fprintf(stderr, "DECRYPTED MESSAGE:\n%s\n",
		String::dumpToString(response).c_str());

      // check, if decryption worked: the next segment must be a HIRMS or a HNSHK or a HIRMG
      if ( (String::nextDEG(segment, 0) != "HIRMS") &&
	  (String::nextDEG(segment, 0) != "HNSHK") &&
	  (String::nextDEG(segment, 0) != "HIRMG")){
	_medium.ref().unmountMedium();
	_medium=(Medium*)0;
	throw Error("MessageQueue::setResponse",
		    "ERROR on decryption !!",0);
      }
    } // HNVSD

    // Signature head ?
    else if  (String::nextDEG(segment, 0) == "HNSHK") {
      startOfSignedData = pos - (segment.length() + 1);
    }

    // segment result
    else if (String::nextDEG(segment, 0) == "HIRMS") {
      bool jobFound = false;
      // check which job is responsible for this segment
      for (iter=_jobqueue.begin(); iter != _jobqueue.end(); iter++) {
	if ( (*iter).ref().containsSegment(segNumber) ) {
	  (*iter).ref().jobSuccess(segment);
	  jobFound = true;
	  break;
	}
      } // for
      if (! jobFound)
	try {
	  GeneralSegResponse resp(segment);
	  //_customer->hbci->msgStateResponse(segment);
	  generalMsg.push_back(resp);
	} catch (Error e) {}
    } // HIRMS

    // signature tail
    else if (String::nextDEG(segment, 0) == "HNSHA") {
      unsigned int spos=0;
      if (endOfSignedData==0)
	endOfSignedData=pos-segment.length() - 1; // last pos
      SEGSignatureTail sigtail(_customer);
      if (!sigtail.parse(segment,spos)) {
	_medium.ref().unmountMedium();
	_medium=(Medium*)0;
	throw Error("MessageQueue::setResponse",
		    "HNSHA: Bad segment.",0);
      }
      signature=sigtail.signature();
      signedData=response.substr(startOfSignedData,
				 endOfSignedData-startOfSignedData);

      // if this is RDH-mode, we might have recieved a new key with
      // this message. if this is the first time, we can overstep
      // the verification because it does not make much sense;-)
      // the verification is done by hand with the ini-letter
      // otherwise, we perform the verification!
      if ((HBCI_SECURITY_DDV == _medium.ref().securityMode()) ||
	  ((HBCI_SECURITY_RDH == _medium.ref().securityMode()) &&
	   _medium.cast<MediumRDHBase>().ref().hasInstSignKey())){
	// do the verification
	herr=_medium.ref().verify(signedData,signature);
	if (!herr.isOk())
	  throw Error("MessageQueue::setResponse",herr);
      } // should we do the verification?
    }

    // message tail
    else if (String::nextDEG(segment, 0) == "HNHBS") {
      // message tail
    }

    // nothing special, try to get the responsible job
    else {
      bool done=false;
      for (iter=_jobqueue.begin(); iter != _jobqueue.end(); iter++) {
	if ( (*iter).ref().containsSegment(segNumber) ) {
	  (*iter).ref().parseResponse(segment);
	  done=true;
	  break;
	}
      } // for
      if (!done) {
	// no job responsible, so I handle it myself
	parseGeneralResponse(segment);
      }
    }

  } // while
  return true;
}


void MessageQueue::parseGeneralResponse(string respMsg){
    unsigned int pos=0;
    instituteMessage msg;
    string tmp;
    Pointer<Bank> bank;

    bank.setDescription("MessageQueue::parseGeneralResponse::bank");
    bank=_customer.ref().user().ref().bank();

    if (String::nextDEG(respMsg, pos) == "HIKIM") {
        // skip segment head
        pos+=String::nextDE(respMsg, pos).length() + 1;
        // read subject
        msg.setSubject(String::nextDE(respMsg, pos));
        pos+=String::nextDE(respMsg, pos).length() + 1;
        // read text
        tmp=String::nextDE(respMsg, pos);
        msg.setText(String::unEscape(tmp));
        tmp=String::date2string();
        msg.setDate(Date(tmp,4));
        tmp=String::time2string();
        msg.setTime(Time(tmp));
        msg.setCountry(bank.ref().countryCode());
        msg.setBankCode(bank.ref().bankCode());
        _bankmessages.push_back(msg);
    }
}


bool MessageQueue::hasErrors(){
    list<Pointer<Job> >::const_iterator i;
    bool he;

    he=false;
    for (i=_jobqueue.begin();
         i!=_jobqueue.end();
         i++) {
        if ((*i).ref().hasErrors())
            he=true;
    } // for
    return he;
}


int MessageQueue::getResult(){
  /*fprintf(stderr,
            "WARNING:\n"
            "Your application uses the deprecated method \n"
            "\"MessageQueue::getResult()\".\n"
            "Please make it use \n"
            "\"MessageQueue::hasErrors()\"\n"
            "instead.");*/
    return _result;
}


bool MessageQueue::empty(){
    return _jobqueue.empty();
}


int MessageQueue::size(){
    return _jobqueue.size();
}


MessageReference MessageQueue::messageReference() const {
  return MessageReference(_dialogid, _msgnumber);
}


} // namespace HBCI

