/*
******************************************************************************

   LibGgiMisc implementation for "fbdev" target -- API functions.

   
  
   Copyright (c) Fri Mar  2 2001 by: 
	Brian S. Julin		bri@calyx.com

  
   
   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   The above copyright notice applies to all files in this package, unless 
   explicitly stated otherwise in the file itself or in a file named COPYING 
   in the same directory as the file.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.





******************************************************************************
*/

#include <stdio.h>
#include <ggi/display/fbdev.h>
#include "fbdevggimisc.h"


/* _All_ non-local (ie not declared 'static') functions and variables _must_
   be prefixed with the extension name, and for sublibs also with a unique
   sublib identifier. This is to keep the namespace clean on systems where
   all symbols are exported by default.
*/

/*-* API Implementation */


int GGIMISC_Fbdev_GetRayPos(ggi_visual *vis, sint32 *x, sint32 *y)
{
	int res;
	struct fb_vblank raypos;
	struct fb_var_screeninfo *s;

	/* This interface is not well defined.  Our best bet
	   is to take anything that says it is both capable
	   of being present, and is present, at face value. 

	   Neither is the GGIMisc interface well defined.
	   What to do when a valid count is available and there
	   is a sync or blank is not established.

	   The interworking between the interfaces is even 
	   worse -- GGI shields the user from knowing about
	   scandoubling/pixeldoubling.  Luckily we have 
	   the fbcon fix info.  Unluckily we don't know 
	   for sure how the counter relates to the fix info,
	   since that could be card dependent.

	   Though it would probably be safe to return both
	   the bit-shifted flags | the counts, due to 
	   screen dimensions not being quite 2^31 large,
	   I would say best bet is to define the flags as so:
	   #define GGI_RP_BORDER   0x7ffffffe
           #define GGI_RP_BLANK    0x7ffffffd
           #define GGI_RP_SYNC     0x7ffffffb
           #define GGI_RP_DONTCARE 0x7ffffff7

	   These are relatively impossible for a counter to 
	   hit, and they leave negative numbers for us to 
	   use when a counter is available -- such that 
	   we put 0/0 at the screen origin.  You either
	   get flags _or_ numbers depending on what you pass
	   in -- some additional flags supplied for requesting:
	   GGI_RP_FLAGSONLY, GGI_RP_PIXELS_ADJUSTED, GGI_RP_PIXELS_RAW
	   GGI_RP_DOTS_ADJUSTED, GGI_RP_DOTS_RAW
	   GGI_RP_RAW_ADJUSTED, GGI_RP_RAW

	   On the fbdev side, the expension feilds in fb_vblank
	   are there by my plea, and hopefully will someday be used 
	   in the future to specify just what is being returned in 
	   the count variables.

	   Comments?

	   For now, if a count is there in either dimension,
	   it is used and Or'd with the flags, if they are present.
	   The fixinfo/count stuff is not done, as the only fbdev
	   device that has this does provide a vcount, so
	   the code would not be reached.

	*/
	*y = 0; *x = 0;
	if ((res = ioctl(LIBGGI_FD(vis),FBIOGET_VBLANK,&raypos))) return(res);
	if (!raypos.flags) return(GGI_ENOFUNC); /* Got functionality? */
	if ((raypos.flags & FB_VBLANK_HAVE_VBLANK)
	    && (raypos.flags & FB_VBLANK_VBLANKING))
	  *y |= GGI_RP_BLANK;
	if ((raypos.flags & FB_VBLANK_HAVE_VSYNC)
	    && (raypos.flags &  FB_VBLANK_VSYNCING))
	  *y |= GGI_RP_SYNC;
	if ((raypos.flags & FB_VBLANK_HAVE_VCOUNT) && raypos.vcount)
	  *y = raypos.vcount & 0x7fffffff;
	if ((raypos.flags & FB_VBLANK_HAVE_HBLANK)
	    && (raypos.flags & FB_VBLANK_HBLANKING))
	  *x |= GGI_RP_BLANK;
	if ((raypos.flags & FB_VBLANK_HAVE_HCOUNT) && raypos.hcount)
	  *x = raypos.hcount & 0x7fffffff;
	if (*x | *y) return 0;
	if (!((raypos.flags & FB_VBLANK_HAVE_COUNT) && raypos.count)) 
	  return(0);
	
	/* Now it gets ugly. */	
	/* Punt!
	   We're going to assume that counter is in pixels from
	   the origin for now.
	*/
	s = &(FBDEV_PRIV(vis)->var);
	if (s->xres == 0) return(0);
	*y = raypos.count / s->xres;
	*x = raypos.count - *y * s->xres;
	return(0);
}


int GGIMISC_Fbdev_WaitRayPos(ggi_visual *vis, sint32 *x, sint32 *y)
{
  struct fb_vblank raypos;
  uint32 flags = 0;
  int res;

  /* This implementation is just wrong.  Needs a lot of work 
     and clearer API definition. */
  if (*x == GGI_RP_DONTCARE && *y == GGI_RP_DONTCARE) return(0);
  if ((res = ioctl(LIBGGI_FD(vis),FBIOGET_VBLANK,&raypos))) return(res);
  if (!raypos.flags) return(GGI_ENOFUNC); /* Got functionality? */
  if (*x & GGI_RP_BLANK) flags |= FB_VBLANK_HAVE_HBLANK;
  if (*y & GGI_RP_BLANK) flags |= FB_VBLANK_HAVE_VBLANK;
  if (*y & GGI_RP_SYNC)  flags |= FB_VBLANK_HAVE_VSYNC;
  if ((flags & raypos.flags) != flags) {
    *x = 0; *y = 0; /* This tells user to try using numbers instead */
    return(GGI_ENOFUNC);
  }
  if (flags) {
    flags = 0;
    if (*x & GGI_RP_BLANK) flags |= FB_VBLANK_HBLANKING;
    if (*y & GGI_RP_BLANK) flags |= FB_VBLANK_VBLANKING;
    if (*y & GGI_RP_SYNC)  flags |= FB_VBLANK_VSYNCING;
    /* Yeah, we busy poll.  
       Weren't expecting something fancy, were you? */   
    while (1) {
      res = ioctl(LIBGGI_FD(vis),FBIOGET_VBLANK,&raypos);
      if (res) return(GGI_EUNKNOWN);
      if ((flags & raypos.flags) == flags) return(0);
    }
  }
  /* I may have hacked getraypos, but this could cause a hang,
     so we won't do FB_VBLANK_HAVE_COUNT this time. */
  if ((*x != GGI_RP_DONTCARE) && !(raypos.flags & FB_VBLANK_HAVE_HCOUNT)) 
    return(GGI_ENOFUNC);
  if ((*y != GGI_RP_DONTCARE) && !(raypos.flags & FB_VBLANK_HAVE_VCOUNT)) 
    return(GGI_ENOFUNC);
  while(1) {
    res = ioctl(LIBGGI_FD(vis),FBIOGET_VBLANK,&raypos);
    if (res) return(GGI_EUNKNOWN);
    if (((*x == GGI_RP_DONTCARE) || (*x < raypos.hcount)) &&
	((*y == GGI_RP_DONTCARE) || (*y < raypos.vcount))) return(0);
  }
}


/* Expecting this to work?  Read the source... the only
   fb driver that seems to really do it now is amifb. */
int GGIMISC_Fbdev_SetSplitline(ggi_visual *vis, int y)
{
  int res;
  struct fb_var_screeninfo var;
  if (FBDEV_PRIV(vis)->orig_fix.ypanstep == 0)
    return(GGI_ENOFUNC);
#warning threadsafe?
  memcpy(&var, &(FBDEV_PRIV(vis)->var), sizeof(var));
  var.yoffset = y;
  var.xoffset = 0;
  var.vmode |= FB_VMODE_YWRAP; 
  if ((res = ioctl(LIBGGI_FD(vis),FBIOGET_VBLANK,&var))) 
    return(res); 
  return(0);
}
