#line 2 "string.c"
/*-
 * C-SaCzech
 * Copyright (c) 1996-2002 Jaromir Dolecek <dolecek@ics.muni.cz>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Jaromir Dolecek
 *	for the CSacek project.
 * 4. The name of Jaromir Dolecek may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY JAROMIR DOLECEK ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL JAROMIR DOLECEK BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: string.c,v 1.55 2002/02/03 11:13:41 dolecek Exp $ */

/* CSacek functions which work with strings */

#include "csacek.h"

/* 
 * tries to find ``search'' in ``str''; the comparison is case insensitive
 * note this routine works for ASCII only
 */                   
const char *
csa_strcasestr( str, search )
  const char *str, *search;
{
	const char *i,*j;

	if (!*search) return str;

   	for(; *str; str++) {
	   if (CSA_UPPER(*str) == CSA_UPPER(*search)) {
		i = str+1;
		j = search+1;
		for(; *i && *j && CSA_UPPER(*i) == CSA_UPPER(*j); i++, j++);
		if (*j == '\0') return str;
	   }
	}

	return NULL;
}

/*
 * fills csa_String with values given; if ``len'' is lower than 0,
 * strlen(val) is supstutited instead of it; if ``maxlen'' is lower
 * than zero, ``len'' is subtituted instead of it
 */
void
csa_fillstring(str, val, len, maxlen)
  csa_String *str;
  const char   *val;
  int len, maxlen;
{
	str->value = val;
	str->len = (len >= 0) ? len : strlen(val);
	str->maxlen = (maxlen >= 0) ? maxlen : str->len;
}

/* creates & fills in new csa_String struct */
csa_String *
csa_createstring(wpool, val)
  struct pool *wpool;
  const char *val;
{
	csa_String *newstr;
	newstr = (csa_String *) ap_palloc(wpool, sizeof(csa_String));
	csa_fillstring(newstr, val, -1, -1);
	return newstr;
}

/*
 * finds out, if ``str'' ends with one of items in ``list'';
 * items in ``list'' are separated by ``sep''
 * returns pointer at suffix found in  string ``str'' or NULL if no suffix
 * was found 
 */
const char * 
csa_has_suffix(str, list, sep_int)
  const char *str, *list;
  int sep_int;
{
   const char *b,  *suffix_pos=NULL;
   char sep=(char)sep_int;
   int str_len = 0, len = 0;

   str_len = strlen(str);
   while( *list )
   {
      b = strchr(list, sep);
      if (b == NULL) b = strchr(list,0); /* last item */
      len = b - list;
      if ( len <= str_len && len > 0 
		/* LINTED */
		&& strncmp(list, &str[str_len-len], len) == 0 )
      {
	suffix_pos = &str[str_len-len];
	break;
      }
      if (*b) list = ++b; /* shift list behind separator */
      else list = b;
   }

   return suffix_pos;
}                                             

/*
 * this finds first __FOO__ string and returns position where it was found,
 * position of text behind it and csa_String structure with what it
 * should be substituted by
 * return length of found item, 0 if nothing has been found
 */
size_t
csa_find_subs(p, str, len, idxp, subs)
  csa_params_t *p;
  const char *str;
  size_t len, *idxp;
  const csa_String **subs;
{
	const char *temp, *newtemp;
	size_t add_len=0, remains=len;

	temp = str;
	for(; remains > 0;) {
		newtemp = memchr(temp, '_', remains);
		if (!newtemp) break; /* stop if '_' not found */

		remains -= (newtemp - temp);
		/* if there is no chance anything would match (the remaining
		 * part is too short), just stop right away */
		if (remains < 8) break;
			
		temp = newtemp;
		if (temp[1] != '_') {
			temp++, remains--;
			continue;
		}
	
		if (remains>=11 && strncasecmp(temp, "__CHARSET__", 11) == 0) {
			add_len = 11;
			*subs	= &p->charset;
		}
		else if (remains>=8 && strncasecmp(temp, "__PART__", 8) == 0) {
			add_len = 8;
			*subs	= &p->part;
		}
		else if (remains>=16
			  && strncasecmp(temp, "__LAMPACHARSET__", 16) == 0) {
			add_len = 16;
			*subs	= &p->lampacharset;
		} else {
			temp++, remains--;
			continue; /* nothing supported */
		}

		/*
		 * Go berserk if add_len is longer than current value of
		 * CSA_FINDSUBS_MAX - cause a SIGSEGV in that case.
		 */
		if (add_len > CSA_FINDSUBS_MAX) {
			char *foo = NULL;
			foo[0] = 0;
		}

		*idxp = temp - str;
		return add_len;
	}

	return 0; /* nothing supported found */
}

/*
 * parses string and substitutes all occurencies of supported __FOO__
 * strings by appropriate values; returns new strings with all
 * __FOO__ strings substituted or original string if no __FOO__ string
 * has been found in it
 * space for new string is allocated in temporary pool, so it will be
 * reused as soon as we reach main loop in loop.c again
 */
char *
csa_subs_string(p, str)
  csa_params_t *p;
  char *str;
{
	char *buff=NULL, *past=NULL;
	const csa_String *subs;
	size_t buff_len=0, need_len, skipped, idx, idx_rest, rest_len;
	size_t str_len = strlen(str), past_len = 0;
	int use_past;

	/*
	 * XXX This migh be slighly optimized by keeping point of where
	 * we stopped searching last time and starting csa_find_subs()
	 * from there. However, csa_subs_string() is typically called
	 * for short strings, so implementing this optimalization is
	 * not worth the extra complexity of code it would introduce.
	 */
	while((skipped = csa_find_subs(p, str, str_len, &idx, &subs)) != 0) {
		idx_rest = idx + skipped;
		rest_len = str_len - idx_rest;
		if (skipped >= subs->len) {
			memcpy(&str[idx], subs->value, subs->len);
			memmove(&str[idx + subs->len], &str[idx_rest],rest_len);
			str_len = str_len - (skipped - subs->len);
		} else {
			need_len = str_len + (subs->len - skipped);
			if (buff_len < need_len) {
				buff = (char *)csa_alloca(need_len + 1,
							p->pool_tmp);
				buff_len = need_len;
				if (idx) memcpy(buff, str, idx);
				use_past = 0;
			} else {
				/*
				 * Temp buffer is long enough to hold the entire
				 * string, but subs string is longer than
				 * original - have to keep the rest of string
				 * so that it can be correctly appended after
				 * the substitution takes place.
				 */
				if (past_len < rest_len)  {
					past = (char *)csa_alloca(rest_len,
							p->pool_tmp);
					past_len = rest_len;
				}
				memcpy(past, &str[idx_rest], rest_len);
				use_past = 1;
			}
			memcpy(&buff[idx], subs->value, subs->len);
			memcpy(&buff[idx + subs->len],
				(use_past) ? past : &str[idx_rest], rest_len);
			str = buff;
			str_len = need_len;
		}
	}
	str[str_len] = '\0';

	return (buff) ? ap_pstrndup(p->pool_tmp, str, (int)str_len) : str;
}

/*
 * It checks whether strings are non-null before comparing them. 
 * Returns: 0 - same, 1 - different or any is null
 */
int
csa_n_strcmp(s1, s2)
  const char *s1, *s2;
{
    return !(s1 && s2 && strcmp(s1, s2) == 0);
}
