// =============================================================================
//
//      --- kvi_taskbarbutton.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviTaskbarButton"

#define _KVI_TASKBARBUTTON_CPP_

#include <qdrawutil.h>
#include <qpainter.h>
#include <qpixmap.h>
#include <qtooltip.h>

#include "kvi_app.h"
#include "kvi_defines.h"
#include "kvi_taskbar.h"
#include "kvi_taskbarbutton.h"
#include "kvi_window.h"

/**
 * KviTaskBarButton
 */
KviTaskBarButton::KviTaskBarButton(QWidget *parent, KviTaskBar *pTaskBar, KviWindow *win_ptr, QPixmap *mem_buf)
	: QWidget(parent)
{
	m_pWindow      = win_ptr;
	m_pMemBuffer   = mem_buf;
	m_pTaskBar     = pTaskBar;
	m_bIsOn        = false;
	m_bAttached    = false;
	m_bHighlighted = false;
	m_bAltColor    = false;
	m_iProgress    = 0;

	setBackgroundMode(NoBackground);
	QToolTip::add(this, win_ptr->caption());
	KviStr tmp(KviStr::Format, "taskbar_button_%s", win_ptr->caption());
	setName(tmp.ptr());
	setFocusPolicy(NoFocus);
}

KviTaskBarButton::~KviTaskBarButton()
{
	// Nothing here
}

void KviTaskBarButton::paintEvent(QPaintEvent *e)
{
	if( !isVisible() ) {
#ifdef _KVI_DEBUG_CHECK_RANGE_
		if( e )
			debug("KviTaskBarButton::paintEvent() while not visible!");
#endif
		return;
	}
	int w = width();
	int h = height();
	QPainter pa(m_pMemBuffer);
	QColorGroup cg = colorGroup();
	QColor qcolor;
	QFont fnt = font();
	if( m_pWindow->isMinimized() ) {
		qcolor = cg.dark();
		fnt.setItalic(true);
	} else {
		qcolor = cg.buttonText();
		fnt.setItalic(false);
	}
	setFont(fnt);
	QFontMetrics fm(fnt);
	pa.setFont(fnt);

	const QPixmap *pix = m_pTaskBar->backgroundPixmap();
	if( pix ) {
		if( !pix->isNull() ) {
			QPoint p = m_pTaskBar->mapFromGlobal(mapToGlobal(QPoint(0, 0)));
			pa.drawTiledPixmap(-p.x(), -p.y(), w, h, *pix);
		} else pix = 0;
	}
	if( !pix ) // Also catch the case that pixmap exists but isNull()
		pa.fillRect(0, 0, w, h, cg.button());
	qDrawShadePanel(&pa, 0, 0, w, h, cg, m_bIsOn, 1, 0);
	int leftPos = (h - 16) >> 1;
	if( m_pWindow->myIconPtr() )
		bitBlt(m_pMemBuffer, leftPos, leftPos, m_pWindow->myIconPtr(), 0, 0, 16, 16);
	if( !m_bAttached )
		qDrawShadeRect(&pa, leftPos + 16, 1, w - 2, h - 2, cg, false, 0, 0);
	leftPos += KVI_TASKBAR_BORDER + 16;
	int maxWidth = w - (leftPos + KVI_TASKBAR_BORDER);

	if( m_iProgress > 0 ) {
		// Paint the progress bar
		int rectW = maxWidth * m_iProgress / 100;
		pa.fillRect(leftPos - 1, KVI_TASKBAR_BORDER, rectW, h - KVI_TASKBAR_BUTTON_DOUBLE_BORDER, cg.mid());
	}
	if( m_bIsOn ) {
		m_bHighlighted = false; // Clear highlight if on
		m_bAltColor    = false;
	}
	QColor *clr;
	if( m_bHighlighted ) {
		pa.setPen(m_bAltColor ? blue : red);
		QColor qcolor = m_bAltColor ? (QColor) blue : (QColor) red;
		clr = &qcolor;
	}
	pa.setPen(qcolor);
	clr = &qcolor;
	// Calculate the length of the string to paint
	int len      = 0;
	int strWidth = 0;
	if( maxWidth > 10 ) {
		pa.setClipRect(leftPos, KVI_TASKBAR_BORDER, maxWidth, h - (KVI_TASKBAR_BORDER << 1));
		pa.setClipping(true);
		register const char *p = m_pWindow->caption();
		while( *p ) {
			strWidth += fm.width(*p);
			if( strWidth < maxWidth ) {
				p++;
				len++;
			} else {
				if( len > 0 ) { // Go back one char to allow space for points
					strWidth -= fm.width(*p);
					len--;
					p--;
					strWidth -= fm.width(*p);
				}
				maxWidth = 0; // This means that we must draw the points
				break;
			}
		}
		int yPos = KVI_TASKBAR_BORDER + fm.ascent();

		if( len )
			pa.drawText(leftPos, yPos, QString(m_pWindow->caption()), len);
		if( maxWidth == 0 ) {
			leftPos += strWidth;
			pa.drawText(leftPos, yPos, QString("..."), 3);
		}
		pa.setClipping(false);
	}
	bitBlt(this, 0, 0, m_pMemBuffer, 0, 0, w, h);
}

void KviTaskBarButton::mousePressEvent(QMouseEvent *e)
{
	m_bHighlighted = false;
	m_bAltColor    = false;
	if( e->button() & LeftButton ) {
		if( e->state() & ShiftButton )
			m_pWindow->delayedClose();
		else
        	m_pTaskBar->buttonLeftClicked(m_pWindow, this);
	} else m_pTaskBar->buttonRightClicked(m_pWindow, this);
}

bool KviTaskBarButton::isOn()
{
	return m_bIsOn;
}

void KviTaskBarButton::setOn(bool bOn)
{
	if( bOn == m_bIsOn ) return;

	m_bIsOn = bOn;
	paintEvent(0);
}

void KviTaskBarButton::setAttachState(bool bAttached)
{
	if( m_bAttached == bAttached ) return;

	m_bHighlighted = false;
	m_bAltColor    = false;
	m_bAttached    = bAttached;
	paintEvent(0);
}

void KviTaskBarButton::highlight(bool bAltColor)
{
	if( !m_bIsOn ) {
		if( (!m_bHighlighted) || (bAltColor && !m_bAltColor) ) {
			m_bHighlighted = true;
			m_bAltColor    = bAltColor;
			paintEvent(0);
		}
	}
	g_pApp->globalHighlight(bAltColor);
}

void KviTaskBarButton::updateButton()
{
	// Update the tooltip
	QToolTip::remove(this);
	QToolTip::add(this, m_pWindow->caption());
	paintEvent(0);
}

#include "m_kvi_taskbarbutton.moc"
