/* @(#) idnsd.c 1.4 @(#) */
/***************************************************************\
*	Copyright (c) 1999 First Step Internet Services, Inc.
*		All Rights Reserved
*	Distributed under the BSD Licenese
*
*	Module: IDNSD
\***************************************************************/

#define _KOALAMUD_MAIN_C "@(#) nitehawk@winghove.1ststep.net|src/idnsd/idnsd.c|20000507155731|37699 @(#)"

/* This daemon reads an IP address on its stdin and writes the resolved name
 * to stdout.  It is expected that all input lines are either IP4 or IP6
 * addresses needing resolved.  If at some point there is a real need for name
 * to IP resolution, it will most likely be implemented in this subserver as
 * well.
 */

#include "autoconf.h"
#include "version.h"

/* Constants controlling operations */
/* Wait 5 seconds between retires */
#define IDNSD_RETRY_PAUSE	1

/* Maximum number of retries */
#define IDNSD_MAX_RETRIES	5

/* Length of a request line */
#define IDNSD_REQUEST_SIZE	512

char cvsstrings[][100] =
{
	_KOALAMUD_VERSION_H,
	_KOALAMUD_MAIN_C,
};

/* Display version string */
void displayversion(void)
{
	printf("%s", COPYRIGHTSTR);
}

/* Display help message */
void displayhelp(void)
{
	printf("%s", COPYRIGHTSTR);
	printf("Usage:\n");
	printf(
"-h             Show this screen\n"
"-V             Display version information\n"
);
}

/* Parse command line parameters */
int parseoptions(int argc, char *argv[])
{
	char opt = EOF;		// Current Option
	const char optionlist[] = "hV";	// Accepted Options

	opterr= 0;

	while ((opt = getopt(argc, argv, optionlist)) != -1)
	{
		switch (opt)
		{
		case 'h':  // User is requesting help
			displayhelp();
			exit(1);

		case 'V':
			displayversion();
			exit(1);

		case '?':
			printf("Bad option '%c' on command line.\n",
				optopt);
			displayhelp();
			exit(1);

		case ':':
			printf("Option '%c' missing argument.\n",
				optopt);
			displayhelp();
			exit(1);

		default:
			printf("Bad option '%c' on command line.\n",
				optopt);
			displayhelp();
			exit(1);
		}

	}

	return 0;
}

/* Lookup an reverse address */
#if AFINDEP
/* AF Independant version */
static void lookup(char *inbuf)
{
	struct addrinfo *ai = NULL;
	int retries = 0;
	int err;
	char hostname[IDNSD_REQUEST_SIZE];

	/* Loop until we either reach our max retry count or successfully resolve
	 * the address */
	while ((err = getaddrinfo(inbuf, NULL, NULL, &ai)) != 0)
	{
		/* Print out an error reply on error */
		switch (err)
		{
		case EAI_ADDRFAMILY:
			printf("$error:  AF Problem resolving address\n");
			return;
		case EAI_FAIL:
			printf("$error:  Nonrecoverable resolution failure\n");
			return;
		case EAI_MEMORY:
			printf("$error:  Memory allocation failure\n");
			return;
		default:
			++retries;
		}

		if (retries >= IDNSD_MAX_RETRIES)
		{
			printf("$error:  Max retry count reached\n");
			return;
		}

		/* Give ourselves a moment to let the error possibly recover */
		sleep(IDNSD_RETRY_PAUSE);
	}

	/* Ok, we now have the information we need in an addrinfo struct, resolve
	 * it to the hostname with getnameinfo */
	while ((err = getnameinfo(ai->ai_addr, ai->ai_addrlen, hostname,
			IDNSD_REQUEST_SIZE, NULL, 0, 0)) != 0)
	{
		/* Print out an error reply on error */
		switch (err)
		{
		case EAI_ADDRFAMILY:
			printf("$error:  AF Problem resolving address\n");
			return;
		case EAI_FAIL:
			printf("$error:  Nonrecoverable resolution failure\n");
			return;
		case EAI_MEMORY:
			printf("$error:  Memory allocation failure\n");
			return;
		default:
			++retries;
		}

		if (retries >= IDNSD_MAX_RETRIES)
		{
			printf("$error:  Max retry count reached\n");
			return;
		}

		/* Give ourselves a moment to let the error possibly recover */
		sleep(IDNSD_RETRY_PAUSE);
	}

	/* Print out reply message */
	printf("$name: %d %s\n", strlen(hostname), hostname);

	/* Free resolver response */
	freeaddrinfo(ai);
}
#else
/* AF_INET version */
static void lookup(char *inbuf)
{
}
#endif

int main(int argc, char *argv[])
{
	int err = 0;
	char rdbuf[IDNSD_REQUEST_SIZE];
	char *el;

	err = parseoptions(argc, argv);

	/* Loop until we run out of data */
	for (;;)
	{
		/* Clear out buffer */
		memset(rdbuf, '\0', IDNSD_REQUEST_SIZE);
		
		/* Read a request line */
		if (fgets(rdbuf, IDNSD_REQUEST_SIZE, stdin) == NULL)
			exit(2);

		if ((el = strrchr(rdbuf, '\n')) == NULL)
			continue;   // Ignore incomplete lines
		*el = '\0';		// Strip the \n character
		if ((el = strrchr(rdbuf, '\r')) != NULL)
			*el = '\0';	// strip \r character

		/* Check for special commands before calling for a lookup */
		if (strcmp(rdbuf, "$hello") == 0)
		{
			printf("$alive\n");
			fflush(stdout);
			continue;
		}
		if (strcmp(rdbuf, "$shutdown") == 0)
		{
			exit(0);
		}

		/* resolve the address */
		lookup(rdbuf);

		/* Flush stdout */
		fflush(stdout);
	}

	return err;
}
