/***************************************************************************
                       iptchain.h  -  description
                          -------------------
 begin                : Mon Jan 28 2002
 copyright            : (C) 2002 by Christian Hubinger
 email                : a9806056@unet.univie.ac.at
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef IPTCHAIN_H
#define IPTCHAIN_H
#include "qptrlist.h"
#include "qstring.h"
#include "qstringlist.h"
class QString;
class KMFError;
class IPTRule;
class IPTable;

/**
  *@author Christian Hubinger
  */
class IPTChain {
public:

	/** The Constructor that will set the 2Name" and the "Table" of the chain.
	  If buildin is set true the chain will be handeled as bild-in chain. Normaly this
	  is only done at the initialklistaion of the whole KMFDoc.
	  */
	IPTChain( const QString& name , IPTable* table, bool buildin = false );
	~IPTChain();

	/** Sets the name of the Chain as it will be used in the script
	*/
	void setName( const QString& );

	/** Sets the table to which this chain will be added
	*/
	void setTable( IPTable* table );

	IPTable* table() const;

	/** Sets the Default Policy of the Chain if it is a build in chain like ( $IPT -P INPUT -j DROP )
	  For user defined chain this function generates a rule like ( $IPT -A CHAINNAME -j DROP )
	  which will be added at the end of the chain to avoid that the packetes jump back to the
	  Chain that feeded this chain.
	  */
	void setDefaultTarget( const QString& );

	/** Enabbles/Disables the build in logging for this Chain. This will generate a Rule
	like: $IPT -A CHAINNAME --match limit --limit <limit> --limit-burst <burst> -j LOG --log-prefix <prefix>
	This rule will be placed right before the Default Policy to log just the Packets that don't match any
	of the rules defined by this chain.
	limit, burst, prefix may be empty strings to disable those features.
	If enable is "false" the logging will be deactivated.
	*/
	void setDropLogging( bool enable, QString& limit , QString& burst , QString& prefix );

	/** Defines weather a Chain is build-in or is user-defined
	*/
	void setBuildIn( bool ); // define if chain is user defined or a built in chain
	/** will set a switch that lets the chain be shown by the GUI
	*/
	void setUsed( bool ); // defines if the Chain is used by its document

	/** Sets if a chain has a "Default Policy" set.
	*/
	void hasCustomDefaultTarget( bool has_target );

	/** Moves the rule up (to beginning of the chain) if how_much < 0 else
	  moves the rule down (to end of the chain)
	  */
	bool moveRule( IPTRule* rule, int how_much );



	/** Simply adds a Rule to the Chains Ruleset.
	*/
	IPTRule* addRule( const QString& name );
	/** Deletes a given Rule in the Ruleset.
	*/
	bool delRule( IPTRule& );

	/** Returns "true" if the logging is enabled, otherwhise "false".
	*/
	bool logging() const {
		return enable_log;
	};

	/** Returns "true" if the Chain is a build-in chain and "fasle" if it is
	  an user defined chain
	  */
	bool isBuildIn() const {
		return is_build_in_chain;
	};

	/** Returns "true" if the Chain has a "Default Policy" set or not.
	*/
	bool hasDefaultTarget() const {
		return has_default_target;
	};

	/** Returns "true" if the Chain should be visible inn the GUI.
	*/
	bool isUsed() const {
		return is_used;
	};

	void setChainRuleset( QPtrList<IPTRule>* rules ) {
		m_ruleset = *rules;
	}

	/** Returns the name of the Chain.
	*/
	const QString& name() const {
		return m_name;
	} ;

	/** Returns the limit of matches for the logging - rule.
	*/
	const QString& logLimit() const {
		return m_log_limit;
	} ;

	/** Returns the log prefix
	*/
	const QString& logPrefix() const {
		return m_log_prefix;
	};

	/** Returns the log limit burst number.
	*/
	const QString& logBurst() const {
		return m_log_burst;
	};

	/** Returns the returns the Chaindefinition ($IPT -N chainname) if this is a user defined
	  chain. if it's a build in chain like INPUT it returnes an empty string.
	  */
	const QString& chainDefinitions() const {
		return m_cmd_chain_definition ;
	} ;

	/** Returns the Chain Policy ( "DROP" "ACCEPT" "LOG" etc )
	*/
	const QString& defaultTarget() const {
		return m_default_target;
	} ;

	/** Returns rules that forward packets to other chains.
	  Those Rules have a chain as their target ( -j <chainname>)
	  */
	QPtrList<IPTRule>& chainFwds();

	/** Returns rules that forward packets to this chain.
	  Those Rules have this chain as their target ( -j <chainname>)
	  */
	QPtrList<IPTRule>& chainFeeds();

	/** Returns the whole ruleset defined by the Chain as
	    an QPtrList<IPTRule>.
	    */
	QPtrList<IPTRule>& chainRuleset() const;

	/** Returns a list of strings like: "$IPT -A CHAINNAME OPTIONS [ all kind of allowed optiions p.e --match
	  state --state NEW,RELATED --source 192.168.0.2 ... ] -j TARGET [target options p.e --log-prefix 'KMF:' ]"
	  */
	QPtrList<QStringList>& createIPTablesChainRules();

	/** Returns a String like: "$IPT -P INPUT DROP" for build-in and "$IPT -A CHAINNAME -j DROP"
	  for user defined chans.
	  */
	QString createIPTablesChainDefaultTarget();

	/** Returns a String like: "$IPT -N CHAINNAME" that is written to the script.
	*/
	QString createIPTablesChainDefinition();  // returns iptables -N <CHAIN_NAME>

	/** Simply adds a Rule to the Chains Ruleset.
	*/
	KMFError* addRule( IPTRule& );

	/** Create a deep copy of this chain
	*/
	IPTChain* createChainClone();

private:    // functions
	/** Sets the rule numbers in the rules to the actual position
	  in the Chain
	  */
	void regenerateRuleNumbers();

private:    // data
	// initialisized by the constructor
	IPTable* m_table;
	KMFError* m_err;

	bool is_used;
	bool enable_log;
	bool is_build_in_chain;
	bool has_default_target;

	QString m_name;
	QString m_log_limit;
	QString m_log_prefix;
	QString m_log_burst;

	QString m_default_target;

	/** this is the QPrtList that stores the IPTRule* objects
	  that belog to this chain
	  */
	QPtrList<IPTRule> m_ruleset;


	QString m_cmd_chain_definition;
	QString m_cmd_default_target;

};

#endif
