
/*
 * config.C -- written for Juice
 *	Copyright (C) 1999, 2000, 2001 Abraham vd Merwe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef CONFIG_CONFIG_C
#define CONFIG_CONFIG_C

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>

#include "typedefs.h"
#include "config.h"

Config::Config ()
{
   entries = NULL;
}

Config::~Config ()
{
   ConfigList *tmp;
   while (entries != NULL)
	 {
		if (entries->next == NULL)
		  {
			 if (entries->type == CFG_TYPE_STRING) free (entries->txt);
			 free (entries->idname);
			 entries = NULL;
		  }
		else
		  {
			 tmp = entries;
			 while (tmp->next->next != NULL) tmp = tmp->next;
			 if (tmp->next->type == CFG_TYPE_STRING) free (tmp->next->txt);
			 free (tmp->next->idname);
			 tmp->next = NULL;
		  }
	 }
}

void Config::AddInteger (const char *idname,int value)
{
   ConfigList *nw,*tmp;
   nw = (ConfigList *) malloc (sizeof (ConfigList));
   nw->type = CFG_TYPE_INTEGER;
   nw->value = value;
   nw->idname = (char *) malloc (strlen (idname) + 1);
   strcpy (nw->idname,idname);
   nw->next = NULL;
   if (entries == NULL)
	 entries = nw;
   else
	 {
		tmp = entries;
		while (tmp->next != NULL) tmp = tmp->next;
		tmp->next = nw;
	 }
}

void Config::AddBoolean (const char *idname,bool state)
{
   ConfigList *nw,*tmp;
   nw = (ConfigList *) malloc (sizeof (ConfigList));
   nw->type = CFG_TYPE_BOOLEAN;
   nw->state = state;
   nw->idname = (char *) malloc (strlen (idname) + 1);
   strcpy (nw->idname,idname);
   nw->next = NULL;
   if (entries == NULL)
	 entries = nw;
   else
	 {
		tmp = entries;
		while (tmp->next != NULL) tmp = tmp->next;
		tmp->next = nw;
	 }
}

void Config::AddString (const char *idname,const char *txt)
{
   ConfigList *nw,*tmp;
   nw = (ConfigList *) malloc (sizeof (ConfigList));
   nw->type = CFG_TYPE_STRING;
   nw->txt = (char *) malloc (strlen (txt) + 1);
   strcpy (nw->txt,txt);
   nw->idname = (char *) malloc (strlen (idname) + 1);
   strcpy (nw->idname,idname);
   nw->next = NULL;
   if (entries == NULL)
	 entries = nw;
   else
	 {
		tmp = entries;
		while (tmp->next != NULL) tmp = tmp->next;
		tmp->next = nw;
	 }
}

void Config::RemoveEntry (const char *idname)
{
   ConfigList *current,*previous;
   bool finished = FALSE;
   current = entries;
   previous = NULL;
   while ((current != NULL) && (!finished))
     {
		if (strcmp (current->idname,idname) == 0)
		  {
			 if ((previous != NULL) && (current->next != NULL)) previous->next = current->next;
			 else if (previous == NULL) entries = current->next;
			 else previous->next = NULL;
			 if (current->type == CFG_TYPE_STRING) free (current->txt);
			 free (current->idname);
			 free (current);
			 finished = TRUE;
		  }
		else
		  {
			 previous = current;
			 current = current->next;
		  }
	 }
}

void Config::RemoveAllEntries ()
{
   ConfigList *tmp;
   while (entries != NULL)
	 {
		if (entries->next == NULL)
		  {
			 if (entries->type == CFG_TYPE_STRING) free (entries->txt);
			 free (entries->idname);
			 entries = NULL;
		  }
		else
		  {
			 tmp = entries;
			 while (tmp->next->next != NULL) tmp = tmp->next;
			 if (tmp->next->type == CFG_TYPE_STRING) free (tmp->next->txt);
			 free (tmp->next->idname);
			 tmp->next = NULL;
		  }
	 }
}

bool Config::IsEmpty ()
{
   return (entries == NULL);
}

bool Config::GetEntry (const char *idname,int &type,int &value,bool &state,char *txt)
{
   ConfigList *tmp;
   tmp = entries;
   while ((tmp != NULL) && (strcmp (tmp->idname,idname) != 0)) tmp = tmp->next;
   if (tmp == NULL) return FALSE;
   type = tmp->type;
   if (type == CFG_TYPE_INTEGER) value = tmp->value;
   else if (type == CFG_TYPE_BOOLEAN) state = tmp->state;
   else strcpy (txt,tmp->txt);
   return TRUE;
}

int Config::Parse (const char *filename)
{
   FILE *handle;	
   char curline[CFG_LINE_LENGTH],variable[CFG_LINE_LENGTH],value[CFG_LINE_LENGTH];
   int src,dest;
   bool started;
   if ((handle = fopen (filename,"r")) == NULL) return CFG_ERROR_ACCESS;
   do
	 {
		if (fgets (curline,CFG_LINE_LENGTH,handle) == NULL) break;
		/* 1. Strip leading spaces */
        src = 0;
		while ((isspace (curline[src])) && (src < (int) strlen (curline))) src++;
		if (src == (int) strlen (curline)) continue;
		/* 2. Skip comment if one is present */
		if (curline[src] == '#') continue;
		/* 3. Read variable name */
		dest = 0;
		while ((!isspace (curline[src])) && (src < (int) strlen (curline)))
		  variable[dest++] = curline[src++];
		if (src == (int) strlen (curline))
		  {
			 fclose (handle);
			 return CFG_ERROR_CORRUPT;
		  }
		variable[dest] = '\0';
		/* 4. Strip spaces */
		while ((isspace (curline[src])) && (src < (int) strlen (curline))) src++;
		if (src == (int) strlen (curline))
		  {
			 fclose (handle);
			 return CFG_ERROR_CORRUPT;
		  }
		/* 5. Get equal sign */
		if (curline[src++] != '=')
		  {
			 fclose (handle);
			 return CFG_ERROR_CORRUPT;
		  }
		/* 6. Strip spaces */
		while ((isspace (curline[src])) && (src < (int) strlen (curline))) src++;
		if (src == (int) strlen (curline))
		  {
			 fclose (handle);
			 return CFG_ERROR_CORRUPT;
		  }
		/* 7. Read value */
		dest = 0;
		started = FALSE;
		while ((curline[src] != '\n') && (src < (int) strlen (curline)))
		  {
			 if (curline[src] == '"') started = started ? FALSE : TRUE;
			 if ((curline[src] == '#') && (!started)) break;
			 value[dest++] = curline[src++];
		  }
		if (src == (int) strlen (curline))
		  {
			 fclose (handle);
			 return CFG_ERROR_LINES;
		  }
		value[dest] = '\0';
		for (src = 0; src < (int) strlen (variable); src++)
		  variable[src] = tolower (variable[src]);
		/* 8. Strip trailing spaces from value */
		src = strlen (value) - 1;
		while ((isspace (value[src])) && (src > 0)) src--;
		value[src + 1] = '\0';
		/* 9. Add statement to entries */
		if (isdigit (*value))
		  {
			 src = 0;
			 while (src < (int) strlen (value)) if (!isdigit (value[src++]))
			   {
				  fclose (handle);
				  return CFG_ERROR_CORRUPT;
			   }
			 AddInteger (variable,atoi (value));
		  }
		else if (*value == '"')
		  {
			 char *tmp = value;
			 tmp++;
			 tmp[strlen (tmp) - 1] = '\0';
			 AddString (variable,tmp);
		  }
		else
		  {
			 if (strcasecmp (value,"true") == 0)
			   AddBoolean (variable,TRUE);
			 else if (strcasecmp (value,"false") == 0)
			   AddBoolean (variable,FALSE);
			 else
			   {
				  fclose (handle);
				  return CFG_ERROR_CORRUPT;
			   }
		  }
	 }
   while (!feof (handle));
   fclose (handle);
   return CFG_ERROR_NONE;
}

int Config::Create (const char *filename,const char *format, ...)
{
   FILE *handle;
   ConfigList *tmp;
   char header[CFG_LINE_LENGTH];
   va_list ap;
   va_start (ap,format);
   if (vsnprintf (header,sizeof (header) - 1,format,ap) == -1) return CFG_ERROR_LINES;
   va_end (ap);
   if ((handle = fopen (filename,"w")) == NULL) return CFG_ERROR_ACCESS;
   fprintf (handle,"# %s\n",header);
   tmp = entries;
   while (tmp != NULL)
	 {
		fprintf (handle,"%s = ",tmp->idname);
		switch (tmp->type)
		  {
		   case CFG_TYPE_INTEGER:
			 fprintf (handle,"%d\n",tmp->value);
			 break;
		   case CFG_TYPE_BOOLEAN:
			 if (tmp->state) fprintf (handle,"True\n"); else fprintf (handle,"False\n");
			 break;
		   case CFG_TYPE_STRING:
			 fprintf (handle,"\"%s\"\n",tmp->txt);
			 break;
		   default:
			 break;
			 /* This shouldn't happen */
		  }
		tmp = tmp->next;
	 }
   fclose (handle);
   return CFG_ERROR_NONE;
}

int Config::Save (const char *filename)
{
   FILE *handle;
   char *buffer;
   char curline[CFG_LINE_LENGTH],variable[CFG_LINE_LENGTH],
   		newline[CFG_LINE_LENGTH],newtxt[CFG_LINE_LENGTH];
   int src,dest,type,newvalue,newstate;
   bool started;
   buffer = NULL;
   if ((handle = fopen (filename,"r")) == NULL) return CFG_ERROR_ACCESS;
   do
	 {
		if (fgets (curline,CFG_LINE_LENGTH,handle) == NULL) break;
		*newline = '\0';
		/* 1. Strip leading spaces */
        src = 0;
		while ((isspace (curline[src])) && (src < (int) strlen (curline)))
		  {
			 newline[src] = curline[src];
			 src++;
		  }
		if (src == (int) strlen (curline))
		  {
			 if (buffer != NULL)
			   {
				  buffer = (char *) realloc (buffer,strlen (buffer) + strlen (curline) + 1);
				  strcat (buffer,curline);
			   }
			 else
			   {
				  buffer = (char *) malloc (strlen (curline) + 1);
				  strcpy (buffer,curline);
			   }
			 continue;
		  }
		/* 2. Skip comment if one is present */
		if (curline[src] == '#')
		  {
			 if (buffer != NULL)
			   {
				  buffer = (char *) realloc (buffer,strlen (buffer) + strlen (curline) + 1);
				  strcat (buffer,curline);
			   }
			 else
			   {
				  buffer = (char *) malloc (strlen (curline) + 1);
				  strcpy (buffer,curline);
			   }
			 continue;
		  }
		/* 3. Read variable name */
		dest = 0;
		while ((!isspace (curline[src])) && (src < (int) strlen (curline)))
		  {
			 newline[src] = variable[dest] = curline[src];
			 dest++;
			 src++;
		  }
		if (src == (int) strlen (curline))
		  {
			 fclose (handle);
			 free (buffer);
			 return CFG_ERROR_CORRUPT;
		  }
		variable[dest] = '\0';
		/* 4. Strip spaces */
		while ((isspace (curline[src])) && (src < (int) strlen (curline)))
		  {
			 newline[src] = curline[src];
			 src++;
		  }
		if (src == (int) strlen (curline))
		  {
			 fclose (handle);
			 free (buffer);
			 return CFG_ERROR_CORRUPT;
		  }
		/* 5. Get equal sign */
		if (curline[src] != '=')
		  {
			 fclose (handle);
			 free (buffer);
			 return CFG_ERROR_CORRUPT;
		  }
		else
		  {
			 newline[src] = curline[src];
			 src++;
		  }
		/* 6. Strip spaces */
		while ((isspace (curline[src])) && (src < (int) strlen (curline)))
		  {
			 newline[src] = curline[src];
			 src++;
		  }
		if (src == (int) strlen (curline))
		  {
			 fclose (handle);
			 free (buffer);
			 return CFG_ERROR_CORRUPT;
		  }
		newline[src] = '\0';
		/* 7. Read value */
		started = FALSE;
		while ((curline[src] != '\n') && (src < (int) strlen (curline)))
		  {
			 if (curline[src] == '"') started = started ? FALSE : TRUE;
			 if ((curline[src] == '#') && (!started)) break;
			 src++;
		  }
		if (src == (int) strlen (curline))
		  {
			 fclose (handle);
			 free (buffer);
			 return CFG_ERROR_LINES;
		  }
		/* 8. Strip trailing spaces from value */
		src--;
		while ((isspace (curline[src])) && (src > 0)) src--;
		src++;
		/* 9. Update value */
		for (int i = 0; i < (int) strlen (variable); i++)
		  variable[i] = tolower (variable[i]);
		if (!GetEntry (variable,type,newvalue,newstate,newtxt))
		  {
			 fclose (handle);
			 free (buffer);
			 return CFG_ERROR_CORRUPT;
		  }
		else
		  {
			 if (type == CFG_TYPE_BOOLEAN)
			   {
				  if (newstate) strcat (newline,"True"); else strcat (newline,"False");
			   }
			 else if (type == CFG_TYPE_INTEGER)
			   {
				  char tmp[20];
				  sprintf (tmp,"%d",newvalue);
				  strcat (newline,tmp);
			   }
			 else
			   {
				  strcat (newline,"\"");
				  strcat (newline,newtxt);
				  strcat (newline,"\"");
			   }
		  }
		/* 10. Append rest of curline */
		dest = strlen (newline);
		while (src < (int) strlen (curline)) newline[dest++] = curline[src++];
		newline[dest] = '\0';
		/* 11. Write newline to buffer */
		if (buffer != NULL)
		  {
			 buffer = (char *) realloc (buffer,strlen (buffer) + strlen (newline) + 1);
			 strcat (buffer,newline);
		  }
		else
		  {
			 buffer = (char *) malloc (strlen (newline) + 1);
			 strcpy (buffer,newline);
		  }
	 }
   while (!feof (handle));
   fclose (handle);
   if ((handle = fopen (filename,"w")) == NULL)
	 {
		free (buffer);
		return CFG_ERROR_ACCESS;
	 }
   fwrite (buffer,strlen (buffer),sizeof (char),handle);
   fclose (handle);
   return CFG_ERROR_NONE;
}

#endif
