//
//      Sound/Timer engine wrappers for Allegro
//

#ifndef _WIN32
	#include <unistd.h>     // Fork
	#include <signal.h>
#else
	#include <process.h>
#endif
#include <stdlib.h>     // atexit(), random
#include <string.h>     // String splicing routines

#include <allegro.h>
#include "ithelib.h"
#include "console.h"
#include "media.h"
#include "loadfile.h"
#include "sound.h"
#include "ogglib.h"

// defines

// variables

extern char mus_chan;                      // Number of music channels
extern char sfx_chan;                      // Number of effects channels
extern int  driftlevel;                    // Amount of frequency variation

int sf_volume = 63;
int mu_volume = 63;

int Songs=0;
int Waves=0;
static int pausedata=0;   // Speed of song before pausing, 0 if not paused

struct SMTab *wavtab;       // Tables of data for each sound and song.
struct SMTab *mustab;       // Filled by script.cc

// functions

void S_Load();              // Load the sound and music
void S_SoundVolume();       // Set the Sound volume
void S_MusicVolume();       // Set the Music volume
void S_PlaySample(char *s,int v);   // Play a sound sample
void S_PlayMusic(char *s);  // Play a song
void S_StopMusic();         // Stop the music

static void LoadMusic();
static void LoadWavs();

static int ForkPlayer_start(char *songfile, long offset);
static void ForkPlayer_stop();
static void ForkPoll();


// Public code

#if 0
/*
 *      S_PollMusic - Keep the music playing
 */

void S_PollMusic()
{
#ifndef NO_AUD
poll_aud();
#endif
}
#endif

/*
 *      S_Load - Load in the music and samples.
 */

void S_Load()
{
atexit(ForkPlayer_stop);
LoadMusic();
LoadWavs();
}


/*
 *      S_MusicVolume - Set the music level
 */

void S_MusicVolume(int vol)
{
mu_volume = vol;
#ifndef NO_AUD
set_aud_volume(vol); // 0-255
#endif
return;
}


/*
 *      S_SoundVolume - Set the effects level
 */

void S_SoundVolume(int vol)
{
sf_volume = vol;
return;
}


/*
 *   S_PlayModule - play a music module from the array song[]
 *                  may disrupt the sound effects slightly
 */

void S_PlayMusic(char *name)
{
IFILE *ifp;
char filename[1024];
long offset,ctr;

#ifdef NO_SOUND
return;
#endif

for(ctr=0;ctr<Songs;ctr++)
	if(!istricmp(mustab[ctr].name,name))
		{
		// If the music is marked Not Present, ignore it (music is optional)
		if(mustab[ctr].fname[0] == 0)
			return;
		// Otherwise, no excuses

		if(!loadfile(mustab[ctr].fname,filename))
			{
			Bug("S_PlayMusic - could play song %s, could not find file '%s'\n",name,mustab[ctr].fname);
			return;
			}

		ifp = iopen(filename);
		offset = ifp->origin;
		strcpy(filename,ifp->truename);
		iclose(ifp);

		if(!ForkPlayer_start(filename,offset))
			{
			// Try again, with system prefix
			ifp = iopen(filename);
			offset = ifp->origin;
			strcpy(filename,ifile_prefix);
			strcat(filename,ifp->truename);
			strslash(filename);
			iclose(ifp);
			if(!ForkPlayer_start(filename,offset))
				Bug("S_PlayMusic - could not play song '%s'\n",name);
			}

        return;
        }

Bug("S_PlayMusic - could not find song '%s'\n",name);
}

/*
 *   S_PauseModule - Stop music, not SFX
 */

void S_PauseMusic()
{
return;
}

/*
 *   S_UnPauseModule - Restart music
 */

void S_UnPauseMusic()
{
return;
}

/*
 *   S_PlaySample - play a sound sample.
 */

void S_PlaySample(char *name, int volume)
{
int freq,num,ctr;

#ifdef NO_SOUND
return;
#endif

for(ctr=0;ctr<Waves;ctr++)
    if(!istricmp(wavtab[ctr].name,name))
        {
        freq=1000;
        if(!wavtab[ctr].nodrift)
            {
            num = 10 - (rand()%20);
            freq+=(num*driftlevel)/10;
            }
		play_sample(wavtab[ctr].sample,volume,128,freq,0); // vol,pan,freq,loop

//        if(!wavtab[ctr].handle)
//                    MIDASerror();
        return;
        }

Bug("S_PlaySample- could not find sound '%s'\n",name);
}


//
//   Private code hereon
//

/*
 *   LoadMusic - Make sure the music files are there, else mark as Not Present
 */

void LoadMusic()
{
char filename[1024];
int ctr;

ilog_printf("  Checking music.. ");     // This line is not terminated!

for(ctr=0;ctr<Songs;ctr++)
	if(!loadfile(mustab[ctr].fname,filename))
		{
		ilog_quiet("Warning: Could not load music file '%s'\n",mustab[ctr].fname);
		mustab[ctr].fname[0]=0;	// Erase the filename to mark N/A
		}

ilog_printf("done.\n");  // This terminates the line above

}


/*
 *   LoadWavs - Load in the sounds
 */

void LoadWavs()
{
char filename[1024];
int ctr;

#ifdef NO_SOUND
return;
#endif

// load in each sound

ilog_printf("  Loading sounds");     // This line is not terminated, for the dots

Plot(Waves);    // Work out how many dots to print

for(ctr=0;ctr<Waves;ctr++)
	{
        if(!loadfile(wavtab[ctr].fname,filename))
            ithe_panic("LoadWavs: Cannot open WAV file",wavtab[ctr].fname);
        wavtab[ctr].sample=iload_wav(filename);
        if(!wavtab[ctr].sample)
            ithe_panic("LoadWavs: Invalid WAV file",filename);
        Plot(0);                                        // Print a dot
	}

ilog_printf("\n");  // End the line of dots

}

#if 0
/*
static pid_t ForkID=0;
static FILE *oggfile;

// Create a fork of the program and run the player.
// Then self-destruct.

int ForkPlayer_start(char *songfile, long offset)
{
int r;
if(ForkID)
	{
	kill(ForkID,SIGKILL); // SANCTIFY
	unload_vorbis_stream();
	oggfile=NULL;
	}

// Open the file up first
oggfile=fopen(songfile,"rb");
if(!oggfile)
	return 0; // Failed

// Now try playing it
fseek(oggfile,offset,SEEK_SET);

ForkID=fork();
if(ForkID)
	return 1;

if(!load_vorbis_stream_offset(oggfile))
		{
		fclose(oggfile);
		raise(SIGKILL); // Ouch!
//		return 0;
		}

while(poll_vorbis_stream() != -1);
unload_vorbis_stream();
oggfile=NULL;
ForkID=0;       // Kinda pointless
raise(SIGKILL); // Suicide
return 0;
}

// Stop any music

void ForkPlayer_stop()
{
if(ForkID)
	{
	kill(ForkID,SIGKILL); // SANCTIFY
	unload_vorbis_stream();
	oggfile=NULL;
	}
ForkID=0;
}
*/
#endif


static volatile FILE *oggfile;
static volatile int Playing;

// Create a fork of the program and run the player.
// Then self-destruct.

int ForkPlayer_start(char *songfile, long offset)
{
int r;
if(Playing)
	{
	remove_int(poll_vorbis_stream);
	unload_vorbis_stream();
	oggfile=NULL;
	Playing=0;
	}

LOCK_VARIABLE(oggfile);
LOCK_VARIABLE(Playing);
LOCK_FUNCTION(ForkPoll);

// Open the file up first
oggfile=fopen(songfile,"rb");
if(!oggfile)
	return 0; // Failed

// Now try playing it
fseek(oggfile,offset,SEEK_SET);

if(!load_vorbis_stream_offset(oggfile))
		{
		fclose(oggfile);
		oggfile=NULL;
		return 0;
		}

Playing=1;
install_int_ex(ForkPoll,BPS_TO_TIMER(44));
return 1;
}

// Stop any music

void ForkPlayer_stop()
{
if(Playing)
	{
	remove_int(ForkPoll);
	unload_vorbis_stream();
	oggfile=NULL;
	Playing=0;
	}
}

void ForkPoll()
{
if(Playing)
	{
	if(poll_vorbis_stream() == -1)
		{
		Playing=0;
		unload_vorbis_stream();
		oggfile=NULL;
		remove_int(ForkPoll);
		}
	}
}
END_OF_FUNCTION(ForkPoll);


