/* gxset - GTK interface to xset(1)

   Copyright (C) 1999 Ren Seindal (rene@seindal.dk)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. 
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <math.h>
#include <string.h>
#include <stdlib.h>

#include <gtk/gtk.h>

#include "support.h"
#include "callbacks.h"
#include "interface.h"

#include "main.h"
#include "guistuff.h"

GtkWidget *
find_widget(GtkWidget *w, const char *name)
{
    GtkWidget *tmp = lookup_widget(w, name);

    if (tmp == NULL)
	g_error(_("Widget lookup failure %s - shouldn't happen!"), name);
    return tmp;
}

ApplicationState *
get_application_state(GtkWidget *w)
{
    return (ApplicationState *)gtk_object_get_user_data(GTK_OBJECT(w));
}

GUI *
get_widget_gui(GtkWidget *w)
{
    return (GUI *)gtk_object_get_data(GTK_OBJECT(w), "gui");
}

void
set_widget_gui(GtkWidget *w, GUI *gui)
{
    gtk_object_set_data(GTK_OBJECT(w), "gui", gui);
}

void
make_gui(GtkWidget *w)
{
    GUI *gui;

    gui = g_new(GUI, 1);

    gui->gxset = w;

    gui->keyrepeat_onoff = find_widget(w, "keyrepeat_onoff");
    gui->keyrepeat_delay = find_widget(w, "keyrepeat_delay");
    gui->keyrepeat_rate = find_widget(w, "keyrepeat_rate");

    gui->keyclick_onoff = find_widget(w, "keyclick_onoff");
    gui->keyclick_volume = find_widget(w, "keyclick_volume");

    gui->bell_onoff = find_widget(w, "bell_onoff");
    gui->bell_pitch = find_widget(w, "bell_pitch");
    gui->bell_volume = find_widget(w, "bell_volume");
    gui->bell_duration = find_widget(w, "bell_duration");
    gui->beep_button = find_widget(w, "beep_button");

    gui->mouse_accel = find_widget(w, "mouse_accel");
    gui->mouse_threshold = find_widget(w, "mouse_threshold");

    gui->scrsaver_onoff = find_widget(w, "scrsaver_onoff");
    gui->scrsaver_delay = find_widget(w, "scrsaver_delay");
    gui->scrsaver_cycle = find_widget(w, "scrsaver_cycle");
    gui->scrsaver_blank = find_widget(w, "scrsaver_blank");
    gui->scrsaver_expose = find_widget(w, "scrsaver_expose");

    gui->dpms_onoff = find_widget(w, "dpms_onoff");
    gui->dpms_standby = find_widget(w, "dpms_standby");
    gui->dpms_suspend = find_widget(w, "dpms_suspend");
    gui->dpms_off = find_widget(w, "dpms_off");

    gui->font_path = find_widget(w, "font_path");
    gui->font_button_new = find_widget(w, "font_button_new");
    gui->font_button_edit = find_widget(w, "font_button_edit");
    gui->font_button_up = find_widget(w, "font_button_up");
    gui->font_button_down = find_widget(w, "font_button_down");
    gui->font_button_remove = find_widget(w, "font_button_remove");

    gui->button_ok = find_widget(w, "button_ok");
    gui->button_apply = find_widget(w, "button_apply");
    gui->button_save = find_widget(w, "button_save");
    gui->button_cancel = find_widget(w, "button_cancel");

    set_widget_gui(w, gui);
}



typedef void (*UpdateFunc)(GtkToggleButton *w, gpointer user_data);

static void
toggle_button_set(GtkWidget *w, gboolean val)
{
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), val);
}

static gboolean
toggle_button_get(GtkWidget *w)
{
    return gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w));
}


static void
spin_button_set(GtkWidget *w, gint val)
{
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(w), val);
}

static gint
spin_button_get(GtkWidget *w)
{
    return gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(w));
}

static void
spin_button_set_fraction(GtkWidget *w, gint num, gint div)
{
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(w), ((gfloat)(num))/div);
}

static void
spin_button_get_fraction(GtkWidget *w, gint *nump, gint *divp)
{
    gfloat val = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(w));

    /* This assumes max one decimal in the spin button */
    *divp = 10;			/* divisors 2 and 5 */
    *nump = floor((val * 10) + 0.5);

    while (*divp%2 == 0 && *nump%2 == 0) {
	*divp /= 2;
	*nump /= 2;
    }

    while (*divp%5 == 0 && *nump%5 == 0) {
	*divp /= 5;
	*nump /= 5;
    }
}


static void
text_entry_set(GtkWidget *w, gchar *val)
{
    gtk_entry_set_text(GTK_ENTRY(w), val);
}

static gchar *
text_entry_get(GtkWidget *w)
{
    return gtk_entry_get_text(GTK_ENTRY(w));
}


static void
clist_set_rows(GtkWidget *w, GPtrArray *vec)
{
    int i;
    gchar *tmp[1];
    GtkCList *clist = GTK_CLIST(w);

    gtk_clist_freeze(clist);
    gtk_clist_clear(clist);

    for (i = 0; i < vec->len; i++) {
	tmp[0] = g_ptr_array_index(vec, i);
	gtk_clist_append(clist, tmp);
    }

    gtk_clist_thaw(clist);
}



void
adjust_font_buttons(GtkWidget *gxset, ApplicationState *appstate)
{
    int row = appstate->font_path_selected;
    GUI *gui = get_widget_gui(gxset);

    gtk_widget_set_sensitive(gui->font_button_edit, row >= 0);
    gtk_widget_set_sensitive(gui->font_button_up, row > 0);
    gtk_widget_set_sensitive(gui->font_button_down, 
			     (row >= 0 
			      && row < appstate->cur->font_path->len-1));
    gtk_widget_set_sensitive(gui->font_button_remove, row >= 0);

}

void
set_font_path(GtkWidget *gxset, ApplicationState *appstate)
{
    GUI *gui = get_widget_gui(gxset);

    clist_set_rows(gui->font_path, appstate->cur->font_path);

    if ((appstate->font_path_selected >= 0
	 && appstate->font_path_selected < appstate->cur->font_path->len))
	gtk_clist_select_row(GTK_CLIST(gui->font_path),
			     appstate->font_path_selected, 0);

    adjust_font_buttons(gxset, appstate);
}

void
set_gui_state(GtkWidget *gxset, ApplicationState *appstate)
{
    XsetInfo *info = appstate->cur;
    GUI *gui = get_widget_gui(gxset);

    /* Key auto-repeat */
    toggle_button_set(gui->keyrepeat_onoff, info->keyrepeat_onoff);
    on_keyrepeat_onoff_toggled(GTK_TOGGLE_BUTTON(gui->keyrepeat_onoff), gxset);
    spin_button_set(gui->keyrepeat_delay, info->keyrepeat_delay);
    spin_button_set(gui->keyrepeat_rate, info->keyrepeat_rate);

    /* Key click */
    toggle_button_set(gui->keyclick_onoff, info->keyclick_volume > 0);
    on_keyclick_onoff_toggled(GTK_TOGGLE_BUTTON(gui->keyclick_onoff), gxset);
    spin_button_set(gui->keyclick_volume, info->keyclick_volume);

    /* Bell */
    toggle_button_set(gui->bell_onoff, info->bell_onoff);
    on_bell_onoff_toggled(GTK_TOGGLE_BUTTON(gui->bell_onoff), gxset);
    spin_button_set(gui->bell_volume, info->bell_volume);
    spin_button_set(gui->bell_pitch, info->bell_pitch);
    spin_button_set(gui->bell_duration, info->bell_duration);

    /* Mouse */
    spin_button_set_fraction(gui->mouse_accel, 
			     info->mouse_acceleration,
			     info->mouse_divisor);
    spin_button_set(gui->mouse_threshold, info->mouse_threshold);

    /* Screen saver */
    toggle_button_set(gui->scrsaver_onoff, info->scrsaver_onoff);
    on_scrsaver_onoff_toggled(GTK_TOGGLE_BUTTON(gui->scrsaver_onoff), gxset);
    toggle_button_set(gui->scrsaver_blank, info->scrsaver_blank);
    toggle_button_set(gui->scrsaver_expose, info->scrsaver_expose);

    spin_button_set(gui->scrsaver_delay, info->scrsaver_delay);
    spin_button_set(gui->scrsaver_cycle, info->scrsaver_cycle);

    /* DPMS */
    toggle_button_set(gui->dpms_onoff, info->dpms_onoff != 0);
    on_dpms_onoff_toggled(GTK_TOGGLE_BUTTON(gui->dpms_onoff), gxset);

    spin_button_set(gui->dpms_standby, info->dpms_standby);
    spin_button_set(gui->dpms_suspend, info->dpms_suspend);
    spin_button_set(gui->dpms_off, info->dpms_off);

    set_font_path(gxset, appstate);
}

void
get_gui_state(GtkWidget *gxset, ApplicationState *appstate)
{
    XsetInfo *info = appstate->cur;
    GUI *gui = get_widget_gui(gxset);

    info->keyrepeat_onoff = toggle_button_get(gui->keyrepeat_onoff);
    info->keyrepeat_delay = spin_button_get(gui->keyrepeat_delay);
    info->keyrepeat_rate = spin_button_get(gui->keyrepeat_rate);

    info->keyclick_volume = spin_button_get(gui->keyclick_volume);

    info->bell_onoff = toggle_button_get(gui->bell_onoff);
    info->bell_volume = spin_button_get(gui->bell_volume);
    info->bell_pitch = spin_button_get(gui->bell_pitch);
    info->bell_duration = spin_button_get(gui->bell_duration);

    spin_button_get_fraction(gui->mouse_accel, 
			     &info->mouse_acceleration,
			     &info->mouse_divisor);
    info->mouse_threshold = spin_button_get(gui->mouse_threshold);

    info->scrsaver_onoff = toggle_button_get(gui->scrsaver_onoff);
    info->scrsaver_blank = toggle_button_get(gui->scrsaver_blank);
    info->scrsaver_expose = toggle_button_get(gui->scrsaver_expose);

    info->scrsaver_delay = (info->scrsaver_onoff 
			    ? spin_button_get(gui->scrsaver_delay)
			    : 0);
    info->scrsaver_cycle = spin_button_get(gui->scrsaver_cycle);

    info->dpms_onoff = toggle_button_get(gui->dpms_onoff);
    info->dpms_standby = spin_button_get(gui->dpms_standby);
    info->dpms_suspend = spin_button_get(gui->dpms_suspend);
    info->dpms_off = spin_button_get(gui->dpms_off);
}




gchar *
font_dialog_get_data(GtkWidget *font_dialog)
{
    gchar *new_path;
    GtkNotebook *nb;

    nb = (GtkNotebook *)find_widget(font_dialog, "notebook");
    if (gtk_notebook_get_current_page(nb) == 0) { /* directory */
	gchar *file;
	gboolean allow_scaling;

	file = text_entry_get(find_widget(font_dialog, "font_dir_entry"));
	allow_scaling = toggle_button_get(find_widget(font_dialog,
						      "font_dir_scaling"));

	if (!(file && *file))
	    new_path = NULL;
	else {
	    file = g_strdup(file); /* GtkEntry doesn't malloc for us */

	    if (!g_path_is_absolute(file)) {
		gchar *cwd = g_get_current_dir();
		gchar *abs_path = g_strjoin("/", cwd, file, NULL);

		g_free(cwd);
		g_free(file);

		file = abs_path;
	    }

	    if (allow_scaling)
		new_path = g_strdup(file);
	    else
		new_path = g_strdup_printf("%s:unscaled", file);

	    g_free(file);
	}
    } else {		/* font server */
	gchar *host;
	gint port;

	host = text_entry_get(find_widget(font_dialog, "font_host_entry"));
	port = spin_button_get(find_widget(font_dialog, "font_port"));

	if (host && *host)
	    new_path = g_strdup_printf("%s/:%d", host, port);
	else
	    new_path = NULL;
    }
    return new_path;
}

GtkWidget *
font_dialog_new(gchar *old_path)
{
    GtkWidget *font_dialog;
    GtkNotebook *nb;

    gint len;
    gchar *tmp;
    gchar *portp;
    gint port;

    font_dialog = create_new_font();
    nb = (GtkNotebook *)find_widget(font_dialog, "notebook");

    if (old_path == NULL) {	/* New entry */
	gtk_notebook_set_page(nb, 0);
    } else {
	tmp = g_strdup(old_path);

	if (g_path_is_absolute(old_path)) {	/* Directory entry */
	    len = strlen(tmp);

	    if (len > 9 && strcmp(tmp+len-9, ":unscaled") == 0) {
		toggle_button_set(find_widget(font_dialog,
					      "font_dir_scaling"), FALSE);
		tmp[len-9] = '\0';
	    } else {
		toggle_button_set(find_widget(font_dialog, 
					      "font_dir_scaling"), TRUE);
	    }

	    text_entry_set(find_widget(font_dialog, "font_dir_entry"), tmp);
	    gtk_notebook_set_page(nb, 0);
	} else if ((portp = strstr(tmp, "/:")) != NULL) { /* Font server */
	    *portp = '\0';
	    portp += 2;

	    port = atoi(portp);

	    if (port > 0)
		spin_button_set(find_widget(font_dialog, "font_port"), port);

	    text_entry_set(find_widget(font_dialog, "font_host_entry"), tmp);
	    gtk_notebook_set_page(nb, 1);
	} else {
	    text_entry_set(find_widget(font_dialog, "font_dir_entry"), tmp);
	    gtk_notebook_set_page(nb, 0);
	}

	g_free(tmp);
    }

    return font_dialog;
}
