/*
 * Copyright (C) 2002,2003 Pascal Haakmat.
 * Licensed under the GNU GPL.
 * Absolutely no warranty.
 */

#include <string.h>
#include <math.h>
#include <signal.h>
#include <config.h>
#include "lib/cpudetect.h"
#include "lib/minmax.h"
#include "pref.h"
#include "mem.h"
#include "gui.h"
#include "module.h"
#include "main.h"

extern GList *shells;
struct func_table funcs;
struct sigaction oldact;
CpuCaps gCpuCaps;

int Emergency;

RETSIGTYPE
sighandler(int sig, 
           siginfo_t *siginfo, 
           void *ptr) {
    char  fname[4096];
    int in_chans, out_chans;
    GList *l;
    mixer *output_mixer;
    shell *shl;
    
    if(Emergency) {
        /* Oh god, crashed in sighandler. */
        FAIL("crashed while trying to recover from previous crash! sorry, bye.\n");
        if(sigaction(siginfo->si_signo, &oldact, NULL) == -1) 
            FAIL("could not install old signal handler\n");
        return;
    }
    Emergency = 1;
    printf("**********************************************************************\n");
    printf("**********************   GNUSOUND CRASHED!   *************************\n");
    printf("**********************************************************************\n");
    printf("GNUsound has crashed! _Trying_ to save unsaved data...\n");
   for(l = shells; l; l = l->next) {
       printf("shell @ %p...", 
              ((shell *)(l->data)));
       if(((shell *)(l->data))->magic[0] == 'O' &&
          ((shell *)(l->data))->magic[1] == 'k') {
           shl = (shell *)(l->data);
           printf("magic ok");
           if(snd_verify(shl->sr)) 
               printf("...snd_verify ok\n");
           else {
               printf("...snd_verify failed\n");
               continue;
           }
           if(!shl->has_changed) {
               printf(" not changed, not saving\n");
               continue;
           }
           output_mixer = mixer_new(shl->sr->channels, shl->sr->channels);        
           if(!output_mixer) {
               gui_alert("Cannot create output mixer for %d tracks. " \
                         "Very low memory.", shl->sr->channels);
               return;
           }
           in_chans = shl->sr->channels;
           out_chans = shl->sr->channels;
           output_mixer->is_unity = 1;
           strcpy(fname, shl->sr->name);
           strcat(fname, ".crash");
           printf(" saving as %s\n", fname);
           file_save_internal(fname, shl, in_chans, out_chans, output_mixer);
       } else
           printf("corrupt!\n");
   }

    printf("crash recovery complete, continuing with normal crash procedure...\n");
    if(sigaction(siginfo->si_signo, &oldact, NULL) == -1) 
        FAIL("could not install old signal handler\n");
}


int
main(int argc, 
     char *argv[]) {
    int r;
    struct sigaction act;

    funcs.peak_max_func = max_int8_c;
    funcs.peak_min_func = min_int8_c;
    funcs.max8_func = max_int8_c;
    funcs.min8_func = min_int8_c;
    funcs.max16_func = max_int16_c;
    funcs.min16_func = min_int16_c;

#ifdef HAVE_ARCH_X86
#ifdef RUNTIME_CPUDETECT
    GetCpuCaps(&gCpuCaps);
#if USE_MMX_MINMAX == 2
    DEBUG("using static MMX minmax\n");
#elif USE_MMX_MINMAX == 1
    if(gCpuCaps.hasMMX2) {
        DEBUG("using MMX2 minmax\n");
        funcs.peak_max_func = max_int8_mmx2;
        funcs.peak_min_func = min_int8_mmx2;
        funcs.max8_func = max_int8_mmx2;
        funcs.min8_func = min_int8_mmx2;
        funcs.max16_func = max_int16_mmx2;
        funcs.min16_func = min_int16_mmx2;
    } else if(gCpuCaps.has3DNow) {
        DEBUG("using 3DNow! minmax\n");
        funcs.peak_max_func = max_int8_3dnow;
        funcs.peak_min_func = min_int8_3dnow;
        funcs.max8_func = max_int8_3dnow;
        funcs.min8_func = min_int8_3dnow;
        funcs.max16_func = max_int16_3dnow;
        funcs.min16_func = min_int16_3dnow;
    }
#endif /* USE_MMX_MINMAX */
#endif /* RUNTIME_CPUDETECT */
#endif /* HAVE_ARCH_X86 */
 
    if(mem_init()) {
        FAIL("fatal error initializing private malloc library.\n");
        return 1;
    }

    if(pref_init()) {
        FAIL("fatal error initializing preferences.\n");
        return 1;
    }

    if(module_init()) {
        FAIL("fatal error initializing modules.\n");
        return 1;
    }

    if(sigaction(SIGSEGV, NULL, &oldact) == -1) 
        FAIL("could not get signal handler\n");

    if(gui_init(argc, argv)) {
        FAIL("fatal error initializing GUI.\n");
        return 1;
    }

    /*
     * gui_init() installs the sucky GNOME signal handler.
     * Replace it with our own ones here.
     */

    act.sa_flags = SA_SIGINFO;
    sigemptyset(&act.sa_mask);
    act.sa_sigaction = sighandler;
    if(sigaction(SIGFPE, &act, NULL) == -1) 
        FAIL("could not install SIGFPE signal handler\n");
    if(sigaction(SIGSEGV, &act, NULL) == -1) 
        FAIL("could not install SIGSEGV signal handler\n");
    if(sigaction(SIGBUS, &act, NULL) == -1) 
        FAIL("could not install SIGBUS signal handler\n");
    if(sigaction(SIGTERM, &act, NULL) == -1) 
        FAIL("could not install SIGTERM signal handler\n");
    if(sigaction(SIGABRT, &act, NULL) == -1) 
        FAIL("could not install SIGABRT signal handler\n");
    if(sigaction(SIGILL, &act, NULL) == -1) 
        FAIL("could not install SIGILL signal handler\n");
    
    r = gui_run(argc, argv);

    module_exit();
    pref_exit();
    mem_exit();
#ifndef HAVE_GNOME2
    gtk_exit(r);
#endif
    return r;

}
