/******************************************************************************\
 gnofin/dialog-file.c   $Revision: 1.7 $
 Copyright (C) 1999-2000 Darin Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

#include "common.h"
#include <gtk/gtksignal.h>
#include <gtk/gtkfilesel.h>
#include <gtk/gtkmain.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkoptionmenu.h>
#include <gtk/gtkmenu.h>
#include <gtk/gtkradiomenuitem.h>
#include <libgnomeui/gnome-uidefs.h>
#include <libgnomeui/gnome-preferences.h>
#include "dialogs.h"
#include "file-utils.h"

/* modal wrapper to the builtin GTK file selection dialog */
typedef struct {
  GtkFileSelection *filesel;
  gboolean accepted;
  gboolean protect;	/* if true, file must not exist */
} FileSelectionInfo;

static void
filesel_ok (GtkWidget *widget, FileSelectionInfo *info)
{
  gchar *name;

  trace("");

  info->accepted = TRUE;
  name = gtk_file_selection_get_filename (info->filesel);

  if (info->protect && file_exists (name))
  {
    gint result;

    name = g_strdup_printf (_("Overwrite existing file '%s'?"), name);
    result = dialog_question_yes_no (GTK_WINDOW (info->filesel), name);
    g_free (name);

    switch (result)
    {
    case DIALOG_YES:
      break;
    case DIALOG_NO:
      info->accepted = FALSE;
      return;
    }
  }

  gtk_main_quit();
}

static void
format_select (GtkMenuItem *item, FileFormatSelection *format)
{
  GList *menu_items;

  trace ("");
  g_return_if_fail (format);

  menu_items = GTK_MENU_SHELL (GTK_WIDGET (item)->parent)->children;
  format->selection = g_list_index (menu_items, item);

  trace ("selection = %d", format->selection);
}

static void
fill_format_selector (GtkOptionMenu *optmenu, GtkMenu *menu, FileFormatSelection *format)
{
  GSList *it;
  GSList *group = NULL;

  trace ("");
  g_return_if_fail (optmenu);
  g_return_if_fail (GTK_IS_OPTION_MENU (optmenu));
  g_return_if_fail (menu);
  g_return_if_fail (GTK_IS_MENU (menu));
  g_return_if_fail (format);
  g_return_if_fail (format->strings);

  for (it=format->strings; it; it=it->next)
  {
    GtkWidget *item;
    const gchar *label = LIST_DEREF (const gchar, it);

    item = gtk_radio_menu_item_new_with_label (group, label);
    group = gtk_radio_menu_item_group (GTK_RADIO_MENU_ITEM (item));

    gtk_widget_show (item);
    gtk_menu_append (menu, item);

    gtk_signal_connect (GTK_OBJECT (item), "activate",
    			GTK_SIGNAL_FUNC (format_select), format);
  }

  gtk_option_menu_set_menu (optmenu, GTK_WIDGET (menu));
  if (format->selection >= 0)
    gtk_option_menu_set_history (optmenu, format->selection);
}

static GtkWidget *
make_format_selector (FileFormatSelection *format)
{
  GtkWidget *hbox, *vbox, *label;
  GtkWidget *optmenu, *menu;
  GtkWidget *frame;

  trace ("");
  g_return_val_if_fail (format, NULL);
  g_return_val_if_fail (format->strings, NULL);

  frame = gtk_frame_new (_("Options"));
  vbox = gtk_vbox_new (0, 0);
  hbox = gtk_hbox_new (0, GNOME_PAD);
  label = gtk_label_new (format->label);
  optmenu = gtk_option_menu_new ();
  menu = gtk_menu_new ();

  fill_format_selector (GTK_OPTION_MENU (optmenu), GTK_MENU (menu), format);

  gtk_container_add (GTK_CONTAINER (frame), vbox);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, GNOME_PAD_SMALL);
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, GNOME_PAD);
  gtk_box_pack_start (GTK_BOX (hbox), optmenu, TRUE, TRUE, GNOME_PAD);
  gtk_widget_show_all (frame);

  return frame;
}

gchar *
dialog_query_filename (GtkWindow *parent, const gchar *title,
		       const gchar *defaultname, gboolean protect,
		       FileFormatSelection *format)
{
  FileSelectionInfo info = {0};
  GtkFileSelection *filesel;
  GtkWidget *formatsel;
  gchar *result;

  trace ("");
  g_return_val_if_fail (title, NULL);

  filesel = GTK_FILE_SELECTION (gtk_file_selection_new (title));
  gtk_file_selection_hide_fileop_buttons (filesel);

  if (format)
  {
    formatsel = make_format_selector (format);
    gtk_box_pack_start (GTK_BOX (filesel->action_area), formatsel, FALSE, TRUE, 0);
  }

  /* connect signals for ok and cancel buttons */
  gtk_signal_connect (GTK_OBJECT (filesel), "delete_event",
		      GTK_SIGNAL_FUNC(gtk_main_quit), NULL);
  gtk_signal_connect (GTK_OBJECT (filesel->ok_button), "clicked",
		      GTK_SIGNAL_FUNC(filesel_ok), &info);
  gtk_signal_connect (GTK_OBJECT (filesel->cancel_button), "clicked",
		      GTK_SIGNAL_FUNC(gtk_main_quit), NULL);

  /* initialize */
  info.filesel = filesel;
  info.protect = protect;

  if (defaultname && (strlen (defaultname) > 0))
    gtk_file_selection_set_filename (filesel, defaultname);
  
  gtk_window_set_modal (GTK_WINDOW (filesel), TRUE);

  if (parent)
  {
    gtk_window_set_transient_for (GTK_WINDOW (filesel), parent);

    if (gnome_preferences_get_dialog_centered ())
    {
      /* Code copied straight out of gnome_dialog_set_parent */
      gint x, y, w, h, dialog_x, dialog_y;

      if (!GTK_WIDGET_VISIBLE (parent))
      	return NULL; /* Can't get its size/pos */
      
      /* Throw out other positioning */
      gtk_window_set_position (GTK_WINDOW (filesel), GTK_WIN_POS_NONE);

      gdk_window_get_origin (GTK_WIDGET (parent)->window, &x, &y);
      gdk_window_get_size (GTK_WIDGET (parent)->window, &w, &h);
      
      /* Just a "guess" at what centered means */
      dialog_x = x + w/4;
      dialog_y = y + h/4;

      gtk_widget_set_uposition (GTK_WIDGET (filesel), dialog_x, dialog_y);
    }
  }

  gtk_widget_show (GTK_WIDGET (filesel));
  gtk_grab_add (GTK_WIDGET (filesel));
  gtk_main ();

  trace ("");

  if (info.accepted)
  {
    gchar *name = gtk_file_selection_get_filename (filesel);

    if (name [strlen (name) - 1] == '/')
      result = NULL;  /* selection is a directory */
    else
      result = g_strdup (name);
  }
  else
    result = NULL;

  gtk_widget_destroy (GTK_WIDGET (filesel));

  return result;
}
