/******************************************************************************\
 gnofin/bankbook.h   $Revision: 1.6 $
 Copyright (C) 1999-2000 Darin Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

#ifndef BANKBOOK_H
#define BANKBOOK_H

#include <gtk/gtkobject.h>
#include "data-types.h"
#include "data-events.h"
#include "cached-string.h"
#include "history.h"


/* Standard Gtk+ object-type cast macros
 */
#define BANKBOOK(obj)            (GTK_CHECK_CAST ((obj), bankbook_get_type (), Bankbook))
#define BANKBOOK_CLASS(klass)    (GTK_CHECK_CLASS_CAST ((klass), bankbook_get_type (), BankbookClass))
#define IS_BANKBOOK(obj)         (GTK_CHECK_TYPE ((obj), bankbook_get_type ())
#define IS_BANKBOOK_CLASS(klass) (GTK_CHECK_CLASS_TYPE ((klass), bankbook_get_type ())

/* Public readonly helper macros
 */
#define BANKBOOK_RECORD_TYPES(b)           ((b)->record_types)
#define BANKBOOK_ACCOUNTS(b)               ((b)->accounts)
#define BANKBOOK_CAN_UNDO(b)               ((gboolean)((b)->history.can_undo == 1))
#define BANKBOOK_CAN_REDO(b)               ((gboolean)((b)->history.can_redo == 1))
#define BANKBOOK_IS_DIRTY(b)               ((gboolean)((b)->dirty != 0))
#define BANKBOOK_IS_REMEMBERING_EVENTS(b)  ((gboolean)((b)->remember_events == 1))


/* Gnofin accounts data structure 
 *
 * In summary, each Gnofin data object may contain several
 * accounts, which may each contain several transaction records.  
 * Each transaction record has a "type" which defines certain
 * rules associated with it.  For example, is the record 
 * numbered?  Or, does the record correspond to a transfer
 * between two accounts?  In the latter case, we say the
 * record has a type that is "linked."  The data structure
 * does not place any limits on how the various type attributes
 * may be mixed; however, each type must have a unique name.
 *
 * A Bankbook object is the root of the accounts tree, with
 * a list of all accounts and a list of all record types.  It
 * also contains a cache of all payee, category, and memo strings
 * used.
 */

struct _Bankbook
{
  GtkObject    object;

  /* private */
  GList       *record_types;
  StringCache  payee_cache;
  StringCache  memo_cache;
  StringCache  category_cache;
  StringCache  link_cache;  /* cache of linked account names (b/c some may not be Gnofin accounts) */
  GList       *accounts;
  History      history;
  const gchar *lasterr;
  guint16      dirty;
  guint        remember_events : 1;
};

typedef struct
{
  GtkObjectClass parent_class;

  /* The index parameter indicates the position of the new/deleted account
   * or record.  If deleted, the actual account/record object no longer exists.
   */
  void (* account_inserted)                (const Bankbook *, guint index);
  void (* account_deleted)                 (const Bankbook *, guint index);

  void (* record_inserted)                 (const Bankbook *, const Account *, guint index);
  void (* record_deleted)                  (const Bankbook *, const Account *, guint index);

  /* When account/record info changes, its position may be changed.  the
   * last_index parameter indicates the previous position of the account/record
   */
  void (* account_info_changed)            (const Bankbook *, const Account *, guint mask, guint last_index);
  void (* record_info_changed)             (const Bankbook *, const Record *, guint mask, guint last_index);

  void (* record_list_invalidated)         (const Bankbook *, const Account *);
  void (* type_list_invalidated)           (const Bankbook *);
  void (* category_list_invalidated)       (const Bankbook *);
  void (* payee_list_invalidated)          (const Bankbook *);
  void (* memo_list_invalidated)           (const Bankbook *);
  void (* linked_account_list_invalidated) (const Bankbook *);

  void (* balance_changed)                 (const Bankbook *, const Account *);
  void (* history_changed)                 (const Bankbook *);
  void (* dirty_flag_changed)              (const Bankbook *);

} BankbookClass;


GtkType      bankbook_get_type  (void);
GtkObject   *bankbook_new       (void);
void         bankbook_set_error (Bankbook *book, char *fmt, ... );
const gchar *bankbook_get_error (Bankbook *book);


#endif
