/* gnobog_arborescence.c
 *
 * Copyright (C) 2000 Frdric LESPEZ  & Renaud CHAILLAT
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <gtk/gtk.h>
#include "gnobog_arborescence.h"
#include "gnobog_dnd.h"
#include "gnobog_pixmaps.h"
#include "gnobog_app.h"

/* to call clist class virtual functions default implementation */
#define GTK_CLIST_CLASS_FW(_widget_)   GTK_CLIST_CLASS (((GtkObject*) (_widget_))->klass)

/* Signals indices */
enum {
  POPUP_MENU,
  OPEN_ROW,
  LAST_SIGNAL
};

/* Signals array */
static guint    gnobog_arborescence_signals[LAST_SIGNAL];

/* Arguments */
enum {
  ARG_0,
  ARG_DND_TIMER,
  ARG_DND_THRESHOLD,
  ARG_SELECTION_MODE
};

/* Parent class pointer */
static GtkCTreeClass *parent_class = NULL;

/* Array containing Target Types for drag&drop.
 * Define the types of data our program can accept as a target
 * Order is important, the source will send the data with the
 * 1st format it knows in this list 
 */
const GtkTargetEntry dnd_target_types_array [] = {
  { GNOBOG_BOOKMARK_ENTRY_ATOM_NAME, GTK_TARGET_SAME_APP, GNOBOG_BOOKMARK_ENTRY},
  { STRING_ATOM_NAME, 0, STRING},
  { MOZILLA_URL_ATOM_NAME, 0, MOZILLA_URL},
  { TEXT_URI_LIST_ATOM_NAME, 0, TEXT_URI_LIST}
};

/* Number of known targets for drag&drop */
const gint dnd_n_target_types = sizeof (dnd_target_types_array) /
                                sizeof(dnd_target_types_array[0]);

/* A popup menu for dnd with middle button or shift+control */
static GnomeUIInfo dnd_actions[] = {
  GNOMEUIINFO_ITEM_NONE (N_("_Move here"), NULL, NULL),
  GNOMEUIINFO_ITEM_NONE (N_("_Copy here"), NULL, NULL),
  GNOMEUIINFO_ITEM_NONE (N_("_Make Alias here"), NULL, NULL),
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_ITEM_NONE (N_("Cancel drag"), NULL, NULL),
  GNOMEUIINFO_END
};


/*********************
 * Private methods
 *********************/

/* Standard gtk+ object system functions */
static void                       gnobog_arborescence_class_init 
                                                     (GnobogArborescenceClass* class);
static void                       gnobog_arborescence_init 
                                                     (GnobogArborescence* arborescence);
static void                       gnobog_arborescence_set_arg 
                                                     (GtkObject* object, 
						      GtkArg* arg, 
						      guint arg_id);
static void                       gnobog_arborescence_get_arg 
                                                     (GtkObject* object, 
						      GtkArg* arg, 
						      guint arg_id);
static void                       gnobog_arborescence_destroy 
                                                     (GtkObject* object);

/* Our own selection mechanisms. */
/* Motion decides on when initiating a drag.
   We don't use gtk_drag_source_set which breaks extended selection.
   Caveat : how will we take gtk user's settings into account ? (drag threshold for instance) */
static gboolean                   gnobog_arborescence_button_press        
                                                     (GtkWidget* widget, 
						      GdkEventButton* event);
static gboolean                   gnobog_arborescence_button_release      
                                                     (GtkWidget* widget, 
						      GdkEventButton* event);
static gboolean                   gnobog_arborescence_motion_notify       
                                                     (GtkWidget* widget, 
						      GdkEventMotion* event);

/* Default dnd behaviour (override clist and ctree default handlers). */
static gboolean                   gnobog_arborescence_scrolling_is_desirable 
                                                     (GnobogArborescence* arborescence, 
						      gint y);
static gboolean                   gnobog_arborescence_drag_scroll 
                                                     (gpointer data);
static void                       gnobog_arborescence_setup_drag_scroll 
                                                     (GnobogArborescence* arborescence, 
						      gint y);
static GdkDragAction              gnobog_arborescence_drag_get_action 
                                                     (GdkDragContext* context);
static GdkDragAction              gnobog_arborescence_drag_get_action 
                                                     (GdkDragContext* context);
static GnobogBookmarksInsertMode  gnobog_arborescence_insert_mode_from_drag_pos 
                                                     (GtkCListDragPos insert_pos);
static GtkCListDestInfo           gnobog_arborescence_drag_dest_cell 
                                                     (GnobogArborescence* arborescence, 
						      gint y);
static void                       gnobog_arborescence_cancel_drag_scroll 
                                                     (GnobogArborescence* arborescence);
static void                       gnobog_arborescence_drag_begin                
                                                     (GtkWidget* widget, 
						      GdkDragContext* context);
static void                       gnobog_arborescence_drag_end          
                                                     (GtkWidget* widget, 
						      GdkDragContext* context);
static void                       gnobog_arborescence_drag_data_get     
                                                     (GtkWidget* widget, 
						      GdkDragContext* context, 
						      GtkSelectionData* selection_data, 
						      guint info, 
						      guint time);
static void                       gnobog_arborescence_drag_data_delete     
                                                     (GtkWidget* widget, 
						      GdkDragContext* context); 
static void                       gnobog_arborescence_drag_leave          
                                                     (GtkWidget* widget, 
						      GdkDragContext* context, 
						      guint time);
static gboolean                   gnobog_arborescence_drag_motion         
                                                     (GtkWidget* widget, 
						      GdkDragContext* context, 
						      gint x, 
						      gint y, 
						      guint time);
static gboolean                   gnobog_arborescence_drag_drop           
                                                     (GtkWidget* widget, 
						      GdkDragContext* context, 
						      gint x, 
						      gint y, 
						      guint time);
static void                       gnobog_arborescence_drag_data_received  
                                                     (GtkWidget* widget, 
						      GdkDragContext* context, 
						      gint x, 
						      gint y, 
						      GtkSelectionData* data, 
						      guint info, 
						      guint time);
//static void                     gnobog_arborescence_drag_data_delete    
//                                      (GtkWidget* widget, 
//                                       GdkDragContext* context, 
//                                       gpointer user_data);

/* Delete all... */
//static void                     gnobog_arborescence_clear 
//                                      (GtkCList* clist);

/* Functions to perform selections */
static void                       gnobog_arborescence_select_range        
                                                     (GnobogArborescence* arborescence, 
						      int row1, 
						      int row2);
static void                       gnobog_arborescence_unselect_range      
                                                     (GnobogArborescence* arborescence, 
						      int row1, 
						      int row2);
static void                       gnobog_arborescence_handle_selection            
                                                     (GnobogArborescence* arborescence, 
						      int row, 
						      guint state);
static gint                       gnobog_arborescence_compare_rows        
                                                     (gconstpointer a, 
						      gconstpointer b);

/* Node edition */
static void                       gnobog_arborescence_node_insert 
                                                     (GnobogArborescence* arborescence, 
						      GnobogBookmarksNode anchor_node, 
						      GnobogBookmarksInsertMode mode, 
						      GnobogBookmarksNode bookmarks_node);
static void                       gnobog_arborescence_node_insert_into 
                                                     (GnobogArborescence* arborescence, 
						      GnobogBookmarksNode anchor_node, 
						      GnobogBookmarksNode bookmarks_node);
static void                       gnobog_arborescence_node_insert_after 
                                                     (GnobogArborescence* arborescence, 
						      GnobogBookmarksNode anchor_node, 
						      GnobogBookmarksNode bookmarks_node);
static void                       gnobog_arborescence_node_insert_before 
                                                     (GnobogArborescence* arborescence, 
						      GnobogBookmarksNode anchor_node, 
						      GnobogBookmarksNode bookmarks_node);
static void                       gnobog_arborescence_node_delete 
                                                     (GnobogArborescence* arborescence, 
						      GnobogBookmarksNode bookmarks_node);
static void                       gnobog_arborescence_node_modify 
                                                     (GnobogArborescence* arborescence, 
						      GnobogBookmarksNode bookmarks_node);

/* Misc */
static void                       gnobog_arborescence_open_row            
                                                     (GnobogArborescence* arborescence, 
						      gint row);
static void                       gnobog_arborescence_draw_drag_highlight 
                                                     (GnobogArborescence* arborescence, 
						      gint row, 
						      GtkCListDragPos drag_pos);


/*****************************************************************************/
/*** END OF DECLARATIONS                                                   ***/
/*****************************************************************************/

/*---------------------------------------------------------------------------*/









/******************************************************************************/
/*      Standard gtk+ object system functions                                 */
/******************************************************************************/
/*
 * gnobog_arborescence_get_type
 * gnobog_arborescence_class_init
 * gnobog_arborescence_init
 * gnobog_arborescence_new_with_titles
 * gnobog_arborescence_set_arg
 * gnobog_arborescence_get_arg
 * gnobog_arborescence_destroy
 */

/* Creates the GtkArborescence class and its type information.
 * @return The type ID for GtkArborescenceClass 
 */
GtkType
gnobog_arborescence_get_type (void)
{
  static GtkType gnobog_arborescence_type = 0; /* or guint ? */

  if ( !gnobog_arborescence_type ) {
    GtkTypeInfo gnobog_arborescence_info = {
      "GnobogArborescence",
      sizeof (GnobogArborescence),
      sizeof (GnobogArborescenceClass),
      (GtkClassInitFunc)  gnobog_arborescence_class_init,
      (GtkObjectInitFunc) gnobog_arborescence_init,
      NULL, 
      NULL, 
      (GtkClassInitFunc) NULL
    };
    gnobog_arborescence_type = gtk_type_unique (gtk_ctree_get_type (), 
                                                &gnobog_arborescence_info);
  }

  return gnobog_arborescence_type;
}

/* Arborescence class initialization function.
 * @param class A pointer to the class structure to initialize. 
 */
static void
gnobog_arborescence_class_init (GnobogArborescenceClass* klass)
{
  GtkObjectClass*  object_class;
  GtkWidgetClass*  widget_class;
  GtkCListClass*   clist_class;
  GtkCTreeClass*   ctree_class;

  object_class =  (GtkObjectClass *) klass;
  widget_class =  (GtkWidgetClass *) klass;
  clist_class  =  (GtkCListClass *)  klass;
  ctree_class  =  (GtkCTreeClass *)  klass;

  parent_class = gtk_type_class (gtk_ctree_get_type ());

  gtk_object_add_arg_type ("GnobogArborescence::dnd_timer",
                           GTK_TYPE_UINT,
                           GTK_ARG_READWRITE,
                           ARG_DND_TIMER);
  gtk_object_add_arg_type ("GnobogArborescence::dnd_threshold",
                           GTK_TYPE_INT,
                           GTK_ARG_READWRITE,
                           ARG_DND_THRESHOLD);
  gtk_object_add_arg_type ("GnobogArborescence::selection_mode",
                           GTK_TYPE_INT,
                           GTK_ARG_READWRITE,
                           ARG_SELECTION_MODE);

  gnobog_arborescence_signals[POPUP_MENU] =
    gtk_signal_new ("popup_menu",
                    GTK_RUN_FIRST,
                    object_class->type,
                    GTK_SIGNAL_OFFSET (GnobogArborescenceClass, popup_menu),
                    gtk_marshal_NONE__POINTER,
                    GTK_TYPE_NONE, 1,
                    GTK_TYPE_GDK_EVENT);
  gnobog_arborescence_signals[OPEN_ROW] =
    gtk_signal_new ("open_row",
                    GTK_RUN_FIRST,
                    object_class->type,
                    GTK_SIGNAL_OFFSET (GnobogArborescenceClass, open_row),
                    gtk_marshal_NONE__INT,
                    GTK_TYPE_NONE, 1,
                    GTK_TYPE_INT);

  gtk_object_class_add_signals (object_class, gnobog_arborescence_signals, LAST_SIGNAL);

  object_class->set_arg = gnobog_arborescence_set_arg;
  object_class->get_arg = gnobog_arborescence_get_arg;
  object_class->destroy = gnobog_arborescence_destroy;

  /* Our own event handlers */
  widget_class->button_press_event        = gnobog_arborescence_button_press;
  widget_class->button_release_event      = gnobog_arborescence_button_release;
  widget_class->motion_notify_event       = gnobog_arborescence_motion_notify;
        
  /* default dnd behaviour */
  widget_class->drag_begin                = gnobog_arborescence_drag_begin;
  widget_class->drag_end                  = gnobog_arborescence_drag_end;
  widget_class->drag_data_get             = gnobog_arborescence_drag_data_get;
  widget_class->drag_data_delete          = gnobog_arborescence_drag_data_delete;
  widget_class->drag_leave                = gnobog_arborescence_drag_leave;
  widget_class->drag_motion               = gnobog_arborescence_drag_motion;
  widget_class->drag_drop                 = gnobog_arborescence_drag_drop;
  widget_class->drag_data_received        = gnobog_arborescence_drag_data_received;
        
  klass->open_row   = gnobog_arborescence_open_row;
  klass->popup_menu = NULL;
}


/* Arborescence object instance initialization. */
static void
gnobog_arborescence_init (GnobogArborescence* arborescence)
{
  arborescence->row_was_selected      = FALSE;
  arborescence->anchor_row            = -1;
  arborescence->previous_row          = -1;
  arborescence->timer_expired         = FALSE;
  arborescence->in_extended_selection = FALSE;
  arborescence->perform_drag          = FALSE;
  arborescence->dest_info.cell.row    = -1;
  arborescence->dest_info.insert_pos  = GTK_CLIST_DRAG_NONE;
  arborescence->drag_scroll_timer_id  = -1;
        
  gnobog_arborescence_set_defaults (arborescence);
}


/* Arborescence constructor. */
GtkWidget *
gnobog_arborescence_new_with_titles (GnobogBookmarks* bookmarks_watched,
                                     int columns, 
                                     char **titles)
{
  GnobogArborescence*  arborescence;
  GtkWidget*           widget;
  gint                 i;
  gint                 titles_num;
  char**               i18n_titles;

  g_return_val_if_fail (bookmarks_watched != NULL, NULL);
  g_return_val_if_fail (columns > 0, NULL);
  g_return_val_if_fail (*titles != NULL, NULL);

  titles_num = sizeof (*titles) / sizeof (*titles[0]);
  g_return_val_if_fail (titles_num != columns, NULL);

  /* i18n stuff */
  i18n_titles = g_new (char*, titles_num);
  for (i=0; i<titles_num; i++) {
    i18n_titles[i] = g_strdup_printf ("%s", _(titles[i]));
  }

  arborescence = gtk_type_new (gnobog_arborescence_get_type ());
  gtk_ctree_construct (GTK_CTREE (arborescence), columns, 0 /* tree column */, i18n_titles);

  for (i=0; i<titles_num; i++) {
    g_free (i18n_titles[i]);
  }
  g_free (i18n_titles);

  arborescence->columns                 = columns;
  arborescence->visible_columns_count   = columns;
  arborescence->titles_visibility       = TRUE;
  arborescence->columns_visibility      = g_new (gboolean, columns);

  for (i=0; i<columns; i++) {
    arborescence->columns_visibility[i] = TRUE;
    /* try to optimize colums width */
    //      gtk_clist_optimal_column_width  (GTK_CLIST (arborescence), i); 
    gtk_clist_set_column_width  (GTK_CLIST (arborescence), 
				 i,
				 150); 
  }

  arborescence->bookmarks_watched       = bookmarks_watched;

  /* Don't forget to add motion-notify event handling */
  widget = GTK_WIDGET (arborescence);
  gtk_widget_add_events (widget, GDK_BUTTON1_MOTION_MASK | GDK_BUTTON2_MOTION_MASK);

  /* Set selection mode */
  gtk_clist_set_selection_mode (GTK_CLIST (arborescence), GTK_SELECTION_EXTENDED);

  /* Connect handlers to bookmarks signals */
  gtk_signal_connect_object_while_alive (GTK_OBJECT (bookmarks_watched), 
                                         "list_created", 
                                         (GtkSignalFunc) gnobog_arborescence_list_insert, 
                                         GTK_OBJECT (arborescence));
  gtk_signal_connect_object_while_alive (GTK_OBJECT (bookmarks_watched), 
                                         "list_deleted", 
                                         (GtkSignalFunc) gnobog_arborescence_list_delete, 
                                         GTK_OBJECT (arborescence));
  gtk_signal_connect_object_while_alive (GTK_OBJECT (bookmarks_watched), 
                                         "list_modified", 
                                         (GtkSignalFunc) gnobog_arborescence_list_modify, 
                                         GTK_OBJECT (arborescence));
  gtk_signal_connect_object_while_alive (GTK_OBJECT (bookmarks_watched), 
                                         "bookmarks_cleared", 
                                         (GtkSignalFunc) gnobog_arborescence_clear, 
                                         GTK_OBJECT (arborescence));

  /* Activate target dnd */
  gtk_drag_dest_set (widget,
                     GTK_DEST_DEFAULT_MOTION |
                                /* GTK_DEST_DEFAULT_HIGHLIGHT | */
                     GTK_DEST_DEFAULT_DROP,
                     dnd_target_types_array,
                     dnd_n_target_types,
                     GDK_ACTION_MOVE | GDK_ACTION_COPY | GDK_ACTION_ASK);

  /* default reorderable dnd mode */
  //      gtk_ctree_set_reorderable (GTK_CTREE(arborescence), TRUE);
  //    GTK_CLIST_SET_FLAG (GTK_CLIST(arborescence), CLIST_DRAW_DRAG_RECT);

  return widget;
}

static void             
gnobog_arborescence_destroy (GtkObject* object)
{
  GnobogArborescence*  arborescence;
        
  g_return_if_fail (object != NULL);
  g_return_if_fail (GNOBOG_IS_ARBORESCENCE (object));
        
  arborescence = GNOBOG_ARBORESCENCE (object);
        
  g_free (arborescence->columns_visibility);
}



static void     
gnobog_arborescence_set_arg (GtkObject* object, 
                             GtkArg* arg, 
                             guint arg_id)
{
  GnobogArborescence  *arborescence;
        
  g_return_if_fail (object != NULL);
  g_return_if_fail (GNOBOG_IS_ARBORESCENCE(object));
        
  arborescence = GNOBOG_ARBORESCENCE (object);
        
  switch (arg_id) {
  case ARG_DND_TIMER:
    arborescence->dnd_timer = GTK_VALUE_UINT (*arg);
    break;
  case ARG_DND_THRESHOLD:
    if (GTK_VALUE_INT (*arg) < 0) break;
    arborescence->dnd_threshold = GTK_VALUE_INT (*arg);
    break;
  case ARG_SELECTION_MODE:
    if (GTK_VALUE_INT (*arg) < 0) break;
    arborescence->selection_mode = GTK_VALUE_INT (*arg);
    break;
  default:
    break;
  }
}

static void     
gnobog_arborescence_get_arg (GtkObject* object, 
                             GtkArg* arg, 
                             guint arg_id)
{
  GnobogArborescence  *arborescence;
        
  g_return_if_fail (object != NULL);
  g_return_if_fail (GNOBOG_IS_ARBORESCENCE(object));
        
  arborescence = GNOBOG_ARBORESCENCE (object);
        
  switch (arg_id) {
  case ARG_DND_TIMER:
    GTK_VALUE_UINT (*arg) = arborescence->dnd_timer;
    break;
  case ARG_DND_THRESHOLD:
    GTK_VALUE_INT (*arg) = arborescence->dnd_threshold;
    break;
  case ARG_SELECTION_MODE:
    GTK_VALUE_INT (*arg) = arborescence->selection_mode;
    break;
  default:
    break;
  }

}













/******************************************************************************/
/*      Settings                                                              */
/******************************************************************************/
/*
 * gnobog_arborescence_set_defaults
 * gnobog_arborescence_set_dnd_timer
 * gnobog_arborescence_get_dnd_timer
 * gnobog_arborescence_set_dnd_threshold
 * gnobog_arborescence_get_dnd_threshold
 * gnobog_arborescence_set_selection_mode
 * gnobog_arborescence_get_selection_mode
 */
void
gnobog_arborescence_set_defaults (GnobogArborescence* arborescence)
{
  gtk_object_set (GTK_OBJECT(arborescence),
                  "dnd_timer", 100,
                  "dnd_threshold", 3,
                  "selection_mode", DND_TIMER_ONLY_MODE,
                  NULL);
}

void
gnobog_arborescence_set_dnd_timer (GnobogArborescence* arborescence, 
                                   guint32 dnd_timer)
{
  gtk_object_set (GTK_OBJECT(arborescence),
                  "dnd_timer", dnd_timer,
                  NULL);
}

guint32
gnobog_arborescence_get_dnd_timer (GnobogArborescence* arborescence)
{
  g_return_val_if_fail (arborescence != NULL, 0);
        
  return arborescence->dnd_timer;
}

void
gnobog_arborescence_set_dnd_threshold (GnobogArborescence* arborescence, 
                                       gint dnd_threshold)
{
  gtk_object_set (GTK_OBJECT(arborescence),
                  "dnd_threshold", dnd_threshold,
                  NULL);
}

gint
gnobog_arborescence_get_dnd_threshold (GnobogArborescence* arborescence)
{
  g_return_val_if_fail (arborescence != NULL, -1);
        
  return arborescence->dnd_threshold;
}

void
gnobog_arborescence_set_selection_mode (GnobogArborescence* arborescence, 
                                        gint selection_mode)
{
  gtk_object_set (GTK_OBJECT(arborescence),
                  "selection_mode", selection_mode,
                  NULL);
}

gint
gnobog_arborescence_get_selection_mode (GnobogArborescence* arborescence)
{
  g_return_val_if_fail (arborescence != NULL, -1);
        
  return arborescence->selection_mode;
}

void
gnobog_arborescence_set_titles_visibility (GnobogArborescence* arborescence,
					   gboolean visibility)
{
  g_return_if_fail (arborescence != NULL);
  g_return_if_fail (GNOBOG_IS_ARBORESCENCE (arborescence));

  arborescence->titles_visibility = visibility;
                
  if ( arborescence->titles_visibility == TRUE ) {
    gtk_clist_column_titles_show (GTK_CLIST (arborescence));
  } else {
    gtk_clist_column_titles_hide (GTK_CLIST (arborescence));
  }
  return;
}

gboolean
gnobog_arborescence_get_titles_visibility (GnobogArborescence* arborescence)
{
  g_return_val_if_fail (arborescence != NULL, FALSE);
        
  return arborescence->titles_visibility;
}

gboolean
gnobog_arborescence_get_column_visibility (GnobogArborescence* arborescence, 
					   gint column)
{
  g_return_val_if_fail (arborescence != NULL, FALSE);
  g_return_val_if_fail (GNOBOG_IS_ARBORESCENCE (arborescence), FALSE);
  g_return_val_if_fail (column > 0, FALSE);
  g_return_val_if_fail (column <= arborescence->columns, FALSE);

  return arborescence->columns_visibility[column-1];
}

void
gnobog_arborescence_set_column_visibility (GnobogArborescence* arborescence, 
					   gint column,
					   gboolean visibility)
{
  //  gint i, width, height;

  g_return_if_fail (arborescence != NULL);
  g_return_if_fail (GNOBOG_IS_ARBORESCENCE (arborescence));
  g_return_if_fail (column > 0);
  g_return_if_fail (column <= arborescence->columns);

  if ( TRUE == visibility  && 
       arborescence->columns_visibility[column - 1] == FALSE ) {
    g_message ("should show column %d", column);
    gtk_clist_set_column_visibility (GTK_CLIST (arborescence), column-1, TRUE);
    arborescence->visible_columns_count++;
    arborescence->columns_visibility[column - 1] = TRUE;
  } else if ( FALSE == visibility &&
	      arborescence->columns_visibility[column - 1] == TRUE ) {
    if ( arborescence->visible_columns_count > 1 ) { /* don't hide all columns */
      g_message ("should hide column %d", column);
      arborescence->visible_columns_count--;
      arborescence->columns_visibility[column - 1] = FALSE;
      gtk_clist_set_column_visibility (GTK_CLIST (arborescence), column-1, FALSE);
    }
  }

  /* try to optimize colums width */
  /* test if arborescence has been mapped before querying window's size */
  //  if ( ! GTK_WIDGET_MAPPED (GTK_WIDGET (arborescence)) ) return;

  //  gdk_window_get_size (GTK_WIDGET (arborescence)->window,
  //		       &width,
  //		       &height);
  //  g_message ("Wanted width: %d\n", width/arborescence->visible_columns_count);
  //  for (i=0; i<arborescence->columns; i++) {
  //    if ( arborescence->columns_visibility[i - 1] == TRUE ) {
      //      gtk_clist_optimal_column_width  (GTK_CLIST (arborescence), i); 
  //      gtk_clist_set_column_width  (GTK_CLIST (arborescence), 
  //				   i,
  //				   width/arborescence->visible_columns_count); 
  //    }
  //}
}

















/******************************************************************************/
/*      'EVENT' SIGNALS                                                       */
/******************************************************************************/
/*
 * gnobog_arborescence_button_press
 * gnobog_arborescence_button_release
 * gnobog_arborescence_motion_notify
 */


/* Our handler for 'button-press-event' signal. */
static gboolean
gnobog_arborescence_button_press (GtkWidget* widget, 
                                  GdkEventButton* event)
{
  GnobogArborescence*     arborescence;
  GnobogArborescenceNode  arborescence_node;
  GtkCList*               clist;
  gint                    cursor_row;
  gint                    x, y;
  gboolean                normal, shift, control, shift_control;
  gboolean                windows_mode, dnd_timer_only_mode, rino_mode;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GNOBOG_IS_ARBORESCENCE (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  arborescence  = GNOBOG_ARBORESCENCE (widget);
  clist         = GTK_CLIST (widget);

  /* CHECK : Can we receive this event if it does not occur in our window */
  if ( event->window != clist->clist_window ) {
    return (* GTK_WIDGET_CLASS (parent_class)->button_press_event) (widget, event);
  }

  normal        = (event->state & GDK_SHIFT_MASK) == 0 &&
                  (event->state & GDK_CONTROL_MASK) == 0;
  shift         = (event->state & GDK_SHIFT_MASK) != 0 &&
                  (event->state & GDK_CONTROL_MASK) == 0;
  control       = (event->state & GDK_SHIFT_MASK) == 0 &&
                  (event->state & GDK_CONTROL_MASK) != 0;
  shift_control = (event->state & GDK_SHIFT_MASK) != 0 &&
                  (event->state & GDK_CONTROL_MASK) != 0;

  windows_mode        = (arborescence->selection_mode == WINDOWS_MODE);
  dnd_timer_only_mode = (arborescence->selection_mode == DND_TIMER_ONLY_MODE);
  rino_mode           = (arborescence->selection_mode == RINO_MODE);

  /* Store coordinates */
  x = event->x;
  y = event->y;   

  /* Get the row and node */
  cursor_row        = gnobog_arborescence_get_row_from_cursor_coords (arborescence, 
								      event->x, 
								      event->y);
  arborescence_node = gnobog_arborescence_get_node_from_row (arborescence, 
							     cursor_row);

  /* React on single-click or double-click */
  switch (event->type) {

  case GDK_BUTTON_PRESS:     /* Single click */
    /* Which button ? */
    switch (event->button) {
    case 1:                  /* left click   */
      //      g_message ("ROW : %d\n", cursor_row);
      if ( cursor_row >= 0 ) {
        /* Save the mouse info (useful in motion handler) */
        arborescence->button_pressed        = event->button;
        arborescence->in_extended_selection = FALSE;
        arborescence->button_pressed_time   = event->time;
        arborescence->button_pressed_x      = x;
        arborescence->button_pressed_y      = y;
        /* Expand the node if we are on a 'hot spot' */
        if (gtk_ctree_is_hot_spot (GTK_CTREE (arborescence), x, y)) {
          gtk_ctree_toggle_expansion (GTK_CTREE (arborescence), arborescence_node);
        } else {
          /* Remember row state */
          if (gnobog_arborescence_is_row_selected (arborescence, cursor_row) ) {
            arborescence->row_was_selected = TRUE;
          } else {
            arborescence->row_was_selected = FALSE;
          }
          /* Unselect all if needed */
          if ( normal && (windows_mode || rino_mode)
               && !gnobog_arborescence_is_row_selected (arborescence, cursor_row) ) {
            gnobog_arborescence_unselect_all (arborescence);
          }
          /* Common to all selection modes */
          if ( normal ) {
            arborescence->anchor_row   = cursor_row;
            arborescence->previous_row = cursor_row;
            gnobog_arborescence_select_row (arborescence, cursor_row);
          }
          /* Shift Key */
          if ( windows_mode && shift ) {
            gnobog_arborescence_unselect_all (arborescence);
            if (arborescence->anchor_row >= 0) {
              gnobog_arborescence_select_range (arborescence, 
                                                arborescence->anchor_row, 
                                                cursor_row);
            } else {
              gnobog_arborescence_select_row (arborescence, cursor_row);
              arborescence->anchor_row = cursor_row;
            }
          }
          if ( windows_mode && shift_control && arborescence->anchor_row >= 0 ) {
            if ( gnobog_arborescence_is_row_selected (arborescence, 
                                                      arborescence->anchor_row) ) {
              gnobog_arborescence_select_range (arborescence, 
                                                arborescence->anchor_row, 
                                                cursor_row);
            } else {
              gnobog_arborescence_unselect_range
                (arborescence, arborescence->anchor_row, cursor_row);
              gnobog_arborescence_select_row (arborescence, cursor_row);
            }                                                               
          }
        }
      } else {
        /* If not on a row, unselect all */
        gtk_clist_unselect_all (clist);
	return FALSE;
      }
      return TRUE;
      //                      break;

    case 2: /* middle click */
      arborescence->button_pressed = event->button;
      if ( !gnobog_arborescence_is_row_selected (arborescence, cursor_row) ) {
        gnobog_arborescence_unselect_all (arborescence);
        gnobog_arborescence_select_row (arborescence, cursor_row);
      }
      break;

    case 3: /* right click, we could emit on release if we want to drag with it */
      arborescence->button_pressed = event->button;
      gtk_signal_emit (GTK_OBJECT (arborescence),
                       gnobog_arborescence_signals[POPUP_MENU],
                       event);
      return TRUE;
      break;

    default: /* wheel ? */
      break;
    }

    break; /* End of single click */
  
  case GDK_2BUTTON_PRESS:   /* Double-click on left button */
    if (event->button != 1)
      break;
    /* Emit 'open-row' signal */
    if (cursor_row >= 0)
      gtk_signal_emit (GTK_OBJECT (arborescence), 
                       gnobog_arborescence_signals[OPEN_ROW], 
                       cursor_row);
    return TRUE;
    break;

  default:
    break;
  }

  return FALSE;
}


/* Our handler for 'motion-notify-event' signal.
 * Here we decide to initiate dnd or not.  
 */
static gboolean
gnobog_arborescence_motion_notify (GtkWidget* widget, 
                                   GdkEventMotion* event)
{
  GnobogArborescence*     arborescence;
  GtkCList*               clist;
  gint                    row;
  gboolean                normal, shift, control, shift_control;
  gboolean                windows_mode, dnd_timer_only_mode, rino_mode;
  gboolean                dnd_threshold;
//GdkPixmap*              pixmap;
//GdkBitmap*              mask;
//GnobogArborescenceNode  arborescence_node;
        
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GNOBOG_IS_ARBORESCENCE (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);
        
  arborescence = GNOBOG_ARBORESCENCE (widget);
  clist        = GTK_CLIST (widget);

  /* CHECK : Can we receive this event if it does not occur in our window ?? */
  if ( event->window != clist->clist_window ) {
    return (* GTK_WIDGET_CLASS (parent_class)->motion_notify_event) (widget, event);
  }

  row = gnobog_arborescence_get_row_from_cursor_coords (arborescence, event->x, event->y);

  normal        = (event->state & GDK_SHIFT_MASK) == 0 &&
                  (event->state & GDK_CONTROL_MASK) == 0;
  shift         = (event->state & GDK_SHIFT_MASK) != 0 &&
                  (event->state & GDK_CONTROL_MASK) == 0;
  control       = (event->state & GDK_SHIFT_MASK) == 0 &&
                  (event->state & GDK_CONTROL_MASK) != 0;
  shift_control = (event->state & GDK_SHIFT_MASK) != 0 &&
                  (event->state & GDK_CONTROL_MASK) != 0;

  windows_mode        = (arborescence->selection_mode == WINDOWS_MODE);
  dnd_timer_only_mode = (arborescence->selection_mode == DND_TIMER_ONLY_MODE);
  rino_mode           = (arborescence->selection_mode == RINO_MODE);

  /* Do nothing (YET) if not on a row */
  if ( row <0 ) { 
    g_message ("Out from motion notify"); 
    return FALSE; 
  } else {
    g_message ("Motion notify, row: %d\n", row);
  }


  /* Move with left or middle button */
  /* If we didn't received a pressed signal in the same window,
   * we do nothing */
  if ( !(((1 == arborescence->button_pressed) && (event->state & GDK_BUTTON1_MASK)) ||  
         ((2 == arborescence->button_pressed) && (event->state & GDK_BUTTON2_MASK))) ) {
    g_message ("Motion notify doesn't handle this event");
    return FALSE;
  }

  /* Handle any pending selections or unselection */
  /* Extended selection in windows mode uses shift and not mouse motion */
  if ( !windows_mode && (arborescence->in_extended_selection) ) { 
    gnobog_arborescence_handle_selection (arborescence, row, event->state);
  }
  /* If we are performing extended selection/unselection we won't drag */
  if ( arborescence->in_extended_selection ) return TRUE;
        

  /* Decide whether initiating a drag or not. */

  /* First discriminating test : coordinate threshold */
  dnd_threshold = ( MAX ( abs (arborescence->button_pressed_x - event->x),
                          abs (arborescence->button_pressed_y - event->y) )
                    > arborescence->dnd_threshold );
  if ( !dnd_threshold && !dnd_timer_only_mode ) return TRUE;
  /* Then timing stuff and in_extended_selection */
  if ( !windows_mode && !arborescence->timer_expired ) {
    arborescence->timer_expired = TRUE;
    if ( (event->time - arborescence->button_pressed_time) <= arborescence->dnd_timer ) {
      arborescence->in_extended_selection = TRUE;
    }
  }
  /* So do we perform a drag_begin ? */
  if ( windows_mode ) {
    arborescence->perform_drag = TRUE;
    if ( control ) gnobog_arborescence_select_row (arborescence, row);
  } else if ( dnd_timer_only_mode ) {
    if ( !arborescence->in_extended_selection ) arborescence->perform_drag = TRUE;
  } else if ( rino_mode ) {
    if (arborescence->row_was_selected  || !arborescence->in_extended_selection)
      arborescence->perform_drag = TRUE;
  }
        
  /* Initiate the drag operation */
  if ( arborescence->perform_drag ) {
    GtkTargetList  *list;
    GdkDragContext *context;
    list = gtk_target_list_new (dnd_target_types_array, dnd_n_target_types);
    /* Set the dnd context on account of pressed key and button */
    if ( (event->state & GDK_BUTTON2_MASK) || shift_control ) {
      context = gtk_drag_begin (widget, 
                                list, 
                                GDK_ACTION_ASK ,
                                arborescence->button_pressed, 
                                (GdkEvent*) event);           
    } else if ( control ) {
      context = gtk_drag_begin (widget, 
                                list, 
                                GDK_ACTION_COPY ,
                                arborescence->button_pressed, 
                                (GdkEvent*) event);
    } else if ( shift ) {
      context = gtk_drag_begin (widget, 
                                list, 
                                GDK_ACTION_MOVE ,
                                arborescence->button_pressed, 
                                (GdkEvent*) event);             
    } else {
      g_message ("GNOBOG DRAG");
      context = gtk_drag_begin (widget, 
                                list, 
                                GDK_ACTION_COPY | GDK_ACTION_MOVE ,
                                arborescence->button_pressed, 
                                (GdkEvent*) event);             
    }
    /* set drag icon */
    /* This line commented out to avoid a bug with XFree 4 */
    /* gtk_drag_set_icon_default (context); */

    /* FIXME : Free GtkTargetList* list */

    //              arborescence_node = gnobog_arborescence_get_node_from_row (arborescence, row);
    //              gtk_ctree_node_get_pixmap (GTK_CTREE(arborescence),
    //                                                                      arborescence_node,
    //                                                                      1,
    //                                                                      &pixmap,
    //                                                                      &mask);
    //              gtk_drag_set_icon_pixmap (context,
    //                                                                      gdk_colormap_get_system (),
    //                                                                      pixmap,
    //                                                                      mask,
    //                                                                      0,
    //                                                                      0);
  }
  return TRUE;
}


/* Our handler for 'button-release-event' signal. */
static gboolean
gnobog_arborescence_button_release (GtkWidget* widget, 
                                    GdkEventButton* event)
{
  GnobogArborescence*  arborescence;
  GtkCList*            clist;
  gint                 row;
  gboolean             normal, shift, control, shift_control;
  gboolean             windows_mode, dnd_timer_only_mode, rino_mode;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GNOBOG_IS_ARBORESCENCE (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  arborescence = GNOBOG_ARBORESCENCE (widget);
  clist        = GTK_CLIST (widget);

  normal        = (event->state & GDK_SHIFT_MASK) == 0 &&
                  (event->state & GDK_CONTROL_MASK) == 0;
  shift         = (event->state & GDK_SHIFT_MASK) != 0 &&
                  (event->state & GDK_CONTROL_MASK) == 0;
  control       = (event->state & GDK_SHIFT_MASK) == 0 &&
                  (event->state & GDK_CONTROL_MASK) != 0;
  shift_control = (event->state & GDK_SHIFT_MASK) != 0 &&
                  (event->state & GDK_CONTROL_MASK) != 0;

  windows_mode        = (arborescence->selection_mode == WINDOWS_MODE);
  dnd_timer_only_mode = (arborescence->selection_mode == DND_TIMER_ONLY_MODE);
  rino_mode           = (arborescence->selection_mode == RINO_MODE);

  // CHECK : Can we receive this event if it does not occur in our window ??
  if ( event->window != clist->clist_window ) {
    return (* GTK_WIDGET_CLASS (parent_class)->button_release_event) (widget, event);
  }

  if ( !(event->button == 1 || event->button == 2) )
    return FALSE;

  if ( !arborescence->perform_drag ) {
    row = gnobog_arborescence_get_row_from_cursor_coords (arborescence, 
                                                          event->x, 
                                                          event->y);
    /* Normal : unselect row(s) in modes which require it */
    if ( normal && arborescence->row_was_selected ) {
      if ( dnd_timer_only_mode || rino_mode )
        gnobog_arborescence_unselect_row (arborescence, row);
      //                      if (compromise_mode)
      //                              gnobog_arborescence_unselect_all (arborescence);
      if ( windows_mode ) {
        gnobog_arborescence_unselect_all (arborescence);
        gnobog_arborescence_select_row (arborescence, row);
      }
    }
    /* if control key is used */
    if ( control && windows_mode ) {
      if ( arborescence->row_was_selected ) {
        gnobog_arborescence_unselect_row (arborescence, row);
      } else {
        gnobog_arborescence_select_row (arborescence, row);
      }
      /* In each case set anchor row ! */
      arborescence->anchor_row = row;
    }
  }

  /* Reset info */
  arborescence->row_was_selected      = FALSE;
  arborescence->timer_expired         = FALSE;
  arborescence->in_extended_selection = FALSE;
  arborescence->perform_drag          = FALSE;
  arborescence->button_pressed        = 0;

  return TRUE;
}
















/******************************************************************************/
/*      DND                                                                   */
/******************************************************************************/
/*
 * gnobog_arborescence_cancel_drag_scroll
 * gnobog_arborescence_setup_drag_scroll
 * gnobog_arborescence_scrolling_is_desirable
 * gnobog_arborescence_drag_scroll
 * gnobog_arborescence_drag_get_action
 * gnobog_arborescence_insert_mode_from_drag_pos
 * gnobog_arborescence_drag_dest_info_destroy
 * gnobog_arborescence_drag_dest_cell
 * gnobog_arborescence_drag_begin
 * gnobog_arborescence_drag_end
 * gnobog_arborescence_drag_data_delete
 * gnobog_arborescence_drag_data_get
 * gnobog_arborescence_drag_leave
 * gnobog_arborescence_drag_motion
 * gnobog_arborescence_drag_drop
 * gnobog_arborescence_drag_data_received
 */

/* adapted from gmc */

/* TODO: make that a customizable parameter */
#define SCROLL_TIMEOUT 100


/* we need to set auto-scrolling with a timer callback,
 * because if we scroll only in drag_motion, we won't scroll
 * when the mouse is idle on the last or first row */
static void
gnobog_arborescence_cancel_drag_scroll (GnobogArborescence* arborescence)
{
  g_return_if_fail (arborescence != NULL);
  g_return_if_fail (GNOBOG_IS_ARBORESCENCE (arborescence));
        
  if ( arborescence->drag_scroll_timer_id != -1 ) {
    gtk_timeout_remove (arborescence->drag_scroll_timer_id);
    arborescence->drag_scroll_timer_id = -1;
  }
}


static gboolean
gnobog_arborescence_scrolling_is_desirable (GnobogArborescence* arborescence, 
                                            gint y)
{
  GtkAdjustment*  va;
  GtkCList*       clist;

  g_return_val_if_fail (arborescence !=  NULL, FALSE);
  g_return_val_if_fail (GNOBOG_IS_ARBORESCENCE (arborescence), FALSE);
        
  clist = GTK_CLIST (arborescence);
        
  /* normalize coordinate */
  y -= (GTK_CONTAINER (clist)->border_width +
        GTK_WIDGET (arborescence)->style->klass->ythickness +
        clist->column_title_area.height);

        
  va = gtk_clist_get_vadjustment (clist);

  if ( y < 10 ) {
    if ( va->value > va->lower )
      return TRUE;
  } else {
    if ( y > (clist->clist_window_height - 10) ) {
      if ( va->value < va->upper - va->page_size )
        return TRUE;
    }
  }

  return FALSE;
}

/* timer callback for auto-scrolling */
static gboolean
gnobog_arborescence_drag_scroll (gpointer data /* arborescence passed as data for timer callback */)
{
  GtkAdjustment*       va;
  double               v;
  GtkCList*            clist;
  GnobogArborescence*  arborescence;
  GtkCListDestInfo*    dest_info;

  g_return_val_if_fail (data !=  NULL, TRUE);
  g_return_val_if_fail (GNOBOG_IS_ARBORESCENCE (data), TRUE);
        
  clist        = GTK_CLIST (data);
  arborescence = GNOBOG_ARBORESCENCE (data);

  dest_info    = &(arborescence->dest_info);
        
  /* erase drag line/rectangle */
  if ( dest_info->cell.row >= 0 ) {
    gnobog_arborescence_draw_drag_highlight (arborescence, 
                                             dest_info->cell.row, 
                                             dest_info->insert_pos);
    dest_info->cell.row = -1;
  }

  /* Get vertical adjustment */
  va = gtk_clist_get_vadjustment (clist);
  /* Change vertical adjustment to scroll */
  if ( arborescence->drag_motion_y < 10 ) {
    v = va->value - va->step_increment;
    /* speed up if we are closer to the edge !*/
    if ( arborescence->drag_motion_y < 5 )
      v = va->value - 4*(va->step_increment);
    if ( v < va->lower )
      v = va->lower;
    gtk_adjustment_set_value (va, v);
  } else {
    v = va->value + va->step_increment;
    /* speed up if we are closer to the edge !*/
    if ( arborescence->drag_motion_y > (clist->clist_window_height - 5) )
      v = va->value + 4*(va->step_increment);
    if ( v > (va->upper - va->page_size) )
      v = va->upper - va->page_size;
    gtk_adjustment_set_value (va, v);
  }

  return TRUE;
}

static void
gnobog_arborescence_setup_drag_scroll (GnobogArborescence* arborescence, 
                                       gint y)
{
  GtkCList*        clist;  

  g_return_if_fail (arborescence !=  NULL);
  g_return_if_fail (GNOBOG_IS_ARBORESCENCE (arborescence));
        
  clist = GTK_CLIST (arborescence);
        
  /* cancel to reevaluate */
  gnobog_arborescence_cancel_drag_scroll (arborescence);
        
  /* store normalized mouse position to use in scroll callback */
  arborescence->drag_motion_y = y - (GTK_CONTAINER (clist)->border_width +
                                     GTK_WIDGET (arborescence)->style->klass->ythickness +
                                     clist->column_title_area.height);

  if ( gnobog_arborescence_scrolling_is_desirable (arborescence, y) )
    arborescence->drag_scroll_timer_id = gtk_timeout_add (SCROLL_TIMEOUT, 
							  gnobog_arborescence_drag_scroll, 
							  arborescence);
}


/* taken from gmc */
static GdkDragAction
gnobog_arborescence_drag_get_action (GdkDragContext* context)
{
  GtkWidget*    menu;
  int           a;
  GdkDragAction action;

  /* Create the menu */
  menu = gnome_popup_menu_new (dnd_actions);
  a    = gnome_popup_menu_do_popup_modal (menu, NULL, NULL, NULL, NULL);

  /* Set action */
  switch (a) {
  case 0:
    action = GDK_ACTION_MOVE;
    break;

  case 1:
    action = GDK_ACTION_COPY;
    break;

  case 2:
    action = GDK_ACTION_LINK; /* Alias */
    break;

  default:
    action = GDK_ACTION_ASK; /* Magic value to indicate cancellation */
  }

  gtk_widget_destroy (menu);

  return action;
}


static GnobogBookmarksInsertMode
gnobog_arborescence_insert_mode_from_drag_pos (GtkCListDragPos insert_pos)
{
  switch (insert_pos) {
  case GTK_CLIST_DRAG_NONE:
    g_log (G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "Arbo: Insert mode: drag_none == default");
    return INSERT_DEFAULT_MODE;
  case GTK_CLIST_DRAG_BEFORE:
    g_log (G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "Arbo: Insert mode: drag_before == before");
    return INSERT_BEFORE;
  case GTK_CLIST_DRAG_INTO:
    g_log (G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "Arbo: Insert mode: drag_into == default");
    return INSERT_DEFAULT_MODE;
  case GTK_CLIST_DRAG_AFTER:
    g_log (G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "Arbo: Insert mode: drag_after == after");
    return INSERT_AFTER;
  default :
    g_log (G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, "Arbo: Insert mode: no insert pos == default");
    return INSERT_DEFAULT_MODE;
  }
}


/* FROM GTK_CLIST.C */
/* TODO: NOT USER CONFIGURABLE ??? */
#define CELL_SPACING 1
#define ROW_FROM_YPIXEL(clist, y)  (((y) - (clist)->voffset) / \
                                    ((clist)->row_height + CELL_SPACING))
#define ROW_TOP_YPIXEL(clist, row) (((clist)->row_height * (row)) + \
                                    (((row) + 1) * CELL_SPACING) + \
                                    (clist)->voffset)

/* determine position of cursor in regards to a row */
static GtkCListDestInfo
gnobog_arborescence_drag_dest_cell (GnobogArborescence* arborescence, 
                                    gint y)
{
  /* Adapted from GTK_CLIST.C */
  GtkWidget*        widget;
  GtkCList*         clist;
  GtkCListDestInfo  dest_info;

  widget = GTK_WIDGET (arborescence);
  clist  = GTK_CLIST (arborescence);

  dest_info.insert_pos = GTK_CLIST_DRAG_NONE;

  /* normalize coordinate */
  y -= (GTK_CONTAINER (clist)->border_width +
        widget->style->klass->ythickness +
        clist->column_title_area.height);

  dest_info.cell.row = ROW_FROM_YPIXEL (clist, y);
  if ( dest_info.cell.row >= clist->rows ) {
    dest_info.cell.row = clist->rows - 1;
    y = ROW_TOP_YPIXEL (clist, dest_info.cell.row) + clist->row_height;
  }
  if ( dest_info.cell.row < -1 )
    dest_info.cell.row = -1;

  if ( dest_info.cell.row >= 0 ) {
    gint y_delta;
    gint h = 0;

    y_delta = y - ROW_TOP_YPIXEL (clist, dest_info.cell.row);

    dest_info.insert_pos = GTK_CLIST_DRAG_INTO;
    h = clist->row_height / 4;

    if ( y_delta < h ) {
      dest_info.insert_pos = GTK_CLIST_DRAG_BEFORE;
    }
    else if ( clist->row_height - y_delta < h ) {
      dest_info.insert_pos = GTK_CLIST_DRAG_AFTER;
    }
  }
        
  return dest_info;
}


static void
gnobog_arborescence_drag_begin (GtkWidget* widget, 
                                GdkDragContext* context)
{
}

static void
gnobog_arborescence_drag_end (GtkWidget* widget, 
                              GdkDragContext* context)
{
  GnobogArborescence*  arborescence;

  g_return_if_fail (GNOBOG_IS_ARBORESCENCE (widget));

  arborescence = GNOBOG_ARBORESCENCE (widget);

  /* Reset info */
  arborescence->row_was_selected      = FALSE;
  arborescence->timer_expired         = FALSE;
  arborescence->in_extended_selection = FALSE;
  arborescence->perform_drag          = FALSE;
  /* reset button pressed info to allow triggering 
   * drag appropriately in motion notify */
  arborescence->button_pressed = 0;
}


static void                       
gnobog_arborescence_drag_data_delete (GtkWidget* widget, 
				      GdkDragContext* context)
{

}


static void
gnobog_arborescence_drag_data_get (GtkWidget* widget, 
                                   GdkDragContext* context,
                                   GtkSelectionData* selection_data, 
                                   guint info, 
                                   guint time)
{
  GList*               selection;
  GList*               iterator;
  GnobogBookmarksNode  bookmarks_node;
  GnobogArborescence*  arborescence;
  gchar*               location;
  gchar*               text_uri_list;
  gchar*               temp_string;

  g_return_if_fail (GNOBOG_IS_ARBORESCENCE (widget));

  arborescence = GNOBOG_ARBORESCENCE (widget);

  switch (info) {
  case GNOBOG_BOOKMARK_ENTRY:
    /* Set selection_data->data to be "Inner Drag"           */
    /* It's not the pertinent data; we'll work directly on   */
    /* arborescence selections in drop.                      */
    /* TODO : try to copy the selection pointer ?            */
                
    gtk_selection_data_set (selection_data,
                            selection_data->target,
                            8, 
                            "Inner Drag", 
                            10);
    break;

  case STRING :
    info = TEXT_URI_LIST;  /* go to next case... */

  case TEXT_URI_LIST : /* TODO */
    /* construct a multi-lines string, one location per line */
    selection     = gnobog_arborescence_get_sorted_bookmarks_node_selection (arborescence);
    iterator      = selection;
    text_uri_list = g_strdup ("");
    if ( NULL == iterator ) break; /* shouldn't be possible ! */

    do {
      bookmarks_node    = (GnobogBookmarksNode) iterator->data;
      location          = gnobog_bookmarks_node_get_location (bookmarks_node);
      temp_string       = g_strconcat (text_uri_list, location, "\r\n", NULL);
      g_free (text_uri_list);
      text_uri_list     = temp_string;
    } while ((iterator = g_list_next (iterator)) != NULL);  

    /* send the list to dnd target */
    gtk_selection_data_set (selection_data,
			    selection_data->target,
			    8,
			    text_uri_list,
			    strlen (text_uri_list));
    /* TODO: free data on drag data delete */
    break;

  case MOZILLA_URL :
    /* Find the bookmark dragged */
    selection = gnobog_arborescence_get_sorted_bookmarks_node_selection (arborescence);
    if ( NULL == selection ) break; 
    bookmarks_node = (GnobogBookmarksNode) (selection->data);

    /* Set the url of the selected bookmark as the dragged data */
    location = gnobog_bookmarks_node_get_location (bookmarks_node);
    gtk_selection_data_set (selection_data,
                            selection_data->target,
                            8 /* format : num bits/unit */,
                            location /* data pointer */,
                            strlen (location) /* length */);
    /* TODO: check if there's no problem with giving a pointer to the internal
     * 'location' field (this pointer *only* is copied and freed says the doc :-)
     */
    break;

  default :
    g_message("Destination data unknown");
    break;
  }

}

/* WARNING : don't make assumptions !!! */
/* leave signal might be treated in this arborescence AFTER motion signal in another arborescence, */
/* while the context remains the same */
static void
gnobog_arborescence_drag_leave (GtkWidget* widget, 
                                GdkDragContext* context, 
                                guint time)
{
  GtkCList*            clist;
  GnobogArborescence*  arborescence;
  GtkCListDestInfo*    dest_info;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (context != NULL);
  g_return_if_fail (GNOBOG_IS_ARBORESCENCE (widget));

  clist        = GTK_CLIST (widget);
  arborescence = GNOBOG_ARBORESCENCE (widget);

  dest_info    = &(arborescence->dest_info);

  /* remove drag line/rectangle */
  if ( dest_info->cell.row >= 0 ) {
    gnobog_arborescence_draw_drag_highlight (arborescence, 
                                             dest_info->cell.row, 
                                             dest_info->insert_pos);
  }
        
  gnobog_arborescence_cancel_drag_scroll (arborescence);
        
  /* init dest_info */
  /* CHECK : is there always a leave signal (after button release for instance ?) */
  dest_info->insert_pos  = GTK_CLIST_DRAG_NONE;
  dest_info->cell.row    = -1;
}

static gboolean
gnobog_arborescence_drag_motion (GtkWidget* widget, 
                                 GdkDragContext* context,
                                 gint x, 
                                 gint y, 
                                 guint time)
{
  GnobogArborescence*  arborescence;
  GtkCList*            clist;
  GtkCListDestInfo*    dest_info;
  GtkCListDestInfo     new_info;
  gboolean             pos_has_changed;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GNOBOG_IS_ARBORESCENCE (widget), FALSE);

  arborescence = GNOBOG_ARBORESCENCE (widget);
  clist        = GTK_CLIST (widget);      

  dest_info    = &(arborescence->dest_info);

  /* Correct y coordinate - check future versions of Gtk+ !
   * Only needed if you use clist/ctree functions to find the row,
   * in which case you have to set coherence regarding row number */
//y -= clist->column_title_area.height;
        
  /* Determine if we are before/on/after a row or not on a row at all */
  new_info = gnobog_arborescence_drag_dest_cell (arborescence, y);
  /* Check if position has changed */
  if ( new_info.cell.row != dest_info->cell.row ||
       (new_info.cell.row == dest_info->cell.row &&
        dest_info->insert_pos != new_info.insert_pos) ) {
    pos_has_changed = TRUE;
  } else {
    pos_has_changed = FALSE;
  }
        
  /* Evalutate if scrolling is desirable.
   * If it is, we'll scroll while mouse is idle (i.e. until we
   * come in here again) */
  gnobog_arborescence_setup_drag_scroll (arborescence, y);
        
  /* Check if we must erase/draw a drag line/rectangle */
  if ( pos_has_changed ) {
    /* XOR previous line/rectangle only if we are on the SAME widget */
    if ( dest_info->cell.row >= 0 ) {
      gnobog_arborescence_draw_drag_highlight (arborescence, 
                                               dest_info->cell.row, 
                                               dest_info->insert_pos);
    }

    dest_info->insert_pos  = new_info.insert_pos;
    dest_info->cell.row    = new_info.cell.row;
    /* TODO : check if we really CAN'T have row < 0 here */ 
    if ( dest_info->cell.row >= 0 ) {
      gnobog_arborescence_draw_drag_highlight (arborescence, 
                                               dest_info->cell.row, 
                                               dest_info->insert_pos);
    }
    /* ??? */
//  gdk_drag_status (context, context->suggested_action, time);
  }


  /* Remember current position */
  dest_info->insert_pos  = new_info.insert_pos;
  dest_info->cell.row    = new_info.cell.row;
  dest_info->cell.column = new_info.cell.column;

  return FALSE; /* Let the other handlers operate or not ? */
}

static gboolean
gnobog_arborescence_drag_drop (GtkWidget* widget, 
                               GdkDragContext* context,
                               gint x, 
                               gint y, 
                               guint time)
{
        
  return FALSE;
}


static void
gnobog_arborescence_drag_data_received (GtkWidget* widget, 
                                        GdkDragContext* context,
                                        gint x, 
                                        gint y, 
                                        GtkSelectionData* selection_data,
                                        guint info, 
                                        guint time)
{
  GnobogArborescence*        arborescence;
  GnobogArborescence*        source_arborescence;
  GnobogArborescenceNode     arborescence_node;
  GnobogBookmarksNode        paste_anchor;
  GtkCListDestInfo           dest_info;
  GnobogBookmarksInsertMode  insert_mode;
  GdkDragAction              action;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GNOBOG_IS_ARBORESCENCE (widget));    

  arborescence = GNOBOG_ARBORESCENCE (widget);

  /* Check whether we are before/on/after a row */
  dest_info = gnobog_arborescence_drag_dest_cell (arborescence, y);
  if ( dest_info.insert_pos != GTK_CLIST_DRAG_NONE ) {
    insert_mode = gnobog_arborescence_insert_mode_from_drag_pos (dest_info.insert_pos);
    /* Get the arborescence_node on this row */
    arborescence_node = gnobog_arborescence_get_node_from_row (arborescence, 
                                                               dest_info.cell.row);
    if (arborescence_node == NULL) return;
        
    /* Get the corresponding paste anchor */
    paste_anchor = gnobog_arborescence_get_bookmarks_node_from_arborescence_node (arborescence, 
                                                                                  arborescence_node);
  } else { /* insert into root node, even if root == NULL (empty arbo) */
    insert_mode       = INSERT_INTO;
    arborescence_node = NULL;
    paste_anchor      = NULL;
  }

  /* Decode the information dropped */
  /* !!!!!!!!! TODO: Netscape can (rarely) send awful garbage, check if url's clean */
  switch (info) {
    gchar*   string_received;
    gchar*   bookmark_name;
    gchar*   pos_bookmark_name;
    gchar**  split_string_array;
    GList*   list;

  case STRING :
    info = MOZILLA_URL; /* go to next case */

  case MOZILLA_URL:

    /* Unselect any possible selection */
    gnobog_arborescence_unselect_all (arborescence);
    /* We receive the URL only... What a pity :-( */
    /* Hurray !! Mozilla seems to send URL and Title :-) */
    string_received = g_strdup ((gchar *)selection_data->data);
    g_message ("String dropped : -*-%s-*-", string_received);
    /* Check the url string and get site address in url */
    if ( (pos_bookmark_name = strstr (string_received, "\n")) != NULL ) {
      g_message ("Mozilla Good Boy !");
      /* Mozilla sends "URL TITLE" */
      bookmark_name = g_strdup (pos_bookmark_name + 1);
      *pos_bookmark_name = '\0';
    } else if ( 0 == g_strncasecmp (string_received, "http://", 7) ) {
      split_string_array = g_strsplit (string_received, "/", 3);
      bookmark_name      = g_strdup (split_string_array[2]);
      g_strfreev (split_string_array);
    } else bookmark_name = g_strdup (string_received);
    /* Put the dropped entry into the folder */
    gnobog_bookmarks_node_insert_new (arborescence->bookmarks_watched,
                                      paste_anchor, 
                                      insert_mode,
                                      BOOKMARK,
                                      bookmark_name /* name */,
                                      string_received /* location */,
                                      "" /* description */);
    g_free (bookmark_name);
    g_free (string_received);
    /* The row will be selected in callback from the 'list-created' signal */

    break;

  case TEXT_URI_LIST : /* TODO: factorize with above */

    /* Unselect any possible selection */
    gnobog_arborescence_unselect_all (arborescence);

    g_message ("Received text_uri_list by drag & drop");

    list = gnome_uri_list_extract_uris (selection_data->data);

    while (list) {
      string_received = g_strdup ((gchar *)list->data);
      g_message ("String dropped : -*-%s-*-", string_received);
      /* Check the url string and get site address in url */
      if ( (pos_bookmark_name = strstr (string_received, "\n")) != NULL ) {
	bookmark_name = g_strdup (pos_bookmark_name + 1);
	*pos_bookmark_name = '\0';
      } else if ( 0 == g_strncasecmp (string_received, "http://", 7) ) {
	split_string_array = g_strsplit (string_received, "/", 3);
	bookmark_name      = g_strdup (split_string_array[2]);
	g_strfreev (split_string_array);
      } else bookmark_name = g_strdup (string_received);
      /* Put the dropped entry into the folder */
      gnobog_bookmarks_node_insert_new (arborescence->bookmarks_watched,
					paste_anchor, 
					insert_mode,
					BOOKMARK,
					bookmark_name /* name */,
					string_received /* location */,
					"" /* description */);
      g_free (bookmark_name);
      g_free (string_received);

      list = list->next;
    }
    gnome_uri_list_free_strings (list);

    break;

  case GNOBOG_BOOKMARK_ENTRY: /* internal dnd */

    /* Get the source widget of the drag */
    source_arborescence = (GnobogArborescence*) gtk_drag_get_source_widget (context);
    /* If row is part of the selection do nothing */
    /* TODO: shouldn't the model (bookmarks object) do all the checks ? */
    if ( source_arborescence == arborescence ) {
      if ( gnobog_arborescence_is_node_selected (arborescence_node) ) {
        gtk_drag_finish (context, 
                         FALSE /* fail */, 
                         FALSE /* no delete */, 
                         time);
        break;
      }
    }
    else {
      /* Check if paste_anchor would be removed by dnd */
      if ( arborescence->bookmarks_watched == source_arborescence->bookmarks_watched && 
           gnobog_arborescence_is_bookmarks_node_selected (source_arborescence, paste_anchor) ) {
        gtk_drag_finish (context, 
                         FALSE /* fail */, 
                         FALSE /* no delete */, 
                         time);
        break;
      }
      /* If view has changed clear target view selection */
      gnobog_arborescence_unselect_all(arborescence);
    }
    /* Get the selection */
    list = gnobog_arborescence_get_sorted_bookmarks_node_selection (source_arborescence);
    /* Determine the action to perform */
    if ( context->action == GDK_ACTION_ASK ) {
      action = gnobog_arborescence_drag_get_action (context);
      if ( action == GDK_ACTION_ASK )
        return;
    } else {
      action = context->actions;
    }

    /* Move the bookmarks dragged IF THE DOCUMENT IS THE SAME 
     * (really the same, not another bookmarks object even if same file) 
     * I.E. : use move ONLY from a view to another view 
     * Else we have to use the clipboard */
    if ( arborescence->bookmarks_watched == source_arborescence->bookmarks_watched ) {
      /* Act on account of key pressed, move is default */
      if ( (action & GDK_ACTION_MOVE) != 0 ) {
        gnobog_bookmarks_list_move (arborescence->bookmarks_watched,
                                    paste_anchor, 
                                    insert_mode, 
                                    list);
      } else {
        gnobog_bookmarks_list_copy (arborescence->bookmarks_watched,
                                    paste_anchor, 
                                    insert_mode, 
                                    list);                               
      }
    } else {
      /* Act on account of key pressed, copy is the default */
      if ( (action & GDK_ACTION_COPY) != 0 ) {
        gnobog_bookmarks_list_copy_from (arborescence->bookmarks_watched,
                                         source_arborescence->bookmarks_watched,
                                         paste_anchor, 
                                         insert_mode,
                                         list);
      } else {
        gnobog_bookmarks_list_move_from (arborescence->bookmarks_watched,
                                         source_arborescence->bookmarks_watched,
                                         paste_anchor, 
                                         insert_mode,
                                         list);                                
      }
    }
    g_list_free (list);
    /* rows are selected in callback */

    break;

  default:
    g_message ("This kind of dropped data is not yet managed, ID : %d\n", info);
  }

  return;
}
















/******************************************************************************/
/*      ROWS                                                                  */
/******************************************************************************/
/*
 * gnobog_arborescence_get_row_from_cursor_coords
 * gnobog_arborescence_get_row_from_arborescence_node
 * gnobog_arborescence_get_row_from_bookmarks_node
 */
gint
gnobog_arborescence_get_row_from_cursor_coords (GnobogArborescence* arborescence, 
                                               gint x, 
                                               gint y)
{
  gint row, col;

  g_return_val_if_fail (arborescence != NULL, -1);
  g_return_val_if_fail (GNOBOG_IS_ARBORESCENCE (arborescence), -1);


  if ( !gtk_clist_get_selection_info (GTK_CLIST (arborescence), x, y, &row, &col) ) {
    return -1;
  }
        
  return row;
}

/* TODO: CHANGE (Use a data structure) */
gint                                    
gnobog_arborescence_get_row_from_arborescence_node (GnobogArborescence* arborescence, 
                                                    GnobogArborescenceNode arborescence_node)
{
  GnobogBookmarksNode  bookmarks_node;

  g_return_val_if_fail (arborescence_node != NULL, -1);

  /* There MUST be a better way !! */
  bookmarks_node = gnobog_arborescence_get_bookmarks_node_from_arborescence_node (arborescence, 
                                                                                  arborescence_node);
  return gnobog_arborescence_get_row_from_bookmarks_node (arborescence, bookmarks_node);
}

/* TODO: CHANGE (Use a data structure) */
gint                                    
gnobog_arborescence_get_row_from_bookmarks_node (GnobogArborescence* arborescence, 
                                                 GnobogBookmarksNode bookmarks_node)
{
  return gtk_clist_find_row_from_data (GTK_CLIST (arborescence), bookmarks_node);
}















/******************************************************************************/
/*      VISUAL FEEDBACK                                                       */
/******************************************************************************/
/*
 * gnobog_arborescence_unhighlight_row
 * gnobog_arborescence_highlight_row
 * gnobog_arborescence_highlight_node
 * gnobog_arborescence_unhighlight_node
 * gnobog_arborescence_set_focus_row
 * gnobog_arborescence_reset_focus_row
 * gnobog_arborescence_draw_drag_highlight
 */


/* Not used anymore ? */ 
void            
gnobog_arborescence_unhighlight_row (GnobogArborescence* arborescence, 
                                     gint row)
{
  g_return_if_fail (arborescence != NULL);
  g_return_if_fail (row > 0);
  g_return_if_fail (GNOBOG_IS_ARBORESCENCE (arborescence));

  gtk_clist_set_background (GTK_CLIST (arborescence), row, NULL);

}

/* Not used anymore ? */ 
void            
gnobog_arborescence_highlight_row (GnobogArborescence* arborescence, 
                                   gint row)
{
  GnobogArborescenceNode  arborescence_node;      
  /* TODO : Make this customizable (or use theme ?) */
  static GdkColor         background_color = {0, 65535,0,0};

  g_return_if_fail (arborescence != NULL);
  g_return_if_fail (row > 0);
  g_return_if_fail (GNOBOG_IS_ARBORESCENCE (arborescence));

  arborescence_node = gnobog_arborescence_get_node_from_row (arborescence, row);

  gtk_clist_set_background (GTK_CLIST (arborescence), row, &background_color);
}

/* Not used anymore ? */ 
void            
gnobog_arborescence_highlight_node (GnobogArborescence* arborescence, 
                                    GnobogArborescenceNode arborescence_node)
{
  /* TODO : Make this customizable (or use theme ?) */
  static GdkColor  background_color = {0, 65535,0,0};

  g_return_if_fail (arborescence != NULL);
  g_return_if_fail (arborescence_node != NULL);
  g_return_if_fail (GNOBOG_IS_ARBORESCENCE (arborescence) );

  gtk_ctree_node_set_background (GTK_CTREE (arborescence), arborescence_node, &background_color);
}

void            
gnobog_arborescence_unhighlight_node (GnobogArborescence* arborescence, 
                                      GnobogArborescenceNode arborescence_node)
{
  g_return_if_fail (arborescence != NULL);
  g_return_if_fail (arborescence_node != NULL);
  g_return_if_fail (GNOBOG_IS_ARBORESCENCE (arborescence));

  gtk_ctree_node_set_background (GTK_CTREE (arborescence), arborescence_node, NULL);
}

void            
gnobog_arborescence_set_focus_row (GnobogArborescence* arborescence, 
                                   gint row)
{
  GTK_CLIST (arborescence)->focus_row = row;
}

void            
gnobog_arborescence_reset_focus_row (GnobogArborescence* arborescence)
{
   GTK_CLIST (arborescence)->focus_row = -1;
}

static void
gnobog_arborescence_draw_drag_highlight (GnobogArborescence* arborescence,
                                         gint row,
                                         GtkCListDragPos drag_pos)
{
  GnobogArborescenceNode  arborescence_node;

  arborescence_node = gnobog_arborescence_get_node_from_row (arborescence, row);

  GTK_CLIST_CLASS_FW (GTK_CLIST (arborescence))->draw_drag_highlight 
    (GTK_CLIST (arborescence),
     (GtkCListRow *) GTK_CTREE_ROW (arborescence_node),
     row,
     drag_pos);
}



















/******************************************************************************/
/*      SELECTIONS                                                            */
/******************************************************************************/
/*
 * gnobog_arborescence_selection_exists
 * gnobog_arborescence_select_row;
 * gnobog_arborescence_unselect_row;
 * gnobog_arborescence_compare_rows
 * gnobog_arborescence_get_sorted_node_selection (USELESS ??)
 * gnobog_arborescence_get_sorted_bookmarks_node_selection
 * gnobog_arborescence_select_all
 * gnobog_arborescence_unselect_all
 * gnobog_arborescence_bookmarks_node_select
 * gnobog_arborescence_select_range
 * gnobog_arborescence_handle_selection
 * gnobog_arborescence_list_select
 * gnobog_arborescence_is_bookmarks_node_selected
 * gnobog_arborescence_node_select
 * gnobog_arborescence_is_row_selected
 * gnobog_arborescence_is_node_selected
 */

gboolean                
gnobog_arborescence_selection_exists (GnobogArborescence* arborescence)
{
  g_return_val_if_fail (arborescence != NULL, FALSE);

  return ( NULL != GTK_CLIST (arborescence)->selection );
}


void            
gnobog_arborescence_unselect_row (GnobogArborescence* arborescence, 
                                  gint row)
{
  g_return_if_fail (arborescence != NULL);
  g_return_if_fail (row >= 0);
        
  gtk_clist_unselect_row (GTK_CLIST (arborescence), row, 0);
  gnobog_arborescence_reset_focus_row (arborescence);
}

void            
gnobog_arborescence_select_row (GnobogArborescence* arborescence, 
                                gint row)
{
  g_return_if_fail (arborescence != NULL);
  g_return_if_fail (row >= 0);
        
  gtk_clist_select_row (GTK_CLIST (arborescence), row, 0);
  gnobog_arborescence_set_focus_row (arborescence, row);
}

static gint
gnobog_arborescence_compare_rows (gconstpointer a, 
                                  gconstpointer b)
{
  GnobogBookmarksNodeWithRow* node_with_row_a = (GnobogBookmarksNodeWithRow*) a;
  GnobogBookmarksNodeWithRow* node_with_row_b = (GnobogBookmarksNodeWithRow*) b;
        
  if ( node_with_row_a->row < node_with_row_b->row ) return 1;
  if ( node_with_row_a->row > node_with_row_b->row ) return -1;
  return 0;
}

/* TODO but not useful ? */
GList*  
gnobog_arborescence_get_sorted_node_selection (GnobogArborescence* arborescence)
{
  GList*  list;

  g_return_val_if_fail (arborescence != NULL, NULL);

  list = NULL;
  return list;
}

/* Get the list of bookmarks nodes corresponding to the rows selected
 * by the user.
 * The selection is always from bottom to top so as to easily preserve the
 * order in later insertions. */
GList*  
gnobog_arborescence_get_sorted_bookmarks_node_selection (GnobogArborescence* arborescence)
{
  GnobogArborescenceNode       arborescence_node;
  GnobogBookmarksNode          bookmarks_node;
  GList*                       list;
  GList*                       iterator;
  GList*                       bookmarks_nodes_rows_list;
  gint                         row = 0;
  GnobogBookmarksNodeWithRow*  bookmarks_node_with_row;

  g_return_val_if_fail (arborescence != NULL, NULL);

  /* CList selection is NOT ordered. Maybe we miss something...
   * Moreover, g_list_sort needs a 'compare function' in which we won't be
   * able to know which arborescence we are in...
   * And there's no sorting possibility in hash tables or data sets...
   * So for now we build a list of bookmarks_node_with_row */
        
  list                      = NULL;
  bookmarks_nodes_rows_list = NULL;
  bookmarks_node_with_row   = NULL;

  /* If CTree selection is null, send back NULL as selection */
  if ( (iterator = GTK_CLIST (arborescence)->selection) == NULL ) {
    return NULL;
  }
                
  do {
    arborescence_node =  (GnobogArborescenceNode) iterator->data;
    /* Get the Bookmark */
    bookmarks_node = gnobog_arborescence_get_bookmarks_node_from_arborescence_node (arborescence, 
                                                                                    arborescence_node);
    /* Check the row */
    row = gnobog_arborescence_get_row_from_bookmarks_node (arborescence, bookmarks_node);
    /* Build the temporary list */
    bookmarks_node_with_row       = g_new (GnobogBookmarksNodeWithRow, 1);
    bookmarks_node_with_row->row  = row;
    bookmarks_node_with_row->node = bookmarks_node;
    bookmarks_nodes_rows_list     = g_list_prepend (bookmarks_nodes_rows_list, 
                                                    bookmarks_node_with_row);
  } while ((iterator = g_list_next (iterator)) != NULL);  

  /* Check bookmarks_nodes_rows_list */
  iterator = bookmarks_nodes_rows_list;
  do {
    bookmarks_node_with_row = (GnobogBookmarksNodeWithRow*) iterator->data;
    bookmarks_node          = bookmarks_node_with_row->node;
    row                     = bookmarks_node_with_row->row;
    g_message ("Bookmarks_nodes_rows_list : %s -> row %d", 
               gnobog_bookmarks_node_get_name (bookmarks_node), 
               row);
  } while ((iterator = g_list_next(iterator)) != NULL);

  /* Create a selection of bookmarks nodes sorted by rows */
  bookmarks_nodes_rows_list = g_list_sort (bookmarks_nodes_rows_list, 
                                           gnobog_arborescence_compare_rows);
  if ( bookmarks_nodes_rows_list == NULL ) g_message ("Alert sorted list null");
  iterator = bookmarks_nodes_rows_list;
  do {
    bookmarks_node_with_row = (GnobogBookmarksNodeWithRow*) iterator->data;
    list                    = g_list_prepend (list, bookmarks_node_with_row->node);
    g_free (bookmarks_node_with_row);        
  } while ((iterator = g_list_next (iterator)) != NULL);
  /* Free the temporary list */
  /* TODO: free data in list (bookmarks_nodes_with_rows) */
  g_list_free (bookmarks_nodes_rows_list);

  /* Check sorted final selection */
  iterator = list;
  do {
    bookmarks_node = (GnobogBookmarksNode) iterator->data;
    g_message ("Selection item : %s", gnobog_bookmarks_node_get_name (bookmarks_node));
  } while ((iterator = g_list_next(iterator)) != NULL);

  return list;
}

void
gnobog_arborescence_unselect_all (GnobogArborescence* arborescence)
{
  g_return_if_fail (arborescence != NULL);

  gtk_clist_unselect_all (GTK_CLIST (arborescence));
}

void
gnobog_arborescence_select_all (GnobogArborescence* arborescence)
{
  g_return_if_fail (arborescence != NULL);

  gtk_clist_select_all (GTK_CLIST (arborescence));
}

void
gnobog_arborescence_bookmarks_node_select (GnobogArborescence* arborescence, 
                                           GnobogBookmarksNode bookmarks_node )
{
  GnobogArborescenceNode arborescence_node;

  g_return_if_fail (arborescence != NULL);
  g_return_if_fail (bookmarks_node != NULL);

  arborescence_node = gnobog_arborescence_get_arborescence_node_from_bookmarks_node (arborescence, 
                                                                                     bookmarks_node);
  gnobog_arborescence_node_select (arborescence, arborescence_node);
}


static void
gnobog_arborescence_select_range (GnobogArborescence* arborescence, 
                                  int row1, 
                                  int row2)
{
  int   i;
  gint  row_min;
  gint  row_max;

  row_min = MIN (row1, row2);
  row_max = MAX (row1, row2);

  for (i = row_min; i <= row_max; i++)
    gtk_clist_select_row (GTK_CLIST (arborescence), i, 0);
}


static void
gnobog_arborescence_unselect_range (GnobogArborescence* arborescence, 
                                    int row1, 
                                    int row2)
{
  int   i;
  gint  row_min;
  gint  row_max;

  row_min = MIN (row1, row2);
  row_max = MAX (row1, row2);

  for (i = row_min; i <= row_max; i++)
    gtk_clist_unselect_row (GTK_CLIST (arborescence), i, 0);
}

/* Handles row selection according to the specified modifier state
 * Used by motion handler */
static void
gnobog_arborescence_handle_selection (GnobogArborescence* arborescence, 
                                      int row, 
                                      guint state)
{
  gint      previous_row, anchor_row;  /* just to ease typing */
  gint      max, min;
  gboolean  normal, shift, control, shift_control;
  gboolean  windows_mode, dnd_timer_only_mode, rino_mode;
        
  normal        = (state & GDK_SHIFT_MASK) == 0 &&
                  (state & GDK_CONTROL_MASK) == 0;
  shift         = (state & GDK_SHIFT_MASK) != 0 &&
                  (state & GDK_CONTROL_MASK) == 0;
  control       = (state & GDK_SHIFT_MASK) == 0 &&
                  (state & GDK_CONTROL_MASK) != 0;
  shift_control = (state & GDK_SHIFT_MASK) != 0 &&
                  (state & GDK_CONTROL_MASK) != 0;

  windows_mode        = (arborescence->selection_mode == WINDOWS_MODE);
  dnd_timer_only_mode = (arborescence->selection_mode == DND_TIMER_ONLY_MODE);
  rino_mode           = (arborescence->selection_mode == RINO_MODE);

  previous_row = arborescence->previous_row;
  anchor_row   = arborescence->anchor_row;
  min          = MIN(anchor_row, row);
  max          = MAX(anchor_row, row);

  if ( arborescence->row_was_selected &&
       (dnd_timer_only_mode || (rino_mode && shift)) ) {
    gnobog_arborescence_unselect_range (arborescence, anchor_row, row);
    return;
  }


  if ( !arborescence->row_was_selected && (dnd_timer_only_mode || rino_mode) ) {
    /* Compare anchor_row, previous_row and row */
    if ( row > previous_row ) {
      if ( previous_row >= anchor_row ) { /* select prev->row */
        gnobog_arborescence_select_range (arborescence, previous_row+1, row);
      } else {                            /* unselect prev->anch and select anch->row */
        gnobog_arborescence_unselect_range (arborescence, previous_row, min-1);
        gnobog_arborescence_select_range (arborescence, min, max);
      }
    } else if ( row <= previous_row ) {
      if ( previous_row <= anchor_row ) { /* select row->prev */
        gnobog_arborescence_select_range (arborescence, row, previous_row);
      } else {                            /* unselect anch->prev and select row->anch */
        gnobog_arborescence_unselect_range (arborescence, max+1, previous_row);
        gnobog_arborescence_select_range (arborescence, min, max);
      }
    }
  }       
  /* remember row */
  arborescence->previous_row = row;
}

/* Select an arbitrary collection of bookmarks in the current arborescence.
 * Useful in dnd_cb_arborescence_drop.
 * TODO: optimize (don't perform a search from root node in each loop) */
void
gnobog_arborescence_list_select (GnobogArborescence* arborescence, 
                                 GList* list)
{
  GList*                  iterator; /* list of BookmarksNode */
  GnobogBookmarksNode     bookmarks_node;
  GnobogArborescenceNode  arborescence_node;
  gint                    row;

  g_return_if_fail (list != NULL);
  g_return_if_fail (arborescence != NULL);
  g_return_if_fail (GNOBOG_IS_ARBORESCENCE (arborescence));

  iterator = list; /* The list of bookmarks nodes to select */

  /* Freeze the arborescence */
  gtk_clist_freeze (GTK_CLIST (arborescence));
        
  /* Perform the selection by searching the bookmarks nodes in the ctree */
  /* TODO : optimize ? */
  do {
    bookmarks_node    = (GnobogBookmarksNode) (iterator->data);
    arborescence_node = gnobog_arborescence_get_arborescence_node_from_bookmarks_node (arborescence, 
                                                                                       bookmarks_node);
    if ( arborescence_node ) {
      gnobog_arborescence_node_select (arborescence, arborescence_node);
    }
  } while ((iterator = g_list_next (iterator)) != NULL) ; 

  /* Put the focus on the first row (last selection item)
   * TODO : replace by gnobog_arborescence_get_row_from_arborescence_node
   *  if we find a way to look for the row more efficiently...  */
  row = gnobog_arborescence_get_row_from_bookmarks_node (arborescence, bookmarks_node);
  gnobog_arborescence_set_focus_row (arborescence, row);
        
  /* Thaw the arborescence */
  gtk_clist_thaw (GTK_CLIST (arborescence));
        
  return;
}

gboolean        
gnobog_arborescence_is_bookmarks_node_selected (GnobogArborescence* arborescence, 
                                                GnobogBookmarksNode bookmarks_node)
{
  GnobogArborescenceNode  arborescence_node;

  g_return_val_if_fail (arborescence != NULL, FALSE);

  if (bookmarks_node == NULL) return FALSE;

  arborescence_node = gnobog_arborescence_get_arborescence_node_from_bookmarks_node (arborescence, 
                                                                                     bookmarks_node);

  return gnobog_arborescence_is_node_selected (arborescence_node);
}

void            
gnobog_arborescence_node_select (GnobogArborescence* arborescence, 
                                 GnobogArborescenceNode arborescence_node )
{
  g_return_if_fail (arborescence != NULL);
  g_return_if_fail (arborescence_node != NULL);
  g_return_if_fail (GNOBOG_IS_ARBORESCENCE (arborescence));

  gtk_ctree_select (GTK_CTREE (arborescence), arborescence_node);
}


gboolean
gnobog_arborescence_is_row_selected (GnobogArborescence* arborescence, 
                                     gint row)
{
  GnobogArborescenceNode  arborescence_node;

  g_return_val_if_fail (arborescence != NULL, FALSE);
  g_return_val_if_fail (row >= 0, FALSE);
        
  arborescence_node = gnobog_arborescence_get_node_from_row (arborescence, row);
  return gnobog_arborescence_is_node_selected (arborescence_node);
}

gboolean
gnobog_arborescence_is_node_selected (GnobogArborescenceNode arborescence_node)
{
  if ( arborescence_node == NULL ) return FALSE;
        
  return (GTK_CTREE_ROW (arborescence_node)->row.state == GTK_STATE_SELECTED);
}



















/******************************************************************************/
/*      CALLBACKS                                                             */
/******************************************************************************/
/*
 * gnobog_arborescence_clear
 * gnobog_arborescence_list_insert
 * gnobog_arborescence_list_delete
 * gnobog_arborescence_list_modify
 */

void    
gnobog_arborescence_clear (GnobogArborescence* arborescence, 
                           gpointer data)
{
  g_return_if_fail (arborescence != NULL);
  g_return_if_fail (GNOBOG_IS_ARBORESCENCE (arborescence));

  gtk_ctree_remove_node (GTK_CTREE (arborescence), NULL);
}

void
gnobog_arborescence_list_insert (GnobogArborescence* arborescence, 
                                 GnobogBookmarksNode anchor_node,
                                 GnobogBookmarksInsertMode mode,
                                 GList* list,
                                 gpointer data)
{
  GList*               iterator;
  GnobogBookmarksNode  bookmarks_node;
  gint                 anchor_row;
  //  gint                 i;

  g_return_if_fail (arborescence != NULL);
  g_return_if_fail (GNOBOG_IS_ARBORESCENCE (arborescence));

  /* If anchor node is null, root will be assumed */

  if (list == NULL) return;
        
  gtk_clist_freeze (GTK_CLIST (arborescence));

  /* Insert all GnobogBookmarksNode (bookmarks subtrees) */
  if ( mode == INSERT_AFTER || mode == INSERT_INTO ) {
    iterator = g_list_last (list);
    do {
      bookmarks_node = (GnobogBookmarksNode) iterator->data;
      gnobog_arborescence_node_insert (arborescence, anchor_node, mode, bookmarks_node);
    } while ((iterator = g_list_previous (iterator)) != NULL);      
  } else {
    iterator = list;
    do{
      bookmarks_node = (GnobogBookmarksNode) iterator->data;
      gnobog_arborescence_node_insert (arborescence, anchor_node, mode, bookmarks_node );
    } while ((iterator = g_list_next (iterator)) != NULL);
  }

  /* try to optimize colums width */
  //  for (i=0; i<arborescence->columns; i++) {
  // if ( arborescence->columns_visibility[i - 1] == TRUE ) {
  //   gtk_clist_optimal_column_width  (GTK_CLIST (arborescence), i); 
  // }
  //}


  gtk_clist_thaw (GTK_CLIST (arborescence));

  /* If there's no selection (i.e. we are in the arborescence where we dropped/pasted entries )
   * we select the inserted rows 
   * WARNING : now the previous statement is rarely true with all our funcionnalities... 
   * TODO: In fact we should select only in the arborescence the mouse is over to not
   * cancel selections in other views */
//if (GTK_CLIST (arborescence)->selection == NULL) {
  gnobog_arborescence_unselect_all (arborescence);
  gnobog_arborescence_list_select (arborescence, list);
  bookmarks_node = (GnobogBookmarksNode) list->data;
  anchor_row = gnobog_arborescence_get_row_from_bookmarks_node (arborescence, bookmarks_node);
  arborescence->anchor_row = anchor_row;
//}
}

void
gnobog_arborescence_list_delete (GnobogArborescence* arborescence, 
                                 GList* list, 
                                 gpointer data)
{
  GList*              iterator;
  GnobogBookmarksNode bookmarks_node;

  g_return_if_fail (arborescence != NULL);
  g_return_if_fail (GNOBOG_IS_ARBORESCENCE (arborescence));

  if ( list == NULL ) return;
        
  gtk_clist_freeze (GTK_CLIST (arborescence));

  /* Delete all GnobogBookmarksNode (bookmarks subtrees) */
  iterator = list;
  do{
    bookmarks_node = (GnobogBookmarksNode) iterator->data;
    gnobog_arborescence_node_delete (arborescence, bookmarks_node);
  } while ((iterator = g_list_next (iterator)) != NULL);

  gtk_clist_thaw (GTK_CLIST (arborescence));
}

void
gnobog_arborescence_list_modify (GnobogArborescence* arborescence, 
                                 GList* list, 
                                 gpointer data )
{
  GList*              iterator;
  GnobogBookmarksNode bookmarks_node;

  g_return_if_fail (arborescence != NULL);
  g_return_if_fail (GNOBOG_IS_ARBORESCENCE (arborescence));

  if ( list == NULL ) return;       
        
  gtk_clist_freeze (GTK_CLIST (arborescence));

  /* Modify all GnobogBookmarksNode (and traverse subtrees ?!?) */
  iterator = list;
  do{
    bookmarks_node = (GnobogBookmarksNode) iterator->data;
    gnobog_arborescence_node_modify (arborescence, bookmarks_node );
  } while ((iterator = g_list_next (iterator)) != NULL);

  gtk_clist_thaw (GTK_CLIST (arborescence));
}
















/******************************************************************************/
/*      NODES                                                                 */
/******************************************************************************/
/*
 * gnobog_arborescence_get_node_from_row
 * gnobog_arborescence_get_node_from_cursor_coords
 * gnobog_arborescence_get_arborescence_node_from_bookmarks_node
 * gnobog_arborescence_get_bookmarks_node_from_cusor_coords
 * gnobog_arborescence_get_bookmarks_node_from_row
 * gnobog_arborescence_get_bookmarks_node_from_arborescence_node
 */

/* NOT SO USEFUL ? ;-) */
GnobogArborescenceNode
gnobog_arborescence_get_node_from_row (GnobogArborescence* arborescence, 
                                       gint row)
{
  GnobogArborescenceNode arborescence_node;
        
  g_return_val_if_fail (arborescence != NULL, NULL);
  g_return_val_if_fail (GNOBOG_IS_ARBORESCENCE (arborescence), NULL);

  if ( row < 0 ) {
    arborescence_node = NULL;
  } else {
    arborescence_node = gtk_ctree_node_nth (GTK_CTREE (arborescence), row);
  }
        
  return arborescence_node;
}



/* THIS ONE MAY BE BETTER (but avoid doing twice the work...) */
/* NOT USED! */
GnobogArborescenceNode
gnobog_arborescence_get_node_from_cursor_coords (GnobogArborescence* arborescence, 
                                                 gint x, 
                                                 gint y)
{
  GnobogArborescenceNode  arborescence_node;
  gint                    row;
        
  g_return_val_if_fail (arborescence != NULL, NULL);
  g_return_val_if_fail (x > 0, NULL);
  g_return_val_if_fail (y > 0, NULL);
  g_return_val_if_fail (GNOBOG_IS_ARBORESCENCE (arborescence), NULL);

  row = gnobog_arborescence_get_row_from_cursor_coords (arborescence, x, y);
  if ( row == -1 )
    return NULL;
  arborescence_node = gnobog_arborescence_get_node_from_row (arborescence, row);        
  return arborescence_node;

}


GnobogArborescenceNode
gnobog_arborescence_get_arborescence_node_from_bookmarks_node (GnobogArborescence* arborescence, 
                                                               GnobogBookmarksNode bookmarks_node )
{
  GnobogArborescenceNode  arborescence_node;

  g_return_val_if_fail (arborescence != NULL, NULL);
  g_return_val_if_fail (bookmarks_node != NULL, NULL);

  return arborescence_node = gtk_ctree_find_by_row_data (GTK_CTREE (arborescence), NULL, bookmarks_node);
}

/* NOT USED! */
GnobogBookmarksNode             
gnobog_arborescence_get_bookmarks_node_from_cusor_coords (GnobogArborescence* arborescence, 
                                                          gint x, 
                                                          gint y)
{
  GnobogArborescenceNode  arborescence_node;

  g_return_val_if_fail (arborescence != NULL, NULL);
  g_return_val_if_fail (x > 0, NULL);
  g_return_val_if_fail (y > 0, NULL);

  arborescence_node = gnobog_arborescence_get_node_from_cursor_coords (arborescence, x, y);
  return gnobog_arborescence_get_bookmarks_node_from_arborescence_node (arborescence, arborescence_node);
}

GnobogBookmarksNode             
gnobog_arborescence_get_bookmarks_node_from_row (GnobogArborescence* arborescence, 
                                                 gint row)
{
  GnobogArborescenceNode  arborescence_node;

  g_return_val_if_fail (arborescence != NULL, NULL);
  g_return_val_if_fail (row >= 0, NULL);

  arborescence_node = gnobog_arborescence_get_node_from_row (arborescence, row);
  return gnobog_arborescence_get_bookmarks_node_from_arborescence_node (arborescence, arborescence_node);
}


GnobogBookmarksNode                     
gnobog_arborescence_get_bookmarks_node_from_arborescence_node (GnobogArborescence* arborescence,
                                                               GnobogArborescenceNode arborescence_node)
{
  g_return_val_if_fail (arborescence != NULL, NULL);
  g_return_val_if_fail (arborescence_node != NULL, NULL);

  return gtk_ctree_node_get_row_data (GTK_CTREE (arborescence), arborescence_node);
}
















/******************************************************************************/
/*       NODES MANIPULATION FUNCTIONS                                         */
/******************************************************************************/
/*
 * gnobog_arborescence_node_set_info
 * gnobog_arborescence_node_create_from_bookmarks_node
 * gnobog_arborescence_create_from_bookmarks_node
 * gnobog_arborescence_node_insert_into
 * gnobog_arborescence_node_insert_after
 * gnobog_arborescence_node_insert_before
 * gnobog_arborescence_node_insert
 * gnobog_arborescence_node_delete
 * gnobog_arborescence_node_modify
 */

/* Set arborescence node info (texts, pixmaps, etc...) */
static void
gnobog_arborescence_node_set_info (GnobogArborescence* arborescence,
                                   GnobogArborescenceNode arborescence_node,
                                   GnobogBookmarksNodeType bookmarks_node_type,
                                   gchar* arborescence_node_text[])
{
  GdkPixmap*  pixmap_open;
  GdkPixmap*  pixmap_closed;
  GdkBitmap*  mask_open;
  GdkBitmap*  mask_closed;
        
  /* Set icons */
  switch (bookmarks_node_type) {
  case BOOKMARK:
    pixmap_open   = pixmap_bookmark;
    pixmap_closed = pixmap_bookmark;
    mask_open     = mask_bookmark;
    mask_closed   = mask_bookmark;
    break;
  case FOLDER:
    pixmap_open   = pixmap_folder_open;
    pixmap_closed = pixmap_folder_closed;
    mask_open     = mask_folder_open;
    mask_closed   = mask_folder_closed;
    break;
  case SEPARATOR:
    pixmap_open   = pixmap_separator;
    pixmap_closed = pixmap_separator;
    mask_open     = mask_separator;
    mask_closed   = mask_separator;
    break;
  case TITLE:
    pixmap_open   = pixmap_title;
    pixmap_closed = pixmap_title;
    mask_open     = mask_title;
    mask_closed   = mask_title;
    break;
  default:
    pixmap_open   = NULL;
    pixmap_closed = NULL;
    mask_open     = NULL;
    mask_closed   = NULL;
    break;          
  }
        
  /* Set node info
   * FIXME : This function is bugged ! How do you change
   *         the text on several columns ????? */
  //      gtk_ctree_set_node_info (arborescence,
  //                               arborescence_node,
  //                               //arborescence_node_text[0],
  //                               arborescence_node_text[1],
  //                               2,
  //                               pixmap_closed /* NULL / pixmap_folder_closed*/,
  //                               mask_closed   /* NULL / mask_folder_closed*/,
  //                               pixmap_open   /* NULL / pixmap_folder_open*/,
  //                               mask_open     /* NULL / mask_folder_open*/,
  //                               FALSE,
  //                               FALSE);
        
  /* A poor replacement... */
  if ( GTK_CTREE_ROW (arborescence_node)->expanded ) {
    gtk_ctree_node_set_pixtext (GTK_CTREE (arborescence),
                                arborescence_node,
                                0,
                                arborescence_node_text[0],
                                2,
                                pixmap_open  /* NULL / pixmap_folder_open*/,
                                mask_open    /* NULL / mask_folder_open*/);
  } else {
    gtk_ctree_node_set_pixtext (GTK_CTREE (arborescence),
                                arborescence_node,
                                0,
                                arborescence_node_text[0],
                                2,
                                pixmap_closed   /* NULL / pixmap_folder_closed*/,
                                mask_closed     /* NULL / mask_folder_closed*/);
  }
  gtk_ctree_node_set_text (GTK_CTREE (arborescence),
                           arborescence_node,
                           1,
                           arborescence_node_text[1]);
  return;
}


/* Creates an ArborescenceNode from a BookmarksNode. 
 * It is inserted in 'arborescence' as child of 'parent' and as sibling of 'sibling'.
 * Returns NULL if bookmarks_node's NULL. */
static GnobogArborescenceNode
gnobog_arborescence_node_create_from_bookmarks_node (GnobogBookmarksNode bookmarks_node,
                                                     GnobogArborescence* arborescence,
                                                     GnobogArborescenceNode parent, 
                                                     GnobogArborescenceNode sibling)
{
  gchar*                  arborescence_node_text[2];
  GnobogArborescenceNode  arborescence_node;
  GdkPixmap*              pixmap_open;
  GdkPixmap*              pixmap_closed;
  GdkBitmap*              mask_open;
  GdkBitmap*              mask_closed;

  g_return_val_if_fail (bookmarks_node != NULL, NULL);
  g_return_val_if_fail (arborescence != NULL, NULL);
                
  /* FIXME : is this good ? _node_get_name() or g_strdup(entry_get_name()) */
  arborescence_node_text[0] = gnobog_bookmarks_node_get_name (bookmarks_node);
  arborescence_node_text[1] = gnobog_bookmarks_node_get_location (bookmarks_node);

  /* Set icons
   * FIXME : Hacked because of gtk_ctree_set trials... */
  switch (gnobog_bookmarks_node_get_type (bookmarks_node))  {
  case BOOKMARK:
    pixmap_open   = pixmap_bookmark;
    pixmap_closed = pixmap_bookmark;
    mask_open     = mask_bookmark;
    mask_closed   = mask_bookmark;
    break;
  case FOLDER:
    pixmap_open   = pixmap_folder_open;
    pixmap_closed = pixmap_folder_closed;
    mask_open     = mask_folder_open;
    mask_closed   = mask_folder_closed;
    break;
  case SEPARATOR:
    pixmap_open   = pixmap_separator;
    pixmap_closed = pixmap_separator;
    mask_open     = mask_separator;
    mask_closed   = mask_separator;
    break;
  case TITLE:
    pixmap_open   = pixmap_title;
    pixmap_closed = pixmap_title;
    mask_open     = mask_title;
    mask_closed   = mask_title;
    break;
  default:
    pixmap_open   = NULL;
    pixmap_closed = NULL;
    mask_open     = NULL;
    mask_closed  = NULL;
    break;          
  }
        
  /* Insert an empty arborescence_node in arborescence as a child of parent and sibling of sibling
   * Get a pointer to the empty arborescence_node in return */
  /* TODO:  WHAT'S HAPPENING IF parent IS NULL ? */
  arborescence_node = gtk_ctree_insert_node (GTK_CTREE (arborescence),
                                             parent,
                                             sibling,
                                             arborescence_node_text,
                                             2,
                                             pixmap_closed,
                                             mask_closed,
                                             pixmap_open,
                                             mask_open,
                                             FALSE,
                                             FALSE);
        
  /* store the bookmarks_node as custom data of arborescence_node in arborescence */
  gtk_ctree_node_set_row_data (GTK_CTREE (arborescence), arborescence_node, bookmarks_node);

  /* Free Arbo text */
  //  g_free (arborescence_node_text[0]);
  //  g_free (arborescence_node_text[1]);

  return arborescence_node;
}



/* Copy a BookmarksNode and its childs (recursively) into an Arborescence of
 * ArborescenceNodes, under the ArborescenceNode parent.
 * Sibling is used only for the root node (depth 0 of recursion). */
static void
gnobog_arborescence_create_from_bookmarks_node (GnobogArborescence* arborescence,
                                                GnobogArborescenceNode parent,
                                                GnobogArborescenceNode sibling,
                                                GnobogBookmarksNode bookmarks_node)
{
  GnobogArborescenceNode  child;
  GnobogBookmarksNode     bookmarks_node_child;
        
  child = gnobog_arborescence_node_create_from_bookmarks_node (bookmarks_node, 
                                                               arborescence, 
                                                               parent, 
                                                               sibling); /* May be NULL */
        
  if ( !G_NODE_IS_LEAF (bookmarks_node) ) {
    bookmarks_node_child = g_node_first_child (bookmarks_node);
    do {
      gnobog_arborescence_create_from_bookmarks_node (arborescence, 
                                                      child, 
                                                      NULL, 
                                                      bookmarks_node_child);       
    } while ((bookmarks_node_child = g_node_next_sibling (bookmarks_node_child)) != NULL);
  }               
        
  return; 
}

static void
gnobog_arborescence_node_insert_into (GnobogArborescence* arborescence, 
                                      GnobogBookmarksNode anchor_node,
                                      GnobogBookmarksNode bookmarks_node)
{
  GnobogBookmarksNodeType  anchor_node_type;
  GnobogArborescenceNode   anchor_arborescence_node;
  GnobogArborescenceNode   anchor_arborescence_node_parent;
  GnobogArborescenceNode   anchor_arborescence_node_sibling;

  g_return_if_fail (arborescence != NULL);
  g_return_if_fail (bookmarks_node != NULL);
  g_return_if_fail (GNOBOG_IS_ARBORESCENCE (arborescence));

  /* Find anchor_node parent and sibling in the arborescence */
  if ( anchor_node == NULL ) {      /* Assume root node */
    anchor_arborescence_node_parent = NULL;
    if ( (anchor_arborescence_node_sibling =
          gnobog_arborescence_get_node_from_row (arborescence, 0)) != NULL ) {
      //          g_message ("Arbo_insert_into, parent NULL and sibling : %s",
      //                 gnobog_bookmarks_node_get_name 
      //                       (gnobog_arborescence_get_bookmarks_node_from_arborescence_node 
      //                              (arborescence, 
      //                               anchor_arborescence_node_sibling)));
    } else {
      //      g_message ("Arbo_insert_into, parent NULL and sibling NULL");
      anchor_arborescence_node_sibling = NULL;
    }
  } else {        
    anchor_node_type = gnobog_bookmarks_node_get_type (anchor_node);
    if ( (anchor_node_type != TITLE) && (anchor_node_type != FOLDER) ) {
      g_message ("Doesn't make sense to paste into something that is neither a FOLDER, nor a TITLE");
      return;
    }
    anchor_arborescence_node = 
      gnobog_arborescence_get_arborescence_node_from_bookmarks_node (arborescence, anchor_node);

    //    g_message ("Arbo Insert Into - Bookmark : %s", gnobog_bookmarks_node_get_name (bookmarks_node));
                
    anchor_arborescence_node_parent = anchor_arborescence_node;   /* 1st diff with after */
    anchor_arborescence_node_sibling = GTK_CTREE_ROW (anchor_arborescence_node)->children;  /* 2nd diff */
  }

  gnobog_arborescence_create_from_bookmarks_node (arborescence,
                                                  anchor_arborescence_node_parent,
                                                  anchor_arborescence_node_sibling,
                                                  bookmarks_node);
                
  return;
}


static void
gnobog_arborescence_node_insert_after (GnobogArborescence* arborescence, 
                                       GnobogBookmarksNode anchor_node,
                                       GnobogBookmarksNode bookmarks_node)
{
  GnobogBookmarksNodeType  anchor_node_type;
  GnobogArborescenceNode   anchor_arborescence_node;
  GnobogArborescenceNode   anchor_arborescence_node_parent;
  GnobogArborescenceNode   anchor_arborescence_node_sibling;

  g_return_if_fail (arborescence != NULL);
  g_return_if_fail (bookmarks_node != NULL);
  g_return_if_fail (GNOBOG_IS_ARBORESCENCE (arborescence));

  /* Find anchor_node parent and sibling in the arborescence */
  if ( anchor_node == NULL ) {      /* Assume root node */
    anchor_arborescence_node_parent  = NULL;
    anchor_arborescence_node_sibling = NULL;                
  } else {        
    anchor_node_type = gnobog_bookmarks_node_get_type (anchor_node);
    if ( anchor_node_type == TITLE ) {
      g_message ("Insert After is not allowed on TITLE");
      return;
    }
    anchor_arborescence_node = 
      gnobog_arborescence_get_arborescence_node_from_bookmarks_node (arborescence, anchor_node);

    //    g_message ("Arbo Insert After - Bookmark : %s", gnobog_bookmarks_node_get_name (bookmarks_node));

    if ( anchor_arborescence_node == NULL ) {
      g_message ("Pb in arborescence node insert");
      return;
    }
                                
    anchor_arborescence_node_parent  = GTK_CTREE_ROW (anchor_arborescence_node)->parent;
    anchor_arborescence_node_sibling = GTK_CTREE_ROW (anchor_arborescence_node)->sibling;
  }

  /* In fact to paste after, we insert before the next sibling
   * And if there is no next sibling ????
   * In this case ->sibling is NULL,
   * so everything goes like we want ;-) */
  gnobog_arborescence_create_from_bookmarks_node (arborescence,
                                                  anchor_arborescence_node_parent,
                                                  anchor_arborescence_node_sibling,
                                                  bookmarks_node);
                
  return;
}


static void
gnobog_arborescence_node_insert_before (GnobogArborescence* arborescence, 
                                        GnobogBookmarksNode anchor_node,
                                        GnobogBookmarksNode bookmarks_node)
{
  GnobogBookmarksNodeType  anchor_node_type;
  GnobogArborescenceNode   anchor_arborescence_node;
  GnobogArborescenceNode   anchor_arborescence_node_parent;
  GnobogArborescenceNode   anchor_arborescence_node_sibling;

  g_return_if_fail (arborescence != NULL);
  g_return_if_fail (bookmarks_node != NULL);
  g_return_if_fail (GNOBOG_IS_ARBORESCENCE (arborescence));

  /* Find anchor_node parent and sibling in the arborescence */
  if ( anchor_node == NULL ) {      /* Assume root node */
    anchor_arborescence_node_parent  = NULL;
    anchor_arborescence_node_sibling = NULL;                
  } else {        
    anchor_node_type = gnobog_bookmarks_node_get_type (anchor_node);
    if ( anchor_node_type == TITLE ) {
      g_message ("Insert Before is not allowed on TITLE");
      return;
    }
    anchor_arborescence_node = 
      gnobog_arborescence_get_arborescence_node_from_bookmarks_node (arborescence, anchor_node);

    //    g_message ("Arbo Insert Before - Bookmark : %s", gnobog_bookmarks_node_get_name (bookmarks_node));
                
    anchor_arborescence_node_parent  = GTK_CTREE_ROW (anchor_arborescence_node)->parent;
    anchor_arborescence_node_sibling = anchor_arborescence_node; /* Only diff with 'after' ... */
  }

  gnobog_arborescence_create_from_bookmarks_node (arborescence,
                                                  anchor_arborescence_node_parent,
                                                  anchor_arborescence_node_sibling,
                                                  bookmarks_node);
                
  return;
}


static void
gnobog_arborescence_node_insert (GnobogArborescence* arborescence, 
                                 GnobogBookmarksNode anchor_node,
                                 GnobogBookmarksInsertMode mode,
                                 GnobogBookmarksNode bookmarks_node)
{
  g_return_if_fail (arborescence != NULL);
  g_return_if_fail (bookmarks_node != NULL);

  /* if anchor_node is NULL, root will be assumed */
        
  switch (mode) {
  case INSERT_AFTER:
    gnobog_arborescence_node_insert_after (arborescence, anchor_node, bookmarks_node);
    break;
  case INSERT_BEFORE:
    gnobog_arborescence_node_insert_before (arborescence, anchor_node, bookmarks_node);
    break;
  case INSERT_INTO:
    gnobog_arborescence_node_insert_into (arborescence, anchor_node, bookmarks_node);
    break;
  case INSERT_DEFAULT_MODE:
    /* If we are here, bookmarks' signals are broken :-( */
  default:
    /* If we are here it's not good at all ;-) */
    g_assert_not_reached();
    break;
  }
  return;
}

static void
gnobog_arborescence_node_delete (GnobogArborescence* arborescence, 
                                 GnobogBookmarksNode bookmarks_node)
{
  GnobogArborescenceNode  arborescence_node;

  g_return_if_fail (arborescence != NULL);
  g_return_if_fail (bookmarks_node != NULL);

  arborescence_node = gnobog_arborescence_get_arborescence_node_from_bookmarks_node (arborescence, 
                                                                                     bookmarks_node);
  /* This must free the row data if there's no allocated memory */
  gtk_ctree_remove_node (GTK_CTREE(arborescence), arborescence_node);     
}

static void
gnobog_arborescence_node_modify (GnobogArborescence* arborescence, 
                                 GnobogBookmarksNode bookmarks_node)
{
  GnobogArborescenceNode  arborescence_node;
  gchar*                  arborescence_node_text[2];
        
  arborescence_node = 
    gnobog_arborescence_get_arborescence_node_from_bookmarks_node (arborescence, bookmarks_node);
                
  g_message ("Node modify - Bookmark : %s", gnobog_bookmarks_node_get_name (bookmarks_node));
                
  /* FIXME : entry_get_name() or g_strdup(entry_get_name()) ? */
  arborescence_node_text[0] = gnobog_bookmarks_node_get_name (bookmarks_node);
  arborescence_node_text[1] = gnobog_bookmarks_node_get_location (bookmarks_node);
  gnobog_arborescence_node_set_info (arborescence,
                                     arborescence_node,
                                     gnobog_bookmarks_node_get_type (bookmarks_node),
                                     arborescence_node_text);
                                
  return;
}










/******************************************************************************/
/*       MISC                                                                 */
/******************************************************************************/

static void
gnobog_arborescence_open_row (GnobogArborescence* arborescence, 
                              gint row)
{
  GnobogArborescenceNode  arborescence_node;

  g_return_if_fail (arborescence != NULL);
  g_return_if_fail (row >= 0);

  arborescence_node = gnobog_arborescence_get_node_from_row (arborescence, row);
  /* FIXME : don't we need to test if it's a leaf ? */
  gtk_ctree_toggle_expansion (GTK_CTREE (arborescence), arborescence_node);
}

