/* gmoo - a gtk+ based graphical MOO/MUD/MUSH/... client
 * Copyright (C) 1999-2000 Gert Scholten
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <ctype.h>
#include <stdlib.h>

#include "config.h"

#include "userlist.h"
#include "toolbar.h"
#include "menu.h"
#include "notebook.h"
#include "settings.h"
#include "list.h"
#include "window.h"
#include "misc.h"
#include "packages.h"

#define NORMAL 0
#define AWAY   1
#define IDLE   2

#define PIXMAP_SIZE 16

typedef struct {
    int state;
    int obj;
    char *name;
    int icon;
    GList *extra;
} userinfo;

typedef struct {
    GdkPixmap *pix;
    GdkBitmap *mask;
    char *name;
} usericon;
GList *usericons = NULL;

/* userlist DO functions predeclarations: */
void userlist_do_clear(userlist_t *u);
void userlist_do_update_visible(userlist_t *u, userinfo *uf);
void userlist_do_add_visible(userlist_t *u, userinfo *uf);
void userlist_do_delete(userlist_t *u, int obj);
void userlist_do_remove(userlist_t *u, userinfo *uf);

void userlist_do_idle(userlist_t *u, int obj);
void userlist_do_unidle(userlist_t *u, int obj);
void userlist_do_away(userlist_t *u, int obj);
void userlist_do_unaway(userlist_t *u, int obj);
void userlist_do_cloak(userlist_t *u, int obj);
void userlist_do_uncloak(userlist_t *u, int obj);

userinfo *userinfo_parse(userlist_t *u, MOOVar *v);
void      userinfo_free(userinfo *uf);
userinfo *userinfo_find_visible(userlist_t *u, int obj, int *pos);
userinfo *userinfo_find_invisible(userlist_t *u, int obj);

int userinfo_compare_name(const userinfo *a, const userinfo *b);
int userinfo_compare_name_i(const userinfo *a, const userinfo *b);
int userinfo_compare_obj(const userinfo *a, const userinfo *b);
int userinfo_compare_obj_i(const userinfo *a, const userinfo *b);

usericon *usericon_load(const char *name);
void      usericon_free(usericon *i);

int is_friend(userlist_t *u, int user);
int is_gagged(userlist_t *u, int user);

#define USERIDLE     "useridle"
#define USERAWAY     "useraway"
#define USERIDLEAWAY "useridleaway"

void gm_userlist_init() {
/*    usericon_load(USERIDLE);
    usericon_load(USERAWAY);
    usericon_load(USERIDLEAWAY);*/
}

void gm_userlist_exit() {
    g_list_foreach(usericons, (GFunc) usericon_free, NULL);
    g_list_free(usericons);
}

void userlist_title_click(GtkWidget *column_but, int col, world *w);
void userlist_mouse(GtkCList *clist, GdkEventButton *e, world *w);

userlist_t *gm_userlist_new(world *w) {
    userlist_t *u;
    
    if(w->userlist) {
        if(debug) printf("Error, UL already loaded, not creating new one !!!\n");
        return NULL;
    } else if(debug) printf("Creating new userlist for world %s\n", w->p->name);

    u = g_malloc(sizeof(userlist_t));
    w->userlist = u;
    u->belongs_to = w;
    u->fields = NULL;
    u->icon_names = NULL;
    u->icons = NULL;
    u->friends = NULL;
    u->gaglist = NULL;
    u->menu = NULL;
    u->me = 0;
    u->compare = (GCompareFunc) userinfo_compare_name;
    u->users = NULL;
    u->users_invisible = NULL;
    
    gm_notebook_force_world_focused(w);
    
    return u;
}

void gm_userlist_free(userlist_t *u) { 
    if(debug) printf("free()ing userlist mcp data for world %s\n",
                     u ? u->belongs_to->p->name : "-not free()ing ??");
    if(!u) return;
    if(debug) printf("\tBeginning...\n");

    userlist_do_clear(u);
    
    u->belongs_to->userlist = NULL;
    g_list_free(u->friends);
    g_list_free(u->gaglist);
    g_list_foreach(u->fields, (GFunc) g_free, NULL);
    g_list_free(u->fields);
    g_list_foreach(u->icon_names, (GFunc) g_free, NULL);
    g_list_free(u->icon_names);
    g_list_free(u->icons);
    
    MOOVar_free(u->menu);
    
    gtk_label_set_text(GTK_LABEL(u->belongs_to->userlist_totals),
                       _("Not initialized"));
    /* Another hack, closing a world now doesn't close a userlist of another
     * world.
     */
    if(gm_notebook_get_pos(u->belongs_to) >= 0 &&
       GTK_WIDGET_VISIBLE(u->belongs_to->userlist_container)) {
        gm_userlist_toggle(u->belongs_to);
    }

    gm_notebook_force_world_focused(u->belongs_to);
    g_free(u);
    if(debug) printf("\tDone.\n");
}

void update_userlist_width(GtkWidget *userlist_container,
			   GtkAllocation *alloc, world *w) {
    w->p->userlist_width = alloc->width;
    if(debug)
	printf("Userlist resized to: %d\n", w->p->userlist_width);
}

void gm_userlist_new_widgets(world *w) {
    char *list_titles[] = {_("I"), _("Name"), _("Obj")};
    GtkWidget *scrollw;

    w->userlist_container = gtk_vbox_new(FALSE, 0);
    gtk_signal_connect(GTK_OBJECT(w->userlist_container), "size_allocate",
		       GTK_SIGNAL_FUNC(update_userlist_width), w);

    w->userlist_totals_frame = gtk_frame_new(NULL);
    gtk_box_pack_start(GTK_BOX(w->userlist_container),
		       w->userlist_totals_frame, FALSE, FALSE, 0);

    w->userlist_totals = gtk_label_new(_("Not initialized"));
    gtk_widget_show(w->userlist_totals);
    gtk_container_add(GTK_CONTAINER(w->userlist_totals_frame),
                      w->userlist_totals);
    
    scrollw = gtk_scrolled_window_new (NULL, NULL);
    gtk_widget_show(scrollw);
    gtk_box_pack_start(GTK_BOX(w->userlist_container),
                       scrollw, TRUE, TRUE, 0);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(scrollw), 
                                    GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);

    w->userlist_clist = gtk_clist_new_with_titles(3, list_titles);
    gtk_widget_show (w->userlist_clist);
    gtk_container_add (GTK_CONTAINER (scrollw), w->userlist_clist);
    GTK_WIDGET_UNSET_FLAGS (w->userlist_clist, GTK_CAN_FOCUS);
    gtk_clist_set_column_width(GTK_CLIST (w->userlist_clist), 0, PIXMAP_SIZE);
    gtk_clist_set_row_height(GTK_CLIST (w->userlist_clist), PIXMAP_SIZE);
    gtk_clist_column_title_passive(GTK_CLIST (w->userlist_clist), 0);

    gtk_signal_connect(GTK_OBJECT(w->userlist_clist),
                       "click_column", userlist_title_click, w);
    gtk_signal_connect(GTK_OBJECT(w->userlist_clist),
                       "button_press_event", userlist_mouse, w);

    gm_world_update_userlist_totals(w);
    gm_world_update_userlist_headers(w);
    gm_world_update_userlist_objects(w);
}

void userlist_update_totals(userlist_t *u) {
    int users = 0, active = 0;
    GList *l;
    char *s;
    
    for(l = u->users; l; l = g_list_next(l)) {
        users++;
        if(!(((userinfo *)(l->data))->state & IDLE))
            active++;
    }
    if(users) {
        if(active)
            s = g_strdup_printf(_("%d users, %d active"), users, active);
        else
            s = g_strdup_printf(_("%d users"), users);
        gtk_label_set_text(GTK_LABEL(u->belongs_to->userlist_totals), s);
        g_free(s);
    } else {
        gtk_label_set_text(GTK_LABEL(u->belongs_to->userlist_totals),
                           _("No users"));
    }
}

void gm_userlist_toggle_world(world *w) {
    if(!w) return;

    if(debug) printf("Toggling the userlist (for world %s)\n", w->p->name);

    if(GTK_WIDGET_VISIBLE(w->userlist_container)) {
        gtk_widget_hide(w->userlist_container);
        gm_toolbar_userlist_off(w);
        gm_menu_userlist_off(w);
    } else {
        gtk_widget_show(w->userlist_container);
        gm_toolbar_userlist_on(w);
        gm_menu_userlist_on(w);
    }
    if(settings->userlist_resizeable) {
        gm_world_update_userlist_resizeable(w);
    }
}

void gm_userlist_toggle() {
    gm_userlist_toggle_world(gm_notebook_current_world());
}

int is_friend(userlist_t *u, int user) {
    GList *l;
    for(l = u->friends; l; l = g_list_next(l)) {
        if(user == GPOINTER_TO_INT(l->data))
            return TRUE;
    }
    return FALSE;
}

int is_gagged(userlist_t *u, int user) {
    GList *l;
    for(l = u->gaglist; l; l = g_list_next(l)) {
        if(user == GPOINTER_TO_INT(l->data))
            return TRUE;
    }
    return FALSE;
}

void do_set_friend(userlist_t *u, int row, int obj) {
    if(is_friend(u, obj) && settings->userlist_friends_color < BRIGHT) {
        gtk_clist_set_foreground(GTK_CLIST(u->belongs_to->userlist_clist),
                                 row,
                                 &colors[CLAMP(settings->userlist_friends_color + 
                                               BRIGHT,
                                               0, N_COLORS)]);
    } else
        gtk_clist_set_foreground(GTK_CLIST(u->belongs_to->userlist_clist),
                                 row, NULL);
}

void gm_userlist_update_friends_color(userlist_t *u) {
    GList *l;
    int i = 0;

    gm_userlist_toggle_world(u->belongs_to);
    for(l = u->users; l; l = g_list_next(l)) {
        do_set_friend(u, i, ((userinfo *)(l->data))->obj);
        i++;
    }
    gm_userlist_toggle(u->belongs_to);
}

void gm_userlist_update_friends(userlist_t *u) {
    GList *l;
    int i = 0;

    for(l = u->users; l; l = g_list_next(l)) {
        do_set_friend(u, i, ((userinfo *)(l->data))->obj);
        i++;
    }
}

userinfo *userinfo_parse(userlist_t *u, MOOVar *v) {
    userinfo *uf;
    if(v->type == LIST && v->i >= 3 &&
       v->list->type == OBJECT &&
       v->list->next->type == STRING &&
       v->list->next->next->type == INT) {
        uf = g_malloc(sizeof(userinfo));
        v = v->list;
        uf->obj = v->i;
        v = v->next;
        uf->name = g_strdup(v->s);
        v = v->next;
        uf->icon = v->i;
        uf->extra = NULL;
        for(v = v->next; v; v = v->next) {
            uf->extra = g_list_append(uf->extra, MOOVar_tostr(v));
        }
        uf->state = NORMAL;
        return uf;
    }
    return NULL;
}

void userinfo_free(userinfo *uf) {
    g_free(uf->name);
    g_list_foreach(uf->extra, (GFunc) g_free, NULL);
    g_list_free(uf->extra);
    g_free(uf);
}

userinfo *userinfo_find_visible(userlist_t *u, int obj, int *pos) {
    GList *l;
    for(l = u->users; l; l = g_list_next(l)) {
        if(((userinfo *)(l->data))->obj == obj) {
            if(pos) *pos = g_list_position(u->users, l);
            return l->data;
        }
    }
    return NULL;
}

userinfo *userinfo_find_invisible(userlist_t *u, int obj) {
    GList *l;
    for(l = u->users_invisible; l; l = g_list_next(l)) {
        if(((userinfo *)(l->data))->obj == obj) {
            return l->data;
        }
    }
    return NULL;
}

char *userinfo_get_nth(const userinfo *uf, int i) {
    char *s = NULL;
    static char *buf = NULL;

    g_free(buf);
    buf = NULL;
    
    if(i == 1) s = buf = g_strdup_printf("#%d", uf->obj);
    else if(i == 2) s = uf->name;
    else if(i == 3) s = buf = g_strdup_printf("%d", uf->icon);
    else if(i > 3) s = g_list_nth_data(uf->extra, i - 4);

    if(!s) s = "";
    return s;
}

char *get_string_on_usermenu(userlist_t *ul, userinfo *uf) {
    stream_t *stream;
    char *s;

    stream = stream_new(80);
    stream_append_string(stream, _("User: "));
    stream_append_string(stream, uf->name);

    stream_append_string(stream, _("\nObject: "));
    s = g_strdup_printf("#%d", uf->obj);
    stream_append_string(stream, s);
    g_free(s);

    stream_append_string(stream, _("\nType: "));
    s = g_list_nth_data(ul->icon_names, MAX(uf->icon - 1, 0));
    stream_append_string(stream, s ? s : _("Unknown"));

    s = stream->buf;
    g_free(stream);
    return s;
}

int userinfo_compare_name(const userinfo *a, const userinfo *b) {
    if((a->state & IDLE) == (b->state & IDLE))
        return g_strcasecmp(a->name, b->name);
    return a->state & IDLE ? 1 : -1;
}

int userinfo_compare_name_i(const userinfo *a, const userinfo *b) {
    if((a->state & IDLE) == (b->state & IDLE))
        return g_strcasecmp(b->name, a->name);
    return a->state & IDLE ? 1 : -1;
}

int userinfo_compare_obj(const userinfo *a, const userinfo *b) {
    if((a->state & IDLE) == (b->state & IDLE))
        return a->obj - b->obj;
    return a->state & IDLE ? 1 : -1;
}

int userinfo_compare_obj_i(const userinfo *a, const userinfo *b) {
    if((a->state & IDLE) == (b->state & IDLE))
        return b->obj - a->obj;
    return a->state & IDLE ? 1 : -1;
}



void userlist_add(userlist_t *u, MOOVar *v) {
    userinfo *uf;
    if((uf = userinfo_parse(u, v))) {
        if(!userinfo_find_visible(u, uf->obj, NULL) &&
           !userinfo_find_invisible(u, uf->obj)) {
            if(debug) printf("Adding user #%d (%s)\n", uf->obj, uf->name);
            if(is_gagged(u, uf->obj)) 
                u->users_invisible = g_list_append(u->users_invisible, uf);
            else
                userlist_do_add_visible(u, uf);
        } else {
            if(debug) printf("User #%d (%s) is alread in the list !\n",
                             uf->obj, uf->name);
            userinfo_free(uf);
        }
    }
}

void userlist_add_many(userlist_t *u, MOOVar *v) {
    if(v->type == LIST)
        for(v = v->list; v; v = v->next)
            userlist_add(u, v);
}

void userlist_delete(userlist_t *u, MOOVar *v) {
    for(v = v->list; v; v = v->next) {
        if(v->type == OBJECT) {
            userlist_do_delete(u, v->i);
        }
    }
}

void userlist_modify(userlist_t *u, MOOVar *v) {
    userinfo *uf, *current;
    
    if((uf = userinfo_parse(u, v))) {
        if((current = userinfo_find_visible(u, uf->obj, NULL))) { 
            if(debug) printf("Updating user #%d (%s)\n", uf->obj, current->name);
            uf->state = current->state;
            userlist_do_delete(u, current->obj);
            userlist_do_add_visible(u, uf);
        } else if((current = userinfo_find_invisible(u, uf->obj))) {
            if(debug) printf("Updating user #%d (%s)\n", uf->obj, current->name);
            uf->state = current->state;
            userlist_do_delete(u, current->obj);
            u->users_invisible = g_list_prepend(u->users_invisible, uf);
        } else {
            userinfo_free(uf);
        }
    }
}

void userlist_idle(userlist_t *u, MOOVar *v) {
    if(v->type == LIST)
        for(v = v->list; v; v = v->next)
            if(v->type == OBJECT)
                userlist_do_idle(u, v->i);
}

void userlist_unidle(userlist_t *u, MOOVar *v) {
    if(v->type == LIST)
        for(v = v->list; v; v = v->next)
            if(v->type == OBJECT)
                userlist_do_unidle(u, v->i);
}

void userlist_away(userlist_t *u, MOOVar *v) {
    if(v->type == LIST)
        for(v = v->list; v; v = v->next)
            if(v->type == OBJECT)
                userlist_do_away(u, v->i);
}

void userlist_unaway(userlist_t *u, MOOVar *v) {
    if(v->type == LIST)
        for(v = v->list; v; v = v->next)
            if(v->type == OBJECT)
                userlist_do_unaway(u, v->i);
}

void userlist_cloak(userlist_t *u, MOOVar *v) {
    if(v->type == LIST)
        for(v = v->list; v; v = v->next)
            if(v->type == OBJECT)
                userlist_do_cloak(u, v->i);
}

void userlist_uncloak(userlist_t *u, MOOVar *v) {
    if(v->type == LIST)
        for(v = v->list; v; v = v->next)
            if(v->type == OBJECT)
                userlist_do_uncloak(u, v->i);
}

void userlist_handle_data(userlist_t *u, const char *line) {
    MOOVar *v;
    
    if(!(u->fields && u->icons)) return;

    while(*line && isspace(*line)) line++;
    if(debug) printf("UL-DATA: %s\n", line);

    if((v = MOOVar_parse(line + 1))) {
        switch(*line) {
        case '=':
            userlist_do_clear(u);
            userlist_add_many(u, v);
            break;
        case '+':
            userlist_add(u, v);
            break;
        case '*':
            userlist_modify(u, v);
            break;
        case '-':
            userlist_delete(u, v);
            break;
        case '<':
            userlist_idle(u, v);
            break;
        case '>':
            userlist_unidle(u, v);
            break;
        case '[':
            userlist_away(u, v);
            break;
        case ']':
            userlist_unaway(u, v);
            break;
        case '(':
            userlist_cloak(u, v);
            break;
        case ')':
            userlist_uncloak(u, v);
            break;
        default:
            if(debug) printf("Error in UL command, %c not reconised\n", *line);
            MOOVar_free(v);
            return;
        }
        MOOVar_free(v);
        userlist_update_totals(u);
    } else if(debug) {
        printf("Error parseing UL argument !\n");
    }
#if 0
    print_ul_sync(u);
#endif
}

void userlist_handle_fields(userlist_t *u, const char *line) {
    MOOVar *v;
    MOOVar *v1, *v2, *v3;

    if(u->fields) return;
    
    v = MOOVar_parse(line);
    v1 = MOOVar_listindex(v, 1);
    v2 = MOOVar_listindex(v, 2);
    v3 = MOOVar_listindex(v, 3);
    
    if(v && v1 && v2 && v3 &&
       v1->type == STRING && g_strcasecmp(v1->s, "Object") == 0 &&
       v2->type == STRING && g_strcasecmp(v2->s, "Name") == 0 &&
       v3->type == STRING && g_strcasecmp(v3->s, "Icon") == 0) {
        for(v2 = v->list; v2; v2 = v2->next) {
            u->fields = g_list_append(u->fields, g_strdup(v2->s));
            if(debug) printf("Added field: %s\n", v2->s ? v2->s : "-error-");
        }
    }
    MOOVar_free(v);
}

void userlist_handle_icons(userlist_t *u, const char *line) {
    MOOVar *v;
    MOOVar *v2;

    if(u->icon_names) return;
    
    if((v = MOOVar_parse(line))) {
        for(v2 = v->list; v2; v2 = v2->next) {
            u->icon_names = g_list_append(u->icon_names, g_strdup(v2->s));
            u->icons = g_list_append(u->icons, usericon_load(v2->s));
            if(debug) printf("Added icon: %s => %s\n", v2->s ? v2->s : "-error-",
                             g_list_last(u->icons)->data 
                             ? "\x1b[32mfound\x1b[0m !" :
                             "\x1b[31mNOT\x1b[0m found :((");
        }
        MOOVar_free(v);
    }
}

void userlist_handle_command(mcp_multiline *m,
                             const char *name, const char *line) {
    userlist_t *u = m->data;

    if(!u) return;

    if(g_strcasecmp(name, "d") == 0)
        userlist_handle_data(u, line);
    else if(g_strcasecmp(name, "fields") == 0)
        userlist_handle_fields(u, line);
    else if(g_strcasecmp(name, "icons") == 0)
        userlist_handle_icons(u, line);
    else if(debug) {
        printf("Unknown userlist command: %s\n", name);
    }
}

void userlist_try_friends(mcp_package *p, int argc, char **argv) {
    int i;
    char *friends = NULL;
    MOOVar *v, *v2;
    userlist_t *u = p->data;
    
    for(i = 0; i < argc; i += 2) {
        if(g_strcasecmp(argv[i], "friends") == 0)
            friends = argv[i + 1];
    }
    if(friends) {
        v = MOOVar_parse(friends);
        if(v && v->type == LIST) {
            g_list_free(u->friends);
            u->friends = NULL;
            for(v2 = v->list; v2; v2 = v2->next) {
                if(v2->type == OBJECT) {
                    if(debug) printf("Adding friend: #%d\n", v2->i);
                    u->friends = g_list_prepend(u->friends,
                                                GINT_TO_POINTER(v2->i));
                }
            }
            gm_userlist_update_friends(u);
        }
        MOOVar_free(v);
    }
}

void do_gag_users(userlist_t *u) {
    userinfo *uf;
    GList *l;
    for(l = u->gaglist; l; l = g_list_next(l)) {
        if((uf = userinfo_find_visible(u, GPOINTER_TO_INT(l->data), NULL))) {
            userlist_do_remove(u, uf);
            u->users_invisible = g_list_append(u->users_invisible, uf);
        }
    }
}


void do_ungag_users(userlist_t *u) {
    userinfo *uf;
    GList *l;
    for(l = u->gaglist; l; l = g_list_next(l)) {
        if((uf = userinfo_find_invisible(u, GPOINTER_TO_INT(l->data)))) {
            u->users_invisible = g_list_remove(u->users_invisible, uf);
            userlist_do_add_visible(u, uf);
        }
    }
}


void userlist_try_gaglist(mcp_package *p, int argc, char **argv) {
    int i;
    char *gaglist = NULL;
    MOOVar *v, *v2;
    userlist_t *u = p->data;
    
    for(i = 0; i < argc; i += 2) {
        if(g_strcasecmp(argv[i], "gaglist") == 0)
            gaglist = argv[i + 1];
    }
    if(gaglist) {
        v = MOOVar_parse(gaglist);
        if(v && v->type == LIST) {
            do_ungag_users(u);
            g_list_free(u->gaglist);
            u->gaglist = NULL;
            for(v2 = v->list; v2; v2 = v2->next) {
                if(v2->type == OBJECT) {
                    if(debug) printf("Adding gagged user: #%d\n", v2->i);
                    u->gaglist = g_list_prepend(u->gaglist,
                                                GINT_TO_POINTER(v2->i));
                }
            }
            do_gag_users(u);
        }
        MOOVar_free(v);
    }
}

void userlist_try_menu(mcp_package *p, int argc, char **argv) {
    int i;
    char *menu = NULL;
    char *def = NULL;
    MOOVar *v;
    userlist_t *u = p->data;
    
    for(i = 0; i < argc; i += 2) {
        if(g_strcasecmp(argv[i], "menu") == 0)
            menu = argv[i + 1];
        else if(g_strcasecmp(argv[i], "default") == 0)
            def = argv[i + 1];
    }
    if(menu) {
        if((v = MOOVar_parse(menu)) && v->type == LIST) {
            MOOVar_free(u->menu);
            u->menu = v;
            u->menudefault = def ? atoi(def) : 0;
            if(debug) printf("\tUserlist menu registered\n");
        }
    }
}

void userlist_try_you(mcp_package *p, int argc, char **argv) {
    int i;
    char *nr = NULL;
    MOOVar *v;
    userlist_t *u = p->data;
    
    for(i = 0; i < argc; i += 2) {
        if(g_strcasecmp(argv[i], "nr") == 0)
            nr = argv[i + 1];
    }
    if(nr) {
        v = MOOVar_parse(nr);
        if(v && v->type == OBJECT) {
            u->me = v->i;
            if(debug) printf("Own object number set to: #%d\n", u->me);
        }
        MOOVar_free(v);
    }
}

void userlist_try_init(mcp_package *p, int argc, char **argv) {
    char *fields = NULL, *icons = NULL, *d = NULL, *data_tag = NULL;
    world *w = ((userlist_t *)(p->data))->belongs_to;
    int i;
    
    for(i = 0; i < argc; i += 2) {
        if(g_strcasecmp(argv[i], "fields*") == 0)
            fields = argv[i + 1];
        else if(g_strcasecmp(argv[i], "icons*") == 0)
            icons = argv[i + 1];
        else if(g_strcasecmp(argv[i], "d*") == 0)
            d = argv[i + 1];
        else if(g_strcasecmp(argv[i], "_data-tag") == 0)
            data_tag = argv[i + 1];
    }
    if(fields && icons && d && data_tag) {
        gm_mcp_multiline_new(p, data_tag, p->data,
                             userlist_handle_command,
                             NULL);
        if(settings->userlist_auto_open &&
           !GTK_WIDGET_VISIBLE(w->userlist_container)) {
            gm_userlist_toggle_world(w);
        }
        return;
    }
}

/* mcp interface */
void userlist_mcp_handle(mcp_package *p, const char *command,
                         int argc, char **argv) {
    if(debug) printf("Got userlist command: %s\n", command);
    
    if(g_strcasecmp(command, "friends") == 0)
        userlist_try_friends(p, argc, argv);
    else if(g_strcasecmp(command, "gaglist") == 0)
        userlist_try_gaglist(p, argc, argv);
    else if(g_strcasecmp(command, "menu") == 0)
        userlist_try_menu(p, argc, argv);
    else if(g_strcasecmp(command, "you") == 0)
        userlist_try_you(p, argc, argv);
    else if(g_strcasecmp(command, "") == 0)
        userlist_try_init(p, argc, argv);
    else if(debug) {
        printf("Unknowuserlist command: %s\n", command);
    }
}

void userlist_mcp_free(mcp_package *p) {
    gm_userlist_free(p->data);
}

void gm_userlist_mcp_init(mcp_package *p) {
    p->data = gm_userlist_new(p->session->belongs_to);
    p->handle = p->data ? userlist_mcp_handle : NULL;
    p->free = p->data ? userlist_mcp_free : NULL;
}

/********************************************************************************/
/****** functions dat actually DO the work :)  */


void userlist_do_clear(userlist_t *u) {
    if(debug) printf("Clearing userlist\n");
    gtk_clist_clear(GTK_CLIST(u->belongs_to->userlist_clist));

    g_list_foreach(u->users, (GFunc) userinfo_free, NULL);
    g_list_foreach(u->users_invisible, (GFunc) userinfo_free, NULL);
    g_list_free(u->users);
    g_list_free(u->users_invisible);
    
    u->users = NULL;
    u->users_invisible = NULL;
}

void userlist_do_update_visible(userlist_t *u, userinfo *uf) {
    int pos;
    GtkCList *clist = GTK_CLIST(u->belongs_to->userlist_clist);
    char unknown[] = "?";
    char s[10];
    usericon *i;
    
    if((pos = g_list_index(u->users, uf)) < 0) {
        if(debug) printf("Update visible(): no sutch user :(\n");
        return;
    }
    
    do_set_friend(u, pos, uf->obj);

    if(uf->state & IDLE) {
        if(uf->state & AWAY) {
            i = usericon_load(USERIDLEAWAY);
        } else {
            i = usericon_load(USERIDLE);
        }
        if(i)
            gtk_clist_set_pixmap(clist, pos, 0, i->pix, i->mask);
        else
            gtk_clist_set_text(clist, pos, 0, unknown);
    } else if(uf->state & AWAY) {
        if((i = usericon_load(USERAWAY)))
            gtk_clist_set_pixmap(clist, pos, 0, i->pix, i->mask);
        else
            gtk_clist_set_text(clist, pos, 0, unknown);
    } else if((i = g_list_nth_data(u->icons, uf->icon - 1))) {
        gtk_clist_set_pixmap(clist, pos, 0, i->pix, i->mask);
    } else {
        if(uf->icon != CLAMP(uf->icon, 1, g_list_length(u->icons)))
            gtk_clist_set_text(clist, pos, 0, "");
        else
            gtk_clist_set_text(clist, pos, 0, unknown);
    }
    
    gtk_clist_set_text(clist, pos, 1, uf->name);

    sprintf(s, "#%d", uf->obj);
    gtk_clist_set_text(clist, pos, 2, s);
}

void userlist_do_add_visible(userlist_t *u, userinfo *uf) {
    int pos;
    char *empty[]  = {"", "", ""};
    GtkWidget *clist = u->belongs_to->userlist_clist;

    u->users = g_list_insert_sorted(u->users, uf, u->compare);

    pos = g_list_index(u->users, uf);    

    if(debug)
        printf("* added user #%d (%s) to pos %d\n", uf->obj, uf->name, pos);

    gtk_clist_insert(GTK_CLIST(clist), pos, empty);
    userlist_do_update_visible(u, uf);
}

void userlist_do_delete(userlist_t *u, int obj) {
    userinfo *uf;
    int pos;
    if((uf = userinfo_find_visible(u, obj, &pos))) {
        if(debug) printf("Deleting user #%d\n", uf->obj);
        gtk_clist_remove(GTK_CLIST(u->belongs_to->userlist_clist), pos);
        u->users = g_list_remove(u->users, uf);
        userinfo_free(uf);
    } else if((uf = userinfo_find_invisible(u, obj))) {
        if(debug) printf("Deleting user #%d\n", uf->obj);
        u->users_invisible = g_list_remove(u->users_invisible, uf);
        userinfo_free(uf);
    }
}

void userlist_do_remove(userlist_t *u, userinfo *uf) {
    int pos = g_list_index(u->users, uf);
    
    if(pos >= 0) {
        gtk_clist_remove(GTK_CLIST(u->belongs_to->userlist_clist), pos);
        u->users = g_list_remove(u->users, uf);
        if(debug)
            printf("* removed user #%d (%s) from pos %d\n",uf->obj,uf->name,pos);
    } else {
        u->users_invisible = g_list_remove(u->users_invisible, uf);
    }
}

void userlist_do_idle(userlist_t *u, int obj) {
    userinfo *uf;
    if((uf = userinfo_find_visible(u, obj, NULL))) {
        if(debug) printf("Idleing user %d (%s)\n", uf->obj, uf->name);
        userlist_do_remove(u, uf);
        uf->state |= IDLE;
        userlist_do_add_visible(u, uf);
    } else if((uf = userinfo_find_invisible(u, obj))) { 
        if(debug) printf("Idleing invisible user %d (%s)\n", uf->obj, uf->name);
        uf->state |= IDLE;
    }
}

void userlist_do_unidle(userlist_t *u, int obj) {
    userinfo *uf;
    if((uf = userinfo_find_visible(u, obj, NULL))) {
        if(debug) printf("Unidleing user %d (%s)\n", uf->obj, uf->name);
        userlist_do_remove(u, uf);
        uf->state &= ~IDLE;
        userlist_do_add_visible(u, uf);
    } else if((uf = userinfo_find_invisible(u, obj))) {
        if(debug) printf("Unidleing invisible user %d (%s)\n", uf->obj, uf->name);
        uf->state &= ~IDLE;
    }
}

void userlist_do_away(userlist_t *u, int obj) {
    userinfo *uf;
    if((uf = userinfo_find_visible(u, obj, NULL))) {
        if(debug) printf("Awaying user %d (%s)\n", uf->obj, uf->name);
        uf->state |= AWAY;
        userlist_do_update_visible(u, uf);
    } else if((uf = userinfo_find_invisible(u, obj))) { 
        if(debug) printf("Awaying invisible user %d (%s)\n", uf->obj, uf->name);
        uf->state |= AWAY;
    }
}

void userlist_do_unaway(userlist_t *u, int obj) {
    userinfo *uf;
    if((uf = userinfo_find_visible(u, obj, NULL))) {
        if(debug) printf("Unawaying user %d (%s)\n", uf->obj, uf->name);
        uf->state &= ~AWAY;
        userlist_do_update_visible(u, uf);
    } else if((uf = userinfo_find_invisible(u, obj))) { 
        if(debug) printf("Unawaying invisible user %d (%s)\n", uf->obj, uf->name);
        uf->state &= ~AWAY;
    }
}

void userlist_do_cloak(userlist_t *u, int obj) {
    userinfo *uf;
    if((uf = userinfo_find_visible(u, obj, NULL))) {
        if(debug) printf("Cloaking user %d (%s)\n", uf->obj, uf->name);
        userlist_do_remove(u, uf);
        u->users_invisible = g_list_prepend(u->users_invisible, uf);
    }
}

void userlist_do_uncloak(userlist_t *u, int obj) {
    userinfo *uf;
    if((uf = userinfo_find_invisible(u, obj)) &&
       !is_gagged(u, obj)) {
        if(debug) printf("Uncloaking user %d (%s)\n", uf->obj, uf->name);
        u->users_invisible = g_list_remove(u->users_invisible, uf);
        userlist_do_add_visible(u, uf);
    } else if (debug)
        printf("User %d sn't cloaked\n", obj);
}


usericon *usericon_load(const char *name_str) {
    char *file = NULL;
    GdkPixmap *pix;
    GdkBitmap *mask;
    usericon *icon;
    GList *l;
    char *name = str_tolower(g_strdup(name_str));

    for(l = usericons; l; l = g_list_next(l)) {
        icon = l->data;
        if(g_strcasecmp(name, icon->name) == 0) {
            if(debug)
                printf("\x1b[36mUsericon %s already loaded.\x1b[0m\n", name);
            g_free(name);
            return icon;
        }
    }
    icon = NULL;
    
    file = g_strconcat(SHAREDIR"/gmoo/pixmaps/", name, ".xpm", NULL);
    pix = gdk_pixmap_create_from_xpm(gm_window_get_GdkWindow(), &mask,
                                     gm_window_get_background_color(),
                                     file);
    if(!pix) {
        if(debug) printf("\x1b[36mUsericon %s NOT found in %s:((\x1b[0m\n",
                         name, file);
        g_free(file);
        file = g_strconcat(gm_settings_get_pixmap_dir(), "/", name, ".xpm", NULL);
        pix = gdk_pixmap_create_from_xpm(gm_window_get_GdkWindow(), &mask,
                                         gm_window_get_background_color(),
                                         file);
    }
    if(pix) {
        if(debug) printf("\x1b[36mUsericon %s found in %s\x1b[0m\n", name, file);
        icon = g_malloc(sizeof(usericon));
        icon->pix = pix;
        icon->mask = mask;
        icon->name = g_strdup(name);
        gdk_pixmap_ref(pix);
        gdk_bitmap_ref(mask);
        usericons = g_list_append(usericons, icon);
    } else if(debug) {
        printf("\x1b[36mUsericon %s NOT found in %s:((\x1b[0m",
               name, file);
    }
    
    g_free(file);
    g_free(name);
    return icon;
}

    
void usericon_free(usericon *i) {
    if(i) {
        g_free(i->name);
        gdk_pixmap_unref(i->pix);
        gdk_bitmap_unref(i->mask);
        g_free(i);
    }
}

void userlist_title_click(GtkWidget *column_but, int col, world *w) {
    GList *l, *list;
    userlist_t *u = w->userlist;
    
    if(col == 1) {
        if(u->compare == (GCompareFunc) userinfo_compare_name)
            u->compare = (GCompareFunc) userinfo_compare_name_i;
        else
            u->compare = (GCompareFunc) userinfo_compare_name;
    } else if(col == 2) {
        if(u->compare == (GCompareFunc) userinfo_compare_obj)
            u->compare = (GCompareFunc) userinfo_compare_obj_i;
        else
            u->compare = (GCompareFunc) userinfo_compare_obj;
    } else {
        u->compare = (GCompareFunc) userinfo_compare_name;
    }
    
    if(u->users) {
        gtk_clist_freeze(GTK_CLIST(w->userlist_clist));
        gtk_clist_clear(GTK_CLIST(w->userlist_clist));
        list = u->users;
        u->users = NULL;
        for(l = list; l; l = g_list_next(l)) {
            userlist_do_add_visible(u, l->data);
        }
        g_list_free(list);
        gtk_clist_thaw(GTK_CLIST(w->userlist_clist));
    }
}

int do_destroy_menu(GtkWidget *menu) {
    if(debug) printf("Userlistmenu gone !\n");
    gtk_widget_destroy(menu);
    return FALSE;
}

void userlist_menu_hidden(GtkWidget *menu, world *w) {
    if(debug) printf("Userlistmenu hidden ...\n");
    gtk_timeout_add(0, (GtkFunction) do_destroy_menu, menu);
}

void userlist_menu_item_destroyed(GtkWidget *item) {
    char *command = gtk_object_get_user_data(GTK_OBJECT(item));
    if(debug) printf("\tMenuitem destroyed\n");
    g_free(command);
}

void userlist_menu_item_clicked(GtkWidget *item, world *w) {
    char *command = gtk_object_get_user_data(GTK_OBJECT(item));
    if(debug) printf("menu item clicked !\n\tSending: %s\n", command);
    gm_world_writeln(w, command);
}

void userlist_update_friends(world *w) {
    stream_t *stream = stream_new(40);
    GList *l;
    char s[10];

    stream_append(stream, '{');
    for(l = w->userlist->friends; l; l = g_list_next(l)) {
        if(l != w->userlist->friends)
            stream_append(stream, ',');
        stream_append(stream, '#');
        sprintf(s, "%d", GPOINTER_TO_INT(l->data));
        stream_append_string(stream, s);
    }
    stream_append(stream, '}');
    gm_mcp_update_friends(w, stream->buf);
    stream_free(stream);
}

void userlist_add_friend(GtkWidget *item, world *w) {
    void *obj = gtk_object_get_user_data(GTK_OBJECT(item));
    if(debug) printf("Marking user #%d as a friend.\n", (int)obj);
    if(w->userlist) {
        w->userlist->friends = g_list_append(w->userlist->friends, obj);
        gm_userlist_update_friends(w->userlist);
        userlist_update_friends(w);
    }
}

void userlist_remove_friend(GtkWidget *item, world *w) {
    void *obj = gtk_object_get_user_data(GTK_OBJECT(item));
    if(debug) printf("UNmarking user #%d as a friend.\n", (int)obj);
    if(w->userlist) {
        w->userlist->friends = g_list_remove(w->userlist->friends, obj);
        gm_userlist_update_friends(w->userlist);
        userlist_update_friends(w);
    }
}


char *do_parse_menu_str(userinfo *uf, const char *unparsed) {
    stream_t *stream;
    char *s;
    int i;

    stream = stream_new(40);
    for(i = 0; *unparsed; i++) {
        if(*unparsed == '$') {
            unparsed++;
            if(!*unparsed)
                break;
            if(*unparsed == '$')
                stream_append(stream, '$');
            else if(*unparsed == '(') {
                unparsed++;
                while(isspace(*unparsed)) unparsed++;
                if(*unparsed == 'n') {
                    stream_append(stream, '\n');
                    unparsed++;
                } else {
                    if(isdigit(*unparsed)) {
                        i = 0;
                        while(isdigit(*unparsed)) {
                            i *= 10;
                            i += *unparsed - '0';
                            unparsed++;
                        }
                        stream_append_string(stream, userinfo_get_nth(uf, i));
                    }
                }
                while(*unparsed && *unparsed != ')')
                    unparsed++;
                if(*unparsed == ')')
                    unparsed++;
            }
        } else if(*unparsed == '&') {
            unparsed++;
            if(*unparsed == '&') {
                stream_append(stream, '&');
                unparsed++;
            }
        } else {
            stream_append(stream, *unparsed);
            unparsed++;
        }
    }
    s = stream->buf;
    g_free(stream);

    return s;
}

void do_add_menu_item(GtkMenu *menu, world *w, userinfo *uf, MOOVar *v) {
    GtkWidget *item;
    char *command, *desc;
    
    if(debug) printf("adding type: %d, length == %d\n", v->type, v->i);
    
    if(v->type == LIST && v->i == 2 &&
       v->list->type == STRING && v->list->next->type == STRING) {
        desc = do_parse_menu_str(uf, v->list->s);
        command = do_parse_menu_str(uf, v->list->next->s);
        
        item = gtk_menu_item_new_with_label(desc);
        gtk_signal_connect(GTK_OBJECT(item), "destroy",
                           GTK_SIGNAL_FUNC(userlist_menu_item_destroyed), NULL);
        gtk_signal_connect(GTK_OBJECT(item), "activate",
                           GTK_SIGNAL_FUNC(userlist_menu_item_clicked), w);
        g_free(desc);
        gtk_object_set_user_data(GTK_OBJECT(item), command);
    } else {
        item = gtk_menu_item_new();
    }
    gtk_widget_show(item);
    gtk_menu_append(menu, item);
}

void userlist_do_doubleclick(world *w, userinfo *uf) {
    userlist_t *u;
    MOOVar *def;
    char *command;

    if(!w) {
        if(debug) printf("NO WORLD !!!!!!!!!!!!!\n");
        return;
    } else if(!(u = w->userlist)) {
        if(debug) printf("NO USERLIST !!!!!!!!!!!!!\n");
        return;
    }
    def = MOOVar_listindex(u->menu, u->menudefault);
    if(!def) {
        if(debug) printf("No default menu found\n");
        return;
    }
    
    if(def->type == LIST && def->i == 2 && def->list->next->type == STRING) {
        command = do_parse_menu_str(uf, def->list->next->s);
        if(debug) printf("\t=> %s", command);
        gm_world_writeln(w, command);
        g_free(command);
    } else if(debug) {
        printf("Error in UL-menu :(\n");
    }
}

void userlist_mouse(GtkCList *clist, GdkEventButton *e, world *w) {
    int row = -1;
    MOOVar *v;
    GtkWidget *menu, *item, *submenu;
    userinfo *uf;
    char *s;
    
#if 0
    print_ul_sync(w->userlist);
#endif

    gtk_clist_get_selection_info(clist, e->x, e->y, &row, NULL);
    if(row >= 0 && e->button == 3) {
        if(debug) printf("Right clicked on row %d\n", row);
        uf = g_list_nth_data(w->userlist->users, row);
        if(!uf) {
            if(debug) printf("-> Error, no such row ! ??\n\n");
            return;
        }
        if(debug) printf("-> Row %d is user #%d (%s)\n", row, uf->obj, uf->name);

        menu = gtk_menu_new();
        gtk_widget_show(menu);
        gtk_signal_connect_after(GTK_OBJECT(menu), "hide",
                                 GTK_SIGNAL_FUNC(userlist_menu_hidden), w);

        item = gtk_menu_item_new_with_label(uf->name);
        gtk_widget_show(item);
        gtk_menu_append(GTK_MENU(menu), item);



        submenu = gtk_menu_new();
        gtk_widget_show(submenu);
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), submenu);

        s = get_string_on_usermenu(w->userlist, uf);
        item = gtk_menu_item_new_with_label(s);
        gtk_label_set_justify(GTK_LABEL(GTK_BIN(item)->child), GTK_JUSTIFY_LEFT);
        gtk_widget_show(item);
        gtk_menu_append(GTK_MENU(submenu), item);
        g_free(s);

        item = gtk_menu_item_new();
        gtk_widget_show(item);
        gtk_menu_append(GTK_MENU(submenu), item);

        item = gtk_menu_item_new_with_label(_("Mark as friend"));
        gtk_widget_show(item);
        gtk_menu_append(GTK_MENU(submenu), item);
        gtk_widget_set_sensitive(item, !is_friend(w->userlist, uf->obj));
        gtk_signal_connect(GTK_OBJECT(item), "activate",
                           GTK_SIGNAL_FUNC(userlist_add_friend), w);
        gtk_object_set_user_data(GTK_OBJECT(item), GINT_TO_POINTER(uf->obj));
        
        item = gtk_menu_item_new_with_label(_("Unmark as friend"));
        gtk_widget_show(item);
        gtk_menu_append(GTK_MENU(submenu), item);
        gtk_widget_set_sensitive(item, is_friend(w->userlist, uf->obj));
        gtk_signal_connect(GTK_OBJECT(item), "activate",
                           GTK_SIGNAL_FUNC(userlist_remove_friend), w);
        gtk_object_set_user_data(GTK_OBJECT(item), GINT_TO_POINTER(uf->obj));

        
        item = gtk_menu_item_new();
        gtk_widget_show(item);
        gtk_menu_append(GTK_MENU(menu), item);
        
        for(v = w->userlist->menu->list; v; v = v->next) {
            do_add_menu_item(GTK_MENU(menu), w, uf, v);
        }
 
        gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, 2, 0);
    } else if(row >= 0 && e->button == 1 &&
              (e->type==GDK_2BUTTON_PRESS || e->type==GDK_3BUTTON_PRESS)) {
        if(debug) printf("Double clicked on row %d\n", row);
        uf = g_list_nth(w->userlist->users, row)->data;
        userlist_do_doubleclick(w, uf);
    }
}

