from DisplayContainer import DisplayContainer
from GlassWindow import GlassWindow
from utils.WindowSnapper import WindowSnapper
from main import ICON, UNSET_COORD

import gtk
import gobject


# the available window flags
_WINDOW_FLAG_NONE =      0
_WINDOW_FLAG_ABOVE =     1 << 0
_WINDOW_FLAG_BELOW =     1 << 1
_WINDOW_FLAG_STICKY =    1 << 2
_WINDOW_FLAG_MANAGED =   1 << 3
_WINDOW_FLAG_DECORATED = 1 << 4


#
# Class for display windows.
#
class Window(GlassWindow, DisplayContainer):

    # mapping: str -> window flag
    __WINDOW_FLAGS = {"below"  : _WINDOW_FLAG_BELOW,
                      "above"  : _WINDOW_FLAG_ABOVE,
                      "sticky" : _WINDOW_FLAG_STICKY,
                      "managed": _WINDOW_FLAG_MANAGED,
                      "decorated": _WINDOW_FLAG_DECORATED}

    __window_snapper = WindowSnapper()

    def __init__(self):

        # ID of the idle handler for initial placement of the window
        self.__init_drag_handler = 0

        # window bbox as it was stored in the window snapper
        self.__window_bbox = (0, 0, 0, 0)

        # the last mouse pointer position
        self.__last_pointer = (0, 0)

        # window position for detecting moves
        self.__window_pos = (UNSET_COORD, UNSET_COORD)
        self.__is_placed = False

        # window size for detecting resizing
        self.__window_size = (0, 0)

        # the window flags
        self.__window_flags = _WINDOW_FLAG_NONE

        # whether we are in managed mode (window manager takes care of the
        # window); managed windows don't snap to desklets and cannot be moved
        # with the middle mouse button
        self.__managed_mode = False

        # temporary data used for dragging windows
        self.__is_dragging = False
        self.__drag_offset = (0, 0)
        self.__dont_snap = False

        self.__reentrance_lock = False
        self.__positioning_lock = False

        self.__shape = None

        GlassWindow.__init__(self, gtk.WINDOW_TOPLEVEL)
        DisplayContainer.__init__(self)

        self.set_size_request(-1, -1)
        self.set_default_size(10, 10)

        # set the icon
        self.set_icon(gtk.gdk.pixbuf_new_from_file(ICON))

        # set up event handlers
        self.connect("key-press-event", self.__on_key, 0)
        self.connect("key-release-event", self.__on_key, 1)
        self.connect("button-press-event", self.__on_button, 0)
        self.connect("button-release-event", self.__on_button, 1)
        self.add_events(gtk.gdk.BUTTON_PRESS_MASK |
                        gtk.gdk.BUTTON_RELEASE_MASK |
                        gtk.gdk.POINTER_MOTION_MASK |
                        gtk.gdk.POINTER_MOTION_HINT_MASK)

        self.realize()
        self.set_window_flags("")



    #
    # Digs a hole into the window at the given position. This is necessary to
    # be able to place windows with embedded bonobo controls.
    #
    def __dig_a_hole(self, x, y):

        w, h = self.__window_size
        if (w != 0 and h != 0):
            mask = gtk.gdk.Pixmap(None, w, h, 1)
            gc = mask.new_gc()
            gc.foreground = gtk.gdk.Color(0, 0, 0, 1)
            mask.draw_rectangle(gc, True, 0, 0, w, h)
            gc.foreground = gtk.gdk.Color(0, 0, 0, 0)
            mask.draw_rectangle(gc, True, x - 4 , y - 4, 8, 8)
            self.get_children()[0].window.shape_combine_mask(mask, 0, 0)



    #
    # Begins a window dragging operation.
    #
    def __begin_dragging(self):

        assert(self.window)

        self.__is_dragging = True
        self.update_observer(self.OBS_LOCK)
        x, y = self.get_pointer()
        w, h = self.__window_size
        w = max(5, w)
        h = max(5, h)
        if (not (0 <= x <= w)): x = w / 2
        if (not (0 <= y <= h)): y = h / 2
        self.__drag_offset = (x, y)
        self.window.set_cursor(gtk.gdk.Cursor(gtk.gdk.FLEUR))

        bx, by, bw, bh = self.__window_bbox
        if (not self.__managed_mode):
            self.__window_snapper.remove(bx, by, bw, bh)

        # grab the window focus to ensure that keypresses work
        self.present()

        gtk.timeout_add(50, self.__do_dragging)
        gtk.timeout_add(50, self.__do_move_window)

        return False



    #
    # Ends a window dragging operation.
    #
    def __end_dragging(self):

        assert(self.window)

        self.get_children()[0].window.shape_combine_mask(None, 0, 0)
        self.__is_dragging = False
        self.update_observer(self.OBS_UNLOCK)
        self.window.set_cursor(None)

        x, y = self.__window_pos
        w, h = self.__window_size
        if (not self.__managed_mode):
            self.__window_snapper.insert(x, y, w, h)
        self.__set_placed()
        self.__window_bbox = (x, y, w, h)

        self.__reentrance_lock = True
        self.update_observer(self.OBS_GEOMETRY)
        self.__reentrance_lock = False

        return False



    #
    # Drags the window.
    #
    def __do_dragging(self):

        if (self.__is_dragging):
            offx, offy = self.__drag_offset
            winx, winy = self.get_position()
            x, y = self.get_pointer()
            rx, ry = winx + x, winy + y

            if ((rx, ry) != self.__last_pointer):
                self.__last_pointer = (rx, ry)
                x += winx; y += winy
            else:
                wx, wy = self.__window_pos
                x = offx + wx; y = offy + wy

            new_x = x - offx
            new_y = y - offy
            if (self.__window_pos != (new_x, new_y)):
                if (not self.__dont_snap and not self.__managed_mode):
                    w, h = self.__window_size
                    new_x, new_y = self.__window_snapper.snap(new_x, new_y,
                                                              w, h)
                self.__window_pos = (new_x, new_y)

            return True

        else:
            return False



    def __do_move_window(self):

        x, y = self.__window_pos
        self.move(x, y)

        return self.__is_dragging



    def __on_key(self, src, event, is_release):

        key = gtk.gdk.keyval_name(event.keyval)
        if (not is_release and self.__is_dragging):
            x, y = self.__window_pos

            # let the user move the display window with the keyboard
            if (key == "Left"): x -= 4
            elif (key == "Right"): x += 4
            elif (key == "Up"): y -= 4
            elif (key == "Down"): y += 4

            # stop dragging if the Return key is pressed
            elif (key == "Return"):
                self.__end_dragging()

            self.__window_pos = (x, y)

        if (not is_release):
            # don't snap while one of the SHIFT keys is pressed
            self.__dont_snap = (key in ("Shift_L", "Shift_R"))
        else:
            self.__dont_snap = False



    def __on_button(self, src, event, is_release):

        if (not is_release and event.button == 2 and not self.__managed_mode):
            self.__begin_dragging()

        elif (not is_release and self.__is_dragging):
            self.__end_dragging()

        elif (is_release and self.__is_dragging):
            self.__end_dragging()



    def close(self):

        wx, wy, w, h = self.__window_bbox
        self.__window_snapper.remove(wx, wy, w, h)
        self.destroy()



    def set_position(self, x, y):

        gtk.idle_add(self.show)

        # enter positioning mode if the values get unset
        if (x == y == UNSET_COORD and not self.__is_dragging):
            if (self.__managed_mode and not self.__is_placed):
                # let the window manager place the window
                self.__set_placed()

            else:
                # let the user place the window
                self.__begin_dragging()
                x, y = self.get_pointer()
                self.__dig_a_hole(x, y)
            return

        # break cycles
        if ((x, y) == self.__window_pos or self.__reentrance_lock
            or self.__is_dragging): return

        ox, oy = self.__window_pos
        if ((x, y) != self.__window_pos):
            self.__positioning_lock = True
            gtk.timeout_add(0, self.move, x, y)
            self.__window_pos = (x, y)

            bx, by, bw, bh = self.__window_bbox
            if (not self.__managed_mode):
                self.__window_snapper.remove(bx, by, bw, bh)
            w, h = self.__window_size
            if (not self.__managed_mode):
                self.__window_snapper.insert(x, y, w, h)
            self.__set_placed()
            self.__window_bbox = (x, y, w, h)
            self.update_observer(self.OBS_GEOMETRY)


    def __set_placed(self):

        self.__is_placed = True
        if (self.__shape): self.set_shape(self.__shape)


    def is_placed(self):

        return self.__is_placed



    def set_size(self, width, height):

        if ((width, height) != self.__window_size):
            bx, by, bw, bh = self.__window_bbox
            if (not self.__managed_mode):
                self.__window_snapper.remove(bx, by, bw, bh)
            self.__window_size = (width, height)
            self.resize(width, height)
            if (not self.__managed_mode):
                self.__window_snapper.insert(bx, by, width, height)
            self.__window_bbox = (bx, by, width, height)
            self.update_observer(self.OBS_GEOMETRY)



    def get_geometry(self):

        x, y = self.__window_pos
        w, h = self.__window_size
        return (x, y, w, h)



    def set_window_flags(self, value):

        is_managed = self.__managed_mode
        flags = _WINDOW_FLAG_NONE
        for p in value:
            p = p.strip()

            try:
                flags |= self.__WINDOW_FLAGS[p]
            except KeyError:
                import sys
                print >> sys.stderr, "%s is not a valid window flag" % (p,)

        if (not flags & _WINDOW_FLAG_ABOVE):
            self._set_flag_below(flags & _WINDOW_FLAG_BELOW)
        if (not flags & _WINDOW_FLAG_BELOW):
            self._set_flag_above(flags & _WINDOW_FLAG_ABOVE)
        self._set_flag_sticky(flags & _WINDOW_FLAG_STICKY)
        self._set_flag_managed(flags & _WINDOW_FLAG_MANAGED)
        self._set_flag_decorated(flags & _WINDOW_FLAG_DECORATED)
        self.__window_flags = flags

        # remove from window snapper
        if (not is_managed and flags & _WINDOW_FLAG_MANAGED):
            self.__managed_mode = True
            bx, by, bw, bh = self.__window_bbox
            self.__window_snapper.remove(bx, by, bw, bh)

        # insert into window snapper
        elif (is_managed and not flags & _WINDOW_FLAG_MANAGED):
            self.__managed_mode = False
            bx, by, bw, bh = self.__window_bbox
            self.__window_snapper.insert(bx, by, bw, bh)



    def set_title(self, value):

        GlassWindow.set_title(self, value)



    def set_icon(self, pixbuf):

        GlassWindow.set_icon(self, pixbuf)



    def set_shape(self, mask):

        if (self.__window_pos == (UNSET_COORD, UNSET_COORD)):
            self.__shape = mask
        else:
            self.shape_combine_mask(mask, 0, 0)
