/*
** Copyright (C) 10 Feb 1999 Jonas Munsin <jmunsin@iki.fi>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <gtk/gtk.h>
#include <stdlib.h>
#include <string.h>

#include "optimize_usage.h"
#include "vector_commands.h"
#include "contractions.h"
#include "common_gtk.h"
#include "filepicker.h"
#include "modify_file_set.h"
#include "command.h"
#include "mainwindow.h"
#include "locks.h"
#include "globals.h"

static GtkWidget *opt_try_harder;

static GtkWidget *cdsize_option_menu, *cdsize_selected_label, *cdsector_selected_label;
static GtkWidget *used_size_label, *used_sectors_label, *wasted_size_label, *wasted_sectors_label;
long int opt_cdsize = 333000;

typedef struct {
	GtkWidget *window;
	GtkWidget *entry;
} entry_info;

static void select_to_be_deleted(file_data *mark_delete[], int no_of_selections) {
	int i, j;
	file_data *info;

	for (i = 0; i < GTK_CLIST(clist)->rows; i++)
		gtk_clist_unselect_row(GTK_CLIST(clist), i, 0);

	for (i = 0; i < GTK_CLIST(clist)->rows; i++) {
		info = gtk_clist_get_row_data(GTK_CLIST(clist), i);
		g_assert(NULL != info);
		for (j = 0; j < no_of_selections; j++)
			if (strcmp(info->realpath, mark_delete[j]->realpath) == 0)
				gtk_clist_select_row(GTK_CLIST(clist), i, 0);
	}
}

void make_estimate_inaccurate(void) {
        gtk_widget_set_sensitive(used_size_label, 0);
        gtk_widget_set_sensitive(used_sectors_label, 0);
        gtk_widget_set_sensitive(wasted_size_label, 0);
        gtk_widget_set_sensitive(wasted_sectors_label, 0);
}

void make_estimate_accurate(void) {
        gtk_widget_set_sensitive(used_size_label, 1);
        gtk_widget_set_sensitive(used_sectors_label, 1);
        gtk_widget_set_sensitive(wasted_size_label, 1);
        gtk_widget_set_sensitive(wasted_sectors_label, 1);
}

/* size is sectors used */
static void update_estimate_labels(long int size) {
	char s[10];

	if (-1 == g_snprintf(s, 10, "%ld MB", (size*2048)/(1024*1024)))
		g_warning("optimize_usage.c::update_estimate_labels: g_snprintf returned -1!");
	gtk_label_set(GTK_LABEL(used_size_label), s);
	if (-1 == g_snprintf(s, 10, "%ld", size))
		g_warning("optimize_usage.c::set_cd_size: g_snprintf returned -1!");
	gtk_label_set(GTK_LABEL(used_sectors_label), s);
	if (-1 == g_snprintf(s, 10, "%ld MB", ((opt_cdsize - size)*2048)/(1024*1024)))
		g_warning("optimize_usage.c::update_estimate_labels: g_snprintf returned -1!");
	gtk_label_set(GTK_LABEL(wasted_size_label), s);
	if (-1 == g_snprintf(s, 10, "%ld", opt_cdsize - size))
		g_warning("optimize_usage.c::set_cd_size: g_snprintf returned -1!");
	gtk_label_set(GTK_LABEL(wasted_sectors_label), s);
}

/* approximate binpacking routine
 * FIXME: the use of execute_recalc_size() might make some of the global
 * variables wrong
 */

static void begin_binpacking(GtkWidget *widget, gpointer data) {
	file_data **sorted_info;
	file_data **to_mark_delete;
	file_data **check_these;
	file_data *info;
	int i, j;
	cmd_v *cmd;
	long int size_sofar = 150; /* cdrecord uses a few sectors for something
				    * else, 150 is for 74m CDs */
	long int old_session_size = get_old_session_size(mainptr->cdr_option_info);

	sorted_info = g_malloc((GTK_CLIST(clist)->rows)*sizeof(file_data *));
	to_mark_delete = g_malloc((GTK_CLIST(clist)->rows)*sizeof(file_data *));
	check_these = g_malloc((GTK_CLIST(clist)->rows)*sizeof(file_data *));

	if (NULL == sorted_info || NULL == to_mark_delete || NULL == check_these) {
		g_free(sorted_info);
		g_free(to_mark_delete);
		g_free(check_these);
		return;
	}

	size_sofar += old_session_size;

	for (i = 0; i < GTK_CLIST(clist)->rows; i++) /* set up array to be sorted */
		sorted_info[i] = gtk_clist_get_row_data(GTK_CLIST(clist), i);

	for (i = 0; i < GTK_CLIST(clist)->rows; i++) { /* sort array */
		info = sorted_info[i];
		for (j = i; (j > 0) && (sorted_info[j-1]->size > info->size); j--)
			sorted_info[j] = sorted_info[j-1];
		sorted_info[j] = info;
	}
	if (GTK_TOGGLE_BUTTON(opt_try_harder)->active) { /* first fit decreasing */
		int k;
		j = 0;
		k = 0;
		mainwindow_disable_tabs();
		for (i = GTK_CLIST(clist)->rows - 1; i >= 0; i--) {
			check_these[k] = sorted_info[i];
			if (NULL == (cmd = make_mkisofs_c_optimize(check_these, k+1)))
				continue;
			sectorsize = 150 + old_session_size;
			if (!(is_running())) {
				execute_recalc_size(cmd);
				not_running();
			}
			destroy_cmd(cmd);
			if (sectorsize < opt_cdsize)
				k++;
			else
				to_mark_delete[j++] = sorted_info[i];
		}
		mainwindow_enable_tabs();
		if (k > 0) {
			if (NULL == (cmd = make_mkisofs_c_optimize(check_these, k))) {
				g_warning("optimize_usage.c::begin_bin_packing: Could not "
						"recheck pack result, estimate size will not be updated!");
				g_free(sorted_info);
				g_free(to_mark_delete);
				g_free(check_these);
				return;
			}
			sectorsize = 150 + old_session_size;
			if (!(is_running())) {
				execute_recalc_size(cmd);
				not_running();
			}
			size_sofar = sectorsize;
		}
	} else { /* fast sort using "cached" size data */
		j = 0;
		for (i = GTK_CLIST(clist)->rows - 1; i >= 0; i--) {
			if (size_sofar + sorted_info[i]->size < opt_cdsize)
				size_sofar += sorted_info[i]->size;
			else
				to_mark_delete[j++] = sorted_info[i];
		}
	}

	update_estimate_labels(size_sofar);
	make_estimate_accurate();
	select_to_be_deleted(to_mark_delete, j);

	g_free(sorted_info);
	g_free(to_mark_delete);
	g_free(check_these);
}

static void set_cd_size(long int sectsize) {
	/* FIXME: magic constant that could fail with NLS support if some language
	 * has long words */
	char s[50];

	if (opt_cdsize != sectsize)
		make_estimate_inaccurate();

	opt_cdsize = sectsize;

	if (-1 == g_snprintf(s, 50, _("Sectors: %ld"), sectsize))
		g_warning("optimize_usage.c::set_cd_size: g_snprintf returned -1!");
	gtk_label_set(GTK_LABEL(cdsector_selected_label), s);

	if (-1 == g_snprintf(s, 50, _("Size: %ld MB"), (sectsize*2048)/(1024*1024)))
		g_warning("optimize_usage.c::set_cd_size: g_snprintf returned -1!");
	gtk_label_set(GTK_LABEL(cdsize_selected_label), s);
}

static gint size_selection_74(GtkWidget *widget, gpointer data) {
	set_cd_size(333000);
	return FALSE;
}

static gint size_selection_80(GtkWidget *widget, gpointer data) {
	set_cd_size(360000);
	return FALSE;
}

static gint size_selection_90(GtkWidget *widget, gpointer data) {
	set_cd_size(405000);
	return FALSE;
}

static gint size_selection_dvd(GtkWidget *widget, gpointer data) {
	set_cd_size(2298300);
	return FALSE;
}

static gint size_selection_atip(GtkWidget *widget, gpointer data) {
	cmd_v *cdr_cmd;
	char *start;
	FILE *f;
	int done = 0;
	long int atip_size = 0;
	char in_buf[BUF_S];

	cdr_cmd = make_cdrecord_read_atip_command(mainptr->cdr_option_info);
	if (NULL == (f = popen_r_stdout(cdr_cmd))) {
		g_warning("%s::%i: popen failed", __FILE__, __LINE__);
		destroy_cmd(cdr_cmd);
		set_cd_size(0);
		return FALSE;
	}
	destroy_cmd(cdr_cmd);

	while (!(done)) {
		if (fgets(in_buf, BUF_S, f) != NULL) {
			if (feof(f)) {
				g_warning("%s::%i: didn't get ATIP info before eof",
						__FILE__, __LINE__);
				done = 1;
			} else if (ferror(f)) {
				g_warning("%s::%i: error reading ATIP", __FILE__, __LINE__);
				done = 1;
			} else if ((start = strstr(in_buf, "ATIP start of lead out:"))) {
				atip_size = atol(start + 23);
				done = 1;
			} else if ((start = strstr(in_buf,"free blocks:"))) {
				atip_size = atol(start + 13);
				done = 1;
			}
		} else {
			g_warning("%s::%i: error executing cdrecord ATIP command",
					__FILE__, __LINE__);
			done = 1;
		}
	}

	pclose(f);
	set_cd_size(atip_size);
	return FALSE;
}

static void set_size_selection_user(GtkWidget *widget, gpointer data) {
	char *s;
	long int size;

	s = gtk_entry_get_text(GTK_ENTRY(((entry_info *)data)->entry));
	size = atol(s);
	set_cd_size(size);

	gtk_widget_destroy(GTK_WIDGET(((entry_info *)data)->window));
	free(data);
}

static gint size_selection_user(GtkWidget *widget, gpointer data) {
	GtkWidget *cd_size_dialog, *entry, *button;

	entry_info *info;

	cd_size_dialog = gtk_dialog_new();
	entry = gtk_entry_new();
	gtk_widget_show(entry);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(cd_size_dialog)->vbox),
			entry, FALSE, FALSE, 0);
	gtk_tooltips_set_tip(tooltips, entry, _("Enter the size of your "
			"CD-R (e.g., 340000)/DVD-R (e.g., 2298000)"), NULL);

	button = gtk_button_new_with_label(_("Make it so"));
	gtk_widget_show(button);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(cd_size_dialog)->action_area), button,
			TRUE, TRUE, 0);

	info = malloc(sizeof(entry_info));
	info->window = cd_size_dialog;
	info->entry = entry;

	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(set_size_selection_user),
			(gpointer) info);

	gtk_window_set_title(GTK_WINDOW(GTK_DIALOG(cd_size_dialog)), _("sectors on disc"));
	gtk_widget_show(cd_size_dialog);

	return FALSE;
}

void optimize_attach(GtkWidget *vbox) {
	GtkWidget *opt_button, *optionmenu, *menuitem;
	GtkWidget *hbox, *used_frame, *wasted_frame, *frame_vbox;

/* start button */

	opt_button = gtk_button_new_with_label(_("Compute estimate"));
	gtk_widget_show(opt_button);
	gtk_box_pack_start(GTK_BOX(vbox), opt_button, FALSE, FALSE, 0);
	gtk_signal_connect(GTK_OBJECT(opt_button), "button_press_event",
			GTK_SIGNAL_FUNC(begin_binpacking), NULL);
	gtk_tooltips_set_tip(tooltips, opt_button, _("Attempt to determine "
			"which files to exclude in order for them to utilize the CD "
			"space to the max, and selects them so you can confirm by "
			"pressing delete"), NULL);

	opt_try_harder = gtk_check_button_new_with_label(_("Try harder"));
	gtk_widget_show(opt_try_harder);
	gtk_tooltips_set_tip(tooltips, opt_try_harder,
			_("Uses a more exact method of packing the files (useful if you "
			"have a lot of small files and in cases when you \"almost\" "
			"can fit everything on disk) - can be _very slow_") , NULL);
	gtk_box_pack_start(GTK_BOX(vbox), opt_try_harder, FALSE, FALSE, 0);

/* cd size optionmenu */

	optionmenu = gtk_option_menu_new();
	gtk_widget_show(optionmenu);
	gtk_box_pack_start(GTK_BOX(vbox), optionmenu, FALSE, FALSE, 0);
	cdsize_option_menu = gtk_menu_new();

	menuitem = gtk_menu_item_new_with_label(_("74 min"));
	gtk_widget_show(menuitem);
	gtk_menu_append(GTK_MENU(cdsize_option_menu), menuitem);
	gtk_signal_connect(GTK_OBJECT(menuitem), "button_release_event",
			GTK_SIGNAL_FUNC(size_selection_74), NULL);

	menuitem = gtk_menu_item_new_with_label(_("80 min"));
	gtk_widget_show(menuitem);
	gtk_menu_append(GTK_MENU(cdsize_option_menu), menuitem);
	gtk_signal_connect(GTK_OBJECT(menuitem), "button_release_event",
			GTK_SIGNAL_FUNC(size_selection_80), NULL);

	menuitem = gtk_menu_item_new_with_label(_("90 min"));
	gtk_widget_show(menuitem);
	gtk_menu_append(GTK_MENU(cdsize_option_menu), menuitem);
	gtk_signal_connect(GTK_OBJECT(menuitem), "button_release_event",
			GTK_SIGNAL_FUNC(size_selection_90), NULL);

	menuitem = gtk_menu_item_new_with_label(_("DVD-R"));
	gtk_widget_show(menuitem);
	gtk_menu_append(GTK_MENU(cdsize_option_menu), menuitem);
	gtk_signal_connect(GTK_OBJECT(menuitem), "button_release_event",
			GTK_SIGNAL_FUNC(size_selection_dvd), NULL);

	menuitem = gtk_menu_item_new_with_label(_("disc"));
	gtk_widget_show(menuitem);
	gtk_menu_append(GTK_MENU(cdsize_option_menu), menuitem);
	gtk_signal_connect(GTK_OBJECT(menuitem), "button_release_event",
			GTK_SIGNAL_FUNC(size_selection_atip), NULL);

	menuitem = gtk_menu_item_new_with_label(_("user"));
	gtk_widget_show(menuitem);
	gtk_menu_append(GTK_MENU(cdsize_option_menu), menuitem);
	gtk_signal_connect(GTK_OBJECT(menuitem), "button_release_event",
			GTK_SIGNAL_FUNC(size_selection_user), NULL);

	gtk_option_menu_set_menu(GTK_OPTION_MENU(optionmenu), cdsize_option_menu);
/* selected size labels */

	cdsize_selected_label = gtk_label_new(_("Size: 650 MB"));
	gtk_widget_show(cdsize_selected_label);
	gtk_box_pack_start(GTK_BOX(vbox), cdsize_selected_label, FALSE, FALSE, 0);

	cdsector_selected_label = gtk_label_new(_("Sectors: 333000"));
	gtk_widget_show(cdsector_selected_label);
	gtk_box_pack_start(GTK_BOX(vbox), cdsector_selected_label, FALSE, FALSE, 0);

/* used/wasted info TODO: make some down-up progressbars here, seems to require gtk1.1.x though */

	hbox = gtk_hbox_new(FALSE, 1);
	gtk_widget_show(hbox);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	used_frame = gtk_frame_new(_("Used:"));
	gtk_widget_show(used_frame);
	gtk_box_pack_start(GTK_BOX(hbox), used_frame, TRUE, TRUE, 0);
	wasted_frame = gtk_frame_new(_("Wasted:"));
	gtk_widget_show(wasted_frame);
	gtk_box_pack_start(GTK_BOX(hbox), wasted_frame, TRUE, TRUE, 0);

	frame_vbox = gtk_vbox_new(FALSE, 1);
	gtk_widget_show(frame_vbox);
	gtk_container_add(GTK_CONTAINER(used_frame), frame_vbox);

	used_size_label = gtk_label_new("0 MB");
	gtk_widget_show(used_size_label);
	gtk_box_pack_start(GTK_BOX(frame_vbox), used_size_label, FALSE, FALSE, 1);
	used_sectors_label = gtk_label_new("0");
	gtk_widget_show(used_sectors_label);
	gtk_box_pack_start(GTK_BOX(frame_vbox), used_sectors_label, FALSE, FALSE, 1);

	frame_vbox = gtk_vbox_new(FALSE, 1);
	gtk_widget_show(frame_vbox);
	gtk_container_add(GTK_CONTAINER(wasted_frame), frame_vbox);

	wasted_size_label = gtk_label_new("650 MB");
	gtk_widget_show(wasted_size_label);
	gtk_box_pack_start(GTK_BOX(frame_vbox), wasted_size_label, FALSE, FALSE, 1);
	wasted_sectors_label = gtk_label_new("333000");
	gtk_widget_show(wasted_sectors_label);
	gtk_box_pack_start(GTK_BOX(frame_vbox), wasted_sectors_label, FALSE, FALSE, 1);

	make_estimate_accurate();
}

