/*
 * Copyright (C) 2002  Bogdan Surdu (tim@rdsnet.ro)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Revision: 1.8 $
 *
 * $Log: flow_engine.c,v $
 * Revision 1.8  2003/06/19 17:11:20  tim
 * new memory management
 * removed the fragment list
 *
 * Revision 1.7  2003/03/20 20:27:44  tim
 * modified a bit the memory allocation
 *
 * Revision 1.6  2003/03/07 18:57:31  tim
 * Applied patch from SUBREDU Manuel <diablo@iasi.roedu.net>
 *
 * Revision 1.5  2003/03/07 18:56:02  tim
 * *** empty log message ***
 *
 * Revision 1.4  2002/12/13 19:35:01  tim
 *  - added support for IP fragments
 *
 * Revision 1.3  2002/12/03 23:28:36  tim
 * added support for snmp interface id
 *
 * Revision 1.2  2002/11/26 09:28:14  tim
 * row level mutex locking
 * cleanup in flow_scan
 *
 * Revision 1.1  2002/11/24 23:13:38  tim
 * Initial revision
 *
 *
 *
 */

#include <stdio.h>
#include <net/ethernet.h>
#include <netinet/ether.h>
#include <netinet/ip.h>
#include <netinet/in.h>


#include "flow_storage.h"
#include "flow_exporter.h"
#include "list.h"
#include "fprobe.h"

extern int export_inactive;
extern int export_active;
extern pthread_mutex_t mutex_scan[HASH_SLOTS];

#ifdef	MEM_STATS
extern unsigned long mem_usage_stats[32];
extern unsigned long mem_alloc[32], mem_alloc_bytes[32];
extern unsigned long mem_dealloc[32], mem_dealloc_bytes[32];
#endif



int flow_check(struct flow_item_t *flow, unsigned long now) {

  if (now - flow->last_packet > export_inactive || flow->last_packet - flow->flow_start > export_active) {
    return 1;
  } else
    return 0;
}

void flow_add_packet(HASH_TABLE h,struct iphdr *ip, unsigned short *data,  unsigned short src_iif, unsigned short dst_iif, unsigned long ts) {
  int len;
  struct flow_item_t *flow;
  unsigned short sport, dport;
  struct tcphdr *tcp;
  struct udphdr *udp;
  int is_fragment;
  unsigned short frag, frag_off;
  unsigned int key;
  
  len = ntohs(ip->tot_len);

  sport = 0; dport = 0;
  /* TODO, find the source/destination port number from L4 */
  if (ip->protocol == IPPROTO_TCP || ip->protocol == IPPROTO_UDP) {
    sport = *data;
    dport = *(data+1);
  }
  key = sport << 16 | dport;

  frag = ntohs(ip->frag_off);
  frag_off = frag & 0x1fff;
  is_fragment = frag & IP_MF == IP_MF;
  if (is_fragment || frag_off) {
    key = ip->id;
  }


  /* lookup flow in our table */
  flow = (struct flow_item_t *)flow_hash_lookup(h, ip->saddr, ip->daddr, ip->protocol, sport, dport, key);

  /*printf("(%s:%d -> ", inet_ntoa(ip->saddr), htons(sport));
  printf("%s:%d) Len: %d, Prot: %d, Frag: %d (MF: %d), Key: %x, Flow: 0x%x\n", inet_ntoa(ip->daddr), htons(dport), len, ip->protocol, frag_off, is_fragment, key, flow);
  */

  if (flow) {
    /* it's a new flow. timestamp it */
    if (!flow->packets) {
      flow->flow_start = ts;
      flow->src_iif = src_iif;
      flow->dst_iif = dst_iif;
    }

    if (is_fragment && !frag_off) {
      /* The 1st fragment has the sport and dport */
      flow->sport = sport;
      flow->dport = dport;
    }

    flow->last_packet = ts;
    flow->bytes += len;
    flow->packets++;

    pthread_mutex_unlock(&mutex_scan[flow->slot]);

  } else {
#ifdef	DEBUG
    fprintf(stderr,"[%s:%d] Something went wrong..we don't have a flow for this packet\n", __FILE__,__LINE__);
#endif
    return;
  }
}


void flow_print(struct flow_item_t *flow) {
  if (flow != NULL) {
    printf("%s:%d -> ", inet_ntoa(flow->src_ip), flow->sport);
    printf("%s:%d Bytes: %lu Packets: %lu\n", inet_ntoa(flow->dst_ip), flow->dport, flow->bytes, flow->packets);
  }
}


void flow_dump(HASH_TABLE h) {
  int i;
  struct flow_item_t *tmp;

#ifdef	DEBUG
  printf("Dumping flow table\n");
#endif
  for(i=0;i<HASH_SLOTS;i++) {
    printf("Slot %d\n", i);
    tmp = h[i].list; 
    while(tmp) {
      printf("Found a flow in list\n");
      flow_print(tmp);
      tmp = tmp->next;
    }
  }
}

void flow_scan(HASH_TABLE h) {
  int i, count;
  struct flow_item_t *tmp, *kill, *old;
  unsigned long now;
  int res;

  now = timestamp(NULL);

  for(i=0;i<HASH_SLOTS;i++) {
    tmp = h[i].list; 
    old = NULL;
    res = pthread_mutex_lock(&mutex_scan[i]);
    if (res) {
      perror("pthread_mutex_lock in flow_engine\n");
      pthread_exit(NULL);
    }
    while(tmp) {
      if (flow_check(tmp, now)) {
        kill = tmp;
        if (old != NULL) { old->next = tmp->next; }
        else h[i].list = tmp->next;
        tmp = tmp->next;
        flow_export_item(kill);
        continue;
      }
      old = tmp;
      tmp = tmp->next;
    }
    res = pthread_mutex_unlock(&mutex_scan[i]);
    if (res) {
      perror("pthread_mutex_unlock in flow_engine\n");
      pthread_exit(NULL);
    }
  }
}


void flow_delete_table(HASH_TABLE h) {
  int i;
  struct flow_item_t *tmp, *old;

  /* free all lists */
  for(i=0;i<HASH_SLOTS;i++) {
    tmp = h[i].list;
    while(tmp != NULL) {
      old = tmp;
      tmp = tmp->next;

      FREE(old,2);
    }
  }
}

void flow_stats(HASH_TABLE h) {
  FILE *fp;
  int i, count;
  unsigned long total_items;
  struct flow_item_t *tmp;

  fp = fopen("/tmp/fprobe.stats","a");
  total_items = 0;
  for(i=0;i<HASH_SLOTS;i++) {
    tmp = h[i].list;
    count = 0;
    while(tmp) {
      count++;
      tmp = tmp->next;
    }
    fprintf(fp,"%5d | %5d items = %6d bytes\n", i, count, count*sizeof(struct flow_item_t));
    total_items+=count;
  }
  fprintf(fp,"Total %lu items in hash\n", total_items);

#ifdef	MEM_STATS
  for(i=0;i<32;i++) {
    fprintf(fp,"%2d | %10lu bytes | %10lu allocs | %10lu bytes in %10lu free | %10lu leaked\n", i, 
      mem_alloc_bytes[i], mem_alloc[i], mem_dealloc_bytes[i], mem_dealloc[i], mem_usage_stats[i]);
  }
#endif
  fclose(fp);
}

void flow_create_table(HASH_TABLE h) {
  unsigned long i;
  struct flow_item_t *flow;

  /* initialize all slots */
  for(i=0;i<HASH_SLOTS;i++) {
    h[i].list = NULL;
    h[i].last_parse = 0;
  }
  flow_pool = NULL;
  for(i=0;i<FLOW_ITEMS;i++) {
    flow = (flow_item_t *)malloc(sizeof(struct flow_item_t));
    flow->particle_id = i;
    flow_particle[i] = flow;
    flow_add_pool(flow);
  }
  reset_active_counter();
}

