/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libs390.so
 *
 *   File: 390segmgr.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>

#include "vtoc.h"
#include "390segmgr.h"
#include "helpers.h"
#include "options.h"
#include "commit.h"
#include "format.h"
#include "dm.h"
#include "move.h"

/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                         PRIVATE DATA AREAS AND SUBROUTINES                           +
+                                                                                      +
+-------------------------------------------------------------------------------------*/

engine_functions_t * EngFncs = NULL;



/***************** FOR DEBUGGING
static void display_format4_dscb( format4_label_t *f4, lba_t vstart, lba_t vend, int blocksize )
{

    LOG_DEBUG("FORMAT 4 DSCB ....\n");
    LOG_DEBUG("     number of avail dscbs : %d\n", f4->DS4DSREC );
    LOG_DEBUG("     number of vtoc extents: %d\n", f4->DS4NOEXT);
    LOG_DEBUG("     vtoc extent info ....\n");
    LOG_DEBUG("          type             : %02X\n", f4->DS4VTOCE.typeind );
    LOG_DEBUG("          seq number       : %02X\n", f4->DS4VTOCE.seqno );
    LOG_DEBUG("          start cchh       : %d %d\n", f4->DS4VTOCE.llimit.cc, f4->DS4VTOCE.llimit.hh );
    LOG_DEBUG("          start 390  lba   : %"PRIu64"\n", vstart );
    LOG_DEBUG("          start evms lba   : %"PRIu64"\n", vstart*blocksize );
    LOG_DEBUG("          end cchh         : %d %d\n", f4->DS4VTOCE.ulimit.cc, f4->DS4VTOCE.ulimit.hh );
    LOG_DEBUG("          end 390  lba     : %"PRIu64"\n", vend );
    LOG_DEBUG("          end evms lba     : %"PRIu64"\n", vend*blocksize );

}
static void display_format5_dscb( format5_label_t *f5 )
{
    int i;

    LOG_DEBUG("FORMAT 5 DSCB ....\n");
    LOG_DEBUG("     key identifier        : %02X%02X%02X%02X\n", f5->DS5KEYID[0],f5->DS5KEYID[1],f5->DS5KEYID[2],f5->DS5KEYID[3] );
    LOG_DEBUG("     FIRST AVAIL EXTENT\n");
    LOG_DEBUG("     --------------------------------------------\n");
    LOG_DEBUG("          rta of 1st track : %d\n", f5->DS5AVEXT.t);
    LOG_DEBUG("          whole cylinders  : %d\n", f5->DS5AVEXT.fc);
    LOG_DEBUG("          remaining tracks : %d\n", f5->DS5AVEXT.ft);

    for (i=0; i<7; i++) {

        LOG_DEBUG("     AVAIL EXTENT %d\n",i);

        if ( (f5->DS5EXTAV[i].t) ||
             (f5->DS5EXTAV[i].fc) ||
             (f5->DS5EXTAV[i].ft)){

            LOG_DEBUG("          rta of 1st track : %d\n", f5->DS5EXTAV[i].t);
            LOG_DEBUG("          whole cylinders  : %d\n", f5->DS5EXTAV[i].fc);
            LOG_DEBUG("          remaining tracks : %d\n", f5->DS5EXTAV[i].ft);

        }
        else{
            LOG_DEBUG("          null null null null null\n");
        }
    }

    LOG_DEBUG("     next F5 dscb cc-hh-b  : %d-%d-%d\n", f5->DS5PTRDS.cc, f5->DS5PTRDS.hh, f5->DS5PTRDS.b );

}
static void display_format1_dscb( format1_label_t *f1 )
{
    char  name[MAX_DSNAME_LENGTH+1];
    char  system_code[S390_SYSTEM_CODE_LENGTH+1];

    memcpy( name, f1->DS1DSNAM, MAX_DSNAME_LENGTH);
    etoa( name, MAX_DSNAME_LENGTH );
    name[MAX_DSNAME_LENGTH]=0x00;

    memcpy( system_code, f1->DS1SYSCD, S390_SYSTEM_CODE_LENGTH);
    etoa(system_code, S390_SYSTEM_CODE_LENGTH);
    system_code[S390_SYSTEM_CODE_LENGTH]=0x00;

    LOG_DEBUG("FORMAT 1 DSCB ....\n");
    LOG_DEBUG("     data set name         : %s\n", name);
    LOG_DEBUG("     format id             : 0x%02X\n", f1->DS1FMTID);
    LOG_DEBUG("     data set SN           : %02X %02X %02X %02X %02X %02X\n",
        f1->DS1DSSN[0], f1->DS1DSSN[1], f1->DS1DSSN[2], f1->DS1DSSN[3], f1->DS1DSSN[4], f1->DS1DSSN[5]);
    LOG_DEBUG("     vol seq number        : 0x%02x\n", f1->DS1VOLSQ);
    LOG_DEBUG("     creation date         : n/a \n");
    LOG_DEBUG("     expiration date       : n/a \n");
    LOG_DEBUG("     number of extents     : %d\n", f1->DS1NOEPV);
    LOG_DEBUG("     bytes in last dir blk : %d\n", f1->DS1NOBDB);
    LOG_DEBUG("     system code           : %s\n", system_code );
    LOG_DEBUG("     date last referenced  : n/a \n");
    LOG_DEBUG("     system storage id     : 0x%02x\n", f1->DS1SMSFG );
    LOG_DEBUG("     sec space ext flag    : 0x%02x\n", f1->DS1SCXTF );
    LOG_DEBUG("     data set org          : %02X %02X\n", f1->DS1DSRG1, f1->DS1DSRG2 );
    LOG_DEBUG("     record format         : 0x%02x\n", f1->DS1RECFM );
    LOG_DEBUG("     option code           : 0x%02x\n", f1->DS1OPTCD );
    LOG_DEBUG("     block length          : %d\n", f1->DS1BLKL );
    LOG_DEBUG("     record length         : %d\n", f1->DS1LRECL);
    LOG_DEBUG("     key length            : %d\n", f1->DS1KEYL );
    LOG_DEBUG("     relative key pos      : 0x%02x\n", f1->DS1RKP );
    LOG_DEBUG("     data set indicators   : 0x%02x\n", f1->DS1DSIND);
    LOG_DEBUG("     secondary alloc flag  : 0x%02x\n", f1->DS1SCAL1);
    LOG_DEBUG("     secondary alloc qty   : %02X%02X%02X\n", f1->DS1SCAL3[0], f1->DS1SCAL3[1], f1->DS1SCAL3[2]);
    LOG_DEBUG("     last used track/blk   : %d/%d \n", f1->DS1LSTAR.tt, f1->DS1LSTAR.r );
    LOG_DEBUG("     space remaining       : %d\n", f1->DS1TRBAL);
    LOG_DEBUG("     EXTENT 1\n");
    LOG_DEBUG("     --------------------------------------------\n");
    LOG_DEBUG("          type             : %02X\n", f1->DS1EXT1.typeind );
    LOG_DEBUG("          seq number       : %02X\n", f1->DS1EXT1.seqno );
    LOG_DEBUG("          start cchh       : %d %d\n", f1->DS1EXT1.llimit.cc, f1->DS1EXT1.llimit.hh );
    LOG_DEBUG("          end cchh         : %d %d\n", f1->DS1EXT1.ulimit.cc, f1->DS1EXT1.ulimit.hh );
    LOG_DEBUG("     EXTENT 2\n");
    LOG_DEBUG("     --------------------------------------------\n");
    LOG_DEBUG("          type             : %02X\n", f1->DS1EXT2.typeind );
    LOG_DEBUG("          seq number       : %02X\n", f1->DS1EXT2.seqno );
    LOG_DEBUG("          start cchh       : %d %d\n", f1->DS1EXT2.llimit.cc, f1->DS1EXT2.llimit.hh );
    LOG_DEBUG("          end cchh         : %d %d\n", f1->DS1EXT2.ulimit.cc, f1->DS1EXT2.ulimit.hh );
    LOG_DEBUG("     EXTENT 3\n");
    LOG_DEBUG("     --------------------------------------------\n");
    LOG_DEBUG("          type             : %02X\n", f1->DS1EXT3.typeind );
    LOG_DEBUG("          seq number       : %02X\n", f1->DS1EXT3.seqno );
    LOG_DEBUG("          start cchh       : %d %d\n", f1->DS1EXT3.llimit.cc, f1->DS1EXT3.llimit.hh );
    LOG_DEBUG("          end cchh         : %d %d\n", f1->DS1EXT3.ulimit.cc, f1->DS1EXT3.ulimit.hh );

}
**************/




/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                            Start Of EVMS Plugin Functions                            +
+                        (exported to engine via function table)                       +
+                                                                                      +
+-------------------------------------------------------------------------------------*/
static int S390_SetupEVMSPlugin( engine_functions_t * engine_functions)
{
        int rc = 0;

        EngFncs  = engine_functions;

        LOG_ENTRY();

        LOG_EXIT_INT(rc);

        return rc;
}


static void S390_Cleanup(void)
{
        list_anchor_t seglist=EngFncs->allocate_list();
        int rc;
        DISKSEG *seg;
        list_element_t iter;
        DISK_PRIVATE_DATA *disk_pdata;

        LOG_ENTRY();

        if (seglist != NULL) {

                // cleanup all seg private data areas
                rc = EngFncs->get_object_list( SEGMENT,
                                               0,
                                               s390_plugin_rec,
                                               NULL,
                                               0,
                                               &seglist );

                if (rc==0) {
                        LIST_FOR_EACH(seglist, iter, seg ) {
                                if (seg->private_data) free(seg->private_data);
                        }
                }

                EngFncs->destroy_list(seglist);
        }

        // cleanup all disk private data areas
        if ( Disk_PrivateData_List != NULL ) {
                LIST_FOR_EACH( Disk_PrivateData_List, iter, disk_pdata ) {
                        free(disk_pdata);
                }
                EngFncs->destroy_list( Disk_PrivateData_List );
        }

        LOG_EXIT_VOID();
}




/*
 *  I will allow the object to be made into a volume (or reverted) if ...
 *
 *  (1) I actually own the object
 *  (2) Which means it is a segment and has necessarey ctl blocks
 *
 */
static int S390_can_set_volume(storage_object_t * seg, boolean flag )
{
        int rc = EINVAL;
        LOGICALDISK *ld;
        DISK_PRIVATE_DATA *disk_pdata;

        LOG_ENTRY();

        if ( ( seg ) &&
             ( seg->object_type == SEGMENT ) &&
             ( seg->data_type   == DATA_TYPE ) &&
             ( i_can_modify( seg ) == TRUE ) ) {

                ld = get_logical_disk(seg);

                if (ld) {

                        if (flag == TRUE) {  // CREATE VOLUME

                                disk_pdata = get_s390_disk_private_data(ld);

                                if (disk_pdata) {

                                        rc = 0;

                                }

                        }
                        else {      // REVERT VOLUME
                                rc = 0;
                        }
                }

        }

        LOG_EXIT_INT(rc);
        return rc;

}


/*
 *  I can delete this segment if ...
 *
 *  (1) I own the segment
 *  (2) It is a data segment
 *  (3) It resides on a CDL formatted disk
 */
static int S390_CanDestroy( storage_object_t * seg )
{
        int rc = EINVAL;
        LOGICALDISK       *ld = get_logical_disk( seg );

        LOG_ENTRY();

        if (ld) {

                if ( ( seg->data_type == DATA_TYPE ) &&
                     ( isa_compatibility_disk(ld) == TRUE )) {

                        rc = 0;

                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  I can expand the object if ...
 *
 *  (1) it is a data segment
 *  (2) I own the segment
 *  (3) the logical disk and segment info is Ok
 *  (4) a freespace segment immediately follows it
 *
 */
static int S390_CanExpand( storage_object_t    *seg,
                           sector_count_t       expand_limit,
                           list_anchor_t        expansion_points )
{
        DISKSEG              *freespace;
        LOGICALDISK          *ld;
        int                   rc = EINVAL;
        expand_object_info_t *expand_object;
        SEG_PRIVATE_DATA     *pdata;
        sector_count_t        cylinder_size=0;
        list_element_t        e;

        LOG_ENTRY();

        if ( ( expansion_points ) &&
             ( seg ) &&
             ( seg->object_type == SEGMENT ) &&
             ( seg->data_type   == DATA_TYPE ) &&
             ( i_can_modify( seg ) == TRUE ) ) {

                freespace = get_freespace_following_s390_segment( seg );
                ld        = get_logical_disk(seg);
                pdata     = (SEG_PRIVATE_DATA *)seg->private_data;

                if ( freespace && ld ) {

                        if ( isa_compatibility_disk(ld) == TRUE ) {

                                cylinder_size = get_cylinder_size(ld);

                                // we can only expand in cylinder size amounts
                                if ( freespace->size >= cylinder_size &&
                                     expand_limit >= cylinder_size ) {

                                        expand_object = (expand_object_info_t *) EngFncs->engine_alloc( sizeof(expand_object_info_t) );
                                        if (expand_object) {

                                                expand_object->object          = seg;
                                                expand_object->max_expand_size = min(freespace->size, expand_limit);

                                                e=EngFncs->insert_thing( expansion_points,
                                                                         expand_object,
                                                                         INSERT_AFTER,
                                                                         NULL);
                                                if (e!=NULL) {
                                                        rc = 0;
                                                }
                                                else {
                                                        EngFncs->engine_free( expand_object );
                                                        rc = EPERM;
                                                }

                                        }
                                        else {
                                                LOG_ERROR("\nerror, alloc of expand object failed\n");
                                                rc = ENOMEM;
                                        }

                                }

                        }

                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 * I can expand the object by size sectors if:
 *
 *  (1) i own the object
 *  (2) the segment can be expanded
 *  (3) the freespace has at least a cylinder of space
 *
 * If I cannot expand because the freespace is too small
 * then I'll reduce the expand sector count to my maximum.
 *
 */
static int S390_CanExpandBy(storage_object_t * seg, sector_count_t *size)
{
        int                rc = EINVAL;
        LOGICALDISK       *ld;
        DISKSEG           *freespace;
        sector_count_t     cylinder_size;
        sector_count_t     max_expand_sectors=0;
        lba_t              freespace_end_lba=0;
        SEG_PRIVATE_DATA  *pdata;


        LOG_ENTRY();

        if ( ( seg ) &&
             ( seg->object_type == SEGMENT ) &&
             ( seg->data_type   == DATA_TYPE ) &&
             ( i_can_modify( seg ) == TRUE ) ) {

                freespace = get_freespace_following_s390_segment( seg );
                ld        = get_logical_disk(seg);
                pdata     = (SEG_PRIVATE_DATA *)seg->private_data;

                if ( freespace && ld ) {

                        if ( isa_compatibility_disk(ld) == TRUE ) {

                                cylinder_size = get_cylinder_size(ld);

                                // partitions end on a cylinder boundary. if freespace doesnt end on
                                // a cylinder bdy then round it down to a cyl bdy before testing if
                                // freespace can handle the ExpandBy.
                                if (ends_on_cylinder_boundary(ld,freespace->start+freespace->size-1)) {
                                        freespace_end_lba = freespace->start+freespace->size-1;
                                }
                                else {
                                        freespace_end_lba = rounddown_to_cylinder_boundary(ld, freespace->start+freespace->size-1) - 1;
                                }

                                // calculate the max useable size of the freespace area, i.e. the
                                // max area that ends on a cylinder boundary.
                                if (freespace_end_lba > freespace->start) {
                                        max_expand_sectors = freespace_end_lba - freespace->start + 1;
                                }
                                else {
                                        max_expand_sectors = 0;
                                }

                                // we expand in cylinder size chunks ... if max useable freespace
                                // size is less than 1 cylinder then we cant do any expand at all.
                                if (max_expand_sectors >= cylinder_size) {

                                        if ( max_expand_sectors >= *size ) {

                                                if ( max_expand_sectors == *size) {
                                                        rc = 0;
                                                }
                                                else {
                                                        freespace_end_lba = roundup_to_cylinder_boundary(ld, freespace->start + *size - 1 );
                                                        *size      = freespace_end_lba - freespace->start + 1;
                                                }

                                        }
                                        else {
                                                *size = max_expand_sectors;
                                                rc = EINVAL;
                                        }

                                }

                        }

                }

        }


        LOG_EXIT_INT(rc);
        return rc;
}


/*
 * I can shrink a seg if ...
 *
 *  (1) i own the object
 *  (2) it is a data segment
 *  (3) if I chop off a cylinder, the seg will still have
 *      a minimum of 1 cylinder of space
 *
 *  If not exact set new_size to closest higher value possible.
 */
static int S390_CanShrink( storage_object_t * seg,       // object to shrink
                           sector_count_t     shrink_limit,  // a delta size
                           list_anchor_t            shrink_points ) // of type shrink_object_info_t
{
        int                   rc = EINVAL;
        sector_count_t        cylinder_size;
        LOGICALDISK          *ld;
        shrink_object_info_t *shrink_object=NULL;
        SEG_PRIVATE_DATA     *pdata;
        list_element_t        e;

        LOG_ENTRY();

        if ( ( seg ) &&
             ( seg->object_type == SEGMENT ) &&
             ( seg->data_type == DATA_TYPE ) &&
             ( i_can_modify( seg ) == TRUE ) ) {


                pdata = (SEG_PRIVATE_DATA *)seg->private_data;
                ld    = get_logical_disk(seg);

                if ( ld ) {

                        if ( isa_compatibility_disk(ld) == TRUE ) {

                                cylinder_size = get_cylinder_size(ld);

                                // see if caller is trying to shrink smaller than min partition size
                                if ( seg->size > cylinder_size &&
                                     shrink_limit >= cylinder_size ) {

                                        if (shrink_points) {

                                                shrink_object = (shrink_object_info_t *) EngFncs->engine_alloc( sizeof(shrink_object_info_t) );
                                                if (shrink_object) {

                                                        // we are suppose to return the max amount we can shrink. since the
                                                        // minimum partition size is 1 cylinder ... then anything more than
                                                        // a cylinder is the max we can shrink.
                                                        shrink_object->object          = seg;
                                                        shrink_object->max_shrink_size = min(seg->size - cylinder_size, shrink_limit);

                                                        e=EngFncs->insert_thing( shrink_points,
                                                                                 shrink_object,
                                                                                 INSERT_AFTER,
                                                                                 NULL);
                                                        if (e!=NULL) {
                                                                rc = 0;
                                                        }
                                                        else {
                                                                EngFncs->engine_free( shrink_object );
                                                                rc = EPERM;
                                                        }

                                                }
                                                else {
                                                        LOG_ERROR("\nerror, failed to alloc shrink_object\n");
                                                        rc = ENOMEM;
                                                }
                                        }
                                }

                        }

                }

        }


        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  I can allow the storage object to shrink by the specified amount if ...
 *
 *  (1) the shrink point is a segment and I own it
 *  (2) the segment is large enough to allow a shrink
 *  (3) i can shrink it and end the segment on a cylinder boundary
 *
 *  If the segment doesnt end on a cylinder boundary I'll return
 *  an error and report the amount that we could shrink by.
 *
 */
static int S390_CanShrinkBy( storage_object_t * seg,
                             sector_count_t   * size )
{
        int               rc = EINVAL;
        sector_count_t    cylinder_size;
        LOGICALDISK      *ld=NULL;
        sector_count_t    delta=0;
        SEG_PRIVATE_DATA *pdata;

        LOG_ENTRY();

        if ( ( seg ) &&
             ( size ) &&
             (*size > 0 ) &&
             ( seg->object_type == SEGMENT ) &&
             ( seg->data_type == DATA_TYPE ) &&
             ( i_can_modify( seg ) == TRUE ) ) {

                pdata = (SEG_PRIVATE_DATA *)seg->private_data;
                ld    = get_logical_disk(seg);

                if ( ld ) {

                        if ( isa_compatibility_disk(ld) == TRUE ) {

                                cylinder_size = get_cylinder_size(ld);

                                // seg cant shrink if it is a cylinder or less in size
                                if (seg->size > cylinder_size) {

                                        if ( *size < seg->size ) {

                                                if (*size < cylinder_size) {
                                                        delta = cylinder_size;
                                                }
                                                else {
                                                        delta = (*size / cylinder_size)*cylinder_size;
                                                }

                                        }
                                        else {
                                                delta = seg->size - cylinder_size;
                                        }

                                        if ( delta == *size  ) {
                                                rc = 0;
                                        }
                                        else {
                                                *size = delta;
                                                rc = EINVAL;
                                        }

                                }

                        }

                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Called to create a metadata segment on the specified disk. This
 *  segment is basically used to protect the label track and
 *  vtoc tracks.
 */
static DISKSEG * create_metadata_segment( list_anchor_t   output_list,
                                          LOGICALDISK    *ld,
                                          lba_t           start,
                                          sector_count_t  size,
                                          int            *seg_count )
{
        DISKSEG  *metadata=NULL;
        list_element_t e;

        LOG_ENTRY();

        metadata = create_s390_metadata_segment( ld, start, size );
        if (metadata) {

                if ( insert_s390_segment_into_list( ld->parent_objects, metadata ) != NULL ) {

                        e = EngFncs->insert_thing( output_list,
                                                   metadata,
                                                   INSERT_AFTER,
                                                   NULL );
                        if (e != NULL) {
                                ++*seg_count;
                        }
                        else {
                                free_s390_disk_segment(metadata);
                                metadata = NULL;
                        }

                }

        }

        LOG_EXIT_PTR(metadata);
        return metadata;
}


/*
 *  Called to create a DATA segment on the specified disk.
 */
static DISKSEG * create_data_segment( list_anchor_t       output_list,
                                      storage_object_t   *disk,
                                      u_int64_t           offset,
                                      u_int64_t           size,
                                      data_type_t         type,
                                      int                 index,
                                      int                *seg_count )
{
        list_element_t e;
        storage_object_t * segment=NULL;


        LOG_ENTRY();

        segment = allocate_s390_disk_segment( disk );

        if (segment) {

                segment->size = size;
                segment->start = offset;
                segment->data_type = type;

                ((SEG_PRIVATE_DATA *)segment->private_data)->minor = index;

                if ( insert_s390_segment_into_list( disk->parent_objects, segment ) == NULL ) {
                        free_s390_disk_segment(segment);
                        segment = NULL;
                }
                else {
                        e = EngFncs->insert_thing( output_list,
                                                   segment,
                                                   INSERT_AFTER,
                                                   NULL );
                        if (e != NULL) {
                                ++*seg_count;
                        }
                        else {
                                free_s390_disk_segment(segment);
                                segment = NULL;
                        }

                }

        }


        LOG_EXIT_PTR(segment);
        return segment;
}



/*
 *  Standard 390 Volume Layout
 *
 *  Track 1 is called the Label track and is layed out like so ...
 *
 *     Cyl Head Blk          Description
 *      0   0    1           IPL1 Record         ... real or dummy ipl record
 *      0   0    2           IPL2 Record         ... real or dummy ipl record
 *      0   0    3           Volume Label Record ... info about volume including location of vtoc
 *      0   0    4-n         IPL text
 *
 *  The Volume Table of Content (VTOC) can be located anywhere but must be in a single
 *  extent on the volume. However, it is usually placed in the next track on the volume.
 *  The VTOC is allocated in track size chunks, starting and ending on a track boundary.
 *  It contains records that indicate which extents on the volume have been allocated as
 *  data sets. Therefore, these records are called Data Set Control Blocks (DSCBs). There
 *  are several forms for the DSCB.
 *
 *  Format 0 - Null Descriptor
 *  Format 1 - Data Set Descriptor
 *  Format 2 - Indexed Data, linked to Format 1 descriptor
 *  Format 3 - Additional Extents, linked to Format 1 descriptor
 *  Format 4 - VTOC Descriptor
 *  Format 5 - Freespace Descriptor
 *
 *  The VTOC is initialized with Format 0 DSCBs.  Then, the first two descriptors are used
 *  to describe the VTOC and the freespace on the volume ... like so ...
 *
 *  VTOC
 *  Record 0  Record 1  Record 2 ................................ Record N
 *  Format 4  Format 5  Format 0 ...
 *
 *  On VM, the VTOC is used to make the volume looked completely allocated -BUT- with no
 *  data sets.  This is done by using a Format 5 that specifies no freespace and not having
 *  any Format 1 DSCBs. Guest operating systems usually do not see the real dasd. Rather,
 *  they see the minidisk and can create their own vtoc as needed. A vm disk is setup in
 *  this fashion to protect the minidisks and the guest operating systems that own them.
 *
 */
static int s390_check_object(LOGICALDISK  *ld, list_anchor_t output_list, int *seg_count)
{
        int rc=0;
        int blocksize = 0; // really hardsector size (in sectors)
        int next_minor = 0;
        int vtoc_index = 0;

        sector_count_t   offset=0, size=0, psize=0;
        lba_t            io_start;

        lba_t            vtoc_lba=0;

        char             vtoc_record[EVMS_VSECTOR_SIZE];
        format1_label_t *f1 = (format1_label_t *) vtoc_record;
        format4_label_t *f4 = (format4_label_t *) vtoc_record;
//  format5_label_t *f5 = (format5_label_t *) vtoc_record;

        char             volume_label_sector[EVMS_VSECTOR_SIZE];
        volume_label_t  *vlabel = (volume_label_t *) volume_label_sector;

        struct hd_geometry  geo;

        DISK_PRIVATE_DATA *disk_pdata = NULL;

        DISKSEG           *md=NULL;
        DISKSEG           *seg;
        SEG_PRIVATE_DATA  *pdata;

        dasd_information_t devinfo;

        boolean            Disk_Is_Formatted = TRUE;


        LOG_ENTRY();


        /*
         *  Setup geometry information
         */
        blocksize        = ld->geometry.bytes_per_sector >> EVMS_VSECTOR_SIZE_SHIFT;

        geo.cylinders    = ld->geometry.cylinders;
        geo.heads        = ld->geometry.heads;
        geo.sectors      = ld->geometry.sectors_per_track;
        geo.start        = 0;


        LOG_DEBUG("disk blocksize = %d sectors \n", blocksize );


        /*
         *  Get dasd info
         */
        memset(&devinfo, 0, sizeof(dasd_information_t) );
        rc = get_390_dev_info( ld, &devinfo );
        if (rc) {
                LOG_ERROR("error, unable to get information about device %s\n", ld->name );
                LOG_EXIT_INT(rc);
                return rc;
        }

        LOG_DEBUG("Device Info ... %s\n", ld->name );
        LOG_DEBUG("     devno       : %d\n", devinfo.devno );
        LOG_DEBUG("     real devno  : %d\n", devinfo.real_devno );
        LOG_DEBUG("     status      : %d\n", devinfo.status );
        LOG_DEBUG("     label block : %d\n", devinfo.label_block);
        LOG_DEBUG("     type        : %02X%02X%02X%02X\n", devinfo.type[0], devinfo.type[1], devinfo.type[2], devinfo.type[3]);
        LOG_DEBUG("     open count  : %d\n", devinfo.open_count );
        LOG_DEBUG("     dev type    : %d\n", devinfo.dev_type );
        LOG_DEBUG("     FBA layout  : %d\n", devinfo.FBA_layout );

/*****  only found in API VERSION 2 struct
        LOG_DEBUG("     features    : %d\n", devinfo.features );
        switch (devinfo.format) {
            case DASD_FORMAT_NONE:
                LOG_DEBUG("     format      : NONE\n");
                break;
            case DASD_FORMAT_LDL:
                LOG_DEBUG("     format      : LDL\n");
                break;
            case DASD_FORMAT_CDL:
                LOG_DEBUG("     format      : CDL\n");
                break;
            default:
                LOG_DEBUG("     format      : UNKNOWN\n");
                break;
        }
*****/

        /*
         *  Read the 390 volume label record
         */

        io_start  = VOLUME_LABEL_BLOCK_NUMBER * blocksize;

        rc = READ( ld, io_start, 1, (void *) vlabel);
        if (rc) {

                rc = READ( ld, io_start, 1, (void *) vlabel);
                if (rc) {

                        rc = READ( ld, io_start, 1, (void *) vlabel);
                        if (rc) {

                                // 3 times a charm ... if we get here then
                                // assume that the disk is unformatted and build
                                // an unformatted VLABEL struct to merge with
                                // code path below

                                char volid[VOLSER_LENGTH+1];

                                // fill with ebcdic spaces
                                memset( vlabel, 0x40, sizeof(volume_label_t) );

                                // use virtual device address as volume id
                                sprintf(volid, "0X%04X", devinfo.devno);
                                memcpy(&vlabel->volid, volid, VOLSER_LENGTH );
                                atoe((char *)&vlabel->volid, VOLSER_LENGTH);

                                // contrive remainder
                                memcpy( &vlabel->volkey, (&(char []){0xc5,0xe5,0xd4,0xe2}), 4);  // ebcdic EVMS
                                memcpy( &vlabel->vollbl, (&(char []){0xc5,0xe5,0xd4,0xe2}), 4);  // ebcdic EVMS
                                vlabel->vtoc.cc = 0;
                                vlabel->vtoc.hh = 1;
                                vlabel->vtoc.b  = 1;

                                // flag this disk
                                Disk_Is_Formatted = FALSE;

                        }

                }

        }


        LOG_DEBUG("Volume Label ....\n");
        LOG_DEBUG("     Volume    Key: %02X %02X %02X %02X\n",
                  vlabel->volkey[0], vlabel->volkey[1], vlabel->volkey[2], vlabel->volkey[3] );
        LOG_DEBUG("     Volume  Label: %02X %02X %02X %02X\n",
                  vlabel->vollbl[0], vlabel->vollbl[1], vlabel->vollbl[2], vlabel->vollbl[3] );
        LOG_DEBUG("     Volume     Id: %02X %02X %02X %02X %02X %02X\n",
                  vlabel->volid[0], vlabel->volid[1], vlabel->volid[2], vlabel->volid[3], vlabel->volid[4], vlabel->volid[5] );

        LOG_DEBUG("     VTOC cc-hh-b : %d-%d-%d\n", vlabel->vtoc.cc, vlabel->vtoc.hh, vlabel->vtoc.b );

        vtoc_lba = cchhb2blk( &vlabel->vtoc, &geo ) * blocksize;


        /*
         *  Create private data area for this new DASD object
         */
        create_s390_disk_private_data(ld);
        disk_pdata = get_s390_disk_private_data(ld);
        if (disk_pdata == NULL) {
                rc = ENOMEM;
                LOG_EXIT_INT(rc);
                return rc;
        }
        else {
                disk_pdata->vsectors_per_block = blocksize;
                memcpy( &disk_pdata->vlabel, vlabel, sizeof(volume_label_t) );
        }


        /*
         *  Determine the disk type
         */
        disk_pdata->disk_layout = get_partition_type( (char *) vlabel );


        /*
         *  Create evms segments for the 390 partitions
         */
        switch ( disk_pdata->disk_layout ) {
        case ibm_unformatted:

                LOG_DEBUG("Disk Type : unformatted\n");

                md = create_metadata_segment(output_list, ld, 0, ld->size, seg_count);
                if ( md ) {
                        disk_pdata->partition_count = 0;
                        disk_pdata->max_partitions_allowed = 0;
                        disk_pdata->md = md;
                        rc = 0;
                }
                else {
                        rc = ENOMEM;
                }
                break;

        case ibm_cms:    // cms minidisk

                LOG_DEBUG("Disk Type : cms\n");

                if (*((long *)vlabel + 13) != 0) {
                        long *label=(long*)vlabel;
                        blocksize = label[3] >> EVMS_VSECTOR_SIZE_SHIFT; // override hard_sector size???????
                        offset = label[13];
                        size = (label[7] - 1) * blocksize;
                        LOG_DEBUG("(MDSK)");
                }
                else {
                        offset = VOLUME_LABEL_BLOCK_NUMBER + 1;
                        size = ld->size;
                }

                offset *= blocksize;
                size   -= offset;

                md = create_metadata_segment(output_list, ld, 0, offset, seg_count);
                if ( md==NULL ) {
                        LOG_ERROR("error, unable to create a metadata segment for %s\n", ld->name );
                        break;
                }

                seg = create_data_segment( output_list,
                                           ld,
                                           offset,
                                           size,
                                           DATA_TYPE,
                                           1,
                                           seg_count);
                if (seg) {
                        disk_pdata->partition_count = 1;
                        disk_pdata->max_partitions_allowed = 1;
                        disk_pdata->md = md;
                        rc = 0;
                }
                else {
                        rc = ENOMEM;
                }
                break;

        case ibm_ldl:    // linux disk layout
        case ibm_none:    // unlabeled disk

                LOG_DEBUG("Disk Type : ldl or unlabeled or unknown label\n");

                offset = VOLUME_LABEL_BLOCK_NUMBER + 1;    // single partition starts immediately after
                offset *= blocksize;                       // the label record in track 1.
                size   = ld->size - offset;

                md = create_metadata_segment( output_list, ld, 0, offset, seg_count );
                if (md==NULL) {
                        LOG_ERROR("error, unable to create a metadata segment for %s\n", ld->name );
                        break;
                }

                seg = create_data_segment( output_list,
                                           ld,
                                           offset,
                                           size,
                                           DATA_TYPE,
                                           1,
                                           seg_count);

                if (seg) {
                        disk_pdata->partition_count = 1;
                        disk_pdata->max_partitions_allowed = 1;
                        disk_pdata->md = md;
                        rc = 0;
                }
                else {
                        rc = ENOMEM;
                }
                break;

        case ibm_cdl:    // common disk layout

                LOG_DEBUG("Disk Type : cdl\n");

                disk_pdata->vtoc_lba               = vtoc_lba;  // give the disk a vtoc
                disk_pdata->vtoc_record_count      = ld->geometry.sectors_per_track;
                disk_pdata->max_partitions_allowed = ld->geometry.sectors_per_track;

                LOG_DEBUG(" vtoc record count = %d\n", disk_pdata->vtoc_record_count);
                LOG_DEBUG(" vtoc lba          = %"PRIu64"\n", vtoc_lba );

                io_start              = vtoc_lba;

                rc = READ(ld, io_start, 1, (void *) vtoc_record );

                disk_pdata->partition_count = 0;

                // read vtoc records ... terminate when :
                // (1) we hit first NULL record
                // (2) we get an error processing a vtoc record
                // (3) we run out of vtoc records to process
                while ( f1->DS1FMTID != 0 && rc == 0 && vtoc_index < disk_pdata->vtoc_record_count ) {

                        if (f1->DS1FMTID == 0xf1) {  // format 1 == data set record

                                //   display_format1_dscb( f1 );

                                if ( cchh2blk(&f1->DS1EXT2.llimit, &geo) != 0 ||
                                     cchh2blk(&f1->DS1EXT3.llimit, &geo) != 0 ) {

                                        LOG_ERROR("error, vtoc format 1 record is using > 1 data extent.\n");
                                        rc = EPROTO;

                                }
                                else {

                                        offset = cchh2blk(&f1->DS1EXT1.llimit, &geo);
                                        psize  = cchh2blk(&f1->DS1EXT1.ulimit, &geo) - offset + geo.sectors;

                                        ++next_minor;

                                        seg = create_data_segment( output_list,
                                                                   ld,
                                                                   offset * blocksize,
                                                                   psize * blocksize,
                                                                   DATA_TYPE,
                                                                   next_minor,
                                                                   seg_count);

                                        if (seg) {
                                                ++disk_pdata->partition_count;
                                                pdata = (SEG_PRIVATE_DATA *)seg->private_data;
                                                memcpy( &pdata->f1, f1, sizeof( format1_label_t ) );
                                                rc = 0;
                                        }
                                        else {
                                                LOG_ERROR("error, unable to create data segment\n");
                                                rc = ENOMEM;
                                        }

                                }

                        }
                        else if (f1->DS1FMTID == 0xf4) {  // format 4 == vtoc descriptor

                                lba_t vstart = cchh2blk(&f4->DS4VTOCE.llimit, &geo);
                                lba_t vend   = cchh2blk(&f4->DS4VTOCE.ulimit, &geo);

                                // display_format4_dscb(f4,vstart,vend,blocksize);

                                memcpy( &disk_pdata->f4, f1, sizeof(format4_label_t) );

                                disk_pdata->vtoc_record_count      = (vend - vstart) + ld->geometry.sectors_per_track;
                                disk_pdata->vtoc_sector_count      = disk_pdata->vtoc_record_count * disk_pdata->vsectors_per_block;
                                disk_pdata->max_partitions_allowed = disk_pdata->vtoc_record_count - 1;

                                LOG_DEBUG(" vtoc record count : %d\n", disk_pdata->vtoc_record_count );
                                LOG_DEBUG(" vtoc sector count : %"PRIu64"\n", disk_pdata->vtoc_sector_count );
                                LOG_DEBUG(" max partitions    : %d\n", disk_pdata->max_partitions_allowed );

                                rc = 0;
                        }
                        else if (f1->DS1FMTID == 0xf5) {  // format 5 == freespace descriptor

                                // display_format5_dscb( f5 );

                                --disk_pdata->max_partitions_allowed;

                                rc = 0;
                        }
                        else {                            // format ? == ignore unknown descriptors

                                LOG_DEBUG("format ? - unknown vtoc record type = 0x%X\n", f1->DS1FMTID );

                                rc = 0;
                        }


                        if (rc == 0) {
                                ++vtoc_index;
                                io_start += blocksize;
                                rc = READ(ld, io_start, 1, (void *) vtoc_record);
                        }

                }

                if (rc == 0) {

                        sector_count_t  md_size = vtoc_lba + disk_pdata->vtoc_sector_count;

                        md = create_metadata_segment( output_list, ld, 0, md_size, seg_count );
                        if (md==NULL) {
                                LOG_ERROR("error, unable to create a metadata segment for %s\n", ld->name );
                                rc = ENOMEM;
                                break;
                        }
                        else {
                                disk_pdata->md = md;
                                find_freespace_on_s390_disk( ld );
                        }

                }

                break;

        default:
                rc = EINVAL;
        }


        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function: get_devmap_info
 *
 *  Called to test if the segment has an active device mapper
 *  node in the kernel and set the object info accordingly.
 */
static int get_devmap_info( DISKSEG *seg )
{
	int rc = 0;
	dm_target_t *targets=NULL;
	dm_device_t *dev=NULL;

	LOG_ENTRY();

	if (seg->data_type != DATA_TYPE) {
		goto out;
	}

	rc = EngFncs->dm_update_status(seg);
	if (rc) {
		goto out;
	}

	if (!(seg->flags & SOFLAG_ACTIVE)) {
		rc = ENODEV;
		goto out;
	}

	rc = EngFncs->dm_get_targets(seg, &targets);
	if (rc) {
		goto out;
	}

	if (!targets ||               // got a target list
	    targets->next != NULL ||  // with 1 target only
	    !targets->data.linear ||  // with device data
	    targets->start != 0) {    // and target starts at Zero
		rc = EINVAL;
		goto out;
	}

	dev = targets->data.linear;

	LOG_DEBUG("seg start = %"PRIu64"   seg size = %"PRIu64"\n", seg->start, seg->size );
	LOG_DEBUG("krnl start = %"PRIu64"  krnl size = %"PRIu64"\n", dev->start, targets->length);

	if (seg->start != dev->start ||
	    seg->size  != targets->length) {
		LOG_DEBUG("metadata doesnt match kernel object\n");
		rc = EINVAL;
		goto out;
	}

	/* Everything matches. We can copy the segment name to the dev_name. */
	strncpy(seg->dev_name, seg->name, EVMS_NAME_SIZE);

out:
	EngFncs->dm_deallocate_targets(targets);
        LOG_EXIT_INT(0);
        return 0;
}


/*
 *  Called to run discovery code on list of evms objects that
 *  the engine has found so far.  We essentially need to walk the
 *  list of objects, looking for Logical Disks, and see if we
 *  recognize partitioning schemes.  If so, consume the logical
 *  disk by removing it from the list, and place all new segment
 *  objects on the output_object list. Any object we dont like in
 *  the input_object list must be copied to the ouput_object list.
 *
 */
static int S390_Discover( list_anchor_t input_objects, list_anchor_t output_objects, boolean final_call)
{
	int  rc = 0;
	int count = 0;
	storage_object_t *disk, *segment;
	list_element_t iter,iter2;

	LOG_ENTRY();

	LOG_DEBUG("input list has %d objects\n", EngFncs->list_count(input_objects));

	LIST_FOR_EACH(input_objects, iter, disk) {

		LOG_DEBUG("examining object %s\n", disk->name);

		if (disk->object_type != DISK) {
			EngFncs->insert_thing(output_objects, disk, INSERT_AFTER, NULL);
			continue;
		}

		rc = s390_check_object(disk, output_objects, &count);
		if (rc == 0) {
			LIST_FOR_EACH(disk->parent_objects, iter2, segment) {
				get_devmap_info(segment);
			}
		} else {
			prune_s390_segments_from_list(disk->parent_objects);
			delete_s390_disk_private_data(disk);
			EngFncs->insert_thing(output_objects, disk, INSERT_AFTER, NULL);
		}
        }

	LOG_DEBUG("created %d segments\n", count);
	LOG_EXIT_INT(count);
	return count;
}



/*
 *  This seg mgr does not support assign/unassign functions
 */
static int S390_Assign( storage_object_t *object, option_array_t *options )
{
        LOG_ENTRY();
        LOG_EXIT_INT(ENOSYS);
        return ENOSYS;
}





/*
 * Can you unassign your plug-in from this object?
 */
static int S390_CanUnassign( LOGICALDISK * ld )
{
        LOG_ENTRY();
        LOG_EXIT_INT(ENOSYS);
        return ENOSYS;
}


/*
 *  Called by engine to have our partitioning scheme removed from
 *  the specified disk storage object.
 */
static int S390_UnAssign( LOGICALDISK * ld )
{
        LOG_ENTRY();
        LOG_EXIT_INT(ENOSYS);
        return ENOSYS;
}


/*  Called to create the specified segment.
 *
 *  - allocate memory for segment,
 *  - add segment to seg list of logical disk.
 *  - fill in seg info
 */
static int CreateDiskSegment( storage_object_t      * freespace,            /* a free space disk segment      */
                              storage_object_t    * * new_segment,          /* resulting data segment         */
                              sector_count_t          size,                 /* size in sectors                */
                              sector_count_t          sector_offset )       /* sector offset in freespace seg */
{
        int                   rc=0;
        storage_object_t     *seg=NULL;
        SEG_PRIVATE_DATA     *pdata=NULL;
        LOGICALDISK          *ld=NULL;
        DISK_PRIVATE_DATA    *disk_pdata=NULL;
        sector_count_t        sizeof_freespace = freespace->size;
        lba_t                 startof_freespace = freespace->start;
        sector_count_t        seg_size;
        lba_t                 seg_start_lba;
        lba_t                 seg_end_lba;
        sector_count_t        cylsize;
        int                   next_minor;


        LOG_ENTRY();

        ld = get_logical_disk(freespace);
        if (ld == NULL) {
                LOG_ERROR("error, unable to get gpt logical disk segment from freespace parm (%s)\n", freespace->name );
                rc = EINVAL;
                LOG_EXIT_INT(rc);
                return rc;
        }
        disk_pdata = get_s390_disk_private_data(ld);
        if (disk_pdata == NULL) {
                LOG_ERROR("error, unable to get gpt logical disk private data from disk (%s)\n", ld->name );
                rc = EINVAL;
                LOG_EXIT_INT(rc);
                return rc;
        }
        cylsize = get_cylinder_size(ld);
        if (cylsize==0) {
                LOG_ERROR("error, unable to get the cylinder size of disk %s.\n", ld->name );
                rc = EINVAL;
                LOG_EXIT_INT(rc);
                return rc;
        }


        /*
         *  Dump debug info to engine log
         */
        LOG_DEBUG("New Seg Parms ...\n");
        LOG_DEBUG("         Size: %"PRIu64"\n", size);
        LOG_DEBUG("       Offset: %"PRIu64"\n", sector_offset );
        LOG_DEBUG("FreeSpace ...\n");
        LOG_DEBUG("    Start LBA: %"PRIu64"\n", freespace->start);
        LOG_DEBUG("         Size: %"PRIu64"\n", freespace->size);
        LOG_DEBUG("     Cyl Size: %"PRIu64"\n", get_cylinder_size(ld) );
        if (starts_on_cylinder_boundary( ld, freespace->start )==TRUE) {
                LOG_DEBUG("  Cyl Bdy: Yes ... starts on cylinder boundary\n" );
        }
        else {
                LOG_DEBUG("  Cyl Bdy: No ... does not start on cylinder boundary\n");
        }

        /*
         *  See if we can create another segment on the specified logical disk
         */
        if ( i_can_add_partition_to_vtoc( ld ) == FALSE ) {
                LOG_DEBUG("disk already has max allowed data partitions\n");
                rc = EINVAL;
                LOG_EXIT_INT(rc);
                return rc;
        }


        /*
         *  Check if we will be creating more segments than FDISK will be able
         *  to see.
         */
        if ( disk_pdata->partition_count >= 3 ) {

                char * choices[] = {_("Yes"), _("No"), NULL};
                int answer = 0;

                QUESTION( &answer, choices,
                          "\nQuestion: You are about to create another partition on drive %s, which already has %d partitions. "
                          "FDISK can only see the first 3 partitions on a CDL disk and will not be able to see the new partition. "
                          "However, EVMS tools will be able to work just fine with the new partition.\n\n"
                          "Do you want to continue and create the new partition?\n",
                          ld->name, disk_pdata->partition_count );

                if (answer == 1) { // if user replied NO
                        rc = EPERM;    // then return ... operation not permitted
                        LOG_EXIT_INT(rc);
                        return rc;
                }

        }


        /*
         *  Always start segment on a track boundary
         *
         */
        seg_start_lba = freespace->start + sector_offset;


        /*
         *  End segment on a cylinder boundary
         */
        seg_end_lba = seg_start_lba + size - 1;

        if ( ends_on_cylinder_boundary(ld, seg_end_lba) == FALSE ) {

                // if less than a cylinder ... round up to next cyl boundary
                if ( ((seg_end_lba - seg_start_lba + 1) / cylsize) == 0) {
                        seg_end_lba = roundup_to_cylinder_boundary(ld, seg_end_lba);
                }
                else {
                        seg_end_lba = rounddown_to_cylinder_boundary(ld, seg_end_lba) - 1;
                }

                // if we goofed somehow and rounded down to start of seg then fix it!
                if ( seg_start_lba >= seg_end_lba ) {
                        seg_end_lba     = roundup_to_cylinder_boundary(ld, seg_start_lba+1);
                }

        }

        /*
         *  Now get its actual size
         */
        seg_size = seg_end_lba - seg_start_lba + 1;


        /*
         * Test starting LBA
         */
        if (  seg_start_lba < freespace->start ) {

                LOG_ERROR("error, with cylinder allignment, the new seg would start before the free_space segment\n");
                *new_segment = NULL;
                rc = EINVAL;
                LOG_EXIT_INT(rc);
                return rc;

        }

        /*
         *  Test ending LBA
         */
        if (  seg_end_lba > (freespace->start + freespace->size -1) ) {

                LOG_ERROR("error, with cylinder allignment the new segment would run past end of free_space segment\n");
                *new_segment = NULL;
                rc = EINVAL;
                LOG_EXIT_INT(rc);
                return rc;

        }


        /*
         *  Calculate new dimensions for the existing freespace segment
         */
        if ( seg_start_lba == freespace->start ) {
                freespace->size  -= seg_size;
                freespace->start += seg_size;
        }
        else {
                freespace->size   = seg_start_lba - freespace->start;
        }


        /*
         *  Dump debug info to engine log
         */
        LOG_DEBUG("Create Parms:\n");
        LOG_DEBUG("FreeSpace: start_lba= %"PRIu64"  sector_offset= %"PRIu64"  size= %"PRIu64"  name= %s\n", freespace->start, sector_offset, freespace->size,freespace->name );
        LOG_DEBUG("   NewSeg: start_lba= %"PRIu64"  end_lba= %"PRIu64"  size= %"PRIu64"\n", seg_start_lba, seg_end_lba, seg_size);


        seg = allocate_s390_disk_segment( ld );
        if (seg) {

                next_minor = get_next_s390_minor(ld);
                if (next_minor != -1) {

                        pdata = (SEG_PRIVATE_DATA *)seg->private_data;

                        seg->size        = seg_size;
                        seg->start       = seg_start_lba;
                        seg->data_type   = DATA_TYPE;

                        pdata->minor     = next_minor;

                        sprintf(seg->name, "%s%d", ld->name, next_minor );

                        rc = init_format1_dscb( ld, seg );
                        if (rc == 0) {

                                if ( insert_s390_segment_into_list( ld->parent_objects, seg ) == NULL ) {

                                        LOG_ERROR("error, some kind of list insert error");
                                        rc = ENOMEM;

                                }
                                else {

                                        if ( freespace->size == 0 ) {
                                                LOG_DEBUG("removing freespace segment from disk list because we used it all up\n");
                                                remove_s390_segment_from_list(ld->parent_objects, freespace );
                                                free_s390_disk_segment(freespace);
                                        }

                                        find_freespace_on_s390_disk( ld );  // re-examine the disk for any free space and expose it by
                                                                            // hanging free space segments.

                                        ++disk_pdata->partition_count;

                                        rc = 0;
                                }
                        }
                        else {
                                LOG_ERROR("error, failed to initialize the new DSCB\n");
                        }

                }
                else {
                        LOG_ERROR("error, no room left in partition table on disk %s.\n",ld->name);
                        rc = EINVAL;
                }

        }

        if (rc) {
                freespace->size  = sizeof_freespace;  // restore freespace segment
                freespace->start = startof_freespace;

                if (seg) free_s390_disk_segment(seg);  // free memory
                seg = NULL;
        }

        *new_segment = seg;            // return the new data segment to the caller

        LOG_EXIT_INT(rc);
        return rc;
}


static int GetCreateOptions(  storage_object_t  * seg,
                              option_array_t    * options,
                              sector_count_t    * size,
                              lsn_t             * sector_offset )
{
        int i;
        int rc = EINVAL;

        LOG_ENTRY();

        // init the options before beginning
        *sector_offset = 0;
        *size = 0;

        // pickup option values
        for (i = 0; i < options->count; i++) {

                if (options->option[i].is_number_based) {

                        switch (options->option[i].number) {

                        case SEG_CREATE_OPTION_SIZE_INDEX:
                                *size = options->option[i].value.ui64;
                                break;

                        case SEG_CREATE_OPTION_OFFSET_INDEX:
                                *sector_offset = options->option[i].value.ui64;
                                break;

                        default:
                                break;

                        }

                }
                else {

                        if (strcmp(options->option[i].name, SEG_CREATE_OPTION_SIZE_NAME) == 0) {
                                *size = options->option[i].value.ui64;
                        }
                        else if (strcmp(options->option[i].name, SEG_CREATE_OPTION_OFFSET_NAME) == 0) {
                                *sector_offset = options->option[i].value.ui64;
                        }

                }
        }

        // test results
        if ( *size != 0 && (*size + *sector_offset) <= seg->size ) {

                rc = 0;

        }


        LOG_EXIT_INT(rc);
        return rc;
}






/*
 * Create storage_object_t(s) from the list of objects using the given
 * options.  Return the newly allocated storage_object_t(s) in new_objects
 * list.
 */
static int S390_CreateSegment( list_anchor_t          input_objects,
                               option_array_t * options,
                               list_anchor_t          new_objects)
{
        int                 rc;
        uint                object_count=0;
        LOGICALDISK        *ld=NULL;
        DISK_PRIVATE_DATA  *disk_pdata=NULL;
        list_element_t      e;
        storage_object_t   *free_space_seg;
        storage_object_t   *newseg;

        //  Options ...
        sector_count_t      size;
        sector_count_t      sector_offset;

        LOG_ENTRY();

        // we should only be called with a single input object
        if ( EngFncs->list_count(input_objects) ==1 ) {

                free_space_seg = EngFncs->first_thing( input_objects, NULL);
                if ( free_space_seg != NULL) {

                        if ( ( i_can_modify(free_space_seg) == TRUE ) &&
                             ( free_space_seg->data_type == FREE_SPACE_TYPE )) {

                                // get seg create options
                                rc = GetCreateOptions( free_space_seg,
                                                       options,
                                                       &size,
                                                       &sector_offset );

                                if (rc) {
                                        LOG_ERROR("invalid options\n");
                                        LOG_EXIT_INT(EINVAL);
                                        return EINVAL;
                                }



                                rc = CreateDiskSegment( free_space_seg,
                                                        &newseg,
                                                        size,
                                                        sector_offset );

                                if (rc == 0) {
                                        e= EngFncs->insert_thing( new_objects,
                                                                  newseg,
                                                                  INSERT_AFTER | EXCLUSIVE_INSERT,
                                                                  NULL );
                                        if (e != NULL) {
                                                rc = 0;
                                                fixup_390_partition_names( get_logical_disk(newseg) );
                                                ld = get_logical_disk(newseg);
                                                disk_pdata = get_s390_disk_private_data(ld);
                                                if (disk_pdata) {
                                                        disk_pdata->flags |= DISK_HAS_CHANGES_PENDING;
                                                }
                                        }
                                        else {
                                                rc = EPERM;
                                        }
                                }


                        }
                        else {
                                LOG_ERROR("object, to be consumed by create, has the wrong data_type\n");
                                rc = EINVAL;
                        }

                }
                else {
                        LOG_ERROR("error returned from list GetObject call\n");
                        rc = EINVAL;
                }

        }
        else {
                LOG_ERROR("expected 1 object in the input list but found %d\n", object_count);
                rc = EINVAL;
        }

        LOG_EXIT_INT(rc);
        return rc;
}



/*
 *  Called to free a data segment.  If the corresponding disk partition
 *  is a logical drive then we also need to free the EBR segment
 *  as well.
 */
static int S390_DestroySegment( storage_object_t * seg, list_anchor_t child_objects )
{
        int                rc = EINVAL;
        DISKSEG           *md=NULL;
        LOGICALDISK       *ld=NULL;
        DISK_PRIVATE_DATA *disk_pdata=NULL;

        LOG_ENTRY();

        if ( ( seg ) &&
             ( seg->object_type == SEGMENT ) &&
             ( seg->data_type   == DATA_TYPE ) &&
             ( i_can_modify( seg ) == TRUE)) {

                LOG_DEBUG("seg: %s\n", seg->name );

                ld = get_logical_disk(seg);
                if (ld) {

                        disk_pdata = get_s390_disk_private_data( ld );
                        if (disk_pdata) {

                                md = get_metadata_segment_from_disk( ld );
                                if ( md ) {

                                        rc = remove_s390_segment_from_list( ld->parent_objects, seg );
                                        if (rc == 0) {

                                                free_s390_disk_segment(seg);

                                                find_freespace_on_s390_disk( ld );

                                                fixup_390_partition_names( ld );

                                                md->flags |= SOFLAG_DIRTY;

                                                disk_pdata->flags |= DISK_HAS_CHANGES_PENDING;

                                                --disk_pdata->partition_count;

                                        }

                                }
                        }

                }

        }


        LOG_EXIT_INT(rc);
        return  rc;
}

/*
 * Forget about these objects.  Don't delete them.  Just clean up any
 * data structures you may have associated with them.  The Engine will
 * call to deactivate the objects durring commit.
 */
static int S390_Discard(list_anchor_t objects)
{
	int rc=0;
	DISKSEG *md=NULL;
	DISKSEG *seg;
	LOGICALDISK *ld=NULL;
	DISK_PRIVATE_DATA *disk_pdata=NULL;
	list_element_t iter;

	LOG_ENTRY();

	LIST_FOR_EACH( objects, iter, seg ) {
		
		if (( seg->object_type == SEGMENT ) &&
			( seg->data_type   == DATA_TYPE ) &&
			( i_can_modify( seg ) == TRUE) ) {

			ld = get_logical_disk(seg);
			if (ld) {
				
				disk_pdata = get_s390_disk_private_data( ld );
				if (disk_pdata) {
					md = get_metadata_segment_from_disk( ld );
					if (md) {
						remove_s390_segment_from_list( ld->parent_objects, seg);
						free_s390_disk_segment(seg);
						
						if (EngFncs->list_empty(ld->parent_objects) == TRUE) {
							// toss our logical disk private data area
							delete_s390_disk_private_data(ld);
						}
					}
				}
			}
		}
		else {
			rc = EINVAL;  // but continue !
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static void get_expand_options( option_array_t * options,  sector_count_t  * size)
{
        int i;

        LOG_ENTRY();

        for (i = 0; i < options->count; i++) {

                if (options->option[i].is_number_based) {

                        if (options->option[i].number == SEG_EXPAND_OPTION_SIZE_INDEX) {
                                *size = options->option[i].value.ui64;
                        }

                }
                else {

                        if (strcmp(options->option[i].name, SEG_EXPAND_OPTION_SIZE_NAME) == 0) {
                                *size = options->option[i].value.ui64;
                        }

                }
        }


        LOG_EXIT_VOID();
}


/*
 *  Called to expand a data segment.  The segment will be expanded
 *  into the freespace segment that follows the data segment.
 */
static int S390_Expand( storage_object_t *seg, storage_object_t *expand_object, list_anchor_t  objects, option_array_t *options )
{
        int               rc = EINVAL;
        sector_count_t    expand_sectors=0;
        DISKSEG          *freespace;
        lba_t             end_lba;
        sector_count_t    old_seg_size;
        LOGICALDISK      *ld=NULL;
        DISK_PRIVATE_DATA *disk_pdata=NULL;
        SEG_PRIVATE_DATA *pdata=NULL;
        sector_count_t    cylinder_size=0;
        sector_count_t    max_expand_sectors=0;
        lba_t             freespace_end_lba;


        LOG_ENTRY();

        // initial checks to see if we can do the expand
        if ( ( seg ) &&
             ( seg == expand_object ) &&
             ( seg->object_type == SEGMENT ) &&
             ( seg->data_type   == DATA_TYPE ) &&
             ( i_can_modify( seg ) == TRUE )) {

                pdata     = (SEG_PRIVATE_DATA *)seg->private_data;
                ld        = get_logical_disk(seg);
                disk_pdata = get_s390_disk_private_data(ld);
                freespace = get_freespace_following_s390_segment( seg );

                get_expand_options( options,  &expand_sectors);

                if ( freespace && ld && expand_sectors && disk_pdata ) {

                        LOG_DEBUG("     Data Seg  Name: %s\n", seg->name);
                        LOG_DEBUG("              Start: %"PRIu64"\n", seg->start);
                        LOG_DEBUG("               Size: %"PRIu64"\n", seg->size);
                        LOG_DEBUG("     Freespace Name: %s\n", freespace->name);
                        LOG_DEBUG("              Start: %"PRIu64"\n", freespace->start);
                        LOG_DEBUG("               Size: %"PRIu64"\n", freespace->size);

                        cylinder_size = get_cylinder_size(ld);

                        // just in case ...
                        if ( freespace->size < cylinder_size ) {
                                LOG_ERROR("error, trying to expand into free space that is less than 1 cylinder\n");
                                LOG_EXIT_INT(rc);
                                return rc;
                        }

                        // partitions end on a cylinder boundary. if freespace doesnt end on
                        // a cylinder bdy then round it down to a cyl bdy before testing if
                        // freespace can handle the ExpandBy.
                        if (ends_on_cylinder_boundary(ld,freespace->start+freespace->size-1)) {
                                freespace_end_lba = freespace->start+freespace->size-1;
                        }
                        else {
                                freespace_end_lba = rounddown_to_cylinder_boundary(ld, freespace->start+freespace->size-1) - 1;
                        }

                        // calculate the max useable size of the freespace area, i.e. the
                        // max area that ends on a cylinder boundary.
                        if (freespace_end_lba > freespace->start) {
                                max_expand_sectors = freespace_end_lba - freespace->start + 1;
                        }
                        else {
                                LOG_ERROR("error, cant cylinder allign end of segment in available freespace segment\n");
                                LOG_EXIT_INT(rc);
                                return rc;
                        }

                        // you just never know ...
                        if ( expand_sectors > max_expand_sectors ) {
                                expand_sectors = max_expand_sectors;
                        }

                        // do cylinder alignment
                        end_lba = seg->start + seg->size + expand_sectors - 1;

                        if (ends_on_cylinder_boundary(ld, end_lba)==FALSE) {
                                end_lba = roundup_to_cylinder_boundary( ld, end_lba );
                        }

                        // now adjust downwards if too big for freespace ...
                        if ( end_lba > (freespace->start + freespace->size - 1) ) {
                                end_lba = rounddown_to_cylinder_boundary(ld, end_lba-1 ) - 1;
                        }

                        // test if we can expand the data seg into the freespace area
                        if ( ( end_lba > freespace->start ) &&
                             ( end_lba <= freespace->start+freespace->size - 1) ) {

                                // calc actual expand sector count
                                expand_sectors = end_lba - freespace->start + 1;

				// ask the engine if this expand amount is ok.
				rc = EngFncs->can_expand_by(seg, &expand_sectors);
				if (rc) {
					LOG_ERROR("Shrink of segment %s rejected by "
						  "the engine.\n", seg->name);
					LOG_EXIT_INT(rc);
					return rc;
				}

                                // expand the data segment
                                old_seg_size = seg->size;
                                seg->size   += expand_sectors;

                                // shrink the freespace segment
                                freespace->start  += expand_sectors;
                                freespace->size   -= expand_sectors;

                                // mark seg dirty for commit and needs DM activate
                                seg->flags |= (SOFLAG_DIRTY | SOFLAG_NEEDS_ACTIVATE);

                                LOG_DEBUG("     Data Seg  Name: %s\n", seg->name);
                                LOG_DEBUG("           New Start: %"PRIu64"\n", seg->start);
                                LOG_DEBUG("           New Size: %"PRIu64"\n", seg->size);
                                LOG_DEBUG("     Freespace Name: %s\n", freespace->name);
                                LOG_DEBUG("           New Start: %"PRIu64"\n", freespace->start);
                                LOG_DEBUG("           New Size: %"PRIu64"\n", freespace->size);

                                // success.
                                rc = 0;

                                disk_pdata->flags |= DISK_HAS_CHANGES_PENDING;

                                // if we used up all the free space then discard the freespace seg
                                if ( freespace->size == 0 ) {

                                        rc = remove_s390_segment_from_list( ld->parent_objects, freespace );
                                        if (rc==0) {
                                                free_s390_disk_segment(freespace);
                                        }
                                        else {   // error ... backout the expand
                                                LOG_ERROR("error, unable to remove the freespace segment from the disk list\n");
                                                seg->size = old_seg_size;
                                                freespace->start  -= expand_sectors;
                                                freespace->size   += expand_sectors;
                                        }

                                }

                        }
                }
        }


        LOG_EXIT_INT(rc);
        return rc;
}

static void get_shrink_options( option_array_t * options, sector_count_t * size)
{
        int i;

        LOG_ENTRY();

        for (i = 0; i < options->count; i++) {

                if (options->option[i].is_number_based) {

                        if (options->option[i].number == SEG_SHRINK_OPTION_SIZE_INDEX) {
                                *size = options->option[i].value.ui64;
                        }

                }
                else {

                        if (strcmp(options->option[i].name, SEG_SHRINK_OPTION_SIZE_NAME) == 0) {
                                *size = options->option[i].value.ui64;
                        }
                }

        }

        LOG_EXIT_VOID();
}


/*
 *  Called to shrink a data segment to new_size or next smaller increment.
 *  Then, update appropriate fields in the segment and make
 *  changes to freespace segments as needed.
 *
 */
static int S390_Shrink( storage_object_t * seg,
                        storage_object_t * shrink_object,
                        list_anchor_t            objects,
                        option_array_t   * options )
{
        int               rc = EINVAL;
        sector_count_t    shrink_sector_count=0;
        u_int64_t         end_lba;
        LOGICALDISK      *ld=NULL;
        DISK_PRIVATE_DATA *disk_pdata=NULL;
        sector_count_t    new_seg_size=0;
        SEG_PRIVATE_DATA *pdata=NULL;
        sector_count_t    cylinder_size=0;

        LOG_ENTRY();

        // initial checks to see if we can do the shrink
        if ( ( seg ) &&
             ( seg == shrink_object ) &&
             ( seg->object_type == SEGMENT ) &&
             ( seg->data_type   == DATA_TYPE ) &&
             ( i_can_modify( seg ) == TRUE ) ) {

                pdata     = (SEG_PRIVATE_DATA *)seg->private_data;
                ld        = get_logical_disk(seg);
                disk_pdata = get_s390_disk_private_data(ld);

                get_shrink_options( options,  &shrink_sector_count);


                if ( (ld != NULL) &&
                     (disk_pdata != NULL) &&
                     (shrink_sector_count > 0) &&
                     (shrink_sector_count < seg->size ) ) {

                        LOG_DEBUG("     Data Seg  Name: %s\n",   seg->name);
                        LOG_DEBUG("              Start: %"PRIu64"\n", seg->start);
                        LOG_DEBUG("               Size: %"PRIu64"\n", seg->size);
                        LOG_DEBUG("Shrink Sector Count: %"PRIu64"\n", shrink_sector_count );

                        cylinder_size = get_cylinder_size(ld);

                        // we shrink in cylinder size chunks
                        if (shrink_sector_count < cylinder_size) {
                                shrink_sector_count = cylinder_size;
                        }
                        else {
                                shrink_sector_count = (shrink_sector_count/cylinder_size)*cylinder_size;
                        }

			// ask the engine if it's ok to shrink by this amount.
			rc = EngFncs->can_shrink_by(seg, &shrink_sector_count);
			if (rc) {
				LOG_ERROR("Shrink of segment %s rejected by "
					  "the engine.\n", seg->name);
				LOG_EXIT_INT(rc);
				return rc;
			}

                        // resulting seg size
                        new_seg_size = seg->size - shrink_sector_count;

                        // make sure it ends on cylinder boundary
                        end_lba = seg->start + new_seg_size - 1;
                        if (ends_on_cylinder_boundary(ld, end_lba)==FALSE) {
                                end_lba = rounddown_to_cylinder_boundary( ld, end_lba ) - 1;
                        }

                        if ( end_lba >= (seg->start + seg->size - 1) ) {
                                end_lba = rounddown_to_cylinder_boundary(ld, end_lba ) - 1;
                        }

                        // final test if we can shrink the segment
                        if (  ( end_lba > seg->start ) &&
                              ( end_lba < (seg->start+seg->size-1)) ) {

                                // actual new seg size
                                new_seg_size = end_lba - seg->start + 1;

                                // shrink the data segment
                                seg->size = new_seg_size;

                                // expose freespace extent on disk
                                find_freespace_on_s390_disk( ld );

                                // mark seg dirty for commit and needs DM activate
                                seg->flags |= (SOFLAG_DIRTY | SOFLAG_NEEDS_ACTIVATE);

                                // success.
                                rc = 0;

                                disk_pdata->flags |= DISK_HAS_CHANGES_PENDING;

                        }

                }
                else {
                        LOG_ERROR("error, something wrong with shrink sector count, cant shrink segment\n");
                        rc = EINVAL;
                }

        }
        else {
                LOG_ERROR("error, something wrong with parms\n");
                rc = EINVAL;
        }


        LOG_EXIT_INT(rc);
        return rc;
}


static int S390_AddSectorsToKillList( storage_object_t *seg, lsn_t lsn, sector_count_t count)
{
        int                         rc = EINVAL;
        storage_object_t * object = get_logical_disk(seg);

        LOG_ENTRY();

        if (lsn+count > seg->size) {
                LOG_ERROR("KillSectors beyond end of segment, lsn=%"PRIu64" count=%"PRIu64" segsize=%"PRIu64"\n",
                          lsn,count,seg->size);
                LOG_EXIT_INT(EINVAL);
                return EINVAL;
        }
        rc = KILL_SECTORS(object, lsn + seg->start, count);

        LOG_EXIT_INT(rc);
        return rc;
}


static int S390_CommitChanges( storage_object_t *object, uint phase )
{
        int                rc = EINVAL;
        LOGICALDISK       *ld = NULL;
        DISKSEG           *seg = NULL;
        SEG_PRIVATE_DATA  *pdata=NULL;
        DISK_PRIVATE_DATA *disk_pdata;
        list_element_t     iter;

        LOG_ENTRY();
        LOG_DEBUG("object= %s\n", object->name );

        ld = get_logical_disk(object);

        if (ld) {

                disk_pdata = get_s390_disk_private_data(ld);

                if (disk_pdata) {

                        if (phase == SETUP) {
                                if (disk_pdata->flags & DISK_HAS_FORMAT_PENDING) {
                                        rc = format_390_disk( ld,
                                                              disk_pdata->disk_layout,
                                                              disk_pdata->vsectors_per_block << EVMS_VSECTOR_SIZE_SHIFT);
                                        if (!rc) {
                                                LIST_FOR_EACH( ld->parent_objects, iter, seg ){
                                                        seg->flags &= ~SOFLAG_DIRTY;
                                                }
                                                disk_pdata->flags &= ~DISK_HAS_FORMAT_PENDING;
                                        }
                                } else {
					rc = 0;
				}
                        }
                        else if (phase == MOVE) {

                                if ( disk_pdata->flags & DISK_HAS_MOVE_PENDING ) {
                                        LOG_DEBUG("committing move on the disk\n");
                                        pdata = (SEG_PRIVATE_DATA *)object->private_data;
                                        rc = s390_move_segment_commit( object, pdata->move_target, disk_pdata->copy_job );
                                        if (disk_pdata->copy_job) free(disk_pdata->copy_job );
                                        disk_pdata->copy_job = NULL;
                                        disk_pdata->flags   &= ~DISK_HAS_MOVE_PENDING;
                                        object->flags       &= ~SOFLAG_DIRTY;
                                }
                                else {
                                        rc = 0;
                                }

                        }
                        else if ( (phase==FIRST_METADATA_WRITE || phase==SECOND_METADATA_WRITE )&&
                                  !(disk_pdata->flags & DISK_HAS_MOVE_PENDING)) {

                                if ( disk_pdata->flags & DISK_HAS_CHANGES_PENDING ) {
                                        rc = commit_390_partition_table( ld, object, phase, FALSE );
                                        if (!rc) {
                                                LIST_FOR_EACH( ld->parent_objects, iter, seg ){
                                                        seg->flags &= ~SOFLAG_DIRTY;
                                                }
                                                disk_pdata->flags &= ~DISK_HAS_CHANGES_PENDING;
                                        }
                                }
                                else {
                                        rc = 0;
                                }

                        }
                        else {
                                rc = 0;
                        }

                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}


static int s390_backup_metadata( storage_object_t *seg )
{
        LOGICALDISK *ld = get_logical_disk(seg);
	int rc = 0;

	LOG_ENTRY();

	if (seg->data_type != FREE_SPACE_TYPE) {
		rc = commit_390_partition_table( ld, seg, 1, TRUE );
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static int S390_Read( storage_object_t  *seg,
                      lsn_t              offset,
                      sector_count_t     count,
                      void              *buffer )
{
        int    rc;
        storage_object_t * ld = get_logical_disk( seg );

        LOG_ENTRY();

        if (offset+count > seg->size) {
                LOG_ERROR("Read beyond end of segment, offset=%"PRIu64" count=%"PRIu64" segsize=%"PRIu64"\n",
                          offset,count,seg->size);
                rc = EINVAL;
        }
        else {
                rc = READ(ld, offset + seg->start, count, buffer);
        }

        LOG_EXIT_INT(rc);
        return rc;
}



static int S390_Write( storage_object_t  *seg,
                       lsn_t              offset,
                       sector_count_t     count,
                       void              *buffer )
{
        int     rc;
        storage_object_t * ld = get_logical_disk( seg );

        LOG_ENTRY();

        if (offset+count > seg->size) {
                LOG_ERROR("Write beyond end of segment, offset=%"PRIu64" count=%"PRIu64" segsize=%"PRIu64"\n",
                          offset,count,seg->size);
                rc = EINVAL;
        }
        else {
                rc = WRITE(ld, offset + seg->start, count, buffer);
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 * This call notifies you that your object is being made into (or part of)
 * a volume or that your object is no longer part of a volume.  The "flag"
 * parameter indicates whether the volume is being created (TRUE) or
 * removed (FALSE).
 */
static void S390_set_volume(storage_object_t * object, boolean flag)
{
        return;
}



static int GetFormatOptions( option_array_t * options, ibm_label_type_t *disk_layout, u_int32_t *blocksize )
{
        int i;
        int rc = EINVAL;

        LOG_ENTRY();

        // init the options before beginning
        *disk_layout = ibm_none;
        *blocksize   = 0;

        // pickup option values
        for (i = 0; i < options->count; i++) {

                if (options->option[i].is_number_based) {

                        switch (options->option[i].number) {

                        case SEG_FORMAT_OPTION_TYPENAME_INDEX:
                                if ( strstr( options->option[i].value.s, "CDL") ) {
                                        *disk_layout = ibm_cdl;
                                }
                                else if ( strstr( options->option[i].value.s, "LDL") ) {
                                        *disk_layout = ibm_ldl;
                                }
                                break;
                        case SEG_FORMAT_OPTION_BLKSIZE_INDEX:
                                *blocksize = options->option[i].value.ui32;
                                break;

                        default:
                                break;

                        }

                }
                else {

                        if (strcmp(options->option[i].name, SEG_FORMAT_OPTION_TYPENAME_NAME) == 0) {

                                if ( strstr( options->option[i].value.s, "CDL") ) {
                                        *disk_layout = ibm_cdl;
                                }
                                else if ( strstr( options->option[i].value.s, "LDL") ) {
                                        *disk_layout = ibm_ldl;
                                }

                        }
                        else if (strcmp(options->option[i].name, SEG_FORMAT_OPTION_BLKSIZE_NAME) == 0) {
                                *blocksize = options->option[i].value.ui32;
                        }

                }
        }

        // test results
        if (  *disk_layout == ibm_cdl || *disk_layout == ibm_ldl ) {

                if (  *blocksize == 512  ||
                      *blocksize == 1024 ||
                      *blocksize == 2048 ||
                      *blocksize == 4096  ) {

                        rc = 0;

                }

        }


        LOG_EXIT_INT(rc);
        return rc;
}



static int S390_Format( storage_object_t * input_object,
                        option_array_t * options)
{

        int                 rc=EINVAL;
        ibm_label_type_t    disk_layout;
        u_int32_t           blocksize;
        LOGICALDISK        *ld=get_logical_disk(input_object);
        DISKSEG            *md=NULL;
        DISK_PRIVATE_DATA  *disk_pdata=NULL;


        LOG_ENTRY();
        LOG_DEBUG("input object = %s\n", input_object->name );


        // see if we are installing on a DISK and therefore are actually
        // assigning the segment manager to the DISK ...
        //
        // added check for METADATA SEGS because that is how we represent
        // an unformatted disk.
        if (ld) {

                disk_pdata = get_s390_disk_private_data( ld );

                if ( i_can_format_disk( ld ) == TRUE ) {

                        // get disk format options
                        rc = GetFormatOptions( options, &disk_layout, &blocksize );
                        if (rc == 0) {

                                prune_s390_segments_from_list( ld->parent_objects );

                                md = create_s390_metadata_segment( ld, 0, ld->size );
                                if (md) {

                                        if ( insert_s390_segment_into_list( ld->parent_objects, md ) ) {
                                                disk_pdata->md = md;
                                                find_freespace_on_s390_disk( ld );
                                                rc = 0;
                                        }
                                        else {
                                                LOG_ERROR("error, unable to add segment to %s\n", ld->name );
                                                free_s390_disk_segment(md);
                                                rc = ENOMEM;
                                        }

                                }
                                else {
                                        LOG_ERROR("error, unable to create segment for %s\n", ld->name );
                                        rc = ENOMEM;
                                }

                        }

                }

        }


        // in case geometry changed ... also cuz we destroyed
        // existing segments on the disk.
        if (rc == 0) {
                disk_pdata->flags              |= DISK_HAS_FORMAT_PENDING;
                disk_pdata->disk_layout         = disk_layout;
                disk_pdata->vsectors_per_block  = blocksize >> EVMS_VSECTOR_SIZE_SHIFT;
        }

        LOG_EXIT_INT(rc);
        return rc;
}



/*
 * Execute the private action on the object.
 */
static  int S390_PluginPrivateFunctions( storage_object_t * object,
                                         task_action_t      action,
                                         list_anchor_t            objects,
                                         option_array_t   * options)
{
        int rc=EINVAL;
        DISKSEG *freespace=NULL;


        LOG_ENTRY();

        switch (action) {
        case EVMS_Task_Format_Disk:
                rc = S390_Format(object,options);
                break;
        case EVMS_Task_Move_Segment:

                if (EngFncs->list_count(objects) == 1) {
                        freespace = EngFncs->first_thing(objects,NULL);
                        if (freespace) {
                                rc = s390_move_segment(object,freespace);
                        }
                }
                break;

        default:
                rc = ENOSYS;
                break;
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 * Return an array of private actions that you support for this object.
 */
static int S390_GetPluginPrivateFunctions( storage_object_t        * object,
                                           function_info_array_t * * actions)
{
        int                    rc = EINVAL;
        LOGICALDISK           *ld=NULL;
        function_info_array_t *func_info=NULL;
        char                   title[EVMS_VOLUME_NAME_SIZE+1];
        int                    index;

        LOG_ENTRY();

        func_info = EngFncs->engine_alloc( sizeof(function_info_array_t) + (sizeof(function_info_t)* 2) );
        if (func_info) {

                func_info->count = 0;
                index = 0;

                ld = get_logical_disk(object);

                if ( i_can_format_disk( ld ) == TRUE ) {

                        func_info->info[index].function = EVMS_Task_Format_Disk;

                        sprintf(title, _("Format Disk %s"), ld->name );
                        func_info->info[index].title = EngFncs->engine_strdup( title );
                        func_info->info[index].verb = EngFncs->engine_strdup( _("Format") );
                        func_info->info[index].name = EngFncs->engine_strdup( _("Format") );
                        func_info->info[index].help = EngFncs->engine_strdup( _("Use this function to format a dasd device prior to use.") );

                        ++func_info->count;
                        ++index;
                }

                if ( s390_can_move_segment( object) == 0 ) {

                        func_info->info[index].function = EVMS_Task_Move_Segment;

                        func_info->info[index].title = EngFncs->engine_strdup( "Move" );
                        func_info->info[index].verb = EngFncs->engine_strdup( _("Move") );
                        func_info->info[index].name = EngFncs->engine_strdup( _("Move") );
                        func_info->info[index].help = EngFncs->engine_strdup( _("Use this function to move a data segment.") );

                        ++func_info->count;
                        ++index;
                }

                rc = 0;

        }
        else {
                rc = ENOMEM;
        }

        *actions = func_info;

        LOG_EXIT_INT(rc);
        return rc;
}



/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                              PLUGIN FUNCTION TABLE                                   +
+                                                                                      +
+--------------------------------------------------------------------------------------*/
static struct plugin_functions_s sft={

        // the following routines are found above
        setup_evms_plugin:                   S390_SetupEVMSPlugin,
        cleanup_evms_plugin:                 S390_Cleanup,
        can_set_volume:                      S390_can_set_volume,
        can_delete:                          S390_CanDestroy,
        can_expand:                          S390_CanExpand,
        can_expand_by:                       S390_CanExpandBy,
        can_shrink:                          S390_CanShrink,
        can_shrink_by:                       S390_CanShrinkBy,
        discover:                            S390_Discover,
        assign:                              S390_Assign,
        can_unassign:                        S390_CanUnassign,
        unassign:                            S390_UnAssign,
        create:                              S390_CreateSegment,
        delete:                              S390_DestroySegment,
        discard:                             S390_Discard,
        expand:                              S390_Expand,
        shrink:                              S390_Shrink,
        add_sectors_to_kill_list:            S390_AddSectorsToKillList,
        commit_changes:                      S390_CommitChanges,
        read:                                S390_Read,
        write:                               S390_Write,
        set_volume:                          S390_set_volume,
        get_option_count:                    S390_GetOptionCount,
        init_task:                           S390_InitTask,
        set_option:                          S390_SetOption,
        set_objects:                         S390_SetObjects,
        get_info:                            S390_GetInfo,
        get_plugin_info:                     S390_GetPluginInfo,
        get_plugin_functions:                S390_GetPluginPrivateFunctions,
        plugin_function:                     S390_PluginPrivateFunctions,
        can_activate:                        s390_can_activate,
        activate:                            s390_activate,
        can_deactivate:                      s390_can_deactivate,
        deactivate:                          s390_deactivate,
        backup_metadata:                     s390_backup_metadata,
};

/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                       BUILD AND EXPORT AN EVMS PLUGIN RECORD                         +
+                                                                                      +
+--------------------------------------------------------------------------------------*/

static plugin_record_t plugin_record = {

        id:                               SetPluginID(EVMS_OEM_IBM, EVMS_SEGMENT_MANAGER, 2 ),

        version:                          {MAJOR_VERSION, MINOR_VERSION, PATCH_LEVEL},

        required_engine_api_version:      {15, 0, 0},
        required_plugin_api_version:      {plugin: {13, 1, 0}},

        short_name:                       "S390SegMgr",
        long_name:                        "S390 Segment Manager",
        oem_name:                         "IBM",

        functions:                        {plugin: &sft},

        container_functions:              NULL

};

// Vector of plugin record ptrs that we export for the EVMS Engine.
plugin_record_t                *s390_plugin_rec = &plugin_record;
plugin_record_t * evms_plugin_records[] = {&plugin_record, NULL};
