/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libdos.so
 *
 *   File: bsd.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "ptables.h"
#include "segs.h"
#include "embedded.h"
#include "bsd.h"



/*
 * Builds a disk segment matching the bsd partition info
 */
static DISKSEG * build_bsd_segment( LOGICALDISK       *ld,
                                    DISKSEG           *msdos_seg,
                                    u_int32_t          start,
                                    u_int32_t          size,
                                    u_int32_t          sys_id,
                                    u_int32_t          ptable_index,
                                    u_int32_t          minor,
                                    u_int32_t          tag,
                                    u_int32_t          flags,
                                    list_anchor_t      recovery_list )
{
        DISKSEG           *seg     = NULL;
        SEG_PRIVATE_DATA  *pdata   = NULL;
        list_element_t     e;
        int                rc;


        LOG_ENTRY();

        seg = build_segment_for_embedded_partition( ld, msdos_seg, start, size, sys_id, ptable_index, minor );
        if (seg) {

                pdata = (SEG_PRIVATE_DATA *) seg->private_data;

                pdata->flags = SEG_IS_BSD_PARTITION | SEG_IS_EMBEDDED;

                pdata->tag         = tag;
                pdata->permissions = flags;

                e = EngFncs->insert_thing(recovery_list,seg,INSERT_BEFORE,NULL);
                if (e) {
                        rc = 0;
                }
                else {
                        rc = EPERM;
                        free(pdata);
                        free_disk_segment( seg );
                        seg = NULL;
                }

        }

        LOG_EXIT_PTR(seg);
        return seg;
}



int do_bsd_partition_discover( LOGICALDISK *ld, Partition_Record *part )
{
        struct bsd_disklabel       *l;
        struct bsd_partition       *p;
        char                        data[EVMS_VSECTOR_SIZE];
        int                         rc = 0;
        struct plugin_functions_s  *dft;
        DISK_PRIVATE_DATA          *disk_pdata;
        DISKSEG                    *seg;
        DISKSEG                    *bsd_seg;
        int                         bsd_partition_count=0;
        u_int32_t                   ptable_index=0;
        u_int32_t                   minor;
        list_anchor_t               recovery_list;

        LOG_ENTRY();


        // get disk info
        disk_pdata = get_disk_private_data(ld);
        if (disk_pdata == NULL) {
                LOG_EXIT_INT(EINVAL);
                return EINVAL;
        }

        // get disk manager function table
        dft = (struct plugin_functions_s *)ld->plugin->functions.plugin;
        if (dft==NULL) {
                LOG_EXIT_INT(EINVAL);
                return EINVAL;
        }

        // read bsd info
        rc = dft->read( ld,
                        DISK_TO_CPU32( START_LBA(part) ) + BSD_DISKLABEL_PART_TABLE_SECTOR_OFFSET,
                        1,
                        (void *) data );

        if (rc) {
                LOG_EXIT_INT(rc);
                return rc;
        }

        // check for BSD magic number
        l = (struct bsd_disklabel *) data;

        if ( DISK_TO_CPU32(l->d_magic) != BSD_DISKMAGIC) {
                LOG_EXIT_INT(0);
                return 0;
        }

        // create a list
        recovery_list = EngFncs->allocate_list();
        if (recovery_list == NULL) {
                LOG_EXIT_INT(ENOMEM);
                return ENOMEM;
        }

        // get evms segment storage object from disk segment list
        // and remove it from the list, making room for bsd partitions.
        seg = get_matching_segment( (list_anchor_t)  ld->parent_objects,
                                    (lba_t)          DISK_TO_CPU32( START_LBA(part) ),
                                    (sector_count_t) DISK_TO_CPU32( NR_SECTS(part)  ));

        if (seg) {
                rc = remove_diskseg_from_list( ld->parent_objects, seg );
                if (rc) {
                        EngFncs->destroy_list(recovery_list);
                        LOG_EXIT_INT(rc);
                        return rc;
                }
        }
        else {
                EngFncs->destroy_list(recovery_list);
                LOG_EXIT_INT(ENODEV);
                return ENODEV;
        }

        // calculate the first minor to be used by bsd embedded partitions
        minor = disk_pdata->logical_drive_count + disk_pdata->embedded_partition_count + 5;

        LOG_DEBUG("BSD Info:\n");
        LOG_DEBUG("     geometry:  C= %d   H= %d  S= %d\n", DISK_TO_CPU32(l->d_ncylinders), DISK_TO_CPU32(l->d_ntracks), DISK_TO_CPU32(l->d_nsectors));
        LOG_DEBUG("     sector size = %d\n", DISK_TO_CPU32(l->d_secsize) );
        LOG_DEBUG("     number of bsd partition table entries: %d\n", DISK_TO_CPU16(l->d_npartitions) );
        LOG_DEBUG("     size of boot area at sn0 in bytes    : %d\n", DISK_TO_CPU32(l->d_bbsize) );
        LOG_DEBUG("     max size of fs superblock in bytes   : %d\n", DISK_TO_CPU32(l->d_sbsize) );

        rc = 0;
        for (p = l->d_partitions; (p - l->d_partitions < DISK_TO_CPU16(l->d_npartitions) )&&(rc==0); p++) {

                if (p->p_fstype != BSD_TAG_UNUSED ) {

                        LOG_DEBUG("  Slice %d: p_size(%u), p_offset(%u), p_fsize(%u), p_fstype(0x%02X)\n", ptable_index, DISK_TO_CPU32(p->p_size), DISK_TO_CPU32(p->p_offset), DISK_TO_CPU32(p->p_fsize), p->p_fstype );

                        bsd_seg = build_bsd_segment( ld,
                                                     seg,
                                                     DISK_TO_CPU32( p->p_offset ),
                                                     DISK_TO_CPU32( p->p_size ),
                                                     DISK_TO_CPU32( p->p_fstype ),
                                                     ptable_index,
                                                     minor,
                                                     (u_int32_t) p->p_fstype,
                                                     0,
                                                     recovery_list );
                        if (bsd_seg) {

                                if ( insert_diskseg_into_list( ld->parent_objects, bsd_seg) ) {
                                        ++minor;
                                        ++bsd_partition_count;
                                        ++disk_pdata->embedded_partition_count;
                                }
                                else {
                                        rc = EPERM;
                                }

                        }
                        else {
                                rc = ENOMEM;
                        }

                }

                if (rc) break;

                ++ptable_index;
        }

        if (rc) {
                LOG_ERROR("error, problems adding solaris partitions for disk %s.", ld->name );
                remove_embedded_partitions_from_disk( ld, recovery_list );
                insert_diskseg_into_list( ld->parent_objects, seg);
                MESSAGE(_("Abandoning effort with embedded bsd partitions found in %s\n"), seg->name);
                rc = 0;
        }
        else {

                // if we produced -ANY- bsd segments ... then consume the container segment
                if ( bsd_partition_count > 0 ) {
                        diskseg_to_container_segment( seg );
                        EngFncs->concatenate_lists(seg->parent_objects, recovery_list);
                        LOG_DEBUG("Info, found %d embedded bsd partitions in %s\n", bsd_partition_count, seg->name);
                }
                else {
                        insert_diskseg_into_list( ld->parent_objects, seg);
                }

        }

        EngFncs->destroy_list(recovery_list);
        LOG_EXIT_INT(rc);
        return rc;
}



int do_bsd_partition_commit( LOGICALDISK *ld, DISKSEG *seg, DISKSEG *parent, boolean backup )
{
        struct bsd_disklabel       *l;
        struct bsd_partition       *p;
        DISKSEG                    *tseg;
        char                        data[EVMS_VSECTOR_SIZE];
        int                         rc = 0;
        struct plugin_functions_s  *dft;
        DISK_PRIVATE_DATA          *disk_pdata;
        SEG_PRIVATE_DATA           *pdata;
        list_element_t              iter;

        LOG_ENTRY();

        // get disk info
        disk_pdata = get_disk_private_data(ld);
        if (disk_pdata == NULL) {
                LOG_EXIT_INT(ENODEV);
                return ENODEV;
        }

        // get disk manager function table
        dft = (struct plugin_functions_s *)ld->plugin->functions.plugin;
        if (dft==NULL) {
                LOG_EXIT_INT(ENODEV);
                return ENODEV;
        }

        // read bsd info
        rc = dft->read( ld,
                        seg->start + BSD_DISKLABEL_PART_TABLE_SECTOR_OFFSET,
                        1,
                        (void *) data );

        if (rc) {
                LOG_EXIT_INT(rc);
                return rc;
        }

        // check for BSD magic number
        l = (struct bsd_disklabel *) data;

        if ( DISK_TO_CPU32(l->d_magic) != BSD_DISKMAGIC) {
                LOG_EXIT_INT(0);
                return 0;
        }

        // walk through the partition table, removing all the data
        // partitions we find.
        for (p = l->d_partitions; p - l->d_partitions < DISK_TO_CPU16(l->d_npartitions); p++) {
                if (p->p_fstype != BSD_TAG_UNUSED ) {
                        memset(p, 0, sizeof(struct bsd_partition) );
                }
        }


        // now walk through the segment list on this drive and find all
        // the BSD partitions that are embedded in this container seg.
        // and add them to the BSD partition table.
        LIST_FOR_EACH(ld->parent_objects, iter, tseg) {

                pdata = (SEG_PRIVATE_DATA *)tseg->private_data;

                if ( ( pdata->flags & SEG_IS_BSD_PARTITION ) &&
                     ( only_child( tseg ) == seg ) ) {

                        p = &l->d_partitions[pdata->ptable_index];

                        p->p_offset  = CPU_TO_DISK32( tseg->start );
                        p->p_size    = CPU_TO_DISK32( tseg->size );
                        p->p_fstype  = (u_int8_t) pdata->tag;
                        p->p_fsize   = CPU_TO_DISK32( (u_int16_t) pdata->permissions );
                        p->p_frag    = (u_int8_t) pdata->frag;
                        p->p_cpg     = CPU_TO_DISK16( (u_int16_t) pdata->cpg );

                }

        }

        // update BSD info
	if (backup) {
		rc = EngFncs->save_metadata(parent->name, ld->name,
					    seg->start + BSD_DISKLABEL_PART_TABLE_SECTOR_OFFSET,
					    1, data);
	} else {
		rc = dft->write( ld,
				 seg->start + BSD_DISKLABEL_PART_TABLE_SECTOR_OFFSET,
				 1,
				 (void *) data );
	}


        LOG_EXIT_INT(rc);
        return rc;
}




