#include <stdlib.h>
#include "entity.h"
#include "SDL.h"

#include "sdl-common.h"


/**sdl_rectangle_render
 * Render a sdl_sprite to the given SDL surface.
 *
 */
void sdl_rectangle_render(SDL_Rect * dest, SDL_Surface * screen,
	sdl_item * item)
{
  dest->x = item->x;
  dest->y = item->y;
  dest->w = item->width;
  dest->h = item->height;
  SDL_FillRect(screen, dest, item->color); 
  return;
}


/**sdl_rectangle_refresh 
 * Set the refresh bit on our sprite to true.
 */
static gint
sdl_rectangle_refresh (ENode * node, EBuf * attr, EBuf * value)
{
  return TRUE;
}


/**sdl_rectangle_new_pos
 * Set the sprite to a new position, and set the refresh bit.
 */
static gint 
sdl_rectangle_new_pos (ENode * node, EBuf * attr, EBuf * value)
{
  char * xpos = NULL;
  char * ypos = NULL;
  char * width = NULL;
  char * height = NULL;
  sdl_item  * item = NULL;

  item = enode_get_kv(node, "item");

  if (NULL == item)
    {
      g_warning ("Sdl-rectangle: No display item found.\n");
      return FALSE;
    }

  xpos = enode_attrib_str (node, "xpos", NULL);
  ypos = enode_attrib_str (node, "ypos", NULL);
  width = enode_attrib_str (node, "width", NULL);
  height = enode_attrib_str (node, "height", NULL);

  if (NULL == xpos || NULL == ypos || NULL == width || NULL == height)
    {
      g_warning ("Sdl-rectangle: Bad x,y,width, or height position\n");
      return FALSE;
    }

  item->x = atoi (xpos);
  item->y = atoi (ypos);
  item->width = atoi (width);
  item->height = atoi (height);

  enode_set_kv (node, "item", item);

  return TRUE;
}


static gint
sdl_rectangle_set_color(ENode * node, EBuf * attr, EBuf * value)
{
  int red, blue, green;
  sdl_item  * item = NULL;
  ENode * parent;
  SDL_Surface * screen;

  /* Pull out the garray. */
  parent = enode_parent (node, "sdl-window");

  if (NULL == parent)
    {
      g_warning("Sdl-rectangle: Need parent of <sdl-window> type!\n");
      return FALSE;
    }

  /* Get our SDL surface */
  screen = enode_get_kv (parent, "surface");

  if (NULL == screen)
    {
      g_warning("Sdl-rectangle: couldn't get valid screen.\n");
      return FALSE;
    }

  item = enode_get_kv(node, "item");

  if (NULL == item)
    {
      g_warning ("Sdl-rectangle: No display item found.\n");
      return FALSE;
    }

  red = atoi (enode_attrib_str (node, "red", NULL) );
  blue = atoi (enode_attrib_str (node, "blue", NULL) );
  green = atoi (enode_attrib_str (node, "green", NULL) );

  item->color = SDL_MapRGB(screen->format, red, green, blue);
  item->needs_refresh = TRUE;

  enode_set_kv (node, "item", item);

  return TRUE;
}

/**sld_rectangle_setup
 * setup the struct.
 */
static void
sdl_rectangle_setup (ENode * node)
{
  ENode * parent;
  GPtrArray * item_list = NULL;
  SDL_Surface * screen;
  char * src_str = NULL;
  char * xpos = NULL;
  char * ypos = NULL;
  char * width = NULL;
  char * height = NULL;
  int red, blue, green;
  sdl_item * item;

  /* Pull out the garray. */
  parent = enode_parent (node, "sdl-window");

  if (NULL == parent)
    {
      g_warning("Sdl-rectangle: Need parent of <sdl-window> type!\n");
      return;
    }

  /* Get our SDL surface */
  screen = enode_get_kv (parent, "surface");

  if (NULL == screen)
    {
      g_warning("Sdl-rectangle: couldn't get valid screen.\n");
      return;
    }

  /* Get item list. */
  item_list = enode_get_kv (parent, "item_list");

  if (NULL == item_list)
    {
      g_warning ("Sdl-rectangle: Couldn't get sdl display list\n");
      return;
    }

  src_str = enode_attrib_str (node, "src", NULL);

  xpos = enode_attrib_str (node, "ypos", NULL);
  ypos = enode_attrib_str (node, "xpos", NULL);
  width = enode_attrib_str (node, "width", NULL);
  height = enode_attrib_str (node, "height", NULL);

  if (NULL == xpos || NULL == ypos || NULL == width || NULL == height)
    {
      g_warning ("Sdl-rectangle: Bad x and y position\n");
      return;
    }

  red = atoi (enode_attrib_str (node, "red", NULL) );
  blue = atoi (enode_attrib_str (node, "blue", NULL) );
  green = atoi (enode_attrib_str (node, "green", NULL) );

  item = malloc (sizeof (sdl_item));

  /* Setup the item */
  item->type = SDL_RECTANGLE;
  item->name = "hi";
  item->width = atoi (width);
  item->height = atoi (height);
  item->x = atoi (xpos);
  item->y = atoi (ypos);
  item->color = SDL_MapRGB(screen->format, red, green, blue);
  item->needs_refresh = TRUE;
  item->surface = NULL;
 
  /* Push it on the display list */ 
  g_ptr_array_add (item_list, item);

  enode_set_kv(node, "item", item);
}


/**sdl_sprite_destroy
 * Free the sprite.
 */
static void
sdl_rectangle_destroy (ENode * node)
{

}


/**sdl_rectangle_register
 * Register the sdl rectangle attributes. 
 */
void sdl_rectangle_register (void)
{
  Element *element;
  ElementAttr *e_attr;

  /* Register sdl-sprite. */
  element = g_new0 (Element, 1);
  element->render_func = sdl_rectangle_setup; 
  element->destroy_func = sdl_rectangle_destroy;
  element->parent_func = NULL;
  element->tag = "sdl-rectangle";
  element->description = "Create a new SDL window.";
  element_register (element);

  /* X position attribute. */
  e_attr = g_new0 (ElementAttr, 1);
  e_attr->attribute = "xpos";
  e_attr->description = "X position of the sprite";
  e_attr->value_desc = "integer";
  e_attr->possible_values = "-1,*";
  e_attr->set_attr_func = sdl_rectangle_new_pos;
  element_register_attrib (element, e_attr);

  /* Y position attribute. */
  e_attr = g_new0 (ElementAttr, 1);
  e_attr->attribute = "ypos";
  e_attr->description = "y position of the sprite.";
  e_attr->value_desc = "integer";
  e_attr->possible_values = "-1,*";
  e_attr->set_attr_func = sdl_rectangle_new_pos;
  element_register_attrib (element, e_attr);

  /* Width attribute. */
  e_attr = g_new0 (ElementAttr, 1);
  e_attr->attribute = "width";
  e_attr->description = "Width.";
  e_attr->value_desc = "integer";
  e_attr->possible_values = "-1,*";
  e_attr->set_attr_func = sdl_rectangle_new_pos;
  element_register_attrib (element, e_attr);

  /* Height position attribute. */
  e_attr = g_new0 (ElementAttr, 1);
  e_attr->attribute = "height";
  e_attr->description = "Height.";
  e_attr->value_desc = "integer";
  e_attr->possible_values = "-1,*";
  e_attr->set_attr_func = sdl_rectangle_new_pos;
  element_register_attrib (element, e_attr);

  /* Red attribute. */
  e_attr = g_new0 (ElementAttr, 1);
  e_attr->attribute = "red";
  e_attr->description = "Red color value.";
  e_attr->value_desc = "integer";
  e_attr->possible_values = "-1,*";
  e_attr->set_attr_func = sdl_rectangle_set_color;
  element_register_attrib (element, e_attr);

  /* Blue attribute. */
  e_attr = g_new0 (ElementAttr, 1);
  e_attr->attribute = "blue";
  e_attr->description = "Blue color value.";
  e_attr->value_desc = "integer";
  e_attr->possible_values = "-1,*";
  e_attr->set_attr_func = sdl_rectangle_set_color;
  element_register_attrib (element, e_attr);

  /* Green attribute. */
  e_attr = g_new0 (ElementAttr, 1);
  e_attr->attribute = "green";
  e_attr->description = "Green color value.";
  e_attr->value_desc = "integer";
  e_attr->possible_values = "-1,*";
  e_attr->set_attr_func = sdl_rectangle_set_color;
  element_register_attrib (element, e_attr);

  /* force-able refresh attribute. */
  e_attr = g_new0 (ElementAttr, 1);
  e_attr->attribute = "_refresh";
  e_attr->description = "A force refresh";
  e_attr->value_desc = "boolean";
  e_attr->possible_values = "true,false";
  e_attr->set_attr_func = sdl_rectangle_refresh;
  element_register_attrib (element, e_attr);
}
