/* 
 * Java binding for Entity
 *
 * Author: Doug McBride <dougm@liberate.com>
 */

/* 
 * This renderer is free software; please see the LICENSE file for
 * specific information as to licensing conditions.
 */

/* 
 * $Source: /home/cvs/entity/renderers/java/java-embed.c,v $
 * $Id: java-embed.c,v 1.3 2001/05/09 00:20:04 giantpez Exp $
 */

#include <stdlib.h>
#include <jni.h>
#include "entity.h"
#include "java-embed.h"
#include "jni-macros.h"

/* TODO this doesn't work yet
 * #include "java-ENode.h"
 */

/* TODO (add WIN32 support, they use ";") */
#define PATH_SEPARATOR ":"
#define MAX_PATH_LEN 1024

typedef JNIEnv *JNIEnvPtr;

/* use JNI to create a new entity.JavaInterpreter object */
static JNIEnvPtr 
java_create_interp ()
{
    JNIEnv *env;
    JavaVM *jvm;
    JavaVMInitArgs vmArgs;

    jclass classId;
    jmethodID methodId;
    JavaVMOption *options;
    char *cpOption;
    char *libOption;

    char classpath[1024] = ".";
    const char *userClasspath;
    const char *userLibPath;

    /* this will hold our options */
    options = g_malloc0(2 * sizeof(JavaVMOption));

    memset(&vmArgs, 0, sizeof(vmArgs));
    vmArgs.version  = JNI_VERSION_1_2;
    vmArgs.nOptions = 0; /* so far */
    vmArgs.options = options;
    vmArgs.ignoreUnrecognized = JNI_FALSE;

    /* determine classpath */
    userClasspath = getenv("CLASSPATH");
    if (userClasspath)
        strcpy(classpath, userClasspath);

    EDEBUG (("java", "classpath is '%s'", classpath));

    /* XXX 40 is paranoid */
    cpOption = g_malloc0(strlen(classpath) + 40);
    sprintf(cpOption, "-Djava.class.path=%s", classpath);

    options[vmArgs.nOptions].optionString = cpOption;
    options[vmArgs.nOptions].extraInfo = NULL;
    ++vmArgs.nOptions;

    /* determine bin library path for dynamic loading */
    userLibPath = getenv("LD_LIBRARY_PATH");
    if (userLibPath)
    {
		EDEBUG (("java", "libpath is '%s'", userLibPath));

		/* XXX 40 is paranoid */
		libOption = g_malloc0(strlen(userLibPath) + 40);
		sprintf(libOption, "-Djava.library.path=%s", userLibPath);

		options[vmArgs.nOptions].optionString = libOption;
		options[vmArgs.nOptions].extraInfo = NULL;
		++vmArgs.nOptions;
	}

    if (1) /* FIXME there's probably a real debugging flag */
    {
        int i = 0;
        EDEBUG (("java", "JavaVM args:\n    "));
        EDEBUG (("java", "version 0x%08lx, ", vmArgs.version));
        EDEBUG (("java", "ignoreUnrecognized is %s, ",
        vmArgs.ignoreUnrecognized ? "JNI_TRUE" : "JNI_FALSE"));
        EDEBUG (("java", "nOptions is %ld\n", vmArgs.nOptions));
        for (i = 0; i < vmArgs.nOptions; i++)
            EDEBUG (("java", "    option[%2d] = '%s'\n", 
			i, vmArgs.options[i].optionString)); 
    }

    if (JNI_CreateJavaVM(&jvm, (void*) &env, &vmArgs) != JNI_OK)
    {
        g_warning ("java: Error creating JVM [JNI_CreateJavaVM()]");
        return (NULL);
    }

    if (JNI_GetDefaultJavaVMInitArgs((void*) &vmArgs) != JNI_OK)
    {
        g_warning ("java: Error creating JVM [JNI_GetDefaultJavaVMInitArgs()]");
        return (NULL);
    }

    /* TODO don't hard-code strings like this */
    EJNI_NULLBAD(NULL,classId, FindClass, "entity/JavaInterpreter");

    /* we want a pointer to the "create" method. */
    EJNI_NULLBAD(NULL,methodId, GetStaticMethodID, classId, "create", "()V");

    /* now call the method.  it returns a java.lang.String
       which is NULL if it suceeds and contains an error messge
       if it fails. */
    (*env)->CallStaticVoidMethod(env, classId, methodId);

    return env;
}

static ENode *
java_find_containing_object (ENode * node)
{
    if (ebuf_equal_str (node->element, "object"))
        return node;
    else
        return enode_parent (node, "object");
}

static char *
java_evaluate (JNIEnvPtr env, const char *expression)
{
    jclass classId;
    jmethodID methodID;
    jstring evalText;
    jstring jRetString;
    char *retval = NULL;

    EJNI_NULLBAD(NULL,classId, FindClass, "entity/JavaInterpreter");
    EJNI_NULLBAD(NULL,evalText, NewStringUTF, expression);

    /* we want a pointer to the "eval" method. */
    EJNI_NULLBAD(NULL, methodID, GetStaticMethodID, classId, 
	    "eval", "(Ljava/lang/String;)Ljava/lang/String;");
    EJNI(NULL, jRetString, CallStaticObjectMethod, classId, methodID, evalText);
    EDEBUG(("java", "jretstring is %i", (int) jRetString));

    if (jRetString)
    {
	EJNI_NULLBAD(NULL, retval, GetStringUTFChars, jRetString, NULL);
	EDEBUG(("java", "JavaInterpreter.eval(): '%s'", retval));
    }
    else
    {
	EDEBUG(("java", "JavaInterpreter.eval(): null", retval));
    }

    return retval;
}


static void
java_node_render (ENode * node)
{
    ENode *containing_object = java_find_containing_object (node);
    EBufConst *data = enode_get_data (node);

    JNIEnvPtr interp;

    if (containing_object) {
        interp = (JNIEnvPtr) enode_get_kv (containing_object, "java-interp");
    } else {
        g_warning ("<java> tags must go within <object>'s");
        return;
    }

    /* Save reference node on stack.  Any functions or methods called dealing
     * * with enode lookups will use this as the reference. */
    enode_call_reference_push (node);

    EDEBUG (("java", "rendering"));

    /* Create our interpreter if we don't have one already. */
    if (interp == NULL) {
        interp = java_create_interp ();
        if (!interp)
            return;
        enode_set_kv (containing_object, "java-interp", interp);
    }

    if (ebuf_not_empty (data)) {
        /* XXX not handling return for now   
         * int result; 
         * */

        EDEBUG (("java-embed", "executing java"));
        java_evaluate (interp, data->str);
	
	/* XXX not handling a return right now
        if (!result) 
        {
            g_warning
            // TODO add the jvm error to this message 
	     / ("java: error evaluating code in node %s.%s: %s",
             / node->element->str, enode_attrib_str (node, "name", NULL),
             / java_error_message (interp));
	     //
            ("java: error evaluating code in node %s.%s: unknown error",
             node->element->str, enode_attrib_str (node, "name", NULL));
        }
	*/
    }

    enode_call_reference_pop ();
    return;
}

static void
java_node_destroy (ENode * node)
{
    EDEBUG (("java", "destroying"));

    /* Really ought to destroy the interpreters here. */

    /* Actually, what I'd probably recommend here, is attaching * a destroy
     * watcher to the containing object when you set * up the interpreter,
     * and use that callback to destroy it. * That way you don't have to
     * worry about multiple <java> * sections and the semantics
     * involved there. */
    return;
}

static EBuf *
java_execute_function (ENode * node, gchar * function, GSList * args)
{
    GSList *tmp;
    LangArg *arg;
    gint n_args;
    ENode *containing_object = java_find_containing_object (node);
    JNIEnvPtr interp = (JNIEnvPtr) enode_get_kv (containing_object, "java-interp");
    gint i = 0;
    gint j = 0;
    char **stringArgs;
    char functionString[2 * 1024];
    char *currentChar;

    if (interp == NULL) {
        g_warning
            ("java function '%s' asked to be executed, but no interpreter has been created for this object.",
             function);
        return (NULL);
    }

    n_args = g_slist_length (args);
    /* stringArgs is an array of pointers to strings */
    stringArgs = g_malloc0 (n_args * sizeof(char*));

    for (tmp = args; tmp; tmp = tmp->next) {
        arg = (LangArg *) tmp->data;

        if (arg->type == LANG_NODE) {
	    /* TODO add real enode type handling */
	    stringArgs[i] = g_malloc0(8 * sizeof(char));
	    strcpy(stringArgs[i], "\"ENODE\"");
        } else if (arg->type == LANG_STRING) {
            char *str = arg->data;
	    /* FIXME we're just sticking double quotes around it for now */
	    /* 3 == 2 quotes plus the \0 at the end */
            stringArgs[i] = g_malloc0((strlen(str) + 3) * sizeof(char));
            *(stringArgs[i]) = '"';
            strcpy(stringArgs[i] + 1, str);
            strcpy(stringArgs[i] + strlen(str) + 1, "\"");
        } else if (arg->type == LANG_INT) {
            /* stringArgs[i] = g_malloc0((strlen(str) + 3) * sizeof(char)); */
            sprintf(stringArgs[i], "%i", arg->intdata);
        }
	/* TODO support other types
        } else if (arg->type == LANG_BINSTRING) {
            char *str = arg->data;
            int len = arg->size;
            js_vm_make_string (interp->vm, &js_args[i], str, len);
            js_args[i].type = JAVA_STRING;
        } else if (arg->type == LANG_DOUBLE) {
            js_args[i].type = JAVA_FLOAT;
            js_args[i].u.vfloat = arg->doubledata;
        }
        */
        EDEBUG (("java", "  added arg %i: '%s'", i, stringArgs[i]));

        i++;
        enode_call_free_arg (arg);
    }
    EDEBUG (("java", "calling function '%s'", function));


    /*
    if (!interp->vm->consts) {
        g_print ("interp->vm->globals is NULL\n");
    }
    */

    currentChar = functionString;
    /* now we build the string that gets evaluated */
    /* function name */
    currentChar += sprintf(functionString, "%s(", function);
    /* args */
    for (j = 0; j < i; ++j)
    {
        currentChar += sprintf(currentChar, "%s", stringArgs[j]);

	if (j < i - 1)
            *(currentChar++) = ',';
    }
    /* end */
    strcpy(currentChar, ");");

    /* TODO add return value handling / error checking */
    EDEBUG (("java", "  final call: '%s'", functionString));
    java_evaluate(interp, functionString);

    EDEBUG (("java", "call complete", function));

    /* TODO  need to free the memory allocated in this function! */

    return NULL;
}

void
/* TODO support more than one option here
 * #ifdef STATIC_JAVA
 * java_init (RendererFlags flags)
 * #else
 * 
 */
renderer_init (RendererFlags flags)
/* TODO support more than one option here
 * #endif
 *
 */
{
    Element *element;

    if (flags & RENDERER_REGISTER) {
        /* Register java as a tag type */
        element = g_malloc0 (sizeof (Element));
        element->render_func = java_node_render;
        element->destroy_func = java_node_destroy;
        element->description = "Embed Java in your application.";
        element->tag = "java";

        element_register (element);

        /* Register java language type */
        language_register ("java", java_execute_function);
    }
}
