#include <gtk/gtk.h>
#include <stdlib.h>
#include <stdio.h>
#include "entity.h"
#include "gtk-common.h"
#include "gtk-widget-attr.h"

static void
rendgtk_dropdown_selchild_callback (GtkWidget *window, gpointer user_data)
{
    ENode *node = user_data;
    gchar *text;
    gchar *function;
    GSList *children;
    EBuf *value;

    GtkWidget *dropdown;

    dropdown = enode_get_kv (node, "top-widget");
    if (!dropdown)
	return;

    EDEBUG (("dropdown-renderer", "In rendgtk_dropdown_selchild_callback"));
    text = gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (dropdown)->entry));
    enode_attrib_str (node, "text", text);

    value = ebuf_new_with_str (text);
    children = enode_children_attrib (node, "text", value);
    ebuf_free (value);

    if (children) {
        ENode *child = children->data;
        value = enode_attrib (child, "value", NULL);
        if (ebuf_not_empty (value)) {
            EDEBUG (("dropdown-renderer", "Setting value of 'value' of dropdown node to '%s'", value->str));
            enode_attrib (node, "value", ebuf_new_with_ebuf (value));
        }
        g_slist_free (children);
    }

    function = enode_attrib_str (node, "onselect", NULL);

    if (function)
	enode_call_ignore_return (node, function, "s", text);
}

static void
rendgtk_dropdown_onenter_callback (GtkWidget * widget, gpointer user_data)
{
    ENode *node = user_data;
    gchar *function = NULL;
    gchar *text;
    GtkWidget *dropdown;

    EDEBUG (("dropdown-renderer", "in rendgtk_dropdown_onenter_callback\n"));

    dropdown = enode_get_kv (node, "top-widget");
    if (!dropdown)
	return;

    gtk_signal_emit_stop_by_name (GTK_OBJECT
				  (GTK_COMBO (dropdown)->entry), "activate");

    function = enode_attrib_str (node, "onenter", NULL);

    text = gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (dropdown)->entry));
    enode_attrib_quiet (node, "text", ebuf_new_with_str (text));

    enode_call_ignore_return (node, function, "");
}

static void
rendgtk_dropdown_onchange_callback (GtkWidget * widget, gpointer user_data)
{
    ENode *node = user_data;
    gchar *function = NULL;
    gchar *text;
    GtkWidget *dropdown;

    EDEBUG (("dropdown-renderer", "in rendgtk_dropdown_onchange_callback\n"));

    dropdown = enode_get_kv (node, "top-widget");
    if (!dropdown)
	return;

    function = enode_attrib_str (node, "onchange", NULL);

    text = gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (dropdown)->entry));
    enode_attrib_quiet (node, "text", ebuf_new_with_str (text));

    enode_call_ignore_return (node, function, "");
}


static gint
rendgtk_dropdown_text_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *dropdown;

    EDEBUG (("dropdown-renderer", "in rendgtk_dropdown_value_attr_set. %s",
	     value->str));

    dropdown = enode_get_kv (node, "top-widget");
    if (!dropdown)
	return TRUE;

    EDEBUG (("dropdown-renderer", "Setting."));

    /* Has to be a cstring... */
    gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (dropdown)->entry), value->str);
    return TRUE;
}


static gint
rendgtk_dropdown_value_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *dropdown;
    GSList *children;

    EDEBUG (("dropdown-renderer", "in rendgtk_dropdown_value_attr_set. %s",
	     value->str));

    dropdown = enode_get_kv (node, "top-widget");
    if (!dropdown)
	return TRUE;

    children = enode_children_attrib (node, "value", value);
    if (children) {
        ENode *child;
        EBuf *text;

        child = children->data;
        text = enode_attrib (child, "text", NULL);
        if (text) {
            gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (dropdown)->entry), text->str);
            EDEBUG (("dropdown-renderer", "Setting."));
        }
        g_slist_free (children);
    }

    return TRUE;
}

static gint
rendgtk_dropdown_editable_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *dropdown;

    dropdown = enode_get_kv (node, "top-widget");

    if (dropdown)
	gtk_entry_set_editable (GTK_ENTRY (GTK_COMBO (dropdown)->entry),
				erend_value_is_true (value));

    return TRUE;
}


static void
dropdown_renderer_reload (GtkWidget * widget, GdkEventButton * event,
			  gpointer user_data)
{
    ENode *node = user_data;
    GSList *tmp;
    GSList *children;
    GList *items = NULL;
    GtkWidget *dropdown;
    gchar *function;

    if (node == NULL)
	return;

    dropdown = enode_get_kv (node, "top-widget");
    if (!dropdown)
	return;

    /* Notify the user if they want to reload the strings.. */
    function = enode_attrib_str (node, "onopen", NULL);
    if (function)
	enode_call_ignore_return (node, function, "");

    EDEBUG (("dropdown-renderer",
	     "Reloading children to check for strings..\n"));

    children = enode_children (node, NULL);
    tmp = children;
    while (tmp) {
        ENode *tmpnode;

	tmpnode = tmp->data;
	if (ebuf_equal_str (enode_type (tmpnode), "string")) {
	    char *text;
	    text = enode_attrib_str (tmpnode, "text", NULL);
	    if (text)
		items = g_list_append (items, text);

	}
	tmp = tmp->next;
    }

    if (items == NULL) {
        items = g_list_append (items, "");
    }
    gtk_combo_set_popdown_strings (GTK_COMBO (dropdown), items);

    g_slist_free (children);
    g_list_free (items);
}



static void
rendgtk_dropdown_render (ENode * node)
{
    GtkWidget *dropdown;	/* The new dropdown. */

    dropdown = gtk_combo_new ();

    enode_set_kv (node, "top-widget", dropdown);
    enode_set_kv (node, "bottom-widget", dropdown);

    EDEBUG (("dropdown-renderer", "in rendgtk_dropdown_render\n"));

    gtk_signal_connect_after (GTK_OBJECT (GTK_COMBO (dropdown)->entry), "changed",
			      rendgtk_dropdown_onchange_callback, node);

    gtk_signal_connect (GTK_OBJECT (GTK_COMBO (dropdown)->entry),
			"activate", rendgtk_dropdown_onenter_callback, node);

    gtk_signal_connect_after (GTK_OBJECT (GTK_COMBO (dropdown)->popwin),
			      "hide",
			      rendgtk_dropdown_selchild_callback, node);

#if 0
    gtk_signal_connect_after (GTK_OBJECT (GTK_COMBO (dropdown)->list),
			      "select-child",
			      rendgtk_dropdown_selchild_callback, node);
#endif
    
    gtk_signal_connect (GTK_OBJECT (GTK_COMBO (dropdown)->button),
			"button_press_event",
			(GtkSignalFunc) dropdown_renderer_reload, node);

    enode_attribs_sync (node);
    rendgtk_show_cond (node, dropdown);
}

void
dropdown_renderer_register (void)
{
    Element *element;
    ElementAttr *e_attr;

    element = g_new0 (Element, 1);
    element->render_func = rendgtk_dropdown_render;
    element->destroy_func = rendgtk_element_destroy;
    element->tag = "dropdown";
    element->description =
	"Create a dropdown menu.  Use the 'string' element to fill it with items.";
    element_register (element);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "text";
    e_attr->description = "The text currently in the entry widget.";
    e_attr->value_desc = "string";
    e_attr->set_attr_func = rendgtk_dropdown_text_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "value";
    e_attr->description = "The value currently in the entry widget.  This generally comes by setting the value attribute of a child string node, which will be assigned when chosen.";
    e_attr->value_desc = "string";
    e_attr->set_attr_func = rendgtk_dropdown_value_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "onenter";
    e_attr->description = "Specify function called on a onenter event.";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(calling_node)";
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "onchange";
    e_attr->description = "Specify function called when the text in the dropdown is changed.";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(calling_node)";
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "editable";
    e_attr->description = "Specify if user can type in their own values.";
    e_attr->value_desc = "boolean";
    e_attr->possible_values = "(calling_node)";
    e_attr->set_attr_func = rendgtk_dropdown_editable_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "onopen";
    e_attr->description = "Function called when the dropdown is opened.";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(calling_node)";
    e_attr->set_attr_func = NULL;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "onselect";
    e_attr->description = "Function called when the user selects an item";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(calling_node, new_selection)";
    e_attr->set_attr_func = NULL;
    element_register_attrib (element, e_attr);

    /* No GTK_TYPE_COMBO! */
    rendgtk_widget_attr_register (element, GTK_TYPE_WIDGET);

}
