#include <glib.h>
#include <string.h>
#include <stdlib.h>

#include "entity.h"

static GHashTable *languages = NULL;
static gint unique_namespace_id = 0;
static GSList *namespace_node_stack = NULL;

#define LANG_SEPERATOR ":"
#define LANG_DEFAULT   "perl"


/* register a language dispatch function..  language_name is the prefix of
 * the language (eg "perl" for perl:some_func) and then the dispatch or eval
 * function */
void				/* Freed MW */
language_register (gchar * language_name, LangDispatchFunc func)
{
    if (!languages)
	languages = g_hash_table_new (x31_hash, g_str_equal);

    EDEBUG (("enode-call", "registered language '%s'", language_name));

    g_hash_table_insert (languages, language_name, func);
}


GSList *
entity_supported_languages (void)
{
    return (eutils_hash_key_list (languages, NULL));
}


/* Populate both data and intdata so that the languges that need strings
 * don't have to. */
GSList *
enode_call_push_int (GSList * arg_list, gint arg)
{
    gchar *string;
    LangArg *argst;

    argst = g_new0 (LangArg, 1);

    string = g_strdup_printf ("%d", arg);

    argst->type = LANG_INT;
    argst->data = string;
    argst->size = strlen (string);
    argst->intdata = arg;

    arg_list = g_slist_append (arg_list, argst);

    return (arg_list);
}

/* Populate both data and intdata so that the languges that need strings
 * don't have to. */
GSList *
enode_call_push_double (GSList * arg_list, gdouble arg)
{
    gchar *string;
    LangArg *argst;

    argst = g_new0 (LangArg, 1);

    string = g_strdup_printf ("%f", arg);

    argst->type = LANG_DOUBLE;
    argst->data = string;
    argst->size = strlen (string);
    argst->doubledata = arg;

    arg_list = g_slist_append (arg_list, argst);

    return (arg_list);
}

GSList *
enode_call_push_str (GSList * arg_list, const gchar * arg)
{
    gchar *string;
    LangArg *argst;

    argst = g_new0 (LangArg, 1);
    if (!arg)
	arg = "";

    string = g_strdup (arg);

    argst->type = LANG_STRING;
    argst->data = string;
    argst->size = strlen (string);
    /* probably souldn't need this, but it can't hurt. MW */
    argst->intdata = atoi (string);

    arg_list = g_slist_append (arg_list, argst);

    return (arg_list);
}


/* Added binary data to the args list. */
GSList *
enode_call_push_data (GSList * arg_list, gchar * arg, gint size)
{
    gchar *string;
    LangArg *argst;

    ECHECK_RETVAL (arg != NULL, arg_list);
    ECHECK_RETVAL (size >= 0, arg_list);

    argst = g_new0 (LangArg, 1);

    string = g_memdup (arg, size);
    argst->type = LANG_BINSTRING;
    argst->data = string;
    argst->size = size;

    arg_list = g_slist_append (arg_list, argst);

    return (arg_list);
}

/* Add a ENode* to the args list */
GSList *
enode_call_push_node (GSList * arg_list, ENode * node)
{
    LangArg *argst;
    argst = g_new0 (LangArg, 1);

    argst->type = LANG_NODE;
    argst->data = node;

    /* Ref node in case someone decides this is a good one to delete.. */
    enode_ref (node);

    arg_list = g_slist_append (arg_list, argst);

    return (arg_list);
}

void
enode_call_free_arg (LangArg * arg)
{
    if (!arg)
	return;

    if (arg->type != LANG_NODE)
	g_free (arg->data);
    else
	enode_unref (arg->data);

    g_free (arg);
}

static EBufFreeMe *
enode_call_real (ENode * calling_node, gchar * function,
		 gchar * fmt, va_list ap)
{
    /* Current format. */
    gchar *cur;

    /* argument list */
    GSList *arg_list = NULL;

    /* Types of data. */
    /* Binary string uses both string and inter. */
    char *string;
    int inter;
    EBuf *ebuffer;
    ENode *node = NULL;

    /* float flt; */
    double dbl;

    if (!function || strlen (function) == 0)
	return NULL;		/* return peacefully. */

    EDEBUG (("enode-call", "format list is '%s'", fmt));

    for (cur = fmt; *cur != '\0'; cur++) {
	EDEBUG (("enode-call", "*cur = %c", *cur));

	if (*cur == 'n') {	/* ENode * */
	    node = va_arg (ap, ENode *);
	    arg_list = enode_call_push_node (arg_list, node);
	} else if (*cur == 'e') {	/* EBuf* */
	    ebuffer = va_arg (ap, EBuf *);
	    arg_list =
		enode_call_push_data (arg_list, ebuffer->str, ebuffer->len);
	} else if (*cur == 's') {	/* string */
	    string = va_arg (ap, char *);
	    arg_list = enode_call_push_str (arg_list, string);
	} else if (*cur == 'i') {	/* int */
	    inter = va_arg (ap, int);
	    arg_list = enode_call_push_int (arg_list, inter);
	} else if (*cur == 'd') {	/* double */
	    dbl = va_arg (ap, double);
	    arg_list = enode_call_push_double (arg_list, dbl);
	} else if (*cur == 'b') {	/* binary string */
	    string = va_arg (ap, char *);
	    inter = va_arg (ap, int);
	    arg_list = enode_call_push_data (arg_list, string, inter);
	} else {
	    g_warning ("Unknown format character '%c' passed to enode_call ().",
		       *cur);
	    enode_call_free_arg_list_items (arg_list);
	    g_slist_free (arg_list);
	    return (NULL);
	}
	/* ... */
    }

    return enode_call_with_list (calling_node, function, arg_list);
}

void
enode_call_ignore_return (ENode * calling_node, gchar * function,
			  gchar * fmt, ...)
{
    EBuf *retval;
    va_list args;

    va_start (args, fmt);
    retval = enode_call_real (calling_node, function, fmt, args);
    va_end (args);

    if (retval)
	ebuf_free (retval);
}


EBufFreeMe *
enode_call (ENode * calling_node, gchar * function, gchar * fmt, ...)
{
    EBuf *retval;
    va_list args;

    va_start (args, fmt);
    retval = enode_call_real (calling_node, function, fmt, args);
    va_end (args);

    return (retval);
}

/* Free all memory associated with the argument list */
void
enode_call_free_arg_list_items (GSList * arg_list)
{
    GSList *tmp;

    tmp = arg_list;
    while (tmp) {
	LangArg *arg = tmp->data;
	enode_call_free_arg (arg);
	tmp = tmp->next;
    }
}



/* Helper function for enode_call_with_list to prepend the calling_node to
 * the argument list */
static GSList *
enode_call_prepend_node (GSList * arg_list, ENode * node)
{
    LangArg *argst;
    argst = g_new0 (LangArg, 1);

    argst->type = LANG_NODE;
    argst->data = node;
    enode_ref (node);

    arg_list = g_slist_prepend (arg_list, argst);

    return (arg_list);
}


ENode *
userrend_composite_implementation_renderer_node (ENode *ci_node);

/* This now looks up the language needed and dispatches the appropriate
 * function to deal with it. */
EBufConst *			/* Freed MW */
enode_call_with_list (ENode * calling_node,
		      gchar * function_name, GSList * arg_list)
{
    EBuf *retbuf = NULL;	/* Send to the callback from the handler. */
    gchar *language_type;
    gchar *tmp;
    gint index;
    gint lang_is_specified = FALSE;
    LangDispatchFunc dispatch_func;
    ENode *objnode;

    language_type = g_strdup (function_name);	/* Freed MW */

    /* search for a : */
    if ((tmp = strstr (language_type, ":"))) {
	/* make sure it's not a double :: as in some namespace thingies */
	index = tmp - language_type;

	if (language_type[index + 1] != '\0' && language_type[index + 1] != ':') {
	    /* we should now be certain that it's a single : and denotes a
	     * language type */
	    language_type[index] = '\0';
	    lang_is_specified = TRUE;

	    function_name = &language_type[index + 1];
	}
    }

    /* Find out what the object node is */
    objnode = calling_node;

    while (objnode && !ebuf_equal_str (objnode->element, "object")) {
        if (ebuf_equal_str (objnode->element, "composite-implementation")) {
            /* We have to figure out where this is implemented now */
            objnode = userrend_composite_implementation_renderer_node (objnode);
        }
        objnode = objnode->parent;
    }

    if (!objnode) {
        g_warning
            ("ACK! Unable to locate parent <object> node when making call for function '%s'",
             function_name);
        return (NULL);
    }

    if (!lang_is_specified) {
	EBuf *tmp;
        do {
            EBuf *path;

            path = enode_path (objnode);
            EDEBUG (("enode-call", "Found object node of %s\n", path->str));
            ebuf_free (path);
        } while (0);
        
	tmp = enode_attrib (objnode, "default-lang", NULL);

	g_free (language_type);	/* going to be reset. */

	if (ebuf_not_empty (tmp))
	    language_type = g_strdup (tmp->str);
	else
	    language_type = g_strdup (LANG_DEFAULT);
    }

    EDEBUG (("enode-call", "language type '%s', function '%s'\n",
	     language_type, function_name));

    /* Add the calling node as the first argument */
    arg_list = enode_call_prepend_node (arg_list, calling_node);

    /* grab the lang dispatch function */
    dispatch_func = g_hash_table_lookup (languages, language_type);

    if (dispatch_func) {
	/* Remember to wrap calling of the function in the push/pop object
	 * node pair */
	enode_call_reference_push (objnode);
	retbuf = dispatch_func (objnode, function_name, arg_list);
	g_slist_free (arg_list);	/* Free List Here. */
	enode_call_reference_pop ();
    } else {
	g_warning ("Failed to find dispatch function for language '%s'",
		   language_type);

	enode_call_free_arg_list_items (arg_list);
	g_slist_free (arg_list);
    }

    g_free (language_type);

    return (retbuf);
}


void
enode_call_reference_push (ENode * node)
{
    ENode *objnode;

    if (!node)
	return;

    if (ebuf_equal_str (node->element, "object"))
	objnode = node;
    else
	objnode = enode_parent (node, "object");

    /* push it on the stack even if it's NULL */
    namespace_node_stack = g_slist_prepend (namespace_node_stack, objnode);
    if (objnode)
	enode_ref (objnode);
}

ENode *
enode_call_reference_pop (void)
{
    ENode *node;

    node = enode_call_reference ();
    namespace_node_stack = g_slist_remove (namespace_node_stack, node);
    if (node)
	enode_unref (node);

    return (node);
}

ENode *
enode_call_reference (void)
{
    if (namespace_node_stack) {
	ENode *node = namespace_node_stack->data;
	return (node);
    } else
	return (NULL);
}


/* utility function to make it easier to deal with namespaces. */
gchar *				/* Freed MW */
enode_call_get_namespace (gchar * language)
{
    ENode *objnode;
    EBuf *namespace;
    EBuf *namespace_attr;

    namespace_attr = ebuf_new_sized (200);

    ebuf_append_str (namespace_attr, "__");
    ebuf_append_str (namespace_attr, language);
    ebuf_append_str (namespace_attr, "-namespace");


    objnode = enode_call_reference ();
    if (!objnode)
	return (NULL);

    namespace = enode_attrib (objnode, namespace_attr->str, NULL);

    /* if one doesn't already exist, we have to create one */
    if (ebuf_empty (namespace)) {
	namespace = ebuf_new_sized (20);
	ebuf_sprintf (namespace, "namespace%d", unique_namespace_id++);

	enode_attrib_quiet (objnode, namespace_attr->str, namespace);

	EDEBUG (
		("enode-call", "New %s namespace = %s", language,
		 namespace->str));
    }

    ebuf_free (namespace_attr);
    EDEBUG (("enode-call", "returning namespace '%s'", namespace->str));
    return (namespace->str);
}


