/*
 *	Notifies Endeavour about objects being added, modified, or
 *	removed.
 */

#include <limits.h>
#include <unistd.h>
#include <glib.h>
#include <endeavour2.h>


static void print_usage(const gchar *prog_name)
{
	g_print(
"Usage: %s [-a|-m|-n|-r] <object(s)...>\n\
\n\
    Where:\n\
\n\
	-a                      Send an object added event.\n\
	-m                      Send an object modified event.\n\
	-n                      Send an object modified event for renamed\n\
				objects (must have two object paths\n\
				specified, one with the old name and\n\
				one with the new name).\n\
	-r                      Send an object removed event.\n\
\n",
	    prog_name
	);
}

int main(int argc, char *argv[])
{
	gint i, op = 0;
	const gchar *arg;
	gchar *old_path, *new_path, cwd[PATH_MAX];

	edv_context_struct *ctx = EDVContextNew();
	EDVContextLoadConfigurationFile(ctx, NULL);

	if(getcwd(cwd, sizeof(cwd)) != NULL)
	    cwd[sizeof(cwd) - 1] = '\0';
	else
	    *cwd = '\0';

	if(argc <= 1)
	    print_usage(argv[0]);

	/* Check arguments to see if we need to notify about an add,
	 * modify, or removal of an object.
	 */
	for(i = 1; i < argc; i++)
	{
	    arg = argv[i];
	    if(!g_strcasecmp(arg, "-a") ||
	       !g_strcasecmp(arg, "--a")
	    )
	    {
		op = 0;
	    }
	    else if(!g_strcasecmp(arg, "-m") ||
		    !g_strcasecmp(arg, "--m")
	    )
	    {
		op = 1;
	    }
	    else if(!g_strcasecmp(arg, "-n") ||
		    !g_strcasecmp(arg, "--n")
	    )
	    {
		op = 2;
	    }
	    else if(!g_strcasecmp(arg, "-r") ||
		    !g_strcasecmp(arg, "--r")
	    )
	    {
		op = 3;
	    }
	    else if(!g_strcasecmp(arg, "--help") ||
		    !g_strcasecmp(arg, "-help") ||
		    !g_strcasecmp(arg, "--h") ||
		    !g_strcasecmp(arg, "-h")
	    )
	    {
		print_usage(argv[0]);
	    }
	    else if(*arg == '-')
	    {
		print_usage(argv[0]);
	    }
	}

	switch(op)
	{
	  case 0:	/* Send object added event */
	    for(i = 1; i < argc; i++)
	    {
		arg = argv[i];
		if(*arg == '-')
		    continue;

		/* Queue an object added event
		 *
		 * It will be sent when EDVContextSync() is called
		 *
		 * All specified paths must be absolute paths
		 */
		if(*arg == '/')
		    new_path = g_strdup(arg);
		else
		    new_path = g_strdup_printf(
			"%s/%s",
			cwd, arg
		    );
		EDVNotifyQueueObjectAdded(ctx, new_path);
		g_free(new_path);
	    }
	    break;

	  case 1:	/* Send object modified event */
	    for(i = 1; i < argc; i++)
	    {
		arg = argv[i];
		if(*arg == '-')
		    continue;

		/* Queue an object modified event
		 *
		 * It will be sent when EDVContextSync() is called
		 */
                if(*arg == '/')
                    new_path = g_strdup(arg);
                else   
                    new_path = g_strdup_printf(
                        "%s/%s",
                        cwd, arg
                    );
		EDVNotifyQueueObjectModified(ctx, new_path, new_path);
                g_free(new_path);
	    }
	    break;

	  case 2:	/* Send object modified event for renamed object */
	    old_path = NULL;
	    new_path = NULL;
	    for(i = 1; i < argc; i++)
	    {
		arg = argv[i];
		if(*arg == '-')
		    continue;

		if(old_path == NULL)
		{
                    if(*arg == '/')
			old_path = g_strdup(arg);
		    else
			old_path = g_strdup_printf(
                            "%s/%s",
                            cwd, arg
                        );
		}
		else if(new_path == NULL)
		{
                    if(*arg == '/')
                        new_path = g_strdup(arg);
                    else
                        new_path = g_strdup_printf(
                            "%s/%s",
                            cwd, arg
                        );
		}
	    }
	    if((old_path != NULL) && (new_path != NULL))
		EDVNotifyQueueObjectModified(ctx, old_path, new_path);
	    g_free(old_path);
	    g_free(new_path);
	    break;

	  case 3:	/* Send object removed event */
	    for(i = 1; i < argc; i++)
	    {
		arg = argv[i];
		if(*arg == '-')
		    continue;

		/* Queue an object modified event
		 *
		 * It will be sent when EDVContextSync() is called
		 */
                if(*arg == '/')
                    new_path = g_strdup(arg);
                else
                    new_path = g_strdup_printf(
                        "%s/%s",
                        cwd, arg
                    );
		EDVNotifyQueueObjectRemoved(ctx, new_path);
                g_free(new_path);
	    }
	    break;

	}

	EDVContextSync(ctx);
	EDVContextDelete(ctx);

	return(0);
}
