/*  $Header: /cvsroot/dvipdfmx/src/pdfdoc.c,v 1.23 2004/03/24 12:10:52 hirata Exp $
 
    This is dvipdfmx, an eXtended version of dvipdfm by Mark A. Wicks.

    Copyright (C) 2002 by Jin-Hwan Cho and Shunsaku Hirata,
    the dvipdfmx project team <dvipdfmx@project.ktug.or.kr>
    
    Copyright (C) 1998, 1999 by Mark A. Wicks <mwicks@kettering.edu>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
    
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
*/

/*
 * TODO: Many things...
 *  {begin,end}_{bead,article}, box stack, name tree (not limited to dests)...
 */
#if HAVE_CONFIG_H
#include "config.h"
#endif

#include <time.h>

#include "system.h"
#include "mem.h"
#include "error.h"
#include "mfileio.h"

#include "numbers.h"
#include "dvi.h"

#include "pdfobj.h"
#include "pdfparse.h"
#include "pdfencrypt.h"
#include "pdfdev.h"
#include "pdfspecial.h"
#include "pdfximage.h"
#include "pdfdev.h"

/* pdflimits.h is not for limitation of PDF format... */
#include "pdflimits.h"

#include "pdfdoc.h"

static int verbose = 0;

void
pdf_doc_set_verbose (void)
{
  verbose++;
  pdf_ximage_set_verbose(); /* Don't known appropriate place. */
}

pdf_obj *docinfo = NULL;

static struct
{
  pdf_obj *catalog; /* Catalog dicrionary */

  /* Entries in Catalog dictionary */
  pdf_obj *viewerpref;
  pdf_obj *pagelabels;
  pdf_obj *pages;   /* Page tree root node */
  pdf_obj *names;   /* Name dictionary */
  pdf_obj *threads; /* articles_array */
} docroot = {NULL, NULL, NULL, NULL, NULL, NULL};

#define USE_MY_MEDIABOX (1 << 0)
struct page_entry
{
  pdf_obj  *page_obj;
  pdf_obj  *page_ref;

  int       flags;

  double    ref_x, ref_y;
  pdf_rect  cropbox;

  pdf_obj  *resources;
  pdf_obj  *bop, *contents; /* Contents */
  pdf_obj  *annots;
  pdf_obj  *beads;
};

static struct 
{
  pdf_obj *bop, *eop;
  pdf_rect mediabox;

  long num_pages; /* This is not actually total number of pages. */
  long max_pages;
  struct page_entry *pages;
} docpages = {
  NULL, NULL,
  {0.0, 0.0, 0.0, 0.0},

  0, 0,
  NULL
};

/*
 * Pages are starting at 1.
 * The page count does not increase until the page is finished.
 */
#define LASTPAGE()  (&(docpages.pages[docpages.num_pages]))
#define FIRSTPAGE() (&(docpages.pages[0]))
#define PAGECOUNT() (docpages.num_pages)
#define MAXPAGES()  (docpages.max_pages)

static void
resize_pages (long size)
{
  if (size > MAXPAGES()) {
    long i;

    docpages.pages = RENEW(docpages.pages, size, struct page_entry);
    for (i = docpages.max_pages; i < size; i++) {
      docpages.pages[i].page_obj = NULL;
      docpages.pages[i].page_ref = NULL;
      docpages.pages[i].flags     = 0;
      docpages.pages[i].bop       = NULL;
      docpages.pages[i].contents  = NULL;
      docpages.pages[i].resources = NULL;
      docpages.pages[i].annots    = NULL;
      docpages.pages[i].beads     = NULL;
    }
    docpages.max_pages = size;
  }
}

static struct page_entry *
doc_page_entry (unsigned long page_no)
{
  struct page_entry *page;

  if (page_no > 65535ul)
    ERROR("Page number %ul too large!", page_no);
  else if (page_no == 0)
    ERROR("Invalid Page number %ul.", page_no);

  if (page_no > MAXPAGES()) {
    resize_pages(page_no + PAGES_ALLOC_SIZE);
  }

  page = &(docpages.pages[page_no-1]);

  return page;
}


static void doc_close_page_tree  (void);

static void doc_init_names       (void);
static void doc_close_names      (void);
static void doc_init_articles    (void);
static void doc_close_articles   (void);
static void doc_init_bookmarks   (int bookmark_open_depth);
static void doc_close_bookmarks  (void);

static char *pref_filename  = NULL;

void
pdf_doc_set_preference (const char *pref_file)
{
  pref_filename = NEW(strlen(pref_file)+1, char);
  strcpy(pref_filename, pref_file);
}

#define WATERMARK_FLAG_ON_TOP   (1 << 0)
#define WATERMARK_FLAG_FIT_PAGE (1 << 1)

static struct {
  int xobj_id;
  int flags;
  double width, height;
  double rotate;
  double ref_x, ref_y;
} watermarks[16];
static int num_watermark_images = 0;

static int
put_watermark (int is_bop)
{
  transform_info info;
  double ref_x, ref_y;
  int i;

  if (num_watermark_images > 0) {
    for (i = 0; i < num_watermark_images; i++) {
      if (watermarks[i].xobj_id < 0)
	continue;

      if ((is_bop  &&  (watermarks[i].flags & WATERMARK_FLAG_ON_TOP)) ||
	  (!is_bop && !(watermarks[i].flags & WATERMARK_FLAG_ON_TOP)))
	continue;
	  
      transform_info_clear(&info);
      /* dev_page_width() disables papersize special */
      if (watermarks[i].flags & WATERMARK_FLAG_FIT_PAGE) {
	pdf_rect mediabox;

	pdf_doc_get_mediabox(pdf_doc_current_page_no(), &mediabox);
	info.width  = mediabox.urx - mediabox.llx;
	info.height = mediabox.ury - mediabox.lly;
	info.rotate = 0.0;
	ref_x = mediabox.llx;
	ref_y = mediabox.lly;
      } else {
	info.width  = watermarks[i].width;
	info.height = watermarks[i].height;
	info.rotate = watermarks[i].rotate;
	ref_x = watermarks[i].ref_x;
	ref_y = watermarks[i].ref_y;
      }
      pdf_ximage_put_image(watermarks[i].xobj_id, &info, ref_x, ref_y);
    }
  }

  return 0;
}

static void
__setwatermark (pdf_obj *dict)
{
  int i;
  pdf_obj *images, *image, *tmp;

  images = pdf_lookup_dict(dict, "Images");
  num_watermark_images = pdf_array_length(images);
  if (num_watermark_images > 16) {
    num_watermark_images = 16;
  }

  for (i = 0; i < num_watermark_images; i++) {
    image = pdf_get_array(images, i);
    if (!PDF_OBJ_DICTTYPE(image)) {
      WARN("Invalid watermark entry");
      return;
    }

    watermarks[i].ref_x = 0.0;
    watermarks[i].ref_y = 0.0;
      
    tmp = pdf_lookup_dict(image, "Path");
    if (!PDF_OBJ_STRINGTYPE(tmp) ||
	pdf_string_length(tmp) <= 0) {
      WARN("Invalid watermark entry");
      watermarks[i].xobj_id = -1;
      continue;
    }

    watermarks[i].xobj_id = pdf_ximage_findresource(pdf_string_value(tmp));
    if (watermarks[i].xobj_id < 0) {
      WARN("Could not load image \"%s\".", pdf_string_value(tmp));
      continue;
    }

    tmp = pdf_lookup_dict(image, "Width");
    if (PDF_OBJ_NUMBERTYPE(tmp))
      watermarks[i].width = pdf_number_value(tmp);

    tmp = pdf_lookup_dict(image, "Height");
    if (PDF_OBJ_NUMBERTYPE(tmp))
      watermarks[i].height = pdf_number_value(tmp);

    tmp = pdf_lookup_dict(image, "Rotate");
    if (PDF_OBJ_NUMBERTYPE(tmp))
      watermarks[i].rotate = pdf_number_value(tmp) * M_PI / 180;

    tmp = pdf_lookup_dict(image, "Offset");
    if (PDF_OBJ_ARRAYTYPE(tmp) &&
	pdf_array_length(tmp) == 2) {
      watermarks[i].ref_x = pdf_number_value(pdf_get_array(tmp, 0));
      watermarks[i].ref_y = pdf_number_value(pdf_get_array(tmp, 1));
    }

    tmp = pdf_lookup_dict(image, "Flags");
    if (PDF_OBJ_NUMBERTYPE(tmp))
      watermarks[i].flags = pdf_number_value(tmp);
  }
}

static void
__setpagedevice (pdf_obj *dict)
{
  pdf_obj *tmp;
  double width, height, hoffset, voffset;

  width = height = 0;
  hoffset =  72.0;
  voffset = -72.0;
  /* PLRM 3rd. ed., sec. 6.2. */
  tmp = pdf_lookup_dict(dict, "PageSize");
  if (PDF_OBJ_ARRAYTYPE(tmp)) {
    pdf_obj *w, *h;

    w = pdf_get_array(tmp, 0);
    h = pdf_get_array(tmp, 1);
    if (PDF_OBJ_NUMBERTYPE(w) && PDF_OBJ_NUMBERTYPE(h)) {
      width  = pdf_number_value(w);
      height = pdf_number_value(h);
    }
  }

  tmp = pdf_lookup_dict(dict, "PageOffset");
  if (PDF_OBJ_ARRAYTYPE(tmp)) {
    pdf_obj *off_x, *off_y;

    off_x = pdf_get_array(tmp, 0);
    off_y = pdf_get_array(tmp, 1);
    if (PDF_OBJ_NUMBERTYPE(off_x) &&
	PDF_OBJ_NUMBERTYPE(off_y)) {
      hoffset = pdf_number_value(off_x);
      voffset = pdf_number_value(off_y);
    }
  }

#if 0
  dev_set_page(width, height, hoffset, -voffset);
#endif

  tmp = pdf_lookup_dict(dict, "Orientation");
  if (PDF_OBJ_NUMBERTYPE(tmp)) {
    int orient;

    orient = (int) pdf_number_value(tmp);
    switch (orient) {
    case 1:
      pdf_add_dict(docroot.pages, pdf_new_name("Rotate"), pdf_new_number(-90.0));
      break;
    case 2:
      pdf_add_dict(docroot.pages, pdf_new_name("Rotate"), pdf_new_number(180.0));
      break;
    case 3:
      pdf_add_dict(docroot.pages, pdf_new_name("Rotate"), pdf_new_number( 90.0));
      break;
    }
  }

  tmp = pdf_lookup_dict(dict, "MediaColor"); /* wrong extension */
  if (PDF_OBJ_STRINGTYPE(tmp)) {
    if (pdf_string_length(tmp) > 0)
      dev_bg_named_color(pdf_string_value(tmp));
  }
}

static int
import_preference (void)
{
  char    *fullname;
  long     filesize, len;
  pdf_obj *tmp;
  char    *start, *end;
  FILE    *fp;

  if (!pref_filename)
    return -1;

  fullname = kpse_find_file(pref_filename, kpse_program_binary_format, 0);
  if (!fullname) {
    WARN("Could not find file \"%s\".", pref_filename);
    return -1;
  }

  fp = MFOPEN(fullname, FOPEN_R_MODE);
  if (!fp) {
    WARN("Could not open file \"%s\".", fullname);
    return -1;
  }

  filesize = file_size(fp);
  if (filesize > WORK_BUFFER_SIZE) {
    WARN("File \"%s\" too large. (%ld bytes)", fullname, filesize);
  }
  len = fread(work_buffer, sizeof(char), filesize, fp);
  MFCLOSE(fp);

  if (len <= 0)
    return -1;

  start = work_buffer;
  end   = work_buffer + len;
  while (start < end &&
	 (tmp = parse_pdf_dict(&start, end)) != NULL) {
    skip_white(&start, end);
    if (start <= end - strlen("setpagedevice") &&
	!strncmp(start, "setpagedevice", strlen("setpagedevice"))) {
      __setpagedevice(tmp);
      pdf_release_obj(tmp);
      start += strlen("setpagedevice");
    } else if (start <= end - strlen("setwatermarks") &&
	       !strncmp(start, "setwatermarks", strlen("setwatermarks"))) {
      __setwatermark(tmp);
      pdf_release_obj(tmp);
      start += strlen("setwatermarks");
    } else if (start <= end - strlen("setpagelabels") &&
	       !strncmp(start, "setpagelabels", strlen("setpagelabels"))) {
      /* A dictionary containing a number tree. */
      docroot.pagelabels = tmp;
      start += strlen("setpagelabels");
    } else if (start <= end - strlen("setdistillerparams") &&
	       !strncmp(start, "setdistillerparams", strlen("setdistillerparams"))) {
      pdf_release_obj(tmp);
      start += strlen("setdistillerparams");
    } else { /* Everything else are ViewerPreferences */
      if (docroot.viewerpref) {
	pdf_release_obj(tmp);
	break;
      }
      docroot.viewerpref = tmp;
      start += strlen("setviewerpreference");
    }
    skip_white(&start, end);
  }

  return 0;
}

void
pdf_doc_bop (const char *str, unsigned length)
{
  if (docpages.bop) {
    pdf_release_obj(docpages.bop);
  }
  if (length > 0) {
    docpages.bop = pdf_new_stream(STREAM_COMPRESS);
    pdf_add_stream(docpages.bop, str, length);
  } else {
    docpages.bop = NULL;
  }
}

void
pdf_doc_eop (const char *str, unsigned length)
{
  if (docpages.eop) {
    pdf_release_obj(docpages.eop);
  }
  if (length > 0) {
    docpages.eop = pdf_new_stream(STREAM_COMPRESS);
    pdf_add_stream(docpages.eop, str, length);
  } else {
    docpages.eop = NULL;
  }
}

void
pdf_doc_set_current_page_background (const char *str, unsigned length)
{
  struct page_entry *currentpage;

  if (length > 0) {
    currentpage = LASTPAGE();
    if (!currentpage->bop)
      currentpage->bop = pdf_new_stream(STREAM_COMPRESS);
    pdf_add_stream(currentpage->bop, str, length);
  }
}

/*
 * Docinfo
 */
static long
asn_date (char *date_string)
{
  time_t      current_time;
  struct tm  *bd_time;
#ifndef HAVE_TIMEZONE
  #ifdef TM_GM_TOFF
     #define timezone (bdtime->gm_toff)
  #else
     #define timezone 0l
  #endif /* TM_GM_TOFF */
#endif /* HAVE_TIMEZONE */

  time(&current_time);
  bd_time = localtime(&current_time);
  sprintf(date_string, "D:%04d%02d%02d%02d%02d%02d%+03ld'%02ld'",
	  bd_time->tm_year + 1900, bd_time->tm_mon + 1, bd_time->tm_mday,
	  bd_time->tm_hour, bd_time->tm_min, bd_time->tm_sec,
	  -timezone/3600, timezone % 3600);

  return strlen(date_string);
}

#if 0
/*
 * Placing copyright notice of the software in Producer string
 * is inappropriate.
 */
#define BANNER "\
 Copyright \251 2002-2004  by Jin-Hwan Cho and Shunsaku Hirata,\
 Copyright \251 1998, 1999 by Mark A. Wicks.\
"
#endif

static void
doc_close_docinfo (void)
{
  /*
   * Excerpt from PDF Reference 4th ed., sec. 10.2.1.
   *
   * Any entry whose value is not known should be omitted from the dictionary,
   * rather than included with an empty string as its value.
   *
   * ....
   *
   * Note: Although viewer applications can store custom metadata in the document
   * information dictionary, it is inappropriate to store private content or
   * structural information there; such information should be stored in the
   * document catalog instead (see Section 3.6.1,  Document Catalog ).
   */
  const char *keys[] = {
    "Title", "Author", "Subject", "Keywords", "Creator", "Producer",
    "CreationDate", "ModDate", /* Date */
    NULL
  };
  pdf_obj *value;
  char    *banner;
  int      i;

  for (i = 0; keys[i] != NULL; i++) {
    value = pdf_lookup_dict(docinfo, keys[i]);
    if (value) {
      if (!PDF_OBJ_STRINGTYPE(value)) {
	WARN("\"%s\" in DocInfo dictionary not string type.", keys[i]);
	pdf_remove_dict(docinfo, keys[i]);
	WARN("\"%s\" removed from DocInfo.", keys[i]);
      } else if (pdf_string_length(value) == 0) {
	/*
	 * The hyperref package often uses emtpy strings.
	 */
	pdf_remove_dict(docinfo, keys[i]);
      }
    }
  }

#if 0
  /* See above */
  banner = NEW(strlen(BANNER)+strlen(PACKAGE)+strlen(VERSION)+16, char);
  sprintf(banner, "%s (%s), %s", PACKAGE, VERSION, BANNER);
#endif
  banner = NEW(strlen(PACKAGE)+strlen(VERSION)+4, char);
  sprintf(banner, "%s (%s)", PACKAGE, VERSION);
  pdf_add_dict(docinfo,
	       pdf_new_name("Producer"),
	       pdf_new_string(banner, strlen(banner)));
  RELEASE(banner);
  
  if (!pdf_lookup_dict(docinfo, "CreationDate")) {
    char now[32];

    asn_date(now);
    pdf_add_dict(docinfo, 
		 pdf_new_name ("CreationDate"),
		 pdf_new_string(now, strlen(now)));
  }

  pdf_release_obj(docinfo);
}

static pdf_obj *
get_currentpage_resources (const char *category)
{
  pdf_obj *resources;
  struct page_entry *currentpage;

  if (!category) {
    return NULL;
  }
  currentpage = LASTPAGE();
  if (!currentpage->resources) {
    currentpage->resources = pdf_new_dict();
  }
  resources = pdf_lookup_dict(currentpage->resources, category);
  if (!resources) {
    resources = pdf_new_dict();
    pdf_add_dict(currentpage->resources, pdf_new_name(category), resources);
  }

  return resources;
}

void
pdf_doc_add_to_page_resources (const char *category,
			       const char *resource_name, pdf_obj *resource_ref)
{
  pdf_obj *resources;

  resources = get_currentpage_resources(category);
  pdf_add_dict(resources, pdf_new_name(resource_name), resource_ref);
}

static void
doc_flush_page (struct page_entry *page, pdf_obj *parent_ref)
{
  pdf_add_dict(page->page_obj,
	       pdf_new_name("Type"), pdf_new_name("Page"));
  pdf_add_dict(page->page_obj,
	       pdf_new_name("Parent"), parent_ref);

  /*
   * Clipping area specified by CropBox is affected by MediaBox which
   * might be inherit from parent node. If MediaBox of the root node
   * does not have enough size to cover all page's imaging area, using
   * CropBox here gives incorrect result.
   */
  if (page->flags & USE_MY_MEDIABOX) {
    pdf_obj *mediabox;

    mediabox = pdf_new_array();
    pdf_add_array(mediabox,
		  pdf_new_number(ROUND(page->cropbox.llx, 0.01)));
    pdf_add_array(mediabox,
		  pdf_new_number(ROUND(page->cropbox.lly, 0.01)));
    pdf_add_array(mediabox,
		  pdf_new_number(ROUND(page->cropbox.urx, 0.01)));
    pdf_add_array(mediabox,
		  pdf_new_number(ROUND(page->cropbox.ury, 0.01)));
    pdf_add_dict(page->page_obj, pdf_new_name("MediaBox"),  mediabox);
  }
  if (page->annots) {
    pdf_add_dict(page->page_obj,
		 pdf_new_name("Annots"), pdf_ref_obj(page->annots));
    pdf_release_obj(page->annots);
  }
  if (page->beads) {
    pdf_add_dict(page->page_obj,
		 pdf_new_name("B"), pdf_ref_obj(page->beads));
    pdf_release_obj(page->beads);
  }
  pdf_release_obj(page->page_obj);
  pdf_release_obj(page->page_ref);

  page->page_obj = NULL;
  page->page_ref = NULL;
  page->annots   = NULL;
  page->beads    = NULL;
}

/* B-tree? */
#define PAGE_CLUSTER 4
static pdf_obj *
build_page_tree (struct page_entry *firstpage,
		 long num_pages, pdf_obj *parent_ref)
{
  pdf_obj *self, *self_ref, *kids;
  long i;

  self = pdf_new_dict();
  /*
   * This is a slight kludge which allow the subtree dictionary
   * generated by this routine to be merged with the real
   * page_tree dictionary, while keeping the indirect object
   * references right.
   */
  self_ref = parent_ref ? pdf_ref_obj(self) : pdf_ref_obj(docroot.pages);

  pdf_add_dict(self, pdf_new_name("Type"),  pdf_new_name("Pages"));
  pdf_add_dict(self, pdf_new_name("Count"), pdf_new_number((double) num_pages));

  if (parent_ref != NULL)
    pdf_add_dict(self, pdf_new_name("Parent"), parent_ref);

  kids = pdf_new_array();
  if (num_pages > 0 && num_pages <= PAGE_CLUSTER) {
    for (i = 0; i < num_pages; i++) {
      struct page_entry *page;

      page = firstpage + i;
      if (!page->page_ref)
	page->page_ref = pdf_ref_obj(page->page_obj);
      pdf_add_array(kids, pdf_link_obj(page->page_ref));
      doc_flush_page(page, pdf_link_obj(self_ref));
    }
  } else if (num_pages > 0) {
    for (i = 0; i < PAGE_CLUSTER; i++) {
      long start, end;

      start = (i*num_pages)/PAGE_CLUSTER;
      end   = ((i+1)*num_pages)/PAGE_CLUSTER;
      if (end - start > 1) {
	pdf_obj *subtree;

	subtree = build_page_tree(firstpage + start, end - start, pdf_link_obj(self_ref));
	pdf_add_array(kids, pdf_ref_obj(subtree));
	pdf_release_obj(subtree);
      } else {
	struct page_entry *page;

	page = firstpage + start;
	if (!page->page_ref)
	  page->page_ref = pdf_ref_obj(page->page_obj);
	pdf_add_array(kids, pdf_link_obj(page->page_ref));
	doc_flush_page(page, pdf_link_obj(self_ref));
      }
    }
  }
  pdf_add_dict(self, pdf_new_name("Kids"), kids);
  pdf_release_obj(self_ref);

  return self;
}

static void
doc_init_page_tree (double media_width, double media_height)
{
  /*
   * Create empty page tree.
   * The docroot.pages is kept open until the document is closed.
   * This allows the user to write to pages if he so choses.
   */
  docroot.pages = pdf_new_dict();

  docpages.bop = NULL;
  docpages.eop = NULL;

  docpages.num_pages = 0;
  docpages.max_pages = 0;
  docpages.pages     = NULL;

  docpages.mediabox.llx = 0.0;
  docpages.mediabox.lly = 0.0;
  docpages.mediabox.urx = media_width;
  docpages.mediabox.ury = media_height;
}

static void
doc_close_page_tree (void)
{
  pdf_obj *page_tree_root;
  pdf_obj *mediabox;
  long     page_no;

  /*
   * Do consistency check on forward references to pages.
   */
  for (page_no = PAGECOUNT() + 1; page_no <= MAXPAGES(); page_no++) {
    struct page_entry *page;

    page = doc_page_entry(page_no);
    if (page->page_obj) {
      WARN("Nonexistent page #%ld refered.", page_no);
      pdf_release_obj(page->page_ref);
      page->page_ref = NULL;
    }
    if (page->page_obj) {
      WARN("Entry for a nonexistent page #%ld created.", page_no);
      pdf_release_obj(page->page_obj);
      page->page_obj = NULL;
    }
    if (page->annots) {
      WARN("Annotation attached to a nonexistent page #%ld.", page_no);
      pdf_release_obj(page->annots);
      page->annots = NULL;
    }
    if (page->beads) {
      WARN("Article beads attached to a nonexistent page #%ld.", page_no);
      pdf_release_obj(page->beads);
      page->beads = NULL;
    }
    if (page->resources) {
      pdf_release_obj(page->resources);
      page->resources = NULL;
    }
  }

  /*
   * Connect page tree to root node.
   */
  page_tree_root = build_page_tree(FIRSTPAGE(), PAGECOUNT(), NULL);
  pdf_merge_dict(docroot.pages, page_tree_root);
  pdf_release_obj(page_tree_root);

  if (docpages.bop) {
    pdf_add_stream (docpages.bop, "\n", 1);
    pdf_release_obj(docpages.bop);
    docpages.bop = NULL;
  }
  if (docpages.eop) {
    pdf_add_stream (docpages.eop, "\n", 1);
    pdf_release_obj(docpages.eop);
    docpages.eop = NULL;
  }

  /* Create media box at root node and let the other pages inherit it. */
  mediabox = pdf_new_array();
  pdf_add_array(mediabox, pdf_new_number(ROUND(docpages.mediabox.llx, 0.01)));
  pdf_add_array(mediabox, pdf_new_number(ROUND(docpages.mediabox.lly, 0.01)));
  pdf_add_array(mediabox, pdf_new_number(ROUND(docpages.mediabox.urx, 0.01)));
  pdf_add_array(mediabox, pdf_new_number(ROUND(docpages.mediabox.ury, 0.01)));
  pdf_add_dict(docroot.pages, pdf_new_name("MediaBox"), mediabox);

  pdf_add_dict(docroot.catalog,
	       pdf_new_name("Pages"),
	       pdf_ref_obj(docroot.pages));
  pdf_release_obj(docroot.pages);
  docroot.pages  = NULL;

  RELEASE(docpages.pages);
  docpages.pages     = NULL;
  docpages.num_pages = 0;
  docpages.max_pages = 0;
}

/*
 * Outline (Bookmark)
 *
 * This doesn't remember outline hierarcy at all...
 */

#ifndef BOOKMARKS_OPEN_DEFAULT
#define BOOKMARKS_OPEN_DEFAULT 0
#endif
static int outline_depth      = 0;
static int outline_open_depth = 0;

static struct bookmark_item
{
  int      count;
  int      is_open;
  pdf_obj *object;
} outline[MAX_OUTLINE_DEPTH];

static void
doc_init_bookmarks (int bookmark_open_depth)
{
  outline_open_depth = ((bookmark_open_depth >= 0) ?
			bookmark_open_depth : MAX_OUTLINE_DEPTH - bookmark_open_depth);
  outline_depth = 0;
  /* Create empty outline tree */
  outline[outline_depth].object  = pdf_new_dict();
  outline[outline_depth].count   = 0;
  outline[outline_depth].is_open = 0;
}

int
pdf_doc_bookmarks_up (void)
{
  int count;

  if (outline_depth == 0)
    ERROR("Can't go up above the root node!");

  /*
   * Terminate lower level entries.
   */
  pdf_add_dict(outline[outline_depth-1].object,
	       pdf_new_name("Last"), pdf_ref_obj(outline[outline_depth].object));
  pdf_release_obj(outline[outline_depth].object);
  outline[outline_depth].object  = NULL;
  outline[outline_depth].count   = 0;
  outline[outline_depth].is_open = 0;

  /*
   * If the item is open, the value of Count is a positive number which
   * represents the total number of its open descendants at all lower
   * levels of the outline hierarchy. If the item is closed, it is a
   * negative integer whose absolute value specifies how many descendants
   * would appear if the item were reopened.
   */
  if (outline_depth > 1) {
    if (outline[outline_depth-1].is_open) {
      count = outline[outline_depth-1].count;
      /*
       * The count of parent node is a number of it's immediate descendant
       * node at this time. Add all open descendants at all lower levels.
       */
      outline[outline_depth-2].count += count;
    } else {
      count = -outline[outline_depth-1].count;
    }
  } else {
    /* Top level. Can't close it. */
    count = outline[0].count;
  }
  pdf_add_dict(outline[outline_depth-1].object,
	       pdf_new_name("Count"), pdf_new_number(count));

  outline_depth--;

  return outline_depth;
}

int
pdf_doc_bookmarks_down (void)
{
  if (outline_depth >= MAX_OUTLINE_DEPTH - 1)
    ERROR("Outline is too deep.");

  if (!outline[outline_depth].object) {
    pdf_obj *bookmark_item, *text_color, *action;

    WARN("Empty bookmark node!");
    WARN("You have tried to jump more than 1 level.");

#define TITLE_STRING "<No Title>"
    bookmark_item = pdf_new_dict();
    pdf_add_dict(bookmark_item,
		 pdf_new_name("Title"),
		 pdf_new_string(TITLE_STRING, strlen(TITLE_STRING)));

    text_color = pdf_new_array();
    pdf_add_array(text_color, pdf_new_number(1.0));
    pdf_add_array(text_color, pdf_new_number(0.0));
    pdf_add_array(text_color, pdf_new_number(0.0));
    pdf_add_dict(bookmark_item, pdf_new_name("C"), text_color);

    pdf_add_dict(bookmark_item, pdf_new_name("F"), pdf_new_number(1));

#define JS_CODE "app.alert(\"The author of this document made this bookmark item empty!\", 3, 0)"
    action = pdf_new_dict();
    pdf_add_dict(action, pdf_new_name("S"), pdf_new_name("JavaScript"));
    pdf_add_dict(action, 
		 pdf_new_name("JS"), pdf_new_string(JS_CODE, strlen(JS_CODE)));
    pdf_add_dict(bookmark_item, pdf_new_name("A"), action);

    pdf_doc_bookmarks_add(bookmark_item, 0);
  }

  outline_depth++;

  return outline_depth;
}

int
pdf_doc_bookmarks_depth (void)
{
  return outline_depth;
}

void
pdf_doc_bookmarks_add (pdf_obj *dict, int is_open)
{
  pdf_obj *new_item;

  if (outline_depth < 1)
    ERROR("Can't add to outline at depth < 1.");

  if (is_open < 0) {
    if (outline_depth <= outline_open_depth)
      is_open = 1;
    else
      is_open = 0;
  }

  new_item = pdf_new_dict();
  /*
   * Caller doesn't know we don't actually use the dictionary,
   * so he *gave* dict to us.  We have to free it.
   */
  pdf_merge_dict (new_item, dict);
  pdf_release_obj(dict);

  pdf_add_dict(new_item,
	       pdf_new_name("Parent"),
	       pdf_ref_obj(outline[outline_depth-1].object));
  /*
   * We don't care about open items at lower levels at this moment.
   */
  outline[outline_depth-1].count += 1;

  if (outline[outline_depth].object == NULL) {
    pdf_add_dict(outline[outline_depth-1].object,
		 pdf_new_name("First"), pdf_ref_obj(new_item));
  } else {
    pdf_add_dict(new_item,
		 pdf_new_name("Prev"),
		 pdf_ref_obj(outline[outline_depth].object));
    pdf_add_dict(outline[outline_depth].object,
		 pdf_new_name("Next"), pdf_ref_obj(new_item));
    pdf_release_obj(outline[outline_depth].object);
  }
  outline[outline_depth].object  = new_item;
  outline[outline_depth].count   = 0;
  outline[outline_depth].is_open = is_open;
}

static void
doc_close_bookmarks (void)
{
  if (outline[0].object) {
    while (outline_depth > 0) {
      /* pdf_doc_bookmarks_up() decreases outline_depth. */
      pdf_doc_bookmarks_up();
    }
    if (outline[0].count > 0) {
      pdf_add_dict(docroot.catalog,
		   pdf_new_name("Outlines"), pdf_ref_obj(outline[0].object));
    }
    pdf_release_obj(outline[0].object);
    outline[0].object = NULL;
  }
  outline_depth = 0;
}

/*
 * PDF string-keyed object.
 */
struct named_object
{
  pdf_obj *key;   /* PDF string. */
  pdf_obj *value;
};

static int CDECL
cmp_key (const void *d1, const void *d2)
{
  struct named_object *sd1, *sd2;

  sd1 = (struct named_object *) d1;
  sd2 = (struct named_object *) d2;

  return strcmp(pdf_string_value(sd1->key), pdf_string_value(sd2->key));
}

/* Names */

#define NAME_CLUSTER 4
static pdf_obj *
build_name_tree (struct named_object *first, long num_leaves, int is_root)
{
  pdf_obj *result;
  int i;

  result = pdf_new_dict();
  /*
   * According to PDF Refrence, Third Edition (p.101-102), a name tree
   * always has exactly one root node, which contains a SINGLE entry:
   * either Kids or Names but not both. If the root node has a Names
   * entry, it is the only node in the tree. If it has a Kids entry,
   * then each of the remaining nodes is either an intermediate node,
   * containing a Limits entry and a Kids entry, or a leaf node,
   * containing a Limits entry and a Names entry.
   */
  if (!is_root) {
    struct named_object *last;
    pdf_obj *limits;

    limits = pdf_new_array();
    last   = first + num_leaves - 1;
    pdf_add_array(limits, pdf_link_obj(first->key)); /* ... */
    pdf_add_array(limits, pdf_link_obj(last->key )); /* ... */
    pdf_add_dict(result, pdf_new_name("Limits"), limits);
  }

  if (num_leaves > 0 && num_leaves <= 2*NAME_CLUSTER) {
    pdf_obj *names;

    /* Create leaf nodes. */
    names = pdf_new_array();
    for (i = 0; i < num_leaves; i++) {
      struct named_object *cur;

      cur = first + i;
      pdf_add_array(names, cur->key);
      pdf_add_array(names, cur->value); /* Indirect reference for array, dict, stream. */
      cur->key   = NULL; /* To avoid freed multiple times. */
      cur->value = NULL; /* To avoid freed multiple times. */
    }
    pdf_add_dict(result, pdf_new_name("Names"), names);
  } else if (num_leaves > 0) {
    pdf_obj *kids;

    /* Intermediate node */
    kids = pdf_new_array();
    for (i = 0; i < NAME_CLUSTER; i++) {
      pdf_obj *subtree;
      long start, end;

      start = (i*num_leaves) / NAME_CLUSTER;
      end   = ((i+1)*num_leaves) / NAME_CLUSTER;
      subtree = build_name_tree(first + start, end - start, 0);
      pdf_add_array(kids, pdf_ref_obj(subtree));
      pdf_release_obj(subtree);
    }
    pdf_add_dict(result, pdf_new_name("Kids"), kids);
  }

  return result;
}

#define NUM_NAME_CATEGORY 10
#define NAMES_ALLOC_SIZE DESTS_ALLOC_SIZE
static struct
{
  const char *category;
  long num_entries;
  long max_entries;
  struct named_object *data;
} docnames[] = {
  {"Dests", 0, 0, NULL},
  {"AP",    0, 0, NULL},
  {"JavaScript", 0, 0, NULL},
  {"Pages", 0, 0, NULL},
  {"Templates", 0, 0, NULL},
  {"IDS",   0, 0, NULL},
  {"URLS",  0, 0, NULL},
  {"EmbeddedFiles", 0, 0, NULL},
  {"AlternatePresentations", 0, 0, NULL},
  {"Renditions", 0, 0, NULL},
  {NULL, 0, 0, NULL}
};

static void
doc_init_names (void)
{
  int i;

  docroot.names   = NULL;
  for (i = 0; docnames[i].category != NULL; i++) {
    docnames[i].num_entries = 0;
    docnames[i].max_entries = 0;
    docnames[i].data = NULL;
  }
}

int
pdf_doc_add_names (const char *category, pdf_obj *key, pdf_obj *value)
{
  struct named_object *name_entry;
  int i;

  if (!PDF_OBJ_STRINGTYPE(key))
    ERROR("Name dictionary entry key must be a PDF string object.");

  for (i = 0; docnames[i].category != NULL; i++) {
    if (!strcmp(docnames[i].category, category)) {
      break;
    }
  }
  if (docnames[i].category == NULL) {
    WARN("Unknown name dictionary category \"%s\".", category);
    return -1;
  }

  if (docnames[i].num_entries >= docnames[i].max_entries) {
    docnames[i].max_entries += NAMES_ALLOC_SIZE;
    docnames[i].data = RENEW(docnames[i].data, docnames[i].max_entries, struct named_object);
  }
  name_entry = &(docnames[i].data[docnames[i].num_entries]);

  name_entry->key = key;

  switch (PDF_OBJ_TYPEOF(value)) {
  case PDF_ARRAY:
  case PDF_DICT:
  case PDF_STREAM:
    name_entry->value = pdf_ref_obj(value);
    pdf_release_obj(value);
    break;
  default:
    name_entry->value = value;
  }

  docnames[i].num_entries += 1;

  return 0;
}

static void
doc_close_names (void)
{
  pdf_obj *tmp;
  int i;

  for (i = 0; docnames[i].category != NULL; i++) {
    if (docnames[i].num_entries > 0) {
      pdf_obj *name_tree;

      qsort(docnames[i].data,
	    docnames[i].num_entries, sizeof(struct named_object), cmp_key);

      name_tree = build_name_tree(docnames[i].data, docnames[i].num_entries, 1);
      if (!docroot.names) {
	docroot.names = pdf_new_dict();
      }
      pdf_add_dict(docroot.names,
		   pdf_new_name(docnames[i].category), pdf_ref_obj(name_tree));
      pdf_release_obj(name_tree);
    }
    if (docnames[i].data)
      RELEASE(docnames[i].data);
    docnames[i].data = NULL;
    docnames[i].num_entries = 0;
    docnames[i].max_entries = 0;
  }

  if (docroot.names) {
    tmp = pdf_lookup_dict(docroot.catalog, "Names");
    if (!tmp) {
      pdf_add_dict(docroot.catalog,
		   pdf_new_name("Names"), pdf_ref_obj(docroot.names));
    } else if (PDF_OBJ_DICTTYPE(tmp)) {
      pdf_merge_dict(docroot.names, tmp);
      pdf_add_dict(docroot.catalog,
		   pdf_new_name("Names"), pdf_ref_obj(docroot.names));
    } else { /* Maybe reference */
      /* What should I do? */
      WARN("Could not modify Names dictionary.");
    }
    pdf_release_obj(docroot.names);
    docroot.names = NULL;
  }
}


static double annot_grow = 0.0; /* ... */

void
pdf_doc_add_annot (unsigned page_no, const pdf_rect *rect, pdf_obj *annot_dict)
{
  struct page_entry *page;
  pdf_obj  *rect_array;
  pdf_rect  mediabox;

  page = doc_page_entry(page_no);
  if (!page->annots)
    page->annots = pdf_new_array();

  pdf_doc_get_mediabox(pdf_doc_current_page_no(), &mediabox);
  if (rect->llx < mediabox.llx ||
      rect->urx > mediabox.urx ||
      rect->lly < mediabox.lly ||
      rect->ury > mediabox.ury) {
    WARN("Annotation out of page boundary.");
    WARN("Current page's MediaBox: [%g %g %g %g]",
	 mediabox.llx, mediabox.lly, mediabox.urx, mediabox.ury);
    WARN("Annotation: [%g %g %g %g]",
	 rect->llx, rect->lly, rect->urx, rect->ury);
    WARN("Maybe incorrect paper size specified.");
  }
  if (rect->llx > rect->urx || rect->lly > rect->ury) {
    WARN("Rectangle with negative width/height: [%g %g %g %g]",
	 rect->llx, rect->lly, rect->urx, rect->ury);
  }

  rect_array = pdf_new_array();
  pdf_add_array(rect_array, pdf_new_number(ROUND(rect->llx - annot_grow, 0.01)));
  pdf_add_array(rect_array, pdf_new_number(ROUND(rect->lly - annot_grow, 0.01)));
  pdf_add_array(rect_array, pdf_new_number(ROUND(rect->urx + annot_grow, 0.01)));
  pdf_add_array(rect_array, pdf_new_number(ROUND(rect->ury + annot_grow, 0.01)));
  pdf_add_dict (annot_dict, pdf_new_name("Rect"), rect_array);

  pdf_add_array(page->annots, pdf_ref_obj(annot_dict));
}

/*
 * PDF Article Thread
 */
struct article_bead
{
  char    *id;
  long     page_no;
  pdf_rect rect;
};

struct article_thread
{
  char    *id;
  pdf_obj *info;
  long num_beads;
  long max_beads;
  struct article_bead *beads;
};

static struct {
  long num_articles;
  long max_articles;
  struct article_thread *articles;
} docarticles = {0, 0, NULL};

#ifndef PDFDOC_ARTICLE_ALLOC_SIZE
#define PDFDOC_ARTICLE_ALLOC_SIZE 16
#endif
#ifndef PDFDOC_BEAD_ALLOC_SIZE
#define PDFDOC_BEAD_ALLOC_SIZE 16
#endif

static void
doc_init_articles (void)
{
  docroot.threads = NULL;

  docarticles.num_articles = 0;
  docarticles.max_articles = 0;
  docarticles.articles     = NULL;
}

void
pdf_doc_start_article (const char *article_id, pdf_obj *article_info)
{
  struct article_thread *article;

  if (article_id == NULL || strlen(article_id) == 0)
    ERROR("Article thread without internal identifier.");

  if (docarticles.num_articles >= docarticles.max_articles) {
    docarticles.max_articles += PDFDOC_ARTICLE_ALLOC_SIZE;
    docarticles.articles = RENEW(docarticles.articles,
				 docarticles.max_articles, struct article_thread);
  }
  article = &(docarticles.articles[docarticles.num_articles]);

  article->id = NEW(strlen(article_id)+1, char);
  strcpy(article->id, article_id);
  article->info = article_info;
  article->num_beads = 0;
  article->max_beads = 0;
  article->beads     = NULL;

  docarticles.num_articles++;
}

static struct article_bead *
find_bead (struct article_thread *article, const char *bead_id)
{
  struct article_bead *bead;
  long i;

  bead = NULL;
  for (i = 0; i < article->num_beads; i++) {
    if (!strcmp(article->beads[i].id, bead_id)) {
      bead = &(article->beads[i]);
      break;
    }
  }

  return bead;
}

void
pdf_doc_add_bead (const char *article_id,
		  const char *bead_id, long page_no, const pdf_rect *rect)
{
  struct article_thread *article;
  struct article_bead   *bead;
  long i;

  if (!article_id) {
    ERROR("No article identifier specified.");
  }

  article = NULL;
  for (i = 0; i < docarticles.num_articles; i++) {
    if (!strcmp(docarticles.articles[i].id, article_id)) {
      article = &(docarticles.articles[i]);
      break;
    }
  }
  if (!article) {
    ERROR("Specified article thread that doesn't exist.");
    return;
  }

  bead = bead_id ? find_bead(article, bead_id) : NULL;
  if (!bead) {
    if (article->num_beads >= article->max_beads) {
      article->max_beads += PDFDOC_BEAD_ALLOC_SIZE;
      article->beads = RENEW(article->beads,
			     article->max_beads, struct article_bead);
      for (i = article->num_beads; i < article->max_beads; i++) {
	article->beads[i].id = NULL;
	article->beads[i].page_no = -1;
      }
    }
    bead = &(article->beads[article->num_beads]);
    if (bead_id) {
      bead->id = NEW(strlen(bead_id)+1, char);
      strcpy(bead->id, bead_id);
    } else {
      bead->id = NULL;
    }
    article->num_beads++;
  }
  bead->rect.llx = rect->llx;
  bead->rect.lly = rect->lly;
  bead->rect.urx = rect->urx;
  bead->rect.ury = rect->ury;
  bead->page_no  = page_no;
}

static pdf_obj *
doc_make_article (struct article_thread *article,
		  const char **bead_ids, int num_beads,
		  pdf_obj *article_info)
{
  pdf_obj *art_dict;
  pdf_obj *first, *prev, *last;
  long i, n;

  if (!article)
    return NULL;

  art_dict = pdf_new_dict();
  first = prev = last = NULL;
  /*
   * The bead_ids represents logical order of beads in an article thread.
   * If bead_ids is not given, we create an article thread in the order of
   * beads appeared.
   */
  n = bead_ids ? num_beads : article->num_beads;
  for (i = 0; i < n; i++) {
    struct article_bead *bead;

    bead = bead_ids ? find_bead(article, bead_ids[i]) : &(article->beads[i]);
    if (!bead || bead->page_no < 0) {
      continue;
    }
    last = pdf_new_dict();
    if (prev == NULL) {
      first = last;
      pdf_add_dict(first,
		   pdf_new_name("T"), pdf_ref_obj(art_dict));
    } else {
      pdf_add_dict(prev,
		   pdf_new_name("N"), pdf_ref_obj(last));
      pdf_add_dict(last,
		   pdf_new_name("V"), pdf_ref_obj(prev));
      /* We must link first to last. */
      if (prev != first)
	pdf_release_obj(prev);
    }

    /* Realize bead now. */
    {
      struct page_entry *page;
      pdf_obj *rect;

      page = doc_page_entry(bead->page_no);
      if (page->beads == NULL) {
	page->beads = pdf_new_array();
      }
      pdf_add_dict(last, pdf_new_name("P"), pdf_link_obj(page->page_ref));
      rect = pdf_new_array();
      pdf_add_array(rect, pdf_new_number(ROUND(bead->rect.llx, 0.01)));
      pdf_add_array(rect, pdf_new_number(ROUND(bead->rect.lly, 0.01)));
      pdf_add_array(rect, pdf_new_number(ROUND(bead->rect.urx, 0.01)));
      pdf_add_array(rect, pdf_new_number(ROUND(bead->rect.ury, 0.01)));
      pdf_add_dict (last, pdf_new_name("R"), rect);
      pdf_add_array(page->beads, pdf_ref_obj(last));
    }

    prev = last;
  }

  if (first && last) {
    pdf_add_dict(last,
		 pdf_new_name("N"), pdf_ref_obj(first));
    pdf_add_dict(first,
		 pdf_new_name("V"), pdf_ref_obj(last));
    pdf_release_obj(last);
    pdf_add_dict(art_dict,
		 pdf_new_name("F"), pdf_ref_obj(first));
    /* If article_info is supplied, we override article->info. */
    if (article_info) {
      pdf_add_dict(art_dict,
		   pdf_new_name("I"), article_info);
    } else if (article->info) {
      pdf_add_dict(art_dict,
		   pdf_new_name("I"), article->info);
      article->info = NULL; /* We do not write as object reference. */
    }
    pdf_release_obj(first);
  } else {
    pdf_release_obj(art_dict);
    art_dict = NULL;
  }

  return art_dict;
}

#if 0
void
pdf_doc_add_article (const char *article_id,
		     const char **bead_ids, int num_beads,
		     pdf_obj *article_info)
{
  long i;

  if (!article_id)
    return;

  for (i = 0; i < docarticles.num_articles; i++) {
    struct article_thread *article;

    article = &(docarticles.articles[i]);
    if (article->id && !strcmp(article->id, article_id)) {
      pdf_obj *art_dict;

      art_dict = doc_make_article(article, bead_ids, num_beads, article_info);
      if (!docroot.threads) {
	docroot.threads = pdf_new_array();
      }
      pdf_add_array(docroot.threads, pdf_ref_obj(art_dict));
      pdf_release_obj(art_dict);

      break;
    }
  }
}
#endif

static void
doc_clear_article (struct article_thread *article)
{
  if (article) {
    if (article->beads) {
      long i;
      for (i = 0; i < article->num_beads; i++) {
	if (article->beads[i].id)
	  RELEASE(article->beads[i].id);
      }
      RELEASE(article->beads);
      article->beads = NULL;
    }
    if (article->id)
      RELEASE(article->id);
    article->id = NULL;
    article->num_beads = 0;
    article->max_beads = 0;
  }
}

static void
doc_close_articles (void)
{
  int i;

  for (i = 0; i < docarticles.num_articles; i++) {
    struct article_thread *article;

    article = &(docarticles.articles[i]);
    if (article->beads) {
      pdf_obj *art_dict;

      art_dict = doc_make_article(article, NULL, 0, NULL);
      if (!docroot.threads) {
	docroot.threads = pdf_new_array();
      }
      pdf_add_array(docroot.threads, pdf_ref_obj(art_dict));
      pdf_release_obj(art_dict);
    }
    doc_clear_article(article);
  }
  RELEASE(docarticles.articles);
  docarticles.articles = NULL;
  docarticles.num_articles = 0;
  docarticles.max_articles = 0;

  if (docroot.threads) {
    pdf_add_dict(docroot.catalog,
		 pdf_new_name("Threads"), pdf_ref_obj(docroot.threads));
    pdf_release_obj(docroot.threads);
    docroot.threads = NULL;
  }
}

static int xobjects_pending = 0;

/* page_no = 0 for root page tree node. */
void
pdf_doc_set_mediabox (unsigned page_no, const pdf_rect *mediabox)
{
  struct page_entry *page;

  if (page_no == 0) {
    docpages.mediabox.llx = mediabox->llx;
    docpages.mediabox.lly = mediabox->lly;
    docpages.mediabox.urx = mediabox->urx;
    docpages.mediabox.ury = mediabox->ury;
  } else {
    page = doc_page_entry(page_no);
    page->cropbox.llx = mediabox->llx;
    page->cropbox.lly = mediabox->lly;
    page->cropbox.urx = mediabox->urx;
    page->cropbox.ury = mediabox->ury;
    page->flags |= USE_MY_MEDIABOX;
  }
}

void
pdf_doc_get_mediabox (unsigned page_no, pdf_rect *mediabox)
{
  struct page_entry *page;

  if (page_no == 0) {
    mediabox->llx = docpages.mediabox.llx;
    mediabox->lly = docpages.mediabox.lly;
    mediabox->urx = docpages.mediabox.urx;
    mediabox->ury = docpages.mediabox.ury;
  } else {
    page = doc_page_entry(page_no);
    if (page->flags & USE_MY_MEDIABOX) {
      mediabox->llx = page->cropbox.llx;
      mediabox->lly = page->cropbox.lly;
      mediabox->urx = page->cropbox.urx;
      mediabox->ury = page->cropbox.ury;
    } else {
      mediabox->llx = docpages.mediabox.llx;
      mediabox->lly = docpages.mediabox.lly;
      mediabox->urx = docpages.mediabox.urx;
      mediabox->ury = docpages.mediabox.ury;
    }
  }
}

pdf_obj *
pdf_doc_current_page_resources (void)
{
  struct page_entry *currentpage;

  currentpage = LASTPAGE();

  if (!currentpage->resources)
    currentpage->resources = pdf_new_dict();

  return currentpage->resources;
}

long
pdf_doc_current_page_no (void)
{
  return (long) (PAGECOUNT() + 1);
}

pdf_obj *
pdf_doc_ref_page (unsigned long page_no)
{
  struct page_entry *page;

  page = doc_page_entry(page_no);
  if (!page->page_obj) {
    page->page_obj = pdf_new_dict();
    page->page_ref = pdf_ref_obj(page->page_obj);
  }

  return pdf_link_obj(page->page_ref);
}

pdf_obj *
pdf_doc_names (void)
{
  if (!docroot.names)
    docroot.names = pdf_new_dict();

  return docroot.names;
}

pdf_obj *
pdf_doc_page_tree (void)
{
  if (!docroot.pages)
    docroot.pages = pdf_new_dict();

  return docroot.pages;
}

pdf_obj *
pdf_doc_catalog (void)
{
  if (!docroot.catalog)
    docroot.catalog = pdf_new_dict();

  return docroot.catalog;
}

pdf_obj *
pdf_doc_docinfo (void)
{
  if (!docinfo)
    docinfo = pdf_new_dict();

  return docinfo;
}

pdf_obj *
pdf_doc_this_page (void)
{
  struct page_entry *currentpage;

  currentpage = LASTPAGE();

  return currentpage->page_obj;
}

pdf_obj *
pdf_doc_this_page_ref (void)
{
  return pdf_doc_ref_page(PAGECOUNT() + 1);
}

pdf_obj *
pdf_doc_prev_page_ref (void)
{
  if (PAGECOUNT() < 1) {
    ERROR("Reference to previous page, but no pages have been completed yet.");
  }

  return pdf_doc_ref_page((PAGECOUNT() > 0) ? PAGECOUNT() : 1);
}

pdf_obj *
pdf_doc_next_page_ref (void)
{
  return pdf_doc_ref_page(PAGECOUNT() + 2);
}

void
pdf_doc_new_page (void)
{
  struct page_entry *currentpage;

  if (PAGECOUNT() >= MAXPAGES()) {
    resize_pages(MAXPAGES() + PAGES_ALLOC_SIZE);
  }

  /*
   * This is confusing. pdf_doc_finish_page() have increased page count!
   */
  currentpage = LASTPAGE();
  /* Was this page already instantiated by a forward reference to it? */
  if (!currentpage->page_ref) {
    currentpage->page_obj = pdf_new_dict();
    currentpage->page_ref = pdf_ref_obj(currentpage->page_obj);
  }

  currentpage->bop       = NULL;
  currentpage->contents  = pdf_new_stream(STREAM_COMPRESS);
  currentpage->resources = pdf_new_dict();

  currentpage->annots = NULL;
  currentpage->beads  = NULL;

  if (num_watermark_images > 0)
    put_watermark(1);
}

/* This only closes contents and resources. */
void
pdf_doc_finish_page (void)
{
  struct page_entry *currentpage;
  pdf_obj *contents_array;

  if (xobjects_pending)
    ERROR("A pending form XObject at end of page.");

  if (num_watermark_images > 0)
    put_watermark(0);

  currentpage = LASTPAGE();
  if (!currentpage->page_obj)
    currentpage->page_obj = pdf_new_dict();

  /*
   * Make Contents array.
   */
  contents_array = pdf_new_array();
  /*
   * Global BOP content stream.
   * pdf_ref_obj() returns reference itself when the object is
   * indirect reference, not reference to the indirect reference.
   * We keep bop itself but not reference to it since it is
   * expected to be small.
   */
  if (docpages.bop &&
      pdf_stream_length(docpages.bop) > 0) {
    pdf_add_array(contents_array, pdf_ref_obj(docpages.bop));
  }

  /*
   * Current page BOP content stream.
   */
  if (currentpage->bop) {
    if (pdf_stream_length(currentpage->bop) > 0) {
      pdf_add_array  (contents_array, pdf_ref_obj(currentpage->bop));
      pdf_add_stream (currentpage->bop, "\n", 1);
    }
    pdf_release_obj(currentpage->bop);
    currentpage->bop = NULL;
  }

  /* Content body of current page */
  pdf_add_array  (contents_array, pdf_ref_obj(currentpage->contents));
  pdf_add_stream (currentpage->contents, "\n", 1);
  pdf_release_obj(currentpage->contents);
  currentpage->contents = NULL;

  /*
   * Global EOP content stream.
   */
  if (docpages.eop &&
      pdf_stream_length(docpages.eop) > 0) {
    pdf_add_array(contents_array, pdf_ref_obj(docpages.eop));
  }

  pdf_add_dict(currentpage->page_obj,
	       pdf_new_name("Contents"), contents_array);

  /*
   * Page resources.
   */
  if (currentpage->resources) {
    pdf_obj *procset;
    /*
     * ProcSet is obsolete in PDF-1.4 but recommended for compatibility.
     */

    procset = pdf_new_array ();
    pdf_add_array(procset, pdf_new_name("PDF"));
    pdf_add_array(procset, pdf_new_name("Text"));
    pdf_add_array(procset, pdf_new_name("ImageC"));
    pdf_add_dict(currentpage->resources, pdf_new_name("ProcSet"), procset);

    pdf_add_dict(currentpage->page_obj,
		 pdf_new_name("Resources"),
		 pdf_ref_obj(currentpage->resources));
    pdf_release_obj(currentpage->resources);
    currentpage->resources = NULL;
  }

  docpages.num_pages++;
}

void
pdf_doc_add_to_page (const char *buffer, unsigned length)
{
  struct page_entry *currentpage;

  currentpage = LASTPAGE();
  if (!currentpage || !currentpage->contents)
    ERROR("No space allocated for current page's content.");

  pdf_add_stream(currentpage->contents, buffer, length);
}

static char *doccreator = NULL; /* Ugh */

void
pdf_doc_init (const char *filename,
	      double media_width, double media_height,
	      double annot_grow_amount, int bookmark_open_depth)
{
  pdf_out_init(filename);

  docroot.catalog = pdf_new_dict();
  pdf_set_root(docroot.catalog);

  annot_grow = annot_grow_amount;

  docinfo = pdf_new_dict();
  if (doccreator) {
    pdf_add_dict(docinfo,
		 pdf_new_name("Creator"),
		 pdf_new_string(doccreator, strlen(doccreator)));
    RELEASE(doccreator);
    doccreator = NULL;
  }
  pdf_set_info(docinfo);

  doc_init_bookmarks(bookmark_open_depth);
  doc_init_articles();
  doc_init_names();
  doc_init_page_tree(media_width, media_height);

  create_encrypt();

  if (pref_filename) {
    import_preference();
    RELEASE(pref_filename);
    pref_filename = NULL;
  }
}

void
pdf_doc_set_creator (const char *creator)
{
  doccreator = NEW(strlen(creator)+1, char);
  strcpy(doccreator, creator); /* Ugh */
}

static void
doc_close_catalog (void)
{
  pdf_obj *tmp;

  if (docroot.viewerpref) {
    tmp = pdf_lookup_dict(docroot.catalog, "ViewerPreferences");
    if (!tmp) {
      pdf_add_dict(docroot.catalog,
		   pdf_new_name("ViewerPreferences"), pdf_ref_obj(docroot.viewerpref));
    } else if (PDF_OBJ_DICTTYPE(tmp)) {
      pdf_merge_dict(docroot.viewerpref, tmp);
      pdf_add_dict(docroot.catalog,
		   pdf_new_name("ViewerPreferences"), pdf_ref_obj(docroot.viewerpref));
    } else { /* Maybe reference */
      /* What should I do? */
      WARN("Could not modify ViewerPreferences.");
    }
    pdf_release_obj(docroot.viewerpref);
    docroot.viewerpref = NULL;
  }

  if (docroot.pagelabels) {
    tmp = pdf_lookup_dict(docroot.catalog, "PageLabels");
    if (!tmp) {
      pdf_add_dict(docroot.catalog,
		   pdf_new_name("PageLabels"), pdf_ref_obj(docroot.pagelabels));
    } else if (PDF_OBJ_DICTTYPE(tmp)) {
      pdf_merge_dict(docroot.pagelabels, tmp);
      pdf_add_dict(docroot.catalog,
		   pdf_new_name("PageLabels"), pdf_ref_obj(docroot.pagelabels));
    } else { /* Maybe reference */
      /* What should I do? */
      WARN("Could not modify PageLabels.");
    }
    pdf_release_obj(docroot.pagelabels);
    docroot.pagelabels = NULL;
  }

  pdf_add_dict(docroot.catalog,
	       pdf_new_name("Type"), pdf_new_name("Catalog"));
  pdf_release_obj(docroot.catalog);
  docroot.catalog = NULL;
}

void
pdf_doc_close (void)
{
  /*
   * Following things were kept around so user can add dictionary items.
   */
  doc_close_articles();
  doc_close_names();
  doc_close_bookmarks();
  doc_close_page_tree();
  doc_close_docinfo();

  doc_close_catalog();

  pdf_ximage_close(); /* Don't known appropriate place. */
  pdf_out_flush();
}

/*
 * All this routine does is give the form a name and add a unity scaling matrix.
 * It fills in required fields.  The caller must initialize the stream.
 */
void
pdf_doc_make_xform (pdf_obj     *xform,
		    pdf_rect    *bbox,
		    pdf_tmatrix *matrix,
		    pdf_obj     *resources)
{
  pdf_obj *xform_dict;
  pdf_obj *tmp;

  xform_dict = pdf_stream_dict(xform);
  pdf_add_dict(xform_dict,
	       pdf_new_name("Type"),     pdf_new_name("XObject"));
  pdf_add_dict(xform_dict,
	       pdf_new_name("Subtype"),  pdf_new_name("Form"));
  pdf_add_dict(xform_dict,
	       pdf_new_name("FormType"), pdf_new_number(1.0));

  if (!bbox)
    ERROR("No BoundingBox supplied.");

  tmp = pdf_new_array();
  pdf_add_array(tmp, pdf_new_number(ROUND(bbox->llx, .001)));
  pdf_add_array(tmp, pdf_new_number(ROUND(bbox->lly, .001)));
  pdf_add_array(tmp, pdf_new_number(ROUND(bbox->urx, .001)));
  pdf_add_array(tmp, pdf_new_number(ROUND(bbox->ury, .001)));
  pdf_add_dict(xform_dict, pdf_new_name("BBox"), tmp);

  if (matrix) {
    tmp = pdf_new_array();
    pdf_add_array(tmp, pdf_new_number(matrix->a));
    pdf_add_array(tmp, pdf_new_number(matrix->b));
    pdf_add_array(tmp, pdf_new_number(matrix->c));
    pdf_add_array(tmp, pdf_new_number(matrix->d));
    pdf_add_array(tmp, pdf_new_number(ROUND(matrix->e, .001)));
    pdf_add_array(tmp, pdf_new_number(ROUND(matrix->f, .001)));
    pdf_add_dict(xform_dict, pdf_new_name("Matrix"), tmp);
  }

  pdf_add_dict(xform_dict, pdf_new_name("Resources"), resources);
}

#define PAGE_GRABBING_NEST_MAX 4
struct resource_stack 
{
  char    *ident;
  int      xform_depth;

  double   ref_x;
  double   ref_y;
  pdf_rect cropbox;
  pdf_obj *contents;
  pdf_obj *resources;
} res_stack[PAGE_GRABBING_NEST_MAX];

/*
 * begin_form_xobj creates an xobject with its "origin" at
 * xpos and ypos that is clipped to the specified bbox. Note
 * that the origin is not the lower left corner of the bbox.
 */

static char *current_form_ident = NULL; /* Ugh */

extern void
pdf_doc_current_form_info (char **ident, pdf_rect *cropbox)
{
  struct page_entry *currentpage;

  if (xobjects_pending < 1)
    ERROR("No pending Form XObject there: %d", xobjects_pending);

  currentpage = LASTPAGE();

  if (ident) {
    *ident = current_form_ident; /* Ugh */
  }
  if (cropbox) {
    cropbox->llx = currentpage->cropbox.llx;
    cropbox->lly = currentpage->cropbox.lly;
    cropbox->urx = currentpage->cropbox.urx;
    cropbox->ury = currentpage->cropbox.ury;
  }
}


void
pdf_doc_start_grabbing (char *ident,
			double ref_x, double ref_y, pdf_rect *cropbox)
{
  struct page_entry *currentpage;

  if (xobjects_pending >= PAGE_GRABBING_NEST_MAX) {
    WARN("Form XObjects nested too deeply. (limit: %d)", PAGE_GRABBING_NEST_MAX);
    return;
  }
  /*
   * This is a real hack.  We basically treat each xobj as a separate mini
   * page unto itself.  Save all the page structures and reinitialize
   * them when we finish this xobject.
   */
  currentpage = LASTPAGE();
  res_stack[xobjects_pending].xform_depth = dev_xform_depth();
  res_stack[xobjects_pending].ref_x = currentpage->ref_x;
  res_stack[xobjects_pending].ref_y = currentpage->ref_y;
  res_stack[xobjects_pending].cropbox.llx = currentpage->cropbox.llx;
  res_stack[xobjects_pending].cropbox.lly = currentpage->cropbox.lly;
  res_stack[xobjects_pending].cropbox.urx = currentpage->cropbox.urx;
  res_stack[xobjects_pending].cropbox.ury = currentpage->cropbox.ury;
  res_stack[xobjects_pending].contents  = currentpage->contents;
  res_stack[xobjects_pending].resources = currentpage->resources;

  res_stack[xobjects_pending].ident = current_form_ident; /* Ugh */

  currentpage->ref_x = ref_x;
  currentpage->ref_y = ref_y;
  currentpage->cropbox.llx = cropbox->llx;
  currentpage->cropbox.lly = cropbox->lly;
  currentpage->cropbox.urx = cropbox->urx;
  currentpage->cropbox.ury = cropbox->ury;

  currentpage->contents  = pdf_new_stream(STREAM_COMPRESS);
  currentpage->resources = pdf_new_dict();

  current_form_ident = ident; /* Ugh */

  /*
   * Make sure the object is self-contained by adding the
   * current font to the object stream
   */
  dev_reselect_font();
  dev_do_color();

  xobjects_pending++;
}

pdf_obj *
pdf_doc_end_grabbing (void)
{
  pdf_obj *xform;
  pdf_obj *tmp;
  pdf_tmatrix matrix;
  struct page_entry *currentpage;

  if (xobjects_pending <= 0) {
    WARN("[doc] end_form_xobj: Tried to close a nonexistent xobject.");
    return NULL;
  }

  currentpage = LASTPAGE();

  xform = currentpage->contents;
  /*
   * ProcSet is obsolete in PDF-1.4 but recommended for compatibility.
   */
  tmp = pdf_new_array();
  pdf_add_array(tmp, pdf_new_name("PDF"));
  pdf_add_array(tmp, pdf_new_name("Text"));
  pdf_add_array(tmp, pdf_new_name("ImageC"));
  pdf_add_dict(currentpage->resources, pdf_new_name("ProcSet"), tmp);

  /*
   * The reference point of an Xobject is at the lower left corner
   * of the bounding box.  Since we would like to have an arbitrary
   * reference point, we use a transformation matrix, translating
   * the reference point to (0,0).
   */

  matrix.a = 1.0; matrix.b = 0.0;
  matrix.c = 0.0; matrix.d = 1.0;
  matrix.e = -(currentpage->ref_x);
  matrix.f = -(currentpage->ref_y);

  pdf_doc_make_xform(xform,
		     &(currentpage->cropbox), &matrix,
		     pdf_ref_obj(currentpage->resources));
  pdf_release_obj(currentpage->resources);

  currentpage = LASTPAGE();

  xobjects_pending--;

  dev_close_all_xforms(res_stack[xobjects_pending].xform_depth);

  currentpage->ref_x = res_stack[xobjects_pending].ref_x;
  currentpage->ref_y = res_stack[xobjects_pending].ref_y;
  currentpage->cropbox.llx = res_stack[xobjects_pending].cropbox.llx;
  currentpage->cropbox.lly = res_stack[xobjects_pending].cropbox.lly;
  currentpage->cropbox.urx = res_stack[xobjects_pending].cropbox.urx;
  currentpage->cropbox.ury = res_stack[xobjects_pending].cropbox.ury;
  currentpage->contents  = res_stack[xobjects_pending].contents;
  currentpage->resources = res_stack[xobjects_pending].resources;

  current_form_ident     = res_stack[xobjects_pending].ident; /* Ugh */

  dev_reselect_font();
  dev_do_color();

  return xform;
}

static struct
{
  int      dirty;
  pdf_obj *annot_dict;
  pdf_rect rect;
} breaking_state = {0, NULL, {0.0, 0.0, 0.0, 0.0}};

void
pdf_doc_set_box (void)
{
  breaking_state.rect.llx = 10000.0; /* large value */
  breaking_state.rect.lly = 10000.0; /* large value */
  breaking_state.rect.urx = 0.0;     /* small value */
  breaking_state.rect.ury = 0.0;     /* small value */
  breaking_state.dirty    = 0;
}

void
pdf_doc_begin_annot (pdf_obj *dict)
{
  breaking_state.annot_dict = dict;
  pdf_doc_set_box();
  dev_tag_depth();
}

void
pdf_doc_end_annot (void)
{
  pdf_doc_flush_annot();
  breaking_state.annot_dict = NULL;
  dev_untag_depth ();
}

void
pdf_doc_flush_annot (void)
{
  if (breaking_state.dirty) {
    pdf_obj  *annot_dict;

    /* Copy dict */
    annot_dict = pdf_new_dict();
    pdf_merge_dict(annot_dict, breaking_state.annot_dict);
    pdf_doc_add_annot(pdf_doc_current_page_no(), &(breaking_state.rect), annot_dict);
    pdf_release_obj(annot_dict);
  }
  pdf_doc_set_box();
}

void
pdf_doc_expand_box (double llx, double lly, double urx, double ury)
{
  breaking_state.rect.llx = MIN(breaking_state.rect.llx, llx);
  breaking_state.rect.lly = MIN(breaking_state.rect.lly, lly);
  breaking_state.rect.urx = MAX(breaking_state.rect.urx, urx);
  breaking_state.rect.ury = MAX(breaking_state.rect.ury, ury);
  breaking_state.dirty    = 1;
}

