/* Copyright (C) 2003 Timo Sirainen / Joshua Goodall */

#include "lib.h"
#include "base64.h"
#include "hex-binary.h"
#include "base64.h"
#include "buffer.h"
#include "md5.h"
#include "md5crypt.h"
#include "mycrypt.h"
#include "randgen.h"
#include "str.h"
#include "password-scheme.h"
#include "safe-memset.h"

#ifdef HAVE_OPENSSL_SHA1
#  include <openssl/sha.h>
#endif

static const char *salt_chars =
	"./0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";

static const char *password_generate_cram_md5(const char *plaintext)
{
	unsigned char digest[16], ipad[64], opad[64], context_digest[32], *cdp;
	struct md5_context ctxo, ctxi;
	size_t len;
	int i;

	memset(ipad, 0, sizeof(ipad));
	memset(opad, 0, sizeof(opad));

	/* Hash excessively long passwords */
	len = strlen(plaintext);
	if (len > 64) {
		md5_get_digest(plaintext, len, digest);
		memcpy(ipad, digest, 16);
		memcpy(opad, digest, 16);
	} else {
		memcpy(ipad, plaintext, len);
		memcpy(opad, plaintext, len);
	}

	/* ipad/opad operation */
	for (i = 0; i < 64; i++) {
		ipad[i] ^= 0x36;
		opad[i] ^= 0x5c;
	}
	md5_init(&ctxi);
	md5_init(&ctxo);
	md5_update(&ctxi, ipad, 64);
	md5_update(&ctxo, opad, 64);

	/* Make HMAC-MD5 hex digest */
#define CDPUT(p,c) STMT_START {	\
	*p++ = (c) & 0xff;	\
	*p++ = (c)>>8 & 0xff;	\
	*p++ = (c)>>16 & 0xff;	\
	*p++ = (c)>>24 & 0xff;	\
} STMT_END
	cdp = context_digest;
	CDPUT(cdp, ctxo.a);
	CDPUT(cdp, ctxo.b);
	CDPUT(cdp, ctxo.c);
	CDPUT(cdp, ctxo.d);
	CDPUT(cdp, ctxi.a);
	CDPUT(cdp, ctxi.b);
	CDPUT(cdp, ctxi.c);
	CDPUT(cdp, ctxi.d);

	return binary_to_hex(context_digest, sizeof(context_digest));
}

int password_verify(const char *plaintext, const char *password,
		    const char *scheme, const char *user)
{
	unsigned char md5_digest[16];
	const char *realm, *str;
	buffer_t *buf;
	size_t size;
	const char *data;
	struct md5_context ctx;

	if (password == NULL)
		return 0;

	if (strcasecmp(scheme, "CRYPT") == 0)
		return strcmp(mycrypt(plaintext, password), password) == 0;

	if (strcasecmp(scheme, "MD5") == 0)
		return strcmp(md5_crypt(plaintext, password), password) == 0;

	if (strcasecmp(scheme, "PLAIN") == 0)
		return strcmp(password, plaintext) == 0;

	if (strcasecmp(scheme, "HMAC-MD5") == 0) {
		str = password_generate_cram_md5(plaintext);
		return strcmp(str, password) == 0;
	}

	if (strcasecmp(scheme, "DIGEST-MD5") == 0) {
		/* user:realm:passwd */
		realm = strchr(user, '@');
		if (realm != NULL) realm++; else realm = "";

		str = t_strconcat(t_strcut(user, '@'), ":", realm,  ":",
				  plaintext, NULL);
		md5_get_digest(str, strlen(str), md5_digest);
		str = binary_to_hex(md5_digest, sizeof(md5_digest));

		return strcasecmp(str, password) == 0;
	}
#ifdef HAVE_OPENSSL_SHA1
	if (strcasecmp(scheme, "SHA1") == 0) {
		unsigned char sha1_digest[SHA_DIGEST_LENGTH];
		string_t *str;

		SHA1(plaintext, strlen(plaintext), sha1_digest);

		str = t_str_new(64);
		base64_encode(sha1_digest, sizeof(sha1_digest), str);
		return strcasecmp(str_c(str), password) == 0;
	}
#endif

	if (strcasecmp(scheme, "PLAIN-MD5") == 0) {
		md5_get_digest(plaintext, strlen(plaintext), md5_digest);
		str = binary_to_hex(md5_digest, sizeof(md5_digest));
		return strcasecmp(str, password) == 0;
	}

	if (strcasecmp(scheme, "LDAP-MD5") == 0) {
		string_t *str;

		md5_get_digest(plaintext, strlen(plaintext), md5_digest);
		str = t_str_new(64);
		base64_encode(md5_digest, sizeof(md5_digest), str);
		return strcasecmp(str_c(str), password) == 0;
	}

	/* format: base64-encoded MD5 hash and salt */
	if (strcasecmp(scheme, "SMD5") == 0) {
		buf = buffer_create_static(data_stack_pool,
				MAX_BASE64_DECODED_SIZE(strlen(password)+1));

		if (base64_decode(password, strlen(password), NULL, buf) == 1) {
			data = buffer_get_data(buf, &size);
			if (size <= 16) {
				i_error("password-verify(%s): invalid SMD5", user);
				return -1;
			}

			md5_init(&ctx);
			md5_update(&ctx, plaintext, strlen(plaintext));
			md5_update(&ctx, &data[16], size-16);
			md5_final(&ctx, md5_digest);
			return memcmp(md5_digest, data, 16) == 0;
                } else {
			i_error("password-verify(%s): couldn't decode SMD5", user);
			return -1;
		}
	}

	return -1;
}

const char *password_get_scheme(const char **password)
{
	const char *p, *scheme;

	if (*password == NULL)
		return NULL;

	if (strncmp(*password, "$1$", 3) == 0) {
		/* skip the salt */
		p = strchr(*password + 3, '$');
		if (p != NULL) {
			/* stop at next '$' */
			p = strchr(p+1, '$');
			if (p != NULL)
				*password = t_strdup_until(*password, p);
			return "MD5";
		}
	}

	if (**password != '{')
		return NULL;

	p = strchr(*password, '}');
	if (p == NULL)
		return NULL;

	scheme = t_strdup_until(*password + 1, p);
	*password = p + 1;

	/* LDAP's RFC2307 specifies the MD5 scheme for what we call LDAP-MD5.
	 * Implementations use base64 encoding for the hash, so we can
	 * detect the genuine {MD5}'s MCF format - base64 doesn't use '$'. */
	if (strncasecmp(scheme, "MD5", 3) == 0 &&
	    strncmp(*password, "$1$", 3) != 0)
		scheme = "LDAP-MD5";
	return scheme;
}

const char *password_generate(const char *plaintext, const char *user,
			      const char *scheme)
{
	const char *realm, *str;
	unsigned char digest[16];
	char salt[9];
	int i;

	if (strcasecmp(scheme, "CRYPT") == 0) {
		random_fill(salt, 2);
		salt[0] = salt_chars[salt[0] % (sizeof(salt_chars)-1)];
		salt[1] = salt_chars[salt[1] % (sizeof(salt_chars)-1)];
		salt[2] = '\0';
		return t_strdup(mycrypt(plaintext, salt));
	}

	if (strcasecmp(scheme, "MD5") == 0) {
		random_fill(salt, 8);
		for (i = 0; i < 8; i++)
			salt[i] = salt_chars[salt[i] % (sizeof(salt_chars)-1)];
		salt[8] = '\0';
		return t_strdup(md5_crypt(plaintext, salt));
	}

	if (strcasecmp(scheme, "PLAIN") == 0)
		return plaintext;

	if (strcasecmp(scheme, "HMAC-MD5") == 0)
		return password_generate_cram_md5(plaintext);

	if (strcasecmp(scheme, "DIGEST-MD5") == 0) {
		/* user:realm:passwd */
		realm = strchr(user, '@');
		if (realm != NULL) realm++; else realm = "";

		str = t_strconcat(t_strcut(user, '@'), ":", realm,  ":",
				  plaintext, NULL);
		md5_get_digest(str, strlen(str), digest);
		return binary_to_hex(digest, sizeof(digest));
	}

	if (strcasecmp(scheme, "PLAIN-MD5") == 0) {
		md5_get_digest(plaintext, strlen(plaintext), digest);
		return binary_to_hex(digest, sizeof(digest));
	}

	return NULL;
}
