/**
 * Discrete Finite Automaton Simulator.
 * @author Shaun Jackman <sdj@sfu.ca>
 * @copyright Copyright 2004 Shaun Jackman
 */


#include <stdio.h>
#include <string.h>
#include "dfa.h"
#include "set.h"
#include "util.h"


/** Clears this DFA. */
void
clear_dfa( DFA* dfa)
{
	memset( dfa->transition, NO_TRANSITION, sizeof dfa->transition);
	clear_set( &dfa->final);
	memset( dfa->tokens, 0, sizeof dfa->tokens);
}


/** Prints this DFA. */
void
print_dfa( const DFA* dfa)
{
	int i;
	char c;
	for( i = 0; i < STATES; i++)
		for( c = '\t'; c < CHAR_MAX; c++)
			if( dfa->transition[i][(int)c] != NO_TRANSITION) {
				printf( "%d %d '", i, dfa->transition[i][(int)c]);
				print_char( c);
				puts( "'");
			}
	for( i = 0; i < STATES; i++)
		if( contains_element( &dfa->final, i))
			printf( "%d %d\n", i, dfa->tokens[i]);
}


/** Reads a DFA from the specified file. */
void
read_dfa( DFA* dfa, FILE* file)
{
	int from;
	clear_dfa( dfa);
	while( fscanf( file, "%d", &from) == 1) {
		assumex( within( from, 0, STATES-1), "Invalid state");
		if( getc( file) != '\n') {
			int to;
			char c;
			assumex( fscanf( file, "%d %c", &to, &c) == 2,
					"Badly formatted line");
			assumex( within( to, 0, STATES-1), "Invalid state");
			if( c == '\'') {
				c = read_char( file);
				assume( getc( file) == '\'', "Unmatched single-quotes");
			}
			assumex( within( c, '\t', CHAR_MAX), "Non-ASCII character");
			dfa->transition[from][(int)c] = to;
		} else
			add_element( &dfa->final, from);
	}
}
