/*
 * PT
 * Copyright (C) 2002, 2003 The Printing Tool Developers
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * interface.c
 */

#include <stdlib.h>
#include <gdk/gdkkeysyms.h>

#include "preferences.h"
#include "support.h"
#include "interface.h"


/*
 * static variables
 */

static preferences_t p;
static gboolean update = FALSE;
static gboolean start = TRUE;
static gint *selected_row = NULL;
static gint r = 0;
static gchar *password = NULL;
static GtkWidget *transient;

/*
 * p_create_program_window ()
 */

GtkWidget* p_create_program_window (void)
{
	GtkWidget *program_window;
	GtkWidget *tmp_icon;
	GdkPixmap *pixmap;
	GdkBitmap *bitmap;

	GtkWidget *vbox;

	GtkWidget *popup_menu;
	GtkWidget *select_popup_item;
	GtkWidget *line_1;
	GtkWidget *delete_popup_item;
	GtkWidget *up_popup_item;
	GtkWidget *down_popup_item;
	GtkWidget *hold_popup_item;
	GtkWidget *release_popup_item;
	GtkWidget *move_popup_item;
	GtkWidget *line_2;
	GtkWidget *exit_popup_item;
	GtkWidget *line_3;
	GtkWidget *preferences_popup_item;
	GtkWidget *line_4;
	GtkWidget *info_popup_item;

	GtkWidget *scrolledwindow;
	GtkWidget *clist;
	gint i;
	GtkWidget *rank_label;
	GtkWidget *id_label;
	GtkWidget *dest_label;
	GtkWidget *title_label;
	GtkWidget *user_label;
	GtkWidget *format_label;
	GtkWidget *state_label;
	GtkWidget *size_label;
	GtkWidget *copies_label;
	GtkWidget *priority_label;
	GtkWidget *creation_time_label;
	GtkWidget *processing_time_label;

	GtkWidget *toolbar;
	GtkWidget *select_button;
	GtkWidget *delete_button;
	GtkWidget *up_button;
	GtkWidget *down_button;
	GtkWidget *hold_button;
	GtkWidget *release_button;
	GtkWidget *move_button;
	GtkWidget *exit_button;

	GtkAccelGroup *accel_group;
	
	accel_group = gtk_accel_group_new ();

	/*
	 * get preferences
	 */

	p_read_preferences (&p);

	/*
	 * set password callback function
	 */

	cupsSetPasswordCB (p_password_callback);

	/*
	 * program_window
	 */

	program_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	transient = program_window;
	gtk_widget_set_events (program_window, GDK_BUTTON_PRESS_MASK |
	GDK_POINTER_MOTION_HINT_MASK);
	gtk_object_set_data (GTK_OBJECT (program_window), "tmp",
		GINT_TO_POINTER (p.refresh_interval));
	tmp_icon = create_pixmap (program_window, "pt.xpm");
	gtk_pixmap_get (GTK_PIXMAP (tmp_icon), &pixmap, &bitmap);
	gtk_object_set_data (GTK_OBJECT (program_window), "pixmap", pixmap);
	gtk_object_set_data (GTK_OBJECT (program_window), "bitmap", bitmap);

	/*
	 * vbox
	 */

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_widget_ref (vbox);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "vbox",
		vbox, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (vbox);
	gtk_container_add (GTK_CONTAINER (program_window), vbox);

	/*
	 * toolbar
	 */

	toolbar = gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_ICONS);
	gtk_widget_ref (toolbar);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "toolbar",
		toolbar, (GtkDestroyNotify) gtk_widget_unref);
	gtk_toolbar_set_button_relief (GTK_TOOLBAR (toolbar), GTK_RELIEF_NONE);
	gtk_toolbar_set_tooltips (GTK_TOOLBAR (toolbar), 1);
	gtk_box_pack_start (GTK_BOX (vbox), toolbar, FALSE, FALSE, 0);
	gtk_widget_show (toolbar);

	tmp_icon = create_pixmap (program_window, "select.xpm");
	select_button = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
		GTK_TOOLBAR_CHILD_BUTTON, NULL, NULL,
		gettext("Select All"), NULL, tmp_icon, NULL, NULL);
	gtk_widget_ref (select_button);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "select_button",
		select_button, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (select_button);

	tmp_icon = create_pixmap (program_window, "delete.xpm");
	delete_button = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
		GTK_TOOLBAR_CHILD_BUTTON, NULL, NULL,
		gettext("Delete"), NULL, tmp_icon, NULL, NULL);
	gtk_widget_ref (delete_button);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "delete_button",
		delete_button, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (delete_button);

	tmp_icon = create_pixmap (program_window, "up.xpm");
	up_button = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
		GTK_TOOLBAR_CHILD_BUTTON, NULL, NULL,
		gettext("Increase Priority"), NULL, tmp_icon, NULL, NULL);
	gtk_widget_ref (up_button);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "up_button",
		up_button, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (up_button);

	tmp_icon = create_pixmap (program_window, "down.xpm");
	down_button = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
		GTK_TOOLBAR_CHILD_BUTTON, NULL, NULL,
		gettext("Decrease Priority"), NULL, tmp_icon, NULL, NULL);
	gtk_widget_ref (down_button);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "down_button",
		down_button, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (down_button);
	
	tmp_icon = create_pixmap (program_window, "hold.xpm");
	hold_button = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
		GTK_TOOLBAR_CHILD_BUTTON, NULL, NULL,
		gettext("Hold"), NULL, tmp_icon, NULL, NULL);
	gtk_widget_ref (hold_button);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "hold_button",
		hold_button, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (hold_button);

	tmp_icon = create_pixmap (program_window, "release.xpm");
	release_button = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
		GTK_TOOLBAR_CHILD_BUTTON, NULL, NULL,
		gettext("Release"), NULL, tmp_icon, NULL, NULL);
	gtk_widget_ref (release_button);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "release_button",
		release_button, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (release_button);

	tmp_icon = create_pixmap (program_window, "move.xpm");
	move_button = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
		GTK_TOOLBAR_CHILD_BUTTON, NULL, NULL,
		gettext("Move"), NULL, tmp_icon, NULL, NULL);
	gtk_widget_ref (move_button);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "move_button",
		move_button, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (move_button);

	tmp_icon = create_pixmap (program_window, "exit.xpm");
	exit_button = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
		GTK_TOOLBAR_CHILD_BUTTON, NULL, NULL,
		gettext("Exit"), NULL, tmp_icon, NULL, NULL);
	gtk_widget_ref (exit_button);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "exit_button",
		exit_button, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (exit_button);

	/*
	 * popup menu
	 */

	popup_menu = gtk_menu_new ();
	gtk_widget_ref (popup_menu);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "popup_menu",
		popup_menu, (GtkDestroyNotify) gtk_widget_unref);

	select_popup_item = gtk_menu_item_new_with_label (gettext("Select All"));
	gtk_widget_ref (select_popup_item);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "select_popup_item",
		select_popup_item, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (select_popup_item);
	gtk_container_add (GTK_CONTAINER (popup_menu), select_popup_item);
	gtk_widget_add_accelerator (select_popup_item, "activate", accel_group,
		GDK_F2, 0, GTK_ACCEL_VISIBLE);
	gtk_widget_lock_accelerators (select_popup_item);

	line_1 = gtk_menu_item_new ();
	gtk_widget_ref (line_1);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "line_1",
		line_1, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (line_1);
	gtk_container_add (GTK_CONTAINER (popup_menu), line_1);
	gtk_widget_set_sensitive (line_1, FALSE);

	delete_popup_item = gtk_menu_item_new_with_label (gettext("Delete"));
	gtk_widget_ref (delete_popup_item);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "delete_popup_item",
		delete_popup_item, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (delete_popup_item);
	gtk_container_add (GTK_CONTAINER (popup_menu), delete_popup_item);
	gtk_widget_add_accelerator (delete_popup_item, "activate", accel_group,
		GDK_F8, 0, GTK_ACCEL_VISIBLE);
	gtk_widget_lock_accelerators (delete_popup_item);

	up_popup_item = gtk_menu_item_new_with_label (gettext("Increase Priority"));
	gtk_widget_ref (up_popup_item);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "up_popup_item",
		up_popup_item, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (up_popup_item);
	gtk_container_add (GTK_CONTAINER (popup_menu), up_popup_item);
	gtk_widget_add_accelerator (up_popup_item, "activate", accel_group,
		GDK_plus, 0, GTK_ACCEL_VISIBLE);
	gtk_widget_lock_accelerators (up_popup_item);

	down_popup_item = gtk_menu_item_new_with_label (gettext("Decrease Priority"));
	gtk_widget_ref (down_popup_item);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "down_popup_item",
		down_popup_item, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (down_popup_item);
	gtk_container_add (GTK_CONTAINER (popup_menu), down_popup_item);
	gtk_widget_add_accelerator (down_popup_item, "activate", accel_group,
		GDK_minus, 0, GTK_ACCEL_VISIBLE);
	gtk_widget_lock_accelerators (down_popup_item);

	hold_popup_item = gtk_menu_item_new_with_label (gettext("Hold"));
	gtk_widget_ref (hold_popup_item);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "hold_popup_item",
		hold_popup_item, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (hold_popup_item);
	gtk_container_add (GTK_CONTAINER (popup_menu), hold_popup_item);
	gtk_widget_add_accelerator (hold_popup_item, "activate", accel_group,
		GDK_F3, 0, GTK_ACCEL_VISIBLE);
	gtk_widget_lock_accelerators (hold_popup_item);
	
	release_popup_item = gtk_menu_item_new_with_label (gettext("Release"));
	gtk_widget_ref (release_popup_item);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "release_popup_item",
		release_popup_item, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (release_popup_item);
	gtk_container_add (GTK_CONTAINER (popup_menu), release_popup_item);
	gtk_widget_add_accelerator (release_popup_item, "activate", accel_group,
		GDK_F4, 0, GTK_ACCEL_VISIBLE);
	gtk_widget_lock_accelerators (release_popup_item);

	move_popup_item = gtk_menu_item_new_with_label (gettext("Move..."));
	gtk_widget_ref (move_popup_item);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "move_popup_item",
		move_popup_item, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (move_popup_item);
	gtk_container_add (GTK_CONTAINER (popup_menu), move_popup_item);
	gtk_widget_add_accelerator (move_popup_item, "activate", accel_group,
		GDK_F5, 0, GTK_ACCEL_VISIBLE);
	gtk_widget_lock_accelerators (move_popup_item);
	
	line_2 = gtk_menu_item_new ();
	gtk_widget_ref (line_2);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "line_2",
		line_2, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (line_2);
	gtk_container_add (GTK_CONTAINER (popup_menu), line_2);
	gtk_widget_set_sensitive (line_2, FALSE);

	exit_popup_item = gtk_menu_item_new_with_label (gettext("Exit"));
	gtk_widget_ref (exit_popup_item);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "exit_popup_item",
		exit_popup_item, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (exit_popup_item);
	gtk_container_add (GTK_CONTAINER (popup_menu), exit_popup_item);
	gtk_widget_add_accelerator (exit_popup_item, "activate", accel_group,
		GDK_F10, 0, GTK_ACCEL_VISIBLE);
	gtk_widget_lock_accelerators (exit_popup_item);

	line_3 = gtk_menu_item_new ();
	gtk_widget_ref (line_3);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "line_3",
		line_3, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (line_3);
	gtk_container_add (GTK_CONTAINER (popup_menu), line_3);
	gtk_widget_set_sensitive (line_3, FALSE);

	preferences_popup_item = gtk_menu_item_new_with_label (gettext("Preferences..."));
	gtk_widget_ref (preferences_popup_item);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "preferences_popup_item",
		preferences_popup_item, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (preferences_popup_item);
	gtk_container_add (GTK_CONTAINER (popup_menu), preferences_popup_item);
	gtk_widget_add_accelerator (preferences_popup_item, "activate", accel_group,
		GDK_F11, 0, GTK_ACCEL_VISIBLE);
	gtk_widget_lock_accelerators (preferences_popup_item);

	line_4 = gtk_menu_item_new ();
	gtk_widget_ref (line_4);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "line_4",
		line_4, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (line_4);
	gtk_container_add (GTK_CONTAINER (popup_menu), line_4);
	gtk_widget_set_sensitive (line_4, FALSE);

	info_popup_item = gtk_menu_item_new_with_label (gettext("Info..."));
	gtk_widget_ref (info_popup_item);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "info_popup_item",
		info_popup_item, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (info_popup_item);
	gtk_container_add (GTK_CONTAINER (popup_menu), info_popup_item);
	gtk_widget_add_accelerator (info_popup_item, "activate", accel_group,
		GDK_F12, 0, GTK_ACCEL_VISIBLE);
	gtk_widget_lock_accelerators (info_popup_item);

	/*
	 * scrolledwindow
	 */

	scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
	gtk_widget_ref (scrolledwindow);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow),
		GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "scrolledwindow",
		scrolledwindow, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (scrolledwindow);
	gtk_box_pack_start (GTK_BOX (vbox), scrolledwindow, TRUE, TRUE, 0);

	/*
	 * clist
	 */

	clist = gtk_clist_new (12);
	gtk_widget_ref (clist);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "clist",
		clist, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (clist);
	gtk_clist_set_selection_mode (GTK_CLIST (clist), GTK_SELECTION_EXTENDED);
	gtk_clist_set_auto_sort (GTK_CLIST (clist), FALSE);
	gtk_container_add (GTK_CONTAINER (scrolledwindow), clist);
	gtk_object_set_data (GTK_OBJECT (clist), "program_window", program_window);
	gtk_object_set_data (GTK_OBJECT (clist), "j_allocation", GINT_TO_POINTER (0));
	gtk_object_set_data (GTK_OBJECT (clist), "dest_allocation", GINT_TO_POINTER (0));
	gtk_object_set_data (GTK_OBJECT (scrolledwindow), "clist", clist);

	for (i = 0; i < 12; i++)
		gtk_clist_set_column_justification (GTK_CLIST (clist), i, GTK_JUSTIFY_LEFT);

	gtk_clist_column_titles_active (GTK_CLIST (clist));
	gtk_clist_column_titles_show (GTK_CLIST (clist));

	rank_label = gtk_label_new (gettext("Rank"));
	gtk_widget_ref (rank_label);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "rank_label",
		rank_label, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (rank_label);
	gtk_clist_set_column_widget (GTK_CLIST (clist), 0, rank_label);

	priority_label = gtk_label_new (gettext("Priority"));
	gtk_widget_ref (priority_label);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "priority_label",
		priority_label, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (priority_label);
	gtk_clist_set_column_widget (GTK_CLIST (clist), 1, priority_label);

	id_label = gtk_label_new (gettext("ID"));
	gtk_widget_ref (id_label);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "id_label",
		id_label, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (id_label);
	gtk_clist_set_column_widget (GTK_CLIST (clist), 2, id_label);

	user_label = gtk_label_new (gettext("User"));
	gtk_widget_ref (user_label);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "user_label",
		user_label, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (user_label);
	gtk_clist_set_column_widget (GTK_CLIST (clist), 3, user_label);

	title_label = gtk_label_new (gettext("Title"));
	gtk_widget_ref (title_label);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "title_label",
		title_label, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (title_label);
	gtk_clist_set_column_widget (GTK_CLIST (clist), 4, title_label);

	format_label = gtk_label_new (gettext("Format"));
	gtk_widget_ref (format_label);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "format_label",
		format_label, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (format_label);
	gtk_clist_set_column_widget (GTK_CLIST (clist), 5, format_label);

	copies_label = gtk_label_new (gettext("Copies"));
	gtk_widget_ref (copies_label);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "copies_label",
		copies_label, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (copies_label);
	gtk_clist_set_column_widget (GTK_CLIST (clist), 6, copies_label);

	dest_label = gtk_label_new (gettext("Printer"));
	gtk_widget_ref (dest_label);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "dest_label",
		dest_label, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (dest_label);
	gtk_clist_set_column_widget (GTK_CLIST (clist), 7, dest_label);

	state_label = gtk_label_new (gettext("State"));
	gtk_widget_ref (state_label);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "state_label",
		state_label, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (state_label);
	gtk_clist_set_column_widget (GTK_CLIST (clist), 8, state_label);

	size_label = gtk_label_new (gettext("Size (kB)"));
	gtk_widget_ref (size_label);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "size_label",
		size_label, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (size_label);
	gtk_clist_set_column_widget (GTK_CLIST (clist), 9, size_label);

	creation_time_label = gtk_label_new (gettext("Creation Time"));
	gtk_widget_ref (creation_time_label);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "creation_time_label",
		creation_time_label, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (creation_time_label);
	gtk_clist_set_column_widget (GTK_CLIST (clist), 10, creation_time_label);

	processing_time_label = gtk_label_new (gettext("Processing Time"));
	gtk_widget_ref (processing_time_label);
	gtk_object_set_data_full (GTK_OBJECT (program_window), "processing_time_label",
		processing_time_label, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (processing_time_label);
	gtk_clist_set_column_widget (GTK_CLIST (clist), 11, processing_time_label);

	/*
	 * apply preferences
	 */

	p_apply_preferences (program_window);
	gtk_clist_set_sort_type (GTK_CLIST (clist), p.sort_toggle);
	gtk_clist_set_sort_column (GTK_CLIST (clist), p.sort_item);
	if ((p.sort_item == 0) || (p.sort_item == 1) || (p.sort_item == 2) ||
		(p.sort_item == 6) || (p.sort_item == 9))
			gtk_clist_set_compare_func (GTK_CLIST (clist), p_gtk_compare);
	else
		gtk_clist_set_compare_func (GTK_CLIST (clist), NULL);

	/*
	 * connect signals
	 */

	/* select_popup_item - "activate" */
	
	gtk_signal_connect_object (GTK_OBJECT (select_popup_item), "activate",
		GTK_SIGNAL_FUNC (gtk_clist_select_all), GTK_OBJECT (clist));

	/* delete_popup_item - "activate" */

	gtk_signal_connect_object (GTK_OBJECT (delete_popup_item), "activate",
		GTK_SIGNAL_FUNC (p_delete), GTK_OBJECT (clist));

	/* up_popup_item - "activate" */

	gtk_signal_connect_object (GTK_OBJECT (up_popup_item), "activate",
		GTK_SIGNAL_FUNC (p_up), GTK_OBJECT (clist));

	/* down_popup_item - "activate" */

	gtk_signal_connect_object (GTK_OBJECT (down_popup_item), "activate",
		GTK_SIGNAL_FUNC (p_down), GTK_OBJECT (clist));

	/* hold_popup_item - "activate" */

	gtk_signal_connect_object (GTK_OBJECT (hold_popup_item), "activate",
		GTK_SIGNAL_FUNC (p_hold), GTK_OBJECT (clist));

	/* release_popup_item - "activate" */

	gtk_signal_connect_object (GTK_OBJECT (release_popup_item), "activate",
		GTK_SIGNAL_FUNC (p_release), GTK_OBJECT (clist));

	/* move_popup_item - "activate" */

	gtk_signal_connect_object (GTK_OBJECT (move_popup_item), "activate",
		GTK_SIGNAL_FUNC (p_show_move_dialog), GTK_OBJECT (clist));

	/* exit_popup_item - "activate" */

	gtk_signal_connect_object (GTK_OBJECT (exit_popup_item), "activate",
		GTK_SIGNAL_FUNC (p_exit), GTK_OBJECT (program_window));

	/* preferences_popup_item - "activate" */

	gtk_signal_connect_object (GTK_OBJECT (preferences_popup_item), "activate",
		GTK_SIGNAL_FUNC (p_show_preferences_dialog), GTK_OBJECT (program_window));

	/* info_popup_item - "activate" */

	gtk_signal_connect_object (GTK_OBJECT (info_popup_item), "activate",
		GTK_SIGNAL_FUNC (p_show_info_dialog), GTK_OBJECT (program_window));

	/* select_button - "clicked" */

	gtk_signal_connect_object (GTK_OBJECT (select_button), "clicked",
		GTK_SIGNAL_FUNC (gtk_clist_select_all), GTK_OBJECT (clist));

	/* delete_button - "clicked" */

	gtk_signal_connect_object (GTK_OBJECT (delete_button), "clicked",
		GTK_SIGNAL_FUNC (p_delete), GTK_OBJECT (clist));

	/* up_button - "clicked" */

	gtk_signal_connect_object (GTK_OBJECT (up_button), "clicked",
		GTK_SIGNAL_FUNC (p_up), GTK_OBJECT (clist));

	/* down_button - "clicked" */

	gtk_signal_connect_object (GTK_OBJECT (down_button), "clicked",
		GTK_SIGNAL_FUNC (p_down), GTK_OBJECT (clist));

	/* hold_button - "clicked" */

	gtk_signal_connect_object (GTK_OBJECT (hold_button), "clicked",
		GTK_SIGNAL_FUNC (p_hold), GTK_OBJECT (clist));

	/* release_button - "clicked" */

	gtk_signal_connect_object (GTK_OBJECT (release_button), "clicked",
		GTK_SIGNAL_FUNC (p_release), GTK_OBJECT (clist));

	/* move_button - "clicked" */

	gtk_signal_connect_object (GTK_OBJECT (move_button), "clicked",
		GTK_SIGNAL_FUNC (p_show_move_dialog), GTK_OBJECT (clist));

	/* exit_button - "clicked" */

	gtk_signal_connect_object (GTK_OBJECT (exit_button), "clicked",
		GTK_SIGNAL_FUNC (p_exit), GTK_OBJECT (program_window));

	/* clist - "click-column" */

	gtk_signal_connect (GTK_OBJECT (clist), "click-column",
		GTK_SIGNAL_FUNC (p_sort), NULL);

	/* clist - "select-row" */

	gtk_signal_connect (GTK_OBJECT (clist), "select-row",
		GTK_SIGNAL_FUNC (p_select), NULL);

	/* clist - "unselect-row" */

	gtk_signal_connect (GTK_OBJECT (clist), "unselect-row",
		GTK_SIGNAL_FUNC (p_unselect), NULL);

	/* scrolledwindow - "button-press-event" */
	
	gtk_signal_connect (GTK_OBJECT (scrolledwindow), "button-press-event",
		GTK_SIGNAL_FUNC (p_left_button_pressed), NULL);

	/* scrolledwindow - "button-release-event" */
	
	gtk_signal_connect (GTK_OBJECT (scrolledwindow), "button-release-event",
		GTK_SIGNAL_FUNC (p_left_button_released), NULL);

	/* clist - "button-press-event" */
	
	gtk_signal_connect_object (GTK_OBJECT (clist), "button-press-event",
		GTK_SIGNAL_FUNC (p_right_button_pressed), GTK_OBJECT (popup_menu));

	/* program_window - "delete_event" */

	gtk_signal_connect (GTK_OBJECT (program_window), "delete_event",
		GTK_SIGNAL_FUNC (p_exit), NULL);

	gtk_window_add_accel_group (GTK_WINDOW (program_window), accel_group);

	return program_window;
}

/*
 * p_apply_preferences ()
 */

void p_apply_preferences (GtkWidget *widget)
{
	GtkCList *clist;
	gchar *tmp;

	if (strcmp (p.filter_dest, NO_ARG) == 0)
		tmp = g_strdup_printf ("%s <%s:%d>", gettext("Print Jobs"),
		p.host, p.port);
	else
		tmp = g_strdup_printf ("%s <%s@%s:%d>", gettext("Print Jobs"),
		p.filter_dest, p.host, p.port);

	gtk_window_set_title (GTK_WINDOW (widget), tmp);
	g_free (tmp);

	clist = gtk_object_get_data (GTK_OBJECT (widget), "clist");

	gtk_clist_set_column_visibility (clist, 0, p.show_rank);
	gtk_clist_set_column_visibility (clist, 1, p.show_priority);
	gtk_clist_set_column_visibility (clist, 2, p.show_id);
	gtk_clist_set_column_visibility (clist, 3, p.show_user);
	gtk_clist_set_column_visibility (clist, 4, p.show_title);
	gtk_clist_set_column_visibility (clist, 5, p.show_format);
	gtk_clist_set_column_visibility (clist, 6, p.show_copies);
	gtk_clist_set_column_visibility (clist, 7, p.show_dest);
	gtk_clist_set_column_visibility (clist, 8, p.show_state);
	gtk_clist_set_column_visibility (clist, 9, p.show_size);
	gtk_clist_set_column_visibility (clist, 10, p.show_creation_time);
	gtk_clist_set_column_visibility (clist, 11, p.show_processing_time);

	if (start)
	{
		gtk_window_set_default_size (GTK_WINDOW (widget), p.width, p.height);
		start = FALSE;
		return;
	}

	p_stop_polling (clist);
	p_rewrite_clist (clist);
	p_start_polling (clist);
}

/*
 * p_rewrite_clist ()
 */

void p_rewrite_clist (GtkCList *clist)
{
	gint i, n, v;
	gboolean found;
	print_job_t *j;
	GtkAdjustment *vertical_adjustment;
	gfloat vertical_value;

	n = p_get_jobs (&j, p.host, p.port, p.user, p.ssl, GPOINTER_TO_INT
		(gtk_object_get_data (GTK_OBJECT (clist), "j_allocation")),
		p.filter_dest);
	gtk_object_set_data (GTK_OBJECT (clist), "j_allocation", GINT_TO_POINTER (n));
	gtk_clist_freeze (clist);
	update = TRUE;
	vertical_adjustment = gtk_clist_get_vadjustment (clist);
	vertical_value = vertical_adjustment->value;
	gtk_clist_clear (clist);
	
	for (i = 0; i < r ; i++)
	{
		found = FALSE;
		for (v = 0; v < n; v++)
		{
			if (selected_row[i] == atoi (j[v][2]))
			{
				found = TRUE;
				break;
			}
		}
		if (!found)
		{
			selected_row[i--] = selected_row[--r];
			selected_row = g_realloc (selected_row, sizeof (gint) * r);
		}
	}

	for (v = 0; v < n; v++)
	{
		gtk_clist_append (clist, j[v]);
		found = FALSE;
		for (i = 0; i < r ; i++)
		{
			if (selected_row[i] == atoi (j[v][2]))
			{
				found = TRUE;
				break;
			}
		}
		if (found)
			gtk_clist_select_row (clist, v, 1);
	}

	gtk_clist_sort (clist);

	/*
	 * avoid flicker of label widgets
	 */
	
	for (i = 0; i < 12; i++)
	{
		gtk_clist_set_column_auto_resize (GTK_CLIST (clist), i, TRUE);
		gtk_clist_set_column_auto_resize (GTK_CLIST (clist), i, FALSE);
	}
	
	gtk_adjustment_set_value (vertical_adjustment, vertical_value);
	update = FALSE;
	gtk_clist_thaw (clist);
	p_free_jobs (j, n);
}

/*
 * p_timer_rewrite_clist ()
 */

gint p_timer_rewrite_clist (GtkCList *clist)
{
	p_stop_polling (clist);
	p_rewrite_clist (clist);
	p_start_polling (clist);
	return TRUE;
}

/*
 * p_start_polling ()
 */

void p_start_polling (GtkCList *clist)
{
	gtk_object_set_data (GTK_OBJECT (clist), "timer_id",
		GUINT_TO_POINTER (gtk_timeout_add ((p.refresh_interval * 1000),
		(GtkFunction) p_timer_rewrite_clist, clist)));
}

/*
 * p_stop_polling ()
 */

void p_stop_polling (GtkCList *clist)
{
	if ((gtk_object_get_data (GTK_OBJECT (clist), "timer_id")) != NULL)
		gtk_timeout_remove (GPOINTER_TO_UINT (gtk_object_get_data
		(GTK_OBJECT (clist), "timer_id")));
}

/*
 * p_call_multi_request ()
 */

void p_call_multi_request (GtkCList *clist, ipp_op_t w,
	const multi_arg_t multi_arg)
{
	p_stop_polling (clist);
	p_multi_request (selected_row, r, p.host, p.port, p.user,
		p.ssl,w , multi_arg);
	p_rewrite_clist (clist);
	p_start_polling (clist);
}

/*
 * p_delete ()
 */

void p_delete (GtkCList *clist)
{	
	multi_arg_t multi_arg;
	
	if (r == 0)
		return;
	multi_arg.dest = NULL;
	p_call_multi_request (clist, IPP_CANCEL_JOB, multi_arg);
}

/*
 * p_up ()
 */

void p_up (GtkCList *clist)
{
	if (r == 0)
		return;
	p_change_priority (clist, p.priority_change);
}

/*
 * p_down ()
 */

void p_down (GtkCList *clist)
{
	if (r == 0)
		return;
	p_change_priority (clist, (-p.priority_change));
}

/*
 * p_change_priority ()
 */

void p_change_priority (GtkCList *clist, gint change_arg)
{
	multi_arg_t multi_arg;
	gint i;
	gint *priority;
	gchar *s = NULL;
	
	if (r == 0)
		return;
	p_stop_polling (clist);
	priority = g_malloc (sizeof (gint) * r);
	for (i = 0; i < r; i++)
	{
		priority [i] = 50;
		clist->selection = g_list_first (clist->selection);
		while (clist->selection != NULL)
		{
			gtk_clist_get_text (clist, GPOINTER_TO_INT
				(clist->selection->data), 2, &s);
			if (atoi (s) == selected_row[i])
			{
				gtk_clist_get_text (clist, GPOINTER_TO_INT
					(clist->selection->data),1, &s);
				priority[i] = atoi (s) + change_arg;
				if (priority[i] < 0)
					priority[i] = 0;
				else if (priority[i] > 100)
					priority[i] = 100;
			}
			if (clist->selection->next == NULL)
				break;
			else
				clist->selection = clist->selection->next;
		}
	}
	multi_arg.priority = priority;
	p_call_multi_request (clist, IPP_SET_JOB_ATTRIBUTES, multi_arg);
}

/*
 * p_hold ()
 */

void p_hold (GtkCList *clist)
{
	multi_arg_t multi_arg;
	
	if (r == 0)
		return;
	multi_arg.dest = NULL;
	p_call_multi_request (clist, IPP_HOLD_JOB, multi_arg);
}

/*
 * p_release ()
 */

void p_release (GtkCList *clist)
{
	multi_arg_t multi_arg;
	
	if (r == 0)
		return;
	multi_arg.dest = NULL;
	p_call_multi_request (clist, IPP_RELEASE_JOB, multi_arg);
}

/*
 * p_show_move_dialog ()
 */

void p_show_move_dialog (GtkCList *clist_program_window)
{
	GtkWidget *dialog;
	
	GtkWidget *vbox;
	
	GtkWidget *scrolledwindow;
	GtkWidget *clist;
	gint n, i;
	dest_t *dest;
	
	GtkWidget *action_area;
	
	GtkWidget *cancel_button;
	
	GtkWidget *program_window;
	
	if (r == 0)
		return;
	
	/*
	 * dialog
	 */
	
	dialog = gtk_dialog_new ();
	transient = dialog;
	gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
	gtk_window_set_title (GTK_WINDOW (dialog), gettext("Move To"));
	gtk_window_set_policy (GTK_WINDOW (dialog), FALSE, FALSE, TRUE);
	gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER);
	program_window = gtk_object_get_data (GTK_OBJECT
		(clist_program_window), "program_window");
	gtk_window_set_transient_for (GTK_WINDOW (dialog), GTK_WINDOW
		(program_window));

	/*
	 * vbox
	 */

	vbox = GTK_DIALOG (dialog)->vbox;
	gtk_object_set_data (GTK_OBJECT (dialog), "vbox", vbox);
	gtk_widget_set_usize (vbox, 200, 150);
	
	/*
	 * scrolledwindow
	 */

	scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
	gtk_widget_ref (scrolledwindow);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow),
		GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "scrolledwindow", scrolledwindow,
		(GtkDestroyNotify) gtk_widget_unref);
	gtk_container_add (GTK_CONTAINER (vbox), scrolledwindow);

	/*
	 * clist
	 */

	clist = gtk_clist_new (1);
	gtk_widget_ref (clist);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "clist",
		clist, (GtkDestroyNotify) gtk_widget_unref);
	gtk_clist_set_selection_mode (GTK_CLIST (clist), GTK_SELECTION_SINGLE);
	gtk_clist_set_sort_type (GTK_CLIST (clist), GTK_SORT_ASCENDING);
	gtk_container_add (GTK_CONTAINER (scrolledwindow), clist);
	gtk_object_set_data (GTK_OBJECT (clist), "dialog", dialog);
	gtk_object_set_data (GTK_OBJECT (clist), "clist_program_window",
		clist_program_window);
	gtk_clist_set_column_justification (GTK_CLIST (clist), 0, GTK_JUSTIFY_LEFT);
	gtk_clist_set_column_auto_resize (GTK_CLIST (clist), 0, TRUE);

	p_stop_polling (clist_program_window);
	n = p_get_dests (&dest, p.host, p.port, p.user, p.ssl, GPOINTER_TO_INT
		(gtk_object_get_data (GTK_OBJECT (clist_program_window),
		"dest_allocation")));
	gtk_object_set_data (GTK_OBJECT (clist_program_window),
		"dest_allocation", GINT_TO_POINTER (n));

	for (i = 0; i < n; i++)
	{
		gtk_clist_append (GTK_CLIST (clist), dest[i]);
		g_free (dest[i][0]);
	}
	g_free (dest);

	gtk_clist_sort (GTK_CLIST (clist));

	/*
	 * action_area
	 */

	action_area = GTK_DIALOG (dialog)->action_area;
	gtk_object_set_data (GTK_OBJECT (dialog), "action_area", action_area);
	gtk_container_set_border_width (GTK_CONTAINER (action_area), 10);

	/*
	 * cancel_button
	 */

	cancel_button = gtk_button_new_with_label (gettext("Cancel"));
	gtk_widget_ref (cancel_button);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "cancel_button",
		cancel_button, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (action_area), cancel_button, TRUE, TRUE, 0);
	
	/*
	 * connect signals
	 */

	/* clist - "select-row" */

	gtk_signal_connect (GTK_OBJECT (clist), "select-row",
		GTK_SIGNAL_FUNC (p_dest_select), NULL);


	/* cancel_button - "clicked" */ 

	gtk_signal_connect_object (GTK_OBJECT (cancel_button), "clicked",
		GTK_SIGNAL_FUNC (p_reset_transient), GTK_OBJECT
		(program_window));

	gtk_signal_connect_object_after (GTK_OBJECT (cancel_button), "clicked",
		GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (dialog)) ;

	gtk_widget_show_all (dialog);
	gdk_window_set_icon (dialog->window, NULL,
		gtk_object_get_data (GTK_OBJECT (program_window), "pixmap"),
		gtk_object_get_data (GTK_OBJECT (program_window), "bitmap"));

	p_start_polling (clist_program_window);
}

/*
 * p_exit ()
 */

void p_exit (GtkWidget *widget)
{
	GtkCList *clist;
	gint i;

	clist = gtk_object_get_data (GTK_OBJECT (widget), "clist");

	p.width = widget->allocation.width;
	p.height = widget->allocation.height;

	p_write_preferences (&p);
	p_stop_polling (clist);
	if (password != NULL)
	{
	for (i = 0; i < strlen (password); i++)
		password[i] = 0;
	g_free (password);
	}
	g_free (selected_row);
	g_free (p.filter_dest);
	g_free (p.host);
	g_free (p.user);
	gtk_main_quit ();
}

/*
 * p_sort ()
 */

void p_sort (GtkCList *clist, gint column)
{
	if (p.sort_item == column)
	{
		if (!p.sort_toggle)
			p.sort_toggle = TRUE;
		else
			p.sort_toggle = FALSE;
	}
	else
	{
		p.sort_toggle = FALSE;
		p.sort_item = column;
	}
	gtk_clist_set_sort_type (clist, p.sort_toggle);
	gtk_clist_set_sort_column (clist, p.sort_item);
	if ((p.sort_item == 0) || (p.sort_item == 1) || (p.sort_item == 2) ||
		(p.sort_item == 6) || (p.sort_item == 9))
			gtk_clist_set_compare_func (clist, p_gtk_compare);
	else
		gtk_clist_set_compare_func (clist, NULL);
	gtk_clist_sort (clist);
}

/*
 * p_gtk_compare ()
 */

gint p_gtk_compare (GtkCList *clist, gconstpointer p_1, gconstpointer p_2)
{
	gint i_1, i_2;
	
	i_1 = atoi (GTK_CELL_TEXT (((GtkCListRow *) p_1)->cell[clist->sort_column])->text);
	i_2 = atoi (GTK_CELL_TEXT (((GtkCListRow *) p_2)->cell[clist->sort_column])->text);

	return ((i_1 < i_2) ? (-1) : (i_1 > i_2) ? 1 : 0);
}

/*
 * p_select ()
 */

void p_select (GtkCList *clist, gint row, gint column,
	GdkEventButton *event)
{
	gchar *s;
	gpointer a;
			
	if (update)
		return;
	gtk_clist_get_text (clist, row, 2, &s);
	if ((a = g_realloc (selected_row, sizeof (gint) * ++r)) == NULL)
	{
		g_free (selected_row);
		r = 0;
		gtk_clist_unselect_all (clist);
		return;
	}
	selected_row = a;
	selected_row[r - 1] = atoi (s);
}

/*
 * p_unselect ()
 */

void p_unselect (GtkCList *clist, gint row, gint column,
	GdkEventButton *event)
{
	gchar *s;
	gint i, id;
			
	gtk_clist_get_text (clist, row, 2, &s);
	id = atoi (s);
	for (i = 0; i < r ; i++)
		if (selected_row[i] == id)
			break;
	selected_row[i] = selected_row[--r];
	selected_row = g_realloc (selected_row, sizeof (gint) * r);
}

/*
 * p_dest_select ()
 */

void p_dest_select (GtkCList *clist, gint row, gint column,
	GdkEventButton *event)
{
	multi_arg_t multi_arg;
	GtkCList *clist_program_window;
	gchar *s, *move_dest;
	
	clist_program_window = gtk_object_get_data (GTK_OBJECT (clist), "clist_program_window");
	gtk_clist_get_text (clist, row, 0, &s);
	move_dest = g_strdup (s);
	multi_arg.dest = move_dest;
	p_call_multi_request (clist_program_window, CUPS_MOVE_JOB, multi_arg);
	g_free (move_dest);
	transient = gtk_object_get_data (GTK_OBJECT (clist_program_window),
		"program_window");
	gtk_widget_destroy (gtk_object_get_data
		(GTK_OBJECT (clist), "dialog"));
}

/*
 * p_left_button_pressed ()
 */

void p_left_button_pressed (GtkWidget *widget, GdkEventButton *event)
{
	GtkCList *clist;
	
	clist = gtk_object_get_data (GTK_OBJECT (widget), "clist");
	if (event->button == 1)
		p_stop_polling (clist);
}

/*
 * p_left_button_released ()
 */

void p_left_button_released (GtkWidget *widget, GdkEventButton *event)
{
	GtkCList *clist;
	
	clist = gtk_object_get_data (GTK_OBJECT (widget), "clist");
	if (event->button == 1)
	{
		p_rewrite_clist (clist);
		p_start_polling (clist);
	}
}

/*
 * p_right_button_pressed ()
 */

void p_right_button_pressed (GtkWidget *widget, GdkEventButton *event)
{
	if (event->button == 3)
	{
		GdkEventButton *button_event = (GdkEventButton *) event;
		gtk_menu_popup (GTK_MENU (widget), NULL, NULL, NULL, NULL,
			button_event->button, button_event->time);
	}
}

/*
 * p_show_info_dialog ()
 */

void p_show_info_dialog (GtkWidget *widget)
{
	GtkWidget *dialog;
	
	GtkWidget *vbox;
	
	GtkWidget *label_vbox;
	gchar *text;
	GtkWidget *label;
	
	GtkWidget *action_area;
	
	GtkWidget *ok_button;
	
	/*
	 * dialog
	 */

	dialog = gtk_dialog_new ();
	transient = dialog;
	gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
	gtk_window_set_title (GTK_WINDOW (dialog), gettext("Info"));
	gtk_window_set_policy (GTK_WINDOW (dialog), FALSE, FALSE, TRUE);
	gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER);
	gtk_window_set_transient_for (GTK_WINDOW (dialog), GTK_WINDOW (widget));

	/*
	 * vbox
	 */

	vbox = GTK_DIALOG (dialog)->vbox;
	gtk_object_set_data (GTK_OBJECT (dialog), "vbox", vbox);

	/*
	 * label_vbox
	 */

	label_vbox = gtk_vbox_new (FALSE, 0);
	gtk_widget_ref (label_vbox);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "label_vbox",
		label_vbox, (GtkDestroyNotify) gtk_widget_unref);
	gtk_container_add (GTK_CONTAINER (vbox), label_vbox);
	gtk_container_set_border_width (GTK_CONTAINER (label_vbox), 5);

	/*
	 * label
	 */

	text = g_strdup_printf ("PT version 1.2.4\n%c 2002, 2003 The Printing Tool Developers\n\nThis program is free software; you can redistribute it and/or modify\nit under the terms of the GNU General Public License as published by\nthe Free Software Foundation; either version 2 of the License, or\n(at your option) any later version.\n\nThis program is distributed in the hope that it will be useful,\nbut WITHOUT ANY WARRANTY; without even the implied warranty of\nMERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\nGNU General Public License for more details.\n\nYou should have received a copy of the GNU General Public License\nalong with this program; if not, write to the Free Software\nFoundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA", 169);
	label = gtk_label_new (text);
	g_free (text);
	gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_CENTER);
	gtk_box_pack_start (GTK_BOX (label_vbox), label, FALSE, FALSE, 0);
	
	/*
	 * action_area
	 */

	action_area = GTK_DIALOG (dialog)->action_area;
	gtk_object_set_data (GTK_OBJECT (dialog), "action_area", action_area);
	gtk_container_set_border_width (GTK_CONTAINER (action_area), 10);

	/*
	 * ok_button
	 */

	ok_button = gtk_button_new_with_label (gettext("OK"));
	gtk_widget_ref (ok_button);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "ok_button",
		ok_button, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (action_area), ok_button, TRUE, TRUE, 0);
	
	/*
	 * connect signal
	 */

	/* ok_button - "clicked" */ 

	gtk_signal_connect_object (GTK_OBJECT (ok_button), "clicked",
		GTK_SIGNAL_FUNC (p_reset_transient), GTK_OBJECT (widget)) ;

	gtk_signal_connect_object_after (GTK_OBJECT (ok_button), "clicked",
		GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (dialog)) ;

	gtk_widget_show_all (dialog);
	gdk_window_set_icon (dialog->window, NULL,
		gtk_object_get_data (GTK_OBJECT (widget), "pixmap"),
		gtk_object_get_data (GTK_OBJECT (widget), "bitmap"));
}

/*
 * p_reset_transient ()
 */

void p_reset_transient (GtkWidget *widget)
{
	transient = widget;
}

/*
 * p_show_preferences_dialog ()
 */

void p_show_preferences_dialog (GtkWidget *widget)
{
	GtkWidget *dialog;
	
	GtkWidget *vbox;
	
	GtkWidget *hbox;
	
	GtkWidget *vbox_1;
	GtkWidget *vbox_2;
	GtkWidget *vbox_3;

	GtkWidget *view_frame;
	GtkWidget *view_vbox;
	GtkWidget *rank_checkbutton;
	GtkWidget *id_checkbutton;
	GtkWidget *dest_checkbutton;
	GtkWidget *title_checkbutton;
	GtkWidget *user_checkbutton;
	GtkWidget *format_checkbutton;
	GtkWidget *state_checkbutton;
	GtkWidget *size_checkbutton;
	GtkWidget *copies_checkbutton;
	GtkWidget *priority_checkbutton;
	GtkWidget *creation_time_checkbutton;
	GtkWidget *processing_time_checkbutton;
	
	GtkWidget *filter_frame;
	GtkWidget *filter_vbox;
	GtkWidget *filter_combo;

	GtkWidget *host_frame;
	GtkWidget *host_vbox;
	GtkWidget *host_entry;
	GtkWidget *default_host_button;

	GtkWidget *port_frame;
	GtkWidget *port_vbox;
	GtkObject *port_spinbutton_adj;
	GtkWidget *port_spinbutton;
	GtkWidget *default_port_button;

	GtkWidget *refresh_frame;
	GtkWidget *refresh_vbox;
	GtkObject *refresh_spinbutton_adj;
	GtkWidget *refresh_spinbutton;
	GtkWidget *default_refresh_button;

	GtkWidget *user_frame;
	GtkWidget *user_vbox;
	GtkWidget *user_entry;
	GtkWidget *default_user_button;

	GtkWidget *ssl_frame;
	GtkWidget *ssl_vbox;
	GSList *ssl_group = NULL;
	GtkWidget *never_button;
	GtkWidget *if_requested_button;
	GtkWidget *required_button;
	GtkWidget *always_button;
	GtkWidget *empty_vbox;
	GtkWidget *default_ssl_button;

	GtkWidget *change_frame;
	GtkWidget *change_vbox;
	GtkObject *change_spinbutton_adj;
	GtkWidget *change_spinbutton;
	GtkWidget *default_change_button;

	GtkWidget *action_area;

	GtkWidget *cancel_button;
	GtkWidget *ok_button;
		
	/*
	 * dialog
	 */

	dialog = gtk_dialog_new ();
	transient = dialog;
	gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
	gtk_window_set_title (GTK_WINDOW (dialog), gettext("Preferences"));
	gtk_window_set_policy (GTK_WINDOW (dialog), FALSE, FALSE, TRUE);
	gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER);
	gtk_object_set_data (GTK_OBJECT (dialog), "program_window", widget);
	gtk_object_set_data (GTK_OBJECT (dialog), "changed", GINT_TO_POINTER (FALSE));
	gtk_object_set_data (GTK_OBJECT (dialog), "init", GINT_TO_POINTER (TRUE));
	gtk_window_set_transient_for (GTK_WINDOW (dialog), GTK_WINDOW (widget));
	gtk_object_set_data (GTK_OBJECT (dialog), "clist",
		gtk_object_get_data (GTK_OBJECT (widget), "clist"));

	/*
	 * vbox
	 */

	vbox = GTK_DIALOG (dialog)->vbox;
	gtk_object_set_data (GTK_OBJECT (dialog), "vbox", vbox);

	/*
	 * hbox
	 */

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_widget_ref (hbox);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "hbox",
		hbox, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (hbox), 5);


	/*
	 * vbox_1
	 */

	vbox_1 = gtk_vbox_new (FALSE, 5);
	gtk_widget_ref (vbox_1);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "vbox_1",
		vbox_1, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (hbox), vbox_1, FALSE, FALSE, 5);

	/*
	 * vbox_2
	 */

	vbox_2 = gtk_vbox_new (FALSE, 5);
	gtk_widget_ref (vbox_2);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "vbox_2",
		vbox_2, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (hbox), vbox_2, FALSE, FALSE, 5);

	/*
	 * vbox_3
	 */

	vbox_3 = gtk_vbox_new (FALSE, 5);
	gtk_widget_ref (vbox_3);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "vbox_3",
		vbox_3, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (hbox), vbox_3, FALSE, FALSE, 5);

	/*
	 * view_frame
	 */

	view_frame = gtk_frame_new (gettext("View"));
	gtk_widget_ref (view_frame);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "view_frame",
		view_frame, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (vbox_1), view_frame, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (view_frame), 0);

	/*
	 * view_vbox
	 */

	view_vbox = gtk_vbox_new (FALSE, 0);
	gtk_widget_ref (view_vbox);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "view_vbox",
		view_vbox, (GtkDestroyNotify) gtk_widget_unref);
	gtk_container_add (GTK_CONTAINER (view_frame), view_vbox);
	gtk_container_set_border_width (GTK_CONTAINER (view_vbox), 5);

	/*
	 * checkbuttons 
	 */

	rank_checkbutton = gtk_check_button_new_with_label (gettext("Rank"));
	gtk_widget_ref (rank_checkbutton);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "rank_checkbutton",
		rank_checkbutton, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (view_vbox), rank_checkbutton, FALSE, FALSE, 0);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (rank_checkbutton), p.show_rank);

	priority_checkbutton = gtk_check_button_new_with_label (gettext("Priority"));
	gtk_widget_ref (priority_checkbutton);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "priority_checkbutton",
		priority_checkbutton, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (view_vbox), priority_checkbutton, FALSE, FALSE, 0);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (priority_checkbutton), p.show_priority);

	id_checkbutton = gtk_check_button_new_with_label (gettext("ID"));
	gtk_widget_ref (id_checkbutton);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "id_checkbutton",
		id_checkbutton, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (view_vbox), id_checkbutton, FALSE, FALSE, 0);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (id_checkbutton), p.show_id);

	user_checkbutton = gtk_check_button_new_with_label (gettext("User"));
	gtk_widget_ref (user_checkbutton);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "user_checkbutton",
		user_checkbutton, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (view_vbox), user_checkbutton, FALSE, FALSE, 0);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (user_checkbutton), p.show_user);

	title_checkbutton = gtk_check_button_new_with_label (gettext("Title"));
	gtk_widget_ref (title_checkbutton);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "title_checkbutton",
		title_checkbutton, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (view_vbox), title_checkbutton, FALSE, FALSE, 0);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (title_checkbutton), p.show_title);

	format_checkbutton = gtk_check_button_new_with_label (gettext("Format"));
	gtk_widget_ref (format_checkbutton);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "format_checkbutton",
		format_checkbutton, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (view_vbox), format_checkbutton, FALSE, FALSE, 0);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (format_checkbutton), p.show_format);

	copies_checkbutton = gtk_check_button_new_with_label (gettext("Copies"));
	gtk_widget_ref (copies_checkbutton);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "copies_checkbutton",
		copies_checkbutton, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (view_vbox), copies_checkbutton, FALSE, FALSE, 0);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (copies_checkbutton), p.show_copies);

	dest_checkbutton = gtk_check_button_new_with_label (gettext("Printer"));
	gtk_widget_ref (dest_checkbutton);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "dest_checkbutton",
		dest_checkbutton, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (view_vbox), dest_checkbutton, FALSE, FALSE, 0);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dest_checkbutton), p.show_dest);

	state_checkbutton = gtk_check_button_new_with_label (gettext("State"));
	gtk_widget_ref (state_checkbutton);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "state_checkbutton",
		state_checkbutton, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (view_vbox), state_checkbutton, FALSE, FALSE, 0);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (state_checkbutton), p.show_state);

	size_checkbutton = gtk_check_button_new_with_label (gettext("Size"));
	gtk_widget_ref (size_checkbutton);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "size_checkbutton",
		size_checkbutton, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (view_vbox), size_checkbutton, FALSE, FALSE, 0);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (size_checkbutton), p.show_size);

	creation_time_checkbutton = gtk_check_button_new_with_label (gettext("Creation Time"));
	gtk_widget_ref (creation_time_checkbutton);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "creation_time_checkbutton",
		creation_time_checkbutton, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (view_vbox), creation_time_checkbutton, FALSE, FALSE, 0);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (creation_time_checkbutton), p.show_creation_time);

	processing_time_checkbutton = gtk_check_button_new_with_label (gettext("Processing Time"));
	gtk_widget_ref (processing_time_checkbutton);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "processing_time_checkbutton",
		processing_time_checkbutton, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (view_vbox), processing_time_checkbutton, FALSE, FALSE, 0);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (processing_time_checkbutton), p.show_processing_time);

	/*
	 * filter_frame
	 */

	filter_frame = gtk_frame_new (gettext("Printer"));
	gtk_widget_ref (filter_frame);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "filter_frame",
		filter_frame, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (vbox_2), filter_frame, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (filter_frame), 0);

	/*
	 * filter_vbox
	 */

	filter_vbox = gtk_vbox_new (FALSE, 0);
	gtk_widget_ref (filter_vbox);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "filter_vbox",
		filter_vbox, (GtkDestroyNotify) gtk_widget_unref);
	gtk_container_add (GTK_CONTAINER (filter_frame), filter_vbox);
	gtk_container_set_border_width (GTK_CONTAINER (filter_vbox), 5);

	/*
	 * filter_combo
	 */
	 
	filter_combo = gtk_combo_new ();
	gtk_widget_ref (filter_combo);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "filter_combo",
		filter_combo, (GtkDestroyNotify) gtk_widget_unref);
	gtk_entry_set_max_length (GTK_ENTRY (GTK_COMBO (filter_combo)->entry),
		MAX_STRING_LENGTH);
	gtk_box_pack_start (GTK_BOX (filter_vbox), filter_combo, FALSE, FALSE, 0);
	gtk_combo_set_case_sensitive (GTK_COMBO (filter_combo), TRUE);
	gtk_combo_set_value_in_list (GTK_COMBO (filter_combo), TRUE, FALSE);

	/*
	 * host_frame
	 */

	host_frame = gtk_frame_new (gettext("Host"));
	gtk_widget_ref (host_frame);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "host_frame",
		host_frame, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (vbox_2), host_frame, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (host_frame), 0);

	/*
	 * host_vbox
	 */

	host_vbox = gtk_vbox_new (FALSE, 5);
	gtk_widget_ref (host_vbox);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "host_vbox",
		host_vbox, (GtkDestroyNotify) gtk_widget_unref);
	gtk_container_add (GTK_CONTAINER (host_frame), host_vbox);
	gtk_container_set_border_width (GTK_CONTAINER (host_vbox), 5);

	/*
	 * host_entry
	 */

	host_entry = gtk_entry_new ();
	gtk_widget_ref (host_entry);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "host_entry",
		host_entry, (GtkDestroyNotify) gtk_widget_unref);
	gtk_entry_set_max_length (GTK_ENTRY (host_entry), MAX_STRING_LENGTH);
	gtk_box_pack_start (GTK_BOX (host_vbox), host_entry, FALSE, FALSE, 0);
	gtk_entry_set_text (GTK_ENTRY (host_entry), p.host);

	/*
	 * default_host_button
	 */

	default_host_button = gtk_button_new_with_label (gettext("Default"));
	gtk_widget_ref (default_host_button);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "default_host_button",
		default_host_button, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (host_vbox), default_host_button, FALSE, FALSE, 0);

	/*
	 * port_frame
	 */

	port_frame = gtk_frame_new (gettext("Port"));
	gtk_widget_ref (port_frame);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "port_frame",
		port_frame, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (vbox_2), port_frame, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (port_frame), 0);

	/*
	 * port_vbox
	 */

	port_vbox = gtk_vbox_new (FALSE, 5);
	gtk_widget_ref (port_vbox);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "port_vbox",
		port_vbox, (GtkDestroyNotify) gtk_widget_unref);
	gtk_container_add (GTK_CONTAINER (port_frame), port_vbox);
	gtk_container_set_border_width (GTK_CONTAINER (port_vbox), 5);

	/*
	 * port_spinnbutton
	 */

	port_spinbutton_adj = gtk_adjustment_new (p.port, 0, MAX_PORT_VALUE, 1, 10, 10);
	port_spinbutton = gtk_spin_button_new (GTK_ADJUSTMENT (port_spinbutton_adj), 1, 0);
	gtk_widget_ref (port_spinbutton);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "port_spinbutton",
		port_spinbutton, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (port_vbox), port_spinbutton, FALSE, FALSE, 0);
	gtk_entry_set_max_length ( GTK_ENTRY (port_spinbutton), 5);
	gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (port_spinbutton), TRUE);
	gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (port_spinbutton), GTK_UPDATE_IF_VALID);

	/*
	 * default_port_button
	 */

	default_port_button = gtk_button_new_with_label (gettext("Default"));
	gtk_widget_ref (default_port_button);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "default_port_button",
		default_port_button, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (port_vbox), default_port_button, FALSE, FALSE, 0);

	/*
	 * refresh_frame
	 */

	refresh_frame = gtk_frame_new (gettext("Refresh Interval (s)"));
	gtk_widget_ref (refresh_frame);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "refresh_frame",
		refresh_frame, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (vbox_2), refresh_frame, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (refresh_frame), 0);

	/*
	 * refresh_vbox
	 */

	refresh_vbox = gtk_vbox_new (FALSE, 5);
	gtk_widget_ref (refresh_vbox);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "refresh_vbox",
		refresh_vbox, (GtkDestroyNotify) gtk_widget_unref);
	gtk_container_add (GTK_CONTAINER (refresh_frame), refresh_vbox);
	gtk_container_set_border_width (GTK_CONTAINER (refresh_vbox), 5);

	/*
	 * refresh_spinnbutton
	 */

	refresh_spinbutton_adj = gtk_adjustment_new (p.refresh_interval, 1, 10, 1, 10, 10);
	refresh_spinbutton = gtk_spin_button_new (GTK_ADJUSTMENT (refresh_spinbutton_adj), 1, 0);
	gtk_widget_ref (refresh_spinbutton);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "refresh_spinbutton",
		refresh_spinbutton, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (refresh_vbox), refresh_spinbutton, FALSE, FALSE, 0);
	gtk_entry_set_max_length ( GTK_ENTRY (refresh_spinbutton), 2);
	gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (refresh_spinbutton), TRUE);
	gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (refresh_spinbutton), GTK_UPDATE_IF_VALID);

	/*
	 * default_refresh_button
	 */

	default_refresh_button = gtk_button_new_with_label (gettext("Default"));
	gtk_widget_ref (default_refresh_button);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "default_refresh_button",
		default_refresh_button, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (refresh_vbox), default_refresh_button, FALSE, FALSE, 0);

	/*
	 * user_frame
	 */

	user_frame = gtk_frame_new (gettext("User"));
	gtk_widget_ref (user_frame);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "user_frame",
		user_frame, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (vbox_3), user_frame, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (user_frame), 0);
#ifndef ALIEN_USER
	gtk_widget_set_sensitive (user_frame, FALSE);
#endif /*ALIEN_USER*/

	/*
	 * user_vbox
	 */

	user_vbox = gtk_vbox_new (FALSE, 5);
	gtk_widget_ref (user_vbox);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "user_vbox",
		user_vbox, (GtkDestroyNotify) gtk_widget_unref);
	gtk_container_add (GTK_CONTAINER (user_frame), user_vbox);
	gtk_container_set_border_width (GTK_CONTAINER (user_vbox), 5);

	/*
	 * user_entry
	 */

	user_entry = gtk_entry_new ();
	gtk_widget_ref (user_entry);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "user_entry",
		user_entry, (GtkDestroyNotify) gtk_widget_unref);
	gtk_entry_set_max_length (GTK_ENTRY (user_entry), MAX_STRING_LENGTH);
	gtk_box_pack_start (GTK_BOX (user_vbox), user_entry, FALSE, FALSE, 0);
	gtk_entry_set_text (GTK_ENTRY (user_entry), p.user);

	/*
	 * default_user_button
	 */

	default_user_button = gtk_button_new_with_label (gettext("Default"));
	gtk_widget_ref (default_user_button);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "default_user_button",
		default_user_button, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (user_vbox), default_user_button, FALSE, FALSE, 0);

	/*
	 * ssl_frame
	 */

	ssl_frame = gtk_frame_new (gettext("GNUTLS"));
	gtk_widget_ref (ssl_frame);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "ssl_frame",
		ssl_frame, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (vbox_3), ssl_frame, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (ssl_frame), 0);
#ifndef HAVE_LIBGNUTLS
	gtk_widget_set_sensitive (ssl_frame, FALSE);
#endif /*HAVE_LIBGNUTLS*/

	/*
	 * ssl_vbox
	 */

	ssl_vbox = gtk_vbox_new (FALSE, 0);
	gtk_widget_ref (ssl_vbox);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "ssl_vbox",
		ssl_vbox, (GtkDestroyNotify) gtk_widget_unref);
	gtk_container_add (GTK_CONTAINER (ssl_frame), ssl_vbox);
	gtk_container_set_border_width (GTK_CONTAINER (ssl_vbox), 5);

	/*
	 * if_requested_button
	 */

	if_requested_button = gtk_radio_button_new_with_label (ssl_group, gettext("If Requested"));
	ssl_group = gtk_radio_button_group (GTK_RADIO_BUTTON (if_requested_button));
	gtk_widget_ref (if_requested_button);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "if_requested_button",
		if_requested_button, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (ssl_vbox), if_requested_button, FALSE, FALSE, 0);

	/*
	 * never_button
	 */

	never_button = gtk_radio_button_new_with_label (ssl_group, gettext("Never"));
	ssl_group = gtk_radio_button_group (GTK_RADIO_BUTTON (never_button));
	gtk_widget_ref (never_button);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "never_button",
		never_button, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (ssl_vbox), never_button, FALSE, FALSE, 0);
#ifndef HAVE_LIBGNUTLS
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (never_button), TRUE);
#endif /*HAVE_LIBGNUTLS*/

	/*
	 * required_button
	 */

	required_button = gtk_radio_button_new_with_label (ssl_group, gettext("Required"));
	ssl_group = gtk_radio_button_group (GTK_RADIO_BUTTON (required_button));
	gtk_widget_ref (required_button);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "required_button",
		required_button, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (ssl_vbox), required_button, FALSE, FALSE, 0);

	/*
	 * always_button
	 */

	always_button = gtk_radio_button_new_with_label (ssl_group, gettext("Always"));
	ssl_group = gtk_radio_button_group (GTK_RADIO_BUTTON (always_button));
	gtk_widget_ref (always_button);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "always_button",
		always_button, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (ssl_vbox), always_button, FALSE, FALSE, 0);

	/*
	 * empty_vbox
	 */

	empty_vbox = gtk_vbox_new (FALSE, 0);
	gtk_widget_ref (empty_vbox);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "empty_vbox",
		empty_vbox, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (ssl_vbox), empty_vbox, FALSE, FALSE, 0);
	gtk_widget_set_usize (empty_vbox, 0, 5);

	/*
	 * default_ssl_button
	 */

	default_ssl_button = gtk_button_new_with_label (gettext("Default"));
	gtk_widget_ref (default_ssl_button);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "default_ssl_button",
		default_ssl_button, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (ssl_vbox), default_ssl_button, FALSE, FALSE, 0);

	/*
	 * change_frame
	 */

	change_frame = gtk_frame_new (gettext("Change Priority By"));
	gtk_widget_ref (change_frame);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "change_frame",
		change_frame, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (vbox_3), change_frame, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (change_frame), 0);

	/*
	 * change_vbox
	 */

	change_vbox = gtk_vbox_new (FALSE, 5);
	gtk_widget_ref (change_vbox);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "change_vbox",
		change_vbox, (GtkDestroyNotify) gtk_widget_unref);
	gtk_container_add (GTK_CONTAINER (change_frame), change_vbox);
	gtk_container_set_border_width (GTK_CONTAINER (change_vbox), 5);

	/*
	 * change_spinnbutton
	 */

	change_spinbutton_adj = gtk_adjustment_new (p.priority_change, 1, 100, 1, 10, 10);
	change_spinbutton = gtk_spin_button_new (GTK_ADJUSTMENT
		(change_spinbutton_adj), 1, 0);
	gtk_widget_ref (change_spinbutton);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "change_spinbutton",
		change_spinbutton, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (change_vbox), change_spinbutton, FALSE, FALSE, 0);
	gtk_entry_set_max_length ( GTK_ENTRY (change_spinbutton), 3);
	gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (change_spinbutton), TRUE);
	gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (change_spinbutton), GTK_UPDATE_IF_VALID);

	/*
	 * default_change_button
	 */

	default_change_button = gtk_button_new_with_label (gettext("Default"));
	gtk_widget_ref (default_change_button);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "default_change_button",
		default_change_button, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (change_vbox), default_change_button, FALSE, FALSE, 0);

#ifdef HAVE_LIBGNUTLS
	/*
	 * activate ssl_button
	 */

	p_get_ssl_buttons (dialog);
#endif /*HAVE_LIBGNUTLS*/

	/*
	 * load filter combo
	 */

	p_get_filter (dialog);

	/*
	 * action_area
	 */

	action_area = GTK_DIALOG (dialog)->action_area;
	gtk_object_set_data (GTK_OBJECT (dialog), "action_area", action_area);
	gtk_container_set_border_width (GTK_CONTAINER (action_area), 10);

	/*
	 * cancel_button
	 */

	cancel_button = gtk_button_new_with_label (gettext("Cancel"));
	gtk_widget_ref (cancel_button);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "cancel_button",
		cancel_button, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (action_area), cancel_button, TRUE, TRUE, 0);

	/*
	 * ok_button
	 */

	ok_button = gtk_button_new_with_label (gettext("OK"));
	gtk_widget_ref (ok_button);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "ok_button",
		ok_button, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (action_area), ok_button, TRUE, TRUE, 0);

	/*
	 * connect signals
	 */

	/* rank_checkbutton - toggled */
	
	gtk_signal_connect_object (GTK_OBJECT (rank_checkbutton), "toggled",
		GTK_SIGNAL_FUNC (p_preferences_change_action), GTK_OBJECT (dialog));

	/* priority_checkbutton - toggled */
	
	gtk_signal_connect_object (GTK_OBJECT (priority_checkbutton), "toggled",
		GTK_SIGNAL_FUNC (p_preferences_change_action), GTK_OBJECT (dialog));

	/* id_checkbutton - toggled */
	
	gtk_signal_connect_object (GTK_OBJECT (id_checkbutton), "toggled",
		GTK_SIGNAL_FUNC (p_preferences_change_action), GTK_OBJECT (dialog));

	/* user_checkbutton - toggled */
	
	gtk_signal_connect_object (GTK_OBJECT (user_checkbutton), "toggled",
		GTK_SIGNAL_FUNC (p_preferences_change_action), GTK_OBJECT (dialog));

	/* title_checkbutton - toggled */
	
	gtk_signal_connect_object (GTK_OBJECT (title_checkbutton), "toggled",
		GTK_SIGNAL_FUNC (p_preferences_change_action), GTK_OBJECT (dialog));

	/* format_checkbutton - toggled */
	
	gtk_signal_connect_object (GTK_OBJECT (format_checkbutton), "toggled",
		GTK_SIGNAL_FUNC (p_preferences_change_action), GTK_OBJECT (dialog));

	/* copies_checkbutton - toggled */
	
	gtk_signal_connect_object (GTK_OBJECT (copies_checkbutton), "toggled",
		GTK_SIGNAL_FUNC (p_preferences_change_action), GTK_OBJECT (dialog));

	/* dest_checkbutton - toggled */
	
	gtk_signal_connect_object (GTK_OBJECT (dest_checkbutton), "toggled",
		GTK_SIGNAL_FUNC (p_preferences_change_action), GTK_OBJECT (dialog));

	/* state_checkbutton - toggled */
	
	gtk_signal_connect_object (GTK_OBJECT (state_checkbutton), "toggled",
		GTK_SIGNAL_FUNC (p_preferences_change_action), GTK_OBJECT (dialog));

	/* size_checkbutton - toggled */
	
	gtk_signal_connect_object (GTK_OBJECT (size_checkbutton), "toggled",
		GTK_SIGNAL_FUNC (p_preferences_change_action), GTK_OBJECT (dialog));

	/* creation_time_checkbutton - toggled */
	
	gtk_signal_connect_object (GTK_OBJECT (creation_time_checkbutton), "toggled",
		GTK_SIGNAL_FUNC (p_preferences_change_action), GTK_OBJECT (dialog));

	/* processing_time_checkbutton - toggled */
	
	gtk_signal_connect_object (GTK_OBJECT (processing_time_checkbutton), "toggled",
		GTK_SIGNAL_FUNC (p_preferences_change_action), GTK_OBJECT (dialog));

	/* filter_combo->entry - "changed" */

	gtk_signal_connect_object (GTK_OBJECT (GTK_COMBO (filter_combo)->entry), "changed",
		GTK_SIGNAL_FUNC (p_preferences_change_action), GTK_OBJECT (dialog));

	/* host_entry - changed */
	
	gtk_signal_connect_object (GTK_OBJECT (host_entry), "changed",
		GTK_SIGNAL_FUNC (p_preferences_change_action), GTK_OBJECT (dialog));

	gtk_signal_connect_object (GTK_OBJECT (host_entry), "changed",
		GTK_SIGNAL_FUNC (p_get_filter), GTK_OBJECT (dialog));

	/* default_host_button - "clicked" */ 

	gtk_signal_connect_object (GTK_OBJECT (default_host_button), "clicked",
		GTK_SIGNAL_FUNC (p_get_host), GTK_OBJECT (host_entry)) ;

	/* port_spinbutton - changed */
	
	gtk_signal_connect_object (GTK_OBJECT (port_spinbutton), "changed",
		GTK_SIGNAL_FUNC (p_preferences_change_action), GTK_OBJECT (dialog));

	gtk_signal_connect_object (GTK_OBJECT (port_spinbutton), "changed",
		GTK_SIGNAL_FUNC (p_get_filter), GTK_OBJECT (dialog));

	/* default_port_button - "clicked" */ 

	gtk_signal_connect_object (GTK_OBJECT (default_port_button), "clicked",
		GTK_SIGNAL_FUNC (p_get_port), GTK_OBJECT (port_spinbutton)) ;

	/* refresh_spinbutton - changed */
	
	gtk_signal_connect_object (GTK_OBJECT (refresh_spinbutton), "changed",
		GTK_SIGNAL_FUNC (p_preferences_change_action), GTK_OBJECT (dialog));

	/* default_refresh_button - "clicked" */ 

	gtk_signal_connect_object (GTK_OBJECT (default_refresh_button), "clicked",
		GTK_SIGNAL_FUNC (p_get_refresh), GTK_OBJECT (refresh_spinbutton)) ;

	/* change_spinbutton - changed */
	
	gtk_signal_connect_object (GTK_OBJECT (change_spinbutton), "changed",
		GTK_SIGNAL_FUNC (p_preferences_change_action), GTK_OBJECT (dialog));

	/* default_change_button - "clicked" */ 

	gtk_signal_connect_object (GTK_OBJECT (default_change_button), "clicked",
		GTK_SIGNAL_FUNC (p_get_change), GTK_OBJECT (change_spinbutton)) ;

#ifdef ALIEN_USER
	/* user_entry - changed */
	
	gtk_signal_connect_object (GTK_OBJECT (user_entry), "changed",
		GTK_SIGNAL_FUNC (p_preferences_change_action), GTK_OBJECT (dialog));

	/* default_user_button - "clicked" */ 

	gtk_signal_connect_object (GTK_OBJECT (default_user_button), "clicked",
		GTK_SIGNAL_FUNC (p_get_user), GTK_OBJECT (user_entry)) ;
#endif /*ALIEN_USER*/

#ifdef HAVE_LIBGNUTLS
	/* if_requested_button - toggled */
	
	gtk_signal_connect_object (GTK_OBJECT (if_requested_button), "toggled",
		GTK_SIGNAL_FUNC (p_preferences_change_action), GTK_OBJECT (dialog));

	gtk_signal_connect_object (GTK_OBJECT (if_requested_button), "toggled",
		GTK_SIGNAL_FUNC (p_get_filter), GTK_OBJECT (dialog));

	/* never_button - toggled */
	
	gtk_signal_connect_object (GTK_OBJECT (never_button), "toggled",
		GTK_SIGNAL_FUNC (p_preferences_change_action), GTK_OBJECT (dialog));

	gtk_signal_connect_object (GTK_OBJECT (never_button), "toggled",
		GTK_SIGNAL_FUNC (p_get_filter), GTK_OBJECT (dialog));

	/* required_button - toggled */
	
	gtk_signal_connect_object (GTK_OBJECT (required_button), "toggled",
		GTK_SIGNAL_FUNC (p_preferences_change_action), GTK_OBJECT (dialog));

	gtk_signal_connect_object (GTK_OBJECT (required_button), "toggled",
		GTK_SIGNAL_FUNC (p_get_filter), GTK_OBJECT (dialog));

	/* always_button - toggled */
	
	gtk_signal_connect_object (GTK_OBJECT (always_button), "toggled",
		GTK_SIGNAL_FUNC (p_preferences_change_action), GTK_OBJECT (dialog));

	gtk_signal_connect_object (GTK_OBJECT (always_button), "toggled",
		GTK_SIGNAL_FUNC (p_get_filter), GTK_OBJECT (dialog));

	/* default_ssl_button - "clicked" */ 

	gtk_signal_connect_object (GTK_OBJECT (default_ssl_button), "clicked",
		GTK_SIGNAL_FUNC (p_get_ssl_buttons), GTK_OBJECT (dialog)) ;
#endif /*HAVE_LIBGNUTLS*/

	/* cancel_button - "clicked" */ 

	gtk_signal_connect_object (GTK_OBJECT (cancel_button), "clicked",
		GTK_SIGNAL_FUNC (p_reset_transient), GTK_OBJECT (widget)) ;

	gtk_signal_connect_object_after (GTK_OBJECT (cancel_button), "clicked",
		GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (dialog)) ;

	/* ok_button - "clicked" */ 

	gtk_signal_connect_object (GTK_OBJECT (ok_button), "clicked",
		GTK_SIGNAL_FUNC (p_preferences_ok), GTK_OBJECT (dialog)) ;

	gtk_widget_show_all (dialog);
	gdk_window_set_icon (dialog->window, NULL,
		gtk_object_get_data (GTK_OBJECT (widget), "pixmap"),
		gtk_object_get_data (GTK_OBJECT (widget), "bitmap"));
}

/*
 * p_preferences_change_action ()
 */

void p_preferences_change_action (GtkWidget *widget)
{
	gtk_object_set_data (GTK_OBJECT (widget), "changed", GINT_TO_POINTER (TRUE));
}

/*
 * p_preferences_ok ()
 */

void p_preferences_ok (GtkWidget *widget)
{
	GtkCList *clist;

	clist = gtk_object_get_data (GTK_OBJECT (widget), "clist");
	
	if (GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (widget),
		"changed")))
	{
		p_stop_polling (clist);
		
		p.show_rank = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
			(gtk_object_get_data (GTK_OBJECT (widget), "rank_checkbutton")));
		p.show_priority = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
			(gtk_object_get_data (GTK_OBJECT (widget), "priority_checkbutton")));
		p.show_id = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
			(gtk_object_get_data (GTK_OBJECT (widget), "id_checkbutton")));
		p.show_user = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
			(gtk_object_get_data (GTK_OBJECT (widget), "user_checkbutton")));
		p.show_title = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
			(gtk_object_get_data (GTK_OBJECT (widget), "title_checkbutton")));
		p.show_format = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
			(gtk_object_get_data (GTK_OBJECT (widget), "format_checkbutton")));
		p.show_copies = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
			(gtk_object_get_data (GTK_OBJECT (widget), "copies_checkbutton")));
		p.show_dest = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
			(gtk_object_get_data (GTK_OBJECT (widget), "dest_checkbutton")));
		p.show_state = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
			(gtk_object_get_data (GTK_OBJECT (widget), "state_checkbutton")));
		p.show_size = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
			(gtk_object_get_data (GTK_OBJECT (widget), "size_checkbutton")));
		p.show_creation_time = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
			(gtk_object_get_data (GTK_OBJECT (widget), "creation_time_checkbutton")));
		p.show_processing_time = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
			(gtk_object_get_data (GTK_OBJECT (widget), "processing_time_checkbutton")));

		g_free (p.filter_dest);
	
		p.filter_dest = g_strdup (gtk_entry_get_text (GTK_ENTRY
			(GTK_COMBO (gtk_object_get_data (GTK_OBJECT (widget),
			"filter_combo"))->entry)));
		
		if ((strcmp (p.filter_dest, gettext ("(All Printers)")) == 0) ||
			(strcmp (p.filter_dest, gettext ("(Not Available)")) == 0))
		{
			g_free (p.filter_dest);
			p.filter_dest = g_strdup (NO_ARG);
		}

		g_free (p.host);

		p.host = g_strdup (gtk_entry_get_text (GTK_ENTRY (gtk_object_get_data (GTK_OBJECT
			(widget), "host_entry"))));

		p.port = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON
			(gtk_object_get_data (GTK_OBJECT (widget), "port_spinbutton")));
		
		p.refresh_interval = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON
			(gtk_object_get_data (GTK_OBJECT (widget), "refresh_spinbutton")));

		p.priority_change = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON
			(gtk_object_get_data (GTK_OBJECT (widget), "change_spinbutton")));

#ifdef ALIEN_USER
		g_free (p.user);

		p.user = g_strdup (gtk_entry_get_text (GTK_ENTRY (gtk_object_get_data (GTK_OBJECT
			(widget), "user_entry"))));
#endif /*ALIEN_USER*/

#ifdef HAVE_LIBGNUTLS
		p.ssl = p_get_ssl_value (widget);
#endif /*HAVE_LIBGNUTLS*/

		p_apply_preferences (gtk_object_get_data (GTK_OBJECT (widget),
			"program_window"));
	}
	p_reset_transient (gtk_object_get_data (GTK_OBJECT (widget),
		"program_window"));
	gtk_widget_destroy (widget);
}

/*
 * p_get_filter ()
 */

void p_get_filter (GtkWidget *widget)
{
	GtkCList *clist;
	dest_t *dest;
	GList *filter_list = NULL;
	gint i, n;
	gboolean in_list = FALSE;
	gchar *tmp_host;
#ifdef ALIEN_USER
	gchar *tmp_user;
#endif /*ALIEN_USER*/

	clist = gtk_object_get_data (GTK_OBJECT (widget), "clist");
	
	tmp_host = g_strdup (gtk_entry_get_text (GTK_ENTRY (gtk_object_get_data (GTK_OBJECT
			(widget), "host_entry"))));

#ifdef ALIEN_USER
	tmp_user = g_strdup (gtk_entry_get_text (GTK_ENTRY (gtk_object_get_data (GTK_OBJECT
			(widget), "user_entry"))));
#endif /*ALIEN_USER*/

	p_stop_polling (clist);

#ifndef ALIEN_USER
#ifndef HAVE_LIBGNUTLS
	n = p_get_dests (&dest, tmp_host, gtk_spin_button_get_value_as_int
		(GTK_SPIN_BUTTON (gtk_object_get_data (GTK_OBJECT (widget),
		"port_spinbutton"))), p.user, p.ssl,
		GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (clist),
		"dest_allocation")));
#endif /*HAVE_LIBGNUTLS*/
#endif /*ALIEN_USER*/

#ifndef ALIEN_USER
#ifdef HAVE_LIBGNUTLS
	n = p_get_dests (&dest, tmp_host, gtk_spin_button_get_value_as_int
		(GTK_SPIN_BUTTON (gtk_object_get_data (GTK_OBJECT (widget),
		"port_spinbutton"))), p.user, p_get_ssl_value (widget),
		GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (clist),
		"dest_allocation")));
#endif /*HAVE_LIBGNUTLS*/
#endif /*ALIEN_USER*/

#ifdef ALIEN_USER
#ifndef HAVE_LIBGNUTLS
	n = p_get_dests (&dest, tmp_host, gtk_spin_button_get_value_as_int
		(GTK_SPIN_BUTTON (gtk_object_get_data (GTK_OBJECT (widget),
		"port_spinbutton"))), tmp_user, p.ssl,
		GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (clist),
		"dest_allocation")));
#endif /*HAVE_LIBGNUTLS*/
#endif /*ALIEN_USER*/

#ifdef ALIEN_USER
#ifdef HAVE_LIBGNUTLS
	n = p_get_dests (&dest, tmp_host, gtk_spin_button_get_value_as_int
		(GTK_SPIN_BUTTON (gtk_object_get_data (GTK_OBJECT (widget),
		"port_spinbutton"))), tmp_user, p_get_ssl_value (widget),
		GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (clist),
		"dest_allocation")));
#endif /*HAVE_LIBGNUTLS*/
#endif /*ALIEN_USER*/

	g_free (tmp_host);
#ifdef ALIEN_USER
	g_free (tmp_user);
#endif /*ALIEN_USER*/

	gtk_object_set_data (GTK_OBJECT (clist), "dest_allocation", GINT_TO_POINTER (n));

	if (n < 0)
	{
		filter_list = g_list_append (filter_list,
			gettext("(Not Available)"));
		gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (gtk_object_get_data
			(GTK_OBJECT (widget), "filter_combo"))->entry),
			gettext ("(Not Available)"));

		gtk_combo_set_popdown_strings (GTK_COMBO (gtk_object_get_data
			(GTK_OBJECT (widget), "filter_combo")), g_list_first (filter_list));

	}
	else
	{
		filter_list = g_list_append (filter_list,
			gettext("(All Printers)"));
		for (i = 0; i < n; i++)
		{
			filter_list = g_list_append (filter_list, dest[i][0]);
			if (strcmp (dest[i][0], p.filter_dest) == 0)
				in_list = TRUE;
		}
		g_list_sort (filter_list, p_g_compare);
		gtk_combo_set_popdown_strings (GTK_COMBO (gtk_object_get_data
			(GTK_OBJECT (widget), "filter_combo")), g_list_first (filter_list));
		for (i = 0; i < n; i++)
			g_free (dest[i][0]);
		g_free (dest);
		if (in_list)
			gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (gtk_object_get_data
				(GTK_OBJECT (widget), "filter_combo"))->entry),
				p.filter_dest);
		else
		{
			gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (gtk_object_get_data
				(GTK_OBJECT (widget), "filter_combo"))->entry),
				gettext ("(All Printers)"));
			p_preferences_change_action (widget);
		}
	}
	g_list_free (filter_list);
	p_start_polling (clist);
}

/*
 * p_g_compare ()
 */

gint p_g_compare (gconstpointer p_1, gconstpointer p_2)
{
	return (strcmp ((gchar *) p_1, (gchar *) p_2));
}

/*
 * p_get_host ()
 */

void p_get_host (GtkEntry *entry)
{
	gtk_entry_set_text (entry, cupsServer ());
}

/*
 * p_get_port ()
 */

void p_get_port (GtkSpinButton *spinbutton)
{
	gtk_spin_button_set_value (spinbutton, (gfloat) ippPort ());
}

/*
 * p_get_refresh ()
 */

void p_get_refresh (GtkSpinButton *spinbutton)
{
	gtk_spin_button_set_value (spinbutton, 1);
}

/*
 * p_get_change ()
 */

void p_get_change (GtkSpinButton *spinbutton)
{
	gtk_spin_button_set_value (spinbutton, 10);
}

#ifdef ALIEN_USER
/*
 * p_get_user ()
 */

void p_get_user (GtkEntry *entry)
{
	gtk_entry_set_text (entry, cupsUser ());
}
#endif /*ALIEN_USER*/

#ifdef HAVE_LIBGNUTLS
/*
 * p_get_ssl_buttons ()
 */

void p_get_ssl_buttons (GtkWidget *widget)
{
	http_encryption_t ssl;
	
	if (GPOINTER_TO_INT (gtk_object_get_data
		(GTK_OBJECT (widget), "init")) == TRUE)
	{
		gtk_object_set_data (GTK_OBJECT (widget), "init", GINT_TO_POINTER (FALSE));
		ssl = p.ssl;
	}
	else
		ssl = cupsEncryption ();

	if (ssl == HTTP_ENCRYPT_IF_REQUESTED)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON
		(gtk_object_get_data (GTK_OBJECT (widget), "if_requested_button")), TRUE);
	else if (ssl == HTTP_ENCRYPT_NEVER)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON
		(gtk_object_get_data (GTK_OBJECT (widget), "never_button")), TRUE);
	else if (ssl == HTTP_ENCRYPT_REQUIRED)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON
		(gtk_object_get_data (GTK_OBJECT (widget), "required_button")), TRUE);
	else
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON
		(gtk_object_get_data (GTK_OBJECT (widget), "always_button")), TRUE);
}

/*
 * p_get_ssl_value ()
 */

gint p_get_ssl_value (GtkWidget *widget)
{
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
		(gtk_object_get_data (GTK_OBJECT (widget), "if_requested_button"))))
		return HTTP_ENCRYPT_IF_REQUESTED;
	else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
		(gtk_object_get_data (GTK_OBJECT (widget), "never_button"))))
		return HTTP_ENCRYPT_NEVER;
	else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
		(gtk_object_get_data (GTK_OBJECT (widget), "required_button"))))
		return HTTP_ENCRYPT_REQUIRED;
	else
		return HTTP_ENCRYPT_ALWAYS;
}
#endif /*HAVE_LIBGNUTLS*/

/*
 * p_password_callback ()
 */

const gchar *p_password_callback (const gchar *prompt)
{
	gint i;

	if (password != NULL)
	{
		for (i = 0; i < strlen (password); i++)
			password[i] = 0;
		g_free (password);
	}

	p_show_password_dialog ();
	gtk_main ();
	return password;
}

/*
 * p_show_password_dialog ()
 */

void p_show_password_dialog (void)
{
	GtkWidget *dialog;
	
	gchar *tmp;

	GtkWidget *vbox;
	
	GtkWidget *entry_frame;
	GtkWidget *entry_vbox;
	GtkWidget *entry;
	
	GtkWidget *action_area;
	
	GtkWidget *cancel_button;

	GtkWidget *ok_button;
	
	GtkWidget *tmp_icon;
	GdkPixmap *pixmap;
	GdkBitmap *bitmap;
	
	/*
	 * dialog
	 */

	dialog = gtk_dialog_new ();
	gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
	gtk_window_set_title (GTK_WINDOW (dialog), gettext("Password"));
	gtk_window_set_policy (GTK_WINDOW (dialog), FALSE, FALSE, TRUE);
	gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER);
	gtk_window_set_transient_for (GTK_WINDOW (dialog), GTK_WINDOW (transient));

	/*
	 * vbox
	 */

	vbox = GTK_DIALOG (dialog)->vbox;
	gtk_object_set_data (GTK_OBJECT (dialog), "vbox", vbox);
	
	/*
	 * entry_frame
	 */

	tmp = g_strdup_printf ("%s %s %s@%s", gettext ("Password"),
		gettext("for"), p.user, p.host);
	entry_frame = gtk_frame_new (tmp);
	g_free (tmp);
	gtk_widget_ref (entry_frame);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "entry_frame",
		entry_frame, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (vbox), entry_frame, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (entry_frame), 5);

	/*
	 * entry_vbox
	 */

	entry_vbox = gtk_vbox_new (FALSE, 0);
	gtk_widget_ref (entry_vbox);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "entry_vbox",
		entry_vbox, (GtkDestroyNotify) gtk_widget_unref);
	gtk_container_add (GTK_CONTAINER (entry_frame), entry_vbox);
	gtk_container_set_border_width (GTK_CONTAINER (entry_vbox), 5);

	/*
	 * entry
	 */

	entry = gtk_entry_new ();
	gtk_widget_ref (entry);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "entry",
		entry, (GtkDestroyNotify) gtk_widget_unref);
	gtk_entry_set_max_length (GTK_ENTRY (entry), 32);
	gtk_box_pack_start (GTK_BOX (entry_vbox), entry, FALSE, FALSE, 0);
	gtk_entry_set_visibility (GTK_ENTRY (entry), FALSE);


	/*
	 * action_area
	 */

	action_area = GTK_DIALOG (dialog)->action_area;
	gtk_object_set_data (GTK_OBJECT (dialog), "action_area", action_area);
	gtk_container_set_border_width (GTK_CONTAINER (action_area), 10);

	/*
	 * cancel_button
	 */

	cancel_button = gtk_button_new_with_label (gettext("Cancel"));
	gtk_widget_ref (cancel_button);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "cancel_button",
		cancel_button, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (action_area), cancel_button, TRUE, TRUE, 0);

	/*
	 * ok_button
	 */

	ok_button = gtk_button_new_with_label (gettext("OK"));
	gtk_widget_ref (ok_button);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "ok_button",
		ok_button, (GtkDestroyNotify) gtk_widget_unref);
	gtk_box_pack_start (GTK_BOX (action_area), ok_button, TRUE, TRUE, 0);
	
	/*
	 * connect signal
	 */

	/* cancel_button - "clicked" */ 

	gtk_signal_connect_object (GTK_OBJECT (cancel_button), "clicked",
		GTK_SIGNAL_FUNC (p_skip_password), GTK_OBJECT (dialog)) ;

	/* ok_button - "clicked" */ 

	gtk_signal_connect_object (GTK_OBJECT (ok_button), "clicked",
		GTK_SIGNAL_FUNC (p_submit_password), GTK_OBJECT (dialog)) ;

	tmp_icon = create_pixmap (dialog, "pt.xpm");
	gtk_pixmap_get (GTK_PIXMAP (tmp_icon), &pixmap, &bitmap);
	
	gtk_widget_show_all (dialog);
	gdk_window_set_icon (dialog->window, NULL, pixmap, bitmap);
}

/*
 * p_skip_password ()
 */

void p_skip_password (GtkWidget *widget)
{
	password = NULL;

	gtk_widget_destroy (widget);
	gtk_main_quit ();
}

/*
 * p_submit_password ()
 */

void p_submit_password (GtkWidget *widget)
{
	password = g_strdup (gtk_entry_get_text (GTK_ENTRY
		(gtk_object_get_data (GTK_OBJECT (widget), "entry"))));

	gtk_widget_destroy (widget);
	gtk_main_quit ();
}
