/* 
 * main.c
 *
 * crafted - a pud editor for the freecraft project.
 * 
 * Copyright (C) 2001-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <gtk/gtk.h>

#include "load_pud.h"
#include "save_pud.h"
#include "crafted.h"
#include "tileset.h"
#include "tile.h"
#include "craftedrc.h"
#include "craftedfileselection.h"
#include "unit.h"
#include "crafted_png.h"
#include "new_pud.h"
#include "craftedintl.h"
#include "mini_map.h"
#include "file-open-dialog.h"
#include "cool_stuff.h"
#include "generate_map.h"
#include "generate_map2.h"
#include "init_pud.h"
#include "edit_pud.h"
#include "load_pud.h"

tPud Pud;

GtkWidget *drawing_area;
GtkWidget *hscroll, *vscroll;
gint x_offset = 0, y_offset = 0;

GdkPixmap *tileset_pixmap;

GtkWidget *cursor_label;

gint old_x = 0, old_y = 0;
pud_unit_t *old_unit = NULL;

gint current_tool;

GtkWidget *pixmap_tool[LAST_SOLID_TILE+1];
GdkPixmap *gdk_pixmap_tool[LAST_SOLID_TILE+1];
GtkWidget *version_label, *description_label;
GtkWidget *status_label;

GtkWidget *main_window;
GtkWidget *main_vbox;

guint verbosity;

guint unit_notebook_page;

gint current_unit_idx[3];
GtkWidget *unit_tool_radio = NULL;
gint orc_player = 0, human_player = 0;

static gboolean scrolling = FALSE;
static gint scroll_start_x = 0;
static gint scroll_start_y = 0;

GtkWidget *progressbar;

static gboolean load_tileset_and_png(int terrain);
static void drawing_area_redraw(gboolean draw_map);

/*************************************************************************
*  progressbar_init
**************************************************************************/
void progressbar_init(void)
{
  gtk_widget_set_sensitive(main_vbox, FALSE);
}

/*************************************************************************
*  progressbar_update
**************************************************************************/
void progressbar_update(gdouble percentage, gchar *text)
{
  gtk_progress_set_percentage(GTK_PROGRESS(progressbar), percentage);
  gtk_label_set_text(GTK_LABEL(status_label), text);
  while (gtk_events_pending()) gtk_main_iteration();
}

/*************************************************************************
*  progressbar_finish
**************************************************************************/
void progressbar_finish(void)
{
  gtk_widget_set_sensitive(main_vbox, TRUE);
  gtk_progress_set_percentage(GTK_PROGRESS(progressbar), 0.0);
  gtk_label_set_text(GTK_LABEL(status_label), "");
}

/*************************************************************************
*  update_window_title
**************************************************************************/
static void update_window_title(void)
{
  gchar *tmp_title;

  tmp_title = g_strdup_printf("%s%s%s", Pud.path, Pud.modified? _(" (modified)"):"",
                              Pud.readonly?" (-ro-)":"");
  gtk_window_set_title(GTK_WINDOW(main_window), tmp_title);
  g_free(tmp_title);
}

/*************************************************************************
*  exit_editor
**************************************************************************/
static void exit_editor(void)
{
  if (Pud.modified)
  {
    GtkWidget *dialog, *label, *button;

    dialog = gtk_dialog_new();
    gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_MOUSE);
    label = gtk_label_new(_("This level isn't saved.\nDo you really want to quit now?"));
    gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), label);
    button = gtk_button_new_with_label(_(" Yes "));
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
                       (GtkSignalFunc)gtk_main_quit, NULL);
    gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->action_area), button);
    button = gtk_button_new_with_label(_(" No "));
    gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                              (GtkSignalFunc)gtk_widget_destroy,
                              GTK_OBJECT(dialog));
    gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->action_area), button);
    gtk_widget_show_all(dialog);
/*    gtk_main(); FIXME: will be modal*/
  } else
    gtk_main_quit();
}

/*************************************************************************
*  load_tileset_and_png
**************************************************************************/
static gboolean load_tileset_and_png(int terrain)
{
  Graphic *graphic;
  gchar   *filename;
  char    *terrainname;
  int      i;

  static char* pathes[] =
  {
    "graphic/tileset",
    "graphics/tilesets/summer/terrain",
    "graphics/tilesets/winter/terrain",
    "graphics/tilesets/wasteland/terrain",
    "graphics/tilesets/swamp/terrain",
    NULL
  };

  if (verbosity)
    g_print("terrain: %d\n", terrain);
  switch (terrain)
  {
    case 0:
      terrainname = "summer";
      break;

    case 1:
      terrainname = "winter";
      break;

    case 2:
      terrainname = "wasteland";
      break;

    case 3:
      terrainname = "swamp";
      break;

    default:
      terrainname = "summer";
      break;
  }

  /* Johns: automatic path detection, for all freecraft versions */
  for( i=0; pathes[i]; ++i )
  {
    filename = g_strdup_printf("%s/%s/%s.png", data_path, pathes[i], terrainname);

    if (!access(filename, F_OK))
    {
      graphic = LoadGraphicPNG(filename);
      g_free(filename);
      if (graphic)
      {
        tileset_pixmap = graphic->pixmap;
        gdk_pixmap_unref(graphic->mask);
        g_free(graphic);
        load_tileset(terrain);
        return TRUE;
      }
    } else
      g_free(filename);
  }
  g_message("cannot load tileset\n");
  return FALSE;
}

/*************************************************************************
*  save_pud_callback
**************************************************************************/
static void save_pud_callback(void)
{
  GtkWidget *file_selector;

  file_selector = gtk_file_selection_new(_("Save Pud File"));
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(file_selector)->ok_button),
                            "clicked", (GtkSignalFunc)save_pud,
                            GTK_OBJECT(file_selector));
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(file_selector)->cancel_button),
                            "clicked", (GtkSignalFunc)gtk_widget_destroy,
                            GTK_OBJECT(file_selector));
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(file_selector)->ok_button),
                            "clicked", (GtkSignalFunc)gtk_widget_destroy,
                            GTK_OBJECT(file_selector));
  gtk_widget_show(file_selector);
}

/*************************************************************************
*  pud_editor_init
**************************************************************************/
static void pud_editor_init(void)
{
  Pud.path        = NULL;
  Pud.is_loaded   = FALSE;
  Pud.readonly    = FALSE;
  Pud.modified    = FALSE;
  Pud.version     = 1;
  Pud.sub_version = 1;
  Pud.description = NULL;
  Pud.terrain     = 0;
  Pud.width       = 32;
  Pud.height      = 32;
  Pud.fields      = NULL;
  Pud.units       = NULL;
  data_path       = g_strdup_printf("/usr/local/data");
  parse_rc();
}

/*************************************************************************
*  update_units_toggles
**************************************************************************/
static void update_units_toggles(void)
{
  gint i;

  GtkWidget **human_unit_toggle, **orc_unit_toggle;
  
  human_unit_toggle = gtk_object_get_data(GTK_OBJECT(main_window),
                                          "human_unit_toggle");
  orc_unit_toggle = gtk_object_get_data(GTK_OBJECT(main_window),
                                        "orc_unit_toggle");
  for (i=0 ; i<15 ; i++)
  {
    switch (Pud.player[i].side)
    {
      case 0: /* Human */
        gtk_widget_set_sensitive(human_unit_toggle[i], TRUE);
        gtk_widget_set_sensitive(orc_unit_toggle[i], FALSE);
        break;

      case 1: /* Orc */
        gtk_widget_set_sensitive(human_unit_toggle[i], FALSE);
        gtk_widget_set_sensitive(orc_unit_toggle[i], TRUE);
        break;

      case 2: /* Neutral */
        gtk_widget_set_sensitive(human_unit_toggle[i], FALSE);
        gtk_widget_set_sensitive(orc_unit_toggle[i], FALSE);
        break;
    }
  }
}

/*************************************************************************
*  after_load
**************************************************************************/
void after_load(void)
{
  gint           x, y;
  gint           index;
  gchar         *str;
  GList         *tmp_list;
  GtkAdjustment *adj;

  load_tileset_and_png(Pud.terrain);
  /* Update the labels */
  str = g_strdup_printf("%d.%d", Pud.version, Pud.sub_version);
  gtk_label_set_text(GTK_LABEL(version_label), str);
  g_free(str);
  gtk_label_set_text(GTK_LABEL(description_label), Pud.description);
  /* Update the title bar */
  update_window_title();

  /* Update scrollbars */
  adj = gtk_range_get_adjustment(GTK_RANGE(hscroll));
  adj->upper = 32 * Pud.width;
  adj->page_size = drawing_area->allocation.width;
  gtk_range_set_adjustment(GTK_RANGE(hscroll), adj);
  gtk_range_slider_update(GTK_RANGE(hscroll));

  adj = gtk_range_get_adjustment(GTK_RANGE(vscroll));
  adj->upper = 32 * Pud.height;
  adj->page_size = drawing_area->allocation.height;
  gtk_range_set_adjustment(GTK_RANGE(vscroll), adj);
  gtk_range_slider_update(GTK_RANGE(vscroll));

  drawing_area_redraw(TRUE);
  update_units_toggles();
}

/*************************************************************************
*  ask_for_data_path_ok
**************************************************************************/
static void ask_for_data_path_ok(GtkWidget *cfs)
{
  if (data_path) g_free(data_path);
  data_path = g_strdup(crafted_file_selection_get_filename(CRAFTED_FILE_SELECTION(cfs)));
  gtk_widget_destroy(gtk_widget_get_toplevel(cfs));
}

/*************************************************************************
*  ask_for_data_path_save
**************************************************************************/
static void ask_for_data_path_save(GtkWidget *cfs)
{
  FILE *fd;
  gchar *filename;
  if (data_path) g_free(data_path);
  data_path = g_strdup(crafted_file_selection_get_filename(CRAFTED_FILE_SELECTION(cfs)));
  gtk_widget_destroy(gtk_widget_get_toplevel(cfs));

  filename = g_strdup_printf("%s/.craftedrc", g_get_home_dir());
  fd = fopen(filename, "w");
  g_free(filename);

  if (fd)
  {
    fprintf(fd, "# Next line define the path to the FreeCraft data\n");
    fprintf(fd, "data_path = %s\n", data_path);
    fclose(fd);
  }
}

/*************************************************************************
*  ask_for_data_path
**************************************************************************/
static void ask_for_data_path(void)
{
  GtkWidget *dialog, *button, *label, *cfs;

  dialog = gtk_dialog_new();
  gtk_signal_connect(GTK_OBJECT(dialog), "destroy",
                     (GtkSignalFunc)gtk_main_quit, NULL);

  label = gtk_label_new(_("Please indicate where your data files are:"));
  gtk_misc_set_padding(GTK_MISC(label), 10, 5);
  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), label);
  
  cfs = crafted_file_selection_new("data path",
                                   data_path,
                                   TRUE, TRUE);
  gtk_container_set_border_width(GTK_CONTAINER(cfs), 5);
  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), cfs);
  
  button = gtk_button_new_with_label(_("OK"));
  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->action_area), button);
  gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                            (GtkSignalFunc)ask_for_data_path_ok,
                            GTK_OBJECT(cfs));
  
  button = gtk_button_new_with_label(_("Save"));
  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->action_area), button);
  gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                            (GtkSignalFunc)ask_for_data_path_save,
                            GTK_OBJECT(cfs));
  
  button = gtk_button_new_with_label(_("Quit"));
  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->action_area), button);
  gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                            (GtkSignalFunc)_exit,
                            (GtkObject*)-1);
  
/*  gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);*/
  gtk_widget_show_all(dialog);
  
  gtk_main();
}

/*************************************************************************
*  drawing_area_mouse_move_callback
**************************************************************************/
static gboolean drawing_area_mouse_move_callback(GtkWidget *area,
                                                 GdkEventMotion *ev)
{
  gchar         *cursor_text;
  gint           x, y;
  GList         *tmp_list;
  gchar         *status;
  gint           i, j; 
  GtkAdjustment *adj, *adjx, *adjy;

  if (!Pud.is_loaded)
    return TRUE;

  x = (ev->x+x_offset)/32;
  y = (ev->y+y_offset)/32;

  if (x == old_x && y == old_y)
    return TRUE;


  if ((ev->state & GDK_BUTTON2_MASK) && (scrolling))
  { /* handle middle button scrolling */
    adj = gtk_range_get_adjustment(GTK_RANGE(hscroll));
    if (scroll_start_x - (gint)(ev->x) <= adj->upper - adj->page_size)
      gtk_adjustment_set_value(adj, scroll_start_x - (gint)(ev->x));
    adj = gtk_range_get_adjustment(GTK_RANGE(vscroll));
    if (scroll_start_y - (gint)(ev->y) <= adj->upper - adj->page_size)
      gtk_adjustment_set_value(adj, scroll_start_y - (gint)(ev->y));
  }
  
  if ((ev->state & GDK_BUTTON1_MASK) && (current_tool < LAST_SOLID_TILE))
  {
    int idx, n, xi, yi, index;
    idx = get_solid_index(current_tool);
  
    index = solid_tiles[current_tool][idx];

    n = 0;
    if (ev->state & GDK_SHIFT_MASK)   n += 1;
    if (ev->state & GDK_CONTROL_MASK) n += 2;
    if (ev->state & GDK_MOD1_MASK)    n += 4;
      
    for (xi = -n ; xi <= n ; ++xi)
    {
      for (yi = -n ; yi <= n ; ++yi)
      {
        if ((x+xi>=0) && (y+yi>=0) && (x+xi<Pud.width) && (y+yi<Pud.height))
        {
          affect_exact_tile(x + xi, y + yi, (current_tool << 4)+idx);
          check_around(x + xi , y + yi, current_tool);
        }
      }
    }
    gdk_draw_rectangle(area->window, area->style->white_gc, FALSE,
                       x * 32-x_offset, y * 32-y_offset, 31, 31);
    /* handle automatic scrolling :) */
    adjx = gtk_range_get_adjustment(GTK_RANGE(hscroll));
    adjy = gtk_range_get_adjustment(GTK_RANGE(vscroll));
    if (ev->x < 0 || ev->y < 0 ||
        ev->x > adjx->page_size ||
        ev->y > adjy->page_size)
    {
      gint off_x, off_y;

      off_x = off_y = 0;
      /*  The cases for scrolling  */
      if (ev->x < 0)
        off_x = ev->x;
      else if (ev->x > adjx->page_size)
        off_x = ev->x - adjx->page_size;
      if (x_offset + off_x <= adjx->upper - adjx->page_size)
        gtk_adjustment_set_value(adjx, x_offset + off_x);

      if (ev->y < 0)
        off_y = ev->y;
      else if (ev->y > adjy->page_size)
        off_y = ev->y - adjy->page_size;
      if (y_offset + off_y <= adjy->upper - adjy->page_size)
        gtk_adjustment_set_value(adjy, y_offset + off_y);
//      gdk_event_put((GdkEvent *) ev);
    }
  }
 
  if (old_unit)
  {
    UnitType unit_type;

    unit_type = default_unit_types[old_unit->type];
    /* redraw the map underneath */
    for (i=0 ; i < unit_type.tile_width ; i++)
      for (j=0 ; j < unit_type.tile_height ; j++)
      {
        gushort tile;
        gint xx, yy, index;

        xx = old_unit->x+i;
        yy = old_unit->y+j;
        tile = Pud.fields[yy*Pud.width+xx];
        if (tile & 0xFF00)
        { /* mixed tiles */
          index = mixed_tiles[(tile & 0xFF00) >> 8][tile & 0xFF];
        } else
        { /* solid tiles */
          index = solid_tiles[(tile & 0x00F0) >> 4][tile & 0xF];
        }
        gdk_window_copy_area(area->window,
                             area->style->white_gc,
                             xx*32-x_offset, yy*32-y_offset,
                             tileset_pixmap,
                             (index % 16)*32, (index / 16)*32,
                             32,32);
      }
    i = old_unit->type;
    if (i<nb_unit_types)
    {
      GdkGC *gc;
      gint xx, yy;

      if (default_unit_types[i].graphic)
      {
        gc = gdk_gc_new(area->window);
        xx = old_unit->x*32-(default_unit_types[i].width -32*default_unit_types[i].tile_width)/2-x_offset;
        yy = old_unit->y*32-(default_unit_types[i].height-32*default_unit_types[i].tile_height)/2-y_offset;
        gdk_gc_set_clip_mask(gc, default_unit_types[i].graphic[(old_unit->player_number)%8]->mask);
        gdk_gc_set_clip_origin(gc, xx, yy);
        gdk_window_copy_area(area->window, gc, xx, yy,
                             default_unit_types[i].graphic[(old_unit->player_number)%8]->pixmap,
                             0, 0, 
                             default_unit_types[i].width,
                             default_unit_types[i].height);
        gdk_gc_destroy(gc);
      } else if (verbosity) g_print("wrong unit type: %d\n", i);
    }
  } else
  {
    guint16 tile;
    gint index;

    tile = Pud.fields[old_y*Pud.width+old_x];
    if (tile & 0xFF00)
    { /* mixed tiles */
      index = mixed_tiles[(tile & 0xFF00) >> 8][tile & 0xFF];
    } else
    { /* solid tiles */
      index = solid_tiles[(tile & 0x00F0) >> 4][tile & 0xF];
    }
    gdk_window_copy_area(area->window,
                         area->style->white_gc,
                         old_x*32-x_offset, old_y*32-y_offset, 
                         tileset_pixmap,
                         (index % 16)*32,
                         (index / 16)*32, 32,32);
  } 
  cursor_text = g_strdup_printf(" %d x %d ", x, y);
  gtk_label_set_text(GTK_LABEL(cursor_label), cursor_text);
  g_free(cursor_text);
  /* redraw the units */
  for (tmp_list = Pud.units ; tmp_list ; tmp_list = g_list_next(tmp_list))
  {
    pud_unit_t *pud_unit;

    pud_unit = tmp_list->data;
    i = pud_unit->type;
    if ((abs(old_x-pud_unit->x) < 6) &&
        (abs(old_y-pud_unit->y) < 6) &&
        (i<nb_unit_types) )
    {
      GdkGC *gc;
      gint x, y;

      if (default_unit_types[i].graphic)
      {
        gc = gdk_gc_new(area->window);
        x = pud_unit->x*32-(default_unit_types[i].width -32*default_unit_types[i].tile_width)/2-x_offset;
        y = pud_unit->y*32-(default_unit_types[i].height-32*default_unit_types[i].tile_height)/2-y_offset;
        gdk_gc_set_clip_mask(gc, default_unit_types[i].graphic[(pud_unit->player_number)%8]->mask);
        gdk_gc_set_clip_origin(gc, x, y);
        gdk_window_copy_area(area->window, gc, x, y,
                             default_unit_types[i].graphic[(pud_unit->player_number)%8]->pixmap,
                             0, 0, 
                             default_unit_types[i].width,
                             default_unit_types[i].height);
        gdk_gc_destroy(gc);
      } else if (verbosity) g_print("wrong unit type: %d\n", i);
    }
  }

  for (tmp_list = Pud.units ; tmp_list ; tmp_list = g_list_next(tmp_list))
  {
    pud_unit_t *pud_unit;
    gint type;

    pud_unit = tmp_list->data;
    type = pud_unit->type;
    if (type < nb_unit_types)
    {
      UnitType unit_type;

      unit_type = default_unit_types[type];
      if ((x >= pud_unit->x) && (x < pud_unit->x+unit_type.tile_width) &&
          (y >= pud_unit->y) && (y < pud_unit->y+unit_type.tile_height))
      {
        GdkGC *gc;
        GdkColor color;
        
        gc = gdk_gc_new(area->window);
        color.red   = 65535;
        color.green = 0;
        color.blue  = 0;
        gdk_color_alloc(gtk_widget_get_colormap(area), &color);
        gdk_gc_set_foreground(gc, &color);
        gdk_draw_rectangle(area->window, gc, FALSE,
                           pud_unit->x * 32-x_offset,
                           pud_unit->y * 32-y_offset,
                           unit_type.tile_width  * 32 - 1,
                           unit_type.tile_height * 32 - 1);
        gdk_gc_destroy(gc);
        old_x = x; old_y = y;
        old_unit = pud_unit;
        gtk_label_set_text(GTK_LABEL(status_label), unit_type.name);
        return TRUE;
      }
    }
  }
  
  gdk_draw_rectangle(area->window, area->style->white_gc, FALSE,
                     x * 32-x_offset, y * 32-y_offset, 31, 31);
  old_unit = NULL;
  old_x = x; old_y = y;
  return TRUE;
}

/*************************************************************************
*  exact_tile_callback
**************************************************************************/
static void exact_tile_callback(GtkWidget *button, guint16 tile)
{
  gint x, y;

  if (!Pud.modified)
  {
    Pud.modified = TRUE;
    update_window_title();
  }
  x = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(button), "x"));
  y = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(button), "y"));
  affect_exact_tile(x, y, tile);
  gtk_widget_destroy(gtk_widget_get_toplevel(button));
}

/*************************************************************************
*  delete_unit
**************************************************************************/
static void delete_unit(GtkWidget *menu_item, pud_unit_t *unit)
{
  UnitType unit_type;
  gint i, j;

  unit_type = default_unit_types[unit->type];

  /* remove the setting dialog, if any */
  if (unit->setting_dialog)
    gtk_widget_destroy(unit->setting_dialog);
  /* remove from the list */
  Pud.units = g_list_remove(Pud.units, unit);
  /* redraw the map underneath */
  for (i=0 ; i < unit_type.tile_width ; i++)
    for (j=0 ; j < unit_type.tile_height ; j++)
    {
      gushort tile;
      gint x, y, index;

      x = unit->x+i;
      y = unit->y+j;
      tile = Pud.fields[y*Pud.width+x];
      if (tile & 0xFF00)
      { /* mixed tiles */
        index = mixed_tiles[(tile & 0xFF00) >> 8][tile & 0xFF];
      } else
      { /* solid tiles */
        index = solid_tiles[(tile & 0x00F0) >> 4][tile & 0xF];
      }
      gdk_window_copy_area(drawing_area->window,
                           drawing_area->style->white_gc,
                           x*32-x_offset, y*32-y_offset, tileset_pixmap,
                           (index % 16)*32,
                           (index / 16)*32, 32,32);
    }
}

/*************************************************************************
*  pud_unit_update_value
**************************************************************************/
static void pud_unit_update_value(GtkAdjustment *adj, pud_unit_t *pud_unit)
{
  pud_unit->value = adj->value / 2500;
}

/*************************************************************************
*  show_gold_mine_setting_dialog
* XXX: move me
**************************************************************************/
static void show_ressource_setting_dialog(pud_unit_t *pud_unit)
{
  if (pud_unit->setting_dialog)
  {
    if (!GTK_WIDGET_VISIBLE(pud_unit->setting_dialog))
      gtk_widget_show(pud_unit->setting_dialog);
    else
      gdk_window_raise(pud_unit->setting_dialog->window);
  } else
  {
    GtkWidget *button, *alignment, *hbox, *label = NULL;
    GtkWidget *spin;
    GtkAdjustment *adj;

    pud_unit->setting_dialog = gtk_dialog_new();
    switch (pud_unit->type)
    {
      case 92:
        gtk_window_set_title(GTK_WINDOW(pud_unit->setting_dialog),
                             "Gold amount setting");
        break;
      case 93:
        gtk_window_set_title(GTK_WINDOW(pud_unit->setting_dialog),
                             "Oil amount setting");
        break;
    }
    gtk_window_set_wmclass(GTK_WINDOW(pud_unit->setting_dialog), "Crafted", "Crafted");
    gtk_window_set_position(GTK_WINDOW(pud_unit->setting_dialog), GTK_WIN_POS_MOUSE);
    gtk_window_set_policy(GTK_WINDOW(pud_unit->setting_dialog), TRUE, TRUE, TRUE);

    gtk_signal_connect(GTK_OBJECT(pud_unit->setting_dialog), "destroy",
                       (GtkSignalFunc)gtk_widget_destroyed,
                       &(pud_unit->setting_dialog));
    gtk_signal_connect(GTK_OBJECT(pud_unit->setting_dialog), "delete_event",
                       (GtkSignalFunc)gtk_widget_destroy, NULL);
    button = gtk_button_new_with_label(_("Close"));
    gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(pud_unit->setting_dialog)->action_area), button);
    gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                              (GtkSignalFunc)gtk_widget_destroy,
                              GTK_OBJECT(pud_unit->setting_dialog));

    alignment = gtk_alignment_new(0.5, 0.5, 0.8, 0.8);
    gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(pud_unit->setting_dialog)->vbox), alignment);

    hbox = gtk_hbox_new(FALSE, 4);
    gtk_container_set_border_width(GTK_CONTAINER(hbox), 6);
    gtk_container_add(GTK_CONTAINER(alignment), hbox);

    switch (pud_unit->type)
    {
      case 92:
        label = gtk_label_new(_("Gold: "));
        break;
      case 93:
        label = gtk_label_new(_("Oil: "));
        break;
    }
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 4);

    adj = (GtkAdjustment *)gtk_adjustment_new(pud_unit->value*2500, 2500, 300000, 2500, 2500, 2500);
    gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
                       (GtkSignalFunc)pud_unit_update_value, pud_unit);
    spin = gtk_spin_button_new(adj, 1, 0);
    gtk_widget_set_usize(spin, 100, -1);
    gtk_box_pack_start_defaults(GTK_BOX(hbox), spin);

    gtk_widget_show_all(pud_unit->setting_dialog);
  }
}

/*************************************************************************
*  drawing_area_button_press_callback
**************************************************************************/
static gboolean drawing_area_button_press_callback(GtkWidget *area,
                                                   GdkEventButton *ev)
{
  gint   x, y, idx, index;
  GList *tmp_list;
  gint   xi, yi, n;

  if (!Pud.is_loaded)
    return TRUE;
  x = (ev->x+x_offset)/32;
  y = (ev->y+y_offset)/32;

  switch (ev->button)
  {
    case 1: /* left button */
      /* first, test if we click on an unit */
      for (tmp_list = Pud.units ; tmp_list ; tmp_list = g_list_next(tmp_list))
      {
        UnitType    unit_type;
        pud_unit_t *pud_unit;

        pud_unit = tmp_list->data;
        unit_type = default_unit_types[pud_unit->type];
        if ((x >= pud_unit->x) && (x < pud_unit->x+unit_type.tile_width) &&
            (y >= pud_unit->y) && (y < pud_unit->y+unit_type.tile_height))
        {
          gdk_beep();
          return TRUE;
        }
      }
      if (current_tool < LAST_SOLID_TILE)
      {
        idx = get_solid_index(current_tool);
  
        index = solid_tiles[current_tool][idx];
 
        n = 0;
        if (ev->state & GDK_SHIFT_MASK)   n += 1;
        if (ev->state & GDK_CONTROL_MASK) n += 2;
        if (ev->state & GDK_MOD1_MASK)    n += 4;
      
        for (xi = -n ; xi <= n ; ++xi)
        {
          for (yi = -n ; yi <= n ; ++yi)
          {
            if ((x+xi>=0) && (y+yi>=0) && (x+xi<Pud.width) && (y+yi<Pud.height))
            {
              affect_exact_tile(x + xi, y + yi, (current_tool << 4)+idx);
              check_around(x + xi , y + yi, current_tool);
            }
          }
        }
        gdk_draw_rectangle(area->window, area->style->white_gc, FALSE,
                           x * 32-x_offset, y * 32-y_offset, 31, 31);
      } else
      { /* Unit for now */
        gint idx = current_unit_idx[unit_notebook_page];
        pud_unit_t *new_unit;

        new_unit = g_new(pud_unit_t, 1);
        new_unit->x = x;
        new_unit->y = y;
        new_unit->type = idx;
        switch (unit_notebook_page)
        {
          case 0:
            new_unit->player_number = 15;
            break;
          case 1:
            new_unit->player_number = human_player;
            break;
          case 2:
            new_unit->player_number = orc_player;
        }
        new_unit->value = 12;
        new_unit->setting_dialog = NULL;

        Pud.units = g_list_append(Pud.units, new_unit);
        if (default_unit_types[idx].graphic)
        {
          GdkGC *gc;
          gint   x, y;

          gc = gdk_gc_new(area->window);
          x = new_unit->x*32-(default_unit_types[idx].width -32*default_unit_types[idx].tile_width)/2-x_offset;
          y = new_unit->y*32-(default_unit_types[idx].height-32*default_unit_types[idx].tile_height)/2-y_offset;
          gdk_gc_set_clip_mask(gc, default_unit_types[idx].graphic[(new_unit->player_number)%8]->mask);
          gdk_gc_set_clip_origin(gc, x, y);
          gdk_window_copy_area(area->window, gc, x, y,
                               default_unit_types[idx].graphic[(new_unit->player_number)%8]->pixmap,
                               0, 0, 
                               default_unit_types[idx].width,
                               default_unit_types[idx].height);
          gdk_gc_destroy(gc);
        }
      }
      if (!Pud.modified)
      {
        Pud.modified = TRUE;
        update_window_title();
      }
      break;

    case 2: /* middle button */
      scrolling = TRUE;

      scroll_start_x = x_offset + (gint)(ev->x);
      scroll_start_y = y_offset + (gint)(ev->y);
      gtk_grab_add(area);

      {
        GdkCursor *cursor;

        cursor = gdk_cursor_new(GDK_FLEUR);
        gdk_window_set_cursor(area->window, cursor);
        gdk_cursor_destroy(cursor);
      }
      break;

    case 3: /* right button */
      { /* right button: popup */
        GtkWidget *pop_window, *toolbar, *icon;
        gint i;
        guint16 tile;

        for (tmp_list = Pud.units ; tmp_list ; tmp_list = g_list_next(tmp_list))
        {
          UnitType unit_type;
          pud_unit_t *pud_unit;

          pud_unit = tmp_list->data;
          unit_type = default_unit_types[pud_unit->type];
          if ((x >= pud_unit->x) && (x < pud_unit->x+unit_type.tile_width) &&
              (y >= pud_unit->y) && (y < pud_unit->y+unit_type.tile_height))
          {
            GtkWidget *menu, *menu_item;
           
            menu = gtk_menu_new();
            menu_item = gtk_menu_item_new_with_label(_("Delete"));
            gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
                               (GtkSignalFunc)delete_unit, pud_unit);
            gtk_menu_append(GTK_MENU(menu), menu_item);
            gtk_widget_show(menu_item);
            switch (pud_unit->type)
            {
              case 92: /* GoldMine */
              case 93: /* Oil patch */
                menu_item = gtk_menu_item_new_with_label(_("Set Amount"));
                gtk_signal_connect_object(GTK_OBJECT(menu_item), "activate",
                                          (GtkSignalFunc)show_ressource_setting_dialog,
                                          (GtkObject*)pud_unit);
                gtk_menu_append(GTK_MENU(menu), menu_item);
                gtk_widget_show(menu_item);
                break;
            }
            gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, ev->button, ev->time);

            return TRUE;
          }
        }
        tile = Pud.fields[y*Pud.width+x];
        pop_window = gtk_window_new(GTK_WINDOW_POPUP);
        gtk_window_set_position(GTK_WINDOW(pop_window), GTK_WIN_POS_MOUSE);
        gtk_window_set_transient_for(GTK_WINDOW(pop_window), GTK_WINDOW(main_window));
        gtk_window_set_modal(GTK_WINDOW(pop_window), TRUE);
        toolbar = gtk_toolbar_new(GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_ICONS);
        gtk_toolbar_set_space_style(GTK_TOOLBAR(toolbar), GTK_TOOLBAR_SPACE_LINE);
        gtk_toolbar_set_space_size(GTK_TOOLBAR(toolbar), 15);
        gtk_container_add(GTK_CONTAINER(pop_window), toolbar);
        for (i=0 ; i<16 ; i++)
        {
          gint index;
          if (tile & 0xFF00)
          { /* mixed */
            index = mixed_tiles[tile >> 8][(tile & 0xF0)+i];
          } else
          { /* solid */
            index = solid_tiles[tile >> 4][i];
          }
          if (index)
          {
            GdkPixmap *gdk_pixmap;
            GtkWidget *gtk_pixmap;
            GtkWidget *wid;
       
            gdk_pixmap = gdk_pixmap_new(area->window, 32, 32, -1);
            gdk_window_copy_area(gdk_pixmap, area->style->white_gc,
                                 0, 0, tileset_pixmap,
                                 (index % 16)*32, (index / 16)*32, 32,32);
            gtk_pixmap = gtk_pixmap_new(gdk_pixmap, NULL);
            wid = gtk_toolbar_append_item(GTK_TOOLBAR(toolbar),
                                          NULL, NULL, NULL, gtk_pixmap,
                                          (GtkSignalFunc)exact_tile_callback,
                                          GINT_TO_POINTER((tile & 0xFFF0)+i ));
            gtk_object_set_data(GTK_OBJECT(wid), "x", GINT_TO_POINTER(x));
            gtk_object_set_data(GTK_OBJECT(wid), "y", GINT_TO_POINTER(y));
          } else
          {
            gint next_index;
            if (i == 15)
              break;
            if (tile & 0xFF00)
            { /* mixed */
              next_index = mixed_tiles[tile >> 8][(tile & 0xF0)+i+1];
            } else
            { /* solid */
              next_index = solid_tiles[tile >> 4][i+1];
            }
            if (next_index == 0)
              break;
            gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
          }
        }
        gtk_widget_show_all(pop_window);
      }
      break;
  }
  return TRUE;
}

/*************************************************************************
*  drawing_area_button_release_callback
**************************************************************************/
static gboolean drawing_area_button_release_callback(GtkWidget *area,
                                                     GdkEventButton *ev)
{
  gint   x, y, idx, index;
  GList *tmp_list;
  gint   xi, yi, n;

  if (!Pud.is_loaded)
    return TRUE;
  x = (ev->x+x_offset)/32;
  y = (ev->y+y_offset)/32;

  switch (ev->button)
  {
    case 1: /* left button */
      /* do nothing */
      break;

    case 2: /* middle button */
      /* stop scrolling */
      scrolling = FALSE;
      scroll_start_x = 0;
      scroll_start_y = 0;
      gtk_grab_remove(area);
      {
        GdkCursor *cursor;

        cursor = gdk_cursor_new(GDK_LEFT_PTR);
        gdk_window_set_cursor(area->window, cursor);
        gdk_cursor_destroy(cursor);
      }
      break;

    case 3: /* right button */
      /* do nothing */
      break;
  }
  update_minimap();
  return TRUE;
}

/*************************************************************************
*  drawing_area_configure
**************************************************************************/
static gboolean drawing_area_configure(GtkWidget         *drawing_area,
                                       GdkEventConfigure *event)
{
  GtkAdjustment *adj;

  adj = gtk_range_get_adjustment(GTK_RANGE(hscroll));
  adj->page_size = event->width;
  gtk_range_set_adjustment(GTK_RANGE(hscroll), adj);
  gtk_range_slider_update(GTK_RANGE(hscroll));

  adj = gtk_range_get_adjustment(GTK_RANGE(vscroll));
  adj->page_size = event->height;
  gtk_range_set_adjustment(GTK_RANGE(vscroll), adj);
  gtk_range_slider_update(GTK_RANGE(vscroll));
  return FALSE;
}

/*************************************************************************
*  drawing_area_redraw
**************************************************************************/
static void drawing_area_redraw(gboolean draw_tile)
{
  gint i, j;
  GList *tmp_list;

  if (!Pud.fields)
    return;
  
  /* redraw the map */
  if (draw_tile)
  {
    for (i=0 ; i < drawing_area->allocation.width/32 +2; i++)
      for (j=0 ; j < drawing_area->allocation.height/32 +2; j++)
      {
        gushort tile;
        gint x, y, index;
  
        x = x_offset/32+i;
        y = y_offset/32+j;
        tile = Pud.fields[y*Pud.width+x];
        if (tile & 0xFF00)
        { /* mixed tiles */
          index = mixed_tiles[(tile & 0xFF00) >> 8][tile & 0xFF];
        } else
        { /* solid tiles */
          index = solid_tiles[(tile & 0x00F0) >> 4][tile & 0xF];
        }
        gdk_window_copy_area(drawing_area->window,
                             drawing_area->style->white_gc,
                             x*32-x_offset, y*32-y_offset, tileset_pixmap,
                             (index % 16)*32,
                             (index / 16)*32, 32,32);
      }
  }
  /* draw units on top */
  for (tmp_list = Pud.units ; tmp_list ; tmp_list = g_list_next(tmp_list))
  {
    pud_unit_t *pud_unit;

    pud_unit = tmp_list->data;
    i = pud_unit->type;
    if (i<nb_unit_types)
    {
      GdkGC *gc;
      gint x, y;

      if (default_unit_types[i].graphic)
      {
        gc = gdk_gc_new(drawing_area->window);
        x = pud_unit->x*32-(default_unit_types[i].width -32*default_unit_types[i].tile_width)/2-x_offset;
        y = pud_unit->y*32-(default_unit_types[i].height-32*default_unit_types[i].tile_height)/2-y_offset;
        gdk_gc_set_clip_mask(gc, default_unit_types[i].graphic[(pud_unit->player_number)%8]->mask);
        gdk_gc_set_clip_origin(gc, x, y);
        gdk_window_copy_area(drawing_area->window, gc, x, y,
                             default_unit_types[i].graphic[(pud_unit->player_number)%8]->pixmap,
                             0, 0, 
                             default_unit_types[i].width,
                             default_unit_types[i].height);
        gdk_gc_destroy(gc);
      } else if (verbosity) g_print("wrong unit type: %d\n", i);
    }
  }
}

/*************************************************************************
*  drawing_area_expose
**************************************************************************/
static gboolean drawing_area_expose(GtkWidget      *drawing_area,
                                    GdkEventExpose *event)
{
  gint i, j;

  if (!Pud.fields)
    return FALSE;

  if (event->count)
    return FALSE;

  drawing_area_redraw(TRUE);
  return TRUE;
}

/*************************************************************************
*  drawing_area_events
**************************************************************************/
static gboolean drawing_area_events(GtkWidget *canvas,
                                    GdkEvent  *ev)
{
  switch (ev->type)
  {
    case GDK_MOTION_NOTIFY:
      return drawing_area_mouse_move_callback(canvas, (GdkEventMotion*)ev);

    case GDK_BUTTON_PRESS:
      return drawing_area_button_press_callback(canvas, (GdkEventButton*)ev);

    case GDK_BUTTON_RELEASE:
      return drawing_area_button_release_callback(canvas, (GdkEventButton*)ev);

    case GDK_CONFIGURE:
      return drawing_area_configure(canvas, (GdkEventConfigure*)ev);

    case GDK_EXPOSE:
     return drawing_area_expose(canvas, (GdkEventExpose*)ev);

    default:
     return FALSE;
  }
}

/*************************************************************************
*  drawing_area_hscroll
**************************************************************************/
static void drawing_area_hscroll(GtkAdjustment *adj)
{
  x_offset = adj->value;
  drawing_area_redraw(TRUE);
}

/*************************************************************************
*  drawing_area_vscroll
**************************************************************************/
static void drawing_area_vscroll(GtkAdjustment *adj)
{
  y_offset = adj->value;
  drawing_area_redraw(TRUE);
}

/*************************************************************************
*  select_tool
**************************************************************************/
static void select_tool(GtkButton *dum, gpointer data)
{
  current_tool = GPOINTER_TO_INT(data);
}

/*************************************************************************
*  unit_notebook_switch_page
**************************************************************************/
static void unit_notebook_switch_page(GtkNotebook     *notebook,
                                      GtkNotebookPage *page,
                                      guint            page_num)
{
  unit_notebook_page = page_num;
  if (unit_tool_radio)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(unit_tool_radio), TRUE);
}

/*************************************************************************
*  list_unit_select_child
**************************************************************************/
static void list_unit_select_child(GtkList   *list,
                                   GtkWidget *child,
                                   gpointer   data)
{
  gint type = GPOINTER_TO_INT(data);
  gint idx;

  idx = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(child), "idx"));
  current_unit_idx[type] = idx;
  if (unit_tool_radio)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(unit_tool_radio), TRUE);
}

/*************************************************************************
*  orc_unit_select_player
**************************************************************************/
static void orc_unit_select_player(GtkToggleButton *radio, gpointer data)
{
  if (gtk_toggle_button_get_active(radio))
  {
    orc_player = GPOINTER_TO_INT(data);
  }
}

/*************************************************************************
*  human_unit_select_player
**************************************************************************/
static void human_unit_select_player(GtkToggleButton *radio, gpointer data)
{
  if (gtk_toggle_button_get_active(radio))
  {
    human_player = GPOINTER_TO_INT(data);
  }
}

/*************************************************************************
* main
**************************************************************************/
int main(int argc, char *argv[])
{
  GtkWidget *vbox, *menu_bar;
  GtkWidget *main_hbox, *scr;
  GtkWidget *left_vbox;
  GtkWidget *general_unit_list, *human_unit_list, *orc_unit_list;
  GtkWidget *unit_vbox;
  GtkAccelGroup *accel_group;
  GtkItemFactory *factory;
  GtkWidget *toolbar, *first_radio = NULL, *radio;
  GtkWidget *statusarea, *status_hbox, *label_frame, *status_frame;
  gint i, idx;
  GtkWidget *label;
  GtkAdjustment *adj;
  GtkWidget *notebook;
  GtkWidget *table;
  GtkWidget *unit_table;
  GSList *group;
  gchar  *pud_to_load = NULL;
  GtkWidget **human_unit_toggle;
  GtkWidget **orc_unit_toggle;

  static GtkItemFactoryEntry main_menu[] =
  {
    { N_("/_File"),                     NULL, NULL,               0, "<Branch>" },
    { N_("/File/_Open Pud"),            NULL, file_open_callback, 0, "<Item>" },
    { N_("/File/_New Pud"),             NULL, new_pud_callback,   0, "<Item>" },
    { N_("/File/_Save Pud"),            NULL, save_pud_callback,  0, "<Item>" },
    { N_("/File/-"),                    NULL, NULL,               0, "<Separator>" },
    { N_("/File/_Quit"),                NULL, exit_editor,        0, "<Item>" },
    { N_("/_Cool Stuff"),               NULL, NULL,               0, "<Branch>" },
    { N_("/Cool Stuff/_Flip map"),      NULL, flip_map,           0, "<Item>" },
    { N_("/Cool Stuff/_Generate map"),  NULL, generate_map,       0, "<Item>" },
    { N_("/Cool Stuff/Generate map_2"), NULL, generate_map2,      0, "<Item>" },
    { N_("/Cool Stuff/_Edit players"),  NULL, edit_pud_callback,  0, "<Item>" },
    { N_("/_Window"),                   NULL, NULL,               0, "<Branch>" },
    { N_("/Window/_Mini Map"),          NULL, show_mini_map,      0, "<Item>" }
  };

  gchar *icon_names[] = { "", "light water", "dark water", "light coast", "dark coast",
                          "light grass", "dark grass", "forest", "rocks",
                          "human closed wall", "orc closed wall", "human open wall",
                          "orc open wall", "bridge", "road", "ford", "unused"};

  static guint nb_entries = sizeof(main_menu) / sizeof(GtkItemFactoryEntry);

  gtk_init(&argc, &argv);
  gdk_rgb_init();

  verbosity = 0;
  for (i=1 ; i<argc ; i++)
  {
    if (!strcmp(argv[i], "--help"))
    {
      g_print(_("%s: A pud editor for the freecraft project\n"
              "Usage: %s [OPTION]... [FILE]\n\n"
              "  --help       display this help and exit\n"
              "  --version    display version and exit\n"
              "  --verbose    increase the verbosity level\n"),
              argv[0], argv[0]);
      exit(0);
    } else if (!strcmp(argv[i], "--version"))
    {
      g_print(_("%s: A pud editor for the freecraft project\n"
              "  Version 0.1.3\n"), argv[0]);
      exit(0);
    } else if (!strcmp(argv[i], "--verbose"))
    {
      verbosity++;
    } else
    {
      pud_to_load = argv[i];
    }
  }   

  pud_editor_init();
  
  main_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_widget_set_usize(main_window, 750, 550);
  gtk_signal_connect(GTK_OBJECT(main_window), "delete_event",
                     (GtkSignalFunc)exit_editor, NULL);

  vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(main_window), vbox);

  main_vbox = gtk_vbox_new(FALSE, 0);
  gtk_box_pack_start_defaults(GTK_BOX(vbox), main_vbox);
  
  accel_group = gtk_accel_group_new();
  factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR, "<Main>", accel_group);
  gtk_item_factory_create_items(factory, nb_entries, main_menu, NULL);
  gtk_accel_group_attach(accel_group, GTK_OBJECT(main_window));
  menu_bar = gtk_item_factory_get_widget(factory, "<Main>");
  gtk_box_pack_start(GTK_BOX(main_vbox), menu_bar, FALSE, FALSE, 0);              

  toolbar = gtk_toolbar_new(GTK_ORIENTATION_HORIZONTAL,
                            GTK_TOOLBAR_ICONS);
  gtk_box_pack_start(GTK_BOX(main_vbox), toolbar, FALSE, FALSE, 0);
  
  main_hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start_defaults(GTK_BOX(main_vbox), main_hbox);

  left_vbox = gtk_vbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(main_hbox), left_vbox, FALSE, FALSE, 0);

  /* The pud version */
  label = gtk_label_new(_("Version"));
  gtk_box_pack_start(GTK_BOX(left_vbox), label, FALSE, FALSE, 0);
  version_label = gtk_label_new("");
  gtk_box_pack_start(GTK_BOX(left_vbox), version_label, FALSE, FALSE, 0);

  /* The Map description */
  label = gtk_label_new(_("Description"));
  gtk_box_pack_start(GTK_BOX(left_vbox), label, FALSE, FALSE, 0);
  description_label = gtk_label_new("");
  gtk_box_pack_start(GTK_BOX(left_vbox), description_label, FALSE, FALSE, 0);

  /* The units lists notebook on the left */
  notebook = gtk_notebook_new();
  unit_notebook_page = 0;
  gtk_signal_connect(GTK_OBJECT(notebook), "switch_page",
                     (GtkSignalFunc)unit_notebook_switch_page, NULL);
  gtk_box_pack_end_defaults(GTK_BOX(left_vbox), notebook);
  /* "General" units */
  scr = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scr),
                                 GTK_POLICY_NEVER,
                                 GTK_POLICY_ALWAYS);
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), scr,
                           gtk_label_new("general"));
  general_unit_list = gtk_list_new();
  gtk_signal_connect(GTK_OBJECT(general_unit_list), "select_child",
                     (GtkSignalFunc)list_unit_select_child, GINT_TO_POINTER(0));
  gtk_list_set_selection_mode(GTK_LIST(general_unit_list),
                              GTK_SELECTION_BROWSE);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scr), general_unit_list);
  /* "Human" units */
  unit_vbox = gtk_vbox_new(FALSE, 2);
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), unit_vbox,
                           gtk_label_new("Human"));

  unit_table = gtk_table_new(4, 4, TRUE);
  gtk_container_set_border_width(GTK_CONTAINER(unit_table), 3);
  gtk_box_pack_start(GTK_BOX(unit_vbox), unit_table, FALSE, FALSE, 0);

  group = NULL;
  human_unit_toggle = g_new(GtkWidget*, 15);
  gtk_object_set_data(GTK_OBJECT(main_window), "human_unit_toggle",
                      human_unit_toggle);
  for (i=0 ; i<15 ; i++)
  {
    gchar  *tmp_txt;

    tmp_txt = g_strdup_printf("%d", i+1);
    human_unit_toggle[i] = gtk_radio_button_new_with_label(group, tmp_txt);
    gtk_signal_connect(GTK_OBJECT(human_unit_toggle[i]), "toggled",
                       (GtkSignalFunc)human_unit_select_player,
                       GINT_TO_POINTER(i));
    group = gtk_radio_button_group(GTK_RADIO_BUTTON(human_unit_toggle[i]));
    g_free(tmp_txt);
    gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(human_unit_toggle[i]), FALSE);
    gtk_misc_set_alignment(GTK_MISC(GTK_BIN(human_unit_toggle[i])->child), 0.5, 0.5);
    gtk_table_attach_defaults(GTK_TABLE(unit_table), human_unit_toggle[i],
                              i%4, i%4+1, i/4, i/4+1);
  }
  
  scr = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scr),
                                 GTK_POLICY_NEVER,
                                 GTK_POLICY_ALWAYS);
  gtk_box_pack_start_defaults(GTK_BOX(unit_vbox), scr);
  human_unit_list = gtk_list_new();
  gtk_signal_connect(GTK_OBJECT(human_unit_list), "select_child",
                     (GtkSignalFunc)list_unit_select_child, GINT_TO_POINTER(1));
  gtk_list_set_selection_mode(GTK_LIST(human_unit_list),
                              GTK_SELECTION_BROWSE);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scr), human_unit_list);
  /* "Orc" units */
  unit_vbox = gtk_vbox_new(FALSE, 2);
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), unit_vbox,
                           gtk_label_new(_("Orc")));

  unit_table = gtk_table_new(4, 4, TRUE);
  gtk_container_set_border_width(GTK_CONTAINER(unit_table), 3);
  gtk_box_pack_start(GTK_BOX(unit_vbox), unit_table, FALSE, FALSE, 0);

  group = NULL;
  orc_unit_toggle = g_new(GtkWidget *,15);
  gtk_object_set_data(GTK_OBJECT(main_window), "orc_unit_toggle",
                      orc_unit_toggle);
  for (i=0 ; i<15 ; i++)
  {
    gchar *tmp_txt;

    tmp_txt = g_strdup_printf("%d", i+1);
    orc_unit_toggle[i] = gtk_radio_button_new_with_label(group, tmp_txt);
    gtk_signal_connect(GTK_OBJECT(orc_unit_toggle[i]), "toggled",
                       (GtkSignalFunc)orc_unit_select_player,
                       GINT_TO_POINTER(i));
    group = gtk_radio_button_group(GTK_RADIO_BUTTON(orc_unit_toggle[i]));
    g_free(tmp_txt);
    gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(orc_unit_toggle[i]), FALSE);
    gtk_misc_set_alignment(GTK_MISC(GTK_BIN(orc_unit_toggle[i])->child), 0.5, 0.5);
    gtk_table_attach_defaults(GTK_TABLE(unit_table), orc_unit_toggle[i],
                              i%4, i%4+1,  i/4, i/4+1);
  }
  scr = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scr),
                                 GTK_POLICY_NEVER,
                                 GTK_POLICY_ALWAYS);
  gtk_box_pack_start_defaults(GTK_BOX(unit_vbox), scr);
  orc_unit_list = gtk_list_new();
  gtk_signal_connect(GTK_OBJECT(orc_unit_list), "select_child",
                     (GtkSignalFunc)list_unit_select_child, GINT_TO_POINTER(2));
  gtk_list_set_selection_mode(GTK_LIST(orc_unit_list),
                              GTK_SELECTION_BROWSE);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scr), orc_unit_list);


  /* We cannot use a scrolled_window since it would use too much memory :-(,
   * so we have to emulate it... */
  table = gtk_table_new(2, 2, FALSE);
  gtk_box_pack_start_defaults(GTK_BOX(main_hbox), table);

  drawing_area = gtk_drawing_area_new();
  gtk_table_attach_defaults(GTK_TABLE(table), drawing_area, 0,1, 0,1);
  gtk_widget_set_events(drawing_area, GDK_EXPOSURE_MASK
                                   | GDK_LEAVE_NOTIFY_MASK
                                   | GDK_BUTTON_PRESS_MASK
                                   | GDK_BUTTON_RELEASE_MASK
                                   | GDK_POINTER_MOTION_MASK
                                   | GDK_KEY_PRESS_MASK);
  gtk_signal_connect(GTK_OBJECT(drawing_area), "event",
                     (GtkSignalFunc)drawing_area_events, NULL);

  adj = GTK_ADJUSTMENT(gtk_adjustment_new(0, 0, 32*32, 32, 32*10, 300));
  gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
                     (GtkSignalFunc)drawing_area_hscroll, NULL);
  hscroll = gtk_hscrollbar_new(adj);
  gtk_table_attach(GTK_TABLE(table), hscroll, 0,1, 1,2,
                   GTK_FILL,0, 0,0);
  adj = GTK_ADJUSTMENT(gtk_adjustment_new(0, 0, 32*32, 32, 32*10, 300));
  gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
                     (GtkSignalFunc)drawing_area_vscroll, NULL);
  vscroll = gtk_vscrollbar_new(adj);
  gtk_table_attach(GTK_TABLE(table), vscroll, 1,2, 0,1,
                   0,GTK_FILL, 0,0);

  /* The status area at the bottom */
  statusarea = gtk_event_box_new();
  gtk_box_pack_start(GTK_BOX(vbox), statusarea, FALSE, FALSE, 0);

  status_hbox = gtk_hbox_new(FALSE, 2);
  gtk_container_add(GTK_CONTAINER(statusarea), status_hbox);

  /*  create the contents of the status area *********************************/

  /*  the cursor label  */
  label_frame = gtk_frame_new(NULL);
  gtk_frame_set_shadow_type(GTK_FRAME(label_frame), GTK_SHADOW_IN);

  cursor_label = gtk_label_new(" 128 x 128 ");
  gtk_container_add(GTK_CONTAINER(label_frame), cursor_label);

  /*  the status label  */
  status_frame = gtk_frame_new(NULL);
  gtk_frame_set_shadow_type(GTK_FRAME(status_frame), GTK_SHADOW_IN);
  
  status_label = gtk_label_new("Crafted");
  gtk_container_add(GTK_CONTAINER(status_frame), status_label);
  gtk_misc_set_alignment(GTK_MISC(status_label), 0.0, 0.5);
  gtk_misc_set_padding(GTK_MISC(status_label), 5, 0);

  /*  the progress bar  */
  adj = (GtkAdjustment*)gtk_adjustment_new(0.0, 0.0, 1.0, 0.0, 0.0, 0.0);
  progressbar = gtk_progress_bar_new_with_adjustment(adj);
  gtk_widget_set_usize(progressbar, 80, -1);
  gtk_box_pack_start(GTK_BOX(status_hbox), label_frame, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(status_hbox), status_frame, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(status_hbox), progressbar, FALSE, FALSE, 0);
  
  gtk_widget_show_all(main_window);
  
  gtk_widget_set_sensitive(main_window, FALSE);
  while (!load_tileset_and_png(0))
  {
    ask_for_data_path();
  }
  gtk_widget_set_sensitive(main_window, TRUE);

  current_tool = 1;

  /* the "tiles" tools */
  radio = NULL;
  for (i=1 ; i<LAST_SOLID_TILE ; i++)
  {
    gdk_pixmap_tool[i] = gdk_pixmap_new(drawing_area->window, 32, 32, -1);
    gdk_window_copy_area(gdk_pixmap_tool[i], drawing_area->style->white_gc,
                         0, 0, tileset_pixmap,
                         (solid_tiles[i][0] % 16)*32,
                         (solid_tiles[i][0] / 16)*32,
                         32,32);
    pixmap_tool[i] = gtk_pixmap_new(gdk_pixmap_tool[i], NULL);
    radio = gtk_toolbar_append_element(GTK_TOOLBAR(toolbar),
                                       GTK_TOOLBAR_CHILD_RADIOBUTTON,
                                       radio,
                                       icon_names[i],
                                       icon_names[i],
                                       icon_names[i],
                                       pixmap_tool[i],
                                       select_tool,
                                       GINT_TO_POINTER(i));
    if (i==1) first_radio = radio;
    // Walls are disabled and empty tile types
    if ( (i>=9 && i<BRIDGE) || solid_tiles[i][0]==0 )
      gtk_widget_set_sensitive((radio), FALSE);
  }

  gtk_widget_set_sensitive(main_vbox, FALSE);
  gtk_label_set_text(GTK_LABEL(status_label), _("loading units..."));
  gtk_progress_set_show_text(GTK_PROGRESS(progressbar), TRUE);

  fill_unit_lists(1, general_unit_list);
  fill_unit_lists(2, human_unit_list);
  fill_unit_lists(3, orc_unit_list);
  update_units_toggles();

  /* the "unit" tool */
  gdk_pixmap_tool[LAST_SOLID_TILE] = gdk_pixmap_new(drawing_area->window, 32, 32, -1);
  gdk_window_copy_area(gdk_pixmap_tool[LAST_SOLID_TILE], drawing_area->style->white_gc,
                       0, 0, tileset_pixmap, /* FIXME: find something better */
                       0, 32, 32, 32);
  pixmap_tool[LAST_SOLID_TILE] = gtk_pixmap_new(
			gdk_pixmap_tool[LAST_SOLID_TILE], NULL);
  unit_tool_radio = gtk_toolbar_append_element(GTK_TOOLBAR(toolbar),
                                     GTK_TOOLBAR_CHILD_RADIOBUTTON,
                                     radio,
                                     _("Units"),
                                     _("Units"),
                                     _("Units"),
                                     pixmap_tool[LAST_SOLID_TILE],
                                     select_tool,
                                     GINT_TO_POINTER(i));
                       
  gtk_progress_set_value(GTK_PROGRESS(progressbar), 0.0);
  gtk_progress_set_show_text(GTK_PROGRESS(progressbar), FALSE);
  gtk_label_set_text(GTK_LABEL(status_label), "Crafted");
  gtk_widget_set_sensitive(main_vbox, TRUE);

  gtk_list_select_item(GTK_LIST(general_unit_list), 0);
  gtk_list_select_item(GTK_LIST(human_unit_list), 0);
  gtk_list_select_item(GTK_LIST(orc_unit_list), 0);
  
  init_pud(256, 256);
  if (pud_to_load)
  {
    g_print("should load '%s'\n", pud_to_load);
    load_pud(pud_to_load);
  }
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(first_radio), TRUE);
  
  g_mem_profile();
  gtk_main();

  g_print("after\n");
  g_mem_profile();
  return 0;
}

