;;; -*- Mode: LISP; Syntax: ANSI-Common-Lisp; Package: CL-USER; -*-
;;;
;;; Filename:    ssl-ffi-lw.lisp
;;; Authors:     Jochen Schmidt <jsc@dataheaven.de>
;;;              Wade Humeniuk <humeniuw@cadvision.com>
;;; Description: Definitions of the needed foreign-functions
;;;              and the byte-vector handling needed for
;;;              simple buffering.
;;;

(in-package :ssl-internal)

(eval-when (:compile-toplevel :load-toplevel :execute)
  (require "comm"))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Register Modules ;;;
;;;;;;;;;;;;;;;;;;;;;;;;

;;; This ssl.so has to be created with
;;; ld -shared -o ssl.so -L/usr/lib -lssl -lcrypto
;;; -- jsc@dataheaven.de
#-win32(fli:register-module "/tmp/ssl.so")

;; OpenSSL dlls compiled by mingw32
#+win32(fli:register-module :libssl32 :real-name "libssl32")
#+win32(fli:register-module :libeay32 :real-name "libeay32")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Connection and Error handling  ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fli:define-foreign-function (ssl-load-error-strings "SSL_load_error_strings" :source)
    ()
  :result-type :void
  :language :ansi-c)

(fli:define-foreign-function (ssl-library-init "SSL_library_init" :source)
    ()
  :result-type :int
  :language :ansi-c)

(fli:define-foreign-function (ssl-v23-method "SSLv23_method" :source)
    ()
  :result-type :pointer
  :language :ansi-c)

(fli:define-foreign-function (ssl-set-connect-state "SSL_set_connect_state" :source)
    ((ssl :pointer))
  :result-type :void
  :language :ansi-c)

(fli:define-foreign-function (ssl-set-accept-state "SSL_set_accept_state" :source)
    ((ssl :pointer))
  :result-type :void
  :language :ansi-c)

(fli:define-foreign-function (ssl-ctx-new "SSL_CTX_new" :source)
    ((method :pointer))
  :result-type :pointer
  :language :ansi-c)

(fli:define-foreign-function (ssl-new "SSL_new" :source)
    ((method :pointer))
  :result-type :pointer
  :language :ansi-c)

(fli:define-foreign-function (ssl-set-fd "SSL_set_fd" :source)
    ((ssl :pointer)
     (fd :int))
  :result-type :int
  :language :ansi-c)

(fli:define-foreign-function (ssl-ctx-set-mode "SSL_CTX_set_mode" :source)
    ((ctx :pointer)
     (fd :long))
  :result-type :long
  :language :ansi-c)

(fli:define-foreign-function (ssl-get-error "SSL_get_error" :source)
    ((ssl :pointer)
     (ret :int))
  :result-type :int
  :language :ansi-c)

(fli:define-foreign-function (err-get-error "ERR_get_error" :source)
    ()
  :result-type (:unsigned :long)
  :language :ansi-c)

(fli:define-foreign-function (ssl-connect "SSL_connect" :source)
    ((ssl :pointer ))
  :result-type :int
  :language :ansi-c)

(fli:define-foreign-function (ssl-accept "SSL_accept" :source)
    ((ssl :pointer ))
  :result-type :int
  :language :ansi-c)

(fli:define-foreign-function (ssl-write "SSL_write" :source)
    ((ssl :pointer)
     (buf (:c-array (:unsigned :byte)))
     (num :int))
  :result-type :int
  :language :ansi-c)


(fli:define-foreign-function (ssl-read "SSL_read" :source)
    ((ssl :pointer)
     (buf (:c-array (:unsigned :byte)))
     (num :int))
  :result-type :int
  :language :ansi-c)

(fli:define-foreign-function (ssl-pending "SSL_pending" :source)
    ((ssl :pointer))
  :result-type :int
  :language :ansi-c)

(fli:define-foreign-function (ssl-shutdown "SSL_shutdown" :source)
    ((ssl :pointer))
  :result-type :int
  :language :ansi-c)

(fli:define-foreign-function (ssl-free "SSL_free" :source)
    ((ssl :pointer))
  :result-type :void
  :language :ansi-c)

(fli:define-foreign-function (ssl-ctx-free "SSL_CTX_free" :source)
    ((ssl :pointer))
  :result-type :void
  :language :ansi-c)

(fli:define-foreign-function (rand-seed "RAND_seed" :source)
    ((buf (:c-array (:unsigned :byte)))
     (num :int))
  :result-type :void
  :language :ansi-c)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Certificate and Privatekey handling ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defconstant +ssl-filetype-pem+ 1)
(defconstant +ssl-filetype-asn1+ 2)
(defconstant +ssl-filetype-default+ 3)

;; Set the set of available ciphers
(fli:define-foreign-function (ssl-set-cipher-list "SSL_set_cipher_list" :source)
    ((ssl :pointer)
     (str (:reference-pass (:ef-mb-string :limit 512)))) ; Limit arbitrary chosen
  :result-type :int
  :language :ansi-c)

;; SSL-USE-RSA-PRIVATEKEY-FILE
;; Load the RSA privatekey from the specified filename into the connection-handle
;; type is either +ssl-filetype-pem+ or +ssl-filetype-asn1+
(fli:define-foreign-function (ssl-use-rsa-privatekey-file "SSL_use_RSAPrivateKey_file" :source)
    ((ssl :pointer)
     (str (:reference-pass (:ef-mb-string :limit 1024)))
     (type :int))
  :result-type :int
  :language :ansi-c)

;; SSL-CTX-USE-RSA-PRIVATEKEY-FILE
;; Load the RSA privatekey from the specified filename into the CTX
;; type is either +ssl-filetype-pem+ or +ssl-filetype-asn1+
(fli:define-foreign-function (ssl-ctx-use-rsa-privatekey-file "SSL_CTX_use_RSAPrivateKey_file" :source)
    ((ctx :pointer)
     (str (:reference-pass (:ef-mb-string :limit 1024)))
     (type :int))
  :result-type :int
  :language :ansi-c)

;; SSL-USE-CERTIFICATE-FILE
;; Load the certificate from the specified filename into the connection-handle
;; type is either +ssl-filetype-pem+ or +ssl-filetype-asn1+
(fli:define-foreign-function (ssl-use-certificate-file "SSL_use_certificate_file" :source)
    ((ssl :pointer)
     (str (:reference-pass (:ef-mb-string :limit 1024)))
     (type :int))
  :result-type :int
  :language :ansi-c)

;; SSL-CTX-LOAD-VERIFY-LOCATIONS
;; Set the location where further CA certificates can be found to the specified
;; directory and/or file
(fli:define-foreign-function (ssl-ctx-load-verify-locations "SSL_CTX_load_verify_locations" :source)
    ((ctx :pointer)
     (CAfile (:reference-pass (:ef-mb-string :limit 1024) :allow-null t))
     (CApath (:reference-pass (:ef-mb-string :limit 1024) :allow-null t)))
  :result-type :int
  :language :ansi-c)
                             
;; SSL-CTX-SET-CLIENT-CA-LIST
;; Set the CTX' list of CAs that are acceptable from the client
(fli:define-foreign-function (ssl-ctx-set-client-ca-list "SSL_CTX_set_client_CA_list" :source)
    ((ctx :pointer)
     (list :pointer))
  :result-type :void
  :language :ansi-c)

;; SSL-SET-CLIENT-CA-LIST
;; Set the connection-handle's list of CAs that are acceptable from the client
(fli:define-foreign-function (ssl-set-client-ca-list "SSL_set_client_CA_list" :source)
    ((ssl :pointer)
     (list :pointer))
  :result-type :void
  :language :ansi-c)

;; SSL-LOAD-CLIENT-CA-FILE
;; Load and return a list of CAs from the specified file
(fli:define-foreign-function (ssl-load-client-ca-file "SSL_load_client_CA_file" :source)
    ((file (:ef-mb-string :limit 1024)))
  :result-type :pointer
  :language :ansi-c)

;;;;;;;;;;;;;;;;;;;;
;;; Byte Vectors ;;;
;;;;;;;;;;;;;;;;;;;;

(defun make-byte-vector (size)
  (fli:allocate-foreign-object :type `(:c-array (:unsigned :byte) ,size)))

(defun free-byte-vector (vector)
  (fli:free-foreign-object vector))

(defmacro bvref (buffer i)
  `(fli:foreign-aref ,buffer ,i))