;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: odcl -*-
;;; $Id: tree.lisp,v 1.19 2003/03/24 21:51:31 adam Exp $
;;;
;;; Copyright (c) 2002 - 2003 onShore Development, Inc.

;;; Rudiments of a threaded AVL tree
;;;
;;; Knuth ACP 6.2.3, Algorithm A

(in-package :odcl)

(defstruct (avl-node (:conc-name avl/)
                     (:constructor avl/create))
  left                                  ; subtree L
  right                                 ; subtree R
  (balance 0 :type fixnum)              ; balance factor
  data)                                 ; data

(defclass avl-tree ()
  ((head       :initarg :head
               :accessor tree-head
               :initform (avl/create))
   (test       :initarg :test
               :reader tree-test
               :initform (error "avl-tree constructor must be supplied a test"))
   (size       :initarg :size
               :initform 0)             ; node count
   (height     :initarg :height
               :initform 0)             ; node count
   (generation :initarg :generation
               :initform 0)))           ; generation count

(defmethod tree-root ((self avl-tree))
  (avl/right (tree-head self)))

(defun avl/link (dir node)
  (ecase dir
    (-1 (avl/left node))
    (1  (avl/right node))))

(defsetf avl/link (dir node) (snode)
  `(ecase ,dir
    (-1 (setf (avl/left ,node) ,snode))
    (1  (setf (avl/right ,node) ,snode))))

(defun avl/mapc (map-function node &aux stack (go-left t))
  "stack based iteration until threads are complete"
  (while node
    (if (and go-left (avl/left node))
        (progn
          (push node stack)
          (setq node (avl/left node)))
        (progn
          (funcall map-function node)
          (if (avl/right node)
              (setq node (avl/right node)
                    go-left t)
              (setq node (pop stack)
                    go-left nil))))))

(defun c-iterator-push (iterator value)
  (funcall iterator value))

(defun c-iterator (avl-tree)
  (let ((node (tree-root avl-tree))
        (go-left t)
        (peeked nil)
        (stack nil))
    (flet ((get-next ()
             (if peeked
                 (pop peeked)
                 (loop
                  (if node
                      (if (and go-left (avl/left node))
                          (progn
                            (push node stack)
                            (setq node (avl/left node)))
                          (let ((data (avl/data node)))
                            (if (avl/right node)
                                (setq node (avl/right node)
                                      go-left t)
                                (setq node (pop stack)
                                  go-left nil))
                            (return-from get-next data)))
                      (return-from get-next nil))))))
      (lambda (&optional value)
        (if value
            (push value peeked)
            (get-next))))))

(defun avl/node-copy (root)
  (when root
    (avl/create :left (avl/node-copy (avl/left root))
                :right (avl/node-copy (avl/right root))
                :data (avl/data root)
                :balance (avl/balance root))))

;; public

(defmethod is-empty? ((self avl-tree))
  (null (tree-root self)))

(defmethod tree-member ((self avl-tree) data &aux found)
  (let ((node (tree-root self))
        (test (tree-test self)))
    (while (and node (not found))
      (cond ((funcall test data (avl/data node))
             (setq node (avl/left node)))
            ((funcall test (avl/data node) data)
             (setq node (avl/right node)))
            (t 
             (setq found t))))
    (when node
      (avl/data node))))

(defmethod tree-map (fn (self avl-tree))
  (avl/mapc (lambda (node)
              (funcall fn (avl/data node)))
            (tree-root self)))

(defmethod c-map (fn (self avl-tree))
  (avl/mapc (lambda (node)
              (funcall fn (avl/data node)))
            (tree-root self)))

(defmethod tree-first ((self avl-tree))
  (when-bind (node (tree-root self))
    (while (avl/left node)
      (setq node (avl/left node)))
    (avl/data node)))

(defmethod tree-last ((self avl-tree))
  (when-bind (node (tree-root self))
    (while (avl/right node)
      (setq node (avl/right node)))
    (avl/data node)))

(defmethod tree->list ((self avl-tree) &aux list)
  (avl/mapc (lambda (node)
               (push (avl/data node) list))
             (tree-root self))
  (nreverse list))

(defmethod get-size ((self avl-tree))
  (slot-value self 'size))

(defmethod clear ((self avl-tree))
  (setf (avl/right (tree-root self)) nil))

;; interface

(defmethod c-probe ((tree avl-tree) item)
  (assert (not (null item)))
  (let ((test (tree-test tree))
        (p (tree-root tree))
        (q nil))
    (unless p
      (return-from c-probe t))
    (loop
     (cond ((funcall test item (avl/data p))
            (setq q (avl/left p))
            (when (null q)
              (return-from c-probe nil))
            (setf p q))
           ((funcall test (avl/data p) item)
            (setq q (avl/right p))
            (when (null q)
              (return-from c-probe nil))
            (setf p q))
           (t
            (return-from c-probe (avl/data p)))))))

(defmethod c-add ((tree avl-tree) item &aux result-data)
  (assert (not (null item)))
  (flet ((merge (data &optional existing-node)
           (unless existing-node
             (setf existing-node (avl/create)))
           (setf result-data (avl/data existing-node)
                 (avl/data existing-node) data)
           existing-node))
    ;; A1 [Initialize.]
    (let* ((test (tree-test tree))
           (_t (tree-head tree))
           (s (avl/right _t))
           (p (avl/right _t))
           (q nil)
           (r nil))
      (unless p
        (setf (avl/right _t) (merge item))
        (return-from c-add result-data))
      (loop
       ;; A2 [Compare.]
       (cond ((funcall test item (avl/data p))
              ;; A3 [Move left.]
              (setq q (avl/left p))
              (when (null q)
                (setf q (merge item)
                      (avl/left p) q)
                (return))
              (unless (= 0 (avl/balance q))
                (setf _t p
                      s q))
              (setf p q))
             ((funcall test (avl/data p) item)
              ;; A4 [Move right.]
              (setq q (avl/right p))
              (when (null q)
                (setf q (merge item)
                      (avl/right p) q)
                (return))
              (unless (= 0 (avl/balance q))
                (setf _t p
                      s q))
              (setf p q))
             (t
              (merge item p)
              (return-from c-add result-data))))
      ;; A5 [Insert.] (initialization done above)
      ;; A6 [Adjust balance factors.]
      (let ((a (if (funcall test item (avl/data s)) -1 1)))
        (setf p (avl/link a s)
              r p)
        (while (not (eql p q))
          (cond ((funcall test item (avl/data p))
                 (setf (avl/balance p) -1
                       p (avl/left p)))
                ((funcall test (avl/data p) item)
                 (setf (avl/balance p) 1
                       p (avl/right p)))
                (t
                 (error "logic error 1"))))
        ;; A7 [Balancing act.]
        (cond ((= (avl/balance s) 0)
               ;; i)
               (setf (avl/balance s) a)
               (incf (slot-value tree 'height))
               (return-from c-add result-data))
              ((= (avl/balance s) (- a))
               ;; ii)
               (setf (avl/balance s) 0)
               (return-from c-add result-data))
              ((= (avl/balance s) a)
               (cond ((= (avl/balance r) a)
                      ;; A8 [Single rotation.]
                      (setf p r
                            (avl/link a s) (avl/link (- a) r)
                            (avl/link (- a) r) s
                            (avl/balance s) 0
                            (avl/balance r) 0))
                     ((= (avl/balance r) (- a))
                      ;; A9 [Double rotation.]
                      (setf p (avl/link (- a) r)
                            (avl/link (- a) r) (avl/link a p)
                            (avl/link a p) r
                            (avl/link a s) (avl/link (- a) p)
                            (avl/link (- a) p) s)
                      (cond ((= a (avl/balance p))
                             (setf (avl/balance s) (- a))
                             (setf (avl/balance r) 0))
                            ((= 0 (avl/balance p))
                             (setf (avl/balance s) 0)
                             (setf (avl/balance r) 0))
                            ((= (- a) (avl/balance p))
                             (setf (avl/balance s) 0)
                             (setf (avl/balance r) a))))
                     ;; (t ;; r = 0, a = 1 ?
                     ;;  (error "logic error: a/b r = ~s a = ~s" (avl/balance r) a))
                     )
               (setf (avl/balance p) 0)
               (if (eql s (avl/right _t))
                   (setf (avl/right _t) p)
                   (setf (avl/left _t) p)))
              (t
               (error "logic error 3")))))))

(defun %avl/rebalance-rotate (node link-dir sense &aux result)
  (let ((sub (avl/link link-dir node)))
    (ecase sense
      (:left
       (cond ((< (avl/balance sub) 0)
              (setf (avl/balance sub) 0)
              t)
             ((= (avl/balance sub) 0)
              (setf (avl/balance sub) 1)
              nil)
             (t
              (let* ((sub-r (avl/right sub))
                     (sub-r-b (avl/balance sub-r)))
                (if (>= sub-r-b 0)
                    (progn
                      (setf (avl/right sub) (avl/left sub-r)
                            (avl/left sub-r) sub)
                      (if (= 0 sub-r-b)
                          (setf (avl/balance sub) 1
                                (avl/balance sub-r) -1
                                result nil)
                          (setf (avl/balance sub) 0
                                (avl/balance sub-r) 0
                                result t))
                      (setf (avl/link link-dir node) sub-r)
                      result)
                    (let* ((sub-r-l (avl/left sub-r))
                           (sub-r-l-b (avl/balance sub-r-l)))
                      (setf (avl/left sub-r) (avl/right sub-r-l)
                            (avl/right sub-r-l) sub-r
                            (avl/right sub) (avl/left sub-r-l)
                            (avl/left sub-r-l) sub)
                      (if (> sub-r-l-b 0)
                          (setf (avl/balance sub) -1)
                          (setf (avl/balance sub) 0))
                      (if (< sub-r-l-b 0)
                          (setf (avl/balance sub-r) 1)
                          (setf (avl/balance sub-r) 0))
                      (setf (avl/link link-dir node) sub-r-l
                            (avl/balance sub-r-l) 0)
                      t))))))
      (:right
       (cond ((> (avl/balance sub) 0)
              (setf (avl/balance sub) 0)
              t)
             ((= (avl/balance sub) 0)
              (setf (avl/balance sub) -1)
              nil)
             (t
              (let* ((sub-l   (avl/left sub))
                     (sub-l-b (avl/balance sub-l)))
                (if (<= sub-l-b 0)
                    (progn
                      (setf (avl/left sub) (avl/right sub-l)
                            (avl/right sub-l) sub)
                      (if (= 0 sub-l-b)
                          (setf (avl/balance sub) -1
                                (avl/balance sub-l) 1
                                result nil)
                          (setf (avl/balance sub) 0
                                (avl/balance sub-l) 0
                                result t))
                      (setf (avl/link link-dir node) sub-l)
                      result)
                    (let* ((sub-l-r   (avl/right sub-l))
                           (sub-l-r-b (avl/balance sub-l-r)))
                      (setf (avl/right sub-l) (avl/left sub-l-r)
                            (avl/left sub-l-r) sub-l
                            (avl/left sub) (avl/right sub-l-r)
                            (avl/right sub-l-r) sub)
                      (if (< sub-l-r-b 0)
                          (setf (avl/balance sub) 1)
                          (setf (avl/balance sub) 0))
                      (if (> sub-l-r-b 0)
                          (setf (avl/balance sub-l) -1)
                          (setf (avl/balance sub-l) 0))
                      (setf (avl/link link-dir node) sub-l-r
                            (avl/balance sub-l-r) 0)
                      t)))))))))

(defun %avl/rebalance (node link-dir q)
  (let ((br (avl/link link-dir node)))
    (if (avl/right br)
        (when (%avl/rebalance br 1 q)
          (%avl/rebalance-rotate node link-dir :right))
	(progn
          (setf (avl/data q) (avl/data br)
                (avl/link link-dir node) (avl/left br))
          t))))

(defmethod c-remove ((tree avl-tree) data)
  (let ((test (tree-test tree)))
    (labels ((%c-remove (root link-dir)
               (when-bind (sub (avl/link link-dir root))
                 (cond ((funcall test data (avl/data sub))
                        ;; look left
                        (when (%c-remove sub -1)
                          (%avl/rebalance-rotate root link-dir :left)))
                       ((funcall test (avl/data sub) data)
                        ;; look right
                        (when (%c-remove sub 1)
                          (%avl/rebalance-rotate root link-dir :right)))
                       (t
                        ;; match: delete, return t if everything is ok
                        (cond ((null (avl/right sub))
                               (setf (avl/link link-dir root)
                                     (avl/left sub))
                               t)
                              ((null (avl/left sub))
                               (setf (avl/link link-dir root)
                                     (avl/right sub))
                               t)
                              (t
                               (when (%avl/rebalance sub -1 sub)
                                 (%avl/rebalance-rotate root link-dir :left)))))))))
      ;; start deleting to the right (top node is a dummy)
      (%c-remove (tree-head tree) 1))))

(defmethod c-copy ((self avl-tree))
  (with-slots (head test)
    self
    (make-instance 'avl-tree
                   :test test
                   :head (avl/node-copy head))))

(defun node-depth (node)
  (+ 1 (max (if-bind (r (avl/right node))
                (node-depth r)
                0)
            (if-bind (l (avl/left node))
                (node-depth l)
                0))))

(defun node-balance (node)
  (let ((rd (if-bind (r (avl/right node))
                (node-depth r)
                0))
        (ld (if-bind (l (avl/left node))
                (node-depth l)
                0)))
    (if (or (= rd ld)
            (> 2 (abs (- ld rd))))
        0
        (if (< rd ld)
            1
            -1))))

(defparameter *test* nil)

(defun test-avl ()
  (dotimes (x 32)
    (setq *test* nil)
    (dotimes (x 32768)
      (push (random 128) *test*))
    (test-inner *test*)))

;;(setq *failing*
;;      '(8 2 1 12 14 4 11 11 6 7 13 6 7 10 12 10 11 6 9 8 9 5 5 7 7 1 15 7 6 11 15 6))
;;      '(2 0 4 0 5 0 6 0 7 0 3 6 7 4 1 5 0 7))

(defun test-inner (test)
  (let ((tree (make-instance 'avl-tree :test #'<))
        (test (copy-list test)))
    (while test
      (c-add tree (pop test))
      (c-remove tree (pop test)))
    tree))
