;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: odcl -*-
;;; $Id: transaction.lisp,v 1.57 2003/09/09 17:21:05 craig Exp $
;;;
;;; Copyright (c) 2000 - 2003 onShore Development, Inc.

(in-package :odcl)

;; variables

(defvar *fascistic-context* nil)

(defvar *recover-context* t)

(defvar *default-editing-context* nil
  "The editing context in which changes occur unless an editing
context is given specifically.  If NIL, then no editing context will
be used.")

(defvar *ec-unwinding* nil
  "T if the editiing context is actively aborting")

(defparameter *default-ec-cache-size* 5000
  "Number of objects to cache per ec")

;; CLASS DEFINITIONS

;; ------------------------------------------------------------
;; transactable-store
;; ------------------------------------------------------------
;;
;; A transactable-store is the persistent storage manager for a
;; collection of transaction-objects.
;;
;; An editing-context synchronizes a transaction across a set of
;; transactable-stores.
;;
;; In order to participate in transactions, the following methods must
;; be implemented:
;;
;; (get-store-instance-by-key #<store> #<key>)
;;  -------------------------
;;
;; Look into the store, and return object identified by key, where nil
;; definitively indicates nonexistence.

(defclass transactable-store ()
  ((name :initarg :name
         :reader name
         :initform nil)
   (state :initform :uninitialized)
   (transaction :initform nil)))

(defmethod get-store-classes ((self transactable-store))
  (error "GET-STORE-CLASSES must be defined for objects of type ~A" (type-of self)))

(defmethod generate-new-oid ((self transactable-store) class)
  (declare (ignore class))
  (error "GENERATE-NEW-OID must be defined for objects of type ~A" (type-of self)))

(defmethod initialize-store ((self transactable-store) state)
  (declare (ignore state))
  (error "INITIALIZE-STORE must be defined for objects of type ~A" (type-of self)))

(defmethod get-store-instance-by-key ((self transactable-store) key)
  (declare (ignore key))
  (error "GET-STORE-INSTANCE-BY-KEY must be defined for objects of type ~A" (type-of self)))

(defmethod commit-phase-1 ((self transactable-store) transaction)
  (declare (ignore transaction))
  (error "COMMIT-PHASE-1 must be defined for objects of type ~A" (type-of self)))

;; ------------------------------------------------------------
;; transaction-object
;; ------------------------------------------------------------
;;
;; A transaction-object is an object managed by a transactable store.
;;
;; In order to participate in transactions, the following methods must
;; be implemented:
;;
;; (instance-key #<tx-object>)
;;  ------------
;;
;; Return a key for tx-object of the form (symbol key-1 ... key-n)
;;
;; (instance-snapshot #<tx-object>) => #<snapshot>
;;  -----------------
;;
;; Capture the rollback-suitable state of an object at the beginning
;; of an edit or delete.
;;
;; (instance-restore #<tx-object> #<snapshot>) => nil
;;  ----------------
;;
;; Destructively modify tx-object to reassume snapshot state.

(defclass transaction-object ()
  ((editing-context :initarg :editing-context
                    :initform nil                       ; signifying "unregistered"
                    :accessor editing-context
                    :documentation "The editing context that provided this object."))
  (:documentation "Superclass of all objects managed by an editing context."))

(defmethod instance-key ((self transaction-object))
  (error "INSTANCE-KEY must be defined for objects of type ~A" (type-of self)))

(defmethod instance-snapshot ((self transaction-object))
  (error "INSTANCE-SNAPSHOT must be defined for objects of type ~A" (type-of self)))

(defmethod instance-restore ((self transaction-object) snapshot)
  (declare (ignore snapshot))
  (error "INSTANCE-RESTORE must be defined for objects of type ~A" (type-of self)))

(defmethod instance-ec-status ((self transaction-object) ec)
  (let ((key (instance-key self)))
    (or (get-unstoreable (car key) (cdr key))
        (progn
          (multiple-value-bind (object status)
              (ec-cache-get ec key)
            (when object
              (return-from instance-ec-status status)))))))

;; Transaction object Protocol

(defun %instance-key-p (key)
  (and (typep (car key) 'symbol)
       (not (null (second key)))))

(deftype instance-key ()
  '(and list
    (satisfies %instance-key-p)))

(defmethod transaction-objects-equal ((a transaction-object)
                                      (b transaction-object))
  (and (eql (class-of a) (class-of b))
       (equal (instance-key a) (instance-key b))))

(defmethod transaction-objects-equal ((a t)
                                      (b t))
  (equal a b))

(defmethod attached-to-transaction? ((self transactable-store) transaction)
  (eql (slot-value self 'transaction) transaction))

(defmethod attach-transaction ((self transactable-store) attach-transaction)
  (with-slots (transaction)
    self
    (when transaction
      (error "Store ~s is already attached to transaction ~s" self transaction))
    (setf transaction attach-transaction)))

(defmethod detach-transaction ((self transactable-store) detach-transaction)
  (with-slots (transaction)
    self
    (unless (eql transaction detach-transaction)
      (error "Store ~s is not attached to transaction ~s: value is ~s" self detach-transaction transaction))
    (setf transaction nil)))

;; stack events

(defstruct tx-event
  (type :undefined :type keyword)
  (store nil)
  (key nil :type list)
  (data nil))

(defmethod print-object ((self tx-event) stream)
  (print-unreadable-object
   (self stream :type t :identity t)
   (format stream "~A key: ~A data: ~A"
           (tx-event-type self)
           (tx-event-key self)
           (tx-event-data self))))

;; transaction

(defclass transaction ()
  ((state             :initform :inactive)
   (stack             :initform nil)
   (edited-objects  :initform (make-instance 'hash-table-cache
                                             :test #'equal
                                             :markable t
                                             :keyfun 'instance-key)
                    :documentation "Markable Hash table containing
modified objects.  Marks indicated the type of change made, and are
limited to :new :deleted and :edited.")
   (abort-hooks       :initform nil)
   (precommit-hooks   :initform nil)
   (commit-hooks      :initform nil)))

(defun set-transaction-active (transaction)
  (with-slots (state stack abort-hooks commit-hooks precommit-hooks edited-objects)
    transaction
    (ecase state
      (:active
       (error "Transaction is already active"))
      (:inactive
       (cache-flush edited-objects)
       (setf state :active
             stack nil
             abort-hooks nil
             precommit-hooks nil
             commit-hooks nil)))))

(defun set-transaction-inactive (transaction)
  (with-slots (state stack abort-hooks commit-hooks precommit-hooks edited-objects)
    transaction
    (ecase state
      (:inactive
       (error "Transaction is already inactive"))
      (:active
       (cache-flush edited-objects)
       (setf state :inactive
             stack nil
             abort-hooks nil
             precommit-hooks nil
             commit-hooks nil)))))

(defun transaction-mark (transaction key mark)
  (with-slots (edited-objects)
    transaction
    (cache-mark edited-objects key mark)))

(defun transaction-put (transaction key value mark)
  (with-slots (edited-objects)
    transaction
    (cache-put edited-objects key value mark)))

(defun transaction-active? (transaction)
  (eql (slot-value transaction 'state) :active))

(defun on-txn-abort (transaction fn)
  (push fn (slot-value transaction 'abort-hooks)))

(defun on-txn-commit (transaction fn)
  (push fn (slot-value transaction 'commit-hooks)))

(defun on-txn-precommit (transaction fn)
  (push fn (slot-value transaction 'precommit-hooks)))

(defmethod run-abort-hooks ((txn transaction))
  (mapcar #'funcall (slot-value txn 'abort-hooks)))

(defmethod run-commit-hooks ((txn transaction))
  (mapcar #'funcall (slot-value txn 'commit-hooks)))

(defmethod run-precommit-hooks ((txn transaction))
  (mapcar #'funcall (slot-value txn 'precommit-hooks)))

;; ------------------------------------------------------------
;; editing context
;; ------------------------------------------------------------

(defclass editing-context ()
  ((stores      :initarg :stores
                :initform nil
                :documentation "A list of objects implementing the
transaction object storage protocol for which this context manages
transactions and caching")
   #+cmu
   (lock        :initform (mp::make-lock "EC Lock"))
   (transaction :initform (make-instance 'transaction)
                :reader current-transaction)
   (config      :initform #*1111)        ; cache, log, authorize, reference
   (cache       :initform (make-instance 'lru-cache
                                         :size *default-ec-cache-size*
                                         :test #'equal))))

(defun %ec-config-bit (keyword)
  (ecase keyword
    (:cache 0)
    (:log 1)
    (:authorize 2)
    (:reference 3)))

(defun get-ec-config (keyword &optional (ec *default-editing-context*))
  (= 1 (aref (slot-value ec 'config) (%ec-config-bit keyword))))

(defmacro with-ec-config (config-bits &rest body)
  (let ((temp-config #*0000))
    (dolist (config-bit config-bits)
      (setf (aref temp-config (%ec-config-bit config-bit)) 1))
    `(with-slots (config)
      *default-editing-context*
      (let ((stored-config config))
        (setf config ,temp-config)
        (unwind-protect
             (let ()
               ,@body)
          (setf config stored-config))))))

(defmethod get-context-store ((self editing-context) instance)
  (get-context-store-by-class self (type-of instance)))

(defmethod get-context-store-by-class ((self editing-context) class)
  (dolist (store (slot-value self 'stores))
    (when (member class (get-store-classes store))
      (return-from get-context-store-by-class store)))
  (error "No store for class ~s" class))

(defmethod attach-store ((self editing-context) (store transactable-store))
  (push store (slot-value self 'stores)))

(defmethod manages-store? ((ec editing-context) (store transactable-store))
  (member store (slot-value ec 'stores)))

(defmethod print-object ((self editing-context) stream)
  (print-unreadable-object
   (self stream :type t :identity t)
   (if (and (slot-boundp self 'transaction)
            (slot-boundp self 'cache))
       (with-slots (transaction cache)
         self
         (write-string "root" stream))
       (write-string "invalid" stream))))

(defmethod initialize-instance :after ((self editing-context) &rest init-options)
  (declare (ignore init-options))
  (with-slots (cache)
    self
    (push (lambda (value)
            (ec-unregister value self))
          (delete-hook cache))
    (push (lambda (value)
            (ec-register value self))
          (insert-hook cache)))
  self)

(defmethod destroy-editing-context ((context editing-context))
  (ec-abort context)
  (with-slots (cache)
    context
    (when (slot-boundp context 'cache)
      (let ((scanner (%cache-scanner cache)))
        (do ((kvm (funcall scanner) (funcall scanner)))
            ((null kvm))
          (when (cadr kvm)
            (setf (slot-value (cadr kvm) 'editing-context) nil))))
      (destroy-cache cache))
    (slot-makunbound context 'stores)))

;;(defmethod editing-context-validp ((context editing-context))
;;  (unless (slot-boundp context 'stores)
;;    (error 'editing-context-error)))

(defmethod editing-context-bindings (store)
  (declare (ignore store))
  nil)

(defun get-context-bindings (context)
  (cons (cons '*default-editing-context* context)
        (mapcan #'editing-context-bindings (slot-value context 'stores))))

#+cmu
(defmacro with-context ((&optional (editing-context *default-editing-context*)) &body body)
  (let ((bindings-sym (gensym "bindings")))
    `(let* ((,bindings-sym (get-context-bindings ,editing-context))
            (lock (slot-value ,editing-context 'lock)))
      (mp::with-lock-held (lock)
        (progv
            (mapcar #'car ,bindings-sym)
            (mapcar #'cdr ,bindings-sym)
          (with-ec-config (:cache :log :authorize :reference)
            (unwind-protect
                 (multiple-value-prog1
                     (progn
                       (ec-begin ,editing-context)
                       ,@body)
                   (ec-commit ,editing-context nil))
              (ec-abort ,editing-context))))))))

#-cmu
(defmacro with-context ((&optional (editing-context *default-editing-context*)) &body body)
  (let ((bindings-sym (gensym "bindings")))
    `(let* ((,bindings-sym (get-context-bindings ,editing-context)))
      (progv
          (mapcar #'car ,bindings-sym)
          (mapcar #'cdr ,bindings-sym)
        (with-ec-config (:cache :log :authorize :reference)
          (unwind-protect
               (multiple-value-prog1
                   (progn
                     (ec-begin ,editing-context)
                     ,@body)
                 (ec-commit ,editing-context nil))
            (ec-abort ,editing-context)))))))

(defmacro without-context (&body body)
  `(let ((*default-editing-context* nil))
    ,@body))

;; Internal methods for behavior modification

(defmethod ec-register ((instance transaction-object) (context editing-context))
  (with-slots (editing-context)
    instance
    (when editing-context
      (when *fascistic-context*
        (error "Instance ~s already registered in context ~s" instance editing-context)))
    (setf editing-context context)))

(defmethod ec-unregister ((instance transaction-object) (context editing-context))
  (with-slots (editing-context)
    instance
    (unless editing-context
      (when *fascistic-context*
        (error "Instance ~s not registered in context ~s" instance context)))
    (unless (eql editing-context context)
      (when *fascistic-context*
        (error "Instance ~s unregistration mismatch: ~s != ~s" instance editing-context context)))
    (setf editing-context nil)))

(defmethod ec-cache ((instance transaction-object) (context editing-context))
  (let ((key (instance-key instance)))
    (with-slots (cache)
      context
      (when (get-ec-config :cache context)
        (cache-put cache key instance)))))

;; called when a new object is created and inserted for the first time

(defmethod ec-insert ((instance transaction-object) (context editing-context))
  (let ((key (instance-key instance))
        (store (get-context-store context instance)))
    (with-slots (cache transaction)
      context
      ;; put it in our transaction edited objects hash
      (setf (editing-context instance) context)
      (transaction-put transaction key instance :new)
      (add-tx-event context :insert store key instance))))

;; we need to look in the present transactions set of edited objects to
;; find any marks on the object and the most recent version
(defmethod ec-cache-get ((self editing-context) key)
  (with-slots (cache transaction)
    self
    (or (cache-get (slot-value transaction 'edited-objects) key)
        (cache-get cache key))))

(defmethod ec-edit ((instance transaction-object) context)
  (unless (slot-value instance 'editing-context)
    (setf (slot-value instance 'editing-context) context))
  (unless *ec-unwinding*
    (let ((key (instance-key instance))
          (store (get-context-store context instance)))
      (with-slots (cache transaction)
        context
        ;; put it in our transaction edited objects hash
        (transaction-put transaction key instance :edited)
        (add-tx-event context :edit store key instance)))))

(defmethod ec-delete ((instance transaction-object) (context editing-context))
  (unless (slot-value instance 'editing-context)
    (ec-cache instance context))
  (let ((key (instance-key instance))
        (store (get-context-store context instance)))
    (with-slots (cache transaction)
      context
      ;; put it in our transaction edited objects hash
      (transaction-put transaction key instance :deleted)
      (add-tx-event context :delete store key instance))))

(defun add-tx-event (context type store key data)
  (unless store
    (error "No store for transaction event"))
  (with-slots (transaction)
    context
    (with-slots (stack)
      transaction
      (let* ((rollback (car (member-if (lambda (stack-item)
                                         (and (eql (tx-event-type stack-item) :rollback)
                                              (equal (tx-event-key stack-item) key))) stack)))
             (edit-state (if rollback
                             (car (member-if (lambda (stack-item)
                                               (and (equal (tx-event-key stack-item) key))) stack)))))
        (ecase type
          (:insert
           (if rollback
               (case (tx-event-type edit-state)
                 (:insert
                  (error "Double insertion of object ~s" data))
                 (:edit
                  (setf (tx-event-data edit-state) data))
                 (:delete
                  (error "reinsertion of deleted object ~s" data))
                 (t
                  (error "transaction inconsistent")))
               (progn
                 (push (make-tx-event :type :rollback :store store :key key :data nil) stack)
                 (push (make-tx-event :type :insert :store store :key key :data data) stack))))
          (:edit
           (if rollback
               (case (tx-event-type edit-state)
                 ((:edit :insert)
                  (setf (tx-event-data edit-state) data))
                 (:delete
                  (error "Edit of deleted object ~s" data))
                 (t
                  (error "transaction inconsistent")))
               (progn
                 (push (make-tx-event :type :rollback :store store :key key :data (instance-snapshot data)) stack)
                 (push (make-tx-event :type :edit :store store :key key :data data) stack))))
          (:delete
           (if rollback
               (case (tx-event-type edit-state)
                 (:insert
                  ;; Deletion of newly inserted object
                  (setf (tx-event-type edit-state) :delete))
                 (:edit
                  (setf (tx-event-type edit-state) :delete))
                 (:delete
                  (error "Delete of deleted object ~s" data))
                 (t
                  (error "transaction inconsistent")))
               (progn
                 (push (make-tx-event :type :rollback :store store :key key :data (instance-snapshot data)) stack)
                 (push (make-tx-event :type :delete :store store :key key :data data) stack)))))))))

(defun ec-on-txn-abort (fn &optional (ec *default-editing-context*))
  (with-slots (transaction)
    ec
    (on-txn-abort transaction fn)))

(defun ec-on-txn-commit (fn &optional (ec *default-editing-context*))
  (with-slots (transaction)
    ec
    (on-txn-commit transaction fn)))

(defun ec-on-txn-precommit (fn &optional (ec *default-editing-context*))
  (with-slots (transaction)
    ec
    (on-txn-precommit transaction fn)))

(defmethod run-abort-hooks ((context editing-context))
  (with-slots (transaction)
    context
    (run-abort-hooks transaction)))

(defmethod run-commit-hooks ((context editing-context))
  (with-slots (transaction)
    context
    (run-commit-hooks transaction)))

(defmethod run-precommit-hooks ((context editing-context))
  (with-slots (transaction)
    context
    (run-precommit-hooks transaction)))

;; External interface of editing context

(defmethod ec-reset ((self editing-context))
  (let ((*fascistic-context* nil))
    (with-slots (cache stores transaction)
      self
      (dolist (store stores)
        (setf (slot-value store 'transaction) nil))
      (setf transaction (make-instance 'transaction))
      (cache-flush cache))))

(defun ec-begin (&optional (ec *default-editing-context*))
  (with-slots (transaction)
    ec
    (set-transaction-active transaction)
    (dolist (store (slot-value ec 'stores))
      (attach-transaction store transaction))
    (publish-event 'transaction-start :transaction transaction)))
                       

(defun ec-commit (&optional (ec *default-editing-context*) (begin-transaction t))
  (with-slots (transaction stores)
    ec
    (when (transaction-active? transaction)
      (editing-context-execute-changes ec)
      ;; after commit we set the transaction inactive, this
      ;; way ec-abort will not try and run the abort-hooks
      ;; and rewind the stack
      (set-transaction-inactive transaction)
      
      (dolist (store stores)
        (when (attached-to-transaction? store transaction)
          (detach-transaction store transaction)))

      (publish-event 'transaction-commit :transaction transaction)
      
      (when begin-transaction
        (ec-begin ec)))))

(defun ec-abort (&optional (ec *default-editing-context*))
  (with-slots (stores cache transaction)
    ec
    ;; run our hooks first, before we detach
    (unwind-protect
        (progn

          ;; catch errors in rollbacks and wrap them in
          ;; ec-rollback-error so that we can perform cleanup
          (handler-case
              (when (transaction-active? transaction)
                (run-abort-hooks ec))
            (error (e)
              (error 'ec-rollback-error
                     :original e)))

          ;; detach in all cases where we are called
          ;; we must ensure we're still attached since ec-abort
          ;; can be called outside of a context (via destroy-editing-context)
          (when (slot-boundp ec 'stores)
	    (dolist (store stores)
	      (when (attached-to-transaction? store transaction)
		(detach-transaction store transaction))))

          ;; only do rollback if we did not succeed
          ;; in executing our changes
          (when (transaction-active? transaction)
            ;; (cmsg "Transaction active: rolling back w/ stack: ~s" (slot-value transaction 'stack))
            (dolist (action (slot-value transaction 'stack))
              (when (eql :rollback (odcl::tx-event-type action))
                (let ((key (odcl::tx-event-key action))
                      (data (odcl::tx-event-data action)))
                  (restore ec key data))))))
      ;; ensure transaction is set inactive
      (when (transaction-active? transaction)
        (publish-event 'transaction-abort :transaction transaction)
        (set-transaction-inactive transaction))
      (when *fascistic-context*
        (unless (= 0 (slot-value cache 'mcount))
          (if *recover-context*
              (progn
                (ec-reset ec)
                (error "Marked entries remained in cache on ec-exit (flushed)"))
              (error "Marked entries remain in cache on ec-exit")))))))

(defun restore (context key data)
  ;; (cmsg "indexed-store abort: roll back object identified by ~s to state ~s" key data)
  (with-slots (cache)
    context
    (if data
        (progn
          (if-bind (obj (ec-cache-get context key))
              (instance-restore obj data)
              (error "instance-restore called on object with key ~A, not in cache." key)))
        (cache-delete cache key))))

(defun ec-invalidate (instance &key (ec *default-editing-context*))
  (when-bind (key (instance-key instance))
    (cache-delete (slot-value ec 'cache) key)))

(defmethod editing-context-execute-changes ((self editing-context) &aux phase-1)
  (with-slots (stores cache transaction)
    self
    (unwind-protect
         (progn
           (dolist (store stores)
             (push (commit-phase-1 store transaction) phase-1))
           (run-precommit-hooks self))
      (while phase-1
        (funcall (pop phase-1) :commit-phase-2))
      (dolist (elt (slot-value transaction 'stack))
        (case (tx-event-type elt)
          (:edit
           (cache-delete cache (tx-event-key elt)))
          (:delete
           ;; remove it from the ECs cache after it has been deleted
           (cache-delete cache (tx-event-key elt)))
          (:rollback
           nil)))
      (setf (slot-value transaction 'stack) nil)
      (run-commit-hooks self))
    (while phase-1
      (funcall (pop phase-1) :rollback))))

(defmethod get-unstoreable (x y)
  (declare (ignore x y))
  nil)

(defun instance-by-key (key &key (from-cache t) (from-store t) (context *default-editing-context*))
  (assert (typep key 'instance-key))
  (or (get-unstoreable (car key) (cdr key))
      (progn
        (when (and from-cache context)
          (multiple-value-bind (object status)
              (ec-cache-get context key)
            (when object
              (return-from instance-by-key
                (unless (eql status :deleted)
                  object)))))
        (when from-store
          (when-bind (store (get-context-store-by-class context (car key)))
            (get-store-instance-by-key store key))))))

(defun describe-ec-object (key &key (context *default-editing-context*))
  (when (get-unstoreable (car key) (cdr key))
    (cmsg "Object not storeable.")
    (return-from describe-ec-object))
  (multiple-value-bind (object status)
      (ec-cache-get context key)
    (when object
      (return-from describe-ec-object
        (cmsg "Known by cache, with status ~s" status)))))
