;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: odcl -*-
;;; $Id: search-forms.lisp,v 1.25 2003/04/23 20:33:50 craig Exp $
;;;
;;; Copyright (c) 2000 - 2003 onShore Development, Inc.

(in-package :odcl)

;; ------------------------------------------------------------
;; Search Form Specifications

(defvar *search-form-specifications* (make-hash-table :test #'equal))

(defun %search-form-spec-hash (class spec-name)
  (list class spec-name))

;; (def-search-spec resource :catalog-search-set
;;     ((:isbn :default t)
;;      (:record-created "Creation Date")
;;      (:director)
;;      (:rating:required :input-type 'media-rating)))

(defmacro def-search-spec (propertied-class spec-name &body spec)
  "Define a set of psheet property specs on PROPERTIED-CLASS under
SPEC-NAME.  search form specification are lists destructured as
follows: (name &key default caption input-type) NAME is the property
name.  If DEFAULT is true, than the property is added to the displayed
criterion list automatically. CAPTION and INPUT-TYPE default to the
values returned from GET-PROPERTY but can be overriden."
  `(setf (gethash (%search-form-spec-hash (quote ,propertied-class) ,spec-name)
          *search-form-specifications*)
    (quote ,spec)))


(defmacro add-terms-to-search-spec (propertied-class spec-name &body spec)
  "Add search terms to a SEARCH-SPEC."
  (let ((spec (merge-search-form-specs (class-search-form-spec propertied-class spec-name)
                                       spec)))
    `(setf (gethash (%search-form-spec-hash (quote ,propertied-class) ,spec-name)
            *search-form-specifications*)
      (quote ,spec))))
  
(defmethod class-search-form-spec (class spec-name)
  "Return the psheet spec for an instance with the given name."
  (copy-list (gethash (%search-form-spec-hash class spec-name)
                      *search-form-specifications*)))

(defun generate-search-form-set (class specs)
  (let ((proplist nil))
    (dolist (property-spec specs)
      (destructuring-bind (name &key default caption invert input-type required def-value &allow-other-keys)
          property-spec
        (destructuring-bind (&optional reader writer type-def caption-def)
            (property-metadata class name)
          (declare (ignore reader writer))
          (unless caption
            (setq caption caption-def))
          (unless input-type
            (setq input-type type-def))
          (push (list name default caption invert input-type required def-value) proplist))))
    (nreverse proplist)))

(defmethod class-search-form-set (class set-name)
  (let ((pspec (class-search-form-spec class set-name)))
    (generate-search-form-set class pspec)))

;; fifth = "required"
(defmethod required? (field-spec)
  (sixth field-spec))

;; second = "default"
(defmethod default? (field-spec)
  (second field-spec))

(defmethod search-form-default-fields (set)
  (remove-if-not #'default? set))

(defmethod search-form-required-fields (set)
  (remove-if-not #'required? set))

;; ------------------------------------------------------------
;; Result Set Specifications

(defvar *result-set-specifications* (make-hash-table :test #'equal))

(defun %result-set-spec-hash (class spec-name)
  (list class spec-name))

(defmacro def-result-spec (propertied-class spec-name &body spec)
  "Define a set of result set specs on PROPERTIED-CLASS under
SPEC-NAME.  search form specifications are lists structured as
follows: (name &key default width sort-on caption display-type) NAME
is the property name.  If SORT-ON is a symbol, than the property can
be sorted on, and uses the SORT-ON value as the comparison function
taking two arguments. CAPTION and DISPLAY-TYPE default to the values
returned from GET-PROPERTY, but can be overridden. Ordering is
important."
  `(setf (gethash (%result-set-spec-hash (quote ,propertied-class) ,spec-name)
          *result-set-specifications*)
    (quote ,spec)))

(defmethod object-result-set-spec (self spec-name)
  "Return the result set spec for INSTANCE with given name."
  (gethash (%result-set-spec-hash (type-of self)
				  spec-name)
	   *result-set-specifications*))

(defmethod result-set-spec-for-class (class spec-name)
  (gethash (%result-set-spec-hash class spec-name)
	   *result-set-specifications*))

(defmethod generate-result-set-for-class (class specs)
  (let ((proplist nil))
    (dolist (property-spec specs)
      (destructuring-bind (name &key default keyfield width sort-default sort-function caption display-type
                                summary selectable &allow-other-keys)
	  property-spec
	(let* ((meta (property-metadata class name))
	       (type-def (nth 2 meta))
	       (caption-def (nth 3 meta)))
	  (unless caption
	    (setq caption caption-def))
	  (unless display-type
	    (setq display-type type-def))
	  (push (list name
		      :default default
		      :caption caption
		      :keyfield keyfield
		      :width width
		      :sort-default sort-default
                      :sort-function sort-function
                      :selectable selectable
                      :summary summary
		      :display-type display-type) proplist))))
    (nreverse proplist)))

(defun generate-result-set (instance specs &optional set-criterion)
  (declare (ignore set-criterion))
  (let ((proplist nil))
    (dolist (property-spec specs)
      (destructuring-bind (name &key default keyfield width
                                property-args
                                sort-default sort-function
                                caption display-type
                                selectable summary
                                &allow-other-keys)
	  property-spec
	(multiple-value-bind (value type-def caption-def)
            (if property-args
                (get-property-n instance name property-args)
                (get-property instance name))
	  (unless caption
	    (setq caption caption-def))
	  (unless display-type
	    (setq display-type type-def))
	  (push (list name value
		      :default default
		      :width width
		      :keyfield keyfield
		      :sort-default sort-default
                      :sort-function sort-function
		      :caption caption
                      :selectable selectable
                      :summary summary
		      :display-type display-type) proplist))))
    (nreverse proplist)))

(defmethod result-set-default-fields (set)
  (remove-if-not #'(lambda (fs)
		     (destructuring-bind (name &key default &allow-other-keys)
			 fs
                       (declare (ignore name))
		       default))
		 set))

(defvar *export-format-set-specifications* (make-hash-table :test #'equal))

(defmacro def-export-format-spec (name &body spec)
  "Define a set of result set specs on PROPERTIED-CLASS under
SPEC-NAME.  search form specification are lists destructured as
follows: (name &key default width sort-on caption display-type) NAME
is the property name.  If SORT-ON is a symbol, than the property can
be sorted on, and uses the SORT-ON value as the compairson func taking
two arguments. CAPTION and DISPLAY-TYPE default to the values returned
from GET-PROPERTY but can be overriden. Ordering is important."
  `(setf (gethash ,name *export-format-set-specifications*)
	 (quote ,spec)))

(defun export-format-spec-named (spec-name)
  (gethash spec-name *export-format-set-specifications*))

(defun merge-search-form-specs (&rest sets &aux newset)
  (dolist (set (copy-list sets))
    (dolist (prop set)
      (destructuring-bind (name &rest args)
          prop
        (if-bind (spec (car (member name newset :key #'car)))
          (dolist (pair (partition-pairwise args
                                            #'(lambda (x y)
                                                (declare (ignore y))
                                                (keywordp x))))
            (unless (member (car pair) (cdr spec))
              (setf (cdr (assoc (car spec) newset))
                    (append (cdr spec) pair))))
          (push prop newset)))))
  (nreverse newset))
