/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2000-2004 The Caudium Group
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: camas_addressbook_global.pike,v 1.1.2.1 2004/01/23 18:45:01 vida Exp $
 */

#include <module.h>
#include <camas/globals.h>
#include <camas/addressbook.h>
inherit "module";

constant cvs_version="$Id: camas_addressbook_global.pike,v 1.1.2.1 2004/01/23 18:45:01 vida Exp $";
constant module_type = MODULE_PROVIDER;
constant module_name = "CAMAS: Addressbook global";
constant module_doc  = "This module is a Global AddressBook. This means "
                       "that entries in this address book will be "
                       "available for all users of this Camas server.";
constant module_unique = 1;
constant thread_safe = 1;

// the contents of this addressbook
array(mapping(int:array(string))) content;
#if constant(thread_create)
object global_lock = Thread.Mutex();
#endif

// the order in which we store attributes 
constant attributes = ({ 
  AD_MAIL,
  AD_NAME,
  AD_SURNAME,
  AD_NICKNAME,
  AD_HOMEPHONE,
  AD_WORKPHONE,
  AD_TITLE,
  AD_COMPANY,
  AD_DEPARTMENT,
  AD_HOMEPOSTALADDRESS,
  AD_WORKPOSTALADDRESS,
  AD_WORKCITY,
  AD_WORKCOUNTRY,
  AD_WORKSTATE,
  AD_PAGER,
  AD_MOBILE,
  AD_URI,
  AD_DESCRIPTION,
  AD_POSTALCODE
});

void create()
{
#ifdef CAMAS_DEBUG
  defvar("debug",0,"Debug",TYPE_FLAG,"Debug the call / errors into Caudium "
         "error log ?");
#endif
  defvar("myname", "Global addressbook", "Name of this addressbook",
         TYPE_STRING, "The name of this addressbook that will appear to the user");

  defvar("isdefault", 0, "Is default ?", TYPE_FLAG,
         "If set to yes, users will have this addressbook selected when they "
         "enter the address book screen the first time during their session.");
  // Replace this one in TYPE_FILE ?
  defvar("globaladdressbookfile", "", "File", TYPE_STRING,
         "A file containing global addressbook addresses in LDIF format");
}

void start(int cnt, object conf)
{
  load_global_addressbook();
}

string status()
{
  string out = "Number of entries loaded from " + QUERY (globaladdressbookfile) + ": ";
  out += sizeof(content);
  return out;
}

// Loads the address book from configinterface or file
private void load_global_addressbook() {
#if constant(thread_create)
  object lock = global_lock->lock();
#endif
  mixed err = catch {
    string fn = QUERY (globaladdressbookfile);
    string filebook = "";
    if (sizeof(fn) > 0)
      filebook=Stdio.read_bytes(fn) || "";
    if(!sizeof(filebook))
      throw(({ "Can't open file " + fn + " for reading\n", backtrace() }));
    array(mapping(int:string)) entries = CAMAS.AddressBook2.LDIF.imp(filebook);
    content = map(entries, lambda(mapping(int:string) attrs) {
       mapping(int:array(string)) newattrs = ([ ]);
       foreach(indices(attrs), int a)
         newattrs[a] = ({ attrs[a] });
       return newattrs; });
    if(!content)
      content = ({ });
    CDEBUG(sprintf("content=%O\n", content));
  };
#if constant(thread_create)
  destruct(lock);
#endif
  if(err)
    report_error("Error in camas_addressbook_global: can't import file "
     "due to the following error: %s\n", describe_backtrace(err));
}
 
string query_provides()
{
  return("camas_addressbook");
}

void stop()
{
}

/*
 * What we provide here
 */

//
//! method: int version(void)
//!  Give the CAMAS_PREFERENCES api version
//!  supported by the module
//! returns:
//!  the version of the API
//
int version()
{
  return 1;
}

//! method: string get_name(void)
//!  get the name of this addressbook
string get_name()
{
  return QUERY(myname);
}

//! method: int writeable(void)
//!  Is this addressbook writeable ?
int writeable()
{
  return 0;
}

//! method: int isdefault(void)
//!  Is this addressbook the default one for users ?
int isdefault()
{
  return QUERY(isdefault);
}

//! method: array(string) get_all_attributes()
//!  Returns every attributes supported by this addressbook
array(string) get_all_attributes()
{
  return attributes;
}

//! method: int|string get_lastprotocol_error()
//!  Returns the last protocol error if any or 0 otherwise
//!  In this module there is no protocol so this function always returns 0
int|string get_lastprotocol_error()
{
  return 0;
}

//
//! method: array(mapping(int:string)) get_all(object id)
//!  Returns every entries on this addressbook, maybe disable by administrator/protocol
//!  Returns an empty string if there are no entries
array(mapping(int:array(string))) get_all(object id)
{
  return content;
}

//
//! method: array(mapping(int:array(string))) get(mapping(int:string) variables2search, object id, void|int or_search, void|int wildcardsearch)
//!   get every entries matching the given values in the variables2search mapping
//! arg: mapping(int:string) variables2search
//!  A mapping indexed by internal attributes value (internals means name will be replaced by cn later) and 
//!  containing for each key the value it must have. If several attributes are provided we simply make an AND
//! arg: object id
//!  Caudium id object
//! arg: void|int or_search
//!  Is this search and'ed or or'ed ?. By default attributes are and'ed
//! arg: void|int wildcardsearch 
//!  For a given attribute, do we search words matching a part of the search filter (wildcard type search) 
//!  or only the words matching ?
//! returns:
//!  Every entries present on this addressbook that match the given criterias
//!  An empty string if none match
array(mapping(int:array(string))) get(mapping(int:string) variables2search, object id, void|int or_search, 
    void|int wildcardsearch)
{
  return CAMAS.AddressBook2.get(variables2search, id, get_all(id), or_search, wildcardsearch);
}

/*
 * If you visit a file that doesn't contain these lines at its end, please     
 * cut and paste everything from here to that file.                            
 */                                                                            

/*
 * Local Variables:                                                            
 * c-basic-offset: 2                                                           
 * End:                                                                        
 *                                                                             
 * vim: softtabstop=2 tabstop=2 expandtab autoindent formatoptions=croqlt smartindent cindent shiftwidth=2
 */

