/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2000-2004 The Caudium Group
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: FolderTools.pmod,v 1.28.2.2 2004/05/10 15:42:19 vida Exp $
 */

//! file: FolderTools.pmod
//!  Folder tools for CAMAS
//! cvs_version: $Id: FolderTools.pmod,v 1.28.2.2 2004/05/10 15:42:19 vida Exp $

#include <module.h>
#include <camas/msg.h>          // Language macros
#include <camas/language.h>     // Language sym names M_*
#include <camas/globals.h>

//! method: int is_a_root_mailbox(array mailbox, string mailpath, int depth)
//!  Checks if a mailbox is a root mailbox given the mailpath and the depth
//! returns:
//!  1 or 0 wether the mailbox is a root mailbox or not
//! arg: array mailbox
//!  The mailbox to test
//! arg: string mailpath
//!  The mailpath to check if the mailbox is a root mailbox for
//! arg: int depth
//!  The depth of the root mailboxes for the given mailpath
static inline private int is_a_root_mailbox(array mailbox, string mailpath, int depth)
{
  // if the mailbox differs from the depth, it can't be a root mailbox
  if(sizeof(mailbox[MB_HIERARCHY_IDX]) != depth)
    return 0;

  // in case of mailpath empty, any folder should be an mailbox
  if(strlen(mailpath) == 0)
    return 1;
 
  // in case the first element equals the mailpath minus the last character, it is a mailbox
  // (mailpath usually contains the separator as a last character which is not part of the mailpath)
  if(mailbox[MB_HIERARCHY_IDX][0] == mailpath[..sizeof(mailpath)-2])
    return 1;

  // in case the first element equals mailpath, it is a mailbox
  if(mailbox[MB_HIERARCHY_IDX][0] == mailpath)
    return 1;

  // in case the only one element equals INBOX, it is a mailbox
  // (INBOX is a keyword in IMAP - see RFC2060 5.1)
  if(mailbox[MB_HIERARCHY_IDX][0] == "INBOX" && sizeof(mailbox[MB_HIERARCHY_IDX])==1)
    return 1;

  return 0;
}

//! method: int is_inbox (array mailbox, object sessobj)
//! returns:
//!  1 or 0 wether the mailbox is the inbox (personal) root
//! arg: array mailbox
//!  Current mailbox
//! example rxml:
//!  INBOX       true
//!  INBOX.iTeam false
//!  shared      false
inline int is_inbox (array mailbox, object sessobj)
{
  if(mailbox[MB_HIERARCHY_IDX][0] == "INBOX" && sizeof(mailbox[MB_HIERARCHY_IDX])==1)
  {
    FOLDERS_DEBUG("mailbox " + mailbox[MB_FOLDERNAME_IDX] + " is inbox mailbox\n");
    return 1;
  }
  FOLDERS_DEBUG("mailbox " + mailbox[MB_FOLDERNAME_IDX] + " is not inbox mailbox\n");
  return 0;
}

//! method: int is_in_inbox (array mailbox, object sessobj)
//! returns:
//!  1 or 0 wether the mailbox is the inbox (personnal) hierarchy
//! arg: array mailbox
//!  Current mailbox
//! arg: object sessobj
//!  The current Camas session 
//! example rxml:
//!  INBOX       true
//!  INBOX.iTeam true
//!  shared      false
inline int is_in_inbox (array mailbox, object sessobj)
{
  // if we are in INBOX, the it's an INBOX type folder of course
  // or
  // if it contains the INBOX namespace
  // and
  // it's not a shared a folder
  // (Cyrus IMAP set sharedpath to "" in its default configuration
  // so we need to check !sizeof(sessobj->sharedpath) as well)
  if (is_inbox(mailbox, sessobj) 
      || (search(mailbox[MB_FOLDERNAME_IDX], sessobj->mailpath) == 0
           && (!sizeof(sessobj->sharedpath) || search(mailbox[MB_FOLDERNAME_IDX],sessobj->sharedpath) != 0)))
  {
    FOLDERS_DEBUG("mailbox " + mailbox[MB_FOLDERNAME_IDX] + " is in inbox\n");
    return 1;
  }
  FOLDERS_DEBUG("mailbox " + mailbox[MB_FOLDERNAME_IDX] + " is not in inbox\n");
  return 0;
}

//! method: int is_shared (array mailbox, object sessobj)
//! returns:
//!  1 or 0 wether the mailbox is the shared root
//! arg: array mailbox
//!  Current mailbox
//! arg: object sessobj
//!  The current Camas session 
//! example (shareddepth=2):
//!  INBOX          false
//!  INBOX.iTeam    false
//!  shared         false
//!  shared.foo     true
//!  shared.foo.bar false
inline int is_shared (array mailbox, object sessobj)
{
  if(stringp(sessobj->sharedpath)
     && is_a_root_mailbox(mailbox, sessobj->sharedpath, sessobj->shareddepth)
     && (!sizeof(sessobj->mailpath) || search(mailbox[MB_FOLDERNAME_IDX],sessobj->mailpath) != 0))
  {
    FOLDERS_DEBUG("mailbox " + mailbox[MB_FOLDERNAME_IDX] + " is shared mailbox\n");
    return 1;
  }
  FOLDERS_DEBUG("mailbox " + mailbox[MB_FOLDERNAME_IDX] + " is not shared mailbox\n");
}

//! method: int is_in_shared (array mailbox, object sessobj)
//! returns:
//!  1 or 0 wether the mailbox is in the shared hierarchy and belong to a shared mailbox
//! arg: array mailbox
//!  Current mailbox
//! arg: object sessobj
//!  The current Camas session 
//! example (for shareddepth==2)
//!  INBOX          false
//!  INBOX.iTeam    false
//!  shared         false
//!  shared.foo     true 
//!  shared.foo.bar true
inline int is_in_shared (array mailbox, object sessobj)
{
  if (stringp(sessobj->sharedpath) 
      && search(mailbox[MB_FOLDERNAME_IDX],sessobj->sharedpath)==0 
      && (!sizeof(sessobj->mailpath) || search(mailbox[MB_FOLDERNAME_IDX],sessobj->mailpath) != 0)
      && sizeof(mailbox[MB_HIERARCHY_IDX])>=sessobj->shareddepth)
  {
    FOLDERS_DEBUG("mailbox " + mailbox[MB_FOLDERNAME_IDX] + " is in shared\n");
    return 1;
  }
  FOLDERS_DEBUG("mailbox " + mailbox[MB_FOLDERNAME_IDX] + " is not in shared\n");
  return 0;
}

//! method: int is_folder(array mailbox, string foldername)
//!  Check if the mailbox is the folder name foldername
//! returns:
//!  1 or 0 wether the actual folder is foldername or not.
//! arg: array mailbox
//!  Current mailbox
//! arg: string folder
//!  CAMAS folder name
inline int is_folder(array mailbox, string foldername)
{ 
  if(sizeof(foldername) && has_value(mailbox[MB_FOLDERNAME_IDX], foldername))
  {
    FOLDERS_DEBUG("mailbox " + mailbox[MB_FOLDERNAME_IDX] + " is a folder named " + foldername + "\n");
    return 1; 
  }
  FOLDERS_DEBUG("mailbox " + mailbox[MB_FOLDERNAME_IDX] + " is not a folder named " + foldername + "\n");
  return 0; 
}

//! method: int is_draft(array mailbox, string draftsfolder)
//!  Check if the mailbox is a draft folder or not
//! returns:
//!  1 or 0 wether the mailbox is a draft folder or not.
//! arg: array mailbox
//!  Current mailbox
//! arg: string draftsfolder
//!   the draftsfolder string
inline int is_draft(array mailbox, string draftsfolder)
{
  return is_folder(mailbox, draftsfolder);
}

//! method: int is_sent(array mailbox, string sentfolder)
//!  Check if the mailbox is a sent folder or not
//! returns:
//!  1 or 0 wether the mailbox is a sent folder or not.
inline int is_sent(array mailbox, string sentfolder)
{
  return is_folder(mailbox, sentfolder);
}

//! method: int is_answered(array mailbox, string answeredfolder)
//!  Check if the mailbox is an answered folder or not
//! returns:
//!  1 or 0 wether the mailbox is an answered folder or not.
inline int is_answered(array mailbox, string answeredfolder)
{
  return is_folder(mailbox, answeredfolder);
}

//! method: int is_trash(array mailbox, string trashfolder)
//!  Check if the mailbox is a trash folder or not
//! returns:
//!  1 or 0 wether the mailbox is a trash folder or not.
inline int is_trash(array mailbox, string trashfolder)
{
  return is_folder(mailbox, trashfolder);
}

//! method: int is_prefsbox(array mailbox, string prefsbox)
//!  Check if the mailbox is a prefs folder or not
//! returns:
//!  1 or 0 wether the mailbox is a preferences folder or not
inline int is_prefsbox(array mailbox, string prefsbox)
{
  return is_folder(mailbox, prefsbox);
}

//! method: int is_a_folder(array mailbox, object sessobj)
//! returns:
//!  1 or 0 wether the current folder is a folder or not
//! arg: array mailbox
//! arg: object sessobj
//!  The current Camas session 
inline int is_a_folder(array mailbox, object sessobj)
{
  if(is_in_shared(mailbox, sessobj) || is_in_inbox(mailbox, sessobj))
    return 1;
  return 0;
}

//! method: int is_a_special_folder(array mailbox, object sessobj)
//! returns:
//!  1 or 0 wether the current folder is a special folder or not.
//!  special folders are: draft, sent, answered and trash
//! arg: array mailbox
//! arg: object sessobj
//!  The current Camas session 
inline int is_a_special_folder(array mailbox, object sessobj)
{
  return (is_draft(mailbox,sessobj->draftsfolder)
    || is_sent(mailbox,sessobj->sentfolder)
    || is_answered(mailbox,sessobj->answeredfolder) 
    || is_trash(mailbox,sessobj->trashfolder)
    || is_prefsbox(mailbox, sessobj->prefsbox));
}

//! method: int is_a_top_level_mailbox(array mailbox, object sessobj)
//! returns:
//!  Return true if the mailbox is a top level mailbox like:
//!    - INBOX for a personnal mailbox
//!    - shared.some.mailbox if the mailbox is a shared mailbox with a shareddepth of 2
//! arg: array mailbox
//!  The current mailbox
//! arg: object sessobj
//!  The current Camas session 
inline int is_a_top_level_mailbox(array mailbox, object sessobj)
{
  if(is_inbox(mailbox, sessobj) || is_shared(mailbox, sessobj))
  {
    FOLDERS_DEBUG("mailbox " + mailbox[MB_FOLDERNAME_IDX] + " is a top level mailbox\n");
    return 1;
  }
  FOLDERS_DEBUG("mailbox " + mailbox[MB_FOLDERNAME_IDX] + " is not a top level mailbox\n");
  return 0;
}

//!   
//! method: string translate_frommboxindex (object id, int index)
//!  Get the "translation" of the name of the box from Current language
//!  module (using sessobj). If the name is not on language module, then
//!  unmodified mailbox name is returned.
//! returns:
//!  Clear-text name for IMAP inbox from current language module.
//! arg: object id
//!  The Caudium id object
//! arg: int index
//!  The index in the mailboxes mapping of the mailbox to translate
inline string translate_frommboxindex (object id, int index)
{
  return translate_frommbox(id, CSESSION->mailboxes[index]);
}

//!   
//! method: string translate_frommbox (object id, array mbox)
//!  Get the "translation" of the name of the box from Current language
//!  module (using sessobj). If the name is not on language module, then
//!  unmodified mailbox name is returned.
//! returns:
//!  Clear-text name for IMAP inbox from current language module.
//! arg: object id
//!  The Caudium id object
//! arg: array mbox
//!  The mailbox to translate
inline string translate_frommbox (object id, array mbox_a)
{
  string mbox = mbox_a[MB_HIERARCHY_IDX][-1];

  if (mbox == "INBOX")
    return MSG(M_INBOX);

  if (is_in_inbox(mbox_a, CSESSION))
  {
    if (mbox == CSESSION->trashfolder)
      return MSG(M_BOXTRASH);

    if (mbox == CSESSION->sentfolder)
      return MSG(M_BOXSENTMAIL);

    if (mbox == CSESSION->draftsfolder)
      return MSG(M_BOXDRAFTS);

    if (mbox == CSESSION->answeredfolder)
      return MSG(M_BOXANSWERED);
  }
  else if (is_in_shared(mbox_a, CSESSION))
  {

    if (mbox ==  CSESSION->trashfolder)
      return MSG(M_SHARED_BOXTRASH);

    if (mbox == CSESSION->sentfolder)
      return MSG(M_SHARED_BOXSENTMAIL);

    if (mbox == CSESSION->draftsfolder)
      return MSG(M_SHARED_BOXDRAFTS);

    if (mbox == CSESSION->answeredfolder)
      return MSG(M_SHARED_BOXANSWERED);
  }

  return mbox;
}

//! method: string getmailboxroot(array mailbox, object sessobj)
//!  Gives the path of the mailbox for the current imap path
//! returns:
//!  INBOX.               for INBOX.some.nested.folders
//!  shared.mailbox.      for shared.mailbox.have.folders.too if shareddepth = 2
//!  shared.mailbox.have. for shared.mailbox.have.folders.too if shareddepth = 3
//! arg: array mailbox
//!  The mailbox array
//! arg: object sessobj
//!  The current session object
inline string getmailboxroot(array mailbox, object sessobj)
{
  if(is_in_inbox(mailbox, sessobj))
  {
    FOLDERS_DEBUG("mailbox root for " + mailbox[MB_FOLDERNAME_IDX] + " is " + 
      sessobj->mailpath + "\n");
    return sessobj->mailpath;
  }
  // is it a shared folder ?
  if(stringp(sessobj->sharedpath) 
      && search(mailbox[MB_FOLDERNAME_IDX],sessobj->sharedpath)==0 
      && search(mailbox[MB_FOLDERNAME_IDX],sessobj->mailpath) != 0)
  {
    // does the folder has more sub directory than shareddepth ?
    if(sizeof(mailbox[MB_HIERARCHY_IDX])>=sessobj->shareddepth)
    {
      string mailboxrootpath = "";
      for(int i = 0; i < sessobj->shareddepth; i++)
      {
        if(i > 0)
        {
          mailboxrootpath += getseparator(mailbox);
        }
       mailboxrootpath += mailbox[MB_HIERARCHY_IDX][i];
      }
      FOLDERS_DEBUG("mailbox root for " + mailbox[MB_FOLDERNAME_IDX] + " is " +
        mailboxrootpath + getseparator(mailbox) + "\n");
      return mailboxrootpath + getseparator(mailbox);
    }
    // it's a root shared folder
    else
      return sessobj->sharedpath;
  }
  // special case for "shared" fake mailbox with courier-imap
  if(mailbox[MB_FOLDERNAME_IDX] == "shared")
    return "shared";
  // default case
  report_warning("CAMAS.FolderTools.getmailboxroot: Current path " + 
     mailbox[MB_HIERARCHY_IDX][0]+ " can't be processed.\n");
  FOLDERS_DEBUG("mailbox root for " + mailbox[MB_FOLDERNAME_IDX] + " is INBOX\n");
  return "INBOX";
}

//! method: string string getseparator(array mailbox)
//!  Gives the current mailbox separator of the mailbox
//!  usual values include "." (Courier-IMAP, Cyrus) or "/" (WU)
//! returns:
//!  The mailbox separator for the current mailbox (for now only personnal and 
//!  and personnal folders).
//! arg: array mailbox
//!  The mailbox array
inline string getseparator(array mailbox)
{
  return mailbox[MB_SEPARATOR_IDX];
}

//! method: string mb_displayname_from_name(string mbox, array mailboxes)
inline string mb_displayname_from_name(string mbox, array mailboxes) {
  foreach(mailboxes,array mbox_a)
  if(mbox_a[MB_FOLDERNAME_IDX]==mbox)
    return mbox_a[MB_DISPLAYNAME_IDX];
  return "";
}

//! method: list_folders(object id, void|int goaccrossmailboxes, void|int showspecialfolders, void|int onlymailboxes, void|int showsent)
//!  returns a list of all folders
//! input :
//!  - id object
//!  - boolean acccrossmailboxes
//!   If false, only lists folders in current mailbox
//!   If true, list all folders
//!  - boolean specialfolders
//!   If false, don't list special folders which commonly are (depending of your CAMAS config):
//!     - sent-mail
//!     - answered
//!     - Trash
//!     - Drafts
//!     - imhoprefs
//!  - boolean onlymailboxes
//!   If true, shows only root mailboxes, not subfolders
//!  - boolean showsent
//!   If true, will show sent folder, even if showspecielfolders is false
//! output : array containing matching id of desired folders
//! note: by default, don't display special folders (ie trash, sent, answered, drafts)
array list_folders(object id, void|int goaccrossmailboxes, void|int showspecialfolders, void|int onlymailboxes,  void|int showsent)
{
  object camas_main = CAMAS_MODULE;
  array  foldersok       = ({ });   // Array of ids of folders we want to return
  
  if (!objectp(camas_main))
  {
    FOLDERS_DEBUG("module camas_main is not present");
    return(0);
  }

  zero_type(goaccrossmailboxes) ? 1 : goaccrossmailboxes;
  zero_type(showspecialfolders) ? 0 : showspecialfolders;
  zero_type(onlymailboxes)      ? 0 : onlymailboxes;
  zero_type(showsent)           ? 0 : showsent;

  int i    = 0,  // Loop counter
      add  = 0;  // Decide wether or not the folder is added to the list

  FOLDERS_DEBUG(sprintf("listFolders begin, mailbox=%O\n", CSESSION->mailbox));

  for(i=0; i<sizeof(CSESSION->mailboxes); i++)
  {
    add = 0;
    array(string|int|array(string)) currentmailbox = CSESSION->mailboxes[i];

    FOLDERS_DEBUG(sprintf("CSESSION->mailboxes[%d]=%O\n", i, currentmailbox));
    if(!goaccrossmailboxes)
    {
      FOLDERS_DEBUG("don't go across mailboxes\n");
      if(CSESSION->mailbox[MB_HIERARCHY_IDX][0]== currentmailbox[MB_HIERARCHY_IDX][0])
      {
        FOLDERS_DEBUG("current mailbox has the same root as CSESSION->mailboxes[i]\n");
        if(is_in_inbox(currentmailbox, CSESSION))
        {
          FOLDERS_DEBUG("It's a personnal mailbox\n");
          add = 1;
        }
        else
        {
          if(is_in_shared(currentmailbox, CSESSION) &&
              search(currentmailbox[MB_FOLDERNAME_IDX] +
                getseparator(currentmailbox),
                getmailboxroot(CSESSION->mailbox, CSESSION)) == 0)
          {
              add = 1;
              FOLDERS_DEBUG("it's a shared mailbox and we will display it\n");
          }
        }
      }
    }
    else
    {
      FOLDERS_DEBUG("go across mailboxes\n");
      if(is_a_folder(currentmailbox, CSESSION))
      {
        FOLDERS_DEBUG("It's a folder\n");
        add = 1;
      }
    }

    if(add && !showspecialfolders)
      if(is_a_special_folder(currentmailbox, CSESSION))
      {
        add = 0;
        FOLDERS_DEBUG("it's a special folder and we won't display it\n");
      }

    if(add && onlymailboxes)
    {
      if(!(
            is_a_top_level_mailbox(currentmailbox, CSESSION)
          ))
      {
          add = 0;
          FOLDERS_DEBUG("it's a top level mailbox and we won't display it\n");
      }
    }

    if(showsent)
    {
      if(is_sent(currentmailbox, CSESSION->sentfolder))
      {
        add = 1;
        FOLDERS_DEBUG("it's a sent mailbox and we will display it\n");
      }
    }

    if(add)
      foldersok += ({ i });
  }

  FOLDERS_DEBUG("listFolders end\n");
  return foldersok;
}

/*
 * If you visit a file that doesn't contain these lines at its end, please
 * cut and paste everything from here to that file.
 */

/*
 * Local Variables:
 * c-basic-offset: 2
 * End:
 *
 * vim: softtabstop=2 tabstop=2 expandtab autoindent formatoptions=croqlt smartindent cindent shiftwidth=2
 */
