<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    html-defs.php                                           */
/* Author:      Paul Waite                                              */
/* Description: Definitions for generic HTML objects such as table,     */
/*              etc. These are objects used to render common HTML       */
/*              entities.                                               */
/*                                                                      */
/*              The classes all inherit RenderableObject, and have a    */
/*              common ancestor defined herein called StylableObject    */
/*              which encapsulates the data and methods to do with      */
/*              the application of styles.                              */
/*                                                                      */
/* ******************************************************************** */
/** @package html */

// ......................................................................
/**
* A renderable tag of some kind. Basically a tag is a language construct
* designed to render at least an identifying name and a value. More
* specific variants might add other properties, and control the way
* the tag is actually rendered.
* @package html
*/
class HTMLtag extends tag {
  // Constructor
  function HTMLtag($name, $value="") {
    $this->tag($name, $value);
  }
  function html() {
    $s = "";
    if ($this->tag_name != "") {
      $s = "<$this->tag_name";
      if (count($this->attributes) > 0) {
        $s .= " ";
        $attrs = array();
        foreach ($this->attributes as $name => $value) {
          $attr = $name;
          if ($value != "") {
            $attr .= "=\"$value\"";
          }
          $attrs[] = $attr;
        }
        $s .= implode(" ", $attrs);
      }
      $s .= ">";
      if ($this->tag_value != "") {
        $s .= $this->tag_value;
        $s .= "</$this->tag_name>";
      }
    }
    return "$s\n";
  }
} // HTMLtag class
// ......................................................................
/**
* StylableObject
* This is a virtual class representing something which can have
* its look changed by applying styles and/or classnames to it.
* @package html
*/
class StylableObject extends RenderableObject {
  /** The ID tag to apply to this object */
  var $id;
  /** The style to apply to this object */
  var $style;
  /** The stylesheet class to apply to this object */
  var $class;
  // ....................................................................
  /**
  * Constructor
  * Create a stylableobject. Sets basic field attributes.
  * @param string  $class_style    Classname or Style setting to apply
  */
  function StylableObject($css="") {
    $this->setcss($css);
  } // StylableObject
  // ....................................................................
  /**
  * Set the ID tag of the object
  * @param string $id The ID to set against this ibject
  */
  function setid($id) {
    $this->id = $id;
  } // setid
  // ....................................................................
  /**
  * Set the class (from stylesheet) for the object
  * @param string $class Class to apply to this object
  */
  function setclass($class) {
    $this->class = $class;
  } // setclass
  // ....................................................................
  /** Clear all existing style settings. This leaves any class
  * settings alone. */
  function clearstyle() {
    $this->style = "";
  } // clearstyle
  // ....................................................................
  /**
  * Append a style to ant exitsing style the object has.
  * @param string $style Style to append to existing.
  */
  function setstyle($style) {
    if (!strstr($this->style, $style)) {
      if ($this->style != "" && substr($this->style, -1) != ";") {
        $this->style .= ";";
      }
      $this->style .= $style;
    }
  } // style
  // ....................................................................
  /**
  * Set class or style for the object. A style is detected by
  * the presence of the colon (:) character in the string.
  * NOTE: styles will be added to the existing style value in
  * cumulative fashion.
  */
  function setcss($css="") {
    // If it contains ":" assume style, else classname..
    if ($css != "") {
      if (strstr($css, ":")) $this->setstyle($css);
      else $this->setclass($css);
    }
    else {
      // Clean 'em out..
      if (isset($this->style)) unset($this->style);
      if (isset($this->class)) unset($this->class);
    }
  } // setcss
  // ....................................................................
  /**
  * This method renders the stylable object attributes. Used in the
  * descendant classes to generate the property tags.
  * $return string Common attribute property tags for current object
  * @access private
  */
  function style_attributes() {
    global $RESPONSE;
    $s = "";
    // Omit the attributes unless we are not part of a reponse,
    // OR we are part of one and the browser is not Netscape..
    if (!isset($RESPONSE) || $RESPONSE->browser != BROWSER_NETSCAPE) {
      if (isset($this->style) && $this->style != "") $s .= " style=\"$this->style\"";
      if (isset($this->id)    && $this->id != "")    $s .= " id=\"$this->id\"";
      if (isset($this->class) && $this->class != "") $s .= " class=\"$this->class\"";
    }
    // Return the HTML..
    return $s;
  } // style_attributes
} // StylableObject class
// ......................................................................
/**
* An HTMLObject is any object which will be rendered in HTML according
* to the basic syntax defined for HTML.
* @package html
*/
class HTMLObject extends StylableObject {
  /** Name of the object */
  var $name;
  /** TAB index of the object */
  var $tabindex;
  /** Access key of the object */
  var $accesskey;
  /** Size of the object */
  var $size;
  /** Width of the object */
  var $width;
  /** Height of the object */
  var $height;
  /** Alignment of the object: left, right, center */
  var $align;
  /** Vertical alignment: top, bottom, middle */
  var $valign;
  /** Title of the object */
  var $title;
  /** ALT text for this object */
  var $alt;
  /** Source URL for this object */
  var $src;
  /** Language code for text content in this object */
  var $lang;
  /** Direction for text - 'LTR' (left-to-right) or 'RTL' (right-to-left) */
  var $langdir;
  /** Traget frame for this object */
  var $target;
  /** Horizontal space around object (pixels) */
  var $hspace;
  /** Vertical space around object (pixels) */
  var $vspace;
  /** Border size (pixels) */
  var $border;
  /** Foreground/text colour of the object */
  var $color;
  /** Background colour of the object */
  var $bgcolor;
  /** Background image url */
  var $bgurl;
  /** Script to call when mouse clicked */
  var $onclick;
  /** Script to call when mouse double-clicked */
  var $ondblclick;
  /** Script to call on key down */
  var $onkeydown;
  /** Script to call on key pressed */
  var $onkeypress;
  /** Script to call on key up */
  var $onkeyup;
  /** Script to call on mouse button down */
  var $onmousedown;
  /** Script to call on mouse button down */
  var $onmousemove;
  /** Script to call on mouse off object */
  var $onmouseout;
  /** Script to call on mouse over object */
  var $onmouseover;
  /** Script to call on mouse button up */
  var $onmouseup;
  /** Script to call onblur */
  var $onblur;
  /** Script to call onfocus */
  var $onfocus;
  /** Script to call onload */
  var $onload;
  /** Script to call onchange */
  var $onchange;
  /** Script to call onselect */
  var $onselect;
  // ....................................................................
  /** User-defined attributes
      @access private */
  var $user_attributes = array();
  // ....................................................................
  /** Text to display in statusbar when mouse over object
      @access private */
  var $linkover_text = "";
  // ....................................................................
  /**
  * Constructor
  * Create a HTMLObject. Sets basic field attributes.
  * @param string  $class_style    Classname or Style setting to apply
  */
  function HTMLObject($css="") {
    $this->StylableObject($css);
  }
  /** Set the name of the object
  @param string $name Name of this object */
  function setname($name)           { $this->name = $name;       }
  /** Set the TAB index of the object (with alias)
  @param integer $ix Tab index for this object */
  function settabindex($ix)         { $this->tabindex = $ix;     }
  /** Alias for settabindex(), deprecated, backward compat.
  @param integer $ix Tab index for this object */
  function set_tabindex($ix)        { $this->settabindex($ix);   }
  /** Set the access key of the object
  @param string $key Access key for this object */
  function setaccesskey($key)       { $this->accesskey = $key;   }
  /** Set the size specifier of the element
  @param integer $size Size of this object */
  function setsize($size)           { $this->size = $size;       }
  /** Set the width specifier of the element
  @param integer $width Width of this object */
  function setwidth($width)         { $this->width = $width;     }
  /** Set the height specifier of the element
  @param integer $height Height of this object */
  function setheight($height)       { $this->height = $height;   }
  /** Set the horizontal alignment eg: left, center, right
  @param string $align Horizontal alignment of this object */
  function setalign($align)         { $this->align = $align;     }
  /** Set the vertical-alignment eg: top, middle, bottom
  @param string $valign Vertical alignment of this object */
  function setvalign($valign)       { $this->valign = $valign;   }
  /** Set the title of this element
  @param string $title Title of this object */
  function settitle($title)         { $this->title = $title;     }
  /** Set the ALT text
  @param string $alt ALT text describing this object */
  function setalt($alt)             { $this->alt = $alt;         }
  /** Set the src URL/relative path
  @param string $src SRC path of this object */
  function setsrc($src)             { $this->src = $src;         }
  /** Set the language for text content
  @param string $lang Language code for this object */
  function setlang($lang)           { $this->lang = $lang;       }
  /** Set the text direction for text content: 'rtl' or 'ltr'
  @param string $langdir Language direction for this object. */
  function setlangdir($langdir)             {
    if (strtolower($langdir) === "rtl") $this->langdir = "RTL";
    else $this->langdir = "LTR";
  }
  /** Set the frame target.
  @param string $target The frame target for this object */
  function settarget($target)       { $this->target = $target;   }
  /** Horizontal spacing (pixels)
  @param integer $px Horizontal spacing in pixels */
  function sethspace($px)           { $this->hspace = $px;       }
  /** Vertical spacing (pixels)
  @param integer $px Vertical spacing in pixels */
  function setvspace($px)           { $this->vspace = $px;       }
  /** Border thickness (pixels)
  @param integer $px Border thickness in pixels */
  function setborder($px)           { $this->border = $px;       }
  /** Set the foreground colour
  @param string $color Foreground colour code */
  function setcolor($color)         { $this->color = $color;     }
  /** Set the background colour
  @param string $bgcolor Background colour code */
  function setbgcolor($bgcolor)     { $this->bgcolor = $bgcolor; }
  /** Set the background image (url)
  @param string $bgurl Background image URL */
  function setbackground($bgurl)    { $this->bgurl = $bgurl;     }
  /** Set the on click script
  @param string $script Name of script to execute */
  function set_onclick($script, $mode=SCRIPT_REPLACE) {
    $this->onclick = inline_script($script, $this->onclick, $mode);
  }
  /** Set the on doubleclick script
  @param string $script Name of script to execute
  @param integer $mode Mode: SCRIPT_REPLACE, SCRIPT_APPEND or SCRIPT_PREFIX */
  function set_ondblclick($script, $mode=SCRIPT_REPLACE) {
    $this->ondblclick = inline_script($script, $this->ondblclick, $mode);
  }
  /** Set the on keydown script
  @param string $script Name of script to execute
  @param integer $mode Mode: SCRIPT_REPLACE, SCRIPT_APPEND or SCRIPT_PREFIX */
  function set_onkeydown($script, $mode=SCRIPT_REPLACE) {
    $this->onkeydown = inline_script($script, $this->onkeydown, $mode);
  }
  /** Set the on keypress script
  @param string $script Name of script to execute
  @param integer $mode Mode: SCRIPT_REPLACE, SCRIPT_APPEND or SCRIPT_PREFIX */
  function set_onkeypress($script, $mode=SCRIPT_REPLACE) {
    $this->onkeypress = inline_script($script, $this->onkeypress, $mode);
  }
  /** Set the on keyup script
  @param string $script Name of script to execute
  @param integer $mode Mode: SCRIPT_REPLACE, SCRIPT_APPEND or SCRIPT_PREFIX */
  function set_onkeyup($script, $mode=SCRIPT_REPLACE) {
    $this->onkeyup = inline_script($script, $this->onkeyup, $mode);
  }
  /** Set the on mousedown script
  @param string $script Name of script to execute
  @param integer $mode Mode: SCRIPT_REPLACE, SCRIPT_APPEND or SCRIPT_PREFIX */
  function set_onmousedown($script, $mode=SCRIPT_REPLACE) {
    $this->onmousedown = inline_script($script, $this->onmousedown, $mode);
  }
  /** Set the on mousemove script
  @param string $script Name of script to execute
  @param integer $mode Mode: SCRIPT_REPLACE, SCRIPT_APPEND or SCRIPT_PREFIX */
  function set_onmousemove($script, $mode=SCRIPT_REPLACE) {
    $this->onmousemove = inline_script($script, $this->onmousemove, $mode);
  }
  /** Set the on mouseout script
  @param string $script Name of script to execute
  @param integer $mode Mode: SCRIPT_REPLACE, SCRIPT_APPEND or SCRIPT_PREFIX */
  function set_onmouseout($script, $mode=SCRIPT_REPLACE) {
    $this->onmouseout = inline_script($script, $this->onmouseout, $mode);
  }
  /** Set the on mouseover script
  @param string $script Name of script to execute
  @param integer $mode Mode: SCRIPT_REPLACE, SCRIPT_APPEND or SCRIPT_PREFIX */
  function set_onmouseover($script, $mode=SCRIPT_REPLACE) {
    $this->onmouseover = inline_script($script, $this->onmouseover, $mode);
  }
  /** Set the on mouseup script
  @param string $script Name of script to execute
  @param integer $mode Mode: SCRIPT_REPLACE, SCRIPT_APPEND or SCRIPT_PREFIX */
  function set_onmouseup($script, $mode=SCRIPT_REPLACE) {
    $this->onmouseup = inline_script($script, $this->onmouseup, $mode);
  }
  /** Set the onblur script
  @param string $script Name of script to execute
  @param integer $mode Mode: SCRIPT_REPLACE, SCRIPT_APPEND or SCRIPT_PREFIX */
  function set_onblur($script, $mode=SCRIPT_REPLACE) {
    $this->onblur = inline_script($script, $this->onblur, $mode);
  }
  /** Set the onfocus script
  @param string $script Name of script to execute
  @param integer $mode Mode: SCRIPT_REPLACE, SCRIPT_APPEND or SCRIPT_PREFIX */
  function set_onfocus($script, $mode=SCRIPT_REPLACE) {
    $this->onfocus = inline_script($script, $this->onfocus, $mode);
  }
  /** Set the onload script
  @param string $script Name of script to execute
  @param integer $mode Mode: SCRIPT_REPLACE, SCRIPT_APPEND or SCRIPT_PREFIX */
  function set_onload($script, $mode=SCRIPT_REPLACE) {
    $this->onload = inline_script($script, $this->onload, $mode);
  }
  /** Set the onchange script
  @param string $script Name of script to execute
  @param integer $mode Mode: SCRIPT_REPLACE, SCRIPT_APPEND or SCRIPT_PREFIX */
  function set_onchange($script, $mode=SCRIPT_REPLACE) {
    $this->onchange = inline_script($script, $this->onchange, $mode);
  }
  /** Set the onselect script
  @param string $script Name of script to execute
  @param integer $mode Mode: SCRIPT_REPLACE, SCRIPT_APPEND or SCRIPT_PREFIX */
  function set_onselect($script, $mode=SCRIPT_REPLACE) {
    $this->onselect = inline_script($script, $this->onselect, $mode);
  }
  //.....................................................................
  /**
  * This defines a key=value pair which will be rendered in the form
  * of '$attrname="$attrvalue"' inside the HTML object tag when rendered.
  * It is provided to cater for attributes not defined explicitly.
  * NOTES: If you omit the attribute value, the attribute will be
  * rendered as just '$attrname', without any value clause.
  * @param string $attrname Name or idenitifier of attribute
  * @param string $attrvalue Value of the attribute (not rendered if omitted)
  */
  function set_attribute($attrname, $attrvalue="???") {
    $this->user_attributes[$attrname] = $attrvalue;
  } // set_attribute
  //.....................................................................
  /**
  * This sets our attributes from another object which is a descendant of
  * HTMLObject. The attributes are specially selected to be only those
  * which are associated with the basic appearance of the object.
  * @param object $obj HTMLObject object instance to inherit attributes from
  */
  function inherit_attributes($obj) {
    if (is_subclass_of($obj, "HTMLObject")) {
      if (isset($obj->style))   $this->setstyle($obj->style);
      if (isset($obj->class))   $this->class   = $obj->class;
      if (isset($obj->width))   $this->width   = $obj->width;
      if (isset($obj->height))  $this->height  = $obj->height;
      if (isset($obj->align))   $this->align   = $obj->align;
      if (isset($obj->valign))  $this->valign  = $obj->valign;
      if (isset($obj->hspace))  $this->hspace  = $obj->hspace;
      if (isset($obj->vspace))  $this->vspace  = $obj->vspace;
      if (isset($obj->border))  $this->border  = $obj->border;
      if (isset($obj->color))   $this->color   = $obj->color;
      if (isset($obj->bgcolor)) $this->bgcolor = $obj->bgcolor;
      if (isset($obj->bgurl))   $this->bgurl   = $obj->bgurl;
      if (isset($obj->lang))    $this->lang    = $obj->lang;
      if (isset($obj->langdir)) $this->langdir = $obj->langdir;
    }
  } // inherit_attributes
  //.....................................................................
  /**
  * Defines the text to show when the mouse moves over the object.
  * @param string $txt Text to show in status area when mouse-over.
  */
  function set_linkover_text($txt="") {
    // Encode so we can allow quotes in this string..
    $this->linkover_text = rawurlencode($txt);
  } // set_linkover_text
  // ....................................................................
  /**
  * Render the full tag. This method renders the given tag including
  * all the attributes, as HTML.
  */
  function taghtml($tag) {
    return "<$tag" . $this->attributes() . ">";
  } // taghtml
  // ....................................................................
  /**
  * Render common field properties
  * This method renders the common object attributes. Used in the
  * descendant classes to generate the property tags.
  * $return string Common attribute property tags for current object
  * @access private
  */
  function attributes() {
    global $RESPONSE;
    $s = $this->style_attributes();
    // Optional linkover statusbar message..
    if ($this->linkover_text != "") {
      $this->set_onmouseover("status=unescape('$this->linkover_text');return true;", SCRIPT_PREFIX);
      $this->set_onmouseout("status='';return true;", SCRIPT_PREFIX);
    }
    // Properties
    if (isset($this->name)    && $this->name != "")    $s .= " name=\"$this->name\"";
    if (isset($this->tabindex))                        $s .= " tabindex=\"$this->tabindex\"";
    if (isset($this->accesskey))                       $s .= " accesskey=\"$this->accesskey\"";
    if (isset($this->size))                            $s .= " size=\"$this->size\"";
    if (isset($this->width)   && $this->width != "")   $s .= " width=\"$this->width\"";
    if (isset($this->height)  && $this->height != "")  $s .= " height=\"$this->height\"";
    if (isset($this->align)   && $this->align != "")   $s .= " align=\"$this->align\"";
    if (isset($this->valign)  && $this->valign != "")  $s .= " valign=\"$this->valign\"";
    if (isset($this->title)   && $this->title != "")   $s .= " title=\"$this->title\"";
    if (isset($this->alt)     && $this->alt != "")     $s .= " alt=\"$this->alt\"";
    if (isset($this->src)     && $this->src != "")     $s .= " src=\"$this->src\"";
    if (isset($this->lang)    && $this->lang != "")    $s .= " lang=\"$this->lang\"";
    if (isset($this->langdir) && $this->langdir != "") $s .= " dir=\"$this->langdir\"";
    if (isset($this->target)  && $this->target != "")  $s .= " target=\"$this->target\"";
    if (isset($this->hspace))                          $s .= " hspace=\"$this->hspace\"";
    if (isset($this->vspace))                          $s .= " vspace=\"$this->vspace\"";
    if (isset($this->border))                          $s .= " border=\"$this->border\"";
    if (isset($this->bgcolor) && $this->bgcolor != "") $s .= " bgcolor=\"$this->bgcolor\"";
    if (isset($this->bgurl)   && $this->bgurl != "")   $s .= " background=\"$this->bgurl\"";
    if (isset($this->color)   && $this->color != "")   $s .= " color=\"$this->color\"";
    // Events
    if (isset($this->onclick))                         $s .= " onclick=\"$this->onclick\"";
    if (isset($this->ondblclick))                      $s .= " ondblclick=\"$this->ondblclick\"";
    if (isset($this->onkeydown))                       $s .= " onkeydown=\"$this->onkeydown\"";
    if (isset($this->onkeypress))                      $s .= " onkeypress=\"$this->onkeypress\"";
    if (isset($this->onkeyup))                         $s .= " onkeyup=\"$this->onkeyup\"";
    if (isset($this->onmousedown))                     $s .= " onmousedown=\"$this->onmousedown\"";
    if (isset($this->onmousemove))                     $s .= " onmousemove=\"$this->onmousemove\"";
    if (isset($this->onmouseout))                      $s .= " onmouseout=\"$this->onmouseout\"";
    if (isset($this->onmouseover))                     $s .= " onmouseover=\"$this->onmouseover\"";
    if (isset($this->onmouseup))                       $s .= " onmouseup=\"$this->onmouseup\"";
    if (isset($this->onblur))                          $s .= " onblur=\"$this->onblur\"";
    if (isset($this->onfocus))                         $s .= " onfocus=\"$this->onfocus\"";
    if (isset($this->onload))                          $s .= " onload=\"$this->onload\"";
    if (isset($this->onchange))                        $s .= " onchange=\"$this->onchange\"";
    if (isset($this->onselect))                        $s .= " onselect=\"$this->onselect\"";
    // Any user-defined attributes..
    foreach ($this->user_attributes as $attrname => $attrvalue) {
      $s .= " $attrname";
      if ($attrvalue != "???") {
        $s .= "=\"$attrvalue\"";
      }
    }
    // Return the HTML..
    return $s;
  } // attributes
} // HTMLObject class
// ......................................................................
/**
* This represents text which is the content of a layout object
* such as a paragraph, a table cell etc. As such is may have an
* associated style and/or classname which is applied with a
* span tag.
* @package html
* @access private
*/
class textcontent extends StylableObject {
  /** Content contained by the object */
  var $content = "";
  // ....................................................................
  /**
  * Set the text content and the style/class for it
  * @param string $content The content to assign to the object
  * @param string $css The style or class to assign to the object
  */
  function textcontent($content="", $css="") {
    $this->StylableObject($css);
    $this->content = $content;
  }
  // ....................................................................
  /** Set the content for the object
  * @param string $content The content to assign to the object
  */
  function setcontent($content) {
    $this->content = $content;
  }
  // ....................................................................
  /** Add to the content for the object
  * @param string $content The content to assign to the object
  */
  function addcontent($content) {
    $this->content .= $content;
  }
  // ....................................................................
  /** Return the HTML for this text content
  * @return string The HTML of this object
  */
  function html() {
    $s = "";
    if (isset($this->style) || isset($this->class) || isset($this->id)) {
      $s .= "<span";
      $s .= $this->style_attributes();
      $s .= ">$this->content</span>";
    }
    else {
      $s .= $this->content;
    }
    // Return the html..
    return $s;
  }
} // textcontent class
// ......................................................................
/**
* tablecell
* This class encapsulates a single cell of a table. As such is is a
* receptacle of content which can be styled. The cell can also optionally
* have permissions defined for it. If defined the rendering as HTML will
* check the logged-in user groups form permission to update content. If
* permission exists, then the content is rendered as a form text field
* instead. The cell variable "cellid" should have been set before this,
* so that the form field is named.
*
* @package html
*/
class tablecell extends HTMLObject {
  /** Optional unique ID for this cell. Used for form field naming. */
  var $cellid = "";
  /** Column span that this cell is anchor of */
  var $colspan = 1;
  /** Row span that this cell is anchor of */
  var $rowspan = 1;
  /** Whether this is a heading cell */
  var $heading = false;
  /** The cell content object */
  var $content;
  /** Whether this cell is colspanned (invisible) */
  var $colspanned = false;
  /** Whether this cell is rowspanned (invisible) */
  var $rowspanned = false;
  /** Force blank content to be non-blank space */
  var $nbsp = false;
  /** Optional access permissions for cell. */
  var $access;
  // ....................................................................
  /** Constructor. Create this new table cell object. */
  function tablecell($content="", $css="") {
    $this->HTMLObject($css);
    $this->content = new textcontent($content);
  } // tablecell
  // ....................................................................
  /**
  * Set permission for this cell for given agent
  * @param mixed $agentids List or Array of unique IDs of agents to assign the permission for
  * @param integer $perm The permission or combination of perms to assign
  * @param string $cellid Identity of the cell, used for rendering cell as textfield in a form
  */
  function permit($agentids, $perm, $cellid="???") {
    if (!isset($this->access)) {
      $this->access = new permissions();
    }
    $this->access->permit($agentids, $perm);
    if ($cellid != "???") {
      $this->setcellid($cellid);
    }
  }
  // ....................................................................
  /**
  * Unset permission for this cell for given agent
  * @param mixed $agentids List of unique IDs of agents to unassign the permission from
  * @param integer $perm The permission or combination of perms to unassign
  */
  function unpermit($agentids, $perm) {
    if (isset($this->access)) {
      $this->access->unpermit($agentids, $perm);
    }
  }
  // ....................................................................
  /** Set the cell ID */
  function setcellid($cellid) {
    $this->cellid = $cellid;
  }
  // ....................................................................
  /** Set the number of columns this cell spans */
  function setcolspan($span) {
    if ($this->rowspanned) {
      if ($span > 1) {
        $this->colspan = $span - 1;
      }
      $this->rowspanned = false;
    }
    else {
      $this->colspan = $span;
    }
  }
  // ....................................................................
  /** Set the number of rows this cell spans */
  function setrowspan($span) {
    if ($this->colspanned) {
      $this->colspanned = false;
    }
    $this->rowspan = $span;
  }
  // ....................................................................
  /** Flag this cell as being spanned */
  function span($type) {
    if ($type == "column") {
      $this->colspan = 1;
      $this->colspanned = true;
    }
    elseif ($type == "row") {
      if ($this->colspan > 1) {
        $this->colspan -= 1;
      }
      else {
        $this->rowspan = 1;
        $this->rowspanned = true;
      }
    }
  }
  // ....................................................................
  /** Flag this cell as being unspanned */
  function unspan($type) {
    if ($type == "column") {
      $this->colspanned = false;
    }
    elseif ($type == "row") {
      $this->rowspanned = false;
    }
  }
  // ....................................................................
  /** Set the style or class for the content of this cell */
  function setcontentcss($css="") {
    $this->content->setcss($css);
  }
  // ....................................................................
  /** Set the content for this cell */
  function setcontent($text="") {
    $this->content->setcontent($text);
  }
  // ....................................................................
  /** Add to the content for this cell */
  function addcontent($text="") {
    $this->content->addcontent($text);
  }
  // ....................................................................
  /** Clear the content from this cell. */
  function clearcontent() {
    $this->content->setcontent("");
  }
  // ....................................................................
  /** Automatically set the alignment of this cell according to content. */
  function autojustify() {
    $content = $this->content->content;
    if ($content != "") {
      $style = $this->style;
      $content_style = $this->content->style;
      if (!stristr($style, "text-align") && !stristr($content_style, "text-align")) {
        //if (is_numeric($content)) {
        $pat = "/^[0-9|\.\$|,]+$/";
        if (preg_match($pat, $content)) {
          $just = "text-align:right;";
          $this->setstyle($just);
        }
      }
    }
  }
  // ....................................................................
  /** Set the alignment of this cell */
  function setalignment($align="", $valign="") {
    if ($align  != "") $this->setalign($align);
    if ($valign != "") $this->setvalign($valign);
  }
  // ....................................................................
  /** Set the width and height of this cell */
  function setmetrics($width="", $height="") {
    $this->setwidth($width);
    $this->setheight($height);
  }
  // ....................................................................
  /**
  * Return the HTML for this cell. Cell content can be displayed as either
  * standard text, or as a form-field. The latter case is only possible if
  * the cell has had permissions defined, and the logged-in user is found
  * to be permitted UPDATE access to the cell.
  */
  function html() {
    global $RESPONSE;
    $s = "";
    // We are only visible if not spanned..
    if (!$this->colspanned && !$this->rowspanned) {
      if ($this->heading) $tag = "th";
      else $tag = "td";
      $s .= "<$tag";
      $s .= $this->attributes();
      if ($this->colspan > 1) {
        $s .= " colspan=\"$this->colspan\"";
      }
      if ($this->rowspan > 1) {
        $s .= " rowspan=\"$this->rowspan\"";
      }
      $s .= ">";
      $sc = $this->content->html();
      if ($sc == "" && $this->nbsp) {
        $sc = "&nbsp;";
      }
      $s .= $sc;
      // Close cell tag..
      $s .= "</$tag>";
    }
    // Return the HTML
    return $s;
  }
  // ....................................................................
  /**
  * Return the CSV content for this cell.
  */
  function csv() {
    $s = "";
    // We are only visible if not spanned..
    if (!$this->colspanned && !$this->rowspanned) {
      $s = strip_tags($this->content->content);
      $s = trim(str_replace("&nbsp;", " ", $s));
      if (strpos($s, "\"") !== false || strpos($s, ",") !== false) {
        $s = str_replace("\"", "\"\"", $s);
        $s = "\"$s\"";
      }
    }
    return $s;
  }
} // tablecell class
// ......................................................................
/**
* headingcell
* @package html
* @access private
*/
class headingcell extends tablecell {
  function headingcell($content="", $css="") {
    $this->tablecell($content, $css);
    $this->heading = true;
  }
} // headingcell class

// ......................................................................
/**
* tablerow
* A container of table cells.
* @package html
* @access private
*/
class tablerow extends HTMLObject {
  /** An array containing the cells of this row */
  var $cells = array();
  // ....................................................................
  /** Constructor. Create a new row object. */
  function tablerow($css="") {
    $this->HTMLObject($css);
  }
  // ....................................................................
  /** Set cell permissions across entire row. If the row number os specified
  * then we assign the cell ID string, which is used when rendering the cell
  * as an editable field, to give the field a name used in form submission.
  * @param mixed $agentids List or Array of unique IDs of agents to assign the permission for
  * @param integer $perm The permission or combination of perms to assign
  * @param integer $row The row number, for cell ID purposes
  * @param string $group The group tag (eg. 'tbody'), for cell ID purposes
  */
  function permit($agentids, $perm, $row="", $group="") {
    for ($c = 0; $c < $this->cellcount(); $c++) {
      // Determine cell ID..
      $cellid = "";
      if ($row != "") {
        $cellid = "_tcell|$row|$c";
        if ($group != "") $cellid .= "|$group";
      }
      // Set cell access permission..
      $cell = $this->cells[$c];
      $cell->permit($agentids, $perm, $cellid);
      $this->cells[$c] = $cell;
    }
  }
  // ....................................................................
  /** Autojustify all cells in this row based on current content */
  function autojustify() {
    for ($c = 0; $c < $this->cellcount(); $c++) {
      $cell = $this->cells[$c];
      $cell->autojustify();
      $this->cells[$c] = $cell;
    }
  }
  // ....................................................................
  /** Return the number of cells in this row
  * @return integer Number of physical cells in the row
  */
  function cellcount() {
    return count($this->cells);
  }
  // ....................................................................
  /**
  * Return the number of columns in this row. This counts spanned
  * cells as well, so a cell with colspan=2 would count as 2 columns.
  * cells whch are col-spanned are omitted, but those which are
  * row-spanned are counted, as they should be.
  * @return integer Number of actual table columns in row
  */
  function colcount() {
    $tot = 0;
    foreach ($this->cells as $cell) {
      if (!$cell->colspanned) {
        $tot += $cell->colspan;
      }
    }
    return $tot;
  }
  // ....................................................................
  /**
  * Return the number of visible cells in this row. This takes account
  * of colspans, and only counts cells that are seen in the row when
  * it is rendered.
  * @return integer Number of visible cells in the row
  */
  function visible_cellsinrow() {
    $tot = 0;
    foreach ($this->cells as $cell) {
      if (!$cell->colspanned) {
        $tot += 1;
      }
    }
    return $tot;
  }
  // ....................................................................
  /** Clear the content from all cells in row. */
  function clearcontent() {
    for ($c = 0; $c < $this->cellcount(); $c++) {
      $cell = $this->cells[$c];
      $cell->clearcontent();
      $this->cells[$c] = $cell;
    }
  }
  // ....................................................................
  /** Set nbsp setting for cells in this row
  * @param boolean mode If true, then all cells will return "&nbsp;" if blank.
  */
  function setnbsp($mode=true) {
    for ($c = 0; $c < $this->cellcount(); $c++) {
      $cell = $this->cells[$c];
      $cell->nbsp = $mode;
      $this->cells[$c] = $cell;
    }
  }
  // ....................................................................
  /** Flag a given column as being spanned.
  * @param string $type Type of span 'column' or 'row'
  * $param integer $col Column (zero-referenced) to span
  */
  function span($type, $col) {
    if (isset($this->cells[$col])) {
      $cell = $this->cells[$col];
      $cell->span($type);
      $this->cells[$col] = $cell;
    }
  }
  // ....................................................................
  /** Flag a given column as being unspanned.
  * @param string $type Type of span 'column' or 'row'
  * @param integer $col Column (zero-referenced) to unspan
  */
  function unspan($type, $col) {
    if (isset($this->cells[$col])) {
      $cell = $this->cells[$col];
      $cell->unspan($type);
      $this->cells[$col] = $cell;
    }
  }
  // ....................................................................
  /** Add a ready-made cell to this row
  * @param object $cell Cell object to add to the row
  */
  function add_cell($cell) {
    $this->cells[] = $cell;
  }
  // ....................................................................
  /**
  * Insert a new cell at the given column position. The new cell
  * takes the place of any cell currently at that position, and
  * the previous cell then occupies the next highest column slot.
  * @param integer $col Column/cell to insert at
  * @param mixed $celltoinsert Optional cell object to use when inserting
  */
  function insert_cell($col, $celltoinsert=false) {
    // Inserting off the end of the row..
    if ($col == $this->cellcount() || $this->cellcount() == 0) {
      $this->append_cells(1, $celltoinsert);
    }
    else {
      if (!$celltoinsert) $celltoinsert = new tablecell();
      $newcells = array();
      $c = 0;
      foreach ($this->cells as $cell) {
        if ($c == $col) {
          if ($cell->colspanned) {
            for ($cc = $c; $cc >= 0; $cc--) {
              if (isset($newcells[$cc])) {
                $prevcell = $newcells[$cc];
                if (!$prevcell->colspanned) {
                  $prevcell->colspan += 1;
                  $newcells[$cc] = $prevcell;
                  break;
                }
              }
            }
            $spannedcell = $celltoinsert;
            $spannedcell->colspanned = true;
            $newcells[] = $spannedcell;
          }
          else {
            $newcells[] = $celltoinsert;
          }
        }
        // Always add original cell afterwards..
        $newcells[] = $cell;
        $c += 1;
      } // foreach

      // Replace row cells with new set..
      $this->cells = $newcells;
    }
  }
  // ....................................................................
  /** Append cells onto the end of this row
  * @param integer $repeat Number of cells to append
  * @param mixed $celltoappend Optional cell object to use when appending
  */
  function append_cells($repeat=1, $celltoappend=false) {
    if (!$celltoappend) $celltoappend = new tablecell();
    for ($i = 0; $i < $repeat; $i++) {
      $this->add_cell( $celltoappend );
    }
  }
  // ....................................................................
  /** Delete a cell from the given column position.
  * @param integer $col Column/cell to delete
  */
  function delete_cell($col) {
    $newcells = array();
    $ctot = $this->cellcount();
    for ($c=0; $c < $ctot; $c++) {
      $cell = $this->cells[$c];
      if ($c == $col) {
        if ($cell->colspanned) {
          for ($cc = $c; $cc >= 0; $cc--) {
            if (isset($newcells[$cc])) {
              $prevcell = $newcells[$cc];
              if (!$prevcell->colspanned) {
                $prevcell->colspan -= 1;
                $newcells[$cc] = $prevcell;
                break;
              }
            }
          }
        }
        elseif ($cell->colspan > 1) {
          if (isset($this->cells[$c + 1])) {
            $nextcell = $this->cells[$c + 1];
            $nextcell->colspanned = false;
            $nextcell->colspan = $cell->colspan - 1;
            $this->cells[$c + 1] = $nextcell;
          }
        }
      }
      else {
        $newcells[] = $cell;
      }
    }
    $this->cells = $newcells;
  }
  // ....................................................................
  /** Poke content into the given cell in this row. The first column
  * in a row being zero (0).
  * @param integer $col Column/cell in row to apply css to
  * @param string $content Cell content to poke into the cell
  * @param string $css Cell css setting, or omit to not set it.
  * @param string $contentcss Cell content css setting, or omit to not set it.
  */
  function poke_cell($col, $content, $css="", $contentcss="") {
    $rc = false;
    if (isset($this->cells[$col])) {
      $cell = $this->cells[$col];
      $cell->setcontent($content);
      if ($css != "") $cell->setcontentcss($css);
      $this->cells[$col] = $cell;
      $rc = true;
    }
    return $rc;
  }
  // ....................................................................
  /** Poke css onto the given cell in this row. The first column
  * in a row being zero (0).
  * @param integer $col Column/cell in row to apply css to
  * @param string $css Cell css setting, or nullstring to leave as is.
  * @param string $contentcss Optional cell content css setting, or omit to not set it.
  */
  function poke_cellcss($col, $css="", $contentcss="") {
    $rc = false;
    if (isset($this->cells[$col])) {
      $cell = $this->cells[$col];
      if ($css != "") $cell->setcss($css);
      if ($contentcss != "") $cell->setcontentcss($css);
      $this->cells[$col] = $cell;
      $rc = true;
    }
    return $rc;
  }
  // ....................................................................
  /** Peek cell content. Returns the content of the given cell.
  * @param integer $col Column/cell in row to retreive content from
  * @return string Cell content of the given column
  */
  function peek_cell($col) {
    $rc = "";
    if (isset($this->cells[$col])) {
      $cell = $this->cells[$col];
      $rc = $cell->content->content;
    }
    return $rc;
  }
  // ....................................................................
  /** Return the given cell in this row. The first column
  * in a row being zero (0). Returns false if no cell present.
  * @param integer $col Column/cell in row to retreive cell object of
  * @return object Cell object of the given column
  */
  function get_cell($col) {
    $cell = false;
    if (isset($this->cells[$col])) {
      $cell = $this->cells[$col];
    }
    return $cell;
  }
  // ....................................................................
  /** Replace the given cell with new table cell. Returns true if ok.
  * @param integer $col Column/cell in row to set cell object of
  * @return object Cell object to insert at the given column
  */
  function set_cell($col, $cell) {
    $rc = false;
    if (isset($this->cells[$col])) {
      $this->cells[$col] = $cell;
      $rc = true;
    }
    return $rc;
  }
  // ....................................................................
  /**
  * Apply a colspan. The column of the cell which has the colspan is
  * given. As a result the relevant spanned cells will be flagged as spanned.
  * If the span goes beyond the row end, it is truncated. NB: If the
  * anchor cell is already spanning, then the new span is added to
  * the original one, up to the limit of the table columns.
  * @param integer $col Number of column to anchor the span (first=0)
  * @param integer $span Number of columns to span across
  */
  function merge_cols($col, $span) {
    if (isset($this->cells[$col])) {
      $cell = $this->cells[$col];
      if (!$cell->colspanned) {
        $oldspan = $cell->colspan;
        $newspan = $oldspan + $span - 1;
        $spancount = $newspan - 1;
        $adj = 0;
        $nxtcol = $col + 1;
        for ($c = $nxtcol; $c < $nxtcol + $spancount; $c++) {
          if (isset($this->cells[$c])) {
            $lacell = $this->cells[$c];
            if ($lacell->colspan > 1) {
              $adj = $adj + $lacell->colspan - 1;
            }
          }
        }
        $spancount += $adj;
        $newspan += $adj;
        $this->span_cols($nxtcol, $spancount);
        $cell->setcolspan($newspan);
        $this->cells[$col] = $cell;
      }
    }
  }
  // ....................................................................
  /**
  * Apply spanning to multiple cells in the row.
  * @param integer $col Number of column to start spanning
  * @param integer $spancount Number of columns to set as spanned
  */
  function span_cols($col, $spancount) {
    for ($c = $col; $c < $spancount + $col; $c++) {
      $this->span("column", $c);
    }
  }
  // ....................................................................
  /**
  * Remove spanning from a colspanned set of cells. This will effectively
  * set the colspan of the orgin cell to 1, and insert new cells after
  * it to make up the difference and maintain the original colcount.
  * @param integer $col Number of column to split at
  */
  function split_cols($col) {
    if (isset($this->cells[$col])) {
      $cell = $this->cells[$col];
      if ($cell->colspan > 1) {
        $unspancount = $cell->colspan - 1;
        $cell->setcolspan(1);
        $this->cells[$col] = $cell;
        for ($i = 1; $i <= $unspancount; $i++) {
          $this->unspan("column", $col + $i);
        }
      }
    }
  }
  // ....................................................................
  /** Set a width profile across the row of cells
  * @param string $prof The width profile eg '33%,67%'
  */
  function set_width_profile($prof) {
    if (!$this->has_colspans()) {
      $ix = 0;
      foreach ($prof as $width) {
        if (isset($this->cells[$ix])) {
          $cell = $this->cells[$ix];
          $cell->setwidth($width);
          $this->cells[$ix] = $cell;
          $ix += 1;
        }
      }
    }
  }
  // ....................................................................
  /** Returns true if this row has colspanned cells
  * @return boolean True if the row has colspanned cells in it
  */
  function has_colspans() {
    $rc = false;
    foreach ($this->cells as $chk) {
      if ($chk->colspanned || $chk->colspan > 1) {
        $rc = true;
        break;
      }
    }
    return $rc;
  }
  // ....................................................................
  /** Set a width profile across the row of cells
  * @return string The current width profile setting for the row
  */
  function get_width_profile() {
    $prof = array();
    if (!$this->has_colspans()) {
      foreach ($this->cells as $cell) {
        $prof[] = (isset($cell->width) ? $cell->width : "");
      }
    }
    return $prof;
  }
  // ....................................................................
  /** Return the HTML for this row and all its cells
  * @return string The HTML for the row
  */
  function html() {
    $s  = "<tr" . $this->attributes() . ">\n";
    foreach ($this->cells as $cell) {
      $s .= $cell->html();
    }
    $s .= "</tr>\n";
    // Return the HTML
    return $s;
  }
  // ....................................................................
  /** Return the CSV for this row and all its cells
  * @return string The CSV rendering of the row
  */
  function csv() {
    $s = "";
    foreach ($this->cells as $cell) {
      $s .= $cell->csv() . ",";
    }
    if ($s != "") {
      $s = substr($s, 0, -1);
    }
    return $s;
  }
} // tablerow class

// ......................................................................
/**
* tablegroup
* This is a virtual class which contains a group of table rows. An
* examples of this would be the <tbody></tbody> or <thead></thead>. It
* is basically an entity which contains and manages a group of rows.
*
* @package html
* @access private
*/
class tablegroup extends StylableObject {
  /** Table group tag, eg: 'tbody' */
  var $tag;
  /** Array containing table rows */
  var $rows = array();
  /** The most recently added cell object */
  var $working_cell;
  /** Column rowspans in effect */
  var $colrowspans = array();
  // ....................................................................
  /** Tablegroup constructor
  * Create a new tablegroup object.
  * @param string $tag The group tag: 'tbody', 'thead' etc.
  * @param string $css The style or class of the group
  */
  function tablegroup($tag, $css="") {
    $this->tag = $tag;
    $this->StylableObject($css);
  }
  // ....................................................................
  /** Set cell permissions across entire group. An optional table name can
  * be passed, which is used with identifying the cell when rendering the cells
  * in th row as an editable field, to give the fields a name used in form
  * submission.
  * @param mixed $agentids List or Array of unique IDs of agents to assign the permission for
  * @param integer $perm The permission or combination of perms to assign
  */
  function permit($agentids, $perm) {
    for ($r = 0; $r < $this->rowcount(); $r++) {
      $row = $this->rows[$r];
      $row->permit($agentids, $perm, $r, $this->tag);
      $this->rows[$r] = $row;
    }
  }
  // ....................................................................
  /** Set cell nbsp setting across entire group.
  * @param boolean mode If true, then all cells will return "&nbsp;" if blank.
  */
  function setnbsp($mode=true) {
    for ($r = 0; $r < $this->rowcount(); $r++) {
      $row = $this->rows[$r];
      $row->setnbsp($mode);
      $this->rows[$r] = $row;
    }
  }
  // ....................................................................
  /**
  * Return the number of rows in this group.
  * @return integer The number of rows in this group
  */
  function rowcount() {
    return count($this->rows);
  }
  // ....................................................................
  /**
  * Return the visible cellcount for a given column in the group. This
  * takes account of any rowspans in effect.
  * @param integer $col Number of the column to count cells in
  * @return integer The visible cell count for the column
  */
  function visible_cellsincol($col) {
    $rc = 0;
    if (isset($this->rows[$col])) {
      foreach ($this->rows as $row) {
        if (isset($row->cells[$col])) {
          $cell = $row->cells[$col];
          if (!$cell->rowspanned) $rc += 1;
        }
      }
    }
    return $rc;
  }
  // ....................................................................
  /** Create a new row
  * @param string $css The style or class of the group
  */
  function tr($css="") {
    // Take care of any pending cell first..
    $this->build();
    $this->insert_rowspanned_cells();
    $this->rows[] = new tablerow($css);
  }
  // ....................................................................
  /** Check the current row and add in any cells which are rowspanned
  * @private
  */
  function insert_rowspanned_cells() {
    if (count($this->colrowspans) > 0 && count($this->rows) > 0) {
      $row = array_pop($this->rows);

      $colrowspans = $this->colrowspans;
      while (list($col, $span) = each($colrowspans)) {
        if ($span > 0) {
          $chk = $row->get_cell($col);
          if (!$chk || $chk->rowspan == 1) {
            $cell = new tablecell();
            $cell->rowspanned = true;
            $row->insert_cell($col, $cell);
            $this->colrowspans[$col] -= 1;
          }
        }
      }
      array_push($this->rows, $row);
    }
  }
  // ....................................................................
  /** Build the current working cell into the row
  * @access private
  */
  function build() {
    if (isset($this->working_cell)) {
      if (count($this->rows) > 0) {
        $row = array_pop($this->rows);

        // Add the working cell first..
        $row->add_cell($this->working_cell);

        // Now record any rowspan being applied..
        if ($this->working_cell->rowspan > 1) {
          $thecol = $row->cellcount() - 1;
          $this->colrowspans[$thecol] = $this->working_cell->rowspan - 1;
        }

        // Finally, add invisible colspanned cells, if any..
        if ($this->working_cell->colspan > 1) {
          $cell = new tablecell();
          $cell->colspanned = true;
          $row->append_cells($this->working_cell->colspan - 1, $cell);
        }
        array_push($this->rows, $row);
      }
      unset($this->working_cell);
    }
  }
  // ....................................................................
  /** Create a new standard cell for the row
  * @param string $content Content to poke into the cell
  * @param string $css Cell css setting, or omit to not set it.
  * @param boolean $heading Whether it is a 'heading' cell or not
  */
  function td($content="", $css="", $heading=false) {
    // Take care of any pending cell..
    $this->build();
    if ($heading) {
      $this->working_cell = new headingcell($content, $css);
    }
    else {
      $this->working_cell = new tablecell($content, $css);
    }
  }
  // ....................................................................
  /** Create a new heading cell for the row
  * @param string $content Content to poke into the cell
  * @param string $css Cell css setting, or omit to not set it.
  */
  function th($content="", $css="") {
    $this->td($content, $css, true);
  }
  // ....................................................................
  /** Clear the content from all rows in the group. */
  function clearcontent() {
    for ($r = 0; $r < $this->rowcount(); $r++) {
      $row = $this->rows[$r];
      $row->clearcontent();
      $this->rows[$r] = $row;
    }
  }
  // ....................................................................
  /** Set the working cell style/class properties
  * @param string $css Cell css setting, or omit to not set it.
  */
  function td_css($css="") {
    $this->working_cell->setcss($css);
  }
  // ....................................................................
  /** Set the working cell content style or class.
  * @param string $css Content css setting, or omit to not set it.
  */
  function td_contentcss($css="") {
    $this->working_cell->content->setcss($css);
  }
  // ....................................................................
  /** Append to the working cell content
  * @param string $text Cell text content
  * @param string $css Cell css setting, or omit to not set it.
  */
  function td_content($text="", $css="") {
    $this->working_cell->addcontent($text);
    $this->working_cell->setcss($css);
  }
  // ....................................................................
  /** Set the working cell alignment properties */
  function td_alignment($align="", $valign="") {
    $this->working_cell->setalignment($align, $valign);
  }
  // ....................................................................
  /** Set the working cell size properties */
  function td_metrics($width="", $height="") {
    $this->working_cell->setmetrics($width, $height);
  }
  // ....................................................................
  /** Set the working cell width property */
  function td_width($width="") {
    $this->working_cell->setwidth($width);
  }
  // ....................................................................
  /** Set the working cell height property */
  function td_height($height="") {
    $this->working_cell->setheight($height);
  }
  // ....................................................................
  /** Set the working cell colspan property */
  function td_colspan($span=1) {
    $this->working_cell->setcolspan($span);
  }
  // ....................................................................
  /** Set the working cell rowspan property */
  function td_rowspan($span=1) {
    $this->working_cell->setrowspan($span);
  }
  // ....................................................................
  /** Poke content into the given cell in this group
  * @param integer $row Row that the column is in
  * @param integer $col Column/cell in row to poke content into
  * @param string $content Cell content to poke into the cell
  * @param string $css Cell css setting, or omit to not set it.
  * @param string $contentcss Cell content css setting, or omit to not set it.
  */
  function poke_cell($row, $col, $content, $css="", $contentcss="") {
    $rc = false;
    if (isset($this->rows[$row])) {
      $r = $this->rows[$row];
      $rc = $r->poke_cell($col, $content, $css, $contentcss);
      $this->rows[$row] = $r;
    }
    return $rc;
  }
  // ....................................................................
  /** Autojustify all rows in this group */
  function autojustify() {
    for ($r = 0; $r < $this->rowcount(); $r++) {
      $row = $this->rows[$r];
      $row->autojustify();
      $this->rows[$r] = $row;
    }
  }
  // ....................................................................
  /** Poke css onto the given cell in this group
  * @param integer $row Row that the column is in
  * @param integer $col Column/cell in row to apply css to
  * @param string $css Cell css setting.
  * @param string $contentcss Cell content css setting, or omit to not set it.
  */
  function poke_cellcss($row, $col, $css, $contentcss="") {
    $rc = false;
    if (isset($this->rows[$row])) {
      $r = $this->rows[$row];
      $rc = $r->poke_cellcss($col, $css, $contentcss);
      $this->rows[$row] = $r;
    }
    return $rc;
  }
  // ....................................................................
  /** Peek content from the given cell in this group */
  function peek_cell($row, $col) {
    $rc = "";
    if (isset($this->rows[$row])) {
      $r = $this->rows[$row];
      $rc = $r->peek_cell($col);
    }
    return $rc;
  }
  // ....................................................................
  /** Return the given cell from this group */
  function get_cell($row, $col) {
    $cell = false;
    if (isset($this->rows[$row])) {
      $r = $this->rows[$row];
      $cell = $r->get_cell($col);
    }
    return $cell;
  }
  // ....................................................................
  /** Replace the given cell with new table cell. Returns true if ok. */
  function set_cell($row, $col, $cell) {
    $rc = false;
    if (isset($this->rows[$row])) {
      $r = $this->rows[$row];
      $rc = $r->set_cell($col, $cell);
      $this->rows[$row] = $r;
    }
    return $rc;
  }
  // ....................................................................
  /** Apply a rowspan. The anchor row,col are given, and the rowspan. */
  function merge_rows($row, $col, $span) {
    $therow = $this->rows[$row];
    $cell = $therow->cells[$col];
    $oldspan = $cell->rowspan;
    $newspan = $oldspan + $span - 1;
    $spancount = $newspan - $oldspan;
    $colspan = $cell->colspan;
    $cell->setrowspan($newspan);
    $therow->cells[$col] = $cell;
    $this->rows[$row] = $therow;
    for ($c = $col; $c < $col + $colspan; $c++) {
      for ($i = 0; $i < $spancount; $i++) {
        $r = $row + $i + $oldspan;
        if (isset($this->rows[$r])) {
          $therow = $this->rows[$r];
          $therow->span("row", $c);
          $this->rows[$r] = $therow;
        }
      }
    }
  }
  // ....................................................................
  /** Apply row-spanning to a number of rows. */
  function span_rows($row, $col, $spancount) {
    for ($r = $row; $r < $spancount + $row; $r++) {
      if (isset($this->rows[$r])) {
        $therow = $this->rows[$r];
        if (isset($therow->cells[$col])) {
          $cell = $therow->cells[$col];
          $cell->span("row", $c);
          $therow->cells[$col] = $cell;
        }
        $this->rows[$r] = $therow;
      }
    }
  }
  // ....................................................................
  /** Remove a rowspan. The anchor row,col are given of the existing span. */
  function split_rows($row, $col) {
    $newrows = $this->rows;
    if (isset($newrows[$row])) {
      $therow = $newrows[$row];
      if (isset($therow->cells[$col])) {
        $cell = $therow->cells[$col];
        $span = $cell->rowspan;
        $colspan = $cell->colspan;
        if ($span > 1) {
          $cell->setrowspan(1);
          $therow->cells[$col] = $cell;
          $newrows[$row] = $therow;
          for ($c = $col; $c < $col + $colspan; $c++) {
            for ($r = $row + 1; $r < $row + $span; $r++) {
              if (isset($newrows[$r])) {
                $therow = $newrows[$r];
                $therow->unspan("row", $c);
                $newrows[$r] = $therow;
              }
            }
          }
          $this->rows = $newrows;
        }
      }
    }
  }
  // ....................................................................
  /**
  * Apply a colspan. The row and column of the cell which has the
  * colspan is given. As a result the relevant spanned cells will be
  * removed from the table. If the span goes beyond the row end,
  * it is truncated.
  */
  function merge_cols($row, $col, $span) {
    if (isset($this->rows[$row])) {
      $r = $this->rows[$row];
      $thecell = $r->cells[$col];
      $rowspan = $thecell->rowspan;
      $r->merge_cols($col, $span);
      $this->rows[$row] = $r;
      if ($rowspan > 1) {
        for ($r = $row + 1; $r < $rowspan + $row; $r++) {
          if (isset($this->rows[$r])) {
            $therow = $this->rows[$r];
            $therow->span_cols($col + 1, $span - 1);
            $this->rows[$r] = $therow;
          }
        }
      }
    }
  }
  // ....................................................................
  /**
  * Remove spanning from a colspanned set of cells. This will effectively
  * set the colspan of the orgin cell to 1, and insert new cells after
  * it to make up the difference and maintain the original colcount.
  */
  function split_cols($row, $col) {
    if (isset($this->rows[$row])) {
      $r = $this->rows[$row];
      $r->split_cols($col);
      $this->rows[$row] = $r;
    }
  }
  // ....................................................................
  /** Insert a column into all rows in the group. */
  function insert_cols($col, $celltoinsert=false) {
    for ($r = 0; $r < $this->rowcount(); $r++) {
      $row = $this->rows[$r];
      $row->insert_cell($col, $celltoinsert);
      $this->rows[$r] = $row;
    }
  }
  // ....................................................................
  /** Append columns onto all rows in the group. */
  function append_cols($repeat=1, $celltoappend=false) {
    for ($r = 0; $r < $this->rowcount(); $r++) {
      $row = $this->rows[$r];
      $row->append_cells($repeat, $celltoappend);
      $this->rows[$r] = $row;
    }
  }
  // ....................................................................
  /** Remove a column from all rows in the group. */
  function delete_cols($col) {
    for ($r = 0; $r < $this->rowcount(); $r++) {
      $row = $this->rows[$r];
      $row->delete_cell($col);
      $this->rows[$r] = $row;
    }
  }
  // ....................................................................
  /** Return a given row from the group. */
  function get_row($row) {
    $therow = false;
    if (isset($this->rows[$row])) {
      $therow = $this->rows[$row];
    }
    return $therow;
  }
  // ....................................................................
  /** Delete a row from the group. */
  function delete_row($row) {
    if (isset($this->rows[$row]) && $this->rowcount() > 0) {
      $newrows = array();
      $r = 0;
      foreach ($this->rows as $therow) {
        if ($r == $row) {
          // Scan this row for rowspans
          $c = 0;
          foreach ($therow->cells as $cell) {
            if ($cell->rowspanned) {
              for ($i = $r - 1; $i >= 0; $i--) {
                $prevrow = $newrows[$i];
                $prevcell = $prevrow->get_cell($c);
                if (!$prevcell->rowspanned) {
                  $prevcell->rowspan -= 1;
                  $prevrow->set_cell($c, $prevcell);
                  $newrows[$i] = $prevrow;
                  break;
                }
              }
            }
            elseif ($cell->rowspan > 1) {
              if (isset($this->rows[$r + 1])) {
                $nextrow = $this->rows[$r + 1];
                $nextcell = $nextrow->get_cell($c);
                $nextcell->rowspanned = false;
                $nextcell->rowspan = $cell->rowspan - 1;
                $nextrow->set_cell($c, $nextcell);
                $this->rows[$r + 1] = $nextrow;
              }
            }
            $c += 1;
          } // foreach
        }
        else {
          $newrows[] = $therow;
        }
        $r += 1;
      } // foreach
      // Establish the new set of rows..
      $this->rows = $newrows;
    }
  }
  // ....................................................................
  /**
  * Insert a row into the table. If the template cell is not provided,
  * then a 'vanilla' cell is used instead. We insert the new row just
  * before the given row.
  * @param integer $row Row position to insert before, in the table
  * @param object $celltoinsert Template cell to use for all row cells
  */
  function insert_row($row, $celltoinsert=false) {
    if ($this->rowcount() == 0) {
      $this->append_row($celltoinsert);
    }
    else {
      $newrow = new tablerow();
      $arow = $this->rows[0];
      $cnt = $arow->cellcount();
      $newrow->append_cells($cnt, $celltoinsert);
      $newrows = array();
      $r = 0;
      foreach ($this->rows as $therow) {
        if ($r == $row) {
          // Scan this row for rowspans
          $c = 0;
          foreach ($therow->cells as $cell) {
            if ($cell->rowspanned) {
              $newcell = $newrow->get_cell($c);
              $newcell->rowspanned = true;
              $newrow->set_cell($c, $newcell);
              for ($i = $r - 1; $i >= 0; $i--) {
                $prevrow = $newrows[$i];
                if (is_object($prevrow)) {
                  $prevcell = $prevrow->get_cell($c);
                  if (!$prevcell->rowspanned) {
                    $prevcell->rowspan += 1;
                    $prevrow->set_cell($c, $prevcell);
                    $newrows[$i] = $prevrow;
                    break;
                  }
                }
              }
            }
            $c += 1;
          } // foreach
          // Insert new row..
          $newrows[] = $newrow;
        }
        // Insert existing row..
        $newrows[] = $therow;
        $r += 1;
      } // foreach
      // Establish the new set of rows..
      $this->rows = $newrows;
    }
  }
  // ....................................................................
  /** Append a row onto the group. */
  function append_row($celltoappend=false) {
    if ($this->rowcount() > 0) {
      $lastrow = $this->rows[$this->rowcount() - 1];
      $cnt = $lastrow->colcount();
    }
    else {
      $cnt = 1;
    }
    $newrow = new tablerow();
    $newrow->append_cells($cnt, $celltoappend);
    $this->rows[] = $newrow;
  }
  // ....................................................................
  /** Set a width profile across the row of cells
  * @param array $prof Array of widths defining the profile
  */
  function set_width_profile($prof) {
    for ($r = 0; $r < $this->rowcount(); $r++) {
      $row = $this->rows[$r];
      $row->set_width_profile($prof);
      $this->rows[$r] = $row;
    }
  }
  // ....................................................................
  /** Return the width profile for this group.
  * @return array Returned array of profile widths
  */
  function get_width_profile() {
    $prof = array();
    if (isset($this->rows)) {
      foreach ($this->rows as $row) {
        $prof = $row->get_width_profile();
        if (count($prof) > 0) {
          break;
        }
      }
    }
    return $prof;
  }
  // ....................................................................
  /**
  * Apply row-stripes to the rows in this group. The parameter
  * passed in is a delimited string list OR array of classes or
  * styles to apply to the rows.
  *
  * @param mixed $csslist Array OR delimited list of styles or classnames
  * @param string $delim Optional delimiter char, defaults to a comma
  */
  function apply_rowstripes($csslist, $delim=",") {
    if (!is_array($csslist)) {
      $csslist = explode($delim, $csslist);
    }
    $totrows = $this->rowcount();
    $totcss = count($csslist);
    if ($totcss > 0 && $totrows > 0) {
      $cssix = 0;
      for ($r = 0; $r < $totrows; $r++) {
        $css = $csslist[$cssix];
        $row = $this->rows[$r];
        $s  = (isset($row->class)) ? $row->class : "";
        $s .= (isset($row->style)) ? $row->style : "";
        if ($css != "" && !strstr($s, $css)) {
          $row->setcss($css);
          $this->rows[$r] = $row;
        }
        $cssix += 1;
        if ($cssix >= $totcss) $cssix = 0;
      }
    }
  }
  // ....................................................................
  /**
  * Return the HTML for this group.
  * @return string HTML rendering for this group of rows.
  */
  function html() {
    // Take care of any pending stuff..
    $this->build();
    $this->insert_rowspanned_cells();
//    $row = array_pop($this->rows);
//    array_push($this->rows, $row);

    // Render all rows inside tag..
    $s = "";
    $s .= " <$this->tag";
    $s .= $this->style_attributes();
    $s .= ">\n";
    foreach ($this->rows as $row) {
      if (is_object($row)) {
        $s .= $row->html();
      }
    }
    $s .= " </$this->tag>\n";
    // Return the HTML..
    return $s;
  }

  // ....................................................................
  /**
  * Return the CSV for this group.
  * @return string CSV rendering for this group of rows.
  */
  function csv() {
    // Take care of any pending stuff..
    $this->build();
    $this->insert_rowspanned_cells();
    $s = "";
    foreach ($this->rows as $row) {
      if (is_object($row)) {
        $s .= $row->csv() . "\n";
      }
    }
    // Return the CSV..
    return $s;
  }
} // tablegroup class
// ......................................................................
/**
* Table.
* A table is a container of tablegroups. Tablegroups are in turn
* containers of table rows, and table rows contain table cells.
*
* This class is predicated on the idea that tables will be built bit
* by bit, in a sequential manner by Php code. That is to say, you will
* be looping through adding to the table structure and content, row
* by row, and within each row, cell by cell.
*
* Here's an example of usage. Copy and paste into a test page in order
* to view the result and see how it works.
*
*  $mytable = new table("test");
*  $mytable->autojustify();
*  $mytable->rowstripes("background-color:#dddddd;,background-color:#efefef;");
*  $mytable->setalign("center");
*  $mytable->setwidth("70%");
*  $mytable->setborder(1);
*  $mytable->thead();
*  $mytable->tr();
*  $mytable->th("Heading 1");
*  $mytable->th("Heading 2", "text-align:right");
*  $mytable->th("Heading 3");
*  $mytable->tbody();
*  $mytable->tr();
*  $mytable->td("Rowspan Text");
*  $mytable->td_rowspan(2);
*  $mytable->td("1234");
*  $mytable->td("Text");
*  $mytable->tr();
*  $mytable->td("Text");
*  $mytable->td("1234");
*  $mytable->tr();
*  $mytable->td("Text");
*  $mytable->td("Text");
*  $mytable->td("Text");
*  $mytable->tr();
*  $mytable->td("Colspan Text");
*  $mytable->td_colspan(3);
* @package html
*/
class table extends HTMLObject {
  // Public
  /** Tablename. Required. An internal name for use in debugging etc. */
  var $tablename = "";
  /** Padding of cells in pixels */
  var $cellpadding = 0;
  /** Spacing between cells in pixels */
  var $cellspacing = 0;
  /** Auto-justify numerics to the right, text to the left */
  var $autojustify = false;

  // Private
  /** Array containing table groups
      @access private */
  var $groups = array();
  /**
  * Array containing css definitions for row striping. Rows
  * will have the css applied repeatedly. Any number of css
  * definitions can be added to the array. These can be either
  * classnames or styles, or a mixture of both.
  * @access private
  */
  var $rowstripes = array();
  // ....................................................................
  /** Table constructor
  * Create a new table object.
  * @param string $tablename A required name string to identify this table.
  * @param string $css The style or classname to apply to the object.
  */
  function table($tablename="", $css="") {
    $this->HTMLObject($css);
    $this->setwidth("100%");
    $this->setborder(0);
    $this->tablename = $tablename;
  }
  // ....................................................................
  /** Set cell permissions across entire table
  * @param mixed $agentids List or Array of unique IDs of agents to assign the permission for
  * @param integer $perm The permission or combination of perms to assign
  */
  function permit($agentids, $perm) {
    foreach ($this->groups as $gp) {
      $gpix = $this->get_group_index($gp->tag);
      $gp->permit($agentids, $perm);
      $this->groups[$gpix] = $gp;
    }
  }
  // ....................................................................
  /** Set nbsp setting for cells in this table
  * @param boolean mode If true, then all cells will return "&nbsp;" if blank.
  */
  function setnbsp($mode=true) {
    foreach ($this->groups as $gp) {
      $gpix = $this->get_group_index($gp->tag);
      $gp->setnbsp($mode);
      $this->groups[$gpix] = $gp;
    }
  }
  // ....................................................................
  /**
  * Return the number of rows in this table.
  * @return integer The number of rows in the table.
  */
  function rowcount() {
    $tot = 0;
    foreach ($this->groups as $group) {
      $tot += $group->rowcount();
    }
    return $tot;
  }
  // ....................................................................
  /**
  * Return the number of cols in this table.
  * This is the number of columns, rather than physical cells, any of
  * which may be spanning multiple columns. In the case of a table which
  * is not properly formatted, this function will return the maximum
  * columns it finds by looking across all of the table rows.
  * @return integer The number of columns in the table.
  */
  function colcount() {
    $tot = 0;
    if (isset($this->groups[0])) {
      $g = $this->groups[0];
      if (isset($g->rows[0])) {
        $r = $g->rows[0];
        $tot = $r->cellcount();
      }
    }
    return $tot;
  }
  // ....................................................................
  /**
  * Return the number of cells in a specific row.
  * This is the number of physical cells, rather than the number of logical
  * number of columns. It should be the same as the colcount() method if
  * the table is properly formatted.
  * @param integer $row The row that the cell is in (0 = first row)
  * @param string $group The group to find the rows
  * @return integer The number of cells in the row.
  */
  function cellcount($row=0, $group="tbody") {
    $rc = 0;
    $gpix = $this->get_group_index($group);
    if ($gpix != -1) {
      $gp = $this->groups[$gpix];
      if (isset($gp->rows[$row])) {
        $r = $gp->rows[$row];
        $rc = $r->cellcount();
      }
    }
    return $rc;
  }
  // ....................................................................
  /**
  * Return the number of visible cells in a specific row.
  * This is the number of viewed cells, rather than the number of physical
  * columns. Ie. if there is a colspan of 2, then only 1 cell will be
  * counted for it, since only one cell appears in the rendered table.
  * @param integer $row The row that the cell is in (0 = first row)
  * @param string $group The group to find the rows
  * @return integer The number of visible cells in the row.
  */
  function visible_cellsinrow($row, $group="tbody") {
    $rc = 0;
    $gpix = $this->get_group_index($group);
    if ($gpix != -1) {
      $gp = $this->groups[$gpix];
      if (isset($gp->rows[$row])) {
        $r = $gp->rows[$row];
        $rc = $r->visible_cellsinrow();
      }
    }
    return $rc;
  }
  // ....................................................................
  /**
  * Return the number of visible cells in a specific column.
  * This is the number of viewed cells, rather than the number of physical
  * cells. Ie. if there is a rowspan of 2, then only 1 cell will be
  * counted for it, since only one cell appears in the rendered table.
  * @return integer The number of visible cells in the given column.
  */
  function visible_cellsincol($col, $group="tbody") {
    $rc = 0;
    $gpix = $this->get_group_index($group);
    if ($gpix != -1) {
      $gp = $this->groups[$gpix];
      $rc = $gp->visible_cellsincol($col);
    }
    return $rc;
  }
  // ....................................................................
  /**
  * Clear the content from all rows in table. This sets the content of
  * every cell in every row in every group to nullstring.
  */
  function clearcontent() {
    foreach ($this->groups as $gp) {
      $gpix = $this->get_group_index($gp->tag);
      $gp->clearcontent();
      $this->groups[$gpix] = $gp;
    }
  }
  // ....................................................................
  /**
  * Set the table padding and spacing amounts. This sets the table
  * 'cellpadding' and 'cellspacing' attributes, in that order.
  * @param integer $pad Amount of cellpadding in pixels
  * @param integer $space Amount of cellspacing in pixels
  */
  function setpadding($pad=0, $space=0) {
    $this->cellpadding = $pad;
    $this->cellspacing = $space;
  }
  // ....................................................................
  /**
  * Close the current group if need be. This just makes sure that the
  * current group stores the working cell in its $rows array.
  * @access private
  */
  function close_group() {
    if (count($this->groups) > 0) {
      $group = array_pop($this->groups);
      $group->build();
      array_push($this->groups, $group);
    }
  }
  // ....................................................................
  /**
  * Define a new table group. A group is simply a container of table
  * rows. We recognize three different kinds of group: 'thead', 'tbody',
  * and 'tfoot'.
  * @param string $tag Type of group: 'thead', 'tbody' or 'tfoot'.
  * @param string $css Style or classname to use for the group.
  * @access private
  */
  function new_group($tag, $css="") {
    $this->close_group();
    $this->groups[] = new tablegroup($tag, $css);
  }
  // ....................................................................
  /**
  * Define a new thead table group.
  * @param string $css Style or classname to use for the group.
  */
  function thead($css="") {
    $this->new_group("thead", $css);
  }
  // ....................................................................
  /**
  * Define a new tbody table group.
  * @param string $css Style or classname to use for the group.
  */
  function tbody($css="") {
    $this->new_group("tbody", $css);
  }
  // ....................................................................
  /**
  * Define a new tfoot table group.
  * @param string $css Style or classname to use for the group.
  */
  function tfoot($css="") {
    $this->new_group("tfoot", $css);
  }
  // ....................................................................
  /**
  * Check if first group is defined and if not then define tbody.
  * @access private
  */
  function check_group() {
    if (count($this->groups) == 0) {
      $this->tbody();
    }
  }
  // ....................................................................
  /**
  * Set table auto-justify mode on or off.
  * If set to on, then when we render the table, cells will have a
  * style automatically applied to justify numeric content to the right
  * and non-numeric content to the left. Handy with simple data tables.
  * @param bool $mode True if we should auto justify all table cells
  */
  function autojustify($mode=true) {
    $this->autojustify = $mode;
  }
  // ....................................................................
  /**
  * Define css for row striping with comma-delimted list of css. Any
  * number of these can be added to style rows in a repeating cycle.
  * @param mixed $csslist Array OR delimited list of styles or classnames
  * @param string $delim Optional delimiter char, defaults to a comma
  */
  function rowstripes($csslist, $delim=",") {
    if (!is_array($csslist)) {
      $csslist = explode($delim, $csslist);
    }
    $this->rowstripes = $csslist;
  }
  // ....................................................................
  /**
  * Check the table cell balance. Ie. make sure we have the
  * same number of cells in each row, taking account of rowspan
  * and of course colspan. NB: Complex mixtures of col and row
  * spanning will not be coped with - this is currently a very
  * simplistic algorithm, intended for fairly basic structures.
  * @return boolean True if the table checks out ok, else false
  */
  function validate() {
    $this->close_group();
    $rowstats = array();
    $totrows = 0;
    $totcols = 0;
    foreach ($this->groups as $group) {
      foreach ($group->rows as $row) {
        $totrows += 1;
        if (is_object($row)) {
          $cols = $row->colcount();
        }
        else $cols = 0;
        if ($cols > $totcols) $totcols = $cols;
        $rowstats[] = $cols;
      }
    }
    if (debugging() && (debug_class() & DBG_TABLES)) {
      if ($this->tablename != "") $name = "'$this->tablename'";
      else $name = "(unnamed)";
      debug("table $name validation: ($totrows rows x $totcols cols) cols by row: ", DBG_TABLES);
      debug(implode(",", $rowstats), DBG_TABLES);
      $first = true; $valid = true;
      foreach($rowstats as $cnt) {
        if ($first) {
          $lastcnt = $cnt;
          $first = false;
        }
        else {
          if ($cnt != $lastcnt) $valid = false;
          $lastcnt = $cnt;
        }
      }
      if ($valid) debugbr(" valid.", DBG_TABLES);
      else debugbr(" <span style='color:red'>error!</span>", DBG_TABLES);
      // Return code
      return $valid;
    }
  } // validate
  // ....................................................................
  /**
  * Define a new row for the current group.
  * @param string $css Style or classname to use for the row.
  */
  function tr($css="") {
    static $stripe_ix = 0;
    $this->check_group();
    $group = array_pop($this->groups);
    $group->tr($css);
    array_push($this->groups, $group);
  }
  // ....................................................................
  /**
  * Define a new standard cell for the current row.
  * @param string $text Content to put in the new cell.
  * @param string $css Style or classname to use for the cell.
  */
  function td($content="", $css="", $heading=false) {
    // Automatically insert row if no groups yet, and this
    // will create the 'tbody' group as well..
    if (count($this->groups) == 0) {
      $this->tr();
    }
    // Deal with auto-justification if required..
    $content = trim($content);

    // Create new cell in this group..
    $group = array_pop($this->groups);
    $group->td($content, $css, $heading);
    array_push($this->groups, $group);
  }
  // ....................................................................
  /**
  * Define a new heading cell for the current row.
  * @param string $text Content to put in the new heading cell.
  * @param string $css Style or classname to use for the cell.
  */
  function th($content="", $css="") {
    // Create new cell in this group..
    $group = array_pop($this->groups);
    $group->td($content, $css, true);
    array_push($this->groups, $group);
  }
  // ....................................................................
  /**
  * Set the working cell css properties. Use this method AFTER you have
  * defined the cell with the td() method. This defines the style or
  * classname to use for the cell. Note this is for the cell, rather
  * than the content inside the cell.
  * @param string $css Style or classname to use for the cell.
  */
  function td_css($css="") {
    $group = array_pop($this->groups);
    $group->td_css($css);
    array_push($this->groups, $group);
  }
  // ....................................................................
  /**
  * Set the working cell content css properties in one hit. Use this
  * method AFTER you have defined the cell with the td() method. This
  * defines the style or classname to use for the cell content. This
  * will be implemented using span tags.
  * @param string $css Style or classname to use for the cell content.
  */
  function td_contentcss($css="") {
    $group = array_pop($this->groups);
    $group->td_contentcss($css);
    array_push($this->groups, $group);
  }
  // ....................................................................
  /**
  * Append content to the working cell. Use this method AFTER you have
  * defined the cell with the td() method.
  * @param string $text Content to append to the current cell.
  * @param string $css Style or classname to use for the cell content.
  */
  function td_content($text="", $css="") {
    $group = array_pop($this->groups);
    $group->td_content($text);
    array_push($this->groups, $group);
  }
  // ....................................................................
  /**
  * Set the working cell alignment properties in one hit. Use
  * this method AFTER you have defined the cell with the td() method.
  * @param string $align Horizontal alignment: 'left', 'center', 'right'.
  * @param string $valign Vertical alignment: 'top', 'middle', 'bottom'.
  */
  function td_alignment($align="", $valign="") {
    $group = array_pop($this->groups);
    $group->td_alignment($align, $valign);
    array_push($this->groups, $group);
  }
  // ....................................................................
  /**
  * Set the working cell width and height properties in one hit. Use
  * this method AFTER you have defined the cell with the td() method.
  * @param string $width Number of pixels or % width of this cell.
  * @param string $height Number of pixels or % height of this cell.
  */
  function td_metrics($width="", $height="") {
    $group = array_pop($this->groups);
    $group->td_metrics($width, $height);
    array_push($this->groups, $group);
  }
  // ....................................................................
  /**
  * Set the working cell width property. Use this method AFTER you
  * have defined the cell with the td() method.
  * @param string $width Number of pixels or % width of this cell.
  */
  function td_width($width="") {
    $group = array_pop($this->groups);
    $group->td_width($width);
    array_push($this->groups, $group);
  }
  // ....................................................................
  /**
  * Set the working cell height property. Use this method AFTER you
  * have defined the cell with the td() method.
  * @param string $height Number of pixels or % height of this cell.
  */
  function td_height($height="") {
    $group = array_pop($this->groups);
    $group->td_height($height);
    array_push($this->groups, $group);
  }
  // ....................................................................
  /**
  * Set the working cell colspan property. Use this method AFTER you
  * have defined the cell with the td() method.
  * @param integer $span Number of columns this cell will span.
  */
  function td_colspan($span=1) {
    $group = array_pop($this->groups);
    $group->td_colspan($span);
    array_push($this->groups, $group);
  }
  // ....................................................................
  /**
  * Set the working cell rowspan property. Use this method AFTER you
  * have defined the cell with the td() method.
  * @param integer $span Number of rows this cell will span.
  */
  function td_rowspan($span=1) {
    $group = array_pop($this->groups);
    $group->td_rowspan($span);
    array_push($this->groups, $group);
  }
  // ....................................................................
  /** Alias for td_css() */
  function th_css($css="")                     { $this->td_css($css);  }
  /** Alias for td_contentcss() */
  function th_contentcss($css="")              { $this->td_contentcss($css); }
  /** Alias for td_content() */
  function th_content($text="", $css="")       { $this->td_content($text, $css); }
  /** Alias for td_alignment() */
  function th_alignment($align="", $valign="") { $this->td_alignment($align, $valign); }
  /** Alias for td_metrics() */
  function th_metrics($width="", $height="")   { $this->td_metrics($width, $height); }
  /** Alias for td_width() */
  function th_width($width="")                 { $this->td_width($width); }
  /** Alias for td_height() */
  function th_height($height="")               { $this->td_height($height); }
  /** Alias for td_colspan() */
  function th_colspan($span="")                { $this->td_colspan($span); }
  /** Alias for td_rowspan() */
  function th_rowspan($span="")                { $this->td_rowspan($span); }
  // ....................................................................
  /**
  * Return the index of the given group type.
  * @param string $tag The tagname of the group to return the index of
  * @return integer The index of the group type
  * @access private
  */
  function get_group_index($tag) {
    for ($gix=0; $gix < count($this->groups); $gix++) {
      $gp = $this->groups[$gix];
      if ($gp->tag == $tag) {
        return $gix;
        break;
      }
    }
    return -1;
  }
  // ....................................................................
  /**
  * Poke some content into the given row,col cell..
  * @param integer $row The row that the cell is in (0 = first row)
  * @param integer $col The column that the cell is in
  * @param string $content The content to put in the cell
  * @param string $css Cell css setting, or omit to not set it.
  * @param string $contentcss Cell content css setting, or omit to not set it.
  * @param string $group The group to find the rows
  * @return bool True if cell exists and was poked with the content
  */
  function poke_cell($row, $col, $content, $css="", $contentcss="", $group="tbody") {
    $rc = false;
    $gpix = $this->get_group_index($group);
    if ($gpix != -1) {
      $gp = $this->groups[$gpix];
      $rc = $gp->poke_cell($row, $col, $content, $css, $contentcss);
      $this->groups[$gpix] = $gp;
    }
    return $rc;
  }
  // ....................................................................
  /**
  * Poke a css class/style onto the given row,col cell..
  * @param integer $row The row that the cell is in (0 = first row)
  * @param integer $col The column that the cell is in
  * @param string $css The class or style to put on the cell
  * @param string $contentcss Cell content css setting, or omit to not set it.
  * @param string $group The group to find the rows
  * @return bool True if cell exists and was poked with the css
  */
  function poke_cellcss($row, $col, $css, $contentcss="", $group="tbody") {
    $rc = false;
    $gpix = $this->get_group_index($group);
    if ($gpix != -1) {
      $gp = $this->groups[$gpix];
      $rc = $gp->poke_cellcss($row, $col, $css, $contentcss);
      $this->groups[$gpix] = $gp;
    }
    return $rc;
  }
  // ....................................................................
  /**
  * Peek content from the given row,col cell..
  * @param integer $row The row that the cell is in (0 = first row)
  * @param integer $col The column that the cell is in
  * @param string $group The group to find the rows
  * @return string The cell contents, or nullstring if none/not valid
  */
  function peek_cell($row, $col, $group="tbody") {
    $rc = "";
    $gpix = $this->get_group_index($group);
    if ($gpix != -1) {
      $gp = $this->groups[$gpix];
      $rc = $gp->peek_cell($row, $col);
    }
    return $rc;
  }
  // ....................................................................
  /**
  * Return the given cell from this group
  * @param integer $row The row that the cell is in (0 = first row)
  * @param integer $col The column that the cell is in (0 = first cell)
  * @param string $group The group to find the rows
  * @return string The table cell object, or false if none/not valid
  */
  function get_cell($row, $col, $group="tbody") {
    $cell = false;
    $gpix = $this->get_group_index($group);
    if ($gpix != -1) {
      $gp = $this->groups[$gpix];
      $cell = $gp->get_cell($row, $col);
    }
    return $cell;
  }
  // ....................................................................
  /**
  * Replace the given cell with new table cell. Returns true if ok.
  * @param integer $row The row that the cell is in (0 = first row)
  * @param integer $col The column that the cell is in
  * @param object $cell The new replacement table cell object
  * @param string $group The group to find the rows
  */
  function set_cell($row, $col, $cell, $group="tbody") {
    $rc = false;
    $gpix = $this->get_group_index($group);
    if ($gpix != -1) {
      $gp = $this->groups[$gpix];
      $rc = $gp->set_cell($row, $col, $cell);
      $this->groups[$gpix] = $gp;
    }
    return $rc;
  }
  // ....................................................................
  /**
  * Check if given cell exists, return True is it does.
  * @param integer $row The row that the cell is in (0 = first row)
  * @param integer $col The column that the cell is in
  * @param string $group The group to find the rows
  * @return bool True if given cell exists, else false.
  */
  function cell_exists($row, $col, $group="tbody") {
    $rc = false;
    $gpix = $this->get_group_index($group);
    if ($gpix != -1) {
      $gp = $this->groups[$gpix];
      if (isset($gp->rows[$row])) {
        $r = $gp->rows[$row];
        $rc = isset($r->cells[$col]);
      }
    }
    return $rc;
  }
  // ....................................................................
  /**
  * Set permission on given table cell. Optionally specify the group
  * which is being targetted (default table body). The specified cell
  * will have its access permission set to the given perm.
  * @param integer $row The row that the cell is in (0 = first row)
  * @param integer $col The column that the cell is in
  * @param mixed $agentids List or Array of unique IDs of agents to assign the permission for
  * @param integer $perm The permission of combination of perms to assign
  * @param string $group The group the cell is found in (defaults to table body)
  */
  function permit_cell($row, $col, $agentids, $perm, $group="tbody") {
    if ($this->cell_exists($row, $col)) {
      $cell = $this->get_cell($row, $col, $group);
      $cell->permit($agentids, $perm);
      $cell->cellid = "_tcell|$row|$col|$group";
      $this->set_cell($row, $col, $cell, $group);
    }
  }
  // ....................................................................
  /**
  * Unset permission on given table cell. Optionally specify the group
  * which is being targetted (default table body). The specified cell
  * will have its access permission unset from the given perms.
  * @param integer $row The row that the cell is in (0 = first row)
  * @param integer $col The column that the cell is in
  * @param mixed $agentids List or Array of unique IDs of agents to unassign the permission for
  * @param integer $perm The permission of combination of perms to unassign
  * @param string $group The group the cell is found in (defaults to table body)
  */
  function unpermit_cell($row, $col, $agentids, $perm, $group="tbody") {
    if ($this->cell_exists($row, $col)) {
      $cell = $this->get_cell($row, $col, $group);
      $cell->unpermit($agentids, $perm);
      $cell->cellid = "_tcell|$row|$col|$group";
      $this->set_cell($row, $col, $cell, $group);
    }
  }
  // ....................................................................
  /**
  * Merge rows at a given cell column.
  * @param integer $row The anchor row for the row merge to begin
  * @param integer $col The column for the row merge
  * @param integer $span The number of rows to merge
  * @param string $group The group to merge rows in
  */
  function merge_rows($row, $col, $span, $group="tbody") {
    $gpix = $this->get_group_index($group);
    if ($gpix != -1) {
      $gp = $this->groups[$gpix];
      $gp->merge_rows($row, $col, $span);
      $this->groups[$gpix] = $gp;
    }
  }
  // ....................................................................
  /**
  * Split rows at a given cell column.
  * @param integer $row The anchor row for the row split to begin
  * @param integer $col The column for the row split
  * @param string $group The group to split rows in
  */
  function split_rows($row, $col, $group="tbody") {
    $gpix = $this->get_group_index($group);
    if ($gpix != -1) {
      $gp = $this->groups[$gpix];
      $gp->split_rows($row, $col);
      $this->groups[$gpix] = $gp;
    }
  }
  // ....................................................................
  /**
  * Apply a colspan in an existing table. The row and column of the
  * cell which has the colspan is given. As a result the relevant
  * spanned cells will be merged in the table. If the span goes
  * beyond the row end, it is truncated. NB: If the spanning cell
  * is already spanning cells, then the span will extend this by the
  * number given.
  * @param integer $row The anchor row for the cell merge to begin
  * @param integer $col The anchor column for the cell merge
  * @param integer $span The number of cells to merge
  * @param string $group The group to merge cells in
  */
  function merge_cols($row, $col, $span, $group="tbody") {
    $gpix = $this->get_group_index($group);
    if ($gpix != -1) {
      $gp = $this->groups[$gpix];
      $gp->merge_cols($row, $col, $span);
      $this->groups[$gpix] = $gp;
    }
  }
  // ....................................................................
  /**
  * Remove a colspan from an existing table. The row and column of the
  * cell which has the colspan is given. As a result the relevant
  * spanned cells will be removed from the table.
  * @param integer $row The anchor row for the cell split to begin
  * @param integer $col The anchor column for the cell split
  * @param string $group The group to split cells in
  */
  function split_cols($row, $col, $group="tbody") {
    $gpix = $this->get_group_index($group);
    if ($gpix != -1) {
      $gp = $this->groups[$gpix];
      $gp->split_cols($row, $col);
      $this->groups[$gpix] = $gp;
    }
  }
  // ....................................................................
  /**
  * Insert a column into all rows in the table. If the template cell is
  * not provided, then a 'vanilla' cell is used instead.
  * @param integer $col Column position to insert at, in the table
  * @param object $celltoinsert Template cell to use for inserting
  */
  function insert_cols($col, $celltoinsert=false) {
    foreach ($this->groups as $gp) {
      $gpix = $this->get_group_index($gp->tag);
      $gp->insert_cols($col, $celltoinsert);
      $this->groups[$gpix] = $gp;
    }
  }
  // ....................................................................
  /**
  * Append columns onto all rows in the table.
  * @param integer $repeat Number of columns to append to table
  * @param object $celltoappend Template cell to use for appending
  */
  function append_cols($repeat=1, $celltoappend=false) {
    foreach ($this->groups as $gp) {
      $gpix = $this->get_group_index($gp->tag);
      $gp->append_cols($repeat, $celltoappend);
      $this->groups[$gpix] = $gp;
    }
  }
  // ....................................................................
  /**
  * Remove a column from all rows in the table
  * @param integer $col Column position to remove from the table
  */
  function delete_cols($col) {
    foreach ($this->groups as $gp) {
      $gpix = $this->get_group_index($gp->tag);
      $gp->delete_cols($col);
      $this->groups[$gpix] = $gp;
    }
  }
  // ....................................................................
  /**
  * Remove a row from the table. If the template cell is not provided
  * then a 'vanilla' cell is used.
  * @param integer $row Row position to remove from the table
  * @param string $group The group to remove row from
  */
  function delete_row($row, $group="tbody") {
    $gpix = $this->get_group_index($group);
    if ($gpix != -1) {
      $gp = $this->groups[$gpix];
      $gp->delete_row($row);
      $this->groups[$gpix] = $gp;
    }
  }
  // ....................................................................
  /**
  * Return a given row from the table.
  * @param integer $row Row position to remove from the table
  * @return object The row reqested, or false if non-existent.
  */
  function get_row($row, $group="tbody") {
    $therow = false;
    $gpix = $this->get_group_index($group);
    if ($gpix != -1) {
      $gp = $this->groups[$gpix];
      $therow = $gp->get_row($row);
    }
    return $therow;
  }
  // ....................................................................
  /**
  * Insert a row into the table. If the template cell is not provided,
  * then a 'vanilla' cell is used instead. We insert the new row just
  * before the given row.
  * @param integer $row Row position to insert before, in the table
  * @param object $celltoinsert Template cell to use for all row cells
  */
  function insert_row($row, $celltoinsert=false) {
    foreach ($this->groups as $gp) {
      $gpix = $this->get_group_index($gp->tag);
      $gp->insert_row($row, $celltoinsert);
      $this->groups[$gpix] = $gp;
    }
  }
  // ....................................................................
  /**
  * Append a row onto the table.
  * @param object $celltoappend Template cell to use for appending the row
  * @param string $group The group (default tbody) to append the row to
  */
  function append_row($celltoappend=false, $group="tbody") {
    $gpix = $this->get_group_index($group);
    if ($gpix != -1) {
      $gp = $this->groups[$gpix];
      $gp->append_row($celltoappend);
      $this->groups[$gpix] = $gp;
    }
  }
  // ....................................................................
  /**
  * Set a width profile across the row of cells. The profile is passed
  * as either a string or an array. If a string, the widths should be
  * separated by commas. The widths can be absolute pixel values or %'s,
  * and applies these widths to the first row of the given group.
  * An example string might for instance be '20%,15%,65%'.
  * @param mixed $prof The width profile, array or comma-delimeted string
  * @param string $group The group (default tbody) to append the row to
  */
  function set_width_profile($prof, $group="tbody") {
    // Close any open group..
    $this->close_group();
    $wprofile = array();
    if (is_array($prof)) {
      $wprofile = $prof;
    }
    elseif (is_string($prof)) {
      $wprofile = explode(",", $prof);
    }
    $gpix = $this->get_group_index($group);
    if ($gpix != -1) {
      $gp = $this->groups[$gpix];
      $gp->set_width_profile($wprofile);
      $this->groups[$gpix] = $gp;
    }
  }
  // ....................................................................
  /**
  * Get the width profile from the table.
  * @param string $group The group (default tbody) to append the row to
  * @return array Width profile as an array of widths
  */
  function get_width_profile($group="tbody") {
    $wprofile = array();
    $gpix = $this->get_group_index($group);
    if ($gpix != -1) {
      $gp = $this->groups[$gpix];
      $wprofile = $gp->get_width_profile();
      $this->groups[$gpix] = $gp;
    }
    return $wprofile;
  }
  // ....................................................................
  /** Render as WML.
  * @return string The table as WML. */
  function wml() {
    return $this->html();
  }
  // ....................................................................
  /**
  * Render the whole table as HTML
  * If dedbugging mode is enabled and the DBG_TABLES flag is set, then
  * we run a simple validation check, and report findings, as well as
  * making sure the table borders are visible.
  * @return string The HTML for this table.
  */
  function html() {
    // Close any open group..
    $this->close_group();

    // If we are debugging tables, make sure the border is displayed..
    if (debugging() && (debug_class() & DBG_TABLES)) {
      $this->setborder(1);
    }
    $s = "";
    if (count($this->groups) > 0) {
      $cols = $this->colcount();
      $rows = $this->rowcount();
      $s .= "\n<!--TBL[$this->tablename]$rows" . "x" . "$cols-->\n";
      $s .= "<table";
      $s .= " cellpadding=\"$this->cellpadding\"";
      $s .= " cellspacing=\"$this->cellspacing\"";
      $s .= $this->attributes();
      $s .= ">\n";

      // Auto-justify heading cells of thead group. This can only
      // be done here, when we have tbody content available to us
      // since we justify heading cells from first body row cells..
      if ($this->autojustify) {
        $gpix = $this->get_group_index("tbody");
        if ($gpix != -1) {
          $gp = $this->groups[$gpix];
          $gp->autojustify();
          $this->groups[$gpix] = $gp;
        }
        // Now deal with any heading row..
        for ($gix=0; $gix < count($this->groups); $gix++) {
          $gp = $this->groups[$gix];
          switch ($gp->tag) {
            case "thead": $gp_head = $gp; $gp_headix = $gix; break;
            case "tbody": $gp_body = $gp; $gp_bodyix = $gix; break;
          }
        }
        if (isset($gp_head) && isset($gp_body)) {
          $headrow = $gp_head->rows[0]; $bodyrow = $gp_body->rows[0];
          for ($col=0; $col < count($headrow->cells); $col++) {
            if (isset($bodyrow->cells[$col])) {
              $cell = $bodyrow->cells[$col];
              if (is_numeric($cell->content->content)) $css = "text-align:right;";
              else $css = "text-align:left;";
              $cell = $headrow->cells[$col];
              $cell->setcss($css);
              $headrow->cells[$col] = $cell;
            }
          }
          $gp_head->rows[0] = $headrow;
          $this->groups[$gp_headix] = $gp_head;
        }
      }

      // Apply rowstriping to tbody if defined..
      if (count($this->rowstripes) > 0) {
        $gpix = $this->get_group_index("tbody");
        if ($gpix != -1) {
          $gp = $this->groups[$gpix];
          $gp->apply_rowstripes($this->rowstripes);
          $this->groups[$gpix] = $gp;
        }
      }

      // Now do each group tag in the order it should be. Note
      // that these may have been added in a different order
      // so we force it to be thead -> tbody -> tfoot here..
      $thead = ""; $tbody = ""; $tfoot = "";
      for ($g = 0; $g < count($this->groups); $g++) {
        $group = $this->groups[$g];
        switch ($group->tag) {
          case "thead":
            $thead .= $group->html();
            break;
          case "tbody":
            $tbody .= $group->html();
            break;
          case "tfoot":
            $tfoot .= $group->html();
            break;
        } // switch
        // Group may have altered itself in the
        // act of being rendered..
        $this->groups[$g] = $group;
      } // for

      // Glue them together in correct order..
      $s .= $thead . $tbody . $tfoot;

      $s .= "</table>\n";
      $s .= "<!--END[$this->tablename]-->";
    }
    // If we are debugging tables, display validation now..
    if (debugging() && (debug_class() & DBG_TABLES)) {
      $this->validate();
    }
    // Return the HTML
    return $s;
  }
  // ....................................................................
  /**
  * Render the table as a CSV formatted stream. This is designed
  * to facilitate the exporting of complex tables of data as CSV format
  * for importing into spreadsheets, or databases etc.
  * @return string The table content in CSV format.
  */
  function csv() {
    $s = "";
    // Close any open group..
    $this->close_group();
    if (count($this->groups) > 0) {
      $thead = ""; $tbody = ""; $tfoot = "";
      for ($g = 0; $g < count($this->groups); $g++) {
        $group = $this->groups[$g];
        switch ($group->tag) {
          case "thead":
            $thead .= $group->csv();
            break;
          case "tbody":
            $tbody .= $group->csv();
            break;
          case "tfoot":
            $tfoot .= $group->csv();
            break;
        } // switch
      } // for
      // Glue them together in correct order..
      $s .= $thead . $tbody . $tfoot;
    }
    // Return the csv stream..
    return $s;
  } // csv
} // table class

// ----------------------------------------------------------------------
/**
* A matrix is a table with no colspans or rowspans. It is an N x M
* rectangle of table cells, and created to the given size from the
* outset. This is just a simple way in which you can obtain s vanilla
* table of given dimensions, with a single statement. All the table
* methods are then available.
* @package html
*/
class matrix extends table {
  /** Rows in table
      @access private */
  var $rows = 1;
  /** Columns in table
      @access private */
  var $cols = 1;
  /**
  * Construct a new matrix.
  * @param integer $rows Number of table rows in the matrix
  * @param integer $cols Number of table columns in the matrix
  * @param string $content Optional content for every cell of the table
  */
  function matrix($rows, $cols, $content="") {
    $this->table();
    for ($r = 0; $r < $rows; $r++) {
      $this->tr();
      for ($c = 0; $c < $cols; $c++) {
        $this->td($content);
      }
    }
    // Close any open group..
    $this->close_group();
  } // matrix
} // matrix class

// ----------------------------------------------------------------------
/**
* [A] - The anchor tag, otheriwse know as a clickable link.
* @package html
*/
class anchor extends HTMLObject {
  // Public
  // Private
  /** URL that clicking the link goes to
      @access private */
  var $href = "";
  /** Link label text
      @access private */
  var $label = "";
  /** Stylesheet class name for highlighted link
      @access private */
  var $highlightclass = false;
  // ....................................................................
  /** Constructor
  * @param string $css The style or classname to apply to the object.
  */
  function anchor($href="", $label="", $css="") {
    $this->HTMLObject($css);
    $this->href = $href;
    $this->label = $label;
  } // anchor
  // ....................................................................
  /**
  * Defines special highlight class name
  * Defines the name of a class in your stylesheet to use for the
  * purpose of highlighting. This property is initialised to be
  * 'false', but if defined, then the link is spanned by a <span>
  * tag with the given class name to highlight it accordingly.
  */
  function highlight($highlightclass) {
    $this->highlightclass = $highlightclass;
  } // highlight
  // ....................................................................
  /** Render the object as HTML */
  function html() {
    if ($this->href != "") {
      $s  = "<a href=\"$this->href\"";
      $s .= $this->attributes();
      $s .= ">";
      if ($this->highlightclass != "") {
        $s .= "<span class=\"$this->highlightclass\">".$this->label."</span>";
      } else {
          $s .= $this->label;
      }
      $s .= "</a>";
    }
    else {
      if ($this->highlightclass != "") {
        $s .= "<span class=\"$this->highlightclass\">".$this->label."</span>";
      } else {
          $s .= $this->label;
      }
    }
    // Return the HTML..
    return $s;
  } // html
} // anchor class

// ----------------------------------------------------------------------
/**
* [HR] - Horizontal Rule
* @package html
*/
class hr extends HTMLObject {
  // Public
  // Private
  /** Whether HR is shaded or not
      @access private */
  var $shade = false;
  /** Colour of the element
      @access private */
  var $colour = "";
  // ....................................................................
  /** Constructor
  * @param string $width The width of the rule eg: '100%', '550' etc.
  * @param integer $size The standard thickness of the rule (px)
  * @param string $colour The colour code of the rule
  * @param string $css The style or classname to apply to the object.
  */
  function hr($width="100%", $size=1, $colour="", $css="") {
    $this->HTMLObject($css);
    $this->setwidth($width);
    $this->setsize($size);
    $this->colour = $colour;
  } // hr
  // ....................................................................
  /** Cause HR to be shaded */
  function shade() {
    $this->shade = true;
  } // ade
  // ....................................................................
  /** Cause HR to be unshaded */
  function noshade() {
    $this->shade = false;
  } // noshade
  // ....................................................................
  /** Render the object as HTML
  * @return string The HTML of this object
  */
  function html() {
    global $RESPONSE;
    $s = "<hr";
    if (!$this->shade) {
      $s .= " noshade";
    }
    if ($this->size > 0) {
      $this->setcss("height:$this->size" . "px;");
    }
    // The different browsers are pretty confused about HR colour..
    if ($this->colour != "") {
      if (isset($RESPONSE) && $RESPONSE->browser == BROWSER_IE) {
        $this->setcss("color:$this->colour;");
      }
      else {
        $this->setcss("background-color:$this->colour;");
      }
    }
    $s .= $this->attributes();
    $s .= ">\n";
    // Return the HTML..
    return $s;
  } // html
} // hr class

// ----------------------------------------------------------------------
/**
* Item list
* Internal list class used to generate list HTML content
* @package html
* @access private
*/
class itemlist extends HTMLobject {
  // Public
  // Private
  /** Array of items (strings) to display
      @access private */
  var $list = array();
  /** Type of list: 'list', 'ordered', or 'bullets'
      @access private */
  var $type = "bullets";
  // ....................................................................
  /** Constructor
  * @param string $type The type of list to create
  * @param string $css The style or classname to apply to the object.
  */
  function itemlist($type, $css="") {
    $this->type = $type;
    $this->HTMLobject($css);
  } // itemlist
  // ....................................................................
  /**
  * Add an item to the list.
  * @param string $item The item to add
  */
  function additem($item="") {
    $this->list[] = $item;
  } // additem
  // ....................................................................
  /**
  * Render the HTML
  * @return string The HTML for this list.
  */
  function html() {
    $s = "";
    if (count($this->items) > 0) {
      $tag = (($this->type == "bullets") ? "ul" : "ol");
      $s = "<$tag" . $this->attributes() . ">";
      foreach ($this->items as $item) {
        if ($item != "") {
          $s .= "<li" . $this->attributes() . ">$item";
        }
      }
      $s .= "</$tag>";
    }
    // Return the HTML..
    return $s;
  } // html
} // itemlist class

/**
* [OL] - Ordered list
* @package html
*/
class ol extends itemlist {
  /** Constructor
  * @param string $css The style or classname to apply to the object.
  */
  function ol($css="") {
    $this->itemlist("ordered", $css);
  }
} // ol
/**
* [UL] - Unordered list
* @package html
*/
class ul extends itemlist {
  /** Constructor
  * @param string $css The style or classname to apply to the object.
  */
  function ul($css="") {
    $this->itemlist("bullets", $css);
  }
} // ul
/**
* [VL] - Vanilla list (suppressed markers)
* @package html
*/
class vl extends itemlist {
  /** Constructor
  * @param string $css The style or classname to apply to the object.
  */
  function vl($css="") {
    $this->itemlist("bullets", $css);
    $this->setcss("list-style:none;");
  }
} // vl

// ----------------------------------------------------------------------
/**
* [IMG] - The img class. This is an object which renders the img tag.
* @package html
*/
class img extends HTMLObject {
  /** Name of an image map */
  var $map;
  /** URL associated with the image */
  var $url;
  /** Target frame for URL */
  var $target;
  /** Image which acts as a clickable icon */
  var $icon;
  /** The image type: "GIF", "JPG", "PNG", "TIFF", "BMP".. */
  var $image_type = "";
  //.....................................................................
  /**
  * Constructor
  * The src field is mandatory. The name is optional but if given it will
  * also be used for the ALT tag. If a tooltip is not given then the name
  * will also be used for that attribute.
  * NB: If the width and/or height are not given and GD is installed it
  * will be used to find the image dimensions, otherwise these attributes
  * will be supressed and the browser left to work it out.
  * @param string $src The path/URL of the image file
  * @param string $name Name of the image object - also used as ALT tag
  * @param string $tooltip The tooltip appears on mouseover for most browsers
  * @param integer $width The width of the image in pixels
  * @param integer $height The height of the image in pixels
  */
  function img($src, $name="", $tooltip="", $width=false, $height=false) {
    $this->HTMLObject();
    $this->set_image($src, $width, $height);
    $this->setborder(0);
    // Set name to image file if not given..
    if ($name == "") {
      $fbits = explode(".", basename($src));
      $name = $fbits[0];
    }
    $this->setname($name);
    $this->setalt($name);
    $this->settitle($tooltip);
    if ($this->title == "") {
      $this->settitle($name);
    }
  } // img
  //.....................................................................
  /**
  * Set the image src and details for this image object.
  * @param string $src The path/URL of the image file
  * @param integer $width The width of the image in pixels
  * @param integer $height The height of the image in pixels
  */
  function set_image($src, $width=false, $height=false) {
    global $RESPONSE;
    $this->setsrc($src);
    // Get image sizing if not specified. Try to ascertain the
    // physical file path. We skip this for remote images..
    $imgpath = $src;
    if ($imgpath != "" && !protocol_prefixed($imgpath)) {
      if (substr($imgpath, 0, 1) == "/") {
        $imgpath = substr($imgpath, 1);
      }
      if (isset($RESPONSE)) {
        $imgpath = "$RESPONSE->site_docroot/$imgpath";
      }
      if (is_readable($imgpath)) {
        $imginfo = getimagesize($imgpath);
        switch ($imginfo[2]) {
          case  1: $this->image_type = "GIF";  break;
          case  2: $this->image_type = "JPG";  break;
          case  3: $this->image_type = "PNG";  break;
          case  4: $this->image_type = "SWF";  break;
          case  5: $this->image_type = "PSD";  break;
          case  6: $this->image_type = "BMP";  break;
          case  7: $this->image_type = "TIFF"; break;
          case 15: $this->image_type = "WBMP"; break;
          default: "??";
        } // switch
        if ($width  === false) $width  = $imginfo[0];
        if ($height === false) $height = $imginfo[1];
      }
    }
    if ($width  !== false) $this->setwidth($width);
    if ($height !== false) $this->setheight($height);
  } // set_image
  //.....................................................................
  /** Set URL. When the image is clicked, the browser will target the
  * URL. Note that this will not work if you have defined the onclick
  * event handler.
  * @param string $url The URL to go to when image is clicked.
  * @param string $target The optional target frame, eg: '_blank' etc.
  */
  function seturl($url, $target="") {
    $this->url = $url;
    $this->target = $target;
  } // seturl
  //.....................................................................
  /** Set image map
  * Defines the image map to use with this image.
  * @param string $map The name of the image map to associate with this image.
  */
  function use_map($map) {
    $this->map = $map;
  } // use_map
  // ....................................................................
  /** Render image as javascript object
  * @return string Javascript code rendering of this image
  */
  function javascript() {
    $js  = "var $this->name=new Image($this->width,$this->height);\n";
    $js .= "var $this->name src=\"$this->src\";\n";
    return $js;
  } // javascript
  // ....................................................................
  /** Render as WML.
  * @return string The image as WML. */
  function wml() {
    return $this->html();
  } // wml
  // ....................................................................
  /**
  * Render the image object as an image 'icon' which can be clicked to
  * display the object in a popup window.
  * @param string $tooltip Optional browser mouseover tooltip text
  * @param object $iconimage A custom image object
  */
  function AsIcon($tooltip="", $iconimage=false) {
    global $RESPONSE, $LIBDIR;
    if ($iconimage) {
      $this->icon = $iconimage;
    }
    else {
      $this->icon = new img("$LIBDIR/img/_image.gif", "Image");
    }
    if ($tooltip != "") {
      $this->icon->title = $tooltip;
    }
    // Display the icon with onclickability..
    if (isset($RESPONSE)) {
      $RESPONSE->body->add_popup_script(
          "vwimg_$this->name",
          $this->width,
          $this->height,
          false,
          false,
          "toolbar=no,status=no,scrollbars=no,resizable=yes"
          );
      $this->icon->set_onclick("vwimg_$this->name" . "('$this->src')");
      //$s = $this->icon->render();
    }
    else {
      // Use vanilla new browser window approach..
      $this->url = $this->src;
      $this->set_image("$LIBDIR/img/_image.gif");
      $this->target = "_new";
      //$s = $this->html();
    }
    //return $s;
  } // AsIcon
  // ....................................................................
  /** Render as HTML
  * @return string The image as HTML. */
  function html() {
    global $RESPONSE;

    if ( isset($this->icon) ) {
      $s = $this->icon->render();
    } else {
      $s = "";
      if (isset($this->url) && $this->url != "") {
        $s .= "<a href=\"$this->url\"";
        if ($this->target != "") {
          $s .= " target=\"$this->target\"";
        }
        $s .= ">";
      }
      $s .= "<img";
      if (isset($this->map) && $this->map != "") {
        $s .= " usemap=\"$this->map\"";
      }
      $s .= $this->attributes();
      $s .= ">";
      if (isset($this->url) && $this->url != "") {
        $s .= "</a>";
      }
    }
    return $s;
  } // html
} // img class

// ----------------------------------------------------------------------
/**
* Deprecated embed Object (HTML3.2)
* This object supports the older embed tag used to embed objects
* into HTML pages. for backward compatibility. In reality there is
* still a lot of mileage left in this tag.
* @package html
*/
class embed extends HTMLObject {
  // Public
  /** Name of Java file, if java applet */
  var $code;
  /** Base URL for plug-in or potenial java applet (IE) */
  var $codebase;
  /** URL of page to acquire the relevant object plugin */
  var $pluginspage;
  /** The MIME type of the object */
  var $mimetype;
  /** If true the object is hidden */
  var $hidden = false;

  // Private
  /** Array of non-std attributes as name/value pairs
      @access private */
  var $other_attributes = array();
  // ....................................................................
  /** Constructor
  * @param string $src The URL of the source data for this object
  * @param string $mimetype The MIMETYPE of this object, eg: 'application/x-sockwave-flash' etc.
  * @param string $pluginspage The URL for downloading the relevant plugin
  * @param string $width Width of object in pixels
  * @param string $height Height of object in pixels
  * @param string $alt ALT text to display if object does not execute
  */
  function embed($src="", $mimetype="", $pluginspage="", $width="", $height="", $alt="", $hidden=false) {
    $this->HTMLObject();
    if ($src != "")         $this->setsrc($src);
    if ($mimetype != "")    $this->mimetype = $mimetype;
    if ($pluginspage != "") $this->pluginspage = $pluginspage;
    if ($width != "")       $this->setwidth($width);
    if ($height != "")      $this->setheight($height);
    if ($alt != "")         $this->setalt($alt);
    $this->hidden = $hidden;
  } // embed
  // ....................................................................
  /**
  * Specify the parameters required for a Java applet, the name
  * of the file containing the applet code, and the URL to find
  * the file at.
  * @param string $code The name of the file contaiing the applet
  * @param string $codebase The URL to locate the code file at
  */
  function java_applet($code, $codebase="") {
    $this->code = $code;
    if ($codebase != "") $this->codebase = $codebase;
  } // java_applet
  // ....................................................................
  /**
  * Add non-standard attribute to be rendered.
  * @param string $name Name of the new attribute
  * @param string $value Value of the new attribute
  */
  function add_attribute($name, $value) {
    $this->other_attributes[$name] = $value;
  } // add_attribute
  // ....................................................................
  function html() {
    $s = "";
    $s .= "<embed";
    if (isset($this->pluginspage)) $s .= " pluginspage=\"$this->pluginspage\"";
    if (isset($this->mimetype))    $s .= " type=\"$this->mimetype\"";
    if (isset($this->code))        $s .= " code=\"$this->code\"";
    if (isset($this->codebase))    $s .= " codebase=\"$this->codebase\"";
    if (isset($this->quality))     $s .= " quality=\"$this->quality\"";
    $s .= " hidden=" . ($this->hidden ? "\"true\"" : "\"false\"");
    $s .= $this->attributes();
    while ( list($name, $value) = each($this->other_attributes)) {
      $s .= " $name=\"$value\"";
    }
    $s .= ">";
    // Return the HTML..
    return $s;
  } // html
} // embed class

// ----------------------------------------------------------------------
/**
* Object
* This HTML4 entity is the preferred method for embedding objects into a
* webpage using the object tag. The only drawback is that a lot of browsers
* don't support it fully, hence the older embed tag is also available for
* use as a fallback mechanism via the add_embed() method.
* @see function add_embed
* @package html
*/
class EmbeddedObject extends HTMLObject {
  // Public
  /** MIME type of data required by the object */
  var $mimetype = "";
  /** URL for the object's implementation */
  var $classid;
  /** URL for relative base for accessing object
      specified by the classid */
  var $codebase;
  /** URL of data object might require */
  var $data;
  /** Object MIME type */
  var $codetype;
  /** Standby message text */
  var $standby;
  /** URL of page to acquire the relevant embed plugin */
  var $pluginspage = "";
  /** MIMETYPE of file by using its extension */
  var $extn_mimetype = "";
  /** Mimetype category, eg: 'movie', 'audio' etc. */
  var $category = "";
  /** Image which acts as a play icon */
  var $icon;

  // Private
  /** Array of parameter key-values
      @access private */
  var $params = array();
  /** If set, contains a EmbeddedObject_Compat object to
      be rendered in this object for compatibility
      @access private */
  var $embed_obj;
  /** Internal name sans spaces
      @access private */
  var $internal_name = "obj";
  /** If not empty, render as a link using $aslink as the text
      @access private */
  var $aslink = "";
  /** Target frame for the $aslink
      @access private */
  var $aslink_target = "";
  // ....................................................................
  /** Constructor
  * @param string $src URL of the source datafor the object, if any
  * @param string $mimetype Mimetypeof the embedded object
  * @param string $classid The classid (CLSID) unique ID of this object
  * @param integer $width Width of object
  * @param integer $height Height of object
  * @param string $codebase URL of code location for this object
  * @param string $standby Standby message whilst loading
  */
  function EmbeddedObject(
      $src="",
      $mimetype="",
      $classid="",
      $width=0,
      $height=0,
      $codebase="",
      $standby="Loading..")
  {
    $this->HTMLobject();
    // Deal with source file, mimetype and category..
    if($src != "") {
      $this->data = $src;
      $file = basename($src);
      $fnbits = explode(".", $file);
      $this->setname($fnbits[0]);
      $this->internal_name = md5(preg_replace("/[ \-,.;:_{}()<>\"\'\`\+\=]/", "", $this->name));
      $mime = mimetype_from_filename($src);
      if ($mime) $this->extn_mimetype = $mime;
      else $this->extn_mimetype = "";
      $this->category = mimetype_category($this->extn_mimetype);
    }
    if ($mimetype != "") {
      $this->mimetype = $mimetype;
    }
    else {
      $this->mimetype = $extn_mimetype;
    }
    if ($classid != "") $this->classid = $classid;
    $this->setwidth($width);
    $this->setheight($height);
    if ($codebase != "") $this->codebase = $codebase;
    if ($standby != "") $this->standby = $standby;
  } // EmbeddedObject
  // ....................................................................
  /**
  * Set up a new object parameter name/value pair. This will be rendered
  * as a param name="foo" value="bar" tag.
  * @param string $name Name of the new param entity
  * @param string $value Value of the new param entity
  */
  function setparam($name, $value) {
    if (trim($name) != "") {
      $this->params[trim($name)] = trim($value);
    }
  } // setparam
  // ....................................................................
  /**
  * Add an embed object to provide fallback for browsers which do not
  * support the object tag properly.
  * @param string $src The URL of the source data for this object
  * @param string $pluginspage The URL for downloading the relevant plugin
  */
  function add_embed($src="", $pluginspage="") {
    $this->embed_obj =
        new embed(
            $src,
            $this->mimetype,
            $pluginspage
            );
  } // add_embed
  // ....................................................................
  /**
  * Render the object as a simple link. The parameter passed will be
  * the text of the link, and the URL is determined by the location
  * of the object as specified in the constructor.
  * @param string $linktext Text to use for the link
  */
  function AsLink($linktext="", $target="") {
    $this->aslink = $linktext;
    $this->aslink_target = $target;
  } // AsLink
  // ....................................................................
  /** Render the HTML
  * @return string HTML for this object
  */
  function html() {
    $s = "";
    if($this->aslink != "") {
      $s .= "<a href=\"$this->data\"";
      if ($this->aslink_target != "") {
        $s .= " target=\"_blank\"";
      }
      $s .= ">";
      $s .= $this->aslink;
      $s .= "</a>";
    }
    else {
      $s .= "<object";
      $s .= $this->attributes();
      if (isset($this->classid))  $s .= " classid=\"$this->classid\"";
      if (isset($this->codebase)) $s .= " codebase=\"$this->codebase\"";
      if (isset($this->data))     $s .= " data=\"$this->data\"";
      if (isset($this->codetype)) $s .= " codetype=\"$this->codetype\"";
      if ($this->mimetype != "")  $s .= " type=\"$this->mimetype\"";
      if (isset($this->standby))  $s .= " standby=\"$this->standby\"";
      $s .= ">";

      // Add param items..
      while ( list($name, $value) = each($this->params)) {
        $s .= "<param name=\"$name\" value=\"$value\">";
      }
      // Add fallback <embed> object..
      if (isset($this->embed_obj)) {
        $this->embed_obj->width  = $this->width;
        $this->embed_obj->height = $this->height;
        $this->embed_obj->mimetype = $this->mimetype;
        if (isset($this->standby)) {
          $this->embed_obj->alt = $this->standby;
        }
        $s .= $this->embed_obj->html();
      }
      $s .= "</object>";
    }
    // Return the html..
    return $s;
  } // html
} // EmbeddedObject class

// ----------------------------------------------------------------------
/** Macromedia Flash Windows CLASS ID */
define("FLASH_CLSID", "clsid:D27CDB6E-AE6D-11cf-96B8-444553540000");
/** Macromedia Flash Codebase - where to download the player */
define("FLASH_CODEBASE", "http://download.macromedia.com/pub/shockwave/cabs/flash/swflash.cab#version=5,0,0,0");
/** Macromedia Flash Plugins plage - support for other platforms */
define("FLASH_PLUGINSPAGE", "http://www.macromedia.com/shockwave/download/index.cgi?P1_Prod_Version=ShockwaveFlash");
/** Macromedia Flash MimeType */
define("FLASH_MIMETYPE", "application/x-shockwave-flash");
/**
* Macromedia Shockwave/Flash Object
* @package html
*/
class FlashObject extends EmbeddedObject {
  /* URL of Flash movie */
  var $movie = "";
  /** Quality setting for the movie */
  var $quality = "high";
  // ....................................................................
  /** Constructor
  * @param string $movie URL of the flash movie file
  * @param integer $width Width in pixels of the movie
  * @param integer $height Height in pixels of the movie
  * @param string $quality Quality spec of the movie
  * @param string $standby Optional message to display while loading
  */
  function FlashObject($movie, $width="", $height="", $quality="high", $standby="") {
    $this->movie = $movie;
    // These attributes are defined by Macromedia for Flash content..
    $classid = FLASH_CLSID;
    $codebase = FLASH_CODEBASE;
    $this->pluginspage = FLASH_PLUGINSPAGE;
    $this->EmbeddedObject($movie, FLASH_MIMETYPE, $classid, $width, $height, $codebase, $standby);
    $this->quality = $quality;
  } // FlashObject
  // ....................................................................
  /**
  * Render the flash object as an image 'icon' which can be clicked to play
  * the movie. If image is specified, it must be a valid 'image' object,
  * otherwise a generic library symbol will be used.
  * @param string $tooltip Optional browser mouseover tooltip text
  * @param object $iconimage A custom image object
  * @see img
  */
  function AsIcon($tooltip="", $iconimage=false) {
    global $RESPONSE, $LIBDIR;
    if ($iconimage) {
      $this->icon = $iconimage;
    }
    else {
      $this->icon = new img("$LIBDIR/img/_flash.gif", "Play");
    }
    if ($tooltip != "") {
      $this->icon->title = $tooltip;
    }
  } // AsIcon
  // ....................................................................
  /** Render the HTML
  * @return string HTML for this object
  */
  function html() {
    global $RESPONSE;
    $s = "";
    if ($RESPONSE->browser != BROWSER_IE) {
      // Need one of these for compatibility..
      $this->embed_obj =
            new embed(
              $this->movie,
              FLASH_MIMETYPE,
              $this->pluginspage,
              $this->width,
              $this->height,
              $this->standby
              );
      $this->embed_obj->add_attribute("quality", $this->quality);
    }
    $this->setparam("movie", $this->movie);
    $this->setparam("quality", $this->quality);

    // If we are doing a clickable image, we need the aid of
    // some handy javascript to make it work..
    if (isset($this->icon)) {
      $this->icon->set_onclick("play_$this->internal_name('$this->name')");
      $embeddedMedia = EmbeddedObject::html();
      $RESPONSE->body->add_popup_script("popup_$this->internal_name", $this->width, $this->height, 100, 40);
      $RESPONSE->body->add_script(
          "var playing=false;\n"
        . "function play_$this->internal_name(pname) {\n"
        . "  popup_$this->internal_name('');\n"
        . "  with (popup_" . $this->internal_name . "Win.document) {\n"
        . "    open();\n"
        . "    write('');\n"
        . "    write('<body bgcolor=\"#333333\" leftmargin=\"0\" rightmargin=\"0\" topmargin=\"0\" marginwidth=\"0\" marginheight=\"0\">');\n"
        . "    write('<center><font face=arial,helvetica size=2>');\n"
        . "    write('$embeddedMedia');\n"
        . "    title='$this->name';\n"
        . "    close();\n"
        . "  }\n"
        . "}\n"
        );
      $s .= $this->icon->render();
    }
    else {
      $s .= EmbeddedObject::html();
    }
    // Return html..
    return $s;
  } // html
} // FlashObject class

// ----------------------------------------------------------------------
/** Windows Media Player CLASS ID */
define("WM_CLSID", "CLSID:22D6f312-B0F6-11D0-94AB-0080C74C7E95");
/** Windows Media Player Codebase - where to download the player */
define("WM_CODEBASE", "http://activex.microsoft.com/activex/controls/mplayer/en/nsmp2inf.cab#Version=6,4,7,1112");
/** Windows Media Player - General plugins page for other platforms */
define("WM_PLUGINSPAGE", "http://www.microsoft.com/Windows/MediaPlayer/");
/** Windows Media Player - MimeType */
define("WM_MIMETYPE", "application/x-mplayer2");
/** Used to indicate sound should start playing on load */
define ("AUTOSTART", true);
/** Used to indicate sound should loop indefinitely */
define ("LOOP", true);
/** Used to hide the object control */
define ("HIDDEN", true);
/**
* Generic Multimedia Object
* @package html
*/
class MediaObject extends EmbeddedObject {
  /* URL of media file */
  var $media = "";
  /** If true the media start immediately it is loaded */
  var $autostart = false;
  /** If true the media replays endlessly */
  var $loop = false;
  /** If true the media object is hidden */
  var $hidden = false;
  /** If true the mediaplayer controls are shown */
  var $showcontrols = true;
  // ....................................................................
  /** Constructor
  * @param string $media URL of the media file
  * @param integer $width Width in pixels of the media object
  * @param integer $height Height in pixels of the media object
  * @param boolean $autostart True if you want the media to auto-start on load
  * @param boolean $loop True if you want the media to repeat endlessly
  * @param boolean $hidden True if you want the media to be hidden from view
  * @param string $standby Standby message whilst loading
  */
  function MediaObject($media, $width="", $height="", $autostart=false, $loop=false, $showcontrols=true, $hidden=false, $standby="Loading..") {
    global $RESPONSE, $LIBDIR;
    $this->media = $media;
    $this->autostart = $autostart;
    $this->loop = $loop;
    $this->showcontrols = $showcontrols;
    $this->hidden = $hidden;
    // Set up parameters according to browser..
    if ($RESPONSE->browser == BROWSER_IE) {
      $classid = WM_CLSID;
      $codebase = WM_CODEBASE;
      $this->pluginspage = WM_PLUGINSPAGE;
      $mimetype = WM_MIMETYPE;
    }
    else {
      $mimetype = $this->extn_mimetype;
    }
    $this->EmbeddedObject($media, $mimetype, $classid, $width, $height, $codebase, $standby);
  } // MediaObject
  // ....................................................................
  /**
  * Render the media object as an image 'icon' which can be clicked to play
  * the media. If image is specified, it must be a valid 'image' object,
  * otherwise a generic library symbol will be used.
  * @param string $tooltip Optional browser mouseover tooltip text
  * @param object $iconimage A custom image object
  * @see img
  */
  function AsIcon($tooltip="", $iconimage=false) {
    global $RESPONSE, $LIBDIR;
    // Make sounds invisible..
    switch ($this->category) {
      case "audio":
        $this->hidden = true;
        $this->setwidth(0);
        $this->setheight(0);
        break;
      default:
        $this->hidden = false;
        break;
    }
    if ($iconimage) {
      $this->icon = $iconimage;
    }
    else {
      switch ($this->category) {
        case "audio":
          $this->icon = new img("$LIBDIR/img/_sound.gif", "Play");
          break;
        default:
          $this->icon = new img("$LIBDIR/img/_movie.gif", "Play");
          break;
      } // switch
    }
    if ($tooltip != "") {
      $this->icon->title = $tooltip;
    }
  } // AsIcon
  // ....................................................................
  /** Render the HTML
  * @return string HTML for this object
  */
  function html() {
    global $RESPONSE, $LIBDIR;
    $s = "";
    // Render the object itself..
    $this->embed_obj =
          new embed(
            $this->media,
            $this->extn_mimetype,
            $this->pluginspage,
            $this->width,
            $this->height,
            $this->standby
            );
    $this->embed_obj->hidden = $this->hidden;
    $this->embed_obj->add_attribute("autostart", ($this->autostart ? "true" : "false"));
    $this->embed_obj->add_attribute("loop", ($this->loop ? "true" : "false"));

    // Do Windows Media Player params for Internet Explorer..
    if ($RESPONSE->browser == BROWSER_IE) {
      $this->setparam("FileName", $this->media);
      $this->setparam("PlayCount", ($this->loop ? "0" : "1"));
      $this->setparam("Volume", "4");
      $this->setparam("ShowControls", ($this->showcontrols ? "true" : "false"));
      $this->setparam("TransparentAtStart", ($this->hidden ? "true" : "false"));
      $this->setparam("AutoStart", ($this->autostart ? "true" : "false"));
      $this->setparam("ShowStatusBar", "false");
    }

    // If we are doing a clickable image, we need the aid of
    // some handy javascript to make it work..
    if (isset($this->icon)) {
      switch ($RESPONSE->browser){
        // For IE we can do the full nine yards and open a special
        // window where we put the embedded object..
        case BROWSER_IE:
          // Play all media in separate window..
          $this->icon->set_onclick("play_$this->internal_name('$this->name')");
          switch ($this->category) {
            // Play movies in a separate window..
            case "movie":
              $embeddedMedia = EmbeddedObject::html();
              $RESPONSE->body->add_popup_script("popup_$this->internal_name", $this->width, $this->height, 100, 40);
              $RESPONSE->body->add_script(
                  "var playing=false;\n"
                . "function play_$this->internal_name(pname) {\n"
                . "  popup_$this->internal_name('');\n"
                . "  with (popup_" . $this->internal_name . "Win.document) {\n"
                . "    open();\n"
                . "    write('');\n"
                . "    write('<body bgcolor=\"#333333\" leftmargin=\"0\" rightmargin=\"0\" topmargin=\"0\" marginwidth=\"0\" marginheight=\"0\">');\n"
                . "    write('<center><font face=arial,helvetica size=2>');\n"
                . "    write('$embeddedMedia');\n"
                . "    title='$this->name';\n"
                . "    close();\n"
                . "  }\n"
                . "}\n"
                );
              break;
            // Play sounds in current window..
            case "audio":
              $RESPONSE->body->add_script(
                  "var playing=false;\n"
                . "function play_$this->internal_name(pname) {\n"
                . "  player=eval('document.' + pname);\n"
                . "  if (player != null) {\n"
                . "    if (playing) {player.Stop();playing=false;}\n"
                . "    else {player.Play();playing=true;}\n"
                . "  }\n"
                . "}\n"
                );
                // Embed the object in this page..
                $s .= EmbeddedObject::html();
              break;
          } //switch category
          $s .= $this->icon->render();
          break;

        // For other browsers we take a slightly more vanilla approach.
        // The click opens a window, with URL set to the media itself.
        // This should get the browser to utilize plugin/helper et al..
        default:
          // Play all media in separate window..
          $RESPONSE->body->add_popup_script("popup_$this->internal_name", $this->width, $this->height, 100, 40);
          $this->icon->set_onclick("popup_$this->internal_name('$this->media')");
          $s .= $this->icon->render();
          break;

      } // switch on browser
    }
    else {
      // No icon, just an embedded object. In this case we will
      // expect problems for browsers other than IE, but c'est la vie..
      $s .= EmbeddedObject::html();
    }
    // Return the HTML..
    return $s;
  } // html
} // MediaObject class

// ----------------------------------------------------------------------
/**
* Document Object
* In actual fact, we don't view Documents as Embedded objects, and instead
* we provide a link to either open a new window, or target a new browser
* window. We point the URL'sto the document, and let the system do
* what it thinks best after that.
* @package html
*/
class DocumentObject extends EmbeddedObject {
  /* URL of document file */
  var $docurl = "";
  // ....................................................................
  /** Constructor
  * @param string $docurl URL of the document file
  * @param integer Width of window showing spreaddocurl
  * @param integer Height of window showing spreaddocurl
  */
  function DocumentObject($docurl, $width="", $height="") {
    global $RESPONSE, $LIBDIR;
    $this->docurl = $docurl;
    $classid = "";
    $this->pluginspage = "";
    $mimetype = "";
    $codebase = "";
    $this->EmbeddedObject($docurl, $mimetype, $classid, $width, $height);
  } // DocumentObject
  // ....................................................................
  /**
  * Render the document object as an image 'icon' which can be clicked to play
  * the media. If image is specified, it must be a valid 'image' object,
  * otherwise a generic library symbol will be used.
  * @param string $tooltip Optional browser mouseover tooltip text
  * @param object $iconimage A custom image object
  * @see img, @see clickable_image
  */
  function AsIcon($tooltip="", $iconimage=false) {
    global $RESPONSE, $LIBDIR;
    if ($iconimage) {
      $this->icon = $iconimage;
    }
    else {
      debugbr("image mime type: ".$this->extn_mimetype);
      switch ($this->extn_mimetype) {
        case CONTENT_MSEXCEL:
          $this->icon = new img("$LIBDIR/img/_excel.gif", "Excel Spreadsheet");
          break;
        case CONTENT_MSWORD:
          $this->icon = new img("$LIBDIR/img/_msword.gif", "Word Document");
          break;
        case CONTENT_PDF:
          $this->icon = new img("$LIBDIR/img/_pdf.gif", "PDF Document");
          break;
        default:
          $this->icon = new img("$LIBDIR/img/_document.gif", "Document");
      } // switch
    }
    if ($tooltip != "") {
      $this->icon->title = $tooltip;
    }
  } // AsIcon
  // ....................................................................
  /** Render the HTML
  * @return string HTML for this object
  */
  function html() {
    global $RESPONSE, $LIBDIR;
    $s = "";
    // If we are doing a clickable image, we need the aid of
    // some handy javascript to make it work..
    if (isset($this->icon)) {
      // Spreaddocurl in separate window..
      $RESPONSE->body->add_popup_script(
        "popup_$this->internal_name",
        $this->width,
        $this->height,
        100,
        40,
        "toolbar=yes,status=yes,scrollbars=yes,resizable=yes"
        );
      $this->icon->set_onclick("popup_$this->internal_name('$this->docurl')");
      $s .= $this->icon->render();
    }
    else {
      // No icon, so we render the object. Note: embedding documents is
      // bound to end in tears, so always render AsLink in your app..
      $s .= EmbeddedObject::html();
    }
    // Return the HTML..
    return $s;
  } // html
} // DocumentObject class

// ----------------------------------------------------------------------
?>