<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    configuration-defs.php                                  */
/* Author:      Paul Waite                                              */
/* Description: Definitions for generic config maintenance              */
/*                                                                      */
/* ******************************************************************** */
/** @package config */

/**
* Configuration
* This is for accessing generic configuration files which store
* useful crappola like GST rate, file paths etc. This object features
* dynamically creatable/deletable configuration fields.
*
* Eg. Usage (single-level configuration):
*    $conf = new configuration("myconfigname");
*    $avar = $conf->value("gstrate");

* Eg. Usage (dual-level config, user preferences example):
*    $conf = new configuration("userprefs", "matthew");
*    $avar = $conf->value("background_colour");
*    // Example of changing it..
*    $conf->set_value("background_colour", "#fefefe");
*    $conf->put();
* @package config
*/
class configuration extends RenderableObject {
  // Public
  /** The name of the current configuration */
  var $config_name = "default";
  /** Identity if the current configuration */
  var $config_id = "default";
  /** Array of configfield objects for the config */
  var $fields;
  /** Set of configuration fields/values in this configuration set */
  var $set;

  // Private
  /** Mode to display the configuration in a form
      @access private */
  var $form_mode = "edit";
  /** Whether config definition exists in database or not
      @access private */
  var $db_config_exists = false;
  /** Whether the configuration exists in database or not
      @access private */
  var $db_configuration_exists = false;
  /** Whether field definitions have changed or not
      @access private */
  var $fieldschanged = false;
  /** Whether configuration data has changed or not
      @access private */
  var $setchanged = false;
  /** Whether to show buttons on forms and auto-process POST or not
      @access private */
  var $autoconfigurate = false;
  // ....................................................................
  /**
  * Constructor
  * Create a new configuration object. Sets basic field attributes.
  * @param string $name The name of the config.
  * @param string $id   The identity of the configuration set in the config
  * @param bool   $auto If true, put the configuration object into autoconfigure mode
  */
  function configuration($name="default", $id="default", $auto=false) {
    // Read it all from disk..
    $this->get($name, $id);
    if ($auto) {
      $this->autoconfigurate = true;
      // Check for POSTings..
      $this->POSTprocess();
    }
  } // configuration
  // ....................................................................
  /**
  * Get the configuration set.
  * Retreives the specified configuration set from database.
  * @param string $name The name of the config.
  * @param string $id   The identity of the configuration set in the config
  */
  function get($name, $id) {
    $this->config_name = $name;
    $this->config_id = $id;
    // Try and find it..
    $q  = "SELECT * FROM ax_config";
    $q .= " WHERE config_name='" . addslashes($name) . "'";
    $stq = dbrecordset($q);
    if ($stq->hasdata) {
      $this->fields = unserialize($stq->field("config_fields"));
      $this->db_config_exists = true;
      // Retrieve possible set..sys_control
      $q  = "SELECT * FROM ax_configuration";
      $q .= " WHERE config_name='" . addslashes($name) . "'";
      $q .= "   AND config_id='" . addslashes($id) . "'";
      $ssq = dbrecordset($q);
      if ($ssq->hasdata) {
        $this->set = unserialize($ssq->field("config_set"));
        $this->db_configuration_exists = true;
      }
      else {
        $this->db_configuration_exists = false;
        if ($this->field_count() > 0) {
          // Re-create bogus empty set from scratch..
          foreach($this->fields as $field) {
            $this->set_value($field->name, $field->defaultvalue());
          }
        }
      }
    }
    else {
      $this->db_config_exists = false;
    }
    // Return true if at least the config exists..
    return $this->db_config_exists;
  } // get
  // ....................................................................
  /**
  * Save the config.
  * Save this config to the database. Create a new one if it
  * doesn't already exist.
  */
  function put() {
    // Deal with brand new config..
    if (!$this->db_config_exists) {
      $sq = new dbinsert("ax_config");
      $sq->set("config_name", $this->config_name);
      $sq->set("config_fields", serialize($this->fields));
      $sq->execute();
      $this->fieldschanged = false;
      $this->db_config_exists = true;
    }
    // Deal with brand new configuration..
    if (!$this->db_configuration_exists) {
      $sq = new dbinsert("ax_configuration");
      $sq->set("config_name", $this->config_name);
      $sq->set("config_id", $this->config_id);
      $sq->set("config_set", serialize($this->set));
      $sq->execute();
      $this->setchanged = false;
      $this->db_configuration_exists = true;
    }

    // Save updates to database copy..
    if ($this->fieldschanged) {
      $sq = new dbupdate("ax_config");
      $sq->set("config_fields", serialize($this->fields));
      $sq->where("config_name='" . addslashes($this->config_name) . "'");
      $sq->execute();
      $this->fieldschanged = false;
    }
    if ($this->setchanged) {
      $sq = new dbupdate("ax_configuration");
      $sq->set("config_set", serialize($this->set));
      $sq->where("config_name='" . addslashes($this->config_name) . "'");
      $sq->where("and config_id='" . addslashes($this->config_id) . "'");
      $sq->execute();
      $this->setchanged = false;
    }
  } // put
  // ....................................................................
  /**
  * Delete the whole config.
  * Delete this config from the database.
  */
  function delete() {
    // RI wil delete the sets of this config..
    $del = new dbdelete("ax_config");
    $del->where("config_name='" . addslashes($this->config_name) . "'");
    $del->execute();
  } // delete
  // ....................................................................
  /**
  * Return fields present count.
  * @return integer Count of fields current defined in this config
  */
  function field_count() {
    $res = 0;
    if (isset($this->fields)) {
      $res = count($this->fields);
    }
    return $res;
  } // field_count
  // ....................................................................
  /**
  * Check if field exists
  * @param string $fname The name of the field to check.
  * @return bool True if named field exists in this configuration
  */
  function field_exists($fname) {
    $fname = str_replace(" ", "_", $fname);
    return (isset($this->fields) && isset($this->fields[$fname]));
  } // field_exists
  // ....................................................................
  /**
  * Insert a new field into this config.
  * This is just a raw routine for inserting a field in this config set.
  * @param string $fname    Name of the new field
  * @param string $ftype    Var type: 'text','numeric','bool', or 'datetime'
  * @param string $flist    List of allowed values, if any
  * @param string $default Default value, if any
  * @access private
  */
  function field_insert($fname, $ftype="text", $flist="", $default="???") {
    $fname = str_replace(" ", "_", $fname);
    if (!$this->field_exists($fname)) {
      $this->fields[$fname] = new configfield($fname, $ftype, $flist, $default);
      $this->fieldschanged = true;
    }
  } // field_insert
  // ....................................................................
  /**
  * Delete a field from this config set.
  * @param string $fname Name of the field to delete.
  * @access private
  */
  function field_delete($fname) {
    $fname = str_replace(" ", "_", $fname);
    if ($this->field_exists($fname)) {
      // Get rid of local variable..
      if (isset($this->set[$fname])) {
        unset($this->set[$fname]);
        $this->setchanged = true;
      }
      // Get rid of the field..
      if (isset($this->fields[$fname])) {
        unset($this->fields[$fname]);
        $this->fieldschanged = true;
      }
    }
  } // field_delete
  // ....................................................................
  /**
  * Create a new field for all configurations
  * This retro-fits all sets of this config with the new field, and
  * will assign the default value to each one.
  * @param string $fname    Name of the new field
  * @param string $ftype    Var type: 'text','numeric','bool', or 'datetime'
  * @param string $flist    List of allowed values, if any
  * @param string $default Default value, if any
  */
  function field_create($fname, $ftype="text", $flist="", $default="???") {
    if ($fname != "") {
      // Insert the field locally..
      $fname = str_replace(" ", "_", $fname);
      $this->field_insert($fname, $ftype, $flist, $default);

      // Insert it for all other configurations..
      $f = $this->fields[$fname];
      $initialval = $f->defaultvalue();
      $this->set_value($fname, $initialval);
      $q  = "SELECT * FROM ax_configuration";
      $q .= " WHERE config_name='" . addslashes($this->config_name) . "'";
      $q .= "   AND config_id <> '" . addslashes($this->config_id) . "'";
      $sets = dbrecordset($q);
      if ($sets->hasdata) {
        start_transaction();
        do {
          $configid = $sets->field("config_id");
          $configuration = new configuration($this->config_name, $configid);
          $configuration->field_insert($fname, $ftype, $flist, $default);
          $configuration->set_value($fname, $initialval);
          $configuration->put();
        } while ($sets->get_next());
        commit();
      }
    }
  } // field_create
  // ....................................................................
  /**
  * Remove a field from all sets of this config.
  * @param string $fname Name of the field to remove from all configurations
  */
  function field_remove($fname) {
    $fname = str_replace(" ", "_", $fname);
    if ($this->field_exists($fname)) {
      // Delete it locally..
      $this->field_delete($fname);
      $this->put();

      // Delete it for all other sets in the configuration..
      $q  = "SELECT * FROM ax_configuration";
      $q .= " WHERE config_name='" . addslashes($this->config_name) . "'";
      $q .= "   AND config_id <> '" . addslashes($this->config_id) . "'";
      $sets = dbrecordset($q);
      if ($sets->hasdata) {
        start_transaction();
        do {
          $configid = $sets->field("config_id");
          $configuration = new configuration($this->config_name, $configid);
          $configuration->field_delete($fname, $ftype, $flist, $default);
          $configuration->put();
        } while ($sets->get_next());
        commit();
      }
    }
  } // field_remove
  // ....................................................................
  /**
  * Put a value in a field, in a set of the config.
  * @param string $name  Name of the field
  * @param string $value Value to assign to the variable
  */
  function set_value($fname, $fvalue="") {
    $fname = str_replace(" ", "_", $fname);
    if ($this->field_exists($fname)) {
      $field = $this->fields[$fname];
      switch ($field->type) {
        case "text":
          break;
        case "textarea":
          break;
        case "numeric":
          if (!is_numeric($fvalue)) {
            $fvalue = 0;
          }
          break;
        case "bool":
          if ($fvalue) {
            $fvalue = true;
          }
          else {
            $fvalue = false;
          }
          break;
        case "datetime":
          if ($fvalue == "") {
            $fvalue = timestamp_to_datetime(); // Now
          }
          else {
            $fvalue = displaydate_to_datetime($fvalue);
          }
          break;
      } // switch
      // Set the value now..
      $this->set[$fname] = $fvalue;
      $this->setchanged = true;
    }
  } // set_value
  // ....................................................................
  /**
  * Get value from a field in a set of the config.
  * @param string $name  Name of the field to get value of
  * @return mixed The value of the given field in the given set
  */
  function value($fname) {
    $fname = str_replace(" ", "_", $fname);
    if ($this->field_exists($fname)) {
      $field = $this->fields[$fname];
      switch ($field->type) {
        case "datetime":
          $fvalue = datetime_to_displaydate(POSTGRES_STD_FORMAT, $this->set[$fname]);
          break;
        default:
          $fvalue = $this->set[$fname];
      } // switch
      return $fvalue;
    }
    return "";
  } // value
  // ....................................................................
  /**
  * Render a set of the config as a subform.
  * Render all the values of the given set of the config as stacked form
  * elements in a table. Since this is a subform the form tags are
  * not rendered.
  * @return string The HTML
  * @access private
  */
  function editform() {
    global $LIBDIR;
    global $RESPONSE;
    $s = "";
    $editForm = new subform();
    if ($this->field_count() > 0) {
      reset($this->fields);
      foreach ($this->fields as $field) {
        $fvalue = $this->value($field->name);
        $fdispname = str_replace("_", " ", $field->name);
        if (isset($f)) unset($f);
        switch ($field->type) {
          case "text":
          case "numeric":
            if ($field->list != "") {
              $options = explode(",", $field->list);
              $f = new form_combofield($field->name, $fdispname, $fvalue);
              $f->setclass("axcombo");
              foreach ($options as $option) {
                $f->additem($option);
              }
            }
            else {
              $f = new form_textfield($field->name, $fdispname, $fvalue);
              if ($field->type == "text") {
                $f->setclass("axtxtbox");
              }
              else {
                $f->setclass("axnumbox");
              }
            }
            break;
          case "textarea":
            $f = new form_memofield($field->name, $fdispname, $fvalue);
            $f->setclass("axmemo");
            break;
          case "bool":
            $f = new form_checkbox($field->name, $fdispname);
            $f->checked = $fvalue;
            $f->setclass("axchkbox");
            break;
          case "datetime":
            $f = new form_textfield($field->name, $fdispname, datetime_to_displaydate(DISPLAY_DATE_FORMAT, $fvalue));
            $f->setclass("axdatetime");
            break;
        } // switch
        // Add form field to our form..
        if (isset($f)) $editForm->add($f);
      } // foreach
    }
    else {
      $s .= "<p>There are currently no data fields defined.</p>";
    }
    $editForm->add(new form_hiddenfield("_configaction", "edit"));
    if ($this->autoconfigurate) {
      if ($this->field_count() > 0) {
        $editForm->add_button(new image_button("_cfgsave","","","", "$LIBDIR/img/_save.gif",57,15,"Save",0));
      }
      if (!isset($RESPONSE) || $RESPONSE->ismemberof_group("Admin")) {
        $editForm->add_button(new image_button("_cfgadd","","","", "$LIBDIR/img/_add.gif",57,15,"Add",0));
        if ($this->field_count() > 0) {
          $editForm->add_button(new image_button("_cfgdelete","","","", "$LIBDIR/img/_delete.gif",57,15,"Add",0));
        }
      }
    }

    // Render the form..
    $s .= $editForm->render();

    // Return the html..
    return $s;
  } // editform
  // ....................................................................
  /**
  * Return a create configfield subform
  * Returns the HTML for a subform which will allow the user to
  * specify a new configfield.
  * @return string The HTML of the subform to create a new config field
  * @access private
  */
  function addform() {
    global $LIBDIR;
    global $RESPONSE;
    $addForm = new subform();
    $addForm->add_text("<b>Add new configuration field:</b>");
    $addForm->add(new form_textfield("new_fname", "Field name"));
    $f = new form_combofield("new_ftype", "Field type", "text");
    $f->additem("text", "Text");
    $f->additem("textarea", "Textarea");
    $f->additem("numeric", "Numeric");
    $f->additem("bool", "Boolean");
    $f->additem("datetime", "Date/time");
    $addForm->add($f);
    $addForm->add(new form_textfield("new_fvalue", "Initial value", "", EDITABLE, "axtxtbox"));
    $addForm->add(new form_textfield("new_allowed_values", "Allowed values", "", EDITABLE, "axtxtbox"));
    $addForm->add(new form_labelfield("", "<small><em>(optional list of comma-delimited values)</em></small>"));
    $addForm->add(new form_textfield("new_fdefault", "Default value", "", EDITABLE, "axtxtbox"));
    $addForm->add(new form_hiddenfield("_configaction", "add"));
    if ($this->autoconfigurate) {
      if (!isset($RESPONSE) || $RESPONSE->ismemberof_group("Admin")) {
        $addForm->add_button(new image_button("_cfgaddit","","","", "$LIBDIR/img/_add.gif",57,15,"Add",0));
      }
      $addForm->add_button(new image_button("_cfgcancel","","","", "$LIBDIR/img/_cancel.gif",57,15,"Cancel",0));
    }
    return $addForm->render();
  } // addform
  // ....................................................................
  /**
  * Return a delete configfield subform
  * Returns the HTML for a subform which will allow the user to
  * specify a field to remove from the configuration set.
  * @return string The HTML of the subform to specify config field to delete
  * @access private
  */
  function deleteform() {
    global $LIBDIR;
    $delForm = new subform();
    $delForm->add_text("<b>Delete configuration field:</b>");
    $delcombo = new form_combofield("delete_fname", "Field name to delete");
    reset($this->fields);
    foreach($this->fields as $field) {
      $fdispname = str_replace("_", " ", $field->name);
      $delcombo->additem($field->name, $fdispname);
    }
    $delForm->add($delcombo);
    $delForm->add(new form_hiddenfield("_configaction", "delete"));
    if ($this->autoconfigurate) {
      if (!isset($RESPONSE) || $RESPONSE->ismemberof_group("Admin")) {
        $delForm->add_button(new image_button("_cfgdeleteit","","","", "$LIBDIR/img/_delete.gif",57,15,"Delete",0));
      }
      $delForm->add_button(new image_button("_cfgcancel","","","", "$LIBDIR/img/_cancel.gif",57,15,"Cancel",0));
    }
    return $delForm->render();
  } // deleteform
  // ....................................................................
  /**
  * Render this configuration.
  * depending on which form_mode we are in, we render the configuration
  * as an edit form, an add form, or a delete form.
  * @return string The HTML of the subform for this configuration action
  */
  function html() {
    // If we are debugging then dump the raw stuff out
    // of our config out at this point in DBG_DUMP mode..
    switch($this->form_mode) {
      case "add":
        return $this->addform();
        break;
      case "delete":
        return $this->deleteform();
        break;
      default:
        return $this->editform();
    } // switch
  } // html
  // ....................................................................
  /**
  * Process a subform POST.
  * Assume that the fields have been submitted in a form as named
  * in the config, and grab the POSTed values.
  * @param text $id The identity of the set to update from POST
  * @access private
  */
  function POSTprocess() {
    global $HTTP_POST_VARS;
    global $_cfgsave_x, $_cfgadd_x, $_cfgdelete_x, $_cfgcancel_x;
    global $_cfgaddit_x, $_cfgdeleteit_x;
    $postingaction = false;
    if (isset($HTTP_POST_VARS["_configaction"])) {
      $action = $HTTP_POST_VARS["_configaction"];
      switch($action) {
        case "add":
          if (!$this->autoconfigurate || isset($_cfgaddit_x)) {
            $fname = $HTTP_POST_VARS["new_fname"];
            if ($fname != "") {
              debugbr("Adding new element '$fname' !");
              $this->field_create(
                      $fname,
                      $HTTP_POST_VARS["new_ftype"],
                      $HTTP_POST_VARS["new_allowed_values"],
                      $HTTP_POST_VARS["new_fdefault"]
                      );
              $this->set_value(
                      $HTTP_POST_VARS["new_fname"],
                      $HTTP_POST_VARS["new_fvalue"]
                      );
              $postingaction = true;
            }
          }
          $this->form_mode = "edit";
          break;

        case "delete":
          if (!$this->autoconfigurate || isset($_cfgdeleteit_x)) {
            $fname = $HTTP_POST_VARS["delete_fname"];
            if ($fname != "") {
              debugbr("Deleting element '$delete_fname' !");
              $this->field_remove($fname);
              $postingaction = true;
            }
          }
          $this->form_mode = "edit";
          break;

        case "edit":
          if (!$this->autoconfigurate || isset($_cfgsave_x)) {
            if (isset($this->fields)) {
              reset($this->fields);
              foreach ($this->fields as $field) {
                $fname = $field->name;
                debugbr("POSTprocess: checking field '$fname'");
                switch ($field->type) {
                  case "bool":
                    if (isset($HTTP_POST_VARS[$fname])) $fvalue = true;
                    else $fvalue = false;
                    break;
                  default:
                    $fvalue = $HTTP_POST_VARS[$fname];
                } // switch
                $this->set_value($fname, $fvalue);
                $postingaction = true;
              } // foreach
            }
            $this->form_mode = "edit";
          }
          elseif (isset($_cfgdelete_x)) {
            $this->form_mode = "delete";
          }
          elseif (isset($_cfgadd_x)) {
            $this->form_mode = "add";
          }
          break;
      } // switch
    }
    // Save the thing if we changed something..
    if ($postingaction) {
      $this->put();
    }
  } // POSTprocess

} // configuration class

// ----------------------------------------------------------------------
/**
* The configurator class is simply a descendant class of configuration
* which sets the parent class into 'autoconfigure' mode. In this mode
* the forms have buttons supplied, and the instantiation of the class
* automatically processes form POSTs.
* @package config
*/
class configurator extends configuration {
  /**
  * Constructor
  *
  * Create a new configurator object.
  * @param string $name The name of the config.
  * @param string $id   The identity of the configuration set in the config
  */
  function configurator($name="default", $id="default") {
    $this->configuration($name, $id, true);
  } // configurator

} // configurator class

// ----------------------------------------------------------------------
/**
** A Configuration field.
* @package config
*/
class configfield {
  var $name;
  var $type;
  var $list;
  var $default;
  /** Constructor.
  * Create a config field.
  * @param string $fname  Name of the field
  * @param string $ftype  Type of field: 'text', 'numeric', 'bool', or 'datetime'
  * @param string $flist  List of permitted values, if appropriate
  */
  function configfield($fname, $ftype="text", $flist="", $default="???") {
    $this->name  = $fname;
    $this->type  = $ftype;
    $this->list  = $flist;
    if ($default != "???") {
      if ($ftype == "bool") {
        $this->default = ($default == "true" || $default == "t" || $default == "1");
      }
      else $this->default = $default;
    }
    else {
      switch ($ftype) {
        case "text":
        case "textarea":
          $this->default = "";
          break;
        case "numeric":
          $this->default = 0;
          break;
        case "bool":
          $this->default = false;
          break;
      } // switch
    }
  }
  /** Return the default field value */
  function defaultvalue() {
    if (isset($this->default)) {
      return $this->default;
    }
    elseif ($this->type == "datetime") {
      return timestamp_to_datetime();
    }
    return "";
  }
}

// ----------------------------------------------------------------------
?>