#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <Ark/ArkSkeleton.h>
#include <Ark/ArkCache.h>
#include <Ark/ArkModelState.h>
#include <Ark/ArkSystem.h>

namespace Ark
{
   
   // ========================================================================
   // Bone controllers
   // ========================================================================
   
   // Assert that value is in the range [start-end], invert it
   // if start > end, etc..
   scalar
   FixController (scalar value, scalar start, scalar end)
   {
      if (value >  180.0) value -= 360.0;
      if (value < -180.0) value += 360.0;
      return value;
   }
   
   
   // Given a bone controller and an adjustment value for this bone (ie an
   // angle if this bone has a rotation channel), this function tries to
   // correct the value.
   void
   BoneController::ComputeSetting(scalar value, scalar *result)
   {
      assert (result != NULL);

      // wrap 0..360 if it's a rotational controller
      if (m_Channel)
	 value = FixController (value, m_Start, m_End);

      if (value < m_Start) value = m_Start;
      if (value > m_End)   value = m_End;
      
      *result = value;
   }


   // ========================================================================
   // Skeleton
   // ========================================================================
   
   Skeleton::Skeleton (const String &name) : 
      Object (name)
   {}

   Skeleton::~Skeleton ()
   {}

   void
   Skeleton::Animate (ModelState &state, 
		      scalar delta_time)
   {
      const SequencePtr& seq = state.m_Sequence;

      if (!seq)
	 return;

      state.m_Time += delta_time;
      if (state.m_Time >= seq->m_EndTime)
      {
	 if (state.m_Mode == ModelState::LOOP)
	 {
	    state.m_SequenceChanged = true;
	    state.m_Time -= seq->m_EndTime;

	    if (state.m_Time < 0.0)
	       state.m_Time = 0.0;

	    if (state.m_Time >= seq->m_EndTime)
	       state.m_Time = 0.0;
	 }
	 else
	    state.m_Time = seq->m_EndTime;
      }

      state.m_SequenceChanged = false;
   }

   void
   Skeleton::SetupBones (const ModelState &state,
			 Matrix44 *matrices,
			 Matrix44 *rootmatx,
			 scalar scalefactor)
   {
#define MAXSTUDIOBONES 128
      static Vector3 pos[MAXSTUDIOBONES];
      static Quat q[MAXSTUDIOBONES];

      const SequencePtr& seq = state.m_Sequence;

      if (!seq)
      {
	 for (size_t i = 0; i < m_Bones.size(); i++)
	 {
	    pos[i] = m_Bones[i].m_DefPosition;
	    q[i] = Quat (m_Bones[i].m_DefRotation.X,
			 m_Bones[i].m_DefRotation.Y,
			 m_Bones[i].m_DefRotation.Z); 
	 }
      }
      else
      {
	 int frame0 = 0, frame1 = 0;
	 scalar f0, f1;
	 
	 if (!seq->ComputeFrameFactors (state.m_Time,
					     &frame0, &f0,
					     &frame1, &f1))
	 {
	    frame0 = frame1 = 0;
	    f0 = 1.0, f1 = 0.0;
	 }

	 const Keyframe &kf0 = seq->m_Keyframes[frame0];
	 const Keyframe &kf1 = seq->m_Keyframes[frame1];

	 for (size_t i = 0; i < m_Bones.size(); i++)
	 {
	     seq->GetTransformation
		(m_Bones[i], kf0[i], f0, kf1[i], f1, 
		 state.m_Controller,
		 &pos[i], &q[i]);
	 }
      }

      for (size_t i = 0; i < m_Bones.size(); ++i)
      {
	 // Compute rotation
	 Matrix44 bonematrix = q[i].GetMatrix();
	 
	 // Set translation
	 bonematrix.M(3,0) = pos[i].X * scalefactor;
	 bonematrix.M(3,1) = pos[i].Y * scalefactor;
	 bonematrix.M(3,2) = pos[i].Z * scalefactor;
	 
	 matrices[i] = bonematrix;
	 if (m_Bones[i].m_Parent != -1)
	 {
	    // Concatenate this matrix with the parent's one.
	    matrices[i].Multiply (matrices[m_Bones[i].m_Parent]);
	 }
	 else
	 {
	    matrices[i].Multiply (m_RootMatrix);

	    if (rootmatx)
	       matrices[i].Multiply (*rootmatx);
	 }
      }   
   }

   // Need to get a pointer to the cache to be able to retrieve sequences 
   // on demand..
   bool
   Skeleton::PostLoad (Cache* cache)
   {
      m_Cache = cache;
      return true;
   }

   SequencePtr
   Skeleton::GetSequence (const String &name)
   {
      SequenceList::iterator it = m_Sequences.find (name);

      if (it == m_Sequences.end())
      {
	 Sys()->Warning("Cannot find a sequence named '%s' in '%s'",
			name.c_str(),Name().c_str());
	 return SequencePtr();
      }
      else
      {
	 SequencePtr seq;
	 
	 m_Cache->Get(V_SEQUENCE, it->second, seq);
	 return seq;
      }
   }

   BBox
   Skeleton::GetBBox (const SequencePtr& seq,
		      scalar scalefactor) const
   {
      Matrix44 mat; 
	  mat.MakeScale(Vector3(scalefactor,scalefactor,scalefactor));

      BBox bb = seq->m_BBox;
      bb.Transform(mat);

      return bb;
   }

}
