/* $Id: ArkLoaderPSys.cpp,v 1.13 2003/02/04 20:18:15 zongo Exp $
** 
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) David Farrell <fdavid@cyberramp.net>
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <sstream>

#include <Ark/ArkParticle.h>
#include <Ark/ArkLoader.h>
#include <Ark/ArkLexer.h>

namespace Ark
{

   static bool ReadEqualScalar (Lexer &lexer, scalar *res)
   {
      lexer.CheckToken ("=");
      const bool result = lexer.ReadScalar( res );
      lexer.CheckToken (";");
      
      return result;
   }
   
   void
   ParticleTmpl::Read (const String& name, Stream &stream, Cache &cache)
   {
      Lexer lexer (name, stream);
      String token;
      
      do
      {
	 token = lexer.GetToken (Ark::Lexer::IDENTIFIER);
	 
	 if (token == "Sprite")
	 {
	    if (m_pType != TYPE_NULL)
	       lexer.Error ("Type is already set (cannot set to sprite)");
	    
	    lexer.CheckToken ("=");
	    String file = lexer.GetToken (Ark::Lexer::STRING);
	    lexer.CheckToken (";");
	    
	    if (file == "")
	       continue;
	    
	    m_pType = TYPE_SPRITE;
	    cache.Get (V_TEXTURE, UnquoteString(file), &m_Sprite);
	    if (m_Sprite)
	    {
		m_Sprite->Configure();
	    }
	 }
	 else if (token == "LineLength")
	 {
	    if (m_pType != TYPE_NULL)
	       lexer.Error ("Type is already set (cannot set to line)");
	    
	    if ( ReadEqualScalar(lexer, &m_LineLength) )
	    {
		m_pType = TYPE_LINE;
	    }
	 }
	 else if (token == "Color")
	 {
	    lexer.CheckToken ("=");
	    lexer.ReadScalarVector (&m_Color.R, 4);
	    lexer.CheckToken (";");
	 }
	 else if (token == "MaxLife")     ReadEqualScalar (lexer,&m_MaxLife);
	 else if (token == "MinLife")     ReadEqualScalar (lexer,&m_MinLife);
	 else if (token == "Weigth")       ReadEqualScalar (lexer,&m_Weigth);
	 else if (token == "Radius")       ReadEqualScalar (lexer,&m_Radius);
	 else if (token == "RandomScale") ReadEqualScalar (lexer,&m_RandomScale);
	 else if (token == "Flags")
	 {
	    if (!lexer.CheckToken ("="))
	       continue;
	    
	    m_Flags = 0;
	    do
	    {
	       String token = lexer.GetToken (Ark::Lexer::IDENTIFIER);
	       
	       if (token == "DIE_COLLISION")
		  m_Flags |= DIE_COLLISION;
	       else if (token == "DIE_AGE")
		  m_Flags |= DIE_AGE;
	       else if (token == "DIE_RESPAWN")
		  m_Flags |= DIE_RESPAWN;
	       else if (token == "FCT_GRAVITY")
		  m_Flags |= FCT_GRAVITY;
	       else if (token == "FCT_RANDOM")
		  m_Flags |= FCT_RANDOM;
	       else if (token == "FCT_WIND")
		  m_Flags |= FCT_WIND;
	       else if (token == "FCT_STRENGTH")
		  m_Flags |= FCT_STRENGTH;
	       
	       Ark::String newtok = lexer.GetToken (Ark::Lexer::SYMBOL);
	       if (newtok != "|")
		  break;
	    } while (1);
	 }
      } while (token != "");
   }

   class LoaderPSys : public Loader
   {
      public:
	 /// Returns true if the file pointed to by \c name does really
	 /// contains a TEX texture file definition.
	 virtual bool CanLoad (ObjectType type, Stream &file,
			       const String &name, const String &args)
	 {
	    if (type != V_PARTICLE)
	       return false;
	    
	    String line;
	    std::getline (file, line);
	    if (line.find("Ark Particle Def", 0) != String::npos)
	       return true;
	    
	    return false;
	 }
	 
	 /// Load the file pointed to by \c name, and read a object in it.
	 /// It will update the progress every \c granularity percents.
	 virtual bool Load (Object *vis, Stream &file, const String &name, const String &args,
			    Cache *cache,
			    Progress *progress, int granularity)
	 {
	    if (vis == NULL || vis->Type() != V_PARTICLE)
	       return false;

	    ((ParticleTmpl*) vis)->Read (name, file, *cache);
	    return true;
	 }
	 
	 /// Returns informations about the formats supported by this loader.
	 virtual String GetInformations()
	 {
	    return "Ark Particle Definition";
	 }
   };
   
   void ark_AddPSysLoader (Loaders *loaders)
   {
      loaders->Add (new LoaderPSys());
   }


}
