/* Copyright (C) 2002, 2003 Vincent Penquerc'h.
   This file is part of the alogg library.
   Written by Vincent Penquerc'h <lyrian -at- kezako -dot- net>.

   Allegro and the alogg grabber plugin are gift-ware. They were created by
   a number of people working in cooperation, and are given to you freely as
   a gift. You may use, modify, redistribute, and generally hack them about
   in any way you like, and you do not have to give us anything in return.
   However, if you like these products you are encouraged to thank us by
   making a return gift to the Allegro/alogg community. This could be by
   writing an add-on package, providing a useful bug report, making an
   improvement to the library and/or the plugin, or perhaps just releasing
   the sources of your program so that other people can learn from them.
   If you redistribute parts of this code or make a game using it, it
   would be nice if you mentioned Allegro and alogg somewhere in the
   credits, but you are not required to do this. We trust you not to abuse
   our generosity. */

#include <stdio.h>

#ifdef ALOGG_USE_TREMOR
#include <tremor/ivorbiscodec.h>
#include <tremor/ivorbisfile.h>
#else
#include <vorbis/codec.h>
#include <vorbis/vorbisfile.h>
#endif

#include "allegro.h"
#include "../datedit.h"

#include <alogg/aloggint.h>
#include <alogg/alogg.h>



/* Used to keep track of Ogg/Vorbis streaming from the grabber interface */
static struct alogg_stream *global_datogg_stream=NULL;

/* updates streaming of the currently playing stream, if any */
static void global_stream_updater()
{
   if (global_datogg_stream) {
      if (!alogg_update_streaming(global_datogg_stream)) {
         alogg_stop_streaming(global_datogg_stream);
         global_datogg_stream=NULL;
      }
   }
}

/* stops streaming - called liberally */
static void stop_streaming()
{
   if (global_datogg_stream) {
      remove_int(&global_stream_updater);
      alogg_stop_streaming(global_datogg_stream);
      global_datogg_stream=NULL;
   }
}



/* creates a new Ogg/Vorbis object */
static void *makenew_ogg(long *size)
{
   datogg_object *data=malloc(sizeof(datogg_object));
   if (data) {
      data->size=0;
      data->position=0;
      data->data=NULL;
      data->pcmdata=NULL;
   }
   return data;
}



/* displays an Ogg/Vorbis object in the grabber object view window */
static void plot_ogg(AL_CONST DATAFILE *dat, int x, int y)
{
#if ALLEGRO_VERSION*0x10000+ALLEGRO_SUB_VERSION*0x100+ALLEGRO_WIP_VERSION>=0x040104
   textout_ex(screen, font, "Double-click in the item list to play it", x, y+32, gui_fg_color, gui_bg_color);
#else
   textout(screen, font, "Double-click in the item list to play it", x, y+32, gui_fg_color);
#endif
}



/* handles double-clicking on an Ogg/Vorbis object in the grabber */
static int dclick_ogg(DATAFILE *dat)
{
   datogg_object *ovd=(datogg_object*)dat->dat;
   if (!ovd) return D_O_K;
   stop_streaming();
   install_int_ex(&global_stream_updater,BPS_TO_TIMER(20));
   global_datogg_stream=aloggint_start_streaming_datogg(ovd,4096);
   return D_O_K;
}



/* returns an information string describing an Ogg/Vorbis object */
static void get_ogg_desc(AL_CONST DATAFILE *dat, char *s)
{
   int ret;
   int channels,freq;
#ifdef ALOGG_USE_TREMOR
   ogg_int64_t length;
#else
   float length;
#endif
   ASSERT(dat);
   ASSERT(s);
   ASSERT(dat->type==DAT_OGG_VORBIS);
   ret=aloggint_get_info((datogg_object*)dat->dat,&channels,&freq,&length);
   if (ret==0) {
     sprintf(
       s,"Ogg/Vorbis stream, %d channels, %d Hz, %2.2f s",
#ifdef ALOGG_USE_TREMOR
       channels,freq,length/1000.0f
#else
       channels,freq,length
#endif
     );
   }
   else {
     sprintf(s,"Invalid Ogg/Vorbis stream");
   }
}



/* exports an Ogg/Vorbis stream into an external file */
static int export_ogg(AL_CONST DATAFILE *dat, AL_CONST char *filename)
{
   datogg_object *ovd;
   PACKFILE *f;

   ASSERT(dat);
   ASSERT(filename);

   ovd=(datogg_object*)dat->dat;
   ASSERT(ovd);
   f=pack_fopen(filename,F_WRITE);
   if (!f) return 0;
   pack_fwrite(ovd->data,ovd->size,f);
   pack_fclose(f);
   return (errno==0);
}



/* imports an Ogg/Vorbis stream from an external file */
static void *grab_ogg(AL_CONST char *filename, long *size, int x, int y, int w, int h, int depth)
{
   datogg_object *ovd;
   PACKFILE *f;

   ASSERT(filename);

   f=pack_fopen(filename,"r");
   if (!f) return NULL;
   ovd=makenew_ogg(NULL);
   ovd->size=f->todo;
   ovd->data=malloc(f->todo);
   if (!ovd->data) {
     pack_fclose(f);
     free(ovd);
     return NULL;
   }
   pack_fread(ovd->data,ovd->size,f);
   pack_fclose(f);
   return ovd;
}



/* saves an Ogg/Vorbis stream into the datafile format */
static
#if ALLEGRO_VERSION*0x10000+ALLEGRO_SUB_VERSION*0x100+ALLEGRO_WIP_VERSION>=0x040104
int save_ogg(DATAFILE *dat, AL_CONST int *fixed_prop, int pack, int pack_kids, int strip, int sort, int verbose, int extra, PACKFILE *f)
#else
void save_ogg(DATAFILE *dat, int packed, int packkids, int strip, int *keeplist, int sort, int verbose, int extra, PACKFILE *f)
#endif
{
   datogg_object *ovd;
   ASSERT(dat);
   ovd=(datogg_object*)dat->dat;
   ASSERT(ovd);
   pack_fwrite(ovd->data,ovd->size,f);
#if ALLEGRO_VERSION*0x10000+ALLEGRO_SUB_VERSION*0x100+ALLEGRO_WIP_VERSION>=0x040104
#if ALLEGRO_VERSION*0x10000+ALLEGRO_SUB_VERSION*0x100+ALLEGRO_WIP_VERSION>=0x040106
   return TRUE;
#else
   return 0;
#endif
#endif
}



static void *load_oggvorvis_object(PACKFILE *f,long size)
{
   datogg_object *ovd;

   ASSERT(f);

   ovd=malloc(sizeof(datogg_object));
   if (!ovd) return NULL;

   ovd->size=size;
   ovd->position=0;
   ovd->data=NULL;
   ovd->pcmdata=NULL;
   if (ovd->size) {
      ovd->data=malloc(ovd->size);
      pack_fread(ovd->data,ovd->size,f);
   }
   return ovd;
}

static void destroy_oggvorbis_object(void *data)
{
   datogg_object *ovd=(datogg_object*)data;
   if (ovd) {
      stop_streaming();
      if (ovd->data) free(ovd->data);
      if (ovd->pcmdata) free(ovd->pcmdata);
      free(ovd);
   }
}

void datogg_init()
{
  register_datafile_object(
    DAT_OGG_VORBIS,&load_oggvorvis_object,&destroy_oggvorbis_object
  );
}


/* plugin interface header */
DATEDIT_OBJECT_INFO datogg_info =
{
   DAT_OGG_VORBIS,
   "Ogg/Vorbis stream", 
   get_ogg_desc,
   makenew_ogg,
   save_ogg,
   plot_ogg,
   dclick_ogg,
   NULL
};



DATEDIT_GRABBER_INFO datogg_grabber =
{
   DAT_OGG_VORBIS,
   "ogg",
   "ogg",
   grab_ogg,
   export_ogg
};

