#| 

   Fully Ordered Finite Sets, Version 0.81
   Copyright (C) 2003, 2004 by Jared Davis <jared@cs.utexas.edu>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public Lic-
   ense along with this program; if not, write to the Free Soft-
   ware Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.



 sort.lisp

   This file provides sorting algorithms to convert lists into sets.

|#

(in-package "SETS")
(include-book "outer")
(set-verify-guards-eagerness 2)
(enable-set-reasoning)


; First I will just give a simple insertion sort.  This is probably 
; good enough for small lists, but it will have quadratic complexity
; and that is no good.

(defun tr-insert-sort (x acc)
  (declare (xargs :guard (and (true-listp x)
                              (setp acc))))
  (if (endp x)
      acc
    (tr-insert-sort (cdr x) (insert (car x) acc))))

(defun insert-sort (x)
  (declare (xargs :guard (true-listp x)
                  :verify-guards nil))
  (mbe :logic (if (endp x)
                  nil
                (insert (car x) (insert-sort (cdr x))))
       :exec (tr-insert-sort x nil)))

(defthm insert-sort-set 
  (setp (insert-sort x)))

(verify-guards insert-sort)
          
(defthm member-x-then-member-sorted
  (iff (SETS::in a (insert-sort x))
       (member-equal a x)))


; So how about a merge sort.

(defun split-list (x)
  (declare (xargs :guard (true-listp x)))
  (cond ((endp x) (mv nil nil))
        ((endp (cdr x)) (mv (list (car x)) nil))
        (t (mv-let (part1 part2)
                   (split-list (cddr x))
                   (mv (cons (car x) part1)
                       (cons (cadr x) part2))))))

(defthm split-list-membership
  (iff (member-equal a x)
       (or (member-equal a (mv-nth 0 (split-list x)))
           (member-equal a (mv-nth 1 (split-list x))))))

(defthm split-list-part1-truelist
  (true-listp (mv-nth 0 (split-list x)))
  :rule-classes :type-prescription)

(defthm split-list-part2-truelist
  (true-listp (mv-nth 1 (split-list x)))
  :rule-classes :type-prescription)

(defthm split-list-length-part1
  (implies (consp (cdr x))
           (equal (len (mv-nth 0 (split-list x)))
                  (+ 1 (len (mv-nth 0 (split-list (cddr x))))))))

(defthm split-list-length-part2
  (implies (consp (cdr x))
           (equal (len (mv-nth 1 (split-list x)))
                  (+ 1 (len (mv-nth 1 (split-list (cddr x))))))))

(defthm split-list-length-less-part1
  (implies (consp (cdr x))
           (< (len (mv-nth 0 (split-list x)))
              (len x))))

(defthm split-list-length-less-part2
  (implies (consp (cdr x))
           (< (len (mv-nth 1 (split-list x)))
              (len x))))

(in-theory (disable split-list-length-part1
                    split-list-length-part2))

(defun mergesort (x)
  (declare (xargs 
    :guard (true-listp x)
    :measure (len x)
    :hints(("Goal" :use ((:instance split-list-length-less-part1)
                         (:instance split-list-length-less-part2))))
    :verify-guards nil))
  (if (endp x) nil
    (if (endp (cdr x)) (insert (car x) nil)
      (mv-let (part1 part2)
              (split-list x)
              (union (mergesort part1) (mergesort part2))))))

(defthm mergesort-set
  (setp (mergesort x)))

(verify-guards mergesort
  :hints(("Goal" :in-theory (disable mv-nth))))

(in-theory (disable split-list-membership))

(defthm mergesort-membership-2
  (implies (member-equal a x)
           (SETS::in a (mergesort x)))
  :hints(("Subgoal *1/3" :use (:instance split-list-membership))))

(defthm mergesort-membership-1
  (implies (SETS::in a (mergesort x))
           (member-equal a x))
  :hints(("Subgoal *1/6" :use (:instance split-list-membership))
         ("Subgoal *1/5" :use (:instance split-list-membership))
         ("Subgoal *1/4" :use (:instance split-list-membership))))

(defthm mergesort-membership
  (iff (SETS::in a (mergesort x))
       (member-equal a x)))

