/*
 * Copyright (c) 2002, 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Liam Stewart
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.vise;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;

import com.redhat.rhdb.misc.RHDBUtils;

/**
 * Implementation of AbstractAction. Added features are a LARGE_ICON
 * key and helper methods for getting/setting values of name, short
 * description, long description, large icon, small icon, mnemonic,
 * and accelerator. Also add methods for retrieving pre-configured
 * buttons, toolbar buttons, and menu items.
 *
 * @author <a href="mailto:liams@redhat.com">Liam Stewart</a>
 * @version 0.0
 */
public abstract class GenericAction extends AbstractAction {
	private EventListenerList listenerList;
	
	/** Large Icon key */
	public static final String LARGE_ICON = "LargeIcon";
	
	/** Index of Mnmemonic key */
	public static final String MNEMONIC_KEY_INDEX = "MnemonicKeyIndex";

	/** Button should be icon-only */
	public static final int ICON = 0;

	/** Button should be text-only */
	public static final int TEXT = 1;

	/** Button should be both icon and text */
	public static final int ICON_TEXT = 2;
	
	/**
	 * Creates a new <code>GenericAction</code> instance.
	 */
	public GenericAction()
	{
		super();
	}
	
	/**
	 * Creates a new <code>GenericAction</code> instance with a given
	 * name.
	 *
	 * @param name a <code>String</code> value
	 */
	public GenericAction(String name)
	{
		super(name);
	}

	/**
	 * Creates a new <code>GenericAction</code> instance with a given
	 * name and icon.
	 *
	 * @param name a <code>String</code> value
	 * @param icon an <code>Icon</code> value
	 */
	public GenericAction(String name, Icon icon)
	{
		super(name, icon);
	}

	/**
	 * Get a JMenuItem that is configured from this GenericAction.
	 *
	 * @return a <code>JMenuItem</code> value
	 */
	public JMenuItem getMenuItem()
	{
		JMenuItem menuitem = new JMenuItem();

		menuitem.setAction(this);
		menuitem.setAccelerator(getAccelerator());
		if ((RHDBUtils.isJava14() && (getValue(MNEMONIC_KEY_INDEX) != null)))
			menuitem.setDisplayedMnemonicIndex(((Integer)getValue(MNEMONIC_KEY_INDEX)).intValue());

		return menuitem;
	}

	/**
	 * Get a JButton that is configured from this GenericAction. The
	 * button is suitable for use in toolbars (icon only).
	 *
	 * @return a <code>JButton</code> value
	 */
	public JButton getToolbarButton()
	{
		JButton button = new JButton();

		button.setAction(this);
		button.setIcon(getLargeIcon());
		button.setText(null);
		button.setMargin(new Insets(0, 0, 0, 0));
		button.setMnemonic(KeyEvent.VK_UNDEFINED);

		return button;
	}

	/**
	 * Configure a JButton to this GenericAction. The
	 * button is suitable for use in toolbars (icon only).
	 *
	 * @param the <code>JButton</code> to configure
	 */
	public void configureToolbarButton(JButton button)
	{
		button.setAction(this);
		button.setIcon(getLargeIcon());
		button.setText(null);
		button.setMargin(new Insets(0, 0, 0, 0));
		button.setMnemonic(KeyEvent.VK_UNDEFINED);
	}

	/**
	 * Get a JButton that is configured from this GenericAction. This
	 * button is a text-only one.
	 *
	 * @return a <code>JButton</code> value
	 */
	public JButton getButton()
	{
		return getButton(TEXT);
	}
	
	/**
	 * Get a JButton that is configured from this GenericAction. The
	 * style of the button should be one of ICON, TEXT, or ICON_TEXT.
	 *
	 * @param style an <code>int</code> value
	 * @return a <code>JButton</code> value
	 */
	public JButton getButton(int style)
	{
		JButton button = new JButton();
		
		button.setAction(this);

		if (getMnemonic() != null)
			button.setMnemonic(getMnemonic().intValue());

		switch (style)
		{
			case ICON:
				button.setIcon(getLargeIcon());
				button.setText(null);
				break;
			case ICON_TEXT:
				button.setIcon(getLargeIcon());
				button.setText(getName());
				break;
			case TEXT:
			default:
				button.setIcon(null);
				button.setText(getName());
				break;
		}

		return button;
	}

	/**
	 * Get the action command.
	 *
	 * @return a <code>String</code> value
	 */
	public String getActionCommand()
	{
		return (String) getValue(ACTION_COMMAND_KEY);
	}

	/**
	 * Set the action commend.
	 *
	 * @param s a <code>String</code> value
	 */
	public void setActionCommand(String s)
	{
		putValue(ACTION_COMMAND_KEY, s);
	}

	/**
	 * Get the short description.
	 *
	 * @return a <code>String</code> value
	 */
	public String getShortDescription()
	{
		return (String) getValue(SHORT_DESCRIPTION);
	}

	/**
	 * Set the short description.
	 *
	 * @param s a <code>String</code> value
	 */
	public void setShortDescription(String s)
	{
		putValue(SHORT_DESCRIPTION, s);
	}

	/**
	 * Get the long description.
	 *
	 * @return a <code>String</code> value
	 */
	public String getLongDescription()
	{
		return (String) getValue(LONG_DESCRIPTION);
	}

	/**
	 * Get the short description.
	 *
	 * @param s a <code>String</code> value
	 */
	public void setLongDescription(String s)
	{
		putValue(LONG_DESCRIPTION, s);
	}

	/**
	 * Get the name.
	 *
	 * @return a <code>String</code> value
	 */
	public String getName()
	{
		return (String) getValue(NAME);
	}

	/**
	 * Set the name.
	 *
	 * @param s a <code>String</code> value
	 */
	public void setName(String s)
	{
		putValue(NAME, s);
	}

	/**
	 * Get the small icon.
	 *
	 * @return an <code>Icon</code> value
	 */
	public Icon getSmallIcon()
	{
		return (Icon) getValue(SMALL_ICON);
	}

	/**
	 * Set the small icon.
	 *
	 * @param i an <code>Icon</code> value
	 */
	public void setSmallIcon(Icon i)
	{
		putValue(SMALL_ICON, i);
	}

	/**
	 * Get the large icon.
	 *
	 * @return an <code>Icon</code> value
	 */
	public Icon getLargeIcon()
	{
		return (Icon) getValue(LARGE_ICON);
	}

	/**
	 * Set the large icon.
	 *
	 * @param i an <code>Icon</code> value
	 */
	public void setLargeIcon(Icon i)
	{
		putValue(LARGE_ICON, i);
	}

	/**
	 * Get the mnemonic.
	 *
	 * @return an <code>Integer</code> value
	 */
	public Integer getMnemonic()
	{
		return (Integer) getValue(MNEMONIC_KEY);
	}

	/**
	 * Set the mnemonic.
	 *
	 * @param i an <code>Integer</code> value
	 */
	public void setMnemonic(Integer i)
	{
		putValue(MNEMONIC_KEY, i);
	}

	/**
	 * Get the accelerator.
	 *
	 * @return a <code>KeyStroke</code> value
	 */
	public KeyStroke getAccelerator()
	{
		return (KeyStroke) getValue(ACCELERATOR_KEY);
	}

	/**
	 * Set the accelerator.
	 *
	 * @param ks a <code>KeyStroke</code> value
	 */
	public void setAccelerator(KeyStroke ks)
	{
		putValue(ACCELERATOR_KEY, ks);
	}

	/**
	 * Action listener. By default, rebounds all events to registered
	 * ActionListeners.
	 *
	 * @param e an <code>ActionEvent</code> value
	 */
	public void actionPerformed(ActionEvent e)
	{
		if (listenerList != null)
		{
			Object[] listeners = listenerList.getListenerList();

			for (int i = listeners.length - 2; i >= 0; i -= 2)
			{
				ActionEvent ev = new ActionEvent(e.getSource(), e.getID(), (String) getValue(ACTION_COMMAND_KEY));
				((ActionListener)listeners[i+1]).actionPerformed(ev);
			}
		}
	}

	/**
	 * Add an action listener.
	 *
	 * @param l an <code>ActionListener</code> value
	 */
	public void addActionListener(ActionListener l)
	{
		if (listenerList == null)
		{
			listenerList = new EventListenerList();
		}

		listenerList.add(ActionListener.class, l);
	}

	/**
	 * Remove an action listner.
	 *
	 * @param l an <code>ActionListener</code> value
	 */
	public void removeActionListener(ActionListener l)
	{
		if (listenerList == null)
			return;

		listenerList.remove(ActionListener.class, l);
	}
		
}// GenericAction
