/*
 * Copyright (c) 2002, 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Liam Stewart
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.treedisplay;

import java.awt.*;
import java.awt.event.*;
import java.awt.geom.*;

import java.util.ArrayList;
import java.util.Iterator;

import javax.swing.*;
import javax.swing.plaf.ComponentUI;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;

/**
 * UI delegate for TreeDisplay objects.
 *
 * @author <a href="mailto:liams@redhat.com">Liam Stewart</a>
 * @version 0.0
 */
public class BasicTreeDisplayUI extends TreeDisplayUI implements MouseListener, KeyListener {

	private TreeDisplay td;

	/**
	 * Creates a new <code>BasicTreeDisplayUI</code> instance.
	 */
	public BasicTreeDisplayUI() { }

	//
	// UI stuff (create, install, uninstall)
	//
	
	/**
	 * Create a new UI element for the given component.
	 *
	 * @param c a <code>JComponent</code> value
	 * @return a <code>ComponentUI</code> value
	 */
	public static ComponentUI createUI(JComponent c)
	{
		return new BasicTreeDisplayUI();
	}

	/**
	 * Install a UI for the given component.
	 *
	 * @param c a <code>JComponent</code> value
	 */
	public void installUI(JComponent c)
	{
		if (!(c instanceof TreeDisplay)) return;
		
		td = (TreeDisplay) c;

		// add event listeners
		td.addMouseListener(this);
		td.addKeyListener(this);
	}
	
	/**
	 * Remove the UI from the given component.
	 *
	 * @param c a <code>JComponent</code> value
	 */
	public void uninstallUI(JComponent c)
	{
		if (!(c instanceof TreeDisplay)) return;
		if (c != td) return;
		
		// remove event listeners
		td.removeMouseListener(this);
	}

	//
	// painting methods
	//
	
	/**
	 * Draw tree. This method really just takes care of setting the
	 * scale, doing translations, etc. and calling relevant node /
	 * edge drawing operations for each node / edge.
	 *
	 * @param g a <code>Graphics</code> value
	 * @param c a <code>JComponent</code> value
	 */
	public void paint(Graphics g, JComponent c)
	{
		if (c != td)
			return;

		int trans_x, trans_y;
		double zoom;

		TreeRenderer renderer = td.getTreeRenderer();

		// we want to use Java2D features
 		Graphics2D g2d = (Graphics2D) g;

		// don't draw over insets & borders
		Insets insets = c.getInsets();
		g2d.translate(insets.left, insets.top);

		Rectangle rect = new Rectangle(td.getTranslateX(),
									   td.getTranslateY(),
									   td.getZoomedWidth(),
									   td.getZoomedHeight());

		if (! rect.intersects(g.getClipBounds()))
			return;

		// anti-alias?
		if (td.isAntiAlias())
		{
			g2d.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
								 RenderingHints.VALUE_ANTIALIAS_ON);
		}
		else
		{
			g2d.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
								 RenderingHints.VALUE_ANTIALIAS_OFF);
		}

		// translate
		g2d.translate(td.getTranslateX(), td.getTranslateY());

		// scale
		zoom = td.getZoomLevel();
		g2d.scale(zoom, zoom);

		// pad
		g2d.translate(td.getPadX(), td.getPadY());

		// draw
		Iterator nodes = td.getTreeLayoutModel().getNodes();
		Iterator edges = td.getTreeLayoutModel().getEdges();

		while (nodes.hasNext())
			renderer.drawNode((TreeLayoutNode)nodes.next(), g2d, td.getSelectionModel());

		while (edges.hasNext())
			renderer.drawEdge((TreeLayoutEdge)edges.next(), g2d, td.getSelectionModel());
	}

	/**
	 * Get the preferred size of the component.
	 *
	 * @param c a <code>JComponent</code> value
	 * @return a <code>Dimension</code> value
	 */
	public Dimension getPreferredSize(JComponent c)
	{
		int x = td.getZoomedWidth();
		int y = td.getZoomedHeight();

		return new Dimension(x, y);
	}

	/**
	 * Get the minimum size of the component.
	 *
	 * @param c a <code>JComponent</code> value
	 * @return a <code>Dimension</code> value
	 */
	public Dimension getMinimumSize(JComponent c)
	{
		return getPreferredSize(c);
	}

	/**
	 * Get the maximum size of the component.
	 *
	 * @param c a <code>JComponent</code> value
	 * @return a <code>Dimension</code> value
	 */
	public Dimension getMaximumSize(JComponent c)
	{
		return getPreferredSize(c);
	}

	//
	// methods implementing KeyListner
	//
	
	public void keyPressed(KeyEvent event)
	{
		int keyCode = event.getKeyCode();
		
		// We only care about a SHIFT-ARROW combination
		if (!event.isShiftDown() ||
			((keyCode != KeyEvent.VK_LEFT) &&
			 (keyCode != KeyEvent.VK_RIGHT) &&
			 (keyCode != KeyEvent.VK_UP) &&
			 (keyCode != KeyEvent.VK_DOWN)))
			return;		// nothing we care about

        TreeLayoutModel tlmodel = td.getTreeLayoutModel();

		// Find the current node
		TreeLayoutNode node = null;
		TreePath tp = td.getSelectionPath();
		if (tp != null)
			node = tlmodel.getNode(tp.getLastPathComponent());
			
		// If we get the node (we should always do), move accordingly
		if (node != null)
		{
		    // Adjust the movement based on the tree orientation
		    switch (tlmodel.getOrientation())
		    {
			    case TreeDisplay.TOP:
				    // Do nothing.  We are fine
				    break;
			    case TreeDisplay.BOTTOM:
			        // Switch UP and DOWN
				    if (keyCode == KeyEvent.VK_UP)
				        keyCode = KeyEvent.VK_DOWN;
					else if (keyCode == KeyEvent.VK_DOWN)
					    keyCode = KeyEvent.VK_UP;
				    break;
			    case TreeDisplay.LEFT:
				    // Here UP means LEFT, RIGHT means DOWN, DOWN means RIGHT, LEFT means UP
				    if (keyCode == KeyEvent.VK_UP)
				        keyCode = KeyEvent.VK_LEFT;
					else if (keyCode == KeyEvent.VK_RIGHT)
					    keyCode = KeyEvent.VK_DOWN;
					else if (keyCode == KeyEvent.VK_DOWN)
					    keyCode = KeyEvent.VK_RIGHT;
					else if (keyCode == KeyEvent.VK_LEFT)
					    keyCode = KeyEvent.VK_UP;
				    break;
			    case TreeDisplay.RIGHT:
				    // Here UP means LEFT, LEFT means DOWN, DOWN means RIGHT, RIGHT means UP
				    if (keyCode == KeyEvent.VK_UP)
				        keyCode = KeyEvent.VK_LEFT;
					else if (keyCode == KeyEvent.VK_LEFT)
					    keyCode = KeyEvent.VK_DOWN;
					else if (keyCode == KeyEvent.VK_DOWN)
					    keyCode = KeyEvent.VK_RIGHT;
					else if (keyCode == KeyEvent.VK_RIGHT)
					    keyCode = KeyEvent.VK_UP;
				    break;
		    }
		
			TreeLayoutNode pnode = node.getParent();
			int numChildren = node.getChildCount();
			
			if (keyCode == KeyEvent.VK_UP)
			{
				if (pnode != null)
				{
					select(pnode, td);
					td.centerOnSelected();
				}
			}
			else if (keyCode == KeyEvent.VK_DOWN)
			{
				if (numChildren > 0)
				{
					TreeLayoutNode n = node.getChild(0);
					select(n, td);
					td.centerOnSelected();
				}
			}
			else if (keyCode == KeyEvent.VK_LEFT)
			{
				if ((pnode != null) &&
					(pnode.getChildCount() > 1))
				{
					int me = pnode.getIndexOfChild(node);
					
					if (me > 0)
					{
						TreeLayoutNode n = pnode.getChild(me - 1);
						select(n, td);
						td.centerOnSelected();
					}
				}
			}
			else if (keyCode == KeyEvent.VK_RIGHT)
			{
				if ((pnode != null) &&
					(pnode.getChildCount() > 1))
				{
					int me = pnode.getIndexOfChild(node);
					
					if (me < (pnode.getChildCount() - 1))
					{
						TreeLayoutNode n = pnode.getChild(me + 1);
						select(n, td);
						td.centerOnSelected();
					}
				}
			}
		}

	}
			
	public void keyReleased(KeyEvent event) {}	
	public void keyTyped(KeyEvent event) {}	
		
	//
	// methods implementing MouseListner
	//
	
	/**
	 * Called whenever a mouse button is clicked.
	 *
	 * @param e a <code>MouseEvent</code> value
	 */
	public void mouseClicked(MouseEvent e)
	{
		if (td != e.getComponent())
			return;

		td.requestFocus();

		if (SwingUtilities.isLeftMouseButton(e))
		{
			switch (td.getMode())
			{
				case TreeDisplay.SELECT:
					TreeLayoutNode n = td.getNode(e.getX(), e.getY());
					TreeLayoutNode s = null;

					TreePath tp = td.getSelectionPath();
					if (tp != null)
						s = td.getTreeLayoutModel().getNode(tp.getLastPathComponent());

					if (n == null)
						break;

					switch (td.getSelectionMode())
					{
						case TreeSelectionModel.SINGLE_TREE_SELECTION:
							if (s != null && n == s && e.isControlDown())
								unselect(n, td);
							else if (s != null && n == s)
								;
							else
								select(n, td);

							// double click centers on selected node
							if (! e.isControlDown() && e.getClickCount() == 2)
								td.centerOnSelected();
							break;
						default:
							// nothing yet
					}

					break;
				case TreeDisplay.ZOOMIN:
					td.setZoomLevel(td.getZoomLevel() * 2);
					break;
				case TreeDisplay.ZOOMOUT:
					td.setZoomLevel(td.getZoomLevel() / 2);
					break;
				case TreeDisplay.PAN:
					break;
				default:
					break;
			}
		}
	}

	/**
	 * Called whenever the mouse pointer enters the component.
	 *
	 * @param e a <code>MouseEvent</code> value
	 */
	public void mouseEntered(MouseEvent e) { }
	

	/**
	 * Called whenever the mouse pointer exits the component.
	 *
	 * @param e a <code>MouseEvent</code> value
	 */
	public void mouseExited(MouseEvent e) { }

	/**
	 * Called whenever a mouse button has been pressed and the mouse
	 * pointer is inside the component.
	 *
	 * @param e a <code>MouseEvent</code> value
	 */
	public void mousePressed(MouseEvent e) { }
	
	/**
	 * Called whenever a mouse button has been released and the mouse
	 * pointer is inside the component.
	 *
	 * @param e a <code>MouseEvent</code> value
	 */
	public void mouseReleased(MouseEvent e) { }

	//
	// private methods
	//

	// unselect given node
	private void unselect(TreeLayoutNode s, TreeDisplay td)
	{
		td.setSelectionPath(null);
	}

	// select given node
	private void select(TreeLayoutNode s, TreeDisplay td)
	{
		if (s == null && td == null)
			return;

		TreePath tp = td.getTreeLayoutModel().getTreePath(s);
		td.setSelectionPath(tp);
	}
}// BasicTreeDisplayUI
